 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>          Data Management Segment Definitions
   >>>>
   >>>>   Static:
   >>>>                 _kdms_construct_segment_defin()
   >>>>                 _kdms_destruct_segment_defin()
   >>>>                 _kdms_hash_segment_defin()
   >>>>                 _kdms_delete_segment_defin()
   >>>>                 _kdms_free_list_segment_defin()
   >>>>  Private:
   >>>>                 kdms_locate_segment_defin()
   >>>>                 kdms_free_segment_defins()
   >>>>   Public:
   >>>>                 kdms_define_segment()
   >>>>                 kdms_undefine_segment()
   >>>>                 kdms_query_segment_definition()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

extern int _kdms_compiled;

/* 
 *    ================================================================== 
 *    Segment Definition Table
 *    ==================================================================
 */
#define TABLESIZE 1024

static int hash_next = 1;      /* -- hash_next must be initially > 0 --*/

static klist         **hash_table = NULL;

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_find_defin - find a defin for a given segment
|
|       Purpose: This routine locates the data services segment
|	         definition structure for a specified segment.
|
|         Input: segment  - the name of the segment to find
|
|        Output: indx   - the hash index into the hash_table
|
|       Returns: the definition or NULL if it doesn't exist
|
|    Written By: Mark Young and Steve Kubica
|          Date: Dec 15, 1994
| Modifications:
|
------------------------------------------------------------*/
static kdms_segment_defin *_kdms_find_defin(
   char *segment,
   int  *indx)
{
   int       temp;
   klist    *list;
   kdms_segment_defin *defin;
   

   /* -- sanity check to make sure that string is not NULL -- */
   if (!segment) return NULL;

   /* -- check to make sure that hash table has been initialized -- */
   if (!hash_table)
      hash_table = (klist **) kcalloc(TABLESIZE, sizeof(klist *));

   /* -- compute the index into the hash table -- */
   temp = khash(segment, -1);
   temp = kabs(temp) % TABLESIZE;
   if (indx) *indx = temp;
   
   for (list = hash_table[temp]; list != NULL; list = klist_next(list))
   {
      defin = (kdms_segment_defin *) list->client_data;
      
      /* -- return if definition matches 
	 --   - this implies that the segment name is identical, and
	 --     and the segment definition is not locked  
       */
      if (defin->lock == FALSE &&
	  strcmp(segment, defin->segment) == 0)
	     return defin;
   }
   return NULL;
}

/*
 *    ================================================================== 
 *    Segment Definition Methods
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kdms_construct_segment_defin
|
|       Purpose: This routine will construct a segment definition
|                according to the given arguments.
|
|		 The segment definition is distinguished by its
|		 segment identifier, which is simply a tokenized
|		 version of the segment name.
|
|		 This segment definition is used to provide
|		 alternate functions for creating, destroying,
|	 	 querying, and referencing, segments.  Alternate
|		 functions can also be provided for getting and
|		 putting data, as well as getting, setting, matching,
|		 copying, querying, and printing that segment's
|		 attributes. 
|
|		 While in one of these alternate functions, the
|		 definition is "locked" so that further kdms calls 
|		 which need to access the segment will _not_ go 
|		 through your handler functions.  Once out of your
|		 handler routine, the definition will be "unlocked".
|
|	 	 This gives you the ability to imbed any needed 
|		 initialization steps into the creation of a given segment.  
|		 For example, the create handler for the "value" segment 
|		 will be called on a kdms_create_segment(obj, "value") call.  
|		 Within this handler, we can again call 
|		 kdms_create_segment("value"), which, because the 
|		 definition is locked, will not call our handler function, 
|		 but instead will create the "value" segment, just as it 
|		 would have done has we not defined it. Inside the create 
|		 handler, after the kdms_create_segment call, we can 
|		 then set the proper dimensionality and index order.
|
|		 If a handler function for a particular action is 
|		 not provided, then data services will simply fall back 
|		 on the action's "default" behavior.
|
|		 The data service functions and their corresponding
|		 handler functions which you can provide are :
|
|		 !   create    - kdms_create_segment
|		 !   destroy   - kdms_destroy_segment
|		 !   query     - kdms_query_segment
|		 !   get_data  - kdms_get_data
|		 !   put_data  - kdms_put_data
|		 !   get       - kdms_set_attribute(s)
|		 !   set       - kdms_set_attribute(s)
|		 !   match     - kdms_match_attribute(s)
|		 !   copy      - kdms_copy_attribute(s)
|		 !   query     - kdms_query_attribute
|		 !   print     - kdms_print_attribute
|
|         Input: segment     - segment identifier 
|		 create()    - create function
|		 destroy()   - destroy function
|		 query()     - query function
|		 get_data()  - get data function
|		 put_data()  - put data function
|		 get_atr()   - attribute get function
|  		 set_atr()   - attribute set function
|		 match_atr() - attribute match function
| 		 copy_atr()  - attribute copy function
|		 query_atr() - attribute query function
|		 print_atr() - attribute print function
|
|        Output: none
|
|       Returns: a pointer to the constructed segment definition 
|		 on success, NULL otherwise
|
|    Written By: Steve Kubica
|          Date: Dec 22, 1994 10:47
| Modifications:
|
------------------------------------------------------------*/
static kdms_segment_defin *
_kdms_construct_segment_defin(
   char   *segment,
   int 	 (*create)    PROTO((kobject, char *)),
   int 	 (*destroy)   PROTO((kobject, char *)),
   int 	 (*query)     PROTO((kobject, char *)),
   kaddr (*get_data)  PROTO((kobject, char *, int *, int *, kaddr)),
   int   (*put_data)  PROTO((kobject, char *, int *, int *, kaddr)),
   int   (*get_atr)   PROTO((kobject, char *, char *, kva_list *)),
   int   (*set_atr)   PROTO((kobject, char *, char *, kva_list *)),
   int   (*match_atr) PROTO((kobject, kobject, char *, char *)),
   int   (*copy_atr)  PROTO((kobject, kobject, char *, char *)),
   int   (*query_atr) PROTO((kobject, char *, char *,int *,int *,int *,int *)),
   int   (*print_atr) PROTO((kobject, char *, char *, kfile *)) )
{
   kdms_segment_defin *defin;

   /* -- allocate a new segment definition structure -- */
   if ((defin = (kdms_segment_defin *)
	kcalloc(1, sizeof(kdms_segment_defin))) == NULL)
   {
      _kdms_set_error(KMEMORY_ALLOCATION);
      return NULL;
   }

   /* -- fill out the segment definition structure -- */
   defin->segment   = kstrdup(segment);
   defin->lock      = FALSE;

   defin->create    = create;
   defin->destroy   = destroy;
   defin->query     = query;
   defin->get_data  = get_data;
   defin->put_data  = put_data;
   defin->get_atr   = get_atr;
   defin->set_atr   = set_atr;
   defin->match_atr = match_atr;
   defin->copy_atr  = copy_atr;
   defin->query_atr = query_atr;
   defin->print_atr = print_atr;

   return defin;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kdms_destruct_segment_defin
|
|       Purpose: Destroy a segment definition.  
|
|         Input: defin - the segment definition to free
|
|        Output: none
|
|       Returns: none
|
|    Written By: Steve Kubica
|          Date: Dec 22, 1994 10:47
| Modifications:
|
------------------------------------------------------------*/
static void
_kdms_destruct_segment_defin(kdms_segment_defin *defin)
{
   /* -- trivial free -- */
   if (defin == NULL)
      return;

   /* -- free the segment name -- */
   kfree(defin->segment);

   /* -- free the segment definition -- */
   kfree(defin);

   return;
}

/*
 *    ================================================================== 
 *    Segment Definition Hash Table Methods
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kdms_hash_segment_defin
|
|       Purpose: Hash a segment defintion to the static
|		 segment definition hash table.
|
|         Input: define - the segment definition to add
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) on failure.
|
|    Written By: Steve Kubica
|          Date: Dec 22, 1994
| Modifications:
|
------------------------------------------------------------*/
static int
_kdms_hash_segment_defin(
   kdms_segment_defin *defin)
{
   int indx;
   int hash;


   /* -- if there is no definition to append, then do nothing -- */
   if (defin == NULL)
   {
      kinfo(KDEBUG, "_kdms_hash_segment_defin : "
	            "can not hash NULL segment definition");
      return FALSE;
   }

   /* -- first try to see if the definition already exists... --  */
   if (defin->segment == NULL)
      return FALSE;
   else if (_kdms_find_defin(defin->segment, &indx) != NULL)
      return FALSE;

   /* -- make a new entry into the hash table -- */
   hash = hash_next++;
   
   /* -- put the entry into the hash table -- */
   hash_table[indx] = klist_add(hash_table[indx], (kaddr) hash,
				  (kaddr) defin);
   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kdms_delete_segment_defin
|
|       Purpose: Given a segment identifier, locate the 
|                segment definition in the segment definition
|		 list and delete it.
|
|         Input: segment - identifier of the segment to delete
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Dec 22, 1994 16:21
| Modifications:
|
------------------------------------------------------------*/
static int
_kdms_delete_segment_defin(
   char *segment)
{
   kdms_segment_defin *defin;
   klist *list;
   int    indx;
   int    id = -1;
   
   
   /* -- make sure we have a valid segment name -- */
   if (segment == NULL)
      return FALSE;

   /* -- get the index into the hash table -- */
   else if ((defin = _kdms_find_defin(segment, &indx)) == NULL)
      return TRUE;

   for (list = hash_table[indx]; list != NULL; list = klist_next(list))
   {
      /* -- find the definition id on the list -- */
      if (defin == list->client_data)
	 id = (int) list->identifier;
   }

   /*
    * If the id was not found, then simply return...
    */
   if (id == -1)
     return (FALSE);
   
   /* -- delete the hash table entry ...  */
   hash_table[indx] = klist_delete(hash_table[indx], (kaddr) id);
   
   /* -- ... and free the definition structure */
   _kdms_destruct_segment_defin(defin);

   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: kdms_locate_segment_defin
|
|       Purpose: Given a segment identifier, locate the 
|                segment in the segment definition list 
|		 and return it.  It will not return a 
|		 segment if it has been locked.
|
|         Input: segment - identifier of the segment to search for
|
|        Output: none
|
|       Returns: the segment definition if it is on the list, 
|	  	 NULL otherwise
|
|    Written By: Steve Kubica
|          Date: Dec 22, 1994 16:21
| Modifications:
|
------------------------------------------------------------*/
kdms_segment_defin *
kdms_locate_segment_defin(
   char *segment)
{
   kdms_segment_defin *defin = NULL;
   int indx;

   if (!segment)
      return NULL;
   else if ((defin = _kdms_find_defin(segment, &indx)) != NULL)
      return defin;
   else 
      return NULL;
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_free_list_segment_defin
|
|       Purpose: Free up a list node from a klist structure
|		 under the assumption that the client_data of
|		 the list node is an segment defintion.
|
|         Input: list
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Steve Kubica
|          Date: Dec 19, 1994 16:21
| Modifications:
|
------------------------------------------------------------*/
static void
_kdms_free_list_segment_defin(klist *list)
{
   if (list == NULL)
      return;
   
   _kdms_destruct_segment_defin((kdms_segment_defin *)list->client_data);

   return;
}

/*-----------------------------------------------------------
|
|  Routine Name: kdms_free_segment_defins
|
|       Purpose: Free up the static segment definition list.
|		 All the definitions are freed and then the list
|		 itself is freed.
|
|         Input: none
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Steve Kubica
|          Date: Mar 21, 1994 16:21
| Modifications:
|
------------------------------------------------------------*/
void
kdms_free_segment_defins(void)
{
   int i;

   for (i = 0; hash_table && i < TABLESIZE; i++)
      klist_free(hash_table[i], _kdms_free_list_segment_defin);

   kfree(hash_table);

   return;
}

/*
 *    ================================================================== 
 *    Public Routines for Segment Definitions
 *    ==================================================================
 */

/************************************************************
*
*  Routine Name: kdms_define_segment - define a segment
*
*       Purpose: This routine is used to define alternate segment
* 		 handler functions for a session.  A segment
* 		 definition is used to provide alternate functions for
* 		 creating, destroying, querying, and referencing,
* 		 segments.  Alternate functions can also be provided
* 		 for getting and putting data, as well as getting,
* 		 setting, matching, copying, querying, and printing
* 		 that segment's attributes.  Segments do not need
*		 to be defined.  This function just provides a
*		 convenient method for encapsulating alternate
*		 functionality for a segment name beneath the
*		 existing data services api.
*
*		 While in one of these alternate functions, the
*		 definition is "locked" so that further kdms calls 
*		 which access the defined segment will _not_ go 
*		 through the defined segment handler functions.  
*		 Once out of the handler function, the definition 
*		 will be "unlocked".
*
*	 	 This provides the ability to imbed any needed 
*		 initialization steps into the creation of a given
*		 segment.  For example, the create handler for the
*		 "value" segment will be called on a
*		 kdms_create_segment(obj, "value") call.  Within this
*		 handler, another kdms_create_segment(obj, "value")
*		 call will not call the create handler becuse the
*		 definition is locked, but instead will create the
*		 "value" segment, just as it would have done had the
*		 "value" segment not been defined.  Additional
*		 initialization steps, for example, setting the
*		 dimensionality of the "value" segment, can be easily
*		 added to the handler after the internal create call.
*
*		 If a handler function for a particular action is 
*		 not provided, then data services will behave as if
*		 the segment has never been defined.
*		 
*		 This segment definition can be undefined with the 
*		 kdms_undefine_segment call.
*
*         Input: segment    - string name of the segment being defined.
*
*   		 create         - create handler function for this segment.
*				  This function will be invoked whenever a
*				  kdms_create_segment function is called
*				  with the defined segment.
*
*       !The create handler declaration is of the form :
*       !
*       !   int create_handler(
*       !       kobject   object,
*       !       char     *segment)
*
*				  The object and segment are passed from
*				  the calling create function.  The return
*			    	  value of the handler will be propogated
*				  back up to the return value of the calling
*				  create function.
*
*   		 destroy        - destroy handler function for this segment.
*				  This function will be invoked whenever a
*				  kdms_destroy_segment function is called
*				  with the defined segment.
*
*       !The destroy handler declaration is of the form :
*       !
*       !   int destroy_handler(
*       !       kobject   object,
*       !       char     *segment)
*
*				  The object and segment are passed from
*				  the calling destroy function.  The return
*			    	  value of the handler will be propogated
*				  back up to the return value of the calling
*				  destroy function.
*
*   		 query          - query handler function for this segment.
*				  This function will be invoked whenever a
*				  kdms_query_segment function is called
*				  with the defined segment.
*
*       !The query handler declaration is of the form :
*       !
*       !   int query_handler(
*       !       kobject   object,
*       !       char     *segment)
*
*				  The object and segment are passed from
*				  the calling query function.  The return
*			    	  value of the handler will be propogated
*				  back up to the return value of the calling
*				  query function.
*
*   		 get_data       - get_data handler function for this segment.
*				  This function will be invoked whenever a
*				  kdms_get_data function is called
*				  with the defined segment.
*
*       !The get_data declaration is of the form :
*       !
*       !   kaddr get_data_handler(
*       !       kobject   object,
*       !       char     *segment,
*	!	int      *begin,
*	!	int      *end
*	!       int       kaddr)
*
*				  The object, segment, begin corner, end
*				  corner, and data pointer are passed from
*				  the calling get_data function.  The return
*			    	  data pointer from the handler will be 
*				  propogated back up to the return value 
*				  of the calling get_data function.
*
*   		 put_data       - put_data handler function for this segment.
*				  This function will be invoked whenever a
*				  kdms_put_data function is called
*				  with the defined segment.
*
*       !The put_data declaration is of the form :
*       !
*       !   int put_data_handler(
*       !       kobject   object,
*       !       char     *segment,
*	!	int      *begin,
*	!	int      *end
*	!       int       kaddr)
*
*				  The object, segment, begin corner, end
*				  corner, and data pointer are passed from
*				  the calling put_data function.  The return
*			    	  value from the handler will be propogated 
*			 	  back up to the return value of the calling
*				  put_data function.
*
*   		 get_atr        - get attribute handler function for this 
*				  segment. This function will be
*				  invoked whenever a kdms_get_attribute,
*				  kdms_get_attributes, kdms_vget_attribute,
*				  or kdms_vget_attributes function is
*				  called with the defined segment.
*
*       !The get attribute handler declaration is of the form :
*       !
*       !   int get_attribute_handler(
*       !       kobject   object,
*       !       char     *segment,
*       !       char     *attribute,
*       !       kva_list *list)
*
*                                 The object, segment, and attribute are 
*				  passed from the calling get function. The
*                                 variable argument list is an already
*                                 opened varargs list which contains the
*                                 pointers with which to return the 
*                                 attribute data.  The variable arguments
*                                 can be pulled off with the kva_arg()
*                                 function.  The list will be closed after
*                                 this handler returns.  The return value
*				  of this handler will also be propogated
*			          up to be the return value of the calling
*				  get function.
*
*		 set_atr        - set attribute handler function for this
*			          segment. This function will be
*			          invoked whenever a kdms_set_attribute,
*			          kdms_set_attributes, kdms_vset_attribute, 
*			          or kdms_vset_attributes function is
*			          called with the defined segment.
*
*	!The set attribute handler declaration is of the form :
*       !
*	!   int set_attribute_handler(
*	!       kobject   object,
*       !       char     *segment,
*       !       char     *attribute,
*       !       kva_list *list)
*
*                                 The object, segment, and attribute are 
*				  passed from the calling set function. The
*                                 variable argument list is an already
*                                 opened varargs list which contains the
*                                 data with which to set the attribute
*                                 data.  The variable arguments
*                                 can be pulled off with the kva_arg()
*                                 function.  The list will be closed after
*                                 this handler returns.  The return value
*                                 of this handler will also be propogated
*                                 up to be the return value of the calling
*                                 set function.
*
*		 match_atr      - match attribute handler function for this
*			          segment. This function will be invoked 
*				  whenever a kdms_match_attribute, 
*				  kdms_match_attributes, or 
*			          kdms_vmatch_attributes
*				  function is called with the defined segment.
*
*	!The match attribute handler declaration is of the form :
*       !
*	!   int match_attribute_handler(
*	!       kobject   object1,
*	!       kobject   object2,
*       !       char     *segment,
*       !       char     *attribute)
*
*                                 The object1, object2, segment, and
*				  attribute arguments are passed
*				  from the calling match function. 
*				  The return value of this
*			          handler will be propogated up to be
*			          the return value of the calling
*			          match function. 
*
*		 copy_atr       - copy attribute handler function for this
*			          segment. This function will be invoked 
*				  whenever a kdms_copy_attribute, 
*				  kdms_copy_attributes, or 
*			          kdms_vcopy_attributes
*				  function is called with the defined segment.
*
*	!The copy attribute handler declaration is of the form :
*       !
*	!   int copy_attribute_handler(
*	!       kobject   object1,
*	!       kobject   object2,
*       !       char     *segment,
*       !       char     *attribute)
*
*                                 The object1, object2, segment, and
*				  attribute arguments are passed
*				  from the calling copy function. 
*                                 copy function. The return value of this
*			          handler will be propogated up to be
*			          the return value of the calling
*			          copy function.  
*
*		 query_atr      - query attribute handler function for this
*			          segment.  This function will be invoked 
*				  whenever a kdms_query_attribute function is 
*				  called with the defined segment.
*
*	!The query attribute handler declaration is of the form :
*       !
*	!   int query_attribute_handler(
*	!       kobject   object,
*       !       char     *segment,
*       !       char     *attribute,
*       !       int      *num_args,
*       !       int      *arg_size,
*       !       int      *data_type,
*       !       int      *permanent)
*
*                                 The object, segment, and attribute are 
*			          passed in from the calling query
*			          function.  The association is the
*			          tokenized representation of the
*			          segment argument from the query
*			          function.  The num_args argument can
*			          be used to return the expected
*			          number of arguments for this
*			          attribute.  The arg_size argument
*			          can be used to return the expected
*			          argument size for this attribute.
*			          The data_type argument can be used
*			          to return the expected data type for
*			          this attribute.  The permanent
*			          argument can be used to indicate if
*			          this attribute is stored or not.
*			          Note that it is up to the programmer
*			          to ensure that the values which the
*			          handler returns for any of those
*			          arguments matches what is going to
*			          be processed in the get and set
*			          handlers.  The return value of this
*			          handler will be propogated up to be
*			          the return value of the calling
*			          query function.  In general, a value
*			          of TRUE is interpreted to mean
*			          that the attribute 'exists'.
*
*		 print_atr      - print attribute handler function for this 
*			          segment. This function will be invoked 
*				  whenever a kdms_print_attribute function is 
*				  called with the defined segment.
*
*	!The print attribute handler declaration is of the form :
*       !
*	!   int print_attribute_handler(
*	!       kobject   object,
*       !       char     *segment,
*       !       char     *attribute,
*       !       kfile    *outfile)
*
*                                 The object, segment, attribute, and
*                                 outfile is passed in from the
*                                 calling print function.  The return
*                                 value of this handler will be
*                                 propogated up to be the return value
*                                 of the calling print function.
*        Output: none
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Steve Kubica
*          Date: Dec 22, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kdms_define_segment(
*                !      char        *segment,
*   		 !      kfunc_int   *create,
*   		 !      kfunc_int   *destroy,
*   		 !      kfunc_int   *query,
*   		 !      kfunc_kaddr *get_data,
*   		 !      kfunc_int   *put_data,
*   		 !      kfunc_int   *get_atr,
*   		 !      kfunc_int   *set_atr,
*   		 !      kfunc_int   *match_atr_atrnn,
*   		 !      kfunc_int   *copy,_atr
*   		 !      kfunc_int   *query,
*		 !      kfunc_int   *print)
*
*************************************************************/
int
kdms_define_segment(
   char  *segment,
   int 	 (*create)    PROTO((kobject, char *)),
   int 	 (*destroy)   PROTO((kobject, char *)),
   int 	 (*query)     PROTO((kobject, char *)),
   kaddr (*get_data)  PROTO((kobject, char *, int *, int *, kaddr)),
   int   (*put_data)  PROTO((kobject, char *, int *, int *, kaddr)),
   int   (*get_atr)   PROTO((kobject, char *, char *, kva_list *)),
   int   (*set_atr)   PROTO((kobject, char *, char *, kva_list *)),
   int   (*match_atr) PROTO((kobject, kobject, char *, char *)),
   int   (*copy_atr)  PROTO((kobject, kobject, char *, char *)),
   int   (*query_atr) PROTO((kobject, char *, char *,int *,int *,int *,int *)),
   int   (*print_atr) PROTO((kobject, char *, char *, kfile *)) )
{
   kdms_segment_defin *defin = NULL;
   kdms_segment_defin *old_defin = NULL;

   if (!_kdms_compiled)
      _kdms_init();

   old_defin = kdms_locate_segment_defin(segment);
   if (old_defin != NULL)
   {
      kinfo(KDEBUG, "kdms_define_segment: attempt to redefine the " 
	            "segment '%s'.", segment);
      return FALSE;
   }

   defin = _kdms_construct_segment_defin(segment, create, destroy, query,
					 get_data, put_data, 
					 get_atr, set_atr, match_atr, 
					 copy_atr, query_atr, print_atr);
   
   return _kdms_hash_segment_defin(defin);
}

/************************************************************
*
*  Routine Name: kdms_undefine_segment - undefine a defined segment
*
*       Purpose: This function will remove a segment definition 
*		 from the segment definition list.  
*
*         Input: segment - the name of the segment to undefine
*
*        Output: none
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Steve Kubica
*          Date: Dec 22, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kdms_undefine_segment(
*                !      char *segment)
*
*************************************************************/
int
kdms_undefine_segment(
   char *segment)
{
   if (!_kdms_compiled)
      _kdms_init();

   return _kdms_delete_segment_defin(segment);
}

/************************************************************
*
*  Routine Name: kdms_query_segment_definition - determines if a segment
*					         is defined.
*
*       Purpose: This function will check to see if a segment is
*		 defined on the segment definition list.
*
*         Input: segment - the name of the segment to query for
*
*        Output: none
*
*       Returns: TRUE (1) if segment is defined, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Steve Kubica
*          Date: Mar 25, 1994 13:37
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kdms_is_defined_attribute(
*                !      char *segment)
*
*************************************************************/
int
kdms_query_segment_definition(char *segment)
{
   if (!_kdms_compiled)
      _kdms_init();

   if (kdms_locate_segment_defin(segment) == NULL)
      return FALSE;
   else
      return TRUE;
}

