 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            
   >>>>
   >>>>  Private:
   >>>>             
   >>>>   Static:
   >>>>             
   >>>>   Public:
   >>>>             	kdms_create()
   >>>>             	kdms_open()
   >>>>             	kdms_close()
   >>>>             	kdms_reference()
   >>>>             	kdms_sync()
   >>>>             	kdms_locate()
   >>>>             	kdms_query_segment()
   >>>>             	kdms_create_segment()
   >>>>             	kdms_destroy_segment()
   >>>>             	kdms_support()
   >>>>             	kdms_update_references()
   >>>>             	kdms_reference_list()
   >>>>             	kdms_get_segment_names()
   >>>>             	kdms_get_segment_list()
   >>>>             	kdms_close_hook()
   >>>>             	kdms_reopen()
   >>>>             	kdms_rename_segment()
   >>>>             	kdms_add_file_format()
   >>>>             
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

int _kdms_compiled = FALSE;

extern int kdms_num_data_formats;
extern DataServiceInformation **services;

/************************************************************
*
*  Routine Name: kdms_create - create a temporary data object.
*
*       Purpose: kdms_create is used to instantiate a data object
*                (kobject) when it will only be used for temporary
*                storage of information.  If you are intending to
*                process an object that already exists as a file or
*                transport (input), or you are planning on saving the
*                kobject to a file or transport (output), then the
*                appropriate routines to use are kdms_input,
*                kdms_output, or kdms_open.
*
*                This function creates an instance of a data object
*                that will have associated with it a temporary
*                transport that will be used for buffering large
*                amounts of data.  This temporary transport will be
*                automatically removed when the process terminates.
*                There is no way to rename the temporary file or
*                replace the temporary file with a permanent one.
*
*                The kdms_create function call creates what is
*                essentially a "blank" object.  That is, the object
*                will initially have no segments, and almost all
*                global attributes will be initialized to default
*                values.  If a default is not appropriate, then the
*                attribute will be uninitialized.  The default values
*                for attributes are described in Chapter 5 of Program
*		 Services Volume 2. of the Khoros 2.0 Manual.
*
*                An object that is created with this function call
*                behaves similarly to an output object that is created
*                with the kdms_output function call.  Thus, it is
*                necessary to create each segment that is desired and
*                initialize attributes such as size and datatype prior
*                to using the object.
*
*         Input: none
*
*        Output: none
*
*       Returns: kobject on success, KOBJECT_INVALID upon failure
*
*  Restrictions: 
*    Written By: Mark Young and Jeremy Worley
*          Date: Jul 13, 1992 15:55
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

kobject
kdms_create(void)
{
   if (!_kdms_compiled)
      _kdms_init();

   return (_kdms_create());
}

/************************************************************
*
*  Routine Name: kdms_open - create an object associated with an input
*                or output transport.
*
*       Purpose: kdms_open is used to instantiate a data object
*                (kobject) that is assocated with a permanent file or
*                transport.  If a permanent file is not desired
*                (i.e. the object is going to be used as temporary
*                storage, and will not be used by any other process)
*                then the kdms_create function call should be used
*                instead.
*
*                The first argument to this function is the transport
*                or file name.  This argument indicates the name of
*                the transport that is associated with the object.
*                The transport name can be any legal khoros transport
*                description.  While this will usually be a regular
*                UNIX file name, it is also possible to specify such
*                things as shared memory pointers, memory map
*                pointers, sockets, streams, and even transports on
*                other machines.  For more information on the syntax
*                of a Khoros transport name, refer to the online man
*                page for the Khoros function kopen.
*
*                The second argument to the kdms_open function call is
*                used to provide data services with specific
*                information about how the object is going to be
*                manipulated.  The flags argument is analogous to
*                kopen's flags argument.  The flags argument is
*                constructed by bitwise OR'ing predefined values from
*                the following list:
*
*		.RS
*       	.IP "KOBJ_READ" 10
*				Open an existing file or transport
*                               for reading (input).  By using this flag,
*                               you are indicating that the file or transport
*                               exists and that it contains valid data.
*                               If it does not exist, or the data is not
*                               recognized, then an error message will
*                               be generated and this function will
*                               return KOBJECT_INVALID.
*        
*        	.IP "KOBJ_WRITE" 10
*				Open a file or transport for writing
*                               (output).  By using this flag, you are
*                               indicating that any data that you write
*                               to the object will be stored in the
*                               file or transport specified.
*        
*        	.IP "KOBJ_STREAM" 10
*				If the transport is a stream then polymorphic
*	                        services does not attempt to buffer the
*	                        data.  By setting this value, you are
*	                        indicating that that process reads and/or
*	                        writes the data strictly sequentially from the
*                               object in its native index order, and
*                               it is not necessary to buffer the data
*                               in a temporary transport that can be
*                               accessed randomly.
*        
*        	.IP "KOBJ_RAW" 10
*				When an object is opened,  data
*	 			services attempts to recognize the file
*	 			format by examining the first part of the
*	 			file (typically there are "magic number"
*	 			or identifiers that make this sort of
*	 			recognition trivial).  If it cannot
*	 			determine the file format, then the open
*	 			operation will fail, unless KOBJ_RAW
*	 			is set.  In other words, if KOBJ_RAW 
*	 			is set, then data services will assume that 
*	 			if it cannot recognize the format, then the
*	 			file contains raw unformatted data.
*		.RE
*
*                These flags can be combined arbitrarily (with the exceptions
*                given above) to alter the interpretation of the file 
*                or transport.  For example, specifying both KOBJ_READ
*                and KOBJ_WRITE will result in a read/write file object.
*                This implies that the file already exists and will be
*                read from using kdms_get_data and written to using
*                kdms_put_data.  When kdms_close is called, the changes
*                that are a result of calls to kdms_put_data will be
*                stored to the file.
*
*                However, if you intend to open an output object, but
*                you need to occationally read data from it that you
*                have already written, it is not necessary to specify 
*                KOBJ_READ (in fact, doing so may result result in an
*                error if the file or transport does not already exist).
*
*                Likewise, it is possible to call kdms_put_data on
*                an input object (one which was opened without the 
*                KOBJ_WRITE flag).  If this is done, then subsequent 
*                calls to kdms_get_data on a region that has been
*                written to will contain the new data.  However,
*                the file that is associated with this input object
*                will not be changed.  Thus, the KOBJ_READ and KOBJ_WRITE
*                flags only indicate what operations are allowed on the
*                permanent file that is associated with the object, not
*                what operations are allowable on the object itself.
*
*                If KOBJ_READ is specified, then the Data Services
*                will attempt to recognize the file format automatically.
*                If it fails, then this function will return KOBJECT_INVALID,
*                indicating that it was unable to open the object,
*                unless the KOBJ_RAW flag was also specified,
*                in which case, it will assume that the input file
*                is simply headerless data.  The structured file formats
*                that are currently recognized are Viff (The Khoros
*                2.0 standard file format), Xvimage (The Khoros 1.0
*                standard file format, which was referred to as Viff
*                in Khoros 1.0), Pnm (Portable Any Map, which includes
*                PBM, PGM, and PNM), and Sun Raster.
*                
*         Input: name - a string that contains the path name
*                       of a file or transport that will be 
*                       associated with the object.
*
*                flags - how the object is to be opened.
*                        a combination of KOBJ_READ, KOBJ_WRITE,
*                        KOBJ_STREAM and KOBJ_RAW as 
*                        described above.
*
*        Output: none
*       Returns: kobject on success, KOBJECT_INVALID upon failure
*
*  Restrictions: The KOBJ_RAW flag will have unpredictable 
*                results if it is combined with the KOBJ_WRITE flag.
*                This limitation will be removed in a later release of 
*                the Khoros 2.0 system.
*
*                The Xvimage file format has a bug in it that prevents
*                it from being used on a stream base output.  This
*                will be fixed in a future release of the Khoros 2.0
*                system.
*    Written By: Jeremy Worley
*          Date: Jul 13, 1992 15:55
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

kobject
kdms_open(
            char *name,
            int flags)
{
   kobject tmp;

   if (!_kdms_compiled)
      _kdms_init();

   tmp = _kdms_open(name, flags);

   return (tmp);
}

/************************************************************
*
*  Routine Name: kdms_close - close an open data object.
*
*       Purpose: 
*                This function is called on an object when all
*                interaction with the object is completed.
*                In addition to freeing resources that were
*                used to manage the object, this function also,
*                performs any "last minute" manipulation on the
*                file or transport that is associated with the
*                object.
*
*                If the object was created with the kdms_reference
*                function call, or if another object was created as
*                a reference of the one being closed, then the object 
*                might be sharing some of its resources with other 
*                objects.  If this is the case, then those shared 
*                resources will not be freed, but rather they will 
*                be disassociated from the object being closed.  Thus, 
*                closing an object does not affect any other object.
*
*         Input: object - the object to be closed.
*
*        Output: none
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young and Jeremy Worley
*          Date: Jul 13, 1992 15:59
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int
kdms_close(kobject object)
{
   if (!_kdms_compiled)
      _kdms_init();

   return ((object == NULL || object->type != KOBJ_DATA) ? FALSE :
           _kdms_close(object, KEXIT_SUCCESS));
}


/************************************************************
*
*  Routine Name: kdms_reference - create a reference of a
*                data object.
*
*       Purpose: 
*                This function is used to create a reference
*                of a data object that can be treated as a second
*                independant data object under most circumstances.
*                A referenced object is similar conceptually to a 
*                symbolic link in a UNIX file system in most 
*                respects.  For example, getting data from an
*                input object and a reference of the object will
*                result in the same data.  Data that is put on an
*                output object can then be gotten from one of its 
*                references.  
*
*                The similarity ends there.  Once an object is
*                referenced, the two resulting objects are 
*                equivelant--there is no way to distinguish the
*                original from the reference.  In fact, closing
*                the original does not in any way affect the 
*                reference, and visa-versa.
*
*                kdms_reference creates a new object that has
*                presentation attributes that are independant
*                of the original object's presentation attributes.
*                The presentation attributes are UNCOUPLED from
*                the physical attributes, see the description
*                found in Chapter 6 of the the Khoros Programmer's
*                Manual on the KDMS_COUPLING attribute for more
*                information.  The two objects (or more if there are
*                future calls to kdms_reference) share all physical
*                resources.
*
*         Input: object - the abstract data object to be
*                         reference.
*
*        Output: none
*       Returns: a kobject that is a reference of the input
*                object on success, KOBJECT_INVALID upon failure
*
*  Restrictions: 
*
*    Written By: Mark Young and Jeremy Worley
*          Date: Aug 20, 1992 15:51
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

kobject
kdms_reference(kobject object)
{
   if (!_kdms_compiled)
      _kdms_init();

   return ((object == NULL || object->type != KOBJ_DATA) ? KOBJECT_INVALID :
           _kdms_reference(object));
}


/************************************************************
*
*  Routine Name: kdms_sync - synchronize physical and presentation
*                layers of a data object.
*
*       Purpose: 
*                This function is used to update physical
*                attributes of a data object to match those
*                of the presentation layer, or visa-versa.
*                When an attribute is set via kdms_set_attribute(s)
*                or kdms_copy_attribute(s) calls, the presentation
*                version of the attribute is the only thing
*                that is directly manipulated.  The KDMS_COUPLING
*                attribute is used at that time to determine if
*                the physical attribute should be updated to
*                correspond to its value at the presentation
*                level.  The KDMS_COUPLING attribute can take
*                on one of three values: KUNCOUPLED, KCOUPLED, or
*                KDEMAND.  If it is set to KUNCOUPLED or KDEMAND,
*                then Data Services will not update the physical
*                layer.  If the attribute is set to KCOUPLED, then
*                data services immediately updates the physical
*                layer.  If the attribute is set to KDEMAND,
*                then this updating will only occur when kdms_sync
*                is called.  If the KDMS_COUPLING attribute is
*                set to KUNCOUPLED, then this routine will simply
*                return, without issuing an error message.
*
*         Input: object  - data object to be resynchronized.
*                segment - segment re-synchronize.  If this is
*                          set to KDMS_OBJECT, then all segments
*                          are re-synchronized.
*                direction - the desired direction of the synchronization.
*                            the legal values are KPRES2PHYS, which
*                            indicates that the physical layer will be
*                            updated to correspond to the presentation
*                            layer; and KPHYS2PRES, which indicates
*                            that the presentation layer will be updated
*                            to correspond to the physical layer.
*
*        Output:
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley
*          Date: Sep 01, 1993 10:41
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int
kdms_sync(
            kobject object,
            char *segment,
            int direction)
{
   if (!_kdms_compiled)
      _kdms_init();

   if (object != NULL)
      return (_kdms_sync(object, segment, direction));

   _kdms_set_error(KDMS_EOBJ_INVALID);
   return (FALSE);
}

/*-----------------------------------------------------------
|
|  Routine Name: kdms_locate
|
|       Purpose:
|
|         Input:
|
|        Output:
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Jun 11, 1993 15:55
| Modifications:
|
------------------------------------------------------------*/

kobject
kdms_locate(char *filename)
{
   if (!_kdms_compiled)
      _kdms_init();

   return (_kdms_locate(filename));
}

/************************************************************
*
*  Routine Name: kdms_query_segment - determine if a data segment is available.
*
*       Purpose: 
*                This function is used to determine if a data segment
*                in a data object is available.  It returns TRUE
*                if the data segment exists, FALSE otherwise.
*
*         Input: object  - data object that segment may exist in.
*                segment - name of segment to determine existence of.
*
*        Output: 
*
*       Returns: TRUE (1) if the data segment exists, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley
*          Date: Aug 23, 1993 12:24
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int
kdms_query_segment(kobject object, char *segment)
{
   kdms_segment_defin *segd = NULL;
   kpresentation *pres = NULL;

   if (!_kdms_compiled)
      _kdms_init();

   /*
    * polymorph.c in appservices takes big advantage of this.
    *   not sure if this is true anymore .... // Steve K
    */
   if (segment == NULL)
      return TRUE;

   /* -- if there is a defined segment query routine, then use that -- */
   if ((segd = kdms_locate_segment_defin(segment)) != NULL)
      if (segd->query)
      {
	 int status;
	 
	 segd->lock = TRUE;
	 status = segd->query(object, segment);
	 segd->lock = FALSE;
	 return status;
      }

   /* -- otherwise, do a normal query -- */   
   if (!_kdms_get_segment_info(object, kstring_to_token(segment), &pres) ||
       !pres)
      return FALSE;

   return TRUE;
}

/************************************************************
*
*  Routine Name: kdms_create_segment - create a segment on a data object.
*
*       Purpose: 
*                This function is used to create a segment that does
*                not already exist in a data object.  If the segment
*                already exists, then this function will generate
*                an error.
*
*         Input: object    - object to create segment in
*                segment   - name of segment to create
*
*        Output: 
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley
*          Date: Aug 23, 1993 10:11
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/
int
kdms_create_segment(
   kobject object,
   char   *segment)
{
   kdms_segment_defin *segd = NULL;
   
   if (!_kdms_compiled)
      _kdms_init();

   if (object == NULL || object->type != KOBJ_DATA)
   {
      _kdms_set_error(KDMS_EOBJ_INVALID);
      return FALSE;
   }
   
   /* -- if there is a defined segment creation routine, then use that -- */
   if ((segd = kdms_locate_segment_defin(segment)) != NULL)
      if (segd->create)
      {
	 int status;
	 
	 segd->lock = TRUE;
	 status = segd->create(object, segment);
	 segd->lock = FALSE;
	 return status;
      }

   /* -- otherwise, do a normal create -- */   
   if (kdms_query_segment(object, segment))
   {
      _kdms_set_error(KDMS_ESEG_EXISTS);
      return FALSE;
   }

   /*
    * Create the segment.
    */
   return _kdms_create_segment(object, kstring_to_token(segment), NULL);
}

/************************************************************
*
*  Routine Name: kdms_destroy_segment - destroy a segment from a data object.
*
*       Purpose: 
*                This function is used to destroy an existing 
*                segment from a data object.  Once a segment
*                has been destroyd, any data or attributes associated
*                with that segment will be lost forever.  A new
*                segment can be created in its place.
*
*                If the segment does not exist, then an error
*                message will be issued.
*
*         Input: object - object containing segment to be destroyd
*                segment - name of segment to be destroyed
*
*        Output: 
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley
*          Date: Aug 23, 1993 10:28
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/
int
kdms_destroy_segment(
   kobject object,
   char   *segment)
{
   kdms_segment_defin *segd = NULL;

   if (!_kdms_compiled)
      _kdms_init();

   if (object == NULL || object->type != KOBJ_DATA)
   {
      _kdms_set_error(KDMS_EOBJ_INVALID);
      return FALSE;
   }

   /* -- if there is a defined segment destroy routine, then use that -- */
   if ((segd = kdms_locate_segment_defin(segment)) != NULL)
      if (segd->destroy)
      {
	 int status;
	 
	 segd->lock = TRUE;
	 status = segd->destroy(object, segment);
	 segd->lock = FALSE;
	 return status;
      }

   /* -- otherwise, do a normal destroy -- */   
   if (!kdms_query_segment(object, segment))
   {
      _kdms_set_error(KDMS_ESEG_NONEXIST);
      return FALSE;
   }

   return _kdms_destroy_segment(object, kstring_to_token(segment));
}

/************************************************************
*
*  Routine Name: kdms_support - obtain a list of file formats
*		 supported by data services.
*
*       Purpose: This function is used to obtain a list of the file
*       	 formats supported by the data abstraction.
*
*         Input: 
*
*        Output: num - the number of formats in the returned
*		 string.
*
*       Returns: An array of strings (char **) containing
*		 a list of all formats that are currently
*		 supported by Data Services.
*
*  Restrictions: 
*    Written By: Jeremy Worley
*          Date: Oct 18, 1993 11:45
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

char **
kdms_support(int *num)
{
   if (!_kdms_compiled)
      _kdms_init();

   return (_kdms_glue_support(num));
}

/************************************************************
*
*  Routine Name: kdms_update_references - update segment presentation
*  of all reference objects.
*
*       Purpose: 
*                This function propagates the values of the
*                presentation attributes from the specified 
*                object and segment to the corresponding segments
*                in all of the object's references.
*
*         Input: object - object containing segment to propagate.
*                segment - name of segment to be propagated.
*
*        Output: 
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley
*          Date: Oct 26, 1993 14:03
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int 
kdms_update_references(
   kobject object,
   char *segment)
{
   if (!_kdms_compiled)
      _kdms_init();

   if (object == NULL || object->type != KOBJ_DATA)
   {
      _kdms_set_error(KDMS_EOBJ_INVALID);
      return (FALSE);
   }

   return (_kdms_update_references(object, segment));
}

/************************************************************
*
*  Routine Name: kdms_reference_list - return a klist of references.
*
*       Purpose: 
*                This function returns a klist of objects that are
*                references of the object passed in as the argument
*                to this function.  The object passed in will also
*                be in this list.  NOTE:  This list is the one
*                actually used by KDMS to manage references.  Destroying
*                this list will cause grief like you've never seen
*                before.
*
*         Input: object - the object to get references of.
*
*        Output: 
*
*       Returns: a klist * on success, NULL on failure.
*
*  Restrictions: 
*    Written By: Jeremy Worley
*          Date: Oct 27, 1993 08:21
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

klist *
kdms_reference_list(kobject object)
{
   if (!_kdms_compiled)
      _kdms_init();

   if (object == NULL || object->type != KOBJ_DATA)
   {
      _kdms_set_error(KDMS_EOBJ_INVALID);
      return (NULL);
   }

   return (klist_head(object->phys->list));
}

/************************************************************
*
*  Routine Name: kdms_get_segment_names - get an array of 
*		 segment names for the object specified.
*
*       Purpose: Given an object, obtain an array of strings
*		 which are the names of all segments which
*		 exist in the object.
*
*         Input: object - the object to get the segment names
*		          from.
*
*        Output: number - the number of segments in the object
*			  (and thus, the number strings in the
*			  array that is returned.
*
*       Returns: an array of strings containing the names of the
*		 segments present in the object on success, NULL
*		 on failure.
*
*  Restrictions: 
*    Written By: Steve Kubica
*          Date: May 02, 1994 11:52
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

char **
kdms_get_segment_names(kobject object, int *number)
{
   klist *seg;
   char *s = NULL;
   char **segments = NULL;
   int nsegments = 0;

   if (!_kdms_compiled)
      _kdms_init();

   for (seg = klist_head(object->presentations);
        seg != NULL; seg = klist_next(seg))
   {
      s = ktoken_to_string(klist_token(seg));
      segments = karray_add(segments, kstrdup(s), nsegments);
      nsegments++;
   }

   if (number)
      *number = nsegments;

   return (segments);
}

/*-----------------------------------------------------------
|
|  Routine Name: kdms_get_segment_list
|       Purpose: I don't really like this function.  Especially,
|		 given the fact that another function (kdms_get_segment_names)
|		 exists which performs a similar function without
|		 jeopardizing the integrity of the object.
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Aug 18, 1994 00:49
| Modifications:
|
------------------------------------------------------------*/

klist *
kdms_get_segment_list(kobject object)
{
   if (!_kdms_compiled)
      _kdms_init();

   return (object->presentations);
}

/************************************************************
*
*  Routine Name: kdms_close_hook - insert a service to be called
*		 when an object is closed.
*
*       Purpose: This function is used to insert a special function that is
*                called immediately before an object is closed (after
*                a call to kdms_close) that can perform any cleanup
*                that may be required before the object is written if
*                it is an output object.
*
*         Input: object - object to have close hook function added
*		          to.
*		 func   - function to set as close hook.  This function 
*			  has the following prototype:  func(kobject).
*
*        Output: 
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley & John Salas
*          Date: Nov 15, 1993 13:59
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

kfunc_int 
kdms_close_hook(kobject object, kfunc_int func)
{
   kfunc_int tmp;

   if (!_kdms_compiled)
      _kdms_init();

   if (object == NULL || object->type != KOBJ_DATA)
      return (NULL);

   tmp = object->phys->close_hook;

   object->phys->close_hook = (kfunc_int) func;

   return (tmp);
}

/************************************************************
*
*  Routine Name: kdms_reopen - associate new data with an existing object
*
*       Purpose: This function is used to associate new stored data
*		 with an already open data object.  This operation can
*		 only be performed on an input object.  This function
*		 is typically used in an interactive environment when
*		 many references to replace the data set being operated
*		 on or visualized with a new data set, without having
*		 to close all of the references and replace them with
*		 new ones.
*
*         Input: object   - data object re-open
*		 filename - transport name of new data set to associate with
*		       	    the object.
*		 flags    - flags to use when opening the data set specified
*			    by the filename argument.
*
*        Output: 
*
*       Returns: object (the input argument) on success, NULL on failure.
*
*  Restrictions: 
*    Written By: Jeremy Worley
*          Date: Nov 15, 1993 13:59
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

kobject
kdms_reopen(
              kobject object,
              char *name,
              int flags)
{
   if (!_kdms_compiled)
      _kdms_init();

   return (_kdms_reopen(object, name, flags));
}

/************************************************************
*
*  Routine Name: kdms_rename_segment - rename a segment
*       Purpose: This function is used to rename a segment in 
*		 an open object.  
*         Input: object - object to rename a segment on.
*		 old_name - the name of the segment to rename.
*		 new_name - the new name of the segment.
*        Output:
*       Returns: TRUE if successful, FALSE otherwise.
*  Restrictions:
*    Written By: Jeremy Worley
*          Date: Aug 14, 1994 18:08
*      Verified:
*  Side Effects:
* Modifications:
*************************************************************/

int
kdms_rename_segment(
   kobject object,
   char *old_name,
   char *new_name)
{
   if (!_kdms_compiled)
      _kdms_init();

   return(_kdms_rename_segment(object, old_name, new_name));
}

/************************************************************
*
*  Routine Name: kdms_add_file_format
*       Purpose: This function is used to add support for a new
*		 file format to data services.  
*         Input: name        - a short name for the file format.  This
*		               is the name that will be used when setting 
*		 	       the KDMS_FORMAT attribute.
*		 description - a description of the file format.
*		 check       - a function that takes an open
*			       kfile pointer and examines (hopefully)
*			       the first few bytes of the file to
*			       determine if it is one of this type.
*		 input       - the function called when check returns
*			       TRUE.  This function opens the file,
*			       reads the header, and sets attributes 
*			       of a data object based on the information
*			       in the header.  
*		 output      - the function called when an output object
*			       is about to be closed.  This function
*			     - will open a permanent output file and 
*			       write a header based on object attributes.
*		 destroy     - this function is when any object is about
*			       to be closed (after output if its an output
*			       object).  This object frees up memory
*			       used by the object for the file format
*			       support.
*		 read_data   - read data out of a file.  This function is
*			       only used if the data set is large, and 
*			       the glue is written to support large data
*			       sets.
*		 write_data  - write datato a file.  This function is
*			       only used if the data set is large, and
*			       the glue is written to support large data
*			       sets.
*		 lock_transport - this function behaves like flock, except
*			       that it is used to lock and unlock a
*			       transport that contains a data set that
*			       associated with an open data object.
*		 order       - set this to NULL.  
*		 get_architecture - get the architecture that the data set
*			       was written on.
*		 set_architecture - for output objects set the archecture
*			       that should be used when writing the data
*			       set.
*
*        Output:
*       Returns:
*  Restrictions:
*    Written By: Jeremy Worley
*          Date: Dec 07, 1994 14:11
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration:
*
*************************************************************/

int
kdms_add_file_format(
   char  *name,
   char  *description,
   int   (*check)            PROTO((int)),
   int   (*input)            PROTO((kobject, int, int)),
   int   (*output)           PROTO((kobject, int, int)),
   int   (*destroy)          PROTO((kobject)),
   kaddr (*read_data)        PROTO((kobject, int, kaddr *, int *, int *)),
   int   (*write_data)       PROTO((kobject, int, kaddr, int *, int *)),
   int   (*lock_transport)   PROTO((kobject, int)),
   int   (*order)            PROTO((char *, int *)),
   int   (*get_architecture) PROTO((kobject)),
   int   (*set_architecture) PROTO((kobject, int)))
{
   DataServiceInformation **tmp = NULL;

   tmp = (DataServiceInformation **)kcalloc(sizeof(DataServiceInformation *), 1);
   *tmp = (DataServiceInformation *)kcalloc(sizeof(DataServiceInformation), 1);

   (*tmp)->identifier = kstrdup(name);
   (*tmp)->label = kstrdup(description);
   (*tmp)->check = check;
   (*tmp)->input = input;
   (*tmp)->output = output;
   (*tmp)->destroy = destroy;
   (*tmp)->read = read_data;
   (*tmp)->write = write_data;
   (*tmp)->order = order;
   (*tmp)->get_architecture = get_architecture;
   (*tmp)->set_architecture = set_architecture;
   (*tmp)->lock = lock_transport;
   
   services = (DataServiceInformation **) karray_add(
      (char **) services, (kaddr) (*tmp),
      kdms_num_data_formats++);

   return(TRUE);
}
