 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";

#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Data Management Services Callbacks
   >>>>
   >>>>   Static:
   >>>>                 _add_callback
   >>>>  Private:
   >>>>                 _kdms_add_callback
   >>>>                 _kdms_remove_callback
   >>>>                 _kdms_process_callback
   >>>>   Public:
   >>>>
   >>>>
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

static int _disable_callbacks = FALSE;
extern int _kdms_compiled;

/*-----------------------------------------------------------
|
|  Routine Name: (static) _add_callback
|
|       Purpose: Add a callback to a generic callback list
|
|         Input: callbacks  - callback array
|                num        - current number of callbacks
|                object     - object
|                seg        - tokenized segment name
|                clientData - clientData...
|
|        Output: num        - increments the number of
|                             callbacks.
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Apr 21, 1993 16:21
| Modifications:
|
------------------------------------------------------------*/

static kcallback **
_add_callback(
                kcallback ** callbacks,
                int *num,
                kfunc_void callback,
                kobject object,
                int seg,
                char *type,
                kaddr clientData)
{
   kcallback *tmp;

   /*
    * only a KDMS_CALLBACK_SAVE is currently allowed for object callbacks.
    */
   if (seg == 0 && 
       kstring_to_token(type) != kstring_to_token(KDMS_CALLBACK_CHANGE))
      return (NULL);
   
   callbacks = (kcallback **) krealloc(callbacks, (*num + 1) *
                                       sizeof(kcallback *));

   if (callbacks == NULL)
      return (NULL);

   tmp = (kcallback *) kcalloc(1, sizeof(kcallback));
   if (tmp == NULL)
      return (NULL);

   tmp->object = object;
   tmp->segment = seg;
   tmp->clientData = clientData;

   /*
    * there are now two types of callbacks: a data callback,
    * associated a segment, and an object callback associated with a
    * physical object. 
    */
   if (seg)
      tmp->callback.data = callback;
   else
      tmp->callback.object = callback;
   
   tmp->type = kstring_to_token(type);

   tmp->whence = NULL;

   callbacks[*num] = tmp;

   (*num)++;

   return (callbacks);
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kdms_add_callback
|
|       Purpose: This function is used to add a callback
|                associated with a segment's data or attributes.
|                the installed callback function is called
|                whenever the data or attribute value is
|                changed.
|
|         Input: object     - object in which to add callback
|                segment    - segment in which to add callback
|                type       - type of callback.  may be one of:
|                               KDMS_CALLBACK_DATA_CHANGE - write of data
|                               KDMS_CALLBACK_DATA_ACCESS - read of data
|                clientData - a pointer to information that
|                             the callback routine needs in order
|                             to service the callback appropriately.
|
|        Output:
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Apr 21, 1993 16:04
| Modifications:
|
------------------------------------------------------------*/

static int
_kdms_add_callback(
                     kobject object,
                     char *segment,
                     char *type,
                     kfunc_void callback,
                     kaddr clientData)
{
   int token = kstring_to_token(segment);
   kpresentation *pres = NULL;

   if (object == NULL || object->type != KOBJ_DATA || type == NULL)
   {
      _kdms_set_error(KDMS_EOBJ_INVALID);
      return (FALSE);
   }

   if (token == 0) 
   {
      object->phys->callbacks = _add_callback(object->phys->callbacks,
               &object->phys->num_callbacks, callback, object, token, type,
					      clientData);
   } 
   else 
   {
      if (!_kdms_get_segment_info(object, token, &pres) || pres == NULL)
	 return (FALSE);

#if 0
kprintf("\nIn Add Callback with the following info\n");
kprintf("    Segment Name    '%s'\n", ktoken_to_string(pres->id));
kprintf("    Segment Address '%x'\n", pres->segment);
kprintf("    Type of callback '%s'\n", type);
kprintf("    Routine '%x'\n", callback);
kprintf("\n");
#endif
      pres->segment->callbacks = _add_callback(pres->segment->callbacks,
					       &pres->segment->num_callbacks, 
					       callback, object, token, type,
					       clientData);
   }
   
   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kdms_remove_callback
|
|       Purpose: Deletes a callback that was previously
|                inserted by a _kdms_add_callback call.
|
|         Input:  object     - object in which to add callback
|                segment    - segment in which to add callback
|                type       - type of callback.  may be one of:
|                               KDMS_CALLBACK_DATA - change of data
|                               KDMS_CALLBACK_ATTR - change of attribute
|                clientData - a pointer to information that
|                             the callback routine needs in order
|                             to service the callback appropriately.
|
|        Output:
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Apr 21, 1993 16:30
| Modifications:
|
------------------------------------------------------------*/

static int
_kdms_remove_callback(
                        kobject object,
                        char *segment,
                        char *type,
                        kfunc_void callback,
                        kaddr clientData)
{
   int token = kstring_to_token(segment);
   int i;
   int j;
   kpresentation *pres;
   kcallback **callbacks;
   int num_callbacks;
   
   if (object == NULL || object->type != KOBJ_DATA || type == NULL)
   {
      _kdms_set_error(KDMS_EOBJ_INVALID);
      return (FALSE);
   }

   if (token == 0) 
   {
      callbacks = object->phys->callbacks;
      num_callbacks = object->phys->num_callbacks;
   } 
   else 
   {
      if (!_kdms_get_segment_info(object, token, &pres) || pres == NULL)
	 return (FALSE);

      callbacks = pres->segment->callbacks;
      num_callbacks = pres->segment->num_callbacks;
   }
   
   for (i = 0; i < num_callbacks; i++)
   {
      if (callbacks[i]->object == object &&
	  callbacks[i]->segment == token &&
	  callbacks[i]->callback.data == callback &&
	  callbacks[i]->clientData == clientData)
      {
	 kfree(callbacks[i]->whence);
	 kfree(callbacks[i]);

	 for (j = i; j < num_callbacks - 1; j++)
	    callbacks[j] = callbacks[j + 1];

	 num_callbacks -= 1;

	 callbacks =(kcallback **) krealloc(callbacks,num_callbacks * 
					    sizeof(kcallback *));
      }
   }
   
   if (token == 0)
   {
      object->phys->callbacks = callbacks;
      object->phys->num_callbacks = num_callbacks;
   } else {
      pres->segment->callbacks = callbacks;
      pres->segment->num_callbacks = num_callbacks;
   }
   
   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: kdms_delete_object_callbacks
|
|       Purpose: Deletes all callbacks.  This is used when objects
|                are closed.
|
|         Input:
|        Output:
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Dec 20, 1993 08:57
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_delete_object_callbacks(
                                kobject object,
                                int segment)
{
   int i,
       j,
       num_callbacks;
   kpresentation *pres;

   if (segment == 0)
   {
      _kdms_set_error(KDMS_ESEG_NONEXIST);
      return (FALSE);
   }

   if (object == NULL || object->type != KOBJ_DATA)
   {
      _kdms_set_error(KDMS_EOBJ_INVALID);
      return (FALSE);
   }

   if (!_kdms_get_segment_info(object, segment, &pres) || pres == NULL)
      return (FALSE);

   /*
    * this loop is backwards because the inside of the loop adjusts
    * the size of the array being checked, and thus the conditional
    * of the for loop changes on each iteration.
    */
   num_callbacks = pres->segment->num_callbacks;
   for (i = pres->segment->num_callbacks - 1; i >= 0; i--)
   {
      if (pres->segment->callbacks[i]->object == object &&
          pres->segment->callbacks[i]->segment == segment)
      {
         kfree(pres->segment->callbacks[i]->whence);
         kfree(pres->segment->callbacks[i]);
         for (j = i + 1; j < pres->segment->num_callbacks; j++)
            pres->segment->callbacks[j - 1] = pres->segment->callbacks[j];

         pres->segment->num_callbacks -= 1;
      }
   }

   if (num_callbacks != pres->segment->num_callbacks)
   {
      if (pres->segment->num_callbacks <= 0)
	 kfree(pres->segment->callbacks);
      else pres->segment->callbacks =
	       (kcallback **) krealloc(pres->segment->callbacks,
				       pres->segment->num_callbacks * 
				       sizeof(kcallback *));
      return (TRUE);
   }
   return (FALSE);              /* not an error state */
}


/*-----------------------------------------------------------
|
|  Routine Name: _kdms_process_data_callbacks
|
|       Purpose: Process the callbacks
|
|         Input:
|
|        Output:
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Apr 21, 1993 17:01
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_process_data_callbacks(
                               kobject object,
                               kpresentation * pres,
                               int *begin,
                               int *end,
                               char *type)
{
   int i;
   kdms_callback callstruct;
   static int disable = FALSE;
   int tmp[KDMS_MAX_DIM];
   kpresentation *tpres;

   /*
    * blindly disable callbacks while callbacks are being serviced
    */
   if (disable == TRUE || _disable_callbacks == TRUE)
      return (TRUE);

   callstruct.segment = ktoken_to_string(pres->id);

   disable = TRUE;

#if 0
kprintf("\nIn process callback with the following info\n");
kprintf("    Segment Name    '%s'\n", ktoken_to_string(pres->id));
kprintf("    Segment Address '%x'\n", pres->segment);
kprintf("    Number of callbacks '%d'\n", pres->segment->num_callbacks);
kprintf("\n");
#endif

   for (i = 0; i < pres->segment->num_callbacks; i++)
   {
      if (pres->segment->callbacks[i]->type == kstring_to_token(type))
      {
         callstruct.type = ktoken_to_string(pres->segment->callbacks[i]->type);

         if (!_kdms_get_segment_info(pres->segment->callbacks[i]->object,
                              pres->segment->callbacks[i]->segment, &tpres))
	    return FALSE;

         _kdms_translate_coords(pres->size, pres->order, begin,
				tpres->size, tpres->order, tmp, 
				tpres->dimension, 1,0);

         callstruct.begin.w = _kdms_get_order(KWIDTH,
                                    tpres->order, tmp, tpres->dimension, 1);
         callstruct.begin.h = _kdms_get_order(KHEIGHT,
                                    tpres->order, tmp, tpres->dimension, 1);
         callstruct.begin.d = _kdms_get_order(KDEPTH,
                                    tpres->order, tmp, tpres->dimension, 1);
         callstruct.begin.t = _kdms_get_order(KTIME,
                                    tpres->order, tmp, tpres->dimension, 1);
         callstruct.begin.e = _kdms_get_order(KELEMENTS,
                                    tpres->order, tmp, tpres->dimension, 1);

         _kdms_translate_coords(pres->size, pres->order, end,
				tpres->size, tpres->order, tmp, 
				tpres->dimension, 1,0);

         callstruct.end.w = _kdms_get_order(KWIDTH,
                                    tpres->order, tmp, tpres->dimension, 1);
         callstruct.end.h = _kdms_get_order(KHEIGHT,
                                    tpres->order, tmp, tpres->dimension, 1);
         callstruct.end.d = _kdms_get_order(KDEPTH,
                                    tpres->order, tmp, tpres->dimension, 1);
         callstruct.end.t = _kdms_get_order(KTIME,
                                    tpres->order, tmp, tpres->dimension, 1);
         callstruct.end.e = _kdms_get_order(KELEMENTS,
                                    tpres->order, tmp, tpres->dimension, 1);

         callstruct.object = pres->segment->callbacks[i]->object;
         pres->segment->callbacks[i]->callback.data(object,
                     ktoken_to_string(pres->segment->callbacks[i]->segment),
                                    pres->segment->callbacks[i]->clientData,
                                                 &callstruct);
      }
   }

   disable = FALSE;

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_process_object_callbacks
|
|       Purpose: Process the callbacks
|
|         Input:
|
|        Output:
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: May 26, 1994 15:04
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_process_object_callbacks(kobject object, char *type)
{
   int i;
   static int disable = FALSE;

   /*
    * blindly disable callbacks while callbacks are being serviced
    */
   if (disable == TRUE || _disable_callbacks == TRUE)
      return (TRUE);

   disable = TRUE;

   for (i = 0; i < object->phys->num_callbacks; i++)
   {
      if (object->phys->callbacks[i]->type == kstring_to_token(type))
      {
         object->phys->callbacks[i]->callback.object(object, type,
                  object->phys->callbacks[i]->clientData);
      }
   }

   disable = FALSE;

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_remove_all_data_callbacks
|
|       Purpose: Removes all callbacks associated with a
|                given object on a specific segment.
|
|         Input:
|
|        Output:
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: May 18, 1993 15:30
| Modifications:
|
------------------------------------------------------------*/

static int
_kdms_remove_all_data_callbacks(
                                  kobject object,
                                  int segment)
{
   kpresentation *pres = NULL;
   int i;

   if (!_kdms_get_segment_info(object, segment, &pres) || !pres)
      return (FALSE);

   for (i = 0; i < pres->segment->num_callbacks; i++)
   {
      if (pres->segment->callbacks[i]->object == object)
         _kdms_remove_callback(object,
                     ktoken_to_string(pres->segment->callbacks[i]->segment),
                        ktoken_to_string(pres->segment->callbacks[i]->type),
                               pres->segment->callbacks[i]->callback.data,
                               pres->segment->callbacks[i]->clientData);
   }

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_remove_all_object_callbacks
|
|       Purpose: Removes all callbacks associated with a
|                given object.
|
|         Input:
|
|        Output:
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: May 26, 1994 13:26
| Modifications:
|
------------------------------------------------------------*/

static int
_kdms_remove_all_object_callbacks(kobject object)
{
   int i;
   int j;
   
   for (i = 0; i < object->phys->num_callbacks; i++)
   {
      if (object->phys->callbacks[i]->object == object)
      {
         kfree(object->phys->callbacks[i]->whence);
         kfree(object->phys->callbacks[i]);

         for (j = i; j < object->phys->num_callbacks - 1; j++)
            object->phys->callbacks[j] = object->phys->callbacks[j + 1];

         object->phys->num_callbacks -= 1;

         object->phys->callbacks =
            (kcallback **) krealloc(object->phys->callbacks,
                        object->phys->num_callbacks * sizeof(kcallback *));
      }
	 
   }

   return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_remove_all_callbacks
|
|       Purpose: Removes all callback associated with a specific
|                object.  This is done so that people (xvisual)
|                who dup objects with callbacks and then close
|                the original dont get huge callback lists
|
|         Input:
|
|        Output:
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: May 18, 1993 15:25
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_remove_all_callbacks(kobject object)
{
   klist *plist;

   plist = object->presentations;

   /*
    * spin through all of the presentations and delete all of their
    * callbacks.
    */
   while (plist != NULL)
   {
      if (!_kdms_remove_all_data_callbacks(object, klist_token(plist)))
         return (FALSE);
      plist = klist_next(plist);
   }

   /*
    * now remove all of the object callbacks.
    */
   _kdms_remove_all_object_callbacks(object);

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_disable_callbacks
|
|       Purpose: External callback control
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Oct 27, 1993 11:13
| Modifications:
|
------------------------------------------------------------*/

int 
_kdms_disable_callbacks(void)
{
   _disable_callbacks = TRUE;
   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_enable_callbacks
|
|       Purpose: External callback control.
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Oct 27, 1993 11:13
| Modifications:
|
------------------------------------------------------------*/

int 
_kdms_enable_callbacks(void)
{
   _disable_callbacks = FALSE;
   return (TRUE);
}

/************************************************************
*
*  Routine Name: kdms_add_callback - add a callback associated
*                with an object's data or attribute.
*
*       Purpose: 
*                This function is used to add a callback on a 
*                segment in a data object so that operations
*                can be performed and monitored in an event
*                driven environment.
*
*                A callback is a mechanism for operating on a data 
*                object whenever an asynchronous event occurs rather 
*                than in a strictly sequential manner.  This function
*                is particularly useful in conjunction with kdms_reference.
*                The callbacks operate on the data, which is shared between
*                referenced objects.  Thus object A can set a callback
*                such that when data get changed via object B, operations
*                can be performed on object A.
*
*                The object and segment arguments are used to
*                specify which segment and object will have
*                the callback added to.
*
*                The type argument specifies what kind of callback
*                should be added.  This argument may take on the 
*                following values:
*
*       !       KDMS_CALLBACK_CHANGE - generate a callback whenever data 
*       !                              on the specified segment is changed 
*       !                              (via kdms_put_data or kdms_copy_data).
*       !       KDMS_CALLBACK_ACCESS - generate a callback whenever data
*       !                              on the specified segment is accessed
*       !                              (via kdms_get_data or kdms_copy_data).
*       !       KDMS_CALLBACK_DELETE - generate a callback whenever the
*       !                              segment specified is about to be
*       !                              deleted (usually
*       !                              via setting the KDMS_AVAILABLE attribute
*       !                              to FALSE or by calling kdms_close).
*       !       KDMS_CALLBACK_SAVE   - generate a callback whenever the
*       !                              specified segment is about to be
*       !                              changed.  This is similar to the
*       !                              KDMS_CALLBACK_CHANGE, except that
*       !                              the callback is generated before
*       !                              the data is changed rather than
*       !                              afterward.
*
*                The callback mechanims in Data Services are analogous to 
*                the callback mechanisms that are available in the Xvwidget 
*                library or in Xt.  The motivation for callbacks in Data 
*                Services is that they facilitate functionality in the 
*                Xvisual library.
*
*         Input: object     - object to add callback to.
*                segment    - segment in object to add callback
*                             to.
*                type       - type of callback to add.
*                callfunc   - function to call when callback
*                             occurs.
*                clientData - data to be passed to callfunc.
*
*        Output:
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Jeremy Worley
*          Date: Apr 21, 1993 16:41
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int
kdms_add_callback(
                    kobject object,
                    char *segment,
                    char *type,
                    kfunc_void callfunc,
                    kaddr clientData)
{

   if (!_kdms_compiled)
      _kdms_init();

   /*
    * Sanity check to make sure that object exists.
    */
   if (object == NULL || object->type != KOBJ_DATA)
   {
      _kdms_set_error(KDMS_EOBJ_INVALID);
      return (FALSE);
   }

   return (_kdms_add_callback(object, segment, type, callfunc, clientData));
}

/************************************************************
*
*  Routine Name: kdms_remove_callback - remove a callback associated
*                with an object's data or attribute.
*
*       Purpose: 
*                This function is used to remove a callback that was
*                previously added on a segment in a data object so that 
*                operations can be performed and monitored in an event
*                driven environment.
*
*                This function will not remove a callback unless all
*                of the arguments that are passed to it are the same
*                as those passed to the kdms_create_callback previously.
*                This allows for multiple callbacks that are similar
*                in nature to be removed without confusion.
*
*                A callback is a mechanism for operating on a data 
*                object whenever an asynchronous event occurs rather 
*                than in a strictly sequential manner.  This function
*                is particularly useful in conjunction with kdms_reference.
*                The callbacks operate on the data, which is shared between
*                referenced objects.  Thus object A can set a callback
*                such that when data get changed via object B, operations
*                can be performed on object A.
*
*                The object and segment arguments are used to
*                specify which segment and object will have
*                the callback added to.
*
*                The type argument specifies what kind of callback
*                should be added.  This argument may take on the 
*                following values:
*
*       !       KDMS_CALLBACK_CHANGE - generate a callback whenever data 
*       !                              on the specified segment is changed 
*       !                              (via kdms_put_data or kdms_copy_data).
*       !       KDMS_CALLBACK_ACCESS - generate a callback whenever data
*       !                              on the specified segment is accessed
*       !                              (via kdms_get_data or kdms_copy_data).
*       !       KDMS_CALLBACK_DELETE - generate a callback whenever the
*       !                              segment specified is about to be
*       !                              deleted (usually
*       !                              via setting the KDMS_AVAILABLE attribute
*       !                              to FALSE or by calling kdms_close).
*       !       KDMS_CALLBACK_SAVE   - generate a callback whenever the
*       !                              specified segment is about to be
*       !                              changed.  This is similar to the
*       !                              KDMS_CALLBACK_CHANGE, except that
*       !                              the callback is generated before
*       !                              the data is changed rather than
*       !                              afterward.
*
*                The callback mechanims in Data Services are analogous to 
*                the callback mechanisms that are available in the Xvwidget 
*                library or in Xt.  The motivation for callbacks in Data 
*                Services is that they facilitate functionality in the 
*                Xvisual library.
*
*         Input: object     - object to add callback to.
*                segment    - segment in object to add callback
*                             to.
*                type       - type of callback to add.
*                callfunc   - function to call when callback
*                             occurs.
*                clientData - data to be passed to callfunc.
*
*        Output:
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Jeremy Worley
*          Date: Apr 21, 1993 16:41
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int
kdms_remove_callback(
                       kobject object,
                       char *segment,
                       char *type,
                       kfunc_void callfunc,
                       kaddr clientData)
{

   if (!_kdms_compiled)
      _kdms_init();

   /*
    * Sanity check to make sure that object exists.
    */
   if (object == NULL || object->type != KOBJ_DATA)
   {
      _kdms_set_error(KDMS_EOBJ_INVALID);
      return (FALSE);
   }

   return (_kdms_remove_callback(object, segment, type, callfunc, clientData));
}
