 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            
   >>>>
   >>>>  Private:
   >>>>             
   >>>>   Static:
   >>>>             
   >>>>   Public:
   >>>>             
   >>>>             
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

#define VIFF_VERSION_TAG "Version"
#define VIFF_COOKIE_TAG  "ViffMagicCookie"
#define VIFF_VERSION_VAL "2.0b"
#define VIFF_COOKIE_VAL  "ThisIsAViff"

#define INCREMENT 10

static kdatum version_tag;
static kdatum version_val;
static kdatum cookie_tag;
static kdatum cookie_val;

static int viff_check PROTO((int));
static int viff_input PROTO((kobject, int, int));
static int viff_output PROTO((kobject, int, int));
static int viff_destroy PROTO((kobject));
static kaddr viff_read_data PROTO((kobject, int, kaddr *, int *, int *));
static int viff_write_data PROTO((kobject, int, kaddr, int *, int *));
static int viff_architecture PROTO((kobject));

static int _local_seek PROTO((kaddr, int, int));
static int _local_read PROTO((kaddr, kaddr, int, int));
static int _local_write PROTO((kaddr, kaddr, int, int));

DataServiceInformation viff_format[] =
{
   {
      "Khoros 2.0 Visualization Image File Format (viff)",
      "viff",
      viff_check,
      viff_input,
      viff_output,
      viff_destroy,
      viff_read_data,
      viff_write_data,
      NULL,
      viff_architecture,
      NULL,
      NULL, /* no need because the dbm stuff keeps the thing locked for us */
   }
};

/*
 *  Internal Resource Structure for the following data services
 *
 *              viff  -  New Khoros 2.0 Visualization Image File Format
 */
typedef struct
{
   kdbm *db;
   kdatum datum;                /* monkey trick */
}
ResourceStruct;

/*-----------------------------------------------------------
|
|  Routine Name: (static) _init_service
|
|       Purpose: 
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 15, 1993 12:39
| Modifications:
|
------------------------------------------------------------*/

static void
_init_service(void)
{
   if (version_tag.dptr != NULL)
      return;

   version_tag.dptr = kstring_copy(VIFF_VERSION_TAG, NULL);
   version_tag.dsize = kstrlen(version_tag.dptr) + 1;

   version_val.dptr = kstring_copy(VIFF_VERSION_VAL, NULL);
   version_val.dsize = kstrlen(version_val.dptr) + 1;

   cookie_tag.dptr = kstring_copy(VIFF_COOKIE_TAG, NULL);
   cookie_tag.dsize = kstrlen(cookie_tag.dptr) + 1;

   cookie_val.dptr = kstring_copy(VIFF_COOKIE_VAL, NULL);
   cookie_val.dsize = kstrlen(cookie_val.dptr) + 1;

   return;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _init - initializes the resource struct
|
|       Purpose: 
|
|         Input: 
|
|        Output: Returns: returns TRUE or FALSE if an error occurs
|
|    Written By: Jeremy Worley
|          Date: Nov 08, 1993 07:46
| Modifications:
|
------------------------------------------------------------*/

static int
_init(kobject object)
{
   kaddr tmp;

   /*
    * Create the image to the object file
    */
   tmp = kcalloc(1, sizeof(ResourceStruct));
   if (tmp == NULL)
      return (FALSE);

   _kdms_glue_set_resources(object, tmp);

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _sort_keys
|
|       Purpose: Maybe this is stupid, but if we encode
|                the keys correctly, then sorting them could
|                be very beneficial
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 08, 1993 08:44
| Modifications:
|
------------------------------------------------------------*/

static int
_stupid(const void *a, const void *b)
{
   kdatum *da = (kdatum *) a,
     *db = (kdatum *) b;

   return (kstrcmp((char *)da->dptr, (char *)db->dptr));
}

static int
_sort_keys(kdatum * keys, int num)
{
   (void)qsort(keys, (size_t)num, sizeof(kdatum), _stupid);
   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _get_all_keys
|
|       Purpose: This silly thing gets a list of all keys
|                in the object.
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 08, 1993 07:57
| Modifications:
|
------------------------------------------------------------*/

static kdatum *
_get_all_keys(kdbm * db, int *num)
{
   kdatum *tmp;
   int n = INCREMENT;

   tmp = kmalloc(n * sizeof(kdatum));

   if (tmp == NULL)
      return (FALSE);

   *num = 0;
   tmp[*num] = kdbm_firstkey(db);
   tmp[*num].dptr = kstrdup(tmp[*num].dptr);

   do
   {
      (*num)++;
      if (*num == n)
      {
         n += INCREMENT;
         tmp = (kdatum *) krealloc(tmp, n * sizeof(kdatum));

         if (tmp == NULL)
            return (FALSE);
      }
      tmp[*num] = kdbm_nextkey(db);
      if (tmp[*num].dptr != NULL)
         tmp[*num].dptr = kstrdup(tmp[*num].dptr);
   }
   while (tmp[*num].dptr != NULL);

   tmp = (kdatum *) krealloc(tmp, (*num) * sizeof(kdatum));
   if (tmp == NULL)
      return (FALSE);

   _sort_keys(tmp, *num);

   return (tmp);
}

/*-----------------------------------------------------------
|
|  Routine Name: _put_attribute
|
|       Purpose: 
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Dec 05, 1993 20:41
| Modifications:
|
------------------------------------------------------------*/

static int
_put_attribute(
                 kdbm * db,
                 kdatum key,
                 int num,
                 int size,
                 int type,
                 kaddr data)
{
   kdbm_lseek(db, key, 0, SEEK_SET);

   kdbm_write(db, key, &num, 1, KINT);
   kdbm_write(db, key, &size, 1, KINT);
   kdbm_write(db, key, &type, 1, KINT);

   kdbm_write(db, key, data, num * size, type);

   return (TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: _compose_key
|
|       Purpose: 
|                segment:
|                segment:attribute
|                       :attribute
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 08, 1993 11:36
| Modifications:
|
------------------------------------------------------------*/

static int
_compose_key(
               char *segment,
               char *attribute,
               kdatum * key)
{
   char tmp[KLENGTH];

   tmp[0] = '\0';

   key->dptr = kstring_3cat(segment, ":", attribute, NULL);

   key->dsize = kstrlen(key->dptr) + 1;

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _decompose_key
|
|       Purpose: 
|                segment:
|                segment:attribute
|                       :attribute
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 08, 1993 11:41
| Modifications:
|
------------------------------------------------------------*/

static int
_decompose_key(
                 kdatum key,
                 char **segment,
                 char **attribute)
{
   char *i1,
     *tmp;

   if (segment != NULL)
      *segment = NULL;

   if (attribute != NULL)
      *attribute = NULL;

   tmp = key.dptr;

   /*
    * rule 1:  if the first substring is > 0, then it is a segment
    */
   if ((i1 = kstrchr(tmp, ':')) == NULL)
      return (FALSE);

   if (i1 - tmp > 0)
   {
      *segment = kstring_ncopy(tmp, i1 - tmp, NULL);
      tmp = i1 + 1;
   }
   else
   {
      tmp = i1 + 1;
   }

   /*
    * rule 2:  everything that is left is an attribute
    */
   if (kstrlen(tmp) > 0)
      *attribute = kstring_copy(tmp, NULL);

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _write_segment_attributes
|
|       Purpose: This routine is called when segment attributes
|                are being written.  All this function does
|                is write the attributes that specifically
|                dictate the physical storage of the segment's
|                data.  Currently, this includes: size, data type,
|                index order, and dimension.
|               
|         Input: object
|                segment
|
|        Output:
|
|       Returns: nothing
|
|    Written By: Jeremy Worley
|          Date: Mar 08, 1994 13:33
| Modifications:
|
------------------------------------------------------------*/

static void
_write_segment_attributes(kobject object, char *segment, kdbm * db)
{
   int dim,
      type,
     *order,
     *size_array;
   kdatum key;

   if (!kdms_get_attributes(object, segment, 
			    KDMS_PHYSICAL_DATA_TYPE, &type,
			    KDMS_PHYSICAL_SIZE, &size_array,
			    KDMS_PHYSICAL_INDEX_ORDER, &order,
			    KDMS_PHYSICAL_DIMENSION, &dim,
			    NULL))
   {
      kinfo(KSYSLIB, "Segment '%s' doesn't exist.\n", segment);
      return;
   }

   _compose_key(segment, KDMS_DIMENSION, &key);
   _put_attribute(db, key, 1, 1, KINT, &dim);
   kfree(key.dptr);             /* compose_key allocates dptr */

   _compose_key(segment, KDMS_DATA_TYPE, &key);
   _put_attribute(db, key, 1, 1, KINT, &type);
   kfree(key.dptr);             /* compose_key allocates dptr */

   _compose_key(segment, KDMS_SIZE, &key);
   _put_attribute(db, key, 1, dim, KINT, size_array);
   kfree(key.dptr);             /* compose_key allocates dptr */

   _compose_key(segment, KDMS_INDEX_ORDER, &key);
   _put_attribute(db, key, 1, dim, KINT, order);
   kfree(key.dptr);             /* compose_key allocates dptr */

   return;
}

/*-----------------------------------------------------------
|
|  Routine Name: _write_attributes
|
|       Purpose: 
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 08, 1993 10:48
| Modifications:
|
------------------------------------------------------------*/
static int
_write_attributes(
                    kobject object,
                    char *segment,
                    kdbm * db,
                    char **atr,
                    int natr)
{
   kdatum key;
   int i,
      num_args,
      size,
      datatype,
      perm;
   kaddr tmp;
   char *name;

   /*
    * Loop through the entire list of attributes and write each
    * of the permanent ones.
    */
   for (i = 0; i < natr; i++)
   {
      /*
       * obtain the string form of the attribute name
       */
      name = atr[i];
      /*
       * get all of the meta information about the attribute.  
       */
      kdms_query_attribute(object, segment,
                           name, &num_args, &size, &datatype, &perm);
      /*
       * If the attribute doesn't have permanence (i.e. it isn't to be
       * stored), or it has no data, then don't bother to store it.
       */
      if (perm && size > 0 && num_args > 0)
      {
         kdms_get_Vattribute(object, segment,
                             name, (kaddr *) & tmp);

         _compose_key(segment, name, &key);

         _put_attribute(db, key, num_args, size, datatype, tmp);

         kfree(key.dptr);       /* compose key allocates dptr */
      }
   }

   /*
    * If this is a segment, then dump all of its "special" attributes.
    */
   if (segment != NULL)
      _write_segment_attributes(object, segment, db);

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _get_attribute
|
|       Purpose: 
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 17, 1993 15:33
| Modifications:
|
------------------------------------------------------------*/

static int 
_get_attribute(
                 kdbm * db,
                 kdatum da,
                 int *num,
                 int *size,
                 int *type,
                 kaddr * data)
{
   int sz;

   kdbm_read(db, da, num, 1, KINT);
   kdbm_read(db, da, size, 1, KINT);
   kdbm_read(db, da, type, 1, KINT);

   sz = (*num) * (*size);

   *data = kcalloc(1, (unsigned)(sz * kdata_size(*type)));

   kdbm_read(db, da, *data, sz, *type);

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) viff_check
|
|       Purpose: check to see if the file is really a viff.
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 08, 1993 07:31
| Modifications:
|
------------------------------------------------------------*/


static int
viff_check(int fid)
{
   kdbm *db;
   kdatum tmp;
   /*
    * kdbm has this cool function that does most of my work for me.
    */
   if (!kdbm_check(fid))
      return (FALSE);

   db = kdbm_fdopen(fid, kfile_flags(fid), kfile_mode(fid));

   if (db == NULL)
      return (FALSE);

   /*
    * since it definitely was a db file, we gotta have the version and
    * cookie information initialized.  So that we can complete the
    * check.
    */
   _init_service();

   /*
    * try to fetch a couple of magic fields out of the viff
    */
   tmp = kdbm_fetch(db, version_tag);
   if (tmp.dptr == 0)
      return (FALSE);

   tmp = kdbm_fetch(db, cookie_tag);
   if (tmp.dptr == 0)
      return (FALSE);

   /*
    * well, it looks like a viff...
    */
   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) viff_input
|
|       Purpose: 
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 08, 1993 07:43
| Modifications:
|
------------------------------------------------------------*/

static int
viff_input(kobject object, int fid, int flags)
{
   ResourceStruct *resources;
   kdatum *keys,
      tmp;
   int num_keys,
      kflags,
      i,
      j,
      num_args,
      size,
      datatype;
   kaddr data;
   int   rorder[KDMS_MAX_DIM];

   _init_service();

   if (!_kdms_initialized(object))
      _init(object);

   /*
    * get the resource structure
    */
   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   /*
    * open the object as a dbm file.  if this fails, then fail.
    */
   kflags = _kdms2kfile_flags(flags);

   resources->db = kdbm_fdopen(fid, kflags | KOPEN_LOCK, 0666);

   if (resources->db == NULL)
   {
      _kdms_set_error(KDMS_EFMT_OPENFAIL);
      return (FALSE);
   }

   /*
    * get a list of keys in the dbm file.  these will be the 
    * attributes and segments.
    */
   keys = _get_all_keys(resources->db, &num_keys);

   for (i = 0; i < num_keys; i++)
   {
      char *seg,
        *attr;

      if (!_decompose_key(keys[i], &seg, &attr))
         continue;

      if (kstrlen(seg) != 0 && !kdms_query_segment(object, seg))
      {
         int size_attr[KDMS_MAX_DIM],
            order[KDMS_MAX_DIM],
            type,
            dim;

         _compose_key(seg, KDMS_DIMENSION, &tmp);
         _get_attribute(resources->db, tmp, &num_args, &size,
                        &datatype, &data);
         dim = *(int *)data;
         kfree(tmp.dptr);       /* compose_key allocs */
         kfree(data);           /* get attribute allocs, but this is a special case */

         _compose_key(seg, KDMS_DATA_TYPE, &tmp);
         _get_attribute(resources->db, tmp, &num_args, &size,
                        &datatype, &data);
         type = *(int *)data;
         kfree(tmp.dptr);       /* compose_key allocs */
         kfree(data);           /* get attribute allocs, but this is a special case */

         _compose_key(seg, KDMS_SIZE, &tmp);
         _get_attribute(resources->db, tmp, &num_args, &size,
                        &datatype, &data);
         for (j = 0; j < dim; j++)
            size_attr[j] = ((int *)data)[j];
         kfree(tmp.dptr);       /* compose_key allocs */
         kfree(data);           /* get attribute allocs, but this is a special case */

         _compose_key(seg, KDMS_INDEX_ORDER, &tmp);
         _get_attribute(resources->db, tmp, &num_args, &size,
                        &datatype, &data);
         for (j = 0; j < dim; j++)
            order[j] = ((int *)data)[j];
         kfree(tmp.dptr);       /* compose_key allocs */
         kfree(data);           /* get attribute allocs, but this is a special case */

         _kdms_set_segment(object, seg, NULL, type, order, size_attr, NULL, NULL,
                           dim);

         if (kstrcmp(seg, KDMS_SEGMENT_VALUE) == 0) {
            rorder[0] = KWIDTH;
            rorder[1] = KHEIGHT;
            rorder[2] = KDEPTH;
            rorder[3] = KTIME;
            rorder[4] = KELEMENTS;
            kdms_set_attribute(object, KDMS_SEGMENT_VALUE, 
				       KDMS_INDEX_ORDER, rorder);

         } else if (kstrcmp(seg, KDMS_SEGMENT_MASK) == 0) {
            rorder[0] = KWIDTH;
            rorder[1] = KHEIGHT;
            rorder[2] = KDEPTH;
            rorder[3] = KTIME;
            rorder[4] = KELEMENTS;
            kdms_set_attribute(object, KDMS_SEGMENT_MASK, 
				       KDMS_INDEX_ORDER, rorder);

         } else if (kstrcmp(seg, KDMS_SEGMENT_LOCATION) == 0) {
            rorder[0] = KWIDTH;
            rorder[1] = KHEIGHT;
            rorder[2] = KDEPTH;
            rorder[3] = KDIMENSION;
            kdms_set_attribute(object, KDMS_SEGMENT_LOCATION, 
				       KDMS_INDEX_ORDER, rorder);

         } else if (kstrcmp(seg, KDMS_SEGMENT_MAP) == 0) {
            rorder[0] = KMAP_WIDTH;
            rorder[1] = KMAP_HEIGHT;
            rorder[2] = KMAP_DEPTH;
            rorder[3] = KMAP_TIME;
            rorder[4] = KMAP_ELEMENTS;
         }
      }

      if (kstrlen(attr) != 0)
      {                         /* (attr != NULL) {   SK & JMAS */
         /*
          * don't bother to write the attribute if its one of the
          * special attributes
          */
         if (kstrcmp(KDMS_SIZE, attr) &&
         kstrcmp(KDMS_INDEX_ORDER, attr) && kstrcmp(KDMS_DATA_TYPE, attr) &&
             kstrcmp(KDMS_DIMENSION, attr))
         {
            _get_attribute(resources->db, keys[i], &num_args, &size,
                           &datatype, &data);
            if (!kdms_query_attribute(object, seg, attr, NULL, NULL, NULL, NULL))
               kdms_create_attribute(object, seg, attr, num_args, size, datatype,
                                     TRUE, TRUE);
            kdms_set_Vattribute(object, seg, attr, data);
            /* the set attribute copies...no need to save */
            if (datatype == KSTRING)
            {
               kstring *cdata = (kstring *) data;
               for (j = 0; j < num_args * size; j++)
                  kfree(cdata[j]);
            }
            kfree(data);
         }

      }

      /* 
       * the decompose_key function allocates this space, so free it 
       */
      kfree(seg);
      kfree(attr);
   }
   _kdms_lock_segments(object);

   /*
    * the get_all_keys function allocates its space, so free it.
    */
   for (i = 0; i < num_keys; i++)
      kfree(keys[i].dptr);

   kfree(keys);

   /*
    * all done
    */
   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) viff_output
|
|       Purpose: 
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 08, 1993 09:01
| Modifications:
|
------------------------------------------------------------*/
static int
viff_output(kobject object, int fid, int flags)
{
   ResourceStruct *resources;
   int kflags;
   klist *seglist,
     *seg;
   char **atr;
   int natr;

   _init_service();

   if (!_kdms_initialized(object))
      _init(object);

   /*
    * get the resource structure
    */
   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   /*
    * open the object as a dbm file.  if this fails, then fail.
    */
   kflags = _kdms2kfile_flags(flags);

   if (!(flags & KOBJ_READ))
      kflags |= O_TRUNC;

   resources->db = kdbm_fdopen(fid, kflags | KOPEN_LOCK, 0666);

   if (resources->db == NULL)
   {
      _kdms_set_error(KDMS_EFMT_OPENFAIL);
      return (FALSE);
   }

   /*
    * dump out the version info
    */
   kdbm_store(resources->db, version_tag, version_val, KDBM_REPLACE);

   /*
    * dump out the magic cookie
    */
   kdbm_store(resources->db, cookie_tag, cookie_val, KDBM_REPLACE);

   /* 
    * get a list of the global attributes and dump 'em out
    */
   atr = kdms_get_attribute_names(object, KDMS_OBJECT, NULL, TRUE, &natr);
   _write_attributes(object, KDMS_OBJECT, resources->db, atr, natr);
   karray_free(atr, natr, NULL);

   /*
    * get a list of segments and whip through all of the segments and
    * create tag 'n stuff on the output kdbm file.
    */
   seglist = kdms_get_segment_list(object);
   for (seg = klist_head(seglist); seg != NULL; seg = klist_next(seg))
   {
      atr = kdms_get_attribute_names(object,
                                   ktoken_to_string(klist_token(seg)), NULL,
                                     TRUE, &natr);
      _write_attributes(object, ktoken_to_string(klist_token(seg)),
                        resources->db, atr, natr);
      karray_free(atr, natr, NULL);
   }

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) viff_destroy
|
|       Purpose: 
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 08, 1993 11:21
| Modifications:
|
------------------------------------------------------------*/

static int
viff_destroy(kobject object)
{
   ResourceStruct *resources;

   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   kdbm_close(resources->db);

   kfree(resources);

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _local_write
|
|       Purpose: 
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 15, 1993 14:24
| Modifications:
|
------------------------------------------------------------*/

static int 
_local_write(kaddr f, kaddr d, int n, int t)
{
   ResourceStruct *res;

   res = (ResourceStruct *) f;

   return (kdbm_write(res->db, res->datum, d, n, t));
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _local_read
|
|       Purpose: 
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 15, 1993 14:24
| Modifications:
|
------------------------------------------------------------*/

static int 
_local_read(kaddr f, kaddr d, int n, int t)
{
   ResourceStruct *res;

   res = (ResourceStruct *) f;

   return (kdbm_read(res->db, res->datum, d, n, t));
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _local_seek
|
|       Purpose: 
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 15, 1993 14:24
| Modifications:
|
------------------------------------------------------------*/

static int 
_local_seek(kaddr f, int o, int w)
{
   ResourceStruct *res;
   kdatum keypoo;

   res = (ResourceStruct *) f;
   keypoo = res->datum;

   return (kdbm_lseek(res->db, keypoo, o, w));
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) viff_read_data
|
|       Purpose: 
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 08, 1993 11:31
| Modifications:
|
------------------------------------------------------------*/

static kaddr
viff_read_data(kobject object, int seg, kaddr * data, int *begin, int *end)
{
   ResourceStruct *resources;
   int dim,
     *size,
      datatype,
      offset = 0,
      mach;
   char *segment = ktoken_to_string(seg);

   if (!kdms_get_attributes(object, segment, 
			    KDMS_PHYSICAL_DATA_TYPE, &datatype,
			    KDMS_PHYSICAL_SIZE, &size,
			    KDMS_PHYSICAL_DIMENSION, &dim,
			    NULL))
      return (FALSE);

   /*
    * get the resource structure
    */
   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   _compose_key(segment, NULL, &resources->datum);

   mach = kdbm_getmachtype(resources->db);

   _kdms_glue_read_segment_data(
                          resources, mach, _local_read, _local_seek, offset,
                              datatype, dim, size, NULL, data, begin, 
			  end);

   kfree(resources->datum.dptr);        /* compose_key allocs */

   return (*data);
}

/*-----------------------------------------------------------
|
|  Routine Name: viff_write_data
|
|       Purpose: 
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 02, 1993 07:09
| Modifications:
|
------------------------------------------------------------*/

static int
viff_write_data(kobject object, int seg, kaddr data, int *begin, int *end)
{
   ResourceStruct *resources;
   int dim,
     *size,
      datatype,
      offset = 0,
      status;
   char *segment = ktoken_to_string(seg);

   if (!kdms_get_attributes(object, segment, 
			    KDMS_PHYSICAL_DATA_TYPE, &datatype,
			    KDMS_PHYSICAL_SIZE, &size,
			    KDMS_PHYSICAL_DIMENSION, &dim,
			    NULL))
      return (FALSE);

   /*
    * get the resource structure
    */
   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   _compose_key(segment, NULL, &resources->datum);

   status = _kdms_glue_write_segment_data(
                  resources, KMACH_LOCAL, _local_write, _local_seek, offset,
                              datatype, dim, size, NULL, data, begin, 
		  end);

   kfree(resources->datum.dptr);        /* compose_key allocs */

   return (status);
}

/*-----------------------------------------------------------
|
|  Routine Name: viff_architecture
|
|       Purpose: Return the architecture that the data was
|                generated on in khoros-speak.
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 08, 1993 11:35
| Modifications:
|
------------------------------------------------------------*/

static int
viff_architecture(kobject object)
{
   ResourceStruct *resources;

   /*
    * get the resource structure
    */
   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   if (resources == NULL)
      return (KMACH_LOCAL);
   else
      return (kdbm_getmachtype(resources->db));

}
