 /*
  * Khoros: $Id$
  */

#undef _POSIX_SOURCE

#if !defined(lint) && !defined(SABER)
static char rcsid[] = "Khoros: $Id$";

#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

#include "copyright.h"	/* Khoros copyright */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Khoros Arf Data Service Routines
   >>>>
   >>>>   Static:
   >>>>  Private:
   >>>>             arf_check()
   >>>>             arf_input()
   >>>>             arf_output()
   >>>>             arf_destroy()
   >>>>             arf_read()
   >>>>             arf_write()
   >>>>             arf_order()
   >>>>             arf_architecture()
   >>>>             _init()
   >>>>   Public:
   >>>>		    None - no public should ever exist as these are
   >>>>			   internal routines only accessible via the Khoros   
   >>>>			   DataServiceInformation *services[] structure.
   >>>>
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

#if !defined(KARF_DEF)
DataServiceInformation arf_format[] = {NULL};
#else

#include "kdatafmt/arf.h"

static int _init PROTO((kobject));
static int arf_check PROTO((int));
static int arf_input PROTO((kobject, int, int));
static int arf_output PROTO((kobject, int, int));
static int arf_destroy PROTO((kobject));
static int arf_order PROTO((char *, int *));
static int arf_architecture PROTO((kobject object));
static kaddr arf_read PROTO((kobject, int, kaddr *, int *, int *));
static int arf_write PROTO((kobject, int, kaddr, int *, int *));

static int define_arf_colormap_attributes PROTO((void));
static int define_arf_comment_attribute PROTO((void));
int malloc_attribute_arrays PROTO((unsigned long, unsigned long,
				   ARF_FOOTER_ATTRIBUTES *));
     
int create_arf_footer_attributes PROTO((kobject, unsigned long, 
					unsigned long));
int set_arf_footer_attributes PROTO((kobject, ARF_FOOTER_ATTRIBUTES *,
				     unsigned long));
int get_arf_footer_attributes PROTO((kobject, ARF_FOOTER_ATTRIBUTES *,
				     unsigned long));
     
int create_arf_info_attributes PROTO((kobject, ARF_INFO *));
int set_arf_info_attributes PROTO((kobject, ARF_INFO *));
int get_arf_info_attributes PROTO((kobject, ARF_INFO *));
    
int create_arf_ladar_param_attributes PROTO((kobject));
int set_arf_ladar_param_attributes PROTO((kobject, ARF_LADAR_PARAM *));
int get_arf_ladar_param_attributes PROTO((kobject, ARF_LADAR_PARAM *));


DataServiceInformation arf_format[] =
{
   {
      "Sun Arf file File Format (arf)",
      "arf",
      arf_check,
      arf_input,
      arf_output,
      arf_destroy,
      arf_read,
      arf_write,
      arf_order,
      arf_architecture,
      NULL,
   }
};

/*
 *  Internal Resource Structure for the following data services
 *
 *		arf  -  ATRWG ARF format
 */
typedef struct 
{
  kfile *file;
  ARF_SUBHEADER *subh;
  ARF_FRAME_FOOTER *frame_footer;
  ARF *arf;
  unsigned long frame_size, info_size, colormap_size, comment_size;
  unsigned long multi_band_size, frame_data_size, ladar_param_size, frame_foot_size;
  unsigned long info_offset, colormap_offset, comment_offset, multi_band_offset;
  unsigned long frame_data_offset, ladar_param_offset;
  char *imagedata;
}  ResourceStruct;

/*

*/
/*-----------------------------------------------------------
|
|  Routine Name: _init - Creates an arf image
|
|       Purpose: This function is used to create an arf image.  Which
|		 means that we use the current dimension values in order
|		 to create a proper arf.
|
|         Input: data - initial sample of data
|                num  - number of bytes in the sample
|
|        Output: Returns: returns TRUE or FALSE if an error occurs
|
|    Written By: James A. Moulton
|          Date: Aug 11, 1993
| Modifications: Apr 11, 1994 Tim Williams - modeled after a version of
|                                            Xvimage.c. It looks as though
|                                            James used Rast.c as a model.
|
------------------------------------------------------------*/
static int
_init(kobject object)
{
   ResourceStruct *resources;


   /*
    * Create the image to the object file
    */
   resources = (ResourceStruct *) kcalloc(1, sizeof(ResourceStruct));
     if (resources == NULL)
      return (FALSE);

   resources->info_size = 0;
   resources->colormap_size = 0;
   resources->comment_size = 0;
   resources->multi_band_size = 0;
   resources->frame_data_size = 0;
   resources->ladar_param_size = 0;
   resources->frame_foot_size = 0;
   
   /* default to size of ARF_MAIN_HDR */
   resources->info_offset = 32;
   resources->colormap_offset = 32;
   resources->comment_offset = 32;
   resources->multi_band_offset = 32;
   resources->frame_data_offset = 32;
   resources->ladar_param_offset = 32;

   _kdms_glue_set_resources(object,resources);

   return (TRUE);
}

/*

*/
/*-----------------------------------------------------------
|
|  Routine Name: arf_check - Checks to see if the file is
|				 an arf
|
|       Purpose: This function is used to check the first 4 bytes of
|		 the stream to see if it is an arf.
|
|         Input: file - the arf file to be checked.
|
|        Output: Returns: returns TRUE or FALSE if an error occurs
|
|    Written By: Tim Williams
|          Date: Apr 22, 1994
| Modifications: 
|
------------------------------------------------------------*/
static int
arf_check(int fid)
{
   unsigned long magic_num;


   /* 
    * set machtype to KARCH_SUN since ARF is written with XDR 
    * (big-endian, IEEE floats)
    */
   kfile_setmachtype(fid, KMACH_SPARC);

   /* read first 4 bytes for magic number */
   kread_generic(fid, &magic_num, 1, KULONG);

   if (magic_num != (unsigned long)ARF_MAGIC_NUM)
     return (FALSE);

   return(TRUE);
}

/*

*/

/*-----------------------------------------------------------
  |
  |  Routine Name: arf_input - Reads an arf image
  |
  |       Purpose: This function is used to read in the image if the
  |		 supplied data is arf image.  The data was checked
  |		 by the arf_check() routine in which the arf
  |		 identifier indicated that the data is a valid arf.
  |		 Since the data was valid we can assume simply read
  |		 the header in and initialize that data segments.
  |
  |         Input: object - the arf object to be initialized
  |
  |        Output: Returns: returns TRUE or FALSE if an error occurs
  |
  |    Written By: James A. Moulton
  |          Date: Aug 11, 1993
  | Modifications: 13 June, 1994 Tim Williams - modeled after a version of
  |                                            Xvimage.c. It looks as though
  |                                            James used Rast.c as a model.
  |
  ------------------------------------------------------------*/
static int
  arf_input(kobject object, int fid, int flags)
{
  ResourceStruct *resources;
  ARF *arf;
  ARF_SUBHEADER *subh = NULL;
  kfile *file;
  int offset, type, i;
  int frame;
  int size[KDMS_MAX_DIM], order[KDMS_MAX_DIM];
  int begin[KDMS_MAX_DIM], end[KDMS_MAX_DIM];
  char ktype[KLENGTH];
  int kflags;
  ARF_FOOTER_ATTRIBUTES footer_attributes;

  if (!_kdms_initialized(object))
    _init(object);
  
  resources = (ResourceStruct *) _kdms_glue_get_resources(object);
  
  /*
   * Read the image from the object file and go ahead and set the arf image
   * within the resource structure.
   */
  kflags = _kdms2kfile_flags(flags);
  file = kfdopen(fid, kflags_to_type(kflags, ktype));
  if (file == NULL) 
  {
    _kdms_set_error(KDMS_EFMT_OPENFAIL);
    return (FALSE);
  }
  resources->file = file;
  
  /*
   * set machtype to KARCH_SUN since ARF uses XDR
   * (big-endian, IEEE floats)
   */
  kfile_setmachtype(kfileno(resources->file), KMACH_SPARC);
  

  if ((arf=(ARF *)kmalloc(sizeof(ARF))) == NULL)
  {
    kerror("data services", "arf_input", 
	   "not enough memory for ARF structure");
    return (FALSE);
  }
  resources->arf = arf;
  /* get 32 byte ARF main header (8 ulongs) */
  arf_get_main_hdr(file, &(resources->arf->hdr));

  /* 
   * create attribute to save ambiguous ARF data type
   * arf_ten_bit, arf_twelve_bit, arf_short_word -> KUSHORT 
   */
  kdms_create_attribute(object, NULL, "ARF_DATATYPE", 1, 1, KINT, 
			TRUE, TRUE);
  kdms_set_attribute(object, NULL, "ARF_DATATYPE", 
		     arf->hdr._image_type);
  
  /*
   * Initialize the data segments according to the data found in the arf
   * structure.
   * 
   * This initializes the arf value data.
   */
  
  /*
   * check for subheaders
   *
   * the arf_get_* routines will read the structure requested and return 
   * the length in the file of the subheader
   */
  
  if (arf->hdr._subheader_flags)
  {
    if ((subh=(ARF_SUBHEADER *)kmalloc(sizeof(ARF_SUBHEADER)))==NULL)
    {
      kerror("data services", "arf_input",
	     "not enough memory for ARF_SUBHEADER structure");
      return (FALSE);
    }
    resources->subh = subh;
    if (arf->hdr._subheader_flags & ARF_INFO_FLAG)
    {
      resources->info_size = arf_get_info(file, subh);
      if (!resources->info_size)
	return (FALSE);

      /* create and set attributes */
      create_arf_info_attributes(object, &subh->info);
      if (!set_arf_info_attributes(object, &subh->info))
      {
	kerror("data services", "arf_input",
	       "Error setting ARF_INFO attriutes");
	return(FALSE);
      }
    }
    
    resources->colormap_offset = resources->info_offset + resources->info_size;
    if (arf->hdr._subheader_flags & ARF_COLORMAP_FLAG)
    {
      unsigned char *mapdata, *mapptr;

      resources->colormap_size = arf_get_colormap(file, subh);
      size[0] = 256;
      size[1] = 3;
      size[2] = size[3] = size[4] = 1;
      
      order[0] = KMAP_HEIGHT;
      order[1] = KMAP_WIDTH;
      order[2] = KMAP_ELEMENTS;
      order[3] = KMAP_DEPTH;
      order[4] = KMAP_TIME;

      begin[0] = begin[1] = begin[2] = begin[3] = begin[4] = 0;
      end[0] = 255;
      end[1] = 2;
      end[2] = end[3] = end[4] = 0;
      
      mapdata = (unsigned char *)kmalloc(3*ARF_COLORMAP_SIZE*sizeof(char));
      mapptr = mapdata;
      for (i=0; i<ARF_COLORMAP_SIZE; i++, mapptr++)
	*mapptr=subh->colormap.red[i];
      for (i=0; i<ARF_COLORMAP_SIZE; i++, mapptr++)
	*mapptr=subh->colormap.green[i];
      for (i=0; i<ARF_COLORMAP_SIZE; i++, mapptr++)
	*mapptr=subh->colormap.blue[i];
      
      define_arf_colormap_attributes();
      if (!kdms_set_attribute(object, NULL, _INTERNAL_COLORSPACE, KRGB))
      {
	kerror("data services", "arf_input",
	       "error setting KIMG_COLORSPACE attribute");
      }
      _kdms_set_segment(object, KDMS_SEGMENT_MAP, mapdata,
			KUBYTE, order, size, begin, end, 5);

      /* need to make the map index order "correct" on loading */
      order[0] = KMAP_WIDTH;
      order[1] = KMAP_HEIGHT;
      order[2] = KMAP_DEPTH;
      order[3] = KMAP_TIME;
      order[4] = KMAP_ELEMENTS;
      kdms_set_attribute(object, KDMS_SEGMENT_MAP, KDMS_INDEX_ORDER, order);

      /*
       * it is vitally important that you include this step because if the
       * data set larger than the buffer threshold, then spurous flushes
       * will ensue that should not happen...it turns out that xvimages and
       * viffs will cope with that situation (though not very efficiently),
       * but file formats that buffer the whole thing in memory won't.
       */
      kdms_set_attribute(object, KDMS_SEGMENT_MAP, KDMS_COUPLING,KUNCOUPLED);

    }

    resources->comment_offset = resources->colormap_offset + 
      resources->colormap_size;

    if (arf->hdr._subheader_flags & ARF_COMMENT_FLAG)
    {
      define_arf_comment_attribute();
      resources->comment_size = arf_get_comment(file, subh);
      if (resources->comment_size)
	kdms_set_attribute(object, NULL, KDMS_COMMENT, subh->comment.bs);
      else
	return (FALSE);
    }
    resources->multi_band_offset = resources->comment_offset + 
      resources->comment_size;
    
    if (arf->hdr._subheader_flags & ARF_MULTI_BAND_FLAG)
    {
      resources->multi_band_size = arf_get_multi_band(file, subh);
      if (resources->multi_band_size)
      {
	/* create and set attributes */
/*
	create_arf_multi_band_attributes(object, &subh->multi_band);
	set_arf_multi_band_attributes(object, &subh->multi_band);
*/
	for (i=0; i < (int)subh->multi_band.num_bands; i++)
	{
	  arf2khoros_datatype(subh->multi_band.band[i].type, &type);
	  resources->frame_size += 
	    arf->hdr._num_rows *
	    arf->hdr._num_cols * 
	      kmach_sizeof(KMACH_SPARC, type);
	}
      } /* ARF_MULTI_BAND_FLAG */
      else
	return (FALSE);
    }
    else
    {
      arf2khoros_datatype(arf->hdr._image_type, &type);
      resources->frame_size = 
	arf->hdr._num_rows *
	arf->hdr._num_cols * 
	  kmach_sizeof(KMACH_SPARC, type);
    } /* not ARF_MULTI_BAND_FLAG */

    resources->frame_data_offset = resources->multi_band_offset + 
      resources->multi_band_size;
   
    if (arf->hdr._subheader_flags & ARF_FRAME_DATA_FLAG) /* footers exist */
    {
      resources->frame_data_size = arf_get_frame_data(file, subh);

      /*
       * Allocate footer structure, attribute arrays,
       * and define footer attributes
       */

      /* create attribute for footer flags */
      kdms_create_attribute(object, NULL, "ARF_FRAME_DATA_footer_flags", 1, 1,
			    KULONG, TRUE, TRUE);
      kdms_set_attribute(object, NULL, 
			 "ARF_FRAME_DATA_footer_flags", 
			 subh->frame_data.footer_flags);

      resources->frame_footer = 
	(ARF_FRAME_FOOTER*)kmalloc(sizeof(ARF_FRAME_FOOTER));
      if (resources->frame_footer == NULL)
      {
	kerror("data services", "arf_input",
	       "not enough memory to create footers");
	return (FALSE);
      }
      /* create footer attributes and allocate arrays */
      if (!malloc_attribute_arrays(subh->frame_data.footer_flags,
				   arf->hdr._num_frames,
				   &footer_attributes))
      {
	kerror("data services", "arf_input", 
	       "error allocating footer attribute arrays");
	return (FALSE);
      }
      create_arf_footer_attributes(object, subh->frame_data.footer_flags, 
				   arf->hdr._num_frames);

      /*
       * read footers AFTER reading all subheaders
       */

    } /* ARF_FRAME_DATA_FLAG */

    resources->ladar_param_offset = resources->frame_data_offset + 
      resources->frame_data_size;
    if (arf->hdr._subheader_flags & ARF_LADAR_PARAM_FLAG)
    {
      resources->ladar_param_size = arf_get_ladar_param(file, subh);
      
      /* create and set attributes */
      create_arf_ladar_param_attributes(object);
      set_arf_ladar_param_attributes(object, &subh->ladar_param);
    }
    
    offset = resources->ladar_param_offset + resources->ladar_param_size;
    /* this should match arf->hdr._image_offset */
    if (offset != arf->hdr._image_offset)
    {
      kerror("data services", "arf_input",
	     "image offset %d calculated from subheaders doesn't match offset in main ARF header %d", offset, arf->hdr._image_offset);
      return (FALSE);
    }
  } /* end subheaders */
  
  /* read the footers */
  if (resources->frame_data_size)
  {
    int here;
    here = kftell(file);

    /* go to start of image data */
    kfseek(file, (long)resources->arf->hdr._image_offset, KSEEK_SET);
    for (frame=0; frame<resources->arf->hdr._num_frames; frame++)
    {
      /* skip past the frame */
      kfseek(file, (long)resources->frame_size, KSEEK_CUR);
      resources->frame_foot_size = 
	arf_get_footer(file, 
		       subh->frame_data.footer_flags, 
		       resources->frame_footer);
      if (subh->frame_data.footer_flags & ARF_FRAME_INFO_FLAG)
      {
	footer_attributes.frame_info_x[frame] = 
	  resources->frame_footer->info.x;
	footer_attributes.frame_info_y[frame] = 
	  resources->frame_footer->info.y;
	footer_attributes.frame_info_az[frame] = 
	  resources->frame_footer->info.azimuth;
	footer_attributes.frame_info_el[frame] = 
	  resources->frame_footer->info.elevation;
	footer_attributes.frame_info_irig_yy[frame] = 
	  resources->frame_footer->info.irig_time.yy;
	footer_attributes.frame_info_irig_dd[frame] = 
	  resources->frame_footer->info.irig_time.dd;
	footer_attributes.frame_info_irig_hh[frame] = 
	  resources->frame_footer->info.irig_time.hh;
	footer_attributes.frame_info_irig_mm[frame] = 
	  resources->frame_footer->info.irig_time.mm;
	footer_attributes.frame_info_irig_ss[frame] = 
	  resources->frame_footer->info.irig_time.ss;
	footer_attributes.frame_info_irig_ms[frame] = 
	  resources->frame_footer->info.irig_time.ms;
      }
      
      if (subh->frame_data.footer_flags & ARF_PLATFORM_ATTITUDE_FLAG)
      {
	footer_attributes.platform_attitude_vel_n[frame] = 
	  resources->frame_footer->attitude.vel_north;
	footer_attributes.platform_attitude_vel_e[frame] = 
	  resources->frame_footer->attitude.vel_east;
	footer_attributes.platform_attitude_vel_v[frame] = 
	  resources->frame_footer->attitude.vel_vertical;
	footer_attributes.platform_attitude_acc_n[frame] = 
	  resources->frame_footer->attitude.acc_north;
	footer_attributes.platform_attitude_acc_e[frame] = 
	  resources->frame_footer->attitude.acc_east;
	footer_attributes.platform_attitude_acc_v[frame] = 
	  resources->frame_footer->attitude.acc_vertical;
	footer_attributes.platform_attitude_lat[frame] = 
	  resources->frame_footer->attitude.latitude;
	footer_attributes.platform_attitude_long[frame] = 
	  resources->frame_footer->attitude.longitude;
	footer_attributes.platform_attitude_alt[frame] = 
	  resources->frame_footer->attitude.altitude;
	footer_attributes.platform_attitude_elev[frame] = 
	  resources->frame_footer->attitude.elevation;
	footer_attributes.platform_attitude_bank[frame] = 
	  resources->frame_footer->attitude.bank;
	footer_attributes.platform_attitude_head[frame] = 
	  resources->frame_footer->attitude.heading;
      }
    } /* frame loop */
    
    /* go back to where you were */
    kfseek(file, here, KSEEK_SET);
    
    /* now set the attributes defined above */
    set_arf_footer_attributes(object, &footer_attributes, 
			      subh->frame_data.footer_flags);
  }
  
  /* check the value data */
  size[0] = arf->hdr._num_cols;
  size[1] = arf->hdr._num_rows;
  if (arf->hdr._image_type == arf_multi_band)
    size[2] = subh->multi_band.num_bands;
  else
    size[2] = 1; 
  size[3] = arf->hdr._num_frames;
  size[4] = 1;
  
  
  order[0] = KWIDTH;
  order[1] = KHEIGHT;
  order[2] = KELEMENTS;
  order[3] = KTIME;
  order[4] = KDEPTH;
  
  arf2khoros_datatype(arf->hdr._image_type, &type);
  
  _kdms_set_segment(object, KDMS_SEGMENT_VALUE, NULL, type,
		    order, size, NULL, NULL, 5);
  
  /* set order to "right" order */
  order[2]=KDEPTH;
  order[4]=KELEMENTS;
  kdms_set_attribute(object, KDMS_SEGMENT_VALUE, KDMS_INDEX_ORDER, order);

  /*
   * it is vitally important that you include this step because if the
   * data set larger than the buffer threshold, then spurous flushes
   * will ensue that should not happen...it turns out that xvimages and
   * viffs will cope with that situation (though not very efficiently),
   * but file formats that buffer the whole thing in memory won't.
   */
  kdms_set_attribute(object, KDMS_SEGMENT_VALUE, KDMS_COUPLING,KUNCOUPLED);
  
  return (TRUE);
}

/*

*/
/*-----------------------------------------------------------
|
|  Routine Name: arf_output - Closes the arf image
|
|       Purpose: This function is used to close the arf image.  Which
|		 means if the image was modified we then re-write image
|		 back out.
|
|         Input: object - the arf object to be closed
|
|        Output: Returns: returns TRUE or FALSE if an error occurs
|
|    Written By: Tim Williams
|          Date: May 16, 1994
| Modifications: ARF_MULTI_BAND data returns FALSE for now - tjw
|
------------------------------------------------------------*/
static int
  arf_output(kobject object, int fid, int flags)
{
   ResourceStruct *resources;

   ARF *arf;
   ARF_SUBHEADER *subh;
   kfile *file;
   int type, arf_type, frame;
   int *size, *order;
   char ktype[KLENGTH];
   int kflags, colorspace;
   ARF_FOOTER_ATTRIBUTES footer_attributes;
   char *tmpcomment=NULL;
   int i;

   if (!_kdms_initialized(object)) 
     _init(object);

   resources = (ResourceStruct *) _kdms_glue_get_resources(object);
   
   kflags = _kdms2kfile_flags(flags);
   resources->file = kfdopen(fid, kflags_to_type(kflags, ktype));
   if (resources->file == NULL) {
     _kdms_set_error(KDMS_EFMT_OPENFAIL);
     return(FALSE);
   }
   file = (kfile *)resources->file;
   
   if (resources->subh == NULL) /* set up ARF_SUBHEADER structure */
   {
     resources->subh=(ARF_SUBHEADER *)kmalloc(sizeof(ARF_SUBHEADER));
     resources->subh=(ARF_SUBHEADER *)kmemset((kaddr)resources->subh, 0, 
					      sizeof(ARF_SUBHEADER));
     if (!resources->subh)
     {
       _kdms_traceback("arf_output");
       _kdms_set_error(KDMS_EFMT_FAILURE);
       return (FALSE);
     }
   }
   subh = (ARF_SUBHEADER *)resources->subh;
   
   /*
    * set machtype to KARCH_SUN since ARF uses XDR
    * (big-endian, IEEE floats)
    */
   kfile_setmachtype(kfileno(resources->file), KMACH_SPARC);
   
   if (resources->arf == NULL) /* set up ARF structure */
   {
     resources->arf = (ARF *)kmalloc(sizeof(ARF));
     resources->arf = (ARF *)kmemset((kaddr)resources->arf, 0, sizeof(ARF));
     if (!resources->arf)
     {
       _kdms_traceback("arf_output");
       _kdms_set_error(KDMS_EFMT_FAILURE);
       return (FALSE);
     }
   }
   arf = resources->arf;
   
   /*
    * check the value data
    */
   if (kdms_get_attributes(object, KDMS_SEGMENT_VALUE, 
			    KDMS_PHYSICAL_DATA_TYPE, &type,
			    KDMS_PHYSICAL_SIZE, &size,
			    KDMS_PHYSICAL_INDEX_ORDER, &order,
			    NULL) == TRUE)
   {       
     arf->hdr._magic_num = (unsigned long)ARF_MAGIC_NUM;
     arf->hdr._version = ARF_CURRENT_VERSION;
     arf->hdr._num_cols = size[0];
     arf->hdr._num_rows = size[1];
     arf->hdr._num_frames = size[3];
     khoros2arf_datatype(type, &arf->hdr._image_type);
     /* 
      * resolve KUSHORT to arf_ten_bit, arf_twelve_bit, or arf_short_word 
      * if possible
      */
     if ((type == KUSHORT) &&
	 (kdms_get_attribute(object, NULL, "ARF_DATATYPE", &arf_type)))
        arf->hdr._image_type = (ARF_IMAGE_TYPE)arf_type;

     arf->hdr._image_offset = 32; 
     arf->hdr._subheader_flags = 0L;
   }
   
   /* get ARF_INFO subheader attributes, if any */
   resources->info_size = get_arf_info_attributes(object, &subh->info);
   if (resources->info_size)
   {
     arf->hdr._subheader_flags |= ARF_INFO_FLAG;
     arf->hdr._image_offset += resources->info_size;
   }
   
   /* get ARF_COLORMAP subheader attributes, if any */
   if (kdms_get_attribute(object, NULL, _INTERNAL_COLORSPACE, &colorspace))
   {
     kaddr data;
     unsigned char *mapptr;
     int begin[KDMS_MAX_DIM] = {0,0,0,0,0};
     int end[KDMS_MAX_DIM];
     
     kdms_get_attributes(object, KDMS_SEGMENT_MAP, 
			 KDMS_PHYSICAL_DATA_TYPE, &type,
			 KDMS_PHYSICAL_SIZE, &size,
			 KDMS_PHYSICAL_INDEX_ORDER, &order,
			 NULL);
     end[0] = size[0] - 1;
     end[1] = size[1] - 1;
     end[2] = size[2] - 1;
     end[3] = size[3] - 1;
     end[4] = size[4] - 1;
     
     data = kdms_get_data(object, NULL, begin, end, NULL);
     
     if ((size[0]<=ARF_COLORMAP_SIZE) &&
	 (size[1] == 3))
     {
       mapptr = data;
       for (i=0; i<size[0]; i++, mapptr++)
	 subh->colormap.red[i] = (unsigned char)*mapptr;
       for (i=0; i<size[0]; i++, mapptr++)
	 subh->colormap.green[i] = (unsigned char)*mapptr;
       for (i=0; i<size[0]; i++, mapptr++)
	 subh->colormap.blue[i] = (unsigned char)*mapptr;
       
       /* 256*4*3=3072 */
       resources->colormap_size = ARF_COLORMAP_SIZE * 4 * 3; 
       arf->hdr._subheader_flags |= ARF_COLORMAP_FLAG;
       arf->hdr._image_offset += resources->colormap_size;
     }
   }     

   /* get ARF_COMMENT subheader attributes, if any */
   if (kdms_get_attribute(object, NULL, KDMS_COMMENT, &tmpcomment))
   {
     int length;
     
     kstrcpy(subh->comment.bs, tmpcomment);
     length = kstrlen(subh->comment.bs);
     if (length != 0) 
     {
       resources->comment_size = 
	 length%4 ? (4+length+(4-length%4)) : (4+length);
       
       arf->hdr._subheader_flags |= ARF_COMMENT_FLAG;
       arf->hdr._image_offset += resources->comment_size;
     }
   }
   
   /* get ARF_MULTI_BAND subheader attributes, if any */
/*
   resources->multi_band_size=
     get_arf_multi_band_attributes(object, &subh->multi_band);
   if (resources->multi_band_size != 0) 
   {
     arf->hdr._subheader_flags |= ARF_MULTI_BAND_FLAG;
     arf->hdr._image_offset += resources->multi_band_size;
     arf->hdr._image_type = arf_multi_band;
     for (i=0; i<subh->multi_band.num_bands; i++)
     {
       arf2khoros_datatype(subh->multi_band.band[i].type, &type);
       resources->frame_size += 
	 arf->hdr._num_rows *
	   arf->hdr._num_cols * 
	     kmach_sizeof(KARCH_SUN, type);
     }
   }
*/
   resources->multi_band_size = 0;
   if ((size[2] == 3) && 
       (type == KUBYTE)) /* only ARF_RGB currently supported */
   {
     long tmpval;
     static char *band_name[]={"red", "green", "blue"};
     
     arf->hdr._subheader_flags |= ARF_MULTI_BAND_FLAG;
     
     kstrcpy(subh->multi_band.name, "rgb");
     tmpval = kstrlen(subh->multi_band.name);
     resources->multi_band_size += tmpval%4 ? 4+tmpval+(4-tmpval%4) : 4+tmpval;
     
     subh->multi_band.num_bands = size[2];
     resources->multi_band_size += 4;

     for (i=0; i < size[2]; i++)
     {
       kstrcpy(subh->multi_band.band[i].name, band_name[i]);
       tmpval = kstrlen(subh->multi_band.band[i].name);
       resources->multi_band_size += 
	 tmpval%4 ? 4+tmpval+(4-tmpval%4) : 4+tmpval;

       subh->multi_band.band[i].type = arf->hdr._image_type;
       resources->multi_band_size += 4;
     }
     arf2khoros_datatype(arf->hdr._image_type, &type);
     resources->frame_size += 
       arf->hdr._num_rows *
	 arf->hdr._num_cols * 
	   kmach_sizeof(KMACH_SPARC, type);
     
     /* increment subhsize for empty bands up to 16 (ARF_MAX_BANDS) */
     resources->multi_band_size += 8*(ARF_MAX_BANDS-
				      subh->multi_band.num_bands);
     
     arf->hdr._image_offset +=  resources->multi_band_size;
     arf->hdr._image_type = arf_multi_band;
   }
   else
   {
     arf2khoros_datatype(arf->hdr._image_type, &type);
     resources->frame_size = 
       arf->hdr._num_rows *
	 arf->hdr._num_cols * 
	   kmach_sizeof(KMACH_SPARC, type);
   }
   
   /* get ARF_FRAME_DATA subheader attributes, if any */
   if (kdms_get_attribute(object, NULL,
			  "ARF_FRAME_DATA_footer_flags",
			  &subh->frame_data.footer_flags))
   {
     resources->frame_data_size = 4;
     arf->hdr._subheader_flags |= ARF_FRAME_DATA_FLAG;
     arf->hdr._image_offset += resources->frame_data_size;
     resources->frame_footer = 
       (ARF_FRAME_FOOTER*)kmalloc(sizeof(ARF_FRAME_FOOTER));
     if (resources->frame_footer == NULL)
     {
       kerror("data services", "arf_output",
	      "not enough memory to create footers");
       return (FALSE);
     }
   }
   
   /* get ARF_LADAR_PARAM subheader attributes, if any */
   resources->ladar_param_size=
     get_arf_ladar_param_attributes(object, &subh->ladar_param);
   if (resources->ladar_param_size != 0) 
   {
     arf->hdr._subheader_flags |= ARF_LADAR_PARAM_FLAG;
     arf->hdr._image_offset += resources->ladar_param_size;
   }
   
   /*
    * Write the arf structure out to the specified filename
    */
   
   /* write 32 byte ARF main header (8 unsigned longs) */
   arf_put_main_hdr(file, &arf->hdr);
   
   /* write sub-headers */
   if (arf->hdr._subheader_flags)
   {
     if (arf->hdr._subheader_flags & ARF_INFO_FLAG)
       arf_put_info(file, &resources->subh->info);
     if (arf->hdr._subheader_flags & ARF_COLORMAP_FLAG)
       arf_put_colormap(file, &resources->subh->colormap);
     if (arf->hdr._subheader_flags & ARF_COMMENT_FLAG)
       arf_put_comment(file, &resources->subh->comment);
     if (arf->hdr._subheader_flags & ARF_MULTI_BAND_FLAG)
       arf_put_multi_band(file, &resources->subh->multi_band);
     if (arf->hdr._subheader_flags & ARF_FRAME_DATA_FLAG)
       arf_put_frame_data(file, &resources->subh->frame_data);
     if (arf->hdr._subheader_flags & ARF_LADAR_PARAM_FLAG)
       arf_put_ladar_param(file, &resources->subh->ladar_param);
   }
   
   /* write footers */
   if (resources->frame_data_size)
   {
     int here;
     here = kftell(file);
     /* go to start of image data */
     kfseek(file, (int)resources->arf->hdr._image_offset, KSEEK_SET);
     
     /* allocate attribute arrays */
     if (!malloc_attribute_arrays(subh->frame_data.footer_flags,
				  arf->hdr._num_frames,
				  &footer_attributes))
     {
       kerror("data services", "arf_output", 
	      "error allocating footer attribute arrays");
       return (FALSE);
     }
     
     /* get attributes */
     get_arf_footer_attributes(object, &footer_attributes, 
			       subh->frame_data.footer_flags);
     
     for (frame=0; frame<resources->arf->hdr._num_frames; frame++)
     {
       /* skip past the frame */
       kfseek(file, (int)resources->frame_size, KSEEK_CUR);
       if (subh->frame_data.footer_flags & ARF_FRAME_INFO_FLAG)
       {
	 resources->frame_footer->info.x = 
	   footer_attributes.frame_info_x[frame] ;
	 resources->frame_footer->info.y = 
	   footer_attributes.frame_info_y[frame] ;
	 resources->frame_footer->info.azimuth = 
	   footer_attributes.frame_info_az[frame];
	 resources->frame_footer->info.elevation = 
	   footer_attributes.frame_info_el[frame] ;
	 resources->frame_footer->info.irig_time.yy =
	   footer_attributes.frame_info_irig_yy[frame];
	 resources->frame_footer->info.irig_time.dd =
	   footer_attributes.frame_info_irig_dd[frame];
	 resources->frame_footer->info.irig_time.hh =
	   footer_attributes.frame_info_irig_hh[frame];
	 resources->frame_footer->info.irig_time.mm = 
	   footer_attributes.frame_info_irig_mm[frame];
	 resources->frame_footer->info.irig_time.ss =
	   footer_attributes.frame_info_irig_ss[frame];
	 resources->frame_footer->info.irig_time.ms =
	   footer_attributes.frame_info_irig_ms[frame];
       }
       
       if (subh->frame_data.footer_flags & ARF_PLATFORM_ATTITUDE_FLAG)
       {
	 resources->frame_footer->attitude.vel_north =
	   footer_attributes.platform_attitude_vel_n[frame];
	 resources->frame_footer->attitude.vel_east =
	   footer_attributes.platform_attitude_vel_e[frame];
	 resources->frame_footer->attitude.vel_vertical =
	   footer_attributes.platform_attitude_vel_v[frame];
	 resources->frame_footer->attitude.acc_north =
	   footer_attributes.platform_attitude_acc_n[frame];
	 resources->frame_footer->attitude.acc_east =
	   footer_attributes.platform_attitude_acc_e[frame];
	 resources->frame_footer->attitude.acc_vertical =
	   footer_attributes.platform_attitude_acc_v[frame];
	 resources->frame_footer->attitude.latitude =
	   footer_attributes.platform_attitude_lat[frame];
	 resources->frame_footer->attitude.longitude =
	   footer_attributes.platform_attitude_long[frame];
	 resources->frame_footer->attitude.altitude =
	   footer_attributes.platform_attitude_alt[frame];
	 resources->frame_footer->attitude.elevation =
	   footer_attributes.platform_attitude_elev[frame];
	 resources->frame_footer->attitude.bank =
	   footer_attributes.platform_attitude_bank[frame];
	 resources->frame_footer->attitude.heading =
	   footer_attributes.platform_attitude_head[frame];
       }
       resources->frame_foot_size =
	 arf_put_footer(file, 
			resources->subh->frame_data.footer_flags, 
			resources->frame_footer);
     } /* frame loop */
     /* go back to where you were */
     kfseek(file, here, KSEEK_SET);
   } /* frame_data_size if */
   
   return (TRUE);
 }


/*

*/
/*-----------------------------------------------------------
|
|  Routine Name: arf_destroy - Frees an arf image
|
|       Purpose: This function is used to destroy an arf image.  
|
|         Input: data - initial sample of data
|                num  - number of bytes in the sample
|
|        Output: Returns: returns TRUE or FALSE if an error occurs
|
|    Written By: Tim Williams (from Mark Young's xvimage_destroy)
|          Date: 18 May 1994
| Modifications:
|
------------------------------------------------------------*/
static int
arf_destroy(kobject object)
{
   ResourceStruct *resources;



   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   /*
    * block and wait until I can have the descriptor to myself
    */

   kfclose(resources->file);

   /* free structures */
   if (resources->arf != NULL)
     kfree(resources->arf);
   
   if (resources->subh != NULL)
     kfree(resources->subh);

   kfree(resources);

   return(TRUE);
}
/*

*/

/*-----------------------------------------------------------
|
|  Routine Name: arf_order
|
|       Purpose: sets order array according to MAP segment
|
|         Input: segment - KDMS_SEGMENT_MAP only valid input segment 
|
|        Output: order
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Tim Williams (from Jeremy Worley's xvimage_order)
|          Date: 18 May 1994
| Modifications:
|
------------------------------------------------------------*/
static int 
arf_order(char *seg, int *ord)
{



   if (kstring_to_token(seg) == kstring_to_token(KDMS_SEGMENT_MAP)) {
      ord[0] = KMAP_HEIGHT;
      ord[1] = KMAP_WIDTH;
      ord[2] = KMAP_ELEMENTS;
      ord[3] = KMAP_DEPTH;
      ord[4] = KMAP_TIME;
   }

   return (TRUE);
}

/*

*/

/*-----------------------------------------------------------
|
|  Routine Name: arf_architecture
|
|       Purpose: Return the architecture that the data was
|                generated on in khoros-speak.
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Tim Williams (from Jeremy Worley's xvimage_architecture)
|          Date: 19 May 1994
| Modifications:
|
------------------------------------------------------------*/

static int
arf_architecture(kobject object)
{
   ResourceStruct *resources;



   /*
    * get the resource structure
    */
   resources = (ResourceStruct *)_kdms_glue_get_resources(object);

   if (resources == NULL)
      return (KMACH_LOCAL);
   else
     /* ARF uses XDR, so set architecture to KMACH_SPARC */
      return (KMACH_SPARC);
}

/*-----------------------------------------------------------
|
|  Routine Name: arf_read
|
|       Purpose: Reads segment data from ARF formatted object
|
|         Input: object - kobject to read segment data from
|                seg    - the data segment to read, normally KDMS_SEGMENT_VALUE
|                data   - pointer to current data segment, or NULL
|                begin  - array of starting points for data segment
|                end    - array of end points for data segment
|
|
|        Output: data   - pointer to data segment
|
|       Returns: pointer to data segment, NULL on error
|
|    Written By: Tim Williams, Jeremy Worley
|          Date: 9 June 1994
| Modifications:
|
------------------------------------------------------------*/

static kaddr
  arf_read(kobject object, int seg, kaddr *data, int *begin, int *end)
{
  ResourceStruct *resources;
  kpresentation *pres;
  int offsets[5];
  
   if (ktoken_check(KDMS_SEGMENT_MAP) == seg)
     return (NULL); /* colormap processed in arf_input */

  if (ktoken_check(KDMS_SEGMENT_VALUE) != seg)
       return (NULL);


  /* get presentation attribute of segment */
  if (!_kdms_get_segment_info(object, seg, &pres) || 
      !pres)
    return (FALSE);
  resources = (ResourceStruct *) _kdms_glue_get_resources(object);
  
  offsets[0] = offsets[1] = offsets[2] = offsets[4] = 0;
  offsets[3] = resources->frame_foot_size;
  
  *data = _kdms_glue_read_segment_data(resources->file,
				      KMACH_SPARC,
				      kfread_generic, 
				      kfseek,
				      (int)resources->arf->hdr._image_offset,
				      pres->segment->datatype,
				      pres->segment->dimension,
				      pres->segment->size,
				      offsets, 
				      data,
				      begin,
				      end);
  return(*data);
}

/*

*/

/*-----------------------------------------------------------
  |
  |  Routine Name: arf_write
  |
  |       Purpose: Writes segment data to ARF formatted object
  |
  |         Input: object - kobject to write segment data to
  |                seg    - the data segment to write, 
  |                         normally KDMS_SEGMENT_VALUE
  |                data   - pointer to current data segment
  |                begin  - array of starting points for data segment
  |                end    - array of end points for data segment
  |
  |
  |        Output: data   - pointer to data segment
  |
  |       Returns: TRUE (1) on success, FALSE (0) otherwise
  |
  |    Written By: Tim Williams, Jeremy Worley
  |          Date: 13 June 1994
  | Modifications:
  |
  ------------------------------------------------------------*/

static int
  arf_write(kobject object, int seg, kaddr data, int *begin, int *end)
{
   ResourceStruct *resources;
   kpresentation *pres;
   int offsets[5];

   if (ktoken_check(KDMS_SEGMENT_MAP) == seg)
     return (TRUE); /* colormap processed in arf_output */

   if (ktoken_check(KDMS_SEGMENT_VALUE) != seg)
       return(FALSE);

  /* if data's NULL, why am I being called? */
  if (data == NULL)
  {
    kerror("data services", "arf_write", "NULL data pointer");
    return (FALSE);
  }

   if (!_kdms_get_segment_info(object, seg, &pres) ||
       !pres)
      return (FALSE);
   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   offsets[0] = offsets[1] = offsets[2] = offsets[4] = 0;
   offsets[3] = resources->frame_foot_size;

   return (_kdms_glue_write_segment_data(resources->file,
					 KMACH_SPARC,
					 kfwrite_generic,
					 kfseek,
					 (int)resources->arf->hdr._image_offset,
					 pres->segment->datatype,
					 pres->segment->dimension,
					 pres->segment->size,
					 offsets, 
					 data,
					 begin,
					 end));
   
 }

/*-----------------------------------------------------------
|
|  Routine Name: define_arf_*_attributes
|
|       Purpose: The following functions are called to predefine
|                all application specific attributes that
|                are specifically needed by arf glue.
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Tim Williams (modeled after Jeremy Worley's
|                xvimage_define_attributes)
|          Date: 25 July 1994
| Modifications:
|
------------------------------------------------------------*/

static int define_arf_colormap_attributes(void)
{
  static int defed=FALSE;

  if (defed)
    return(TRUE);

   if (!kdms_query_attribute_definition(KDMS_OBJECT, _INTERNAL_COLORSPACE))
      kdms_define_attribute(KDMS_OBJECT, _INTERNAL_COLORSPACE, 1, 1, KINT,
                            TRUE, TRUE, KCOLOR_COLORSPACE_DEFAULT);

  defed = TRUE;
  return(TRUE);
}
  
static int define_arf_comment_attribute(void)
{
  static int defed=FALSE;

  if (defed)
    return (TRUE);

  if (!kdms_query_attribute_definition(KDMS_OBJECT, KDMS_COMMENT))
    kdms_define_attribute(KDMS_OBJECT, KDMS_COMMENT, 1, 1, KSTRING,
			  TRUE, TRUE,
                            NULL);
  defed = TRUE;
  return (TRUE);
}


/*-----------------------------------------------------------
  |
  |  Routine Name: malloc_attribute_arrays
  |
  |       Purpose: allocates space for attribute arrays
  |                needed for ARF footers
  |                
  |
  |         Input: footer_flags - footer flags
  |                num_frames - number of frames in ARF image
  |                footer_attributes - ARF_FOOTER_ATTRIBUTES
  |                                    containing arrays of arf footer
  |                                    attributes
  |
  |	 Output: pointers to attribute arrays
  |
  |       Returns: TRUE (1), FALSE (0) on failure
  |
  |    Written By: Tim Williams 
  |          Date:  1 July 1994
  | Modifications:
  |                18 July 1994 - Tim Williams
  |                               created ARF_FOOTER_ATTRIBUTES structure
  |                               to hold footer attribute arrays
  |
  ------------------------------------------------------------*/

int malloc_attribute_arrays(unsigned long footer_flags, 
			    unsigned long num_frames,
			    ARF_FOOTER_ATTRIBUTES *footer_attributes)
{
  if (footer_flags & ARF_FRAME_INFO_FLAG)
  {
    footer_attributes->frame_info_x = 
      (unsigned long *)kmalloc(num_frames * 
			       sizeof(unsigned long));
    if (footer_attributes->frame_info_x == NULL) return (FALSE);
    
    footer_attributes->frame_info_y = 
      (unsigned long *)kmalloc(num_frames * 
			       sizeof(unsigned long));
    if (footer_attributes->frame_info_y == NULL) return (FALSE);
    
    footer_attributes->frame_info_az = 
      (float *)kmalloc(num_frames * 
		       sizeof(float));
    if (footer_attributes->frame_info_az == NULL) return (FALSE);
    
    footer_attributes->frame_info_el = 
      (float *)kmalloc(num_frames * 
		       sizeof(float));
    if (footer_attributes->frame_info_el == NULL) return (FALSE);
    
    footer_attributes->frame_info_irig_yy = 
      (unsigned short *)kmalloc(num_frames * 
				sizeof(unsigned short));
    if (footer_attributes->frame_info_irig_yy == NULL) return (FALSE);
    
    footer_attributes->frame_info_irig_dd = 
      (unsigned short *)kmalloc(num_frames * 
				sizeof(unsigned short));
    if (footer_attributes->frame_info_irig_dd == NULL) return (FALSE);
    
    footer_attributes->frame_info_irig_hh = 
      (unsigned short *)kmalloc(num_frames * 
				sizeof(unsigned short));
    if (footer_attributes->frame_info_irig_hh == NULL) return (FALSE);
    
    footer_attributes->frame_info_irig_mm = 
      (unsigned short *)kmalloc(num_frames * 
				sizeof(unsigned short));
    if (footer_attributes->frame_info_irig_mm == NULL) return (FALSE);
    
    footer_attributes->frame_info_irig_ss = 
      (unsigned short *)kmalloc(num_frames * 
				sizeof(unsigned short));
    if (footer_attributes->frame_info_irig_ss == NULL) return (FALSE);
    
    footer_attributes->frame_info_irig_ms = 
      (unsigned short *)kmalloc(num_frames * 
				sizeof(unsigned short));
    if (footer_attributes->frame_info_irig_ms == NULL) return (FALSE);
  } /* ARF_FRAME_INFO_FLAG */
  
  if (footer_flags & ARF_PLATFORM_ATTITUDE_FLAG)
  {
    footer_attributes->platform_attitude_vel_n = 
      (double *)kmalloc(num_frames * 
			sizeof(double));
    if (footer_attributes->platform_attitude_vel_n == NULL) return (FALSE);
    
    footer_attributes->platform_attitude_vel_e = 
      (double *)kmalloc(num_frames * 
			sizeof(double));
    if (footer_attributes->platform_attitude_vel_e == NULL) return (FALSE);
    
    footer_attributes->platform_attitude_vel_v = 
      (double *)kmalloc(num_frames * 
			sizeof(double));
    if (footer_attributes->platform_attitude_vel_v == NULL) return (FALSE);
    
    footer_attributes->platform_attitude_acc_n = 
      (double *)kmalloc(num_frames * 
			sizeof(double));
    if (footer_attributes->platform_attitude_acc_n == NULL) return (FALSE);
    
    footer_attributes->platform_attitude_acc_e = 
      (double *)kmalloc(num_frames * 
			sizeof(double));
    if (footer_attributes->platform_attitude_acc_e == NULL) return (FALSE);
    
    footer_attributes->platform_attitude_acc_v = 
      (double *)kmalloc(num_frames * 
			sizeof(double));
    if (footer_attributes->platform_attitude_acc_v == NULL) return (FALSE);
    
    footer_attributes->platform_attitude_lat = 
      (double *)kmalloc(num_frames * 
			sizeof(double));
    if (footer_attributes->platform_attitude_lat == NULL) return (FALSE);
    
    footer_attributes->platform_attitude_long = 
      (double *)kmalloc(num_frames * 
			sizeof(double));
    if (footer_attributes->platform_attitude_long == NULL) return (FALSE);
    
    footer_attributes->platform_attitude_alt = 
      (double *)kmalloc(num_frames * 
			sizeof(double));
    if (footer_attributes->platform_attitude_alt == NULL) return (FALSE);
    
    footer_attributes->platform_attitude_elev = 
      (double *)kmalloc(num_frames * 
			sizeof(double));
    if (footer_attributes->platform_attitude_elev == NULL) return (FALSE);
    
    footer_attributes->platform_attitude_bank = 
      (double *)kmalloc(num_frames * 
			sizeof(double));
    if (footer_attributes->platform_attitude_bank == NULL) return (FALSE);
    
    footer_attributes->platform_attitude_head = 
      (double *)kmalloc(num_frames * 
			sizeof(double));
    if (footer_attributes->platform_attitude_head == NULL) return (FALSE);
  } /* ARF_PLATFORM_ATTITUDE_FLAG */
  
  return (TRUE);
}

/*-----------------------------------------------------------
  |
  |  Routine Name: create_arf_footer_attributes
  |
  |       Purpose: creates ARF footer attributes
  |
  |         Input: object - the ARF object containing the footers
  |                footer_flags - footer flags
  |                num_frames - number of frames in ARF image
  |
  |	 Output: None
  |
  |       Returns: TRUE (1), FALSE (0) on failure
  |
  |    Written By: Tim Williams 
  |          Date:  1 July 1994
  | Modifications:
  |
  ------------------------------------------------------------*/
int create_arf_footer_attributes(kobject object, unsigned long footer_flags, 
				 unsigned long num_frames)
{
  if (footer_flags && ARF_FRAME_INFO_FLAG)
  {
    if (!kdms_create_attribute(object, NULL, "ARF_FRAME_INFO_x",
			       1, (int)num_frames, KULONG,
			       TRUE, TRUE)) 
      return (FALSE);
    if (!kdms_create_attribute(object, NULL, "ARF_FRAME_INFO_y",
			       1, (int)num_frames, KULONG,
			       TRUE, TRUE)) 
      return (FALSE);
    if (!kdms_create_attribute(object, NULL, "ARF_FRAME_INFO_az",
			       1, (int)num_frames, KFLOAT,
			       TRUE, TRUE)) 
      return (FALSE);
    if (!kdms_create_attribute(object, NULL, "ARF_FRAME_INFO_el",
			       1, (int)num_frames, KFLOAT,
			       TRUE, TRUE))  
      return (FALSE);
    if (!kdms_create_attribute(object, NULL, "ARF_FRAME_INFO_irig_yy",
			       1, (int)num_frames, KUSHORT,
			       TRUE, TRUE)) 
      return (FALSE);
    if (!kdms_create_attribute(object, NULL, "ARF_FRAME_INFO_irig_dd",
			       1, (int)num_frames, KUSHORT,
			       TRUE, TRUE))
      return (FALSE);
    if (!kdms_create_attribute(object, NULL, "ARF_FRAME_INFO_irig_hh",
			       1, (int)num_frames, KUSHORT,
			       TRUE, TRUE)) 
      return (FALSE);
    if (!kdms_create_attribute(object, NULL, "ARF_FRAME_INFO_irig_mm",
			       1, (int)num_frames, KUSHORT,
			       TRUE, TRUE)) 
      return (FALSE);
    if (!kdms_create_attribute(object, NULL, "ARF_FRAME_INFO_irig_ss",
			       1, (int)num_frames, KUSHORT,
			       TRUE, TRUE)) 
      return (FALSE);
    if (!kdms_create_attribute(object, NULL, "ARF_FRAME_INFO_irig_ms",
			       1, (int)num_frames, KUSHORT,
			       TRUE, TRUE)) 
      return (FALSE);
  } /* ARF_FRAME_INFO_FLAG */
  
  if (footer_flags && ARF_PLATFORM_ATTITUDE_FLAG)
  {
    if (!kdms_create_attribute(object, NULL, "ARF_PLATFORM_ATTITUDE_vel_n",
			       1, (int)num_frames, KDOUBLE,
			       TRUE, TRUE)) 
      return (FALSE);
    if (!kdms_create_attribute(object, NULL, "ARF_PLATFORM_ATTITUDE_vel_e",
			       1, (int)num_frames, KDOUBLE,
			       TRUE, TRUE))
      return (FALSE);
    if (!kdms_create_attribute(object, NULL, "ARF_PLATFORM_ATTITUDE_vel_v",
			       1, (int)num_frames, KDOUBLE,
			       TRUE, TRUE))
      return (FALSE);
    if (!kdms_create_attribute(object, NULL, "ARF_PLATFORM_ATTITUDE_acc_n",
			       1, (int)num_frames, KDOUBLE,
			       TRUE, TRUE)) return (FALSE);
    
    if (!kdms_create_attribute(object, NULL, "ARF_PLATFORM_ATTITUDE_acc_e",
			       1, (int)num_frames, KDOUBLE,
			       TRUE, TRUE)) 
      return (FALSE);
    if (!kdms_create_attribute(object, NULL, "ARF_PLATFORM_ATTITUDE_acc_v",
			       1, (int)num_frames, KDOUBLE,
			       TRUE, TRUE)) 
      return (FALSE);
    if (!kdms_create_attribute(object, NULL, "ARF_PLATFORM_ATTITUDE_lat",
			       1, (int)num_frames, KDOUBLE,
			       TRUE, TRUE)) 
      return (FALSE);
    if (!kdms_create_attribute(object, NULL, "ARF_PLATFORM_ATTITUDE_long",
			       1, (int)num_frames, KDOUBLE,
			       TRUE, TRUE)) 
      return (FALSE);
    if (!kdms_create_attribute(object, NULL, "ARF_PLATFORM_ATTITUDE_alt",
			       1, (int)num_frames, KDOUBLE,
			       TRUE, TRUE))
      return (FALSE);
    if (!kdms_create_attribute(object, NULL, "ARF_PLATFORM_ATTITUDE_elev",
			       1, (int)num_frames, KDOUBLE,
			       TRUE, TRUE)) 
      return (FALSE);
    if (!kdms_create_attribute(object, NULL, "ARF_PLATFORM_ATTITUDE_bank",
			       1, (int)num_frames, KDOUBLE,
			       TRUE, TRUE))
      return (FALSE);
    if (!kdms_create_attribute(object, NULL, "ARF_PLATFORM_ATTITUDE_head",
			       1, (int)num_frames, KDOUBLE,
			       TRUE, TRUE)) 
      return (FALSE);
  } /* ARF_PLATFORM_ATTITUDE_FLAG */
  return (TRUE);
}

/*-----------------------------------------------------------
  |
  |  Routine Name: set_arf_footer_attributes
  |
  |       Purpose: sets ARF footer attributes
  |
  |         Input: object - the ARF object containing the footers
  |                footer_attributes - ARF_FOOTER_ATTRIBUTES structure
  |                                    containing footer attribute arrays
  |                footer_flags - footer flags
  |
  |	 Output: None
  |
  |       Returns: TRUE (1), FALSE (0) on failure
  |
  |    Written By: Tim Williams 
  |          Date:  18 July 1994
  | Modifications:
  |
  ------------------------------------------------------------*/
int set_arf_footer_attributes(kobject object,
			      ARF_FOOTER_ATTRIBUTES *footer_attributes,
			      unsigned long footer_flags)
{
  if (footer_flags & ARF_FRAME_INFO_FLAG)
  {
    if (!kdms_set_attribute(object, NULL, 
			    "ARF_FRAME_INFO_x", 
			    footer_attributes->frame_info_x)) 
      return (FALSE);
    if (!kdms_set_attribute(object, NULL, 
			    "ARF_FRAME_INFO_y", 
			    footer_attributes->frame_info_y)) 
      return (FALSE);
    if (!kdms_set_attribute(object, NULL, 
			    "ARF_FRAME_INFO_el", 
			    footer_attributes->frame_info_el)) 
      return (FALSE);
    if (!kdms_set_attribute(object, NULL, 
			    "ARF_FRAME_INFO_az", 
			    footer_attributes->frame_info_az)) 
      return (FALSE);
    if (!kdms_set_attribute(object, NULL, 
			    "ARF_FRAME_INFO_irig_yy", 
			    footer_attributes->frame_info_irig_yy)) 
      return (FALSE);
    if (!kdms_set_attribute(object, NULL, 
			    "ARF_FRAME_INFO_irig_dd", 
			    footer_attributes->frame_info_irig_dd)) 
      return (FALSE);
    if (!kdms_set_attribute(object, NULL, 
			    "ARF_FRAME_INFO_irig_hh", 
			    footer_attributes->frame_info_irig_hh)) 
      return (FALSE);
    if (!kdms_set_attribute(object, NULL, 
			    "ARF_FRAME_INFO_irig_mm",
			    footer_attributes->frame_info_irig_mm))
      return (FALSE);
    if (!kdms_set_attribute(object, NULL, 
			    "ARF_FRAME_INFO_irig_ss",
			    footer_attributes->frame_info_irig_ss))
      return (FALSE);
    if (!kdms_set_attribute(object, NULL, 
			    "ARF_FRAME_INFO_irig_ms",
			    footer_attributes->frame_info_irig_ms))
      return (FALSE);
  }
  if (footer_flags & ARF_PLATFORM_ATTITUDE_FLAG)
  {
    if (!kdms_set_attribute(object, NULL,
			    "ARF_PLATFORM_ATTITUDE_vel_n",
			    footer_attributes->platform_attitude_vel_n))
      return (FALSE);
    if (!kdms_set_attribute(object, NULL,
			    "ARF_PLATFORM_ATTITUDE_vel_e",
			    footer_attributes->platform_attitude_vel_e))
      return (FALSE);
    if (!kdms_set_attribute(object, NULL,
			    "ARF_PLATFORM_ATTITUDE_vel_v",
			    footer_attributes->platform_attitude_vel_v))
      return (FALSE);
    if (!kdms_set_attribute(object, NULL,
			    "ARF_PLATFORM_ATTITUDE_acc_n",
			    footer_attributes->platform_attitude_acc_n))
      return (FALSE);
    if (!kdms_set_attribute(object, NULL,
			    "ARF_PLATFORM_ATTITUDE_acc_e",
			    footer_attributes->platform_attitude_acc_e))
      return (FALSE);
    if (!kdms_set_attribute(object, NULL,
			    "ARF_PLATFORM_ATTITUDE_acc_v",
			    footer_attributes->platform_attitude_acc_v))
      return (FALSE);
    if (!kdms_set_attribute(object, NULL, 
			    "ARF_PLATFORM_ATTITUDE_lat",
			    footer_attributes->platform_attitude_lat))
      return (FALSE);
    if (!kdms_set_attribute(object, NULL, 
			    "ARF_PLATFORM_ATTITUDE_long",
			    footer_attributes->platform_attitude_long))
      return (FALSE);
    if (!kdms_set_attribute(object, NULL, 
			    "ARF_PLATFORM_ATTITUDE_alt",
			    footer_attributes->platform_attitude_alt))
      return (FALSE);
    if (!kdms_set_attribute(object, NULL, 
			    "ARF_PLATFORM_ATTITUDE_elev",
			    footer_attributes->platform_attitude_elev))
      return (FALSE);
    if (!kdms_set_attribute(object, NULL, 
			    "ARF_PLATFORM_ATTITUDE_bank",
			    footer_attributes->platform_attitude_bank))
      return (FALSE);
    if (!kdms_set_attribute(object, NULL, 
			    "ARF_PLATFORM_ATTITUDE_head",
			    footer_attributes->platform_attitude_head))
      return (FALSE);
  }
  return (TRUE);
}
/*-----------------------------------------------------------
  |
  |  Routine Name: get_arf_footer_attributes
  |
  |       Purpose: gets ARF footer attributes
  |
  |         Input: object - the ARF object containing the footers
  |                footer_attributes - ARF_FOOTER_ATTRIBUTES structure
  |                                    containing footer attribute arrays
  |                footer_flags - footer flags
  |
  |	 Output: None
  |
  |       Returns: TRUE (1), FALSE (0) on failure
  |
  |    Written By: Tim Williams 
  |          Date:  18 July 1994
  | Modifications:
  |
  ------------------------------------------------------------*/
int get_arf_footer_attributes(kobject object,
			      ARF_FOOTER_ATTRIBUTES *footer_attributes,
			      unsigned long footer_flags)
{
  if (footer_flags & ARF_FRAME_INFO_FLAG)
  {
    if (!kdms_get_attribute(object, NULL, 
			    "ARF_FRAME_INFO_x", 
			    &footer_attributes->frame_info_x)) return (FALSE);
    if (!kdms_get_attribute(object, NULL, 
			    "ARF_FRAME_INFO_y", 
			    &footer_attributes->frame_info_y)) return (FALSE);
    if (!kdms_get_attribute(object, NULL, 
			    "ARF_FRAME_INFO_el", 
			    &footer_attributes->frame_info_el)) return (FALSE);
    if (!kdms_get_attribute(object, NULL, 
			    "ARF_FRAME_INFO_az", 
			    &footer_attributes->frame_info_az)) return (FALSE);
    if (!kdms_get_attribute(object, NULL, 
			    "ARF_FRAME_INFO_irig_yy",
			    &footer_attributes->frame_info_irig_yy)) 
      return (FALSE);
    if (!kdms_get_attribute(object, NULL, 
			    "ARF_FRAME_INFO_irig_dd",
			    &footer_attributes->frame_info_irig_dd)) 
      return (FALSE);
    if (!kdms_get_attribute(object, NULL, 
			    "ARF_FRAME_INFO_irig_hh",
			    &footer_attributes->frame_info_irig_hh))
      return (FALSE);
    if (!kdms_get_attribute(object, NULL, 
			    "ARF_FRAME_INFO_irig_mm",
			    &footer_attributes->frame_info_irig_mm)) 
      return (FALSE);
    if (!kdms_get_attribute(object, NULL, 
			    "ARF_FRAME_INFO_irig_ss",
			    &footer_attributes->frame_info_irig_ss)) 
      return (FALSE);
    if (!kdms_get_attribute(object, NULL, 
			    "ARF_FRAME_INFO_irig_ms",
			    &footer_attributes->frame_info_irig_ms)) 
      return (FALSE);
  }
  if (footer_flags & ARF_PLATFORM_ATTITUDE_FLAG)
  {
    if (!kdms_get_attribute(object, NULL,
			    "ARF_PLATFORM_ATTITUDE_vel_n",
			    &footer_attributes->platform_attitude_vel_n))
      return (FALSE);
    if (!kdms_get_attribute(object, NULL,
			    "ARF_PLATFORM_ATTITUDE_vel_e",
			    &footer_attributes->platform_attitude_vel_e))
      return (FALSE);
    if (!kdms_get_attribute(object, NULL,
			    "ARF_PLATFORM_ATTITUDE_vel_v",
			    &footer_attributes->platform_attitude_vel_v))
      return (FALSE);
    if (!kdms_get_attribute(object, NULL,
			    "ARF_PLATFORM_ATTITUDE_acc_n",
			    &footer_attributes->platform_attitude_acc_n))
      return (FALSE);
    if (!kdms_get_attribute(object, NULL,
			    "ARF_PLATFORM_ATTITUDE_acc_e",
			    &footer_attributes->platform_attitude_acc_e))
      return (FALSE);
    if (!kdms_get_attribute(object, NULL,
			    "ARF_PLATFORM_ATTITUDE_acc_v",
			    &footer_attributes->platform_attitude_acc_v))
      return (FALSE);
    if (!kdms_get_attribute(object, NULL, 
			    "ARF_PLATFORM_ATTITUDE_lat",
			    &footer_attributes->platform_attitude_lat))
      return (FALSE);
    if (!kdms_get_attribute(object, NULL, 
			    "ARF_PLATFORM_ATTITUDE_long",
			    &footer_attributes->platform_attitude_long))
      return (FALSE);
    if (!kdms_get_attribute(object, NULL, 
			    "ARF_PLATFORM_ATTITUDE_alt",
			    &footer_attributes->platform_attitude_alt))
      return (FALSE);
    if (!kdms_get_attribute(object, NULL, 
			    "ARF_PLATFORM_ATTITUDE_elev",
			    &footer_attributes->platform_attitude_elev))
      return (FALSE);
    if (!kdms_get_attribute(object, NULL, 
			    "ARF_PLATFORM_ATTITUDE_bank",
			    &footer_attributes->platform_attitude_bank))
      return (FALSE);
    if (!kdms_get_attribute(object, NULL, 
			    "ARF_PLATFORM_ATTITUDE_head",
			    &footer_attributes->platform_attitude_head))
      return (FALSE);
  }
  
  return (TRUE);
}
/*-----------------------------------------------------------
  |
  |  Routine Name: create_arf_info_attributes
  |
  |       Purpose: creates ARF_INFO subheader attributes
  |
  |         Input: object - the ARF object containing the footers
  |
  |	 Output: None
  |
  |       Returns: TRUE (1), FALSE (0) on failure
  |
  |    Written By: Tim Williams 
  |          Date:  18 July 1994
  | Modifications:
  |
  ------------------------------------------------------------*/
/*ARGSUSED*/
int create_arf_info_attributes(kobject object, ARF_INFO *info)
{
  if (!kdms_create_attribute(object, NULL, "ARF_INFO_image_source", 1, 1,
			     KULONG, TRUE, TRUE)) 
    return (FALSE);
  if (!kdms_create_attribute(object, NULL, "ARF_INFO_start_x", 1, 1,
			     KULONG, TRUE, TRUE)) 
    return (FALSE);
  if (!kdms_create_attribute(object, NULL, "ARF_INFO_start_y", 1, 1,
			     KULONG, TRUE, TRUE)) 
    return (FALSE);
  if (!kdms_create_attribute(object, NULL, "ARF_INFO_num_avg", 1, 1,
			     KULONG, TRUE, TRUE)) 
    return (FALSE);
  if (!kdms_create_attribute(object, NULL, "ARF_INFO_capture_rate", 1, 1,
			     KULONG, TRUE, TRUE)) 
    return (FALSE);
  if (!kdms_create_attribute(object, NULL, "ARF_INFO_capture_time_yy", 1, 1,
			     KUSHORT, TRUE, TRUE)) 
    return (FALSE);
  if (!kdms_create_attribute(object, NULL, "ARF_INFO_capture_time_dd", 1, 1,
			     KUSHORT, TRUE, TRUE))
    return (FALSE);
  if (!kdms_create_attribute(object, NULL, "ARF_INFO_capture_time_hh", 1, 1,
			     KUSHORT, TRUE, TRUE)) 
    return (FALSE);
  if (!kdms_create_attribute(object, NULL, "ARF_INFO_capture_time_mm", 1, 1,
			     KUSHORT, TRUE, TRUE)) 
    return (FALSE);
  if (!kdms_create_attribute(object, NULL, "ARF_INFO_capture_time_ss", 1, 1,
			     KUSHORT, TRUE, TRUE)) return (FALSE);
  if (!kdms_create_attribute(object, NULL, "ARF_INFO_capture_time_ms", 1, 1,
			     KUSHORT, TRUE, TRUE)) return (FALSE);
  if (!kdms_create_attribute(object, NULL, "ARF_INFO_capture_loc", 1, 1,
			     KSTRING, TRUE, TRUE)) 
    return (FALSE);
  if (!kdms_create_attribute(object, NULL, "ARF_INFO_sensor_name", 1, 1,
			     KSTRING, TRUE, TRUE)) 
    return (FALSE);
  if (!kdms_create_attribute(object, NULL, "ARF_INFO_digitizer", 1, 1,
			     KSTRING, TRUE, TRUE)) 
    return (FALSE);
  if (!kdms_create_attribute(object, NULL, "ARF_INFO_sensor_hor_fov", 1, 1,
			     KFLOAT, TRUE, TRUE)) 
    return (FALSE);
  if (!kdms_create_attribute(object, NULL, "ARF_INFO_sensor_ver_fov", 1, 1,
			     KFLOAT, TRUE, TRUE)) 
    return (FALSE);
  if (!kdms_create_attribute(object, NULL, "ARF_INFO_samples_per_dwell", 1, 1,
			     KULONG, TRUE, TRUE)) 
    return (FALSE);
  
  return (TRUE);
}

/*-----------------------------------------------------------
  |
  |  Routine Name: set_arf_info_attributes
  |
  |       Purpose: sets ARF_INFO subheader attributes
  |
  |         Input: object - the ARF object containing the footers
  |                info   - ARF_INFO subheader
  |
  |	 Output: None
  |
  |       Returns: TRUE (1), FALSE (0) on failure
  |
  |    Written By: Tim Williams 
  |          Date:  18 July 1994
  | Modifications:
  |
  ------------------------------------------------------------*/

int set_arf_info_attributes(kobject object, ARF_INFO *info)
{
  if (!kdms_set_attribute(object, NULL, 
			  "ARF_INFO_image_source", info->image_source))
    return (FALSE);
  if (!kdms_set_attribute(object, NULL, 
			  "ARF_INFO_start_x", info->start_x)) 
    return (FALSE);
  if (!kdms_set_attribute(object, NULL, 
			  "ARF_INFO_start_y", info->start_y))
    return (FALSE);
  if (!kdms_set_attribute(object, NULL, 
			  "ARF_INFO_num_avg", info->num_avg)) 
    return (FALSE);
  if (!kdms_set_attribute(object, NULL, 
			  "ARF_INFO_capture_rate", info->capture_rate))
    return (FALSE);
  if (!kdms_set_attribute(object, NULL, 
			  "ARF_INFO_capture_time_yy", 
			  info->capture_time.yy)) 
    return (FALSE);
  if (!kdms_set_attribute(object, NULL, 
			  "ARF_INFO_capture_time_dd", 
			  info->capture_time.dd)) 
    return (FALSE);
  if (!kdms_set_attribute(object, NULL, 
			  "ARF_INFO_capture_time_hh", 
			  info->capture_time.hh)) 
    return (FALSE);
  if (!kdms_set_attribute(object, NULL, 
			  "ARF_INFO_capture_time_mm", 
			  info->capture_time.mm)) 
    return (FALSE);
  if (!kdms_set_attribute(object, NULL, 
			  "ARF_INFO_capture_time_ss", 
			  info->capture_time.ss)) 
    return (FALSE);
  if (!kdms_set_attribute(object, NULL, 
			  "ARF_INFO_capture_time_ms", 
			  info->capture_time.ms)) 
    return (FALSE);
  if (!kdms_set_attribute(object, NULL, 
			  "ARF_INFO_capture_loc",
			  info->capture_loc))
    return (FALSE);
  if (!kdms_set_attribute(object, NULL, 
			  "ARF_INFO_sensor_name",
			  info->sensor_name)) 
    return (FALSE);
  if (!kdms_set_attribute(object, NULL, 
			  "ARF_INFO_digitizer", 
			  info->digitizer)) 
    return (FALSE);
  if (!kdms_set_attribute(object, NULL, 
			  "ARF_INFO_sensor_hor_fov", 
			  info->sensor_hor_fov)) 
    return (FALSE);
  if (!kdms_set_attribute(object, NULL, 
			  "ARF_INFO_sensor_ver_fov", 
			  info->sensor_ver_fov))
    return (FALSE);
  if (!kdms_set_attribute(object, NULL, 
			  "ARF_INFO_samples_per_dwell", 
			  info->samples_per_dwell)) 
    return (FALSE);
  
  return (TRUE);
}
/*-----------------------------------------------------------
  |
  |  Routine Name: get_arf_info_attributes
  |
  |       Purpose: gets ARF_INFO subheader attributes
  |
  |         Input: object - the ARF object containing the footers
  |                info   - ARF_INFO subheader
  |
  |	 Output: None
  |
  |       Returns: size in bytes of ARF_INFO subheader 
  |                (with XDR padding on strings)
  |                FALSE (0) on failure
  |
  |    Written By: Tim Williams 
  |          Date:  18 July 1994
  | Modifications:
  |
  ------------------------------------------------------------*/

int get_arf_info_attributes(kobject object, ARF_INFO *info)
{
  unsigned long tmpval;
  int subh_length;
  char *string;
  
  if (!kdms_get_attribute(object, NULL, 
			  "ARF_INFO_image_source", &info->image_source))
    return (FALSE);
  if (!kdms_get_attribute(object, NULL, 
			  "ARF_INFO_start_x", &info->start_x)) 
    return (FALSE);
  if (!kdms_get_attribute(object, NULL, 
			  "ARF_INFO_start_y", &info->start_y))
    return (FALSE);
  if (!kdms_get_attribute(object, NULL, 
			  "ARF_INFO_num_avg", &info->num_avg)) 
    return (FALSE);
  if (!kdms_get_attribute(object, NULL, 
			  "ARF_INFO_capture_rate", &info->capture_rate))
    return (FALSE);
  if (!kdms_get_attribute(object, NULL, 
			  "ARF_INFO_capture_time_yy", 
			  &info->capture_time.yy)) 
    return (FALSE);
  if (!kdms_get_attribute(object, NULL, 
			  "ARF_INFO_capture_time_dd", 
			  &info->capture_time.dd)) 
    return (FALSE);
  if (!kdms_get_attribute(object, NULL, 
			  "ARF_INFO_capture_time_hh", 
			  &info->capture_time.hh)) 
    return (FALSE);
  if (!kdms_get_attribute(object, NULL, 
			  "ARF_INFO_capture_time_mm", 
			  &info->capture_time.mm)) 
    return (FALSE);
  if (!kdms_get_attribute(object, NULL, 
			  "ARF_INFO_capture_time_ss", 
			  &info->capture_time.ss)) 
    return (FALSE);
  if (!kdms_get_attribute(object, NULL, 
			  "ARF_INFO_capture_time_ms", 
			  &info->capture_time.ms)) 
    return (FALSE);
  if (!kdms_get_attribute(object, NULL, 
			  "ARF_INFO_capture_loc", &string))
    return (FALSE);
  else
    kstrcpy(info->capture_loc, string);
  if (!kdms_get_attribute(object, NULL, 
			  "ARF_INFO_sensor_name", &string))
    return (FALSE);
  else
    kstrcpy(info->sensor_name, string);
  if (!kdms_get_attribute(object, NULL, 
			  "ARF_INFO_digitizer", &string))
    return (FALSE);
  else
    kstrcpy(info->digitizer, string);
  if (!kdms_get_attribute(object, NULL, 
			  "ARF_INFO_sensor_hor_fov", 
			  &info->sensor_hor_fov)) 
    return (FALSE);
  if (!kdms_get_attribute(object, NULL, 
			  "ARF_INFO_sensor_ver_fov", 
			  &info->sensor_ver_fov))
    return (FALSE);
  if (!kdms_get_attribute(object, NULL, 
			  "ARF_INFO_samples_per_dwell", 
			  &info->samples_per_dwell)) 
    return (FALSE);
  
  /** update the subhsize ***/
  /* 12 longs + 2 floats + 3 string lengths (ints) */
  subh_length = 68;
  
  tmpval = kstrlen(info->capture_loc);
  if (tmpval%4)
    tmpval += (4 - (tmpval%4));
  subh_length += tmpval;
  
  tmpval = kstrlen(info->sensor_name);
  if (tmpval%4) tmpval += (4 - (tmpval%4));
  subh_length += tmpval;
  
  tmpval = kstrlen(info->digitizer);
  if (tmpval%4) tmpval += (4 - (tmpval%4));
  subh_length += tmpval;
  
  return (subh_length);
}


/*-----------------------------------------------------------
  |
  |  Routine Name: create_arf_ladar_param_attributes
  |
  |       Purpose: creates ARF_LADAR_PARAM subheader attributes
  |
  |       Input: object - the ARF object containing the footers
  |
  |	 Output: None
  |
  |       Returns: TRUE (1), FALSE (0) on failure
  |
  |    Written By: Tim Williams 
  |          Date:  18 July 1994
  | Modifications:
  |
  ------------------------------------------------------------*/

int create_arf_ladar_param_attributes(kobject object)
{
  if (!kdms_create_attribute(object, NULL, "ARF_LADAR_PARAM_fm_scale", 1, 1,
			     KFLOAT, TRUE, TRUE)) 
    return (FALSE);
  if (!kdms_create_attribute(object, NULL, 
			     "ARF_LADAR_PARAM_resolved_range_bias",
			     1, 1,
			     KFLOAT, TRUE, TRUE)) 
    return (FALSE);
  if (!kdms_create_attribute(object, NULL, "ARF_LADAR_PARAM_hor_ifov", 1, 1,
			     KFLOAT, TRUE, TRUE)) 
    return (FALSE);
  if (!kdms_create_attribute(object, NULL, "ARF_LADAR_PARAM_ver_ifov", 1, 1,
			     KFLOAT, TRUE, TRUE)) 
    return (FALSE);
  if (!kdms_create_attribute(object, NULL, "ARF_LADAR_PARAM_coarse_res", 1, 1,
			     KFLOAT, TRUE, TRUE)) 
    return (FALSE);
  if (!kdms_create_attribute(object, NULL, "ARF_LADAR_PARAM_fine_res", 1, 1,
			     KFLOAT, TRUE, TRUE)) 
    return (FALSE);
  
  return(TRUE);
}

/*-----------------------------------------------------------
  |
  |  Routine Name: set_arf_ladar_param_attributes
  |
  |       Purpose: sets ARF_LADAR_PARAM subheader attributes
  |
  |         Input: object      - the ARF object containing the footers
  |                ladar_param - ARF_LADAR_PARAM subheader
  |
  |	 Output: None
  |
  |       Returns: TRUE (1), FALSE (0) on failure
  |
  |    Written By: Tim Williams 
  |          Date:  18 July 1994
  | Modifications:
  |
  ------------------------------------------------------------*/

int set_arf_ladar_param_attributes(kobject object, 
				   ARF_LADAR_PARAM *ladar_param)
{
  if (!kdms_set_attribute(object, NULL, 
			  "ARF_LADAR_PARAM_fm_scale", 
			  ladar_param->fm_scale)) 
    return (FALSE);
  if (!kdms_set_attribute(object, NULL, 
			  "ARF_LADAR_PARAM_resolved_range_bias", 
			  ladar_param->resolved_range_bias)) 
    return (FALSE);
  if (!kdms_set_attribute(object, NULL, 
			  "ARF_LADAR_PARAM_hor_ifov", 
			  ladar_param->hor_ifov)) 
    return (FALSE);
  if (!kdms_set_attribute(object, NULL, 
			  "ARF_LADAR_PARAM_ver_ifov", 
			  ladar_param->ver_ifov)) 
    return (FALSE);
  if (!kdms_set_attribute(object, NULL, 
			  "ARF_LADAR_PARAM_coarse_res", 
			  ladar_param->coarse_res)) 
    return (FALSE);
  if (!kdms_set_attribute(object, NULL, 
			  "ARF_LADAR_PARAM_fine_res", 
			  ladar_param->fine_res)) 
    return (FALSE);
  
  return(TRUE);
}

/*-----------------------------------------------------------
  |
  |  Routine Name: get_arf_ladar_param_attributes
  |
  |       Purpose: gets ARF_LADAR_PARAM subheader attributes
  |
  |         Input: object      - the ARF object containing the footers
  |                ladar_param - ARF_LADAR_PARAM subheader
  |
  |	 Output: None
  |
  |       Returns: size in bytes of ARF_LADAR_PARAM subheader
  |                FALSE (0) on failure
  |
  |    Written By: Tim Williams 
  |          Date:  18 July 1994
  | Modifications:
  |
  ------------------------------------------------------------*/

int get_arf_ladar_param_attributes(kobject object, 
				   ARF_LADAR_PARAM *ladar_param)
{
  int subh_length;
  
  subh_length=0;
  if (kdms_get_attribute(object, NULL, 
			 "ARF_LADAR_PARAM_fm_scale", 
			  ladar_param->fm_scale)) 
    subh_length +=4;
  else
    return (FALSE);

  if (kdms_get_attribute(object, NULL, 
			 "ARF_LADAR_PARAM_resolved_range_bias", 
			 ladar_param->resolved_range_bias)) 
    subh_length += 4;
  else
    return (FALSE);
  
  if (kdms_get_attribute(object, NULL, 
			 "ARF_LADAR_PARAM_hor_ifov", 
			 ladar_param->hor_ifov)) 
    subh_length += 4;
  else
    return (FALSE);

  if (kdms_get_attribute(object, NULL, 
			 "ARF_LADAR_PARAM_ver_ifov", 
			 ladar_param->ver_ifov)) 
    subh_length += 4;
  else
    return (FALSE);
  
  if (kdms_set_attribute(object, NULL, 
			 "ARF_LADAR_PARAM_coarse_res", 
			 ladar_param->coarse_res)) 
    subh_length += 4;
  else
    return (FALSE);
  
  if (kdms_set_attribute(object, NULL, 
			 "ARF_LADAR_PARAM_fine_res", 
			 ladar_param->fine_res)) 
    subh_length += 4;
  else
    return (FALSE);
  
  return(subh_length);
}


#endif	/* KARF_DEF */

/* don`t add after the endif */
