 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for appending objects
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: kapu_append
   >>>> 	
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

static kobject  *src_object, dst_object=NULL;
static int	*w_size=NULL, *h_size=NULL, *d_size=NULL, 
		*t_size=NULL, *e_size=NULL, *d_type=NULL;
static kaddr	*data=NULL;

static void kapu_append_internal_cleanup PROTO((int));


/****************************************************************
* 
*  Routine Name: kapu_append - combine source data objects into single destination object
* 
*       Purpose: The kapu_append function combines all data objects
*		 provided in the input data object array.  The 
*		 objects are combined in the order in which they
*		 appear in the array.  The dimension flags (w_flag, 
*		 h_flag, d_flag, t_flag, e_flag) specify which 
*		 dimension the objects will be stacked along.  One 
*		 and only one of these flags can be true.  
*
*		 .IP "Value Data"
*		 Size & padding: the value (and mask) size of each 
*		 source object will be padded to the maximum of all 
*		 sizes - except in the stacking dimension.  The pad
*		 value used for each source object will be the value
*		 defined by that object's pad value attribute. The 
*		 highest data type of source data objects will 
*		 be propagated to the destination value data.
*		 Any source object that has no value data will be skipped.
*
*		 .IP "Map Data"
*		 When some or all of the source objects have map data, 
*		 the treatment of the maps, and how the data is 
*		 represented in the destination object, depends on the 
*		 mapping parameter passed in.  Possible mapping options 
*		 are KAPU_MAP_ALL_DATA, KAPU_CREATE_AUTOCOLOR_MAPS, and
*		 KAPU_USE_COMMON_MAP.  If no source objects have maps, 
*		 the mapping option is ignored.  If there are doubts about 
*		 which mapping option to use, the safest bet is to map 
*		 the data thru the maps.
*		 .RS
*		 .IP KAPU_MAP_ALL_DATA
*		 Map Data Thru Maps:
*		 All data will be mapped before the data objects are 
*		 combined.  The destination object will not have a map.
*
*		 .IP KAPU_CREATE_AUTOCOLOR_MAPS 
*		 Create a Map for Each Input:
*		 (All source objects may have maps.) For each source 
*		 object that does not have a map, one will be created 
*		 for it.  The the destination object color attributes 
*		 will be copied from the first source object that has 
*		 a map.  New map values will be assigned using the 
*		 KCOLOR_AUTOCOLOR attribute.  
*	
*		 .IP
*		 The map dimension corresponding to the stacking dimension 
*		 (d_flag, t_flag, or e_flag) will reflect the new size 
*		 of the value data in that dimension, and the maps from 
*		 each object will be appended accordingly.  The maximum 
*		 width and height of all maps will be propagated to the 
*		 destination, with zero-padding being applied where 
*		 necessary.  Padding in the other map dimensions may occur 
*		 as well.  For example, if objects with the following 
*		 map and value Depth sizes are being stacked along the 
*		 Depth dimension,
*		 !   obj1  DEP=3  MAP_DEP = 3
*		 !   obj2  DEP=4  MAP_DEP = 1
*		 !   obj2  DEP=2  MAP_DEP = 2
*		 the resulting value data will have a depth of 9, so the 
*		 map depth will be 9 too.  In this example, the second 
*		 object's map will be zero-padded to have a map depth of four.
*
*		 .IP
*		 If the objects are being combined along the width or 
*		 height dimensions, the KAPU_CREATE_AUTOCOLOR_MAPS 
*		 mapping option will fail.
*
*		 .IP KAPU_USE_COMMON_MAP 
*		 Use First Map Only:
*		 In this case, the map data and color attributes of the 
*		 first source object that has a map are directly transferred 
*		 to the destination object. The depth, time, and elements 
*		 dimensions of that map must be 1, otherwise this mapping 
*		 option will fail.  Note that by selecting this mapping 
*		 option, you are assuming that the value segments of all 
*		 objects being appended have valid indices into that map.  
*		 .RE
*	
*		 .IP
*		 The destination object map data type will be the highest 
*		 of the input object map data types.
*
*		 .IP "Mask Data"
*		 If any source object has mask data, then the destination 
*		 object will also have mask data.  The data will be resized
*		 in the same manner as value data is resized.  The highest 
*		 data type of source masks will be propagated to the 
*		 destination mask.
*
*		 .IP "Location and Time Data"
*		 In all cases, the time and location data and/or attributes 
*		 of the first object will transferred to the destination 
*		 object.  If location/time data is propagated, information 
*		 along the stacking dimension will not be changed, and 
*		 therefore, will most likely be corrupt.
*	
*		 .IP "Source & Destination Object Attributes"
*		 This function only sets the data type, mapping mode, and 
*		 the position attributes on references to the source objects.  
*		 Therefore, other attributes already set for the source 
*		 or destination object are used.  For example, if the 
*		 interpolate type is set to zero order (instead of the 
*		 default of pad), when the object size is changed, pixel 
*		 replication, not padding, will occur.  This gives the 
*		 calling routine more control over the append operation.
*
*         Input: object_array - array of kobjects 
*                num_objects - number of objects in object_array
*		 w_flag	- stack objects along width dimension
*		 h_flag	- stack objects along height dimension
*		 d_flag	- stack objects along depth dimension
*		 t_flag	- stack objects along time dimension
*		 e_flag	- stack objects along elements dimension
*		 mapping - if sources have maps, defines how dest map is built
*
*        Output: dst_obj - resulting destination object
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Donna Koechner
*          Date: Jan 20, 1995
*      Verified: 
*  Side Effects: Destination object (dst_obj) only: The size of 
*		 the value, mask and map data may be modified.  
*		 Map, location, and time data may be removed or added.
* Modifications: 9 Feb 1995 - Scott Wilson
*		              Set up to skip any objects with no value data
*                             segment.
****************************************************************/
/* -library_def */
int kapu_append(
	kobject *object_array,
	int	num_objects,
	int	w_flag,
	int	h_flag,
	int	d_flag,
	int	t_flag,
	int	e_flag,
	int	mapping,
	kobject dst_obj)
/* -library_def_end */

/* -library_code */
{
	char		*lib="kapputils", *rtn="kapu_append";
	int		i, j;
	int		has_mask=0, has_location=0, has_time=0;
	int		map_object=0;
	int		w_max=0, h_max=0, d_max=0, t_max=0, e_max=0;
	int		w_sum=0, h_sum=0, d_sum=0, t_sum=0, e_sum=0;
        int             w_rgn=0, h_rgn=0, d_rgn=0, t_rgn=0, e_rgn=0, num_rgns=0;
	int		w_off=0, h_off=0, d_off=0, t_off=0, e_off=0;
	int		max_type, num_maps=0, grid=KNONE;
        int             objcount; /* Number of sources with a value segment */

/*------------------------------------------------------------------------
|  INITIAL ERROR CHECKING 
------------------------------------------------------------------------*/
	
        /*----------------------------------------------------------------
	| if an unknown mapping type is passed in, set to KAPU_MAP_ALL_DATA
	| and print warning message.
        ----------------------------------------------------------------*/
	if ((mapping != KAPU_USE_COMMON_MAP) && 
	    (mapping != KAPU_MAP_ALL_DATA) && 
	    (mapping != KAPU_CREATE_AUTOCOLOR_MAPS))
	{
	   kinfo(KSTANDARD, "Warning: unknown mapping type (%d) passed to "
			    "kapu_append.  Setting to KAPU_MAP_ALL_DATA (%d)",
			    mapping, KAPU_MAP_ALL_DATA);
	   mapping = KAPU_MAP_ALL_DATA;
	}

        /*----------------------------------------------------------------
	| verify that one and only one dimension flag is set 
        ----------------------------------------------------------------*/
	if ((!w_flag && !h_flag && !d_flag && !t_flag && !e_flag) ||
	    (w_flag && (h_flag || d_flag || t_flag || e_flag)) ||
	    (h_flag && (w_flag || d_flag || t_flag || e_flag)) ||
	    (d_flag && (h_flag || w_flag || t_flag || e_flag)) ||
	    (t_flag && (h_flag || d_flag || w_flag || e_flag)) ||
	    (e_flag && (h_flag || d_flag || t_flag || w_flag)) )
	{
	   kerror(lib, rtn,
		"One and only one of the dimension flags must be set:\n"
		"w_flag = %d, h_flag = %d, d_flag = %d, t_flag = %d, "
		"e_flag = %d, ", w_flag, h_flag, d_flag, t_flag, e_flag);
	   return FALSE;
	}

	src_object = (kobject *)kmalloc((unsigned)num_objects*sizeof(kobject));
	if (src_object == NULL)
	{
	   kerror(lib, rtn, "unable to allocate source object array");
	   return FALSE;
	}

	/*----------------------------------------------------------------
	| Verify that all source objects are valid data objects, and 
	| Query each source object for value data. If an object lacks value
        | data, just skip it. Make a list of references for those source 
        | objects that *do* have a value segment.
	----------------------------------------------------------------*/
        objcount = 0;
	for (i=0; i<num_objects; i++)
	{
	   if (object_array[i] == KOBJECT_INVALID)
	   {
	      kerror(lib,rtn,"source object %d is an invalid data object",i);
	      kfree(src_object);
              return FALSE;
	   }
	   src_object[objcount] = NULL;
           if (kpds_query_value(object_array[i]))
           {
	     if ((src_object[objcount] = kpds_reference_object(object_array[i]))
			== KOBJECT_INVALID)
	     {
	       kerror(lib,rtn,"failed to reference source data object [%d]",i);
	       kapu_append_internal_cleanup(num_objects);
               return FALSE;
	     }
             objcount++;
           }
	}

        /* Set the number of objects to now correspond to the number of
           *references* to source objects that had a value segment */
        num_objects = objcount;

        /*----------------------------------------------------------------
	| allocate size arrays 
        ----------------------------------------------------------------*/
	if (((w_size=(int *)kmalloc((unsigned)num_objects*sizeof(int)))==NULL)||
	    ((h_size=(int *)kmalloc((unsigned)num_objects*sizeof(int)))==NULL)||
	    ((d_size=(int *)kmalloc((unsigned)num_objects*sizeof(int)))==NULL)||
	    ((t_size=(int *)kmalloc((unsigned)num_objects*sizeof(int)))==NULL)||
	    ((e_size=(int *)kmalloc((unsigned)num_objects*sizeof(int)))==NULL)||
	    ((d_type=(int *)kmalloc((unsigned)num_objects*sizeof(int)))==NULL))
	{
	   kerror(lib, rtn, "unable to allocate size or data type arrays");
	   kapu_append_internal_cleanup(num_objects);
	   return FALSE;
	}

        /*----------------------------------------------------------------
	| verify that destination object is a valid data object 
        ----------------------------------------------------------------*/
	if (dst_obj == KOBJECT_INVALID)
	{
	   kerror(lib, rtn, "dst_obj is an invalid data object");
	   kapu_append_internal_cleanup(num_objects);
	   return FALSE;
	}

/*----------------------------------------------------------------
|  SOURCE AND DESTINATION OBJECT CHECKS AND SET UP 
----------------------------------------------------------------*/

        /*----------------------------------------------------------------
	| Location and Time Data:
	| ----------------------
	| Query first source object for location and time. If it has 
	| it, transfer the information to the destination object (copy 
	| data later).  setting the location attributes will create 
	| location in the destination object if it does not already 
	| exist. Set up location and time before setting value size, 
	| since setting any size attribute could mess up our final 
	| results.
        ----------------------------------------------------------------*/
	if ((has_location = kpds_query_location(src_object[0])) == TRUE)
	{
	   if (!kpds_copy_location_attr(src_object[0], dst_obj))
	   {
		 kerror(lib, rtn, "unable to copy location attributes");
		 kapu_append_internal_cleanup(num_objects);
              	 return FALSE;
	   }	
	}
	if ((has_time = kpds_query_time(src_object[0])) == TRUE)
	{
           if (!kpds_copy_time_attr(src_object[0], dst_obj))
	   {
	      kerror(lib, rtn, "unable to copy time attributes");
	      kapu_append_internal_cleanup(num_objects);
              return FALSE;
	   }
	}
	   
        /*----------------------------------------------------------------
	| Map Data: 
	|  --------
	| Query each object for map data and get a count of the objects 
	| that have maps, and their sizes. Determine the maximum map size 
	| and data type.  
        ----------------------------------------------------------------*/
	num_maps = 0;
	w_max = h_max = d_max = t_max = e_max = 1;
	max_type = KBIT;
	for (i=(num_objects-1); i>=0; i--)
	{
	   if (kpds_query_map(src_object[i]))
	   {
	      num_maps ++;
	      map_object = i;
	      if (!kpds_get_attributes(src_object[i], KPDS_MAP_SIZE, 
		   &w_size[i], &h_size[i], &d_size[i], &t_size[i], &e_size[i],
		   KPDS_MAP_DATA_TYPE, &d_type[i], NULL))
	      {
		 kerror(lib,rtn,"unable to get map size and type for "
				   "source object %d", i);
		 kapu_append_internal_cleanup(num_objects);
                 return FALSE;
	      }
	      w_max = kmax(w_max, w_size[i]);
	      h_max = kmax(h_max, h_size[i]);
	      d_max = kmax(d_max, d_size[i]);
	      t_max = kmax(t_max, t_size[i]);
	      e_max = kmax(e_max, e_size[i]);
	      max_type = kmax(max_type, d_type[i]);
	   }
	}

        /*----------------------------------------------------------------
	|
        ----------------------------------------------------------------*/
	if (num_maps > 0)
	{
           /*----------------------------------------------------------------
	   | If all inputs have maps, and mapping is "KAPU_MAP_ALL_DATA", 
	   | and are appending along depth, time, or elements, set mapping
	   | to KAPU_CREATE_AUTOCOLOR_MAPS so that data isn't really mapped.  This
	   | may not be the best route to go, but for now, if you always
	   | call kapu_append with KAPU_MAP_ALL_DATA, you will get reasonable
	   | behavior.
           ----------------------------------------------------------------*/
/*
	   if ((num_maps == num_objects) && 
	       (mapping == KAPU_MAP_ALL_DATA) && 
	       (d_flag || t_flag || e_flag))
	      mapping = KAPU_CREATE_AUTOCOLOR_MAPS;
*/

	   if (mapping == KAPU_USE_COMMON_MAP) 
	   {
	      w_max = w_size[map_object];
	      h_max = h_size[map_object];
	      d_max = d_size[map_object];
	      t_max = t_size[map_object];
	      e_max = e_size[map_object];
	      max_type = d_type[map_object];
	      num_maps = 1;
	   }

           /*----------------------------------------------------------------
	   | If the w_flag or h_flag is TRUE, only KAPU_MAP_ALL_DATA or
	   | KAPU_USE_COMMON_MAP is legal.  If KAPU_MAP_ALL_DATA, set 
	   | mapping mode of the source objects to KMAPPED and destroy 
	   | destination map if it exists.  If KAPU_USE_COMMON_MAP and source 1
	   | has a map, set up the destination map.  otherwise destroy it.
           ----------------------------------------------------------------*/
	   else if (mapping == KAPU_MAP_ALL_DATA)
	   {
	      for (i=0; i<num_objects; i++)
	      {
	         if (kpds_query_map(src_object[i]))
	         {
	            if (!kpds_set_attribute(src_object[i], 
				   KPDS_MAPPING_MODE, KMAPPED))
		    {
                       kerror(lib, rtn, "unable to set mapping mode on "
                                           "source object %d", i);
                       kapu_append_internal_cleanup(num_objects);
                       return FALSE;
	            }
	         }   
	      }
	      if ((kpds_query_map(dst_obj)) && (!kpds_destroy_map(dst_obj)))
	      {
                 kerror(lib, rtn, "unable to destroy destination map");
                 kapu_append_internal_cleanup(num_objects);
                 return FALSE;
	      }
	      num_maps = 0;
	   }

	   else if (mapping == KAPU_CREATE_AUTOCOLOR_MAPS) 
	   {
	      if (w_flag || h_flag)
	      {
                 kerror(lib,rtn,"when appending along width or height, "
 			        "creating a map for each input is not a "
				"legal option.");
                 kapu_append_internal_cleanup(num_objects);
                 return FALSE;
	      }

	      if (num_maps != num_objects) 
	      {
	         for (i=0; i<num_objects; i++)
	         {
	            if (!kpds_query_map(src_object[i]))
		    {
		       if (!kpds_create_map(src_object[i]))
		       {
                          kerror(lib,rtn,"cannot create map for object %d",i);
                          kapu_append_internal_cleanup(num_objects);
                          return FALSE;
		       }
		       if (!kpds_set_attributes(src_object[i], 
				KPDS_MAP_DATA_TYPE, max_type,
				KPDS_MAP_SIZE, w_max,h_max,d_max,t_max,e_max,
				NULL))
		       {
                          kerror(lib, rtn, "unable to set map attributes for "
					      "source object %d", i);
                          kapu_append_internal_cleanup(num_objects);
                          return FALSE;
		       }
		       w_size[i] = w_max;
		       h_size[i] = h_max;
		       d_size[i] = d_max;
		       t_size[i] = t_max;
		       e_size[i] = e_max;
		       if ((!kcolor_copy_attribute(src_object[map_object], 
				   src_object[i], KCOLOR_MAP_AUTOCOLOR)) ||
			    (!kcolor_copy_attribute(src_object[map_object], 
				   src_object[i], KCOLOR_COLORSPACE)) ||
			    (!kcolor_copy_attribute(src_object[map_object], 
				   src_object[i], KCOLOR_HAS_ALPHA)) )
		       {
                          kerror(lib,rtn,"failed to copy color attributes from "
				 "map source %d to object %d",map_object,i);
                          kapu_append_internal_cleanup(num_objects);
                          return FALSE;
		       }
		    }
	         }
	         num_maps = num_objects;
	      }
	   }

           /*-------------------------------------------------------------
	   | If all objects have maps, all of the maps will be transferred 
	   | to the destination.  Set up destination using correct map size 
	   | for the stacking dimension.  (Checks were made above so that 
	   | this type of mapping arrangement only happens if we are 
	   | appending along depth, time, or elements.) 
           -------------------------------------------------------------*/
	   if (num_maps == num_objects) 
	   {
	      w_sum = 0; h_sum = 0; d_sum = 0; t_sum = 0; e_sum = 0;
	      for (i=0; i<num_objects; i++)
	      {
	         w_sum += w_size[i];
	         h_sum += h_size[i];
	         d_sum += d_size[i];
	         t_sum += t_size[i];
	         e_sum += e_size[i];
	      }
	      for (i=0; i<num_objects; i++)
	      {
	         if (w_flag)   w_max = w_size[i];
	         if (h_flag)   h_max = h_size[i];
	         if (d_flag)   d_max = d_size[i];
	         if (t_flag)   t_max = t_size[i];
	         if (e_flag)   e_max = e_size[i];
	         if (!kpds_set_attributes(src_object[i],
			   KPDS_MAP_SIZE, w_max, h_max, d_max, t_max, e_max,
			   KPDS_MAP_DATA_TYPE, max_type, NULL)) 
	         {
                    kerror(lib,rtn,"cannot set map attributes on object %d",
				i);
                    kapu_append_internal_cleanup(num_objects);
                    return FALSE;
	         } 
	      }
	      if (w_flag)   w_max = w_sum;
              if (h_flag)   h_max = h_sum;
              if (d_flag)   d_max = d_sum;
              if (t_flag)   t_max = t_sum;
              if (e_flag)   e_max = e_sum;
	   }

           /*----------------------------------------------------------
	   | create the destination map and set attributes 
           ----------------------------------------------------------*/
	   if (num_maps > 0)
	   {
	      if (!kpds_query_map(dst_obj))
	      {
	         if (!kpds_create_map(dst_obj))
	         {
		    kerror(lib,rtn,"unable to create destination map data");
		    kapu_append_internal_cleanup(num_objects);
              	    return FALSE;
	         }
	      }
	      if (!kpds_copy_map_attr(src_object[map_object], dst_obj))
	      {
                 kerror(lib, rtn, "unable to copy attributes from map source "
				     "%d to destination", map_object);
                 kapu_append_internal_cleanup(num_objects);
                 return FALSE;
	      }

	      if (!kpds_set_attributes(dst_obj, 
                        KPDS_MAP_SIZE, w_max, h_max, d_max, t_max, e_max,
                        KPDS_MAP_DATA_TYPE, max_type, NULL))
	      {
	         kerror(lib,rtn,"unable to set destination map attributes");
	         kapu_append_internal_cleanup(num_objects);
                 return FALSE;
	      }
	      kcolor_copy_attributes(src_object[map_object], dst_obj,
				KCOLOR_MAP_AUTOCOLOR, KCOLOR_COLORSPACE,
				KCOLOR_HAS_ALPHA, NULL);
	   }

	}  /* end of map set up case */
	

        /*----------------------------------------------------------------
	| Value Data:  (do this part after location, time, and mapping 
	| ----------
	| are determined) 
	|    * determine the highest data type and set on all source objects
	|    * determine the max size of all dimensions and set on all source
	|      objects.
	| Check that the destination object has value data.  
	| Create it if it does not. 
	|    Set the output size based on dimension flag and maximum 
	|    dimension sizes
	|    Set output data type 
        ----------------------------------------------------------------*/
	for (i=0; i<num_objects; i++)
	{
	   if (!kpds_get_attributes(src_object[i], 
			KPDS_VALUE_SIZE, &w_size[i], &h_size[i], &d_size[i], 
				       &t_size[i], &e_size[i],
			KPDS_VALUE_DATA_TYPE, &d_type[i], NULL))
	   {
	      kerror(lib,rtn,"unable to get value size and data type for "
				   "source object %d", i);
	      kapu_append_internal_cleanup(num_objects);
              return FALSE;
	   }
	}
	
        /*----------------------------------------------------------------
	| Set value sizes, including correct size for the stacking 
	| dimension. 
        ----------------------------------------------------------------*/
	w_max = w_size[0];
	h_max = h_size[0];
	d_max = d_size[0];
	t_max = t_size[0];
	e_max = e_size[0];
	max_type = KBIT;

	w_sum = 0; h_sum = 0; d_sum = 0; t_sum = 0; e_sum = 0;
	for (i=0; i<num_objects; i++)
	{
	   w_max = kmax(w_max, w_size[i]);
	   h_max = kmax(h_max, h_size[i]);
	   d_max = kmax(d_max, d_size[i]);
	   t_max = kmax(t_max, t_size[i]);
	   e_max = kmax(e_max, e_size[i]);
	   max_type = kmax(max_type, d_type[i]);
	   w_sum += w_size[i];
	   h_sum += h_size[i];
	   d_sum += d_size[i];
	   t_sum += t_size[i];
	   e_sum += e_size[i];
	}
	for (i=0; i<num_objects; i++)
	{
           /*-------------------------------------------------------------
	   | Need to keep the original size of the stacking dimension
	   | for each source object (do not want to set it to the 
	   | maximum determined above).  
           -------------------------------------------------------------*/
/* comment out stuff below for ds bug
*/
#if 0
	   if (w_flag)   w_max = w_size[i];
	   if (h_flag)   h_max = h_size[i];
	   if (d_flag)   d_max = d_size[i];
	   if (t_flag)   t_max = t_size[i];
	   if (e_flag)   e_max = e_size[i];
	   if (!kpds_set_attributes(src_object[i],
			KPDS_VALUE_SIZE, w_max, h_max, d_max, t_max, e_max,
			KPDS_VALUE_DATA_TYPE, max_type, NULL)) 
#endif
/* comment out stuff above for ds bug
*/
	   if (!kpds_set_attribute(src_object[i],
			KPDS_VALUE_DATA_TYPE, max_type)) 
	   {
               kerror(lib,rtn,"cannot set value attributes on object %d",i);
               kapu_append_internal_cleanup(num_objects);
               return FALSE;
	   } 

           /*-------------------------------------------------------------
	   | Put the correct values in the size arrays for later use.
	   | This will be the max for all dimensions except the stacking 
	   | dimension, for which max was just set to *_size[i].
           -------------------------------------------------------------*/
/* comment out stuff below for ds bug
*/
#if 0
	   w_size[i] = w_max;
	   h_size[i] = h_max;
	   d_size[i] = d_max;
	   t_size[i] = t_max;
	   e_size[i] = e_max;
#endif
/* comment out stuff above for ds bug
*/
	}

        /*----------------------------------------------------------------
	| create the destination value data and set attributes 
        ----------------------------------------------------------------*/
	if (!kpds_query_value(dst_obj))
	{
	   if (!kpds_create_value(dst_obj))
	   {
	      kerror(lib,rtn,"unable to create destination value data");
	      kapu_append_internal_cleanup(num_objects);
              return FALSE;
	   }
	}
        /*----------------------------------------------------------------
	| The output size should be the max size of all inputs along
	| all dimensions except the stacking dimension.  Along this 
	| dimension, it should be the sum of all source object sizes.
        ----------------------------------------------------------------*/
	if (w_flag)   w_max = w_sum;
        if (h_flag)   h_max = h_sum;
        if (d_flag)   d_max = d_sum;
        if (t_flag)   t_max = t_sum;
        if (e_flag)   e_max = e_sum;
	if (!kpds_set_attributes(dst_obj, 
                        KPDS_VALUE_SIZE, w_max, h_max, d_max, t_max, e_max,
                        KPDS_VALUE_DATA_TYPE, max_type, NULL))
	{
	   kerror(lib,rtn,"unable to set destination value attributes");
	   kapu_append_internal_cleanup(num_objects);
           return FALSE;
	}

        /*----------------------------------------------------------------
	| If we have map data, need to make sure that the depth, time and
	| elements sizes are either 1, or match the value dimensions.
	| (Temporarily using the region size variables)
        ----------------------------------------------------------------*/
	if (num_maps > 0)
	{
	   if (!kpds_get_attribute(dst_obj, KPDS_MAP_SIZE, 
			&w_rgn, &h_rgn, &d_rgn, &t_rgn, &e_rgn))
	   {
	      kerror(lib,rtn,"unable to get destination map size");
	      kapu_append_internal_cleanup(num_objects);
              return FALSE;
	   }
	   if ( ((d_rgn != 1) && (d_rgn != d_max)) ||
	        ((t_rgn != 1) && (t_rgn != t_max)) ||
	        ((e_rgn != 1) && (e_rgn != e_max)) )
	   {
	      kerror(lib,rtn,"Resulting map depth, time, and elements must "
			     "equal 1, or \n equal the value depth, time, or "
		             "elements size\n "
			     "\tmap depth = %d, value depth = %d \n"
			     "\tmap time = %d, value time = %d \n"
			     "\tmap elements = %d, value elements = %d\n."
			     "Use one of the other two mapping options.",
			     d_rgn, d_max, t_rgn, t_max, e_rgn, e_max);
	      kapu_append_internal_cleanup(num_objects);
              return FALSE;
	   }
	}

        /*----------------------------------------------------------------
	| Mask Data;
	| ---------
	| Query all source objects.  If any contain mask, create it in 
	| the destination object.  Instead of creating mask in all objects,
	| keep track of the position, and put mask data in the right
	| places in the destination object.  Query for highest data type,
	| and set it on all source masks and on the destination mask.
        ----------------------------------------------------------------*/
	max_type = KBIT;
	for (i=0; i<num_objects; i++)
	{
	   if (kpds_query_mask(src_object[i]))
	   {
	      if (!kpds_get_attribute(src_object[i], 
			KPDS_MASK_DATA_TYPE, &d_type[i]))
	      {
	         kerror(lib,rtn,"unable to get mask type of object %d", i);
	         kapu_append_internal_cleanup(num_objects);
                 return FALSE;
	      }
	      max_type = kmax(max_type, d_type[i]);
	      has_mask = TRUE;
	   }
	}
	if (has_mask)
	{
	   for (i=0; i<num_objects; i++)
	   {
	      if (!kpds_query_mask(src_object[i]))
	      {
	         if (!kpds_create_mask(src_object[i]))
		 {
	            kerror(lib,rtn,"unable to create mask for object %d", i);
	            kapu_append_internal_cleanup(num_objects);
                    return FALSE;
		 }
		 if (!kpds_set_attribute(src_object[i], 
			KPDS_MASK_DATA_TYPE, max_type))
		 {
	            kerror(lib,rtn,"unable to set mask type for object %d", i);
	            kapu_append_internal_cleanup(num_objects);
                    return FALSE;
		 }
		 if (!kpds_initialize_mask(src_object[i], 1.0))
		 {
	            kerror(lib,rtn,"unable to initialize mask for object %d",i);
	            kapu_append_internal_cleanup(num_objects);
                    return FALSE;
		 }
	      }
	   }
	   if (!kpds_query_mask(dst_obj))
	   {
	      if (!kpds_create_mask(dst_obj))
	      {
	         kerror(lib,rtn,"unable to create destination mask");
	         kapu_append_internal_cleanup(num_objects);
                 return FALSE;
	      }
	   }
	   if (!kpds_set_attribute(dst_obj, KPDS_MASK_DATA_TYPE, max_type))
	   {	
	      kerror(lib,rtn,"unable to set destination mask attributes");
	      kapu_append_internal_cleanup(num_objects);
              return FALSE;
	   }
	   if (!kpds_initialize_mask(dst_obj, 1.0))
	   {	
	      kerror(lib,rtn,"unable to initialize destination mask");
	      kapu_append_internal_cleanup(num_objects);
              return FALSE;
	   }
	}	

        /*----------------------------------------------------------------
	| Reference the destination object 
        ----------------------------------------------------------------*/
	if ((dst_object = kpds_reference_object(dst_obj)) == KOBJECT_INVALID)
	{
	   kerror(lib,rtn,"unable to reference destination object");
	   kapu_append_internal_cleanup(num_objects);
           return FALSE;
	}

/*------------------------------------------------------------------------
|  APPENDING DATA 
------------------------------------------------------------------------*/

        /*----------------------------------------------------------------
	| Use the optimal region size of the value data for each source 
	| object.  To make sure that the data is written to the right 
	| place in the destination object, instantiate the position to 
	| (0,0,0,0,0) for each object, set the offset to all zeros
	| except for the stacking dimension, which should be the sum
	| of the size (in that dim) all objects that have already been 
	| stacked.  Set the increment size to the size of the object
	| that is being appended.
        ----------------------------------------------------------------*/
	w_off = h_off = d_off = t_off = e_off = 0;
	
	for (i=0; i<num_objects; i++)
	{
	   if (!kpds_get_attribute(src_object[i], 
			KPDS_VALUE_OPTIMAL_REGION_SIZE,
			&w_rgn, &h_rgn, &d_rgn, &t_rgn, &e_rgn, &num_rgns))
           {
              kerror(lib,rtn,"cannot get value optimal region size for (%d)",i);
	      kapu_append_internal_cleanup(num_objects);
              return FALSE;
           }
	
	   if (!kpds_set_attributes(dst_object, 
		KPDS_VALUE_REGION_SIZE, w_rgn, h_rgn, d_rgn, t_rgn, e_rgn,
		KPDS_VALUE_OFFSET, w_off,h_off,d_off,t_off,e_off,
		KPDS_VALUE_INCREMENT_SIZE, 
		   w_size[i], h_size[i], d_size[i], t_size[i], e_size[i],
		KPDS_VALUE_POSITION, 0, 0, 0, 0, 0, NULL))
           {
              kerror(lib,rtn,"unable to set value attributes of destination "
				"object for appending source object %d",i);
	      kapu_append_internal_cleanup(num_objects);
              return FALSE;
           }
	   if (!kpds_set_attributes(src_object[i], 
			KPDS_VALUE_REGION_SIZE, w_rgn,h_rgn,d_rgn,t_rgn,e_rgn,
			KPDS_VALUE_POSITION, 0,0,0,0,0,
			NULL))
           {
              kerror(lib,rtn,"unable to set region size of object %d",i);
	      kapu_append_internal_cleanup(num_objects);
              return FALSE;
           }

/*
kinfo(KSTANDARD, "optimal region: %d %d %d %d %d (%d of these)", 
		  w_rgn, h_rgn, d_rgn, t_rgn, e_rgn, num_rgns);
kinfo(KSTANDARD, "offset: %d %d %d %d %d", 
		  w_off, h_off, d_off, t_off, e_off);
*/
	   for (j=0; j<num_rgns; j++)
	   {
/*
kinfo(KSTANDARD, " ");
kpds_print_attribute(src_object[i], KPDS_VALUE_POSITION, kstdout);
kinfo(KSTANDARD, " <---- src_object KPDS_VALUE_POSITION, i=%d, j=%d",i, j);
kpds_print_attribute(dst_object, KPDS_VALUE_POSITION, kstdout);
kinfo(KSTANDARD, " <---- dst_object KPDS_VALUE_POSITION, i=%d, j=%d",i, j);
kpds_print_attribute(dst_object, KPDS_VALUE_OFFSET, kstdout);
kinfo(KSTANDARD, " <---- dst_object KPDS_VALUE_OFFSET, i=%d, j=%d",i, j);
kpds_print_attribute(dst_object, KPDS_VALUE_INCREMENT_SIZE, kstdout);
kinfo(KSTANDARD, " <---- dst_object KPDS_VALUE_INCREMENT_SIZE, i=%d, j=%d",i,j);
*/
	      if ((data = kpds_get_data(src_object[i],KPDS_VALUE_REGION,data))
			== NULL)
              {
                 kerror(lib,rtn,"get data for region %d of object %d failed",
				j,i);
	         kapu_append_internal_cleanup(num_objects);
                 return FALSE;
              }
	      if (!kpds_put_data(dst_object, KPDS_VALUE_REGION, data))
              {
                 kerror(lib,rtn,"put data from region %d of object %d failed",
				j,i);
	         kapu_append_internal_cleanup(num_objects);
                 return FALSE;
              }
	   }
	
	   w_off = (w_flag) ? w_off + w_size[i] : w_off;
	   h_off = (h_flag) ? h_off + h_size[i] : h_off;
	   d_off = (d_flag) ? d_off + d_size[i] : d_off;
	   t_off = (t_flag) ? t_off + t_size[i] : t_off;
	   e_off = (e_flag) ? e_off + e_size[i] : e_off;

	   kfree(data); 
	}

        /*----------------------------------------------------------------
	| If mask exists, get and put all mask data from objects that 
	| have it - keep track of destination position. 
        ----------------------------------------------------------------*/
	if (has_mask)
	{
	   w_off = h_off = d_off = t_off = e_off = 0;
	   for (i=0; i<num_objects; i++)
	   {
	      if (kpds_query_mask(src_object[i]))
	      {
	         if (!kpds_get_attribute(src_object[i], 
			KPDS_MASK_OPTIMAL_REGION_SIZE, 
			&w_rgn, &h_rgn, &d_rgn, &t_rgn, &e_rgn, &num_rgns))
                 {
                    kerror(lib,rtn,"cannot get mask opt region size, (%d)",i);
	            kapu_append_internal_cleanup(num_objects);
                    return FALSE;
                 }
           	 if (!kpds_set_attributes(dst_object,
                	KPDS_MASK_REGION_SIZE, w_rgn,h_rgn,d_rgn,t_rgn,e_rgn,
                	KPDS_MASK_OFFSET, w_off,h_off,d_off,t_off,e_off,
                	KPDS_MASK_INCREMENT_SIZE,
                   	w_size[i], h_size[i], d_size[i], t_size[i], e_size[i],
                	KPDS_MASK_POSITION, 0, 0, 0, 0, 0, NULL))
                 {
                    kerror(lib,rtn,"cannot set destination mask "
				   "attributes (source object %d)",i);
                    kapu_append_internal_cleanup(num_objects);
                    return FALSE;
                 }
	         if (!kpds_set_attributes(src_object[i], 
			KPDS_MASK_REGION_SIZE, w_rgn,h_rgn,d_rgn,t_rgn,e_rgn,
			KPDS_MASK_POSITION, 0,0,0,0,0,
			NULL))
                 {
                    kerror(lib,rtn,"cannot set mask region size, object %d",i);
	            kapu_append_internal_cleanup(num_objects);
                    return FALSE;
                 }

	         for (j=0; j<num_rgns; j++)
	         {
	            if ((data = kpds_get_data(src_object[i],
				KPDS_MASK_REGION, data)) == NULL)
                    {
                       kerror(lib,rtn,"get mask data for region %d of "
					"object %d failed",j,i);
	               kapu_append_internal_cleanup(num_objects);
                       return FALSE;
                    }
	            if (!kpds_put_data(dst_object, KPDS_MASK_REGION, data))
                    {
                       kerror(lib,rtn, "put mask data of region %d of "
					"object %d failed" ,j,i);
	               kapu_append_internal_cleanup(num_objects);
                       return FALSE;
                    }
	         }
	      }
              w_off = (w_flag) ? w_off + w_size[i] : w_off;
              h_off = (h_flag) ? h_off + h_size[i] : h_off;
              d_off = (d_flag) ? d_off + d_size[i] : d_off;
              t_off = (t_flag) ? t_off + t_size[i] : t_off;
              e_off = (e_flag) ? e_off + e_size[i] : e_off;
   
	      kfree(data); 
	   }
	}
	
	if (num_maps > 0)
	{
           /*-------------------------------------------------------------
	   | If only one map, just copy data from source to destination 
           -------------------------------------------------------------*/
	   if (num_maps == 1) 
	   {
	      if (!kpds_copy_map_data(src_object[map_object], dst_object, TRUE))
	      {
                 kerror(lib,rtn,"Cannot copy map from object %d to destination",
				   map_object);
	         kapu_append_internal_cleanup(num_objects);
                 return FALSE;
	      }
	   }

           /*-------------------------------------------------------------
	   | If all sources have maps, get optimal map region each object, 
	   | and transfer from all source objects to destination object. 
           -------------------------------------------------------------*/
	   else
	   {
	      w_off = h_off = d_off = t_off = e_off = 0;
	      for (i=0; i<num_objects; i++)
	      {
	         if (!kpds_get_attribute(src_object[i], 
			KPDS_MAP_OPTIMAL_REGION_SIZE,
			&w_rgn, &h_rgn, &d_rgn, &t_rgn, &e_rgn, &num_rgns))
                 {
                    kerror(lib,rtn,"cannot get map optimal region size "
				   "for (%d)",i);
	            kapu_append_internal_cleanup(num_objects);
                    return FALSE;
                 }
	         if (!kpds_set_attributes(dst_object, 
		      KPDS_MAP_REGION_SIZE, w_rgn, h_rgn, d_rgn, t_rgn, e_rgn,
		      KPDS_MAP_OFFSET, w_off,h_off,d_off,t_off,e_off,
		      KPDS_MAP_INCREMENT_SIZE, 
		         w_size[i], h_size[i], d_size[i], t_size[i], e_size[i],
		      KPDS_MAP_POSITION, 0, 0, 0, 0, 0, NULL))
                 {
                    kerror(lib,rtn,"unable to set map attributes of dest "
				   "object for appending source object %d",i);
	            kapu_append_internal_cleanup(num_objects);
                    return FALSE;
                 }
	         if (!kpds_set_attributes(src_object[i], 
			   KPDS_MAP_REGION_SIZE, w_rgn,h_rgn,d_rgn,t_rgn,e_rgn,
			   KPDS_MAP_POSITION, 0,0,0,0,0,
			   NULL))
                 {
                    kerror(lib,rtn,"unable to set map region size of "
				   "object %d",i);
	            kapu_append_internal_cleanup(num_objects);
                    return FALSE;
                 }

	         for (j=0; j<num_rgns; j++)
	         {
	            if ((data=kpds_get_data(src_object[i],KPDS_MAP_REGION,data))
			== NULL)
                    {
                       kerror(lib,rtn,"get map data for region %d of object "
				      "%d failed",j,i);
	               kapu_append_internal_cleanup(num_objects);
                       return FALSE;
                    }
	            if (!kpds_put_data(dst_object, KPDS_MAP_REGION, data))
                    {
                       kerror(lib,rtn,"put map data from region %d of object "
				      "%d failed", j,i);
	               kapu_append_internal_cleanup(num_objects);
                       return FALSE;
                    }
	         }
	
	         w_off = (w_flag) ? w_off + w_size[i] : w_off;
	         h_off = (h_flag) ? h_off + h_size[i] : h_off;
	         d_off = (d_flag) ? d_off + d_size[i] : d_off;
	         t_off = (t_flag) ? t_off + t_size[i] : t_off;
	         e_off = (e_flag) ? e_off + e_size[i] : e_off;

	         kfree(data); 
	      }
	   }
	}

        /*----------------------------------------------------------------
	| If location data, copy data and/or attributes from first object 
	| that has it. 
        ----------------------------------------------------------------*/
	if (has_location)
	{
	   if (!kpds_get_attribute(src_object[0], 
			KPDS_LOCATION_GRID, &grid))
           {
              kerror(lib, rtn, "Unable to retrieve location grid attribute "
			"from first source object.");
	      kapu_append_internal_cleanup(num_objects);
              return FALSE;
           }

	   if ((grid == KRECTILINEAR) || (grid == KCURVILINEAR))
	   {
	      if (!kpds_copy_location_data(src_object[0], 
			dst_object, TRUE))
	      {
	         kerror(lib,rtn,"failed to copy location data from first "
			   "object to destination object");
	         kapu_append_internal_cleanup(num_objects);
                 return FALSE;
	      }
	   }
	}

        /*----------------------------------------------------------------
	| If time data, copy data and/or attributes from first object 
	| that has it. 
        ----------------------------------------------------------------*/
	if (has_time)
	{
	   if (!kpds_copy_time_data(src_object[0], dst_object, TRUE))
	   {
	      kerror(lib,rtn,"failed to copy time data from first object to "
			"destination object");
	      kapu_append_internal_cleanup(num_objects);
              return FALSE;
	   }
	}

	kapu_append_internal_cleanup(num_objects);
	return TRUE;
}

static void
kapu_append_internal_cleanup(
	int num_objects)
{
        /*----------------------------------------------------------------
	| CLEANUP Function
	| Frees pointers and closes reference objects for kapu_append
        ----------------------------------------------------------------*/
	int	i;

	if (w_size)	kfree(w_size);
	if (h_size)	kfree(h_size);
	if (d_size)	kfree(d_size);
	if (t_size)	kfree(t_size);
	if (e_size)	kfree(e_size);
	for (i=0; i<num_objects; i++)
	{
	   if (src_object[i] != KOBJECT_INVALID)  
	      kpds_close_object(src_object[i]);
	}
	if (src_object != NULL) kfree(src_object);
	if (dst_object != KOBJECT_INVALID)
	   kpds_close_object(dst_object);
	kfree(data);
}
/* -library_code_end */
