 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Polymophic Model Data Handlers
   >>>>
   >>>>   Static:
   >>>>                 _size_get()
   >>>>                 _size_set()
   >>>>                 _size_match()
   >>>>                 _size_copy()
   >>>>                 _size_query()
   >>>>                 _size_print() 
   >>>>             
   >>>>                 _type_get()
   >>>>                 _type_set()
   >>>>                 _type_match()
   >>>>                 _type_copy()
   >>>>                 _type_query()
   >>>>                 _type_print() 
   >>>>
   >>>>			_create_time()
   >>>>  Private:
   >>>>			kpds_init_time_segment()
   >>>>   Public:
   >>>>			kpds_create_time()
   >>>>			kpds_query_time()
   >>>>			kpds_destroy_time()
   >>>>			kpds_create_time()
   >>>>			kpds_copy_time_data()
   >>>>			kpds_copy_time_attr()
   >>>>			kpds_copy_time()  
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

/* -- pds_attrib.c -- */
extern int sync_sizes;

/* 
 *    ================================================================== 
 *    TIME attribute handlers
 *    ==================================================================
 */

/*********** ---------------------------------------------------------------
 ***********  KPDS_TIME_SIZE
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _size_get
|       Purpose: get the time size attribute
|    Written By: Steve Kubica and Jeremy Worley
|          Date: Apr 27, 1994 21:34
------------------------------------------------------------*/
/* ARGSUSED */
static int
_size_get(kobject obj, int assoc, int attr, kaddr clientData, kva_list *list)
{
   int *t = kva_arg(*list, int *);
   int *size;
   

   if (!kdms_query_segment(obj, KPDS_SEGMENT_TIME))
      return FALSE; /* error -- must create time first */

   if (!kdms_get_attribute(obj, KDMS_SEGMENT_TIME, KDMS_SIZE, &size))
      return FALSE; 
    
   if (t) *t = size[0];  /* -- size is only 1D -- */

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _size_set
|       Purpose: set the time size attribute
|    Written By: Steve Kubica and Jeremy Worley
|          Date: Apr 27, 1994 21:34
------------------------------------------------------------*/
/* ARGSUSED */
static int
_size_set(kobject obj, int assoc, int attr, kaddr clientData, 
	  kva_list *list)
{
   int  t = kva_arg(*list, int);
   int  size[KPDS_TIME_DIM] = {-1};

   if (!kdms_query_segment(obj, KPDS_SEGMENT_TIME))
      return FALSE; /* error -- must create time first */

   size[0] = t;
   
   if (sync_sizes)
   {
      sync_sizes = FALSE;  /* -- avoid calling here again -- */

      /* -- set the shared size of the value segment -- */
      if (kdms_query_segment(obj, KPDS_SEGMENT_VALUE))
	 if (!kpds_set_attribute(obj, KPDS_VALUE_SIZE, -1, -1, -1, t, -1))
	    return FALSE;
      
      /* -- set the shared size of the mask segment -- */
      if (kdms_query_segment(obj, KPDS_SEGMENT_MASK))
	 if (!kpds_set_attribute(obj, KPDS_MASK_SIZE, -1, -1, -1, t, -1))
	    return FALSE;

      /* -- set the shared size of the time segment -- */
      if (kdms_query_segment(obj, KPDS_SEGMENT_MAP))
      {
	 int mt;
	 
	 /*
	  *   track the size of the mask time only if the previous map 
	  *   size is not 1
	  */
	 if (!kpds_get_attribute(obj, KPDS_MAP_SIZE, NULL, NULL, NULL, 
				 &mt, NULL))
	    return FALSE;
      
	 if (!kpds_set_attribute(obj, KPDS_MAP_SIZE, -1, -1, -1,
				 ((mt == 1) ? -1 : t), -1))
	    return FALSE;
      }

      sync_sizes = TRUE;
   }

   /* set the size of the time segment */
   if (!kdms_set_attribute(obj, KPDS_SEGMENT_TIME, KDMS_SIZE, size))
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _size_match
|       Purpose: match the time size attribute
|    Written By: Steve Kubica
|          Date: Apr 28, 1994 17:37
------------------------------------------------------------*/
/* ARGSUSED */
static int
_size_match(kobject obj1, kobject obj2, int assoc, int attrib,
	    kaddr clientData1, kaddr clientData2)
{
   int t1;
   int t2;

   if (!kpds_get_attribute(obj1, KPDS_TIME_SIZE, &t1))
      return FALSE;

   if (!kpds_get_attribute(obj2, KPDS_LOCATION_SIZE, &t2))
      return FALSE;

   return (t1 == t2);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _size_copy
|       Purpose: copy the time size attribute
|    Written By: Steve Kubica
|          Date: Apr 28, 1994 17:37
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_size_copy(kobject obj1, kobject obj2, int assoc, int attrib,
	   kaddr clientData1, kaddr clientData2)
{
   int t;

   if (!kpds_get_attribute(obj1, KPDS_TIME_SIZE, &t))
      return FALSE;

   if (!kpds_set_attribute(obj2, KPDS_TIME_SIZE, t))
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _size_query
|       Purpose: query the time size attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_size_query(kobject obj, int assoc, int attrib, kaddr clientData,
	    int *num_args, int *arg_size, int *data_type, int *permanent)
{
   if (!kdms_query_segment(obj, KPDS_SEGMENT_TIME))
      return FALSE;		/* not an error */

   if (num_args)
      *num_args = 1;
   if (arg_size)
      *arg_size = 1;
   if (data_type)
      *data_type = KINT;
   if (permanent)
      *permanent = TRUE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _size_print
|       Purpose: print the time size attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_size_print(kobject obj, int assoc, int attrib, kaddr clientData,
	    kfile * outfile)
{
   int t;


   if (!kpds_get_attribute(obj, KPDS_TIME_SIZE, &t))
      return FALSE;

   if (outfile)
      kfprintf(outfile, "%d", t);

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KPDS_TIME_DATA_TYPE
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _type_get
|       Purpose: get the type attribute
|    Written By: Jeremy Worley, John Salas, and Steve Kubica
|          Date: Apr 28, 1994 17:39
------------------------------------------------------------*/
/* ARGSUSED */
static int
_type_get(kobject obj, int assoc, int attr, kaddr clientData, kva_list *list)
{
   int *rtyp = kva_arg(*list, int *);
   int  typ;


   if (!kdms_query_segment(obj, KPDS_SEGMENT_TIME))
      return FALSE;

   if (!kdms_get_attribute(obj, KPDS_SEGMENT_TIME, KDMS_DATA_TYPE, &typ))
      return FALSE;

   if (rtyp) *rtyp = typ;
   
   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _type_set
|       Purpose: set the type attribute
|    Written By: Jeremy Worley, John Salas, and Steve Kubica
|          Date: Apr 28, 1994 17:39
------------------------------------------------------------*/
/* ARGSUSED */
static int
_type_set(kobject obj, int assoc, int attr, kaddr clientData, kva_list *list)
{
   int typ = kva_arg(*list, int);

   if (!kdms_query_segment(obj, KPDS_SEGMENT_TIME))
      return FALSE;

   if (!kdms_set_attribute(obj, KPDS_SEGMENT_TIME, KDMS_DATA_TYPE, typ))
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _type_match
|       Purpose: match the type attribute
|    Written By: Steve Kubica and John Salas
|          Date: Apr 28, 1994 17:39
------------------------------------------------------------*/
/* ARGSUSED */
static int
_type_match(kobject obj1, kobject obj2, int assoc, int attrib,
	    kaddr clientData1, kaddr clientData2)
{
   int typ1;
   int typ2;

   if (!kdms_query_segment(obj1, KPDS_SEGMENT_TIME) ||
       !kdms_query_segment(obj2, KPDS_SEGMENT_TIME))
      return FALSE;

   if (!kdms_get_attribute(obj1, KPDS_SEGMENT_TIME, KDMS_DATA_TYPE, &typ1))
      return FALSE;

   if (!kdms_get_attribute(obj2, KPDS_SEGMENT_TIME, KDMS_DATA_TYPE, &typ2))
      return FALSE;
   
   return (typ1 == typ2);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _type_copy
|       Purpose: copy the type attribute
|    Written By: Steve Kubica
|          Date: Apr 28, 1994 17:43
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_type_copy(kobject obj1, kobject obj2, int assoc, int attrib,
	   kaddr clientData1, kaddr clientData2)
{
   int typ;

   if (!kdms_query_segment(obj1, KPDS_SEGMENT_TIME) ||
       !kdms_query_segment(obj2, KPDS_SEGMENT_TIME))
      return FALSE;

   if (!kdms_get_attribute(obj1, KPDS_SEGMENT_TIME, KDMS_DATA_TYPE, &typ))
      return FALSE;

   if (!kdms_set_attribute(obj2, KPDS_SEGMENT_TIME, KDMS_DATA_TYPE, typ))
      return FALSE;
   
   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _type_query
|       Purpose: query the type attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_type_query(kobject obj, int assoc, int attrib, kaddr clientData,
	    int *num_args, int *arg_size, int *data_type, int *permanent)
{
   if (!kdms_query_segment(obj, KPDS_SEGMENT_TIME))
      return FALSE;

   if (num_args)
      *num_args = 1;
   if (arg_size)
      *arg_size = 1;
   if (data_type)
      *data_type = KINT;
   if (permanent)
      *permanent = TRUE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _type_print
|       Purpose: print the type attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_type_print(kobject obj, int assoc, int attrib, kaddr clientData,
	    kfile * outfile)
{
   int typ;

   if (!kdms_query_segment(obj, KPDS_SEGMENT_TIME))
      return FALSE;

   if (!kdms_get_attribute(obj, KPDS_SEGMENT_TIME, KDMS_DATA_TYPE, &typ))
      return FALSE;

   if (outfile)
      kfprintf(outfile, "%s (%d)", kdefine_to_datatype(typ), typ);

   return TRUE;
}

/* 
 *    ================================================================== 
 *    TIME segment handlers
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: (static) _create_time
|
|       Purpose: This routine will create the time segment and
|		 initialize its dimensionality and index order.
|
|         Input: obj     - the obj in which to create the time segment
|		 segment - the name of the time segment.
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica & Jeremy Worley
|          Date: Jun 01, 1994 16:23
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static int
_create_time(
   kobject  obj,
   char    *segment)
{
   int order[KPDS_TIME_DIM] = {KTIME};
   int t = -1;

   if (!kdms_create_segment(obj, KPDS_SEGMENT_TIME))
      return FALSE;
   
   if (!kdms_set_attributes(obj, KDMS_SEGMENT_TIME, 
			    KDMS_DIMENSION, KPDS_TIME_DIM,
			    KDMS_INDEX_ORDER, order, 
			    NULL))
      return FALSE;

   /*  -- set initial size according to existing polymorphic segments -- */
   if (kdms_query_segment(obj, KPDS_SEGMENT_VALUE))
      kpds_get_attribute(obj, KPDS_VALUE_SIZE, NULL, NULL, NULL, &t, NULL);

   else if (kdms_query_segment(obj, KPDS_SEGMENT_MASK))
      kpds_get_attribute(obj, KPDS_MASK_SIZE, NULL, NULL, NULL, &t, NULL);

   else if (kdms_query_segment(obj, KPDS_SEGMENT_MAP))
      kpds_get_attribute(obj, KPDS_MAP_SIZE, NULL, NULL, NULL, &t, NULL);
   
   kpds_set_attribute(obj, KPDS_TIME_SIZE, t);

   return TRUE;
}

/* 
 *    ================================================================== 
 *    PRIVATE TIME functions
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: kpds_init_time_segment()
|
|       Purpose: Define the time segment.
|
|         Input: none
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Dec 26, 1994
| Modifications:
|
------------------------------------------------------------*/
int
kpds_init_time_segment(void)
{
   int status = TRUE;

   status &= kdms_define_quasi_attribute(KDMS_OBJECT, 
					 KPDS_TIME_SIZE, NULL,
					 _size_get,   _size_set, 
					 _size_match, _size_copy, 
					 _size_query, _size_print);


   status &= kdms_define_quasi_attribute(KDMS_OBJECT, 
					 KPDS_TIME_DATA_TYPE, NULL,
					 _type_get,   _type_set, 
					 _type_match, _type_copy,
					 _type_query, _type_print);


   status &= kdms_define_segment(KPDS_SEGMENT_TIME, _create_time,
				 NULL, NULL, NULL, NULL, 
				 NULL, NULL, NULL, NULL, NULL, NULL);

   return status;
}

/* 
 *    ================================================================== 
 *    PUBLIC TIME functions
 *    ==================================================================
 */

/************************************************************
*
*  Routine Name: kpds_create_time - create a time segment within a 
*				    data object.
*
*       Purpose: This function is used to create a time segment within
*		 a specified data object.  The size of the time
*		 segment will be initialized to match any the time
*		 size shared with the other polymorphic segments.
*
*		 It is considered an error to create a time segment
*		 if the object already contains one.
*
*         Input: object  - object in which to create the time segment.
*
*        Output: none
*
*       Returns: TRUE (1) if the time segment was successfully created, 
*		 FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley & John Salas
*          Date: Sep 20, 1993 08:33
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kpds_create_time(
*		!   kobject object)
*
*************************************************************/
int
kpds_create_time(kobject object)
{
   return (kpds_init() && kdms_create_segment(object, KPDS_SEGMENT_TIME));
}

/************************************************************
*
*  Routine Name: kpds_query_time - determine if the time segment
*				   exists in a data object.
*
*       Purpose: This function is used to determine if the time
*		 segment exists in a data object.  If time segment
*		 exists in the specified object, then this function
*		 will return TRUE.  If the object is invalid, or time
*		 data does not exist in the object, then this function
*		 will return FALSE.
*
*         Input: object  - data object to be queried.
*
*        Output: none
*
*       Returns: TRUE (1) if the time segment exists, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley & John Salas
*          Date: Sep 20, 1993 08:33
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kpds_query_time(
*		!   kobject object)
*
*************************************************************/
int
kpds_query_time(kobject object)
{
   return (kpds_init() && kdms_query_segment(object, KPDS_SEGMENT_TIME));
}

/************************************************************
*
*  Routine Name: kpds_destroy_time - destroy the time segment in a
*				     data object.
*
*       Purpose: This function is used to destroy the time segment
*		 contained within an object.  Once the time segment
*		 has been destroyed, any data or attributes associated
*		 with the time data will be lost forever.  A new
*		 time segment can be created in its place with the
*		 function kpds_create_time.
*
*		 If the time segment does not exist in the specified
*		 object, it is considered to be an error.
*
*         Input: object  - object from which to remove the time segment.
*
*        Output: time segment is destroyed
*
*       Returns: TRUE (1) if the time segment is successfully destroyed, 
*		 FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley & John Salas
*          Date: Sep 20, 1993 08:33
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kpds_destroy_time(
*		!   kobject object)
*
*************************************************************/
int
kpds_destroy_time(kobject object)
{
   return (kpds_init() && kdms_destroy_segment(object, KPDS_SEGMENT_TIME));
}

/************************************************************
*
*  Routine Name: kpds_initialize_time - preset time segment to constant
*
*       Purpose: This routine initializes the entire time segment to
*		 a specified value.  The last two arguments are real
*		 and imaginary components of a complex number that is
*		 the value that the data will be initialized to.  The
*		 data type of the time segment at the time this
*		 function is called will determine how the complex
*		 number is used.  A non-complex datatype means that
*		 the real part only will be used.  A
*		 non-floating-point data type means that the integer
*		 part of the real argument will be used.
*
*         Input: object - object to initialize time data on
*		 real   - real part of constant
*		 imag   - imaginary part of constant
*
*        Output: time segment is modified in place
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Jeremy Worley
*          Date: Aug 03, 1994 14:53
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kpds_initialize_time(
*                !   kobject object,
*                !   double  real,
*                !   double  imag)
*
*************************************************************/
int
kpds_initialize_time(
   kobject object, 
   double  real, 
   double  imag)
{
   return (kpds_init() && 
	   kdut_initialize_segment(object, KPDS_SEGMENT_TIME, real, imag));
}

/************************************************************
*
*  Routine Name: kpds_copy_time_data - copy all time data from
*                                      one object to another object.
*
*       Purpose: This function copies all of the data contained in the
*		 time segment of one object into the time segment
*		 contained in another object.  If the time segment
*		 exists in the destination object, then its data will
*		 be replaced with the data from the source object.  If
*		 the time segment does not exist in the destination
*		 object, it will be created.
*
*		 The time data can be optionally copied through the
*		 presentations of the two data objects.  This implies
*		 that any presentation stages which are normally
*		 invoked during a kpds_get_data on the source object
*		 or a kpds_put_data call on the destination object
*		 will be used for the copy.
*
*		 Any of the following presentation stages can be
*		 invoked on either the source object or the
*		 destination object : casting, scaling, normalizing,
*		 padding, and interpolating.
*		 
*		 The following presentation stages can be invoked on
*		 only the source object : position, and offset. The
*		 resulting copy will appear to be shifted by the
*		 source position and offset.
*
*		 These presentation stages are brought into the data
*		 pipeline by setting the appropriate presentation
*		 attributes on the data objects. Please see the
*		 chapter on polymorphic data services in the data
*		 services manual for more information on the various
*		 presentation attributes.
*
*		 The following example may help in visualizing the
*		 process.  The source object has a presentation data
*		 type different from the physical data type and a
*		 presentation size different from the physical size.
*		 The destination object only has a different
*		 presentation and physical data type.
*
*		 !	        _______________________
*		 !	       | _____________________ |
*		 !	  _____||______  ---->        ||
*		 !	 |interpolating|              ||
*		 !	  _____||______          _____||______
*		 !	 |___casting___|        |___casting___|
*		 !	   ____||_____            ____||_____
*		 !	  |           |          |           |
*		 !	  |  source   |        	 |destination|
*		 !	  |   data    |        	 |   data    |
*		 !	  |___________|        	 |___________|
*	         !
*
*		 In the above example, the data resulting from the
*		 copy will be interpolated to a new size and cast to a
*		 new data type before being copied into the
*		 destination object.  When being copied into the
*		 destination object, the data will be cast yet again
*		 before it finally is stored.
*
*		 If the presentation and physical layers of the
*		 destination object are coupled then the destination
*		 attributes will be ignored and the source
*		 presentation attributes will be propogated to the
*		 destination physical layer.  The copy, in essence,
*		 will be performed only through the source
*		 presentation, with the destination physical and
*		 presentation layers taking on the characteristics of
*		 the source presentation.  By default, output objects
*		 are not coupled, so this behavior is typical.
*		 
*		 The copy need not be performed using the presentation.
*		 If the data is not copied through the presentation,
*		 the destination data will be a direct copy of the
*		 physical source data.  The physical size and data
*		 type of the time segment will be reflected in the
*		 destination data object from the source to the
*		 destination since they describe the physical state of
*		 the data.
*
*         Input: object1 - the object that serves as a source for the data.
*
*		 copy_through_presentation - if set to TRUE, the copy will
*			 		     be performed through the
*				  	     presentation of the source 
*					     and destination objects.
*					     if set to FALSE, the copy will
*					     be a direct copy of the physical
*					     data.
*
*        Output: object2 - the object that will serve as a destination for
*			   the copy operation.
*
*       Returns: TRUE (1) if copy was successful, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Jeremy Worley and Steve Kubica
*          Date: Jan 2, 1995
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kpds_copy_time_data(
*               !   kobject object1,
*               !   kobject object2,
*		!   int     copy_through_presentation)
*
*************************************************************/
int
kpds_copy_time_data(
   kobject object1,
   kobject object2,
   int     copy_through_presentation)
{
   kobject src = NULL;
   kobject dest = object2;
   
   int t;
   int ot;
   int tps;
   int tof;
   int i;
   int typ;
   int num_regions;
   int old_coupling;
   kaddr data = NULL;
   int order[KPDS_TIME_DIM] = {KTIME};

   if (!kpds_init())
      return FALSE;

   /* -- sanity check -- */
   if (object1 == NULL || object2 == NULL)
   {
      /* -- set invalid object errorno here -- */
      return FALSE;
   }

   if (!copy_through_presentation)
      return kdut_copy_segment_data(object1, object2, KPDS_SEGMENT_TIME);

   /* -- make sure time exists in the source -- */ 
   if (kdms_query_segment(object1, KPDS_SEGMENT_TIME))
   {
      /* -- set cool errno here -- */
      return FALSE;
   }

   /* -- make sure the segment exists in the destination -- */
   if (!kdms_query_segment(object2, KPDS_SEGMENT_TIME))
      if (!kdms_create_segment(object2, KPDS_SEGMENT_TIME))
	 return FALSE;

   /* -- get the position and offset before we make the reference -- */
   if (!kpds_get_attributes(object1,
			    KPDS_TIME_POSITION, &tps, 
			    KPDS_TIME_OFFSET,   &tof, 
			    NULL))
      return FALSE;

   /* -- create a reference to avoid side effects on the source -- */
   src  = kdms_reference(object1);

   /* -- get the old coupling on the dest so we can restore it later -- */
   if (!kdms_get_attribute(dest, KPDS_SEGMENT_TIME, 
			         KDMS_COUPLING, &old_coupling))
      return FALSE;
   
   /* -- want to affect physical layer on the destination -- */
   if (!kdms_set_attribute(dest, KPDS_SEGMENT_TIME, KDMS_COUPLING, KCOUPLED))
      return FALSE;

   /* -- the resulting copy will have the default index order -- */
   kdms_set_attribute(dest, KPDS_SEGMENT_TIME, KDMS_INDEX_ORDER, order);

   /* -- set up region sizes - also copy size and data type -- */
   if (!kpds_get_attributes(src,
			    KPDS_TIME_SIZE, &t,
			    KPDS_TIME_DATA_TYPE, &typ,
			    KPDS_TIME_OPTIMAL_REGION_SIZE, &ot, &num_regions, 
			    NULL))
      return FALSE;
   if (!kpds_set_attributes(src,
			    KPDS_TIME_POSITION, tps, 
			    KPDS_TIME_OFFSET,   tof,
			    KPDS_TIME_REGION_SIZE, ot,
			    NULL))
      return FALSE;
   if (!kpds_set_attributes(dest,
			    KPDS_TIME_SIZE, t,
			    KPDS_TIME_DATA_TYPE, typ,
			    KPDS_TIME_REGION_SIZE, ot,
			    KPDS_TIME_POSITION, 0, 
			    KPDS_TIME_OFFSET, 0, 
			    NULL))
      return FALSE;

   /* -- copy the data using the pds get'n'put data calls -- */
   for (i = 0; i < num_regions; i++)
   {
      data = kpds_get_data(src, KPDS_TIME_REGION, data);
      kpds_put_data(dest, KPDS_TIME_REGION, data);
   }
   kfree(data);
   
   /* -- reset the position and offset to be zero -- */
   if (!kpds_set_attributes(dest, 
			    KPDS_TIME_POSITION, 0, 
			    KPDS_TIME_OFFSET, 0, 
			    NULL))
      return FALSE;

   /* -- set the old coupling back on the segment -- */
   kdms_set_attribute(dest, KPDS_SEGMENT_TIME, KDMS_COUPLING, old_coupling);

   /* -- close the source reference -- */
   kdms_close(src);

   return TRUE;
}

/************************************************************
*
*  Routine Name: kpds_copy_time_attr - copy all time attributes from
*	                               one object to another object.
*
*       Purpose: This function copies all the time attributes from 
*		 one object to another object.
*
*		 If the destination object does not contain a time
*		 segment, then this function will create a time
*		 segment, and initialize its size and data type
*		 attributes to those in the source object.  If the
*		 time data already exists in the destination object,
*		 then the presentation attributes will be set to the
*		 source object's settings.
*
*		 The destination's physical attributes will change
*		 depending on the coupling of the data objects.
*		 If the destination is coupled, then the physical
*		 attributes will be changed as well.  Any data
*		 contained in the destination will be cast, rescaled,
*		 or resized to match its new physical attributes.
*
*         Input: source_object - the object that serves as a
*				 source for the attributes.
*
*        Output: destination_object - the object that serves as
*				      a destination for the
*				      operation.
*
*       Returns: TRUE (1) if copy was successful, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley & John Salas
*          Date: Sep 19, 1993 15:43
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kpds_copy_time_attr(
*		!   kobject object1,
*		!   kobject object2)
*
*************************************************************/
int
kpds_copy_time_attr(
   kobject object1,
   kobject object2)
{
   return (kpds_init() && 
	   kdut_copy_segment_attr(object1, object2, KPDS_SEGMENT_TIME));
}

/************************************************************
*
*  Routine Name: kpds_copy_time - copy the time segment from
*		                  one object to another.
*
*       Purpose: This function copies all of the attributes and data
*		 contained in the time segment of one object into the
*		 time segment contained in another object.  If the
*		 time segment exists in the destination object, then
*		 its data will be replaced with the data from the
*		 source object.  If the time segment does not exist
*		 in the destination object, it will be created.
*
*		 All time segment attributes will be copied from one
*		 object to the other.  This includes presentation
*		 attributes, such as position and offset.
*
*		 The time data can be optionally copied through the
*		 presentations of the two data objects.  This implies
*		 that any presentation stages which are normally
*		 invoked during a kpds_get_data on the source object
*		 or a kpds_put_data call on the destination object
*		 will be used for the copy.
*
*		 Any of the following presentation stages can be
*		 invoked on either the source object or the
*		 destination object : casting, scaling, normalizing,
*		 padding, and interpolating.
*		 
*		 The following presentation stages can be invoked on
*		 only the source object : position, and offset. The
*		 resulting copy will appear to be shifted by the
*		 source position and offset.
*
*		 These presentation stages are brought into the data
*		 pipeline by setting the appropriate presentation
*		 attributes on the data objects. Please see the
*		 chapter on polymorphic data services in the data
*		 services manual for more information on the various
*		 presentation attributes.
*
*		 The following example may help in visualizing the
*		 process.  The source object has a presentation data
*		 type different from the physical data type and a
*		 presentation size different from the physical size.
*		 The destination object only has a different
*		 presentation and physical data type.
*
*		 !	        _______________________
*		 !	       | _____________________ |
*		 !	  _____||______  ---->        ||
*		 !	 |interpolating|              ||
*		 !	  _____||______          _____||______
*		 !	 |___casting___|        |___casting___|
*		 !	   ____||_____            ____||_____
*		 !	  |           |          |           |
*		 !	  |  source   |        	 |destination|
*		 !	  |   data    |        	 |   data    |
*		 !	  |___________|        	 |___________|
*	         !
*
*		 In the above example, the data resulting from the
*		 copy will be interpolated to a new size and cast to a
*		 new data type before being copied into the
*		 destination object.  When being copied into the
*		 destination object, the data will be cast yet again
*		 before it finally is stored.
*
*		 If the presentation and physical layers of the
*		 destination object are coupled then the destination
*		 attributes will be ignored and the source
*		 presentation attributes will be propogated to the
*		 destination physical layer.  The copy, in essence,
*		 will be performed only through the source
*		 presentation, with the destination physical and
*		 presentation layers taking on the characteristics of
*		 the source presentation.  By default, output objects
*		 are not coupled, so this behavior is typical.
*		 
*		 The copy need not be performed using the presentation.
*		 If the data is not copied through the presentation,
*		 the destination data will be a direct copy of the
*		 physical source data.  The physical size and data
*		 type of the time segment will be reflected in the
*		 destination data object from the source to the
*		 destination since they describe the physical state of
*		 the data.
*
*		 This function is equivalent to performing successive
*		 calls to kpds_copy_time_attr and kpds_copy_time_data.
*
*         Input: object1 - the object that serves as a source for the time
*			   segment.
*
*		 copy_through_presentation - if set to TRUE, the copy will
*			 		     be performed through the
*				  	     presentation of the source 
*					     and destination objects.
*					     if set to FALSE, the copy will
*					     be a direct copy of the physical
*					     data.
*
*        Output: object2 - the object that will serve as a destination for
*			   the copy operation.
*
*       Returns: TRUE (1) if copy was successful, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Jeremy Worley & Steve Kubica
*          Date: Sep 19, 1993 15:40
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kpds_copy_time(
*               !   kobject object1,
*               !   kobject object2,
*		!   int     copy_through_presentation)
*
*************************************************************/
int
kpds_copy_time(
   kobject object1,
   kobject object2,
   int     copy_through_presentation)
{
   return (kpds_init() && 
	   kpds_copy_time_attr(object1, object2) &&
	   kpds_copy_time_data(object1, object2, copy_through_presentation));
}
