 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Geometry Data Services
   >>>>
   >>>>   Static:
   >>>>
   >>>>  Private:
   >>>>             
   >>>>   Public:
   >>>>			kgeom_open_input_object()
   >>>>			kgeom_open_output_object()
   >>>>			kgeom_create_object()
   >>>>			kgeom_get_data()  
   >>>>			kgeom_put_data()  
   >>>>			kgeom_create_primitive_list()
   >>>>			kgeom_query_primitive_list() 
   >>>>			kgeom_destroy_primitive_list()
   >>>>			kgeom_set_attribute()  
   >>>>			kgeom_set_attributes() 
   >>>>			kgeom_get_attribute()  
   >>>>			kgeom_get_attributes()  
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "geom_internals.h"

/************************************************************
*
*  Routine Name: kgeom_open_object - create an object associated
*  		       		     with an input or output transport
*
*       Purpose: This function is used to instantiate a data object
*                (\f(CWkobject\fP) that is associated with a permanent
*                file or transport.  If a permanent file is not
*                desired (i.e. the object is going to be used as
*                temporary storage, and will not be used by any other
*                process) then the \f(CWkgeom_create_object()\fP
*                function call should be used instead.
*
*                The first argument to this function is the transport
*                or file name.  This argument indicates the name of
*                the transport that is associated with the object.
*                The transport name can be any legal khoros transport
*                description.  While this will usually be a regular
*                Unix file name, it is also possible to specify such
*                things as shared memory pointers, memory map
*                pointers, sockets, streams, and even transports on
*                other machines.  The construction of transport
*		 descriptions will be explained later.
*
*                The second argument to this function is a flags
*                argument which indicates how then the object is going
*                to used.  The flags argument is analogous to the
*                flags for kopen.  The following flags are available :
*
*                \f(CWKOBJ_READ\fP : 
*                Open an existing file or transport for reading.  By
*                using this flag, you are indicating that the file or
*                transport exists and that it contains valid data.  If
*                it does not exist, then an error message will be
*                generated and this function will return
*                \f(CWKOBJECT_INVALID\fP.
*        	 
*                \f(CWKOBJ_WRITE\fP  :
*                Open a file or transport for writing.  By using
*                this flag, you are indicating that any data that
*                you write to the object will be stored in the 
*                file or transport specified.
*        
*		 This simple example illustrates how an input data
*		 object would be opened.
*	
*  !  \f(CWkobject obj;\fP
*  !
*  !  \f(CWobj = kgeom_open_object("existing.viff",KOBJ_READ);\fP
*
* 		 It is possible to write data to an input object.  If
* 		 data is written to an input object, then subsequent
* 		 calls to get data will contain the new data.
* 		 However, the file or transport that is associated
* 		 with this input object will not be changed.  The
* 		 \f(CWKOBJ_READ\fP and \f(CWKOBJ_WRITE\fP flags only
* 		 indicate what operations are allowed on the permanent
* 		 file or transport that is associated with the object,
* 		 not what operations are allowable on the object
* 		 itself.
*
*                If \f(CWKOBJ_READ\fP is specified, then data services
*                will attempt to recognize the file format
*                automatically.  If it fails, then this function will
*                return \f(CWKOBJECT_INVALID\fP, indicating that it
*                was unable to open the object.
*
*   	         Khoros transport names are constructed using a 
*		 transport token and an identifier.  The transport name
*		 simply a string consisting of "token=identifier".
*
*		 The following tokens are available :
*
* !   \fIfile\fP   (Unix file  :  permanent storage)
* !   \fIpipe\fP   (Unix pipe  :  no permanent storage)
* !   \fImmap\fP   (virtual memory  :  permanent storage)
* !   \fIshm\fP    (shared memory  :  permanent storage)
* !   \fIsocket\fP (sockets  :  no permanent storage)
* !   \fIstream\fP (named stream/pipe  :  no permanent storage)
*
*                If no token is provided, the file transport is used.  
*		 The identifier is specific to the type of transport;
*		 for files, it is simply a filename, for a socket, it is
*		 the socket number.  See \f(CWkopen\fP for more 
*		 information.
*
*		 Only the file transport is reliably supported at this time.
*
*         Input: name   - a string that contains the path name
*                         of a file or transport that will be
*                         associated with the object
*
*                flags  - how the object is to be opened
*
*        Output: none
*       Returns: \f(CWkobject\fP on success, 
*		 \f(CWKOBJECT_INVALID\fP on failure
*
*  Restrictions: 
*
*    Written By: Steve Kubica
*          Date: Sep 16, 1993 17:22
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: kobject kgeom_open_object(
*		!   char *name,
*		!   int   flags)
*
*************************************************************/
kobject
kgeom_open_object(
   char *name, 
   int   flags)
{
   kobject obj;
   int wid;
   int hgt;
   int dep;
   

   KGEOM_INIT;

   obj = kdms_open(name, flags);

   if (obj == NULL)
      return KOBJECT_INVALID;

   /* add a close hook to transcribe the primitive manifest */
   kdms_close_hook(obj, kgeom_close_hook);

   if (flags & KOBJ_READ)
   {
      /* if no primitive list, see what we have available */
      if (!kgeom_query_primitive_list(obj))
      {
	 /* meshes must have a location segment */
	 if (kdms_query_segment(obj, KDMS_SEGMENT_LOCATION))
	 {
	    kgeom_create_primitive_list(obj);
	    kgeom_set_attribute(obj, KGEOM_OBJECT, KGEOM_NUMBER_PRIMITIVES, 1);


	    kpds_get_attribute(obj, KPDS_LOCATION_SIZE, 
			       &wid, &hgt, &dep, NULL);

	    if (dep == 1)
	       kgeom_set_attribute(obj, KGEOM_QUADMESH, 
				   KGEOM_QUADMESH_SIZE, wid, hgt);
	    else
	       kgeom_set_attribute(obj, KGEOM_OCTMESH, 
				   KGEOM_OCTMESH_SIZE, wid, hgt, dep);
	 }

         /* well maybe we have a texture */
	 else if (kdms_query_segment(obj, KDMS_SEGMENT_VALUE))
	 {
	    kgeom_create_primitive_list(obj);
	    kgeom_set_attribute(obj, KGEOM_OBJECT, KGEOM_NUMBER_PRIMITIVES, 1);

	    kpds_get_attribute(obj, KPDS_VALUE_SIZE, 
			       &wid, &hgt, &dep, NULL, NULL);

	    if (dep == 1)
	       kgeom_set_attribute(obj, KGEOM_TEXTURE2D, 
				   KGEOM_TEXTURE2D_SIZE, wid, hgt);
	    else
	       kgeom_set_attribute(obj, KGEOM_TEXTURE3D, 
				   KGEOM_TEXTURE3D_SIZE, wid, hgt, dep);
	 }
      }
   }
   

   if (flags & KOBJ_WRITE)
   {
      if (!kcolor_set_attribute(obj, KCOLOR_COLORSPACE, 
				KGEOM_COLORSPACE_DEFAULT))
      {
	 kdms_close(obj);
	 return NULL;
      }

      /* -- set a geometry magic cookie so I can id this later -- */
      if (!kdms_create_attribute(obj, NULL, KGEOM_MAGIC_COOKIE, 
				 1, 1, KINT, TRUE, TRUE))
      {
	 kdms_close(obj);
	 return NULL;
      }
      
      if (!kcolor_set_attribute(obj, KGEOM_MAGIC_COOKIE, TRUE))
      {
	 kdms_close(obj);
	 return NULL;
      }
   }

   return obj;
}

/************************************************************
*
*  Routine Name: kgeom_open_input_object - open an object associated 
*		                           with an input data transport.
*
*       Purpose: This function is a simplified interface to the
*		 \f(CWkgeom_open_object()\fP function.  It differs in
*		 that it assumes that the object is read-only and its
*		 transport has permanence.  If a permanent file is not
*		 desired (i.e.  the object is going to be used as
*		 temporary storage, and will not be used by any other
*		 process) then the \f(CWkgeom_create_object()\fP
*		 function call should be used instead.
*
*		 The argument to this function is the transport or
*		 file name.  This argument indicates the name of the
*		 transport that is associated with the object.  The
*		 transport name can be any legal khoros transport
*		 description.  While this will usually be a regular
*		 Unix file name, it is also possible to specify such
*		 things as shared memory pointers, memory map
*		 pointers, sockets, streams, and even transports on
*		 other machines.  For additional information, please
*                see \f(CWkgeom_open_object()\fP.
*
*		 This simple example illustrates how an input data
*		 object would be opened.
*	
*  !  \f(CWkobject obj;\fP
*  !
*  !  \f(CWobj = kgeom_open_input_object("existing.viff");\fP
*
*		 This function is equivalent to:
*	!	\f(CWkgeom_open_object(name, KOBJ_READ)\fP
*
*	  Input: name - a string that contains the path name
*			of a file or transport that will be 
*			associated with the object
*
*        Output: none
*
*       Returns: a \f(CWkobject\fP on success, \f(CWKOBJECT_INVALID\fP
*                on failure
*
*  Restrictions:
*
*    Written By: Steve Kubica
*          Date: Sep 16, 1993 17:14
*      Verified:
*  Side Effects: 
* Modifications: 
* 
*   Declaration: kobject kgeom_open_input_object( 
*		!   char *name) 
* 
*************************************************************/
kobject
kgeom_open_input_object(char *name)
{
   kobject obj;


   KGEOM_INIT;

   if ((obj = kgeom_open_object(name, KOBJ_READ)) == NULL)
      return NULL;

   return obj;
}

/************************************************************
*
*  Routine Name: kgeom_open_output_object - open an object associated 
*		                            with an output transport.
*
*       Purpose: This function is a simplified interface to the
*		 \f(CWkgeom_open_object()\fP function.  It differs in
*		 that it assumes that the object is write-only and its
*		 transport has permanence.  If a permanent file is not
*		 desired (i.e.  the object is going to be used as
*		 temporary storage, and will not be used by any other
*		 process) then the \f(CWkgeom_create_object()\fP
*		 function call should be used instead.
*
*		 The argument to this function is the transport or
*		 file name.  This argument indicates the name of the
*		 transport that is associated with the object.  The
*		 transport name can be any legal khoros transport
*		 description.  While this will usually be a regular
*		 Unix file name, it is also possible to specify such
*		 things as shared memory pointers, memory map
*		 pointers, sockets, streams, and even transports on
*		 other machines.  For additional information, please
*                see \f(CWkgeom_open_object()\fP.
*
*		 This simple example illustrates how an input data
*		 object would be opened.
*	
*  !  \f(CWkobject obj;\fP
*  !
*  !  \f(CWobj = kgeom_open_output_object("brand_new.viff");\fP
*
*		 This function is equivalent to:
*	!	\f(CWkgeom_open_object(name, KOBJ_WRITE)\fP
*
*	  Input: name - a string that contains the path name
*			of a file or transport that will be 
*			associated with the object
*
*        Output: none
*
*       Returns: a \f(CWkobject\fP on success, \f(CWKOBJECT_INVALID\fP
*                on failure
*
*  Restrictions:
*
*    Written By: Steve Kubica
*          Date: Sep 16, 1993 17:14
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: kobject kgeom_open_output_object(
*		!   char *name)
*
*************************************************************/
kobject
kgeom_open_output_object(char *name)
{
   kobject obj;


   KGEOM_INIT;

   if ((obj = kgeom_open_object(name, KOBJ_WRITE)) == NULL)
      return NULL;

   return obj;
}

/************************************************************
*
*  Routine Name: kgeom_create_object - create a temporary data object.
*
*       Purpose: This function is used to instantiate a data object
*                (\f(CWkobject\fP) when it will only be used for
*                temporary storage of information.  If you are
*                intending to process an object that already exists as
*                a file or transport (input), or you are planning on
*                saving the object to a file or transport (output),
*                then the appropriate routines to use are
*                \f(CWkgeom_open_input_object()\fP,
*                \f(CWkgeom_open_output_object()\fP, or
*                \f(CWkgeom_open_object()\fP.
*
*                This function call creates what is essentially an
*                empty object.  That is, the object will initially
*                have no data and all global attributes will be
*                initialized to default values.  If a default is not
*                appropriate, then the attribute will be
*                uninitialized.  An object that is created with this
*                function call behaves similarly to an output object
*                that is created with the
*                \f(CWkgeom_open_output_object()\fP function call.
*
*		 This simple example illustrates how a data object
*		 would be created.
*	
*  !  \f(CWkobject obj;\fP
*  !
*  !  \f(CWobj = kgeom_create_object();\fP
*
*                For large data sets, a temporary transport will be
*                created in order to buffer the data out of memory.
*                This temporary transport will be automatically
*                removed when the process terminates. 
*
*         Input: none
*
*        Output: none
*
*       Returns: \f(CWkobject\fP on success, \f(CWKOBJECT_INVALID\fP
*       	 on failure
*
*  Restrictions:
*    Written By: Jeremy Worley & John Salas
*          Date: Sep 16, 1993 17:14
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: kobject kgeom_create_object(
*		!   void)
*
*************************************************************/
kobject
kgeom_create_object(void)
{
   kobject obj;


   KGEOM_INIT;

   if ((obj = kdms_create()) == NULL)
      return NULL;

   if (!kcolor_set_attribute(obj, KCOLOR_COLORSPACE, KGEOM_COLORSPACE_DEFAULT))
   {
      kdms_close(obj);
      return NULL;
   }


   /* -- set a geometry magic cookie so I can id this later -- */
   if (!kdms_create_attribute(obj, NULL, KGEOM_MAGIC_COOKIE, 
			      1, 1, KINT, TRUE, TRUE))
   {
      kdms_close(obj);
      return NULL;
   }
   
   if (!kcolor_set_attribute(obj, KGEOM_MAGIC_COOKIE, TRUE))
   {
      kdms_close(obj);
      return NULL;
   }
   
   return obj;
}

/************************************************************
*
*  Routine Name: kgeom_get_data - get data from a data object.
*
*       Purpose: This function is used to retrieve the data associated 
*                with a given primitive from a data object.   
*
*                Geometry primitives typically consist of multiple
*                data components, with different primitives having
*                different data components associated with them.
*
*		 These data components will be returned via the provided
*		 data arguments. The data arguments can be thought of
*		 as both input and output arguments.  As input, these
*		 arguments dictate whether or not space needs to be
*		 allocated for the returning data.  As output, these
*		 arguments will point to the blocks of data being
*		 returned.
*
*                Specifically, if the data arguments are pointing to NULL,
*                then memory will be allocated to store the requested data
*                If these arguments do not point to NULL, then this function
*		 will assume that the data arguments point to a sufficient 
*                amount of memory for returning the requested data. 
*
*                For example, the following call will retrieve the data
*                contained from a geometry object containing spheres as
*                well as allocate space for the data being returned.
*
* ! \f(CWfloat *locations = NULL;\fP
* ! \f(CWfloat *colors    = NULL;\fP
* ! \f(CWfloat *radii     = NULL;\fP
* !
* ! \f(CWkgeom_get_data(object,KGEOM_SPHERES,&locations,&colors,&radii);\fP
*
*                The amount of data being returned for each primitive
*                component is determined by a number of object level
*                attributes, but is primarily based on the setting
*                of the primitive-level attribute
*                \f(CWKGEOM_NUMBER_OF_VERTICES\fP.
*
*		 If NULL is passed in for a data argument, that data
*		 data component will be skipped.
*		 
*		 Some object level attributes may be set to affect the
*		 presentation of the returned data without affecting
*		 the stored data itself.  For example, setting the
*		 object level attribute
*		 \f(CWKGEOM_LOCATION_DATA_TYPE\fP to \f(CWKFLOAT\fP
*		 will cause all location data to be returned as
*		 floating point numbers regardless of how the data may
*		 be physically stored within the data object.
*
*                A complete list of geometry attributes and primitives
*                can be found in Chapter 3 of Programming Services
*                Volume II.
*
*         Input: object    - the data object that is the source of  
*                            the requested data
*
*                primitive - the primitive to retrieve the data from. 
*                            The data will be retrieved from the primitive
*			     located at the position indicated by the
*			     \f(CWKGEOM_PRIMITIVE_POSITION\fP attribute.
*			     If the entry in the primitive list at this
*			     position does not match this primitive, an
*			     error will result.
*
*        Output: kvalist   - a variable argument list that contains a list
*                            of data pointers to be used for returning the 
*                            various data segments associated with a 
*                            primitive.  If there is no data available for
*                            a segment on the requested primitive, then 
*                            the pointer argument for that segment will point
*                            to NULL.  The variable argument list takes 
*                            the form:
*
*                            !  \f(CW&data1, &data2, ... \fP
*
*                            The number of data pointers in the
*                            variable argument list for each primitive
*                            depends on the primitive.
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise 
*
*  Restrictions: This function assumes that if the pointers to be used
*                for returning the data are not pointing to NULL, then 
*                they are pointing to a preallocated block of memory with 
*                sufficient space for the amount of data which is being 
*                returned.  If they are not, a segmentation fault will
*                probably occur. 
*
*    Written By: Steven Kubica
*          Date: Feb 04, 1993 16:28
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: kobject kgeom_get_data(
*		!   kobject object,
*		!   int     primitive,
*		!   kvalist)
*
*************************************************************/
int 
kgeom_get_data(
   kobject object, 
   int     primitive, 
   kvalist)
{
   kva_list list;
   int status;


   KGEOM_INIT;

   if (object == NULL)
      kgeom_fail_error(KGEOM_EOBJ_INVALID);

   /* grab the variable argument list */
   kva_start(list, primitive);

   if (primitive < KGEOM_MINIMUM_GEOMETRIC_PRIMITIVE)
      status = kgeom_get_mesh(object, primitive, &list);
   else
      status = kgeom_get_primitive(object, primitive, &list);      
   

   /* close down the variable argument list */
   kva_end(list);

   return status; 
}


/************************************************************
*
*  Routine Name: kgeom_put_data - put data into a data object.
*
*       Purpose: This function is used to put data for a given
*                primitive into a data services object.
*
*                Geometry primitives typically consist of multiple
*                data segments, with different primitives having
*                different data segments associated with them. These
*                data components are provided to this function via
*                multiple data arguments.
*
*                For example, the following call will assign the data
*                a list of sphere data to a geometry primitive.
*
* ! \f(CWfloat *locations = (float *) get_some_location_data();\fP
* ! \f(CWfloat *colors    = (float *) get_some_color_data();\fP
* ! \f(CWfloat *radii     = (float *) get_some_radii_data();\fP
* !
* ! \f(CWkgeom_put_data(object,KGEOM_SPHERES,locations,colors,radii);\fP
*
*                Note that data does not have to be provided for all
*                segments.  In such cases, a NULL may be provided as
*                the data argument.  For example, to put a spheres
*                primitive with no color data, the following call
*                would be used.
*
* ! \f(CWkgeom_put_data(object,KGEOM_SPHERES,locations,NULL,radii);\fP
*
*                The amount of data and data type expected for each
*                primitive component is determined by a number of
*                object level attributes.  The primitive-level attribute
*                \f(CWKGEOM_NUMBER_OF_VERTICES\fP should be set before
*		 any data is put to indicate how much data is going to 
*		 be put.  This is not true for the mesh primitives, 
*		 which each have their different size attributes.
*            
*                A complete list of geometry attributes and primitives
*                can be found in Chapter 3 of Programming Services
*                Volume II.
*
*         Input: object    - the data object that is the destination for 
*                            the provided data
*
*                primitive - the primitive to assign the data to. 
*                            The data will be assigned to the 
*                            this primitive located at the
*                            position indicated by the 
*			     \f(CWKGEOM_PRIMITIVE_POSITION\fP attribute.
*			     If the entry in the primitive list at this
*			     position does not match this primitive, an
*			     error will result.
*                   
*                kvalist   - a variable argument list that contains a list
*                            of pointers to data to assign to the different 
*			     components.  Data components may be left 
*			     "empty" by passing in NULL.
*
*			     The variable argument list takes the form: 
*
*                            !  \f(CWdata1, data2, data3, ... \fP
*
*                            The number of data arguments in the
*                            variable argument list for each attribute
*                            depends on the primitive.
*
*        Output: None 
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: This function assumes that a sufficient number of pointers
*                are given with the variable argument list to supply data
*                for all segments of the primitive.  If they are not, 
*                unpredictable behavior may occur.  Also, once a primitive
*		 has been put, it can not be "re-put".  It may be destroyed
*		 along with the rest of the primitive list with a call
*		 to \f(CWkgeom_destroy_primitive_list()\fP.
*
*    Written By: Steven Kubica
*          Date: Feb 04, 1993 16:27
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: kobject kgeom_put_data(
*		!   kobject object,
*		!   int     primitive,
*		!   kvalist)
*
*************************************************************/
int 
kgeom_put_data(
   kobject object,
   int     primitive,
   kvalist)
{
   kva_list list;
   int status;


   KGEOM_INIT;

   if (object == NULL)
      kgeom_fail_error(KGEOM_EOBJ_INVALID);

   /* Grab the variable argument list */
   kva_start(list, primitive);

   if (primitive < KGEOM_MINIMUM_GEOMETRIC_PRIMITIVE)
      status = kgeom_put_mesh(object, primitive, &list);
   else
      status = kgeom_put_primitive(object, primitive, &list);      

   /* close down the variable argument list */
   kva_end(list);

   return status;
}

/************************************************************
*
*  Routine Name: kgeom_create_primitive_list - create a primitive list 
*					       on a data object.
*
*       Purpose: This function is used to instantiate an empty 
*                primitive list within a data services object.
*		 Geometry Service primitives are stored and
*                retrieved from this primitive list.
*		
*		 Once created, the primitive list is accessed via the
*		 \f(CWKGEOM_PRIMITIVE_LIST\fP attribute.  The length
*		 of the primitive list is determined by the
*		 \f(CWKGEOM_NUMBER_PRIMITIVES\fP attribute.
*	
*		 The Geometry Services attribute and data functions
*		 will always operate on the current primitive in the
*		 primitive list.  The current primitive is dictated
*		 by the \f(CWKGEOM_PRIMITIVE_POSITION\fP attribute.
*		 
*		 The following example illustrates how this call would
*		 be used to create a primitive list on an output
*		 object.  In this example, a primitive list with two
*		 primitives, a spheres list and a disjoint polyline, is
*		 created.
*
*  !  \f(CWkobject obj;\fP
*  !
*  !  \f(CWobj = kgeom_open_output_object("newgeometry.viff");\fP
*  !  \f(CWkgeom_create_primtive_list(obj);\fP
*  !  \f(CWkgeom_set_attribute(obj, KGEOM_OBJECT,\fP
*  !  \f(CW                         KGEOM_NUMBER_PRIMITIVES, 2);\fP
*  !  \f(CWkgeom_put_data(obj, KGEOM_SPHERES, loc, NULL, rad);\fP
*  !  \f(CWkgeom_put_data(obj, KGEOM_POLYLINE_DISJOINT, loc, NULL);\fP
*
*		 As geometry primitive data is put into the data object
*		 with put data calls, the primitive position will be
*		 automatically incremented.
*
*         Input: object   - the data object in which to
*                           create the primitive list 
*
*        Output: none
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Steven Kubica 
*          Date: Dec 10, 1993 16:17
*      Verified:
*  Side Effects: 
* Modifications:
*
*   Declaration: kobject kgeom_create_primitive_list(
*		!   kobject object)
*
*************************************************************/
int 
kgeom_create_primitive_list(kobject object)
{
   int size[1] = {1};
   int ord[1]  = {KWIDTH};
   int beg[1]  = {0};
   int end[1]  = {0};
   int lst[1]  = {0};
   
   kgeom_manifest *manifest;

   KGEOM_INIT;

   if (object == NULL)
      kgeom_fail_error(KGEOM_EOBJ_INVALID);

   /* -- if we already have one, then we're done -- */
   if ((manifest = kgeom_get_manifest(object)) != NULL)
      return TRUE;

   /* -- create manifest -- */
   if (!kdms_query_attribute(object, NULL, KGEOM_MANIFEST, 
			     NULL, NULL, NULL, NULL))
      kdms_create_attribute(object, NULL, KGEOM_MANIFEST, 1, 1, KSTRUCT, 
			    FALSE, TRUE);

   manifest = (kgeom_manifest *) kmalloc(sizeof(kgeom_manifest));
   
   manifest->num_primitives = 0;
   manifest->primitives     = NULL;
   
   if (!kdms_set_attribute(object, NULL, KGEOM_MANIFEST, manifest))
      return FALSE; 

   return TRUE;
}

/************************************************************
*
*  Routine Name: kgeom_query_primitive_list - determine the existence of 
*					      a primitive list.
*
*       Purpose: This function checks for the existence of a primitive
*                list within a given data services object.
*
*		 The primitive list is accessed via the
*		 \f(CWKGEOM_PRIMITIVE_LIST\fP attribute. The length of
*		 the primitive list is determined by the
*		 \f(CWKGEOM_NUMBER_PRIMITIVES\fP attribute.  If a data
*		 object does not contain a primitive list, neither of
*		 these attributes can be accessed.
*
*		 This function typically one of the first functions
*		 called after opening an input data object.  
*
*         Input: object   - the data object to check for 
*                           the existence of a primitive list
*
*        Output: none
*
*       Returns: TRUE (1) if primitive list exists , FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Steven Kubica 
*          Date: Feb 13, 1993 16:17
*      Verified:
*  Side Effects: 
* Modifications:
*   Declaration: kobject kgeom_query_primitive_list(
*		!   kobject object)
*
*************************************************************/
int kgeom_query_primitive_list(kobject object)
{
   KGEOM_INIT;

   if (object == NULL)
      kgeom_fail_error(KGEOM_EOBJ_INVALID);

   return (kgeom_get_manifest(object) != NULL);
}

/************************************************************
*
*  Routine Name: kgeom_destroy_primitive_list - destroy the primitive list
*
*       Purpose: This function destroys a primitive list contained in
*                a given data services object.  All data and attributes
*		 associated with the primitives stored on the primitive
*		 list will be destoryed as well.
*
*         Input: object - the data object which contains the
*                         primitive list to destroy
*
*        Output: none
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: This function may have unpredicatable results if
*		 the primitive list contains a mesh or texture primitive.
*    Written By: Steven Kubica 
*          Date: Feb 13, 1993 16:17
*      Verified:
*  Side Effects: 
* Modifications:
*   Declaration: kobject kgeom_destroy_primitive_list(
*		!   kobject object)
*
*************************************************************/
int 
kgeom_destroy_primitive_list(kobject object)
{
   KGEOM_INIT;

   if (object == NULL)
      kgeom_fail_error(KGEOM_EOBJ_INVALID);

   /* -- maybe should blast all segments here?  // SK -- */

   return kgeom_destroy_manifest(object);
}

/************************************************************
*
*  Routine Name: kgeom_set_attribute - set the values of a geometry
*                                      attribute in a data object.
*
*       Purpose: This function is used to assign the value of an attribute 
*                associated with a data object containing geometry data.
*
*                Attributes can be either global to the object, or
*                specific to a primitive contained within the object.
*		 Attributes which are global are termed \fIobject-level\fP, 
*		 and attributes specific to a primitive are termed
*		 \fIprimitive-level\fP.	    
*
*		 Object-level attributes apply globally to all
*		 primitives contained within a data object. 
*		 Two examples of object-level attributes are
*		 \f(CWKGEOM_NAME\fP, which is the name of the object,
*		 and \f(CWKGEOM_AMBIENT\fP, which dictates how the
*		 geometry should react to ambient light sources.
*
*                Primitive-level attributes apply only to the current
*                primitive as indicated by the
*                \f(CWKGEOM_PRIMITIVE_POSITION\fP attribute. An example
*		 of such an attribute is the \f(CWKGEOM_NUMBER\fP attribute
*		 which determines the number of vertices in the current
*		 primitive.
*
*                Attributes may have multiple components.  For
*                example, the attribute \f(CWKGEOM_BOUNDING_BOX\fP
*                consists of two components, one component being the
*                minimum point of the bounding box, and the second
*                component being the maximum point of the bounding box.
*
*                Some of the geometry attributes have varying data
*                types and sizes depending on the values of other
*                attributes.  For example, the \f(CWKGEOM_CENTER\fP
*                attribute must be set with an array of the data type
*                indicated by the attribute
*                \f(CWKGEOM_LOCATION_DATA_TYPE\fP that contains the
*                number of values indicated by the attribute
*                \f(CWKGEOM_LOCATION_SIZE\fP.
*
*                The following example illustrates the use of a set  
*                attribute call to assign two different attributes.
*
* !   \f(CWfloat min[3] = {0.0, 0.0, 0.0};\fP
* !   \f(CWfloat max[3] = {200.0, 123.0, 300.0};\fP
* !
* !   \f(CWkgeom_set_attribute(object, KGEOM_OBJECT, \fP
* !   \f(CW                            KGEOM_NAME, "Isosurface 1");\fP
* !   \f(CWkgeom_set_attribute(object, KGEOM_OBJECT, \fP
* !   \f(CW                            KGEOM_BOUNDING_BOX, min, max);\fP
*
*                A complete list of geometry attributes and primitives
*                can be found in Chapter 3 of Programming Services
*                Volume II.
*
*         Input: object    - the data object into which the attribute's 
*                            value will be assigned
*
*                primitive - the primitive the attribute is related
*                            to.  Object-level attributes can be set
*                            by by passing \f(CWKGEOM_OBJECT\fP for
*                            this argument.  Primitive-level
*                            attributes can be set by passing in the
*                            appropriate primitive for this argument.
*                            The current primitive is determined by
*                            the \f(CWKGEOM_PRIMITIVE_POSITION\fP attribute.
*			     If the primitive at the current position has
*			     not yet been set, this set attribute call
*			     will set it.
*
*                attribute - the attribute to set
*                
*                kvalist   - a variable argument list that contains the
*                            values that will be assigned to the 
*			     different components associated with
*                            that attribute.
*
*                            The variable argument list takes the form:
*
*                            !  \f(CWATTRIBUTE_NAME, value1 [, value2, ...]\fP
*
*                            The number of value arguments in the variable 
*                            argument list corresponds to the number of
*			     arguments needed to set the attribute.
*        Output: none 
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: Calling this function with an incorrect number of arguments 
*                in the variable argument list will not cause any compiler 
*                errors, but will often generate a segmentation fault.
*
*    Written By: Steven Kubica
*          Date: Feb 13, 1993 15:55
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: kobject kgeom_set_attribute(
*		!   kobject object,
*		!   int     primitive,
*		!   char    *attribute,
*		!   kvalist)
*
*************************************************************/
int 
kgeom_set_attribute(
   kobject object,
   int     primitive,
   char    *attribute,
   kvalist)
{
   kva_list list;
   int      status;
   int      position;
   char    *name = NULL;


   KGEOM_INIT;

   if (object == NULL)
      kgeom_fail_error(KGEOM_EOBJ_INVALID);

   /* start the variable arguments list */
   kva_start(list, attribute);
 
   /* attribute will be at the object level, unless it is listable */
   if (primitive == KGEOM_OBJECT)
   {
     status = kdms_vset_attribute(object, KDMS_OBJECT, attribute, &list);
   }
   else
   {
      /* attribute depends on the primitive position */
      kdms_get_attribute(object, NULL,_INTERNAL_PRIMITIVE_POSITION, &position);

      /* make sure attribute is okay for primitive at primitive position */
      if (!kgeom_verify_manifest_primitive(kgeom_get_manifest(object), 
					   primitive, position, NULL))
         kgeom_fail_error(KGEOM_EINTERNAL);

      status = kdms_vset_attribute(object, KDMS_OBJECT, attribute, &list);
   }

   kva_end(list);
   
   return status;
}

/************************************************************
*
*  Routine Name: kgeom_set_attributes - set the values of a number
*                                       of geometry attributes.
*
*       Purpose: This function is used to assign the values of an 
*		 arbitrary number of attributes associated with a 
*		 data object containing geometry data.
*
*                Attributes can be either global to the object, or
*                specific to a primitive contained within the object.
*		 Attributes which are global are termed \fIobject-level\fP, 
*		 and attributes specific to a primitive are termed
*		 \fIprimitive-level\fP.	    
*
*		 Object-level attributes apply globally to all
*		 primitives contained within a data object. 
*		 Two examples of object-level attributes are
*		 \f(CWKGEOM_NAME\fP, which is the name of the object,
*		 and \f(CWKGEOM_AMBIENT\fP, which dictates how the
*		 geometry should react to ambient light sources.
*
*                Primitive-level attributes apply only to the current
*                primitive as indicated by the
*                \f(CWKGEOM_PRIMITIVE_POSITION\fP attribute. An example
*		 of such an attribute is the \f(CWKGEOM_NUMBER\fP attribute
*		 which determines the number of vertices in the current
*		 primitive.
*
*                Attributes may have multiple components.  For
*                example, the attribute \f(CWKGEOM_BOUNDING_BOX\fP
*                consists of two components, one component being the
*                minimum point of the bounding box, and the second
*                component being the maximum point of the bounding box.
*
*                Some of the geometry attributes have varying data
*                types and sizes depending on the values of other
*                attributes.  For example, the \f(CWKGEOM_CENTER\fP
*                attribute must be set with an array of the data type
*                indicated by the attribute
*                \f(CWKGEOM_LOCATION_DATA_TYPE\fP that contains the
*                number of values indicated by the attribute
*                \f(CWKGEOM_LOCATION_SIZE\fP.
*
*                The following example illustrates the use of a single set
*                attributes call to assign two different attributes.
*
* !   \f(CWfloat min[3] = {0.0, 0.0, 0.0};\fP
* !   \f(CWfloat max[3] = {200.0, 123.0, 300.0};\fP
* !
* !   \f(CWkgeom_set_attributes(object, KGEOM_OBJECT, \fP
* !   \f(CW                             KGEOM_BOUNDING_BOX, min, max,\fP
* !   \f(CW                             KGEOM_NAME, "Isosurface 1",\fP
* !   \f(CW                             NULL);\fP
*
*                A complete list of geometry attributes and primitives
*                can be found in Chapter 3 of Programming Services
*                Volume II.
*
*         Input: object    - the data object into which the values of the
*                            attributes will be assigned
*
*                primitive - the primitive the attributes are related to. 
*                            Object-level attributes can be set
*                            by by passing \f(CWKGEOM_OBJECT\fP for
*                            this argument.  Primitive-level
*                            attributes can be set by passing in the
*                            appropriate primitive for this argument.
*                            The current primitive is determined by
*                            the \f(CWKGEOM_PRIMITIVE_POSITION\fP attribute.
*			     If the primitive at the current position has
*			     not yet been set, this set attributes call
*			     will set it.
*                            
*                kvalist   - NULL terminated variable argument list
*                            which contains a list of attributes,
*                            each attribute followed by the values
*                            to assign to that attribute.  
* 
*      			     The variable argument list takes the form:
*
*                         !  \f(CWATTRIBUTE_NAME1, value1 [, value2, ...],\fP
*                         !  \f(CWATTRIBUTE_NAME2, value1,[, value2, ...],\fP
*                         !  \f(CW..., NULL \fP
*
*                            The number of value arguments in the
*                            variable argument list for each attribute
*                            depends on the attribute. The NULL at the
*                            end of the variable argument list serves
*                            as a flag indicating the end of the list.
*
*			     Be careful not to forget the NULL at the
*			     end of the list.  This is a common programming
*			     error which unfortunately will not generate
*			     any compiler warnings. 
*        Output: none 
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: Calling this function with an incorrect number of arguments 
*                in the variable argument list will not cause any compiler 
*                errors, but will often generate a segmentation fault.
*                Also, mixing object and primitive level attributes on a 
*		 single \f(CWkgeom_set_attributes\fP call will produce
*		 incorrect results.
*
*    Written By: Steven Kubica
*          Date: Feb 13, 1993 15:55
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: kobject kgeom_set_attributes(
*		!   kobject object,
*		!   int     primitive,
*		!   kvalist)
*
*************************************************************/
int 
kgeom_set_attributes(
   kobject object,
   int     primitive,
   kvalist)
{
   kva_list list;
   int      position;
   int      status;
 

   KGEOM_INIT;

   if (object == NULL)
      kgeom_fail_error(KGEOM_EOBJ_INVALID);
 
   /* start the variable arguments list */
   kva_start(list, primitive);

   /* attribute will be at the object level, unless it is listable */
   if (primitive == KGEOM_OBJECT)
   {
     status = kdms_vset_attributes(object, KDMS_OBJECT, &list);
   }
   else
   {
      /* attribute depends on the primitive position */
      kdms_get_attribute(object, NULL,_INTERNAL_PRIMITIVE_POSITION, &position);

      /* make sure attribute is okay for primitive at primitive position */
      if (!kgeom_verify_manifest_primitive(kgeom_get_manifest(object), 
					   primitive, position, NULL))
         kgeom_fail_error(KGEOM_EINTERNAL);

      status = kdms_vset_attributes(object, KDMS_OBJECT, &list);
   }

   kva_end(list);
 
   return status;
}

/************************************************************
*
*  Routine Name: kgeom_get_attribute - get the value of a geometry
*                                      attribute in a data object.
*
*       Purpose: This function is used to retrieve the value of an attribute 
*                associated with a data object containing geometry data.
*
*                Attributes can be either global to the object, or
*                specific to a primitive contained within the object.
*		 Attributes which are global are termed \fIobject-level\fP, 
*		 and attributes specific to a primitive are termed
*		 \fIprimitive-level\fP.	    
*
*		 Object-level attributes apply globally to all
*		 primitives contained within a data object. 
*		 Two examples of object-level attributes are
*		 \f(CWKGEOM_NAME\fP, which is the name of the object,
*		 and \f(CWKGEOM_AMBIENT\fP, which dictates how the
*		 geometry should react to ambient light sources.
*
*                Primitive-level attributes apply only to the current
*                primitive as indicated by the
*                \f(CWKGEOM_PRIMITIVE_POSITION\fP attribute. An
*                example of such an attribute is the
*                \f(CWKGEOM_NUMBER\fP attribute which determines the
*                number of vertices in the current primitive.
*
*                Attributes may have multiple components.  For
*                example, the attribute \f(CWKGEOM_BOUNDING_BOX\fP
*                consists of two components, one component being the
*                minimum point of the bounding box, and the second
*                component being the maximum point of the bounding box.
*
*                Some of the geometry attributes have varying data
*                types and sizes depending on the values of other
*                attributes.  For example, the \f(CWKGEOM_CENTER\fP
*                attribute must be retrieved with a pointer to the
*                data type indicated by the attribute
*                \f(CWKGEOM_LOCATION_DATA_TYPE\fP. The returned array
*                will contain the number of values indicated by the
*                attribute \f(CWKGEOM_LOCATION_SIZE\fP. Changes in the
*                attributes which determine the type or the size of
*                another attribute will cause the dependent attribute
*                to be recast or resized.
*
*		 Note that any array attributes, such as strings,
*		 which are retrieved should not be altered or freed.
*		 The pointer returned points to the actual internal
*		 storage array.  A copy should be made if the values
*		 need to be changed.
*
*                The following example illustrates the use of the get
*                attribute call to retrieve two different attributes.
*
* !   \f(CWfloat *min;\fP
* !   \f(CWfloat *max;\fP
* !   \f(CWchar  *name;\fP
* !
* !   \f(CWkgeom_get_attribute(object, KGEOM_OBJECT, \fP
* !   \f(CW                            KGEOM_NAME, &name);\fP
* !   \f(CWkgeom_get_attribute(object, KGEOM_OBJECT, \fP
* !   \f(CW                            KGEOM_BOUNDING_BOX, &min, &max);\fP
*
*                A complete list of geometry attributes and primitives
*                can be found in Chapter 3 of Programming Services
*                Volume II.
*
*         Input: object    - the data object from which the attribute's 
*                            value will be retrieved
*
*                primitive - the primitive the attribute is related
*                            to.  Object-level attributes can be
*                            retrieved by by passing
*                            \f(CWKGEOM_OBJECT\fP for this argument.
*                            Primitive-level attributes can be
*                            retrieved by passing in the appropriate
*                            primitive for this argument.  The current
*                            primitive is determined by the
*                            \f(CWKGEOM_PRIMITIVE_POSITION\fP
*                            attribute.
*
*                attribute - the attribute to get
*                
*        Output: kvalist   - a variable argument list that contains 
*                            the addresses of variables which will be 
*			     used to return the different components 
*			     associated with that attribute.
*
*                            The variable argument list takes the form:
*
*                          !  \f(CWATTRIBUTE_NAME, &value1 [, &value2, ...]\fP
*
*                            The number of value arguments in the variable 
*                            argument list corresponds to the number of
*			     arguments needed to retrieve the attribute.
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: Calling this function with an incorrect number of arguments 
*                in the variable argument list will not cause any compiler 
*                errors, but will often generate a segmentation fault.
*
*    Written By: Steven Kubica 
*          Date: Feb 13, 1993 15:52
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: kobject kgeom_get_attribute(
*		!   kobject object,
*		!   int     primitive,
*		!   int     attribute,
*		!   kvalist)
*
*************************************************************/
int 
kgeom_get_attribute(
   kobject object,
   int     primitive,
   char   *attribute,
   kvalist)
{
   kva_list list;
   int      position;
   int      status;


   KGEOM_INIT;

   if (object == NULL)
      kgeom_fail_error(KGEOM_EOBJ_INVALID);

   kva_start(list, attribute);

   /* attribute will be at the object level, unless it is listable */
   if (primitive == KGEOM_OBJECT)
   {
     status = kdms_vget_attribute(object, KDMS_OBJECT, attribute, &list);
   }
   else
   {
      /* attribute depends on the primitive position */
      kdms_get_attribute(object, NULL,_INTERNAL_PRIMITIVE_POSITION, &position);

      /* make sure attribute is okay for primitive at primitive position */
      if (!kgeom_verify_manifest_primitive(kgeom_get_manifest(object), 
					   primitive, position, NULL))
         kgeom_fail_error(KGEOM_EINTERNAL);

      status = kdms_vget_attribute(object, KDMS_OBJECT, attribute, &list);
   }
 
   kva_end(list);

   return status;
}

/************************************************************
*
*  Routine Name: kgeom_get_attributes - get the values of a number
*                                       of geometry attributes.
*
*       Purpose: This function is used to retrieve the values of an 
*		 arbitrary number of attributes associated with a 
*		 data object containing geometry data.
*
*                Attributes can be either global to the object, or
*                specific to a primitive contained within the object.
*		 Attributes which are global are termed \fIobject-level\fP, 
*		 and attributes specific to a primitive are termed
*		 \fIprimitive-level\fP.	    
*
*		 Object-level attributes apply globally to all
*		 primitives contained within a data object. 
*		 Two examples of object-level attributes are
*		 \f(CWKGEOM_NAME\fP, which is the name of the object,
*		 and \f(CWKGEOM_AMBIENT\fP, which dictates how the
*		 geometry should react to ambient light sources.
*
*                Primitive-level attributes apply only to the current
*                primitive as indicated by the
*                \f(CWKGEOM_PRIMITIVE_POSITION\fP attribute. An example
*		 of such an attribute is the \f(CWKGEOM_NUMBER\fP attribute
*		 which determines the number of vertices in the current
*		 primitive.
*
*                Attributes may have multiple components.  For
*                example, the attribute \f(CWKGEOM_BOUNDING_BOX\fP
*                consists of two components, one component being the
*                minimum point of the bounding box, and the second
*                component being the maximum point of the bounding box.
*
*                Some of the geometry attributes have varying data
*                types and sizes depending on the values of other
*                attributes.  For example, the \f(CWKGEOM_CENTER\fP
*                attribute must be set with an array of the data type
*                indicated by the attribute
*                \f(CWKGEOM_LOCATION_DATA_TYPE\fP that contains the
*                number of values indicated by the attribute
*                \f(CWKGEOM_LOCATION_SIZE\fP. Changes in the
*                attributes which determine the type or the size of
*                another attribute will cause the dependent attribute
*                to be recast or resized.
*
*		 Note that any array attributes, such as strings,
*		 which are retrieved should not be altered or freed.
*		 The pointer returned points to the actual internal
*		 storage array.  A copy should be made if the values
*		 need to be changed.
*
*                The following example illustrates the use of a single get
*                attributes call to retrieve two different attributes.
*
* !   \f(CWfloat *min;\fP
* !   \f(CWfloat *max;\fP
* !   \f(CWchar  *name;\fP
* !
* !   \f(CWkgeom_get_attributes(object, KGEOM_OBJECT, \fP
* !   \f(CW                             KGEOM_BOUNDING_BOX, &min, &max,\fP
* !   \f(CW                             KGEOM_NAME, &name,\fP
* !   \f(CW                             NULL);\fP
*
*                A complete list of geometry attributes and primitives
*                can be found in Chapter 3 of Programming Services
*                Volume II.
*
*         Input: object    - the data object from which the values of the
*                            attributes will be retrieved
*
*                primitive - the primitive the attributes are related to. 
*                            Object-level attributes can be retrieved
*                            by by passing \f(CWKGEOM_OBJECT\fP for
*                            this argument.  Primitive-level
*                            attributes can be retrieved by passing in the
*                            appropriate primitive for this argument.
*                            The current primitive is determined by
*                            the \f(CWKGEOM_PRIMITIVE_POSITION\fP attribute.
*                            
*        Output: kvalist   - NULL terminated variable argument list
*                            which contains a list of attributes,
*			     each attribute followed by the addresses of
*			     variables which will be used to return the
*                            different components associated with that
*                            attribute. 
*
*			     The variable argument list takes the form:
*
*                         !  \f(CWATTRIBUTE_NAME1, &value1 [, &value2, ...],\fP
*                         !  \f(CWATTRIBUTE_NAME2, &value1,[, &value2, ...],\fP
*                         !  \f(CW..., NULL \fP
*
*                            The number of value arguments in the
*                            variable argument list for each attribute
*                            depends on the attribute. The NULL at the
*                            end of the variable argument list serves
*                            as a flag indicating the end of the list.
*
*			     Be careful not to forget the NULL at the
*			     end of the list.  This is a common programming
*			     error which unfortunately will not generate
*			     any compiler warnings. 
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: Calling this function with an incorrect number of arguments 
*                in the variable argument list will not cause any compiler 
*                errors, but will often generate a segmentation fault.
*                Also, mixing object and primitive level attributes on a 
*		 single \f(CWkgeom_get_attributes\fP call will produce
*		 incorrect results.
*
*    Written By: Steven Kubica
*          Date: Feb 13, 1993 15:52
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: kobject kgeom_get_attributes(
*		!   kobject object,
*		!   int     primitive,
*		!   kvalist)
*
*************************************************************/
int 
kgeom_get_attributes(
   kobject object,
   int     primitive,
   kvalist)
{
   kva_list list;
   int position;
   int status;


   KGEOM_INIT;

   if (object == NULL)
      kgeom_fail_error(KGEOM_EOBJ_INVALID);

   kva_start(list, primitive);
 
   /* attribute will be at the object level, unless it is listable */
   if (primitive == KGEOM_OBJECT)
   {
     status = kdms_vget_attributes(object, KDMS_OBJECT, &list);
   }
   else
   {
      /* attribute depends on the primitive position */
      kdms_get_attribute(object, NULL,_INTERNAL_PRIMITIVE_POSITION, &position);

      /* make sure attribute is okay for primitive at primitive position */
      if (!kgeom_verify_manifest_primitive(kgeom_get_manifest(object), 
					   primitive, position, NULL))
         kgeom_fail_error(KGEOM_EINTERNAL);

      status = kdms_vget_attributes(object, KDMS_OBJECT,&list);
   }
 
   kva_end(list);

   return status;
}

/************************************************************
*
*  Routine Name: kgeom_copy_attributes - copy all presentation
*                attributes from one data object to another.
*
*       Purpose: kgeom_copy_attributes is part of the Geometry
*                Data Services.  This function copies all
*                presentation attributes from the source_object
*                to the destination_object.  This means that all
*                the attributes of the object will be copied not
*                just those for the value, mask and map data.
*                For example, the source object may contain
*                attributes that one of the other services (geometry,
*                numerical, etc.) use.  These will also be copied.
*
*                For more information on the behavior of attributes,
*                please refer to kgeom_sync_object, kgeom_get_attribute
*                and kgeom_set_attribute.
*
*         Input: source_object - the object that serves as the
*                                source for the attributes.
*
*        Output: destination_object - the object that serves as
*                                     the destination for the
*                                     operation.
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: Attributes that describe the physical representation
*                of the data will not change.  For example, the
*                physical size, data type and index order will not
*                change.  The presentation of these attributes, however
*                will change.
*    Written By: Jeremy Worley & John Salas
*          Date: Sep 19, 1993 15:43
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kgeom_copy_object_attr(
*               !   kobject source_object,
*               !   kobject destination_object)
*
*************************************************************/
int 
kgeom_copy_attributes(
   kobject source_object,
   kobject destination_object)
{
   KGEOM_INIT;

   kdms_copy_segment_attributes(source_object ,NULL,
				destination_object, NULL);

   /* -- delete all the attributes which should not be copied -- */
   kdms_destroy_attribute(destination_object, NULL, KGEOM_MANIFEST);
   kdms_destroy_attribute(destination_object, NULL, KGEOM_MANIFEST_TRANSCRIPT);
   kdms_destroy_attribute(destination_object, NULL, KGEOM_MANIFEST_TEXTURE);
   kdms_destroy_attribute(destination_object, NULL, KGEOM_MANIFEST_TYPE);
   kdms_destroy_attribute(destination_object, NULL, KGEOM_MANIFEST_WIDTH);

   return TRUE;
}



