 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Geometry Application Services
   >>>>             Attribute Handling Routines
   >>>>
   >>>>   Static:
   >>>>			_data_type_get()   
   >>>>			_data_type_set()   
   >>>>			_data_type_match()   
   >>>>			_data_type_copy()   
   >>>>			_data_type_query()
   >>>>			_data_type_print()
   >>>>
   >>>>			_loc_size_get()   
   >>>>			_loc_size_set()   
   >>>>			_loc_size_match()   
   >>>>			_loc_size_copy()   
   >>>>			_loc_size_query()
   >>>>			_loc_size_print()
   >>>>
   >>>>			_tex_coord_size_get()   
   >>>>			_tex_coord_size_set()   
   >>>>			_tex_coord_size_match()   
   >>>>			_tex_coord_size_copy()   
   >>>>			_tex_coord_size_query()
   >>>>			_tex_coord_size_print()
   >>>>
   >>>>			_bounding_box_get()   
   >>>>			_bounding_box_set()   
   >>>>			_bounding_box_match()   
   >>>>			_bounding_box_copy()   
   >>>>			_bounding_box_query()
   >>>>			_bounding_box_print()
   >>>>
   >>>>			_center_get()
   >>>>			_center_set()
   >>>>			_center_match()
   >>>>			_center_copy()
   >>>>			_center_query()
   >>>>			_center_print()
   >>>>
   >>>>			_color_get()
   >>>>			_color_set()
   >>>>			_color_match()
   >>>>			_color_copy()
   >>>>			_color_query()
   >>>>			_color_print()
   >>>>
   >>>>			_matrix_get()
   >>>>			_matrix_set()
   >>>>			_matrix_match()
   >>>>			_matrix_copy()
   >>>>			_matrix_query()
   >>>>			_matrix_print()
   >>>>
   >>>>			_map_get()
   >>>>			_map_set()
   >>>>			_map_match()
   >>>>			_map_copy()
   >>>>			_map_query()
   >>>>			_map_print()
   >>>>
   >>>>			_map_length_get()
   >>>>			_map_length_set()
   >>>>			_map_length_match()
   >>>>			_map_length_copy()
   >>>>			_map_length_query()
   >>>>			_map_length_print()
   >>>>
   >>>>			_primitive_list_get()
   >>>>			_primitive_list_set()  
   >>>>			_primitive_list_match()
   >>>>			_primitive_list_copy()
   >>>>			_primitive_list_query()
   >>>>			_primitive_list_print()
   >>>>
   >>>>			_number_primitives_get()
   >>>>			_number_primitives_set()  
   >>>>			_number_primitives_match()
   >>>>			_number_primitives_copy() 
   >>>>			_number_primitives_query()
   >>>>			_number_primitives_print()
   >>>>
   >>>>			_primitive_pos_get()
   >>>>			_primitive_pos_set()
   >>>>			_primitive_pos_match()
   >>>>			_primitive_pos_copy() 
   >>>>			_primitive_pos_query()
   >>>>			_primitive_pos_print()
   >>>>
   >>>>			_number_vertices_get()
   >>>>			_number_vertices_set()
   >>>>			_number_vertices_match()
   >>>>			_number_vertices_copy() 
   >>>>			_number_vertices_query()
   >>>>			_number_vertices_print()
   >>>>  Private:
   >>>>			kgeom_define_attributes()
   >>>>             	
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "geom_internals.h"

int _kgeom_attributes_initialized = FALSE;

/*********** ---------------------------------------------------------------
 ***********  KGEOM_DATA_TYPE
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: _data_type_get
|       Purpose: get a data type attribute 
|    Written By: Steve Kubica
|          Date: Jul 09, 1994 
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_data_type_get(kobject obj, int assoc, int attr, kaddr clientData, 
	       kva_list *list)
{
   int *ret_t;
   int  t;
   

   /* -- if this attribute has been set before, then get it -- */
   if (kdms_query_attribute(obj, NULL, (char *)clientData,
			     NULL, NULL, NULL, NULL))
      kdms_get_attribute(obj, NULL, (char *)clientData, &t);
   else
   {
      /* -- do some fancy footwork to see if a segment exists for it -- */

      if (kstrcmp((char *)clientData, _INTERNAL_LOCATION_DATA_TYPE) == 0)
      {
	 if (kdms_query_segment(obj, KGEOM_SEGMENT_LOCATION))
	    kdms_get_attribute(obj, KGEOM_SEGMENT_LOCATION, 
			       KDMS_DATA_TYPE, &t);
	 else
	    t = KGEOM_LOCATION_DATA_TYPE_DEFAULT;

	 /* -- for case if the segment exists, but type hasn't been set -- */
	 if (!kaps_legal_data_type(t))
	    t = KGEOM_LOCATION_DATA_TYPE_DEFAULT;
      }
      else if (kstrcmp((char *)clientData, _INTERNAL_COLOR_DATA_TYPE) == 0)
      {
	 if (kdms_query_segment(obj, KGEOM_SEGMENT_COLOR))
	    kdms_get_attribute(obj, KGEOM_SEGMENT_COLOR, 
			       KDMS_DATA_TYPE, &t);
	 else
	    t = KGEOM_COLOR_DATA_TYPE_DEFAULT;

	 /* -- for case if the segment exists, but type hasn't been set -- */
	 if (!kaps_legal_data_type(t))
	    t = KGEOM_LOCATION_DATA_TYPE_DEFAULT;
      }
      else if (kstrcmp((char *)clientData, _INTERNAL_NORMAL_DATA_TYPE) == 0)
      {
	 if (kdms_query_segment(obj, KGEOM_SEGMENT_NORMAL))
	    kdms_get_attribute(obj, KGEOM_SEGMENT_NORMAL, 
			       KDMS_DATA_TYPE, &t);
	 else
	    t = KGEOM_NORMAL_DATA_TYPE_DEFAULT;

	 /* -- for case if the segment exists, but type hasn't been set -- */
	 if (!kaps_legal_data_type(t))
	    t = KGEOM_LOCATION_DATA_TYPE_DEFAULT;
      }
      else if (kstrcmp((char *)clientData, 
		       _INTERNAL_TEXTURE_COORD_DATA_TYPE) == 0)
      {
	 if (kdms_query_segment(obj, KGEOM_SEGMENT_TEXTURE_COORD))
	    kdms_get_attribute(obj, KGEOM_SEGMENT_TEXTURE_COORD, 
			       KDMS_DATA_TYPE, &t);
	 else
	    t = KGEOM_TEXTURE_COORD_DATA_TYPE_DEFAULT;

	 /* -- for case if the segment exists, but type hasn't been set -- */
	 if (!kaps_legal_data_type(t))
	    t = KGEOM_LOCATION_DATA_TYPE_DEFAULT;
      }
      else if (kstrcmp((char *)clientData, _INTERNAL_RADIUS_DATA_TYPE) == 0)
	 t = KGEOM_RADIUS_DATA_TYPE_DEFAULT;
      else
	 t = KNONE;
   }
   
   ret_t = kva_arg(*list, int *);
   if (ret_t) *ret_t = t;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: _data_type_set
|       Purpose: sets a data type attribute
|    Written By: Steve Kubica
|          Date: Dec 3, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_data_type_set(kobject obj, int assoc, int attr, kaddr clientData, 
	       kva_list *list)
{
   int t = kva_arg(*list, int);

   
   /* -- if this attribute has not been set before, then create it -- */
   if (!kdms_query_attribute(obj, NULL, (char *)clientData,
			     NULL, NULL, NULL, NULL))
      kdms_create_attribute(obj, NULL, (char *)clientData, 1, 1, KINT, 
			    TRUE, TRUE);


   /* -- do some automatic setting if this is a mesh -- */
   if (kstrcmp((char *)clientData, _INTERNAL_LOCATION_DATA_TYPE) == 0)
   {
      if (kdms_query_segment(obj, KGEOM_SEGMENT_LOCATION))
	 kpds_set_attribute(obj, KPDS_LOCATION_DATA_TYPE, t);
   }
   else if (kstrcmp((char *)clientData, _INTERNAL_COLOR_DATA_TYPE) == 0)
   {
      if (kdms_query_segment(obj, KGEOM_SEGMENT_COLOR))
	 kpds_set_attribute(obj, KPDS_VALUE_DATA_TYPE, t);
   }
   else if (kstrcmp((char *)clientData, _INTERNAL_NORMAL_DATA_TYPE) == 0)
   {
      if (kdms_query_segment(obj, KGEOM_SEGMENT_NORMAL))
	 kdms_set_attribute(obj, KGEOM_SEGMENT_NORMAL, KDMS_DATA_TYPE, t);

      /* -- for case if the segment exists, but type hasn't been set -- */
      if (!kaps_legal_data_type(t))
	 t = KGEOM_LOCATION_DATA_TYPE_DEFAULT;
   }
   else if (kstrcmp((char *)clientData, 
		    _INTERNAL_TEXTURE_COORD_DATA_TYPE) == 0)
   {
      if (kdms_query_segment(obj, KGEOM_SEGMENT_TEXTURE_COORD))
	 kdms_set_attribute(obj, KGEOM_SEGMENT_TEXTURE_COORD, 
			    KDMS_DATA_TYPE, t);
   }
   
   /* -- go ahead and set it -- */
   return(kdms_set_attribute(obj, NULL, (char *)clientData, t));
}

/*-----------------------------------------------------------
|  Routine Name: _data_type_match
|       Purpose: match a data type attribute
|    Written By: Steve Kubica 
|          Date: Dec 3, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_data_type_match(kobject obj1, kobject obj2, int assoc, int attrib,
		 kaddr clientData1, kaddr clientData2)
{
   int t1;
   int t2;

 
   if (!kgeom_get_attribute(obj1, KGEOM_OBJECT, ktoken_to_string(attrib), &t1))
      return FALSE;

   if (!kgeom_get_attribute(obj2, KGEOM_OBJECT, ktoken_to_string(attrib), &t2))
      return FALSE;
 
   return (t1 == t2);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _data_type_copy
|       Purpose: copy a data type attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_data_type_copy(kobject obj1, kobject obj2, int assoc, int attrib,
		kaddr clientData1, kaddr clientData2)
{
   int t;
   

   if (!kgeom_get_attribute(obj1, KGEOM_OBJECT, ktoken_to_string(attrib), &t))
      return FALSE;
   
   if (!kgeom_set_attribute(obj2, KGEOM_OBJECT, ktoken_to_string(attrib), t))
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _data_type_query
|       Purpose: query a data type attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_data_type_query(kobject obj, int assoc, int attrib, kaddr clientData,
		 int *num_args, int *arg_size, int *data_type, 
		 int *persistence)
{
   /* these are fixed quantities - simply fill them out */
   if (num_args)    *num_args    = 1;
   if (persistence) *persistence = TRUE;
   if (arg_size)    *arg_size    = 1;
   if (data_type)   *data_type   = KINT;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _data_type_print
|       Purpose: print a data type attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_data_type_print(kobject obj, int assoc, int attrib, kaddr clientData,
		  kfile *outfile)
{
   int t;


   kgeom_get_attribute(obj, KGEOM_OBJECT, ktoken_to_string(attrib), &t);
   
   if (outfile)
      kfprintf(outfile,"%s (%d)", kdefine_to_datatype(t), t);

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KGEOM_LOCATION_SIZE
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: _loc_size_get
|       Purpose: get the location size attribute 
|    Written By: Steve Kubica
|          Date: Jul 09, 1994 
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_loc_size_get(kobject obj, int assoc, int attr, kaddr clientData, 
	      kva_list *list)
{
   int *ret_s;
   int  s;
   

   /* -- if this attribute has been set before, then get it -- */
   if (kdms_query_attribute(obj, NULL, _INTERNAL_LOCATION_SIZE,
			     NULL, NULL, NULL, NULL))
      kdms_get_attribute(obj, NULL, _INTERNAL_LOCATION_SIZE, &s);
   else
   {
      /* -- if we have a location segment, use that -- */
      if (kdms_query_segment(obj, KGEOM_SEGMENT_LOCATION))
	 kpds_get_attribute(obj, KPDS_LOCATION_SIZE, NULL, NULL, NULL, &s);
#if 0
      /* -- damn kludge cause location size is in width position -- */
      if (kdms_query_segment(obj, KGEOM_SEGMENT_LOCATION))
	 kpds_get_attribute(obj, KPDS_LOCATION_SIZE, &s, NULL, NULL, NULL);
#endif
      else /* -- use the default -- */
	 s = KGEOM_LOCATION_SIZE_DEFAULT;
   }
   
   ret_s = kva_arg(*list, int *);
   if (ret_s) *ret_s = s;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: _loc_size_set
|       Purpose: sets the location size attribute
|    Written By: Steve Kubica
|          Date: Dec 3, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_loc_size_set(kobject obj, int assoc, int attr, kaddr clientData, 
	      kva_list *list)
{
   /* -- if this attribute has not been set before, then create it -- */
   if (!kdms_query_attribute(obj, NULL, _INTERNAL_LOCATION_SIZE,
			     NULL, NULL, NULL, NULL))
      kdms_create_attribute(obj, NULL, _INTERNAL_LOCATION_SIZE, 1, 1, 
			    KINT, TRUE, TRUE);
   
   /* -- go ahead and set it -- */
   return(kdms_set_attribute(obj, NULL, 
			     _INTERNAL_LOCATION_SIZE, kva_arg(*list, int)));
}

/*-----------------------------------------------------------
|  Routine Name: _loc_size_match
|       Purpose: match the location size attribute
|    Written By: Steve Kubica 
|          Date: Dec 3, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_loc_size_match(kobject obj1, kobject obj2, int assoc, int attrib,
		kaddr clientData1, kaddr clientData2)
{
   int s1;
   int s2;

 
   if (!kgeom_get_attribute(obj1, KGEOM_OBJECT, KGEOM_LOCATION_SIZE, &s1))
      return FALSE;

   if (!kgeom_get_attribute(obj2, KGEOM_OBJECT, KGEOM_LOCATION_SIZE, &s2))
      return FALSE;
 
   return (s1 == s2);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _loc_size_copy
|       Purpose: copy the location size attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_loc_size_copy(kobject obj1, kobject obj2, int assoc, int attrib,
	       kaddr clientData1, kaddr clientData2)
{
   int s;
   
   if (!kgeom_get_attribute(obj1, KGEOM_OBJECT, KGEOM_LOCATION_SIZE, &s))
      return FALSE;
   
   if (!kgeom_set_attribute(obj2, KGEOM_OBJECT, KGEOM_LOCATION_SIZE, s))
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _loc_size_query
|       Purpose: query the location size attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_loc_size_query(kobject obj, int assoc, int attrib, kaddr clientData,
		int *num_args, int *arg_size, int *data_type, 
		int *persistence)
{
   /* these are fixed quantities - simply fill them out */
   if (num_args)    *num_args    = 1;
   if (persistence) *persistence = TRUE;
   if (arg_size)    *arg_size    = 1;
   if (data_type)   *data_type   = KINT;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _loc_size_print
|       Purpose: print the location size attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_loc_size_print(kobject obj, int assoc, int attrib, kaddr clientData,
		kfile *outfile)
{
   int s;


   kgeom_get_attribute(obj, KGEOM_OBJECT, KGEOM_LOCATION_SIZE, &s);

   if (outfile)
      kfprintf(outfile,"%d", s);

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KGEOM_TEXTURE_COORD_SIZE
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: _tex_coord_size_get
|       Purpose: get the texture coord size attribute 
|    Written By: Steve Kubica
|          Date: Jul 09, 1994 
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_tex_coord_size_get(kobject obj, int assoc, int attr, kaddr clientData, 
		    kva_list *list)
{
   int *ret_s;
   int  s;
   

   /* -- if this attribute has been set before, then get it -- */
   if (kdms_query_attribute(obj, NULL, _INTERNAL_TEXTURE_COORD_SIZE,
			    NULL, NULL, NULL, NULL))
      kdms_get_attribute(obj, NULL, _INTERNAL_TEXTURE_COORD_SIZE, &s);
   else
   {
      /* -- if we have a texture_coord segment, use that -- */
      if (kdms_query_segment(obj, KGEOM_SEGMENT_TEXTURE_COORD))
      {
	 int *size;
      
	 kdms_get_attribute(obj, KGEOM_SEGMENT_TEXTURE_COORD, 
			    KDMS_SIZE, &size);
	 s = size[KDIM];
      }
      else /* -- use the default -- */
	 s = KGEOM_TEXTURE_COORD_SIZE_DEFAULT;
   }
   
   ret_s = kva_arg(*list, int *);
   if (ret_s) *ret_s = s;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: _tex_coord_size_set
|       Purpose: sets the texture coord size attribute
|    Written By: Steve Kubica
|          Date: Dec 3, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_tex_coord_size_set(kobject obj, int assoc, int attr, kaddr clientData, 
		    kva_list *list)
{
   /* -- if this attribute has not been set before, then create it -- */
   if (!kdms_query_attribute(obj, NULL, _INTERNAL_TEXTURE_COORD_SIZE,
			     NULL, NULL, NULL, NULL))
      kdms_create_attribute(obj, NULL, _INTERNAL_TEXTURE_COORD_SIZE, 1, 1, 
			    KINT, TRUE, TRUE);
   
   /* -- go ahead and set it -- */
   return(kdms_set_attribute(obj, NULL, _INTERNAL_TEXTURE_COORD_SIZE, 
			     kva_arg(*list, int)));
}

/*-----------------------------------------------------------
|  Routine Name: _tex_coord_size_match
|       Purpose: match the texture coord size attribute
|    Written By: Steve Kubica 
|          Date: Dec 3, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_tex_coord_size_match(kobject obj1, kobject obj2, int assoc, int attrib,
		      kaddr clientData1, kaddr clientData2)
{
   int s1;
   int s2;

 
   if (!kgeom_get_attribute(obj1, KGEOM_OBJECT, KGEOM_TEXTURE_COORD_SIZE, &s1))
      return FALSE;

   if (!kgeom_get_attribute(obj2, KGEOM_OBJECT, KGEOM_TEXTURE_COORD_SIZE, &s2))
      return FALSE;
 
   return (s1 == s2);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _tex_coord_size_copy
|       Purpose: copy the texture coord attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_tex_coord_size_copy(kobject obj1, kobject obj2, int assoc, int attrib,
		     kaddr clientData1, kaddr clientData2)
{
   int s;
   

   if (!kgeom_get_attribute(obj1, KGEOM_OBJECT, KGEOM_TEXTURE_COORD_SIZE, &s))
      return FALSE;

   if (!kgeom_set_attribute(obj2, KGEOM_OBJECT, KGEOM_TEXTURE_COORD_SIZE, s))
      return FALSE;
   
   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _tex_coord_size_query
|       Purpose: query the texture coord size attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_tex_coord_size_query(kobject obj, int assoc, int attrib, kaddr clientData,
		      int *num_args, int *arg_size, int *data_type, 
		      int *persistence)
{
   /* these are fixed quantities - simply fill them out */
   if (num_args)    *num_args    = 1;
   if (persistence) *persistence = TRUE;
   if (arg_size)    *arg_size    = 1;
   if (data_type)   *data_type   = KINT;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _tex_coord_size_print
|       Purpose: print the texture coord size attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_tex_coord_size_print(kobject obj, int assoc, int attrib, kaddr clientData, 
		      kfile *outfile)
{
   int s;


   kgeom_get_attribute(obj, KGEOM_OBJECT, KGEOM_TEXTURE_COORD_SIZE, &s);

   if (outfile)
      kfprintf(outfile,"%d", s);

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KGEOM_BOUNDING_BOX
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: _bounding_box_get
|       Purpose: get the bounding box attribute
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_bounding_box_get(kobject obj, int assoc, int attr, kaddr clientData,
		  kva_list *list)
{
   float **bmin = kva_arg(*list, float **); /* -- this is an address -- */
   float **bmax = kva_arg(*list, float **); /* -- this is an address -- */

   if (kdms_query_attribute(obj, NULL, _INTERNAL_BOUNDING_BOX,
			    NULL, NULL, NULL, NULL))
      return kdms_get_attribute(obj, NULL, _INTERNAL_BOUNDING_BOX, bmin, bmax);
   else
      return FALSE;  /* -- not an error, just no bounding box -- */
}

/*-----------------------------------------------------------
|  Routine Name: _bounding_box_set
|       Purpose: set the bounding box attribute
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_bounding_box_set(kobject obj, int assoc, int attr, kaddr clientData, 
  	          kva_list *list)
{
   float *bmin = kva_arg(*list, float *);
   float *bmax = kva_arg(*list, float *);

   if (!kdms_query_attribute(obj, NULL, _INTERNAL_BOUNDING_BOX,
			     NULL, NULL, NULL, NULL))
      if (!kdms_create_attribute(obj, NULL, _INTERNAL_BOUNDING_BOX,
				 2, 3, KFLOAT, TRUE, TRUE))
	  return FALSE;
   
   if (!kdms_set_attribute(obj, NULL, _INTERNAL_BOUNDING_BOX, bmin, bmax))
      return FALSE;
   
   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: _bounding_box_match
|       Purpose: match the bounding box attribute
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_bounding_box_match(kobject obj1, kobject obj2, int assoc, int attrib,
                    kaddr clientData1, kaddr clientData2)
{
   if (!kdms_query_attribute(obj1, NULL, _INTERNAL_BOUNDING_BOX,
			     NULL, NULL, NULL, NULL))
      return FALSE;

   if (!kdms_query_attribute(obj2, NULL, _INTERNAL_BOUNDING_BOX,
			     NULL, NULL, NULL, NULL))
      return FALSE;
   
   return kdms_match_attribute(obj1, obj2, NULL, _INTERNAL_BOUNDING_BOX);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _bounding_box_copy
|       Purpose: copy the bounding box attribute
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_bounding_box_copy(kobject obj1, kobject obj2, int assoc, int attrib,
                   kaddr clientData1, kaddr clientData2)
{
   /* -- must exist in source! -- */
   if (!kdms_query_attribute(obj1, NULL, _INTERNAL_BOUNDING_BOX,
			     NULL, NULL, NULL, NULL))
      return FALSE;

   if (!kdms_query_attribute(obj2, NULL, _INTERNAL_BOUNDING_BOX,
			     NULL, NULL, NULL, NULL))
      if (!kdms_create_attribute(obj2, NULL, _INTERNAL_BOUNDING_BOX,
				 2, 3, KFLOAT, TRUE, TRUE))
	 return FALSE;

   return kdms_copy_attribute(obj1, obj2, NULL, _INTERNAL_BOUNDING_BOX);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _bounding_box_query
|       Purpose: query the bounding box attribute
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_bounding_box_query(kobject obj, int assoc, int attrib, kaddr clientData,
                    int *num_args, int *arg_size, int *data_type, 
                    int *persistence)
{
   /* these are fixed quantities - simply fill them out */
   if (num_args)    *num_args    = 2;
   if (persistence) *persistence = TRUE;
   if (arg_size)    *arg_size    = 3;
   if (data_type)   *data_type   = KFLOAT;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _bounding_box_print
|       Purpose: print the bounding box attribute
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_bounding_box_print(kobject obj, int assoc, int attrib, kaddr clientData,
                    kfile *outfile)
{
   float *bmin = NULL; 
   float *bmax = NULL;
   int    has_bounding_box = TRUE;
   int    i;

   if (!kgeom_get_attribute(obj, KGEOM_OBJECT, KGEOM_BOUNDING_BOX,&bmin,&bmax))
      has_bounding_box = FALSE;

   if (outfile)
   {
      if (has_bounding_box) 
      {
         kfprintf(outfile,"bounding box :");
         for (i = 0; i < 3; i++) kfprintf(outfile," %g ",bmin[i]);
         kfprintf(outfile,"<->");        
         for (i = 0; i < 3; i++) kfprintf(outfile," %g ",bmax[i]);
      }
      else
         kfprintf(outfile,"NO bounding box");
   }
  
   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KGEOM_CENTER
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: _center_get
|       Purpose: get the center attribute
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_center_get(kobject obj, int assoc, int attr, kaddr clientData, 
            kva_list *list)
{
   float **center = kva_arg(*list, float **); /* -- this is an address -- */

   if (kdms_query_attribute(obj, NULL, _INTERNAL_CENTER,
			    NULL, NULL, NULL, NULL))
      return kdms_get_attribute(obj, NULL, _INTERNAL_CENTER, center); 
   else
      return FALSE;  /* -- not an error, just no center -- */
}

/*-----------------------------------------------------------
|  Routine Name: _center_set
|       Purpose: set the center attribute
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_center_set(kobject obj, int assoc, int attr, kaddr clientData, 
            kva_list *list)
{
   float *center = kva_arg(*list, float *);

   if (!kdms_query_attribute(obj, NULL, _INTERNAL_CENTER,
			     NULL, NULL, NULL, NULL))
      if (!kdms_create_attribute(obj, NULL, _INTERNAL_CENTER,
				 1, 3, KFLOAT, TRUE, TRUE))
	  return FALSE;
   
   if (!kdms_set_attribute(obj, NULL, _INTERNAL_CENTER, center))
      return FALSE;
   
   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: _center_match
|       Purpose: match the center attribute
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_center_match(kobject obj1, kobject obj2, int assoc, int attrib,
              kaddr clientData1, kaddr clientData2)
{
   if (!kdms_query_attribute(obj1, NULL, _INTERNAL_CENTER,
			     NULL, NULL, NULL, NULL))
      return FALSE;

   if (!kdms_query_attribute(obj2, NULL, _INTERNAL_CENTER,
			     NULL, NULL, NULL, NULL))
      return FALSE;
   
   return kdms_match_attribute(obj1, obj2, NULL, _INTERNAL_CENTER);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _center_copy
|       Purpose: copy the center attribute
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_center_copy(kobject obj1, kobject obj2, int assoc, int attrib,
             kaddr clientData1, kaddr clientData2)
{
   /* -- must exist in source! -- */
   if (!kdms_query_attribute(obj1, NULL, _INTERNAL_CENTER,
			     NULL, NULL, NULL, NULL))
      return FALSE;

   if (!kdms_query_attribute(obj2, NULL, _INTERNAL_CENTER,
			     NULL, NULL, NULL, NULL))
      if (!kdms_create_attribute(obj2, NULL, _INTERNAL_CENTER,
				 1, 3, KFLOAT, TRUE, TRUE))
	 return FALSE;

   return kdms_copy_attribute(obj1, obj2, NULL, _INTERNAL_CENTER);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _center_query
|       Purpose: query the center attribute
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_center_query(kobject obj, int assoc, int attrib, kaddr clientData,
              int *num_args, int *arg_size, int *data_type, 
              int *persistence)
{
   /* these are fixed quantities - simply fill them out */
   if (num_args)    *num_args    = 1;
   if (persistence) *persistence = TRUE;
   if (arg_size)    *arg_size    = 3;
   if (data_type)   *data_type   = KFLOAT;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _center_print
|       Purpose: print the center attribute
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_center_print(kobject obj, int assoc, int attrib, kaddr clientData,
              kfile *outfile)
{
   float *ctr = NULL;
   int    has_center = TRUE;
   int    i;


   if (!kgeom_get_attribute(obj, KGEOM_OBJECT, KGEOM_CENTER, &ctr))
      has_center = FALSE;

   if (outfile)
   {
      if (has_center) 
      {
         kfprintf(outfile,"center :");
         for (i = 0; i < 3 ; i++) kfprintf(outfile," %g ",ctr[i]);
      }
      else
         kfprintf(outfile,"NO center");
   }
  
   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KGEOM_COLOR
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: _color_get
|       Purpose: get the color attribute
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_color_get(kobject obj, int assoc, int attr, kaddr clientData, 
           kva_list *list)
{
   float **color = kva_arg(*list, float **);  /* -- this is an address -- */

   if (kdms_query_attribute(obj, NULL, _INTERNAL_COLOR,
			    NULL, NULL, NULL, NULL))
      return kdms_get_attribute(obj, NULL, _INTERNAL_COLOR, color);
   else
      return FALSE;  /* -- not an error, just no center -- */
}

/*-----------------------------------------------------------
|  Routine Name: _color_set
|       Purpose: set the color attribute
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_color_set(kobject obj, int assoc, int attr, kaddr clientData, 
           kva_list *list)
{
   float *color = kva_arg(*list, float *);

   if (!kdms_query_attribute(obj, NULL, _INTERNAL_COLOR, 
			     NULL, NULL, NULL, NULL))
      if (!kdms_create_attribute(obj, NULL, _INTERNAL_COLOR,
				 1, 3, KFLOAT, TRUE, TRUE))
	  return FALSE;
   
   if (!kdms_set_attribute(obj, NULL, _INTERNAL_COLOR, color))
      return FALSE;
   
   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: _color_match
|       Purpose: match the color attribute
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_color_match(kobject obj1, kobject obj2, int assoc, int attrib,
             kaddr clientData1, kaddr clientData2)
{
   if (!kdms_query_attribute(obj1, NULL, _INTERNAL_COLOR,
			     NULL, NULL, NULL, NULL))
      return FALSE;

   if (!kdms_query_attribute(obj2, NULL, _INTERNAL_COLOR,
			     NULL, NULL, NULL, NULL))
      return FALSE;
   
   return kdms_match_attribute(obj1, obj2, NULL, _INTERNAL_COLOR);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _color_copy
|       Purpose: copy the color attribute
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_color_copy(kobject obj1, kobject obj2, int assoc, int attrib,
            kaddr clientData1, kaddr clientData2)
{
   /* -- must exist in source! -- */
   if (!kdms_query_attribute(obj1, NULL, _INTERNAL_COLOR,
			     NULL, NULL, NULL, NULL))
      return FALSE;

   if (!kdms_query_attribute(obj2, NULL, _INTERNAL_COLOR,
			     NULL, NULL, NULL, NULL))
      if (!kdms_create_attribute(obj2, NULL, _INTERNAL_COLOR,
				 1, 3, KFLOAT, TRUE, TRUE))
	 return FALSE;

   return kdms_copy_attribute(obj1, obj2, NULL, _INTERNAL_COLOR);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _color_query
|       Purpose: query the color attribute
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_color_query(kobject obj, int assoc, int attrib, kaddr clientData,
             int *num_args, int *arg_size, int *data_type, 
             int *persistence)
{
   /* these are fixed quantities - simply fill them out */
   if (num_args)    *num_args    = 1;
   if (persistence) *persistence = TRUE;
   if (arg_size)    *arg_size    = 3;
   if (data_type)   *data_type   = KFLOAT;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _color_print
|       Purpose: print the color attribute
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_color_print(kobject obj, int assoc, int attrib, kaddr clientData,
             kfile *outfile)
{
   float *clr = NULL;
   int    has_color = TRUE;
   int    i;

   if (!kgeom_get_attribute(obj, KGEOM_OBJECT, KGEOM_COLOR, &clr))
      has_color = FALSE;

   if (outfile)
   {
      if (has_color) 
      {
         kfprintf(outfile,"color :");
         for (i = 0; i < 3; i++) kfprintf(outfile," %g ",clr[i]);
      }
      else
         kfprintf(outfile,"NO color");
   }
  
   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KGEOM_MATRIX
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: _matrix_get
|       Purpose: get the matrix attribute
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_matrix_get(kobject obj, int assoc, int attr, kaddr clientData, 
           kva_list *list)
{
   float **matrix = kva_arg(*list, float **); /* -- this is an address -- */

   if (kdms_query_attribute(obj, NULL, _INTERNAL_MATRIX,
			    NULL, NULL, NULL, NULL))

      return kdms_get_attribute(obj, NULL, _INTERNAL_MATRIX, matrix);
   else
      return FALSE;  /* -- not an error, just no center -- */
}

/*-----------------------------------------------------------
|  Routine Name: _matrix_set
|       Purpose: set the matrix attribute
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_matrix_set(kobject obj, int assoc, int attr, kaddr clientData, 
           kva_list *list)
{
   float *matrix = kva_arg(*list, float *);

   if (!kdms_query_attribute(obj, NULL, _INTERNAL_MATRIX, 
			     NULL, NULL, NULL, NULL))
      if (!kdms_create_attribute(obj, NULL, _INTERNAL_MATRIX,
				 1, 16, KFLOAT, TRUE, TRUE))
	  return FALSE;
   
   if (!kdms_set_attribute(obj, NULL, _INTERNAL_MATRIX, matrix))
      return FALSE;
   
   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: _matrix_match
|       Purpose: match the matrix attribute
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_matrix_match(kobject obj1, kobject obj2, int assoc, int attrib,
             kaddr clientData1, kaddr clientData2)
{
   if (!kdms_query_attribute(obj1, NULL, _INTERNAL_MATRIX,
			     NULL, NULL, NULL, NULL))
      return FALSE;

   if (!kdms_query_attribute(obj2, NULL, _INTERNAL_MATRIX,
			     NULL, NULL, NULL, NULL))
      return FALSE;
   
   return kdms_match_attribute(obj1, obj2, NULL, _INTERNAL_MATRIX);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _matrix_copy
|       Purpose: copy the matrix attribute
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_matrix_copy(kobject obj1, kobject obj2, int assoc, int attrib,
            kaddr clientData1, kaddr clientData2)
{
   /* -- must exist in source! -- */
   if (!kdms_query_attribute(obj1, NULL, _INTERNAL_MATRIX,
			     NULL, NULL, NULL, NULL))
      return FALSE;

   if (!kdms_query_attribute(obj2, NULL, _INTERNAL_MATRIX,
			     NULL, NULL, NULL, NULL))
      if (!kdms_create_attribute(obj2, NULL, _INTERNAL_MATRIX,
				 1, 16, KFLOAT, TRUE, TRUE))
	 return FALSE;

   return kdms_copy_attribute(obj1, obj2, NULL, _INTERNAL_MATRIX);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _matrix_query
|       Purpose: query the matrix attribute
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_matrix_query(kobject obj, int assoc, int attrib, kaddr clientData,
             int *num_args, int *arg_size, int *data_type, 
             int *persistence)
{
   /* these are fixed quantities - simply fill them out */
   if (num_args)    *num_args    = 1;
   if (persistence) *persistence = TRUE;
   if (arg_size)    *arg_size    = 16;
   if (data_type)   *data_type   = KFLOAT;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _matrix_print
|       Purpose: print the matrix attribute
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_matrix_print(kobject obj, int assoc, int attrib, kaddr clientData,
             kfile *outfile)
{
   float *mtx = NULL;
   int    has_matrix = TRUE;
   int    i, j;

   if (!kgeom_get_attribute(obj, KGEOM_OBJECT, KGEOM_MATRIX, &mtx))
      has_matrix = FALSE;

   if (outfile)
   {
      if (has_matrix) 
      {
         kfprintf(outfile,"matrix : \n");
         for (i = 0; i < 4; i++) 
	 {
	    for (j = 0; j < 4; j++) 
	       kfprintf(outfile," %g ",mtx[j+i*4]);
	       kfprintf(outfile,"\n");
	 }
      }
      else
         kfprintf(outfile,"NO matrix");
   }
  
   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KGEOM_MAP
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: _map_get
|       Purpose: get the map attribute 
|    Written By: Steve Kubica
|          Date: Jul 09, 1994 
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_map_get(kobject obj, int assoc, int attr, kaddr clientData, 
	 kva_list *list)
{
   kaddr *map;
   int    maplen; 
   int    esize; 
   int    coltype; 
   int    size[5] = {1,1,1,1,1};
   int    beg[5]  = {0,0,0,0,0}; 
   int    end[5]  = {0,0,0,0,0}; 
   

   /* the map is stored in the segment KGEOM_SEGMENT_MAP */

   /* trivial check for existence of map segment */
   if (!kdms_query_segment(obj, KGEOM_SEGMENT_MAP))
      return FALSE;  /* not an error - just no map! */

   /* get the current color size and type */
   if (!kdms_get_attributes(obj, KDMS_OBJECT,
                                 KGEOM_MAP_LENGTH,      &maplen,
                                 KGEOM_COLOR_DATA_TYPE, &coltype, 
				 NULL) )
      kgeom_fail_error(KGEOM_EINTERNAL_GET_ATTRIBUTES);

   /* determine the color vector size (this will be 1 if there is a map) */
   esize = kgeom_color_size(obj, KGEOM_SEGMENT_MAP);

   /* set up the presentation size */
   size[0] = esize;
   size[1] = maplen;

   /* set up the beg and end arrays determining where to get the data */
   end[0]  = esize - 1;
   end[1]  = maplen - 1; 

   /* present the map segment according to the current size and type */ 
   if (!kdms_set_attributes(obj, KGEOM_SEGMENT_MAP,
                                 KDMS_DATA_TYPE,   coltype, 
                                 KDMS_SIZE,        size,
			         KDMS_INTERPOLATE, KPAD,
				 NULL))
      kgeom_fail_error(KGEOM_EINTERNAL_SET_ATTRIBUTES);

   /* get the data */
   map = kva_arg(*list, kaddr *);
   if (map != NULL)
      *map = kdms_get_data(obj, KGEOM_SEGMENT_MAP, beg, end, *map);

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: _map_set
|       Purpose: set the map attribute
|    Written By: Steve Kubica
|          Date: Dec 3, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_map_set(kobject obj, int assoc, int attr, kaddr clientData, 
	 kva_list *list)
{
   kaddr  map;
   int   *old_size; 
   int    old_type;
   int    need_to_create = FALSE;
   int    coltype;
   int    size[5] = {1,1,1,1,1};
   int    beg[5]  = {0,0,0,0,0}; 
   int    end[5]  = {0,0,0,0,0};


   /* the map is stored in the segment KGEOM_SEGMENT_MAP */

   /* get the current location size and type */
   if (!kdms_get_attributes(obj, KDMS_OBJECT,
                                 KGEOM_MAP_LENGTH,      &(size[1]),
                                 KGEOM_COLOR_DATA_TYPE, &coltype, 
				 NULL))
      kgeom_fail_error(KGEOM_EINTERNAL_GET_ATTRIBUTES);

   /* map size is really (location_size+1) X (location_size+1) */
   size[0] = kgeom_color_size(obj, KGEOM_SEGMENT_MAP);

   /* does the map segment exist? */ 
   if (kdms_query_segment(obj, KGEOM_SEGMENT_MAP))
   {
      /* it exists - is it the correct size and data type? */
      if (!kdms_get_attributes(obj, KGEOM_SEGMENT_MAP,
                                    KDMS_SIZE,      &old_size,
                                    KDMS_DATA_TYPE, &old_type, 
				    NULL))
         kgeom_fail_error(KGEOM_EINTERNAL_SET_ATTRIBUTES);
 
      /* if size or type are wrong, need to destroy so we can recreate */ 
      if ((old_size[0] != size[0]) || (old_type != coltype) ||
          (old_size[1] != size[1]))
      {
         if ( !kdms_destroy_segment(obj, KGEOM_SEGMENT_MAP))
            kgeom_fail_error(KGEOM_EINTERNAL);

         need_to_create = TRUE;
      }
   }
   else
     need_to_create = TRUE;

   if (need_to_create)
   {
      if (!kdms_create_segment(obj, KGEOM_SEGMENT_MAP))
         kgeom_fail_error(KGEOM_EINTERNAL); 
      
      if (!kdms_set_attributes(obj, KGEOM_SEGMENT_MAP, 
			       KDMS_SIZE, size,
			       KDMS_DATA_TYPE, coltype, 
			       NULL))
         kgeom_fail_error(KGEOM_EINTERNAL); 
   }
   
   /* 
    *   we can now assume that the map segment exists and 
    *   is the correct size and type.
    */

   /* set up the beg and end arrays determining where to put the data */
   end[0]  = size[0]-1;
   end[1]  = size[1]-1;

   map = kva_arg(*list, kaddr);

   /* put the map */
   if (!kdms_put_data(obj, KGEOM_SEGMENT_MAP, beg, end, map) )
      kgeom_fail_error(KGEOM_EINTERNAL);

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: _map_match
|       Purpose: match the map attribute
|    Written By: Steve Kubica 
|          Date: Dec 3, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_map_match(kobject obj1, kobject obj2, int assoc, int attrib,
	   kaddr clientData1, kaddr clientData2)
{
   kobject  robj1; 
   kobject  robj2;
   int      equiv = TRUE;
   double  *map1 = NULL; 
   double  *map2 = NULL;
   int      len1;
   int      len2;
   int      esize1;
   int      esize2;
   int      i;


   /* first check for the existence of all map segments */
   if ( !kdms_query_segment(obj1, KGEOM_SEGMENT_MAP) ||
        !kdms_query_segment(obj1, KGEOM_SEGMENT_MAP))
      return FALSE; /* not an error - just no map! */
        
   /* 
    *   all segments are present :  
    *       we need to now actually compare the maps.
    *       we will create our own reference objects so we can
    *       present the maps in the same size and data
    *       type --- the comparison will then be simple.
    */

   /* create reference objects so we can compare the actual data */
   if ((robj1 = kdms_reference(obj1)) == NULL)
      kgeom_fail_error(KGEOM_EINTERNAL);

   if ((robj2 = kdms_reference(obj2)) == NULL)
      kgeom_fail_error(KGEOM_EINTERNAL);


   /* see if the map sizes are the same -- 
    * 	if they aren't then the maps do not match 
    */

   /* compare the map length first */
   if (!kgeom_get_attribute(robj1, KGEOM_OBJECT, 
				   KGEOM_MAP_LENGTH, &len1) ||    
       !kgeom_get_attribute(robj2, KGEOM_OBJECT,
				   KGEOM_MAP_LENGTH, &len2)) 
      kgeom_fail_error(KGEOM_EINTERNAL_GET_ATTRIBUTES);

   if (len1 != len2)
      return FALSE;

   /* compare the map width next */
   esize1 = kgeom_color_size(robj1, KGEOM_SEGMENT_MAP);
   esize2 = kgeom_color_size(robj2, KGEOM_SEGMENT_MAP);

   if (esize1 != esize2)
      return FALSE;
   
   /* 
    * set the data types on the reference objects maps to KDOUBLE 
    * for comparison
    */
   if (!kdms_set_attributes(robj1, KDMS_OBJECT,
			           KGEOM_COLOR_DATA_TYPE, KDOUBLE, NULL) ||
       !kdms_set_attributes(robj2, KDMS_OBJECT,
			           KGEOM_COLOR_DATA_TYPE, KDOUBLE, NULL))
      kgeom_fail_error(KGEOM_EINTERNAL_SET_ATTRIBUTES);

   /* get the maps */
   if (!kgeom_get_attribute(robj1, KGEOM_OBJECT, KGEOM_MAP, &map1) ||
       !kgeom_get_attribute(robj2, KGEOM_OBJECT, KGEOM_MAP, &map2))
      return FALSE; /* not an error - just no maps! */

   /* perform the comparison */
   if (map1 && map2)
   {
      for (i = 0; i < esize1*len1; i++)
         equiv &= (map1[i] == map2[i]);
   } else 
      return FALSE;  /* not an error - just no maps! */

   /* tidy up these reference objects */
   kdms_close(robj1);  kdms_close(robj2);  

   return equiv;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _map_copy
|       Purpose: copy the map attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_map_copy(kobject obj1, kobject obj2, int assoc, int attrib,
	  kaddr clientData1, kaddr clientData2)
{
   /* first check for the existence of the map segment to copy */
   if ( !kdms_query_segment(obj1, KGEOM_SEGMENT_MAP))
      return FALSE;  /* not an error - just no map! */

   /* simply copy the segment over */
   if (!kdms_copy_segment_data(obj1, KGEOM_SEGMENT_MAP,
                               obj2, KGEOM_SEGMENT_MAP))
      kgeom_fail_error(KGEOM_EINTERNAL);
   
   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _map_query
|       Purpose: query the map attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_map_query(kobject obj, int assoc, int attrib, kaddr clientData,
	   int *num_args, int *arg_size, int *data_type, 
	   int *persistence)
{
   int coltype;
   int maplen;


   /* first check for the existence of the map segment */
   if (!kdms_query_segment(obj, KGEOM_SEGMENT_MAP))
      return FALSE; /* not an error - just no map! */

   kdms_get_attribute(obj, KDMS_OBJECT, KGEOM_COLOR_DATA_TYPE, &coltype);
   kdms_get_attribute(obj, KDMS_OBJECT, KGEOM_MAP_LENGTH, &maplen);

   /* these are fixed quantities - simply fill them out */
   if (num_args)    *num_args = 1;
   if (persistence) *persistence = TRUE;
   if (data_type)   *data_type = coltype;
   if (arg_size)    *arg_size  = maplen * kgeom_color_size(obj, 
							   KGEOM_SEGMENT_MAP);


   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _map_print
|       Purpose: print the map attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_map_print(kobject obj, int assoc, int attrib, kaddr clientData,
	   kfile *outfile)
{
   kobject robj;
   int     has_map = TRUE;
   double *map = NULL;
   int     esize;
   int     maplen;
   int     i;
   int     j;
   

   /* we create this so we can set our own presentation */
   robj = kdms_reference(obj);

   /* set the presentation location type to double */
   if (!kgeom_set_attribute(robj, KGEOM_OBJECT, 
				  KGEOM_COLOR_DATA_TYPE, KDOUBLE))
      kgeom_fail_error(KGEOM_EINTERNAL_SET_ATTRIBUTES);

   kgeom_get_attribute(robj, KGEOM_OBJECT, KGEOM_MAP_LENGTH, &maplen);
   esize = kgeom_color_size(obj, KGEOM_SEGMENT_MAP);

   if (!kgeom_get_attribute(robj, KGEOM_OBJECT, KGEOM_MAP, &map))
      has_map = FALSE;

   if (outfile)
   {
      if (has_map) 
      {
         for (i = 0; i < maplen; i++) 
	 {
	    for (j = 0; j < esize; j++) 
	       kfprintf(outfile," %g ",map[i*esize+j]);
	    kfprintf(outfile,"\n");
	 }
      }
      else
         kfprintf(outfile,"NO map");
   }
  
   kdms_close(robj);

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KGEOM_MAP_LENGTH
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: _map_length_get
|       Purpose: get the map length attribute 
|    Written By: Steve Kubica
|          Date: Jul 09, 1994 
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_map_length_get(kobject obj, int assoc, int attr, kaddr clientData, 
		kva_list *list)
{
   if (!kdms_query_segment(obj, KGEOM_SEGMENT_MAP))
      return FALSE;   /* need to create map first */

   return kpds_get_attribute(obj, KPDS_MAP_SIZE, NULL, kva_arg(*list, int *),
			     NULL, NULL, NULL);
}

/*-----------------------------------------------------------
|  Routine Name: _map_length_set
|       Purpose: set the map length attribute
|    Written By: Steve Kubica
|          Date: Dec 3, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_map_length_set(kobject obj, int assoc, int attr, kaddr clientData, 
		kva_list *list)
{
   int esize;
   int maplen;
   int type;
   

   /*  if the map does not yet exist, then create it and set it's 
    *  width and height (or esize and maplen, to use geometry parlance)
    */
   if (!kdms_query_segment(obj, KGEOM_SEGMENT_MAP))
      if (!kdms_create_segment(obj, KGEOM_SEGMENT_MAP))
         return FALSE; /* error -- must create a map first */

   maplen = (int) kva_arg(*list, int);
   esize  = kgeom_color_size(obj, KGEOM_SEGMENT_MAP);   
   kgeom_get_attribute(obj, KGEOM_OBJECT, KGEOM_COLOR_DATA_TYPE, &type);

   if (maplen == 0)
      kdms_destroy_segment(obj, KGEOM_SEGMENT_MAP);
   
   return (kpds_set_attributes(obj, 
			       KPDS_MAP_DATA_TYPE, type,
			       KPDS_MAP_SIZE, esize, maplen, NULL, NULL, NULL,
			       NULL));
}

/*-----------------------------------------------------------
|  Routine Name: _map_length_match
|       Purpose: match the map length attribute
|    Written By: Steve Kubica 
|          Date: Dec 3, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_map_length_match(kobject obj1, kobject obj2, int assoc, int attrib,
		  kaddr clientData1, kaddr clientData2)
{
   int h1;
   int h2;

 
   if (!kpds_get_attribute(obj1, 
			   KPDS_MAP_SIZE, NULL, &h1, NULL, NULL, NULL, NULL))
      return FALSE;
 
   if (!kpds_get_attribute(obj2, 
			   KPDS_MAP_SIZE, NULL, &h2, NULL, NULL, NULL, NULL))
      return FALSE;
 
   return (h1 == h2);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _map_length_copy
|       Purpose: copy the map length attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_map_length_copy(kobject obj1, kobject obj2, int assoc, int attrib,
		 kaddr clientData1, kaddr clientData2)
{
   int h;
   int status = TRUE;
   

   status &= kgeom_get_attribute(obj1, KGEOM_OBJECT, 
				       KGEOM_MAP_LENGTH, &h);
   status &= kgeom_set_attribute(obj2, KGEOM_OBJECT, 
				       KGEOM_MAP_LENGTH, h);
   return status;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _map_length_query
|       Purpose: query the map length attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_map_length_query(kobject obj, int assoc, int attrib, kaddr clientData,
		  int *num_args, int *arg_size, int *data_type, 
		  int *persistence)
{
   /* these are fixed quantities - simply fill them out */
   if (num_args)    *num_args    = 1;
   if (persistence) *persistence = TRUE;
   if (arg_size)    *arg_size    = 1;
   if (data_type)   *data_type   = KINT;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _map_length_print
|       Purpose: print the map length attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_map_length_print(kobject obj, int assoc, int attrib, kaddr clientData,
		  kfile *outfile)
{
   int h;


   kgeom_get_attribute(obj, KGEOM_OBJECT, KGEOM_MAP_LENGTH, &h);

   if (outfile)
      kfprintf(outfile,"%d", h);

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KGEOM_PRIMITIVE_LIST
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: _primitive_list_get
|       Purpose: get the primitive list attribute
|    Written By: Steve Kubica
|          Date: Dec 3, 1993
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_primitive_list_get(kobject obj, int assoc, int attr, kaddr clientData,
		    kva_list *list)
{
   kgeom_manifest *manifest = kgeom_get_manifest(obj); 
   
   if (manifest == NULL)
      kgeom_fail_error(KGEOM_ENO_PRIMITIVE_LIST);

   *(kva_arg(*list, int **)) = kgeom_get_manifest_primitive_list(manifest);

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: _primitive_list_set
|       Purpose: set the primitive list attribute
|    Written By: Steve Kubica
|          Date: Dec 3, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_primitive_list_set(kobject obj, int assoc, int attr, kaddr clientData, 
  	            kva_list *list)
{
   kgeom_manifest *manifest = kgeom_get_manifest(obj); 
   int            *plist    = kva_arg( *list, int *);

   if (manifest == NULL)
      kgeom_fail_error(KGEOM_ENO_PRIMITIVE_LIST);

   return kgeom_set_manifest_primitive_list(manifest, plist);
}

/*-----------------------------------------------------------
|  Routine Name: _primitive_list_match
|       Purpose: match the primitive list attribute
|    Written By: Steve Kubica
|          Date: Mar 07, 1994 18:48
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_primitive_list_match(kobject obj1, kobject obj2, int assoc, int attrib,
                      kaddr clientData1, kaddr clientData2)
{
   kgeom_manifest *manifest1 = kgeom_get_manifest(obj1); 
   kgeom_manifest *manifest2 = kgeom_get_manifest(obj2); 
   int *plist1;
   int *plist2;
   int  i;
   
   if (manifest1 == NULL || manifest2 == NULL)
      return FALSE;
   
   if (manifest1->num_primitives != manifest2->num_primitives)
      return FALSE;

   plist1 = kgeom_get_manifest_primitive_list(manifest1);
   plist2 = kgeom_get_manifest_primitive_list(manifest2);
   
   for (i = 0; i < manifest1->num_primitives; i++)
      if (plist1[i] != plist2[i])
	 return FALSE;
   
   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _primitive_list_copy
|       Purpose: copy the primitive list attribute
|    Written By: Steve Kubica
|          Date: May 10, 1994 19:18
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_primitive_list_copy(kobject obj1, kobject obj2, int assoc, int attrib,
                     kaddr clientData1, kaddr clientData2)
{
   /* this amounts to a destroy primitives and a copy primitives */
  
   /* saving this for later */
   kgeom_fail_error(KGEOM_ELIMITATION) /* no ; */
}

/*-----------------------------------------------------------
|  Routine Name: (static) _primitive_list_query
|       Purpose: query the primitive list attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_primitive_list_query(kobject obj, int assoc, int attrib, kaddr clientData,
                      int *num_args, int *arg_size, int *data_type, 
                      int *persistence)
{
   kgeom_manifest *manifest = kgeom_get_manifest(obj); 


   /* -- first check for the existence of the primitive list -- */
   if (manifest == NULL)
      return FALSE; /* not an error - just no primitive list */

   /* these are fixed quantities - simply fill them out */
   if (num_args)    *num_args    = 1;
   if (persistence) *persistence = TRUE;
   if (data_type)   *data_type   = KINT;
   if (arg_size)    *arg_size    = manifest->num_primitives;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _primitive_list_print
|       Purpose: print the primitive list attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_primitive_list_print(kobject obj, int assoc, int attrib, kaddr clientData,
                      kfile *outfile)
{
   /* saving this for later */
   kgeom_fail_error(KGEOM_ELIMITATION) /* no ; */
}

/*********** ---------------------------------------------------------------
 ***********  KGEOM_NUMBER_PRIMITIVES
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: _number_primitives_get
|       Purpose: get the number of primitives attribute
|    Written By: Steve Kubica
|          Date: Dec 3, 1993
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_number_primitives_get(kobject obj, int assoc, int attr, kaddr clientData, 
            	       kva_list *list)
{
   kgeom_manifest *manifest = kgeom_get_manifest(obj); 

   if (manifest == NULL)
      kgeom_fail_error(KGEOM_ENO_PRIMITIVE_LIST);

   *(kva_arg(*list, int *)) = kgeom_get_manifest_number_primitives(manifest);

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: _number_primitives_set
|       Purpose: set the number of primitives attribute
|    Written By: Steve Kubica
|          Date: Dec 3, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_number_primitives_set(kobject obj, int assoc, int attr, kaddr clientData, 
            	       kva_list *list)
{
   kgeom_manifest *manifest = kgeom_get_manifest(obj); 
   int num = kva_arg( *list, int );
   int position;
   int state;

   if (manifest == NULL)
      kgeom_fail_error(KGEOM_ENO_PRIMITIVE_LIST);

   if (!kgeom_set_manifest_number_primitives(manifest, num))
      kgeom_fail_error(KGEOM_ENO_PRIMITIVE_LIST);

   /* get the primitive position and the increment state */ 
   kdms_get_attributes(obj, NULL,
		       _INTERNAL_PRIMITIVE_POSITION, &position,
		       _INTERNAL_GEOM_INC_STATE,     &state, NULL);
 
   /* -- if increment has been failing, then increment it... 
         mark the state as nudged -- */ 
   if (state == KGEOM_INC_STATE_FAIL)
   {
      if ( (position++) < num)
         kdms_set_attribute(obj, NULL, _INTERNAL_PRIMITIVE_POSITION, position);
      else
         kdms_set_attribute(obj, NULL,
			    _INTERNAL_GEOM_INC_STATE, KGEOM_INC_STATE_NUDGE);
   }
 
   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: _number_primitives_match
|       Purpose: match the number of primitives attribute
|    Written By: Steve Kubica
|          Date: Mar 07, 1994 18:48
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_number_primitives_match(kobject obj1, kobject obj2, int assoc, int attrib,
              		 kaddr clientData1, kaddr clientData2)
{
   kgeom_manifest *manifest1 = kgeom_get_manifest(obj1); 
   kgeom_manifest *manifest2 = kgeom_get_manifest(obj2); 
   int num1;
   int num2;
   
   if (manifest1 == NULL || manifest2 == NULL)
      return FALSE;
   
   num1 = kgeom_get_manifest_number_primitives(manifest1);
   num2 = kgeom_get_manifest_number_primitives(manifest2);
   
   return (num1 == num2);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _number_primitives_copy
|       Purpose: copy the number of primitives attribute
|    Written By: Steve Kubica
|          Date: May 10, 1994 19:18
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_number_primitives_copy(kobject obj1, kobject obj2, int assoc, int attrib,
                        kaddr clientData1, kaddr clientData2)
{
   kgeom_manifest *manifest1 = kgeom_get_manifest(obj1); 
   kgeom_manifest *manifest2 = kgeom_get_manifest(obj2); 
   int num;
   
   if (manifest1 == NULL || manifest2 == NULL)
      return FALSE;
   
   num = kgeom_get_manifest_number_primitives(manifest1);
   return kgeom_set_manifest_number_primitives(manifest2, num);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _number_primitives_query
|       Purpose: query the number of primitives attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_number_primitives_query(kobject obj, int assoc, int attrib, kaddr clientData,
                         int *num_args, int *arg_size, int *data_type, 
                         int *persistence)
{
   if (num_args)    *num_args    = 1;
   if (persistence) *persistence = TRUE;
   if (arg_size)    *arg_size    = 1;
   if (data_type)   *data_type   = KINT;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _number_primitives_print
|       Purpose: print the number of primitives attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_number_primitives_print(kobject obj, int assoc, int attrib, kaddr clientData,
                         kfile *outfile)
{
   kgeom_manifest *manifest = kgeom_get_manifest(obj); 
   int num;

   if (manifest == NULL)
      kgeom_fail_error(KGEOM_ENO_PRIMITIVE_LIST);

   num = kgeom_get_manifest_number_primitives(manifest);

   kfprintf(outfile, "%d", num);

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KGEOM_PRIMITIVE_POSITION
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: _primitive_pos_get
|       Purpose: get the primitive position attribute
|    Written By: Steve Kubica
|          Date: Dec 3, 1993
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_primitive_pos_get(kobject obj, int assoc, int attr, kaddr clientData, 
            	   kva_list *list)
{
   int pos; 


   if (!kdms_get_attribute(obj, NULL, _INTERNAL_PRIMITIVE_POSITION, &pos))
      kgeom_fail_error(KGEOM_EINTERNAL_GET_ATTRIBUTES);

   *(kva_arg(*list, int *)) = pos;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: _primitive_pos_set
|       Purpose: set the primitive position attribute
|    Written By: Steve Kubica
|          Date: Dec 3, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_primitive_pos_set(kobject obj, int assoc, int attr, kaddr clientData, 
            	   kva_list *list)
{
   kgeom_manifest *manifest = kgeom_get_manifest(obj);
   int pos = 0;
   int num = 0; 

   pos = (int) kva_arg(*list, int);

   if (manifest == NULL)
      kgeom_fail_error(KGEOM_ENO_PRIMITIVE_LIST);

   /* -- make sure that the position is in range -- */
   num = kgeom_get_manifest_number_primitives(manifest);
   
   if (pos < 0 || pos >= num)
      kgeom_fail_error(KGEOM_EINVALID_PRIMITIVE_POSITION);  
 
   /* -- reassign the position... -- */
   if (!kdms_set_attribute(obj, NULL, _INTERNAL_PRIMITIVE_POSITION, pos))
      kgeom_fail_error(KGEOM_EINTERNAL_SET_ATTRIBUTES);

   /* -- ... and reset the increment state -- */
   if (!kdms_set_attribute(obj, NULL,
			   _INTERNAL_GEOM_INC_STATE, KGEOM_INC_STATE_OK))
      kgeom_fail_error(KGEOM_EINTERNAL_SET_ATTRIBUTES );

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: _primitive_pos_match
|       Purpose: match the primitive position attribute
|    Written By: Steve Kubica
|          Date: Mar 07, 1994 18:48
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_primitive_pos_match(kobject obj1, kobject obj2, int assoc, int attrib,
                     kaddr clientData1, kaddr clientData2)
{
   return (kdms_match_attribute(obj1, obj2, NULL,
			       _INTERNAL_PRIMITIVE_POSITION));
}

/*-----------------------------------------------------------
|  Routine Name: (static) _primitive_pos_copy
|       Purpose: copy the primitive position attribute
|    Written By: Steve Kubica
|          Date: May 10, 1994 19:18
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_primitive_pos_copy(kobject obj1, kobject obj2, int assoc, int attrib,
                    kaddr clientData1, kaddr clientData2)
{
   int pos;

   
   if (!kgeom_get_attribute(obj1, KGEOM_OBJECT, KGEOM_PRIMITIVE_POSITION,&pos))
      kgeom_fail_error(KGEOM_EINTERNAL_GET_ATTRIBUTES);

   if (!kgeom_set_attribute(obj2, KGEOM_OBJECT, KGEOM_PRIMITIVE_POSITION,pos))
      kgeom_fail_error(KGEOM_EINTERNAL_SET_ATTRIBUTES);

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _primitive_pos_query
|       Purpose: query the primitive position attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_primitive_pos_query(kobject obj, int assoc, int attrib, kaddr clientData,
                     int *num_args, int *arg_size, int *data_type, 
                     int *persistence)
{
   if (num_args)    *num_args    = 1;
   if (persistence) *persistence = FALSE;
   if (arg_size)    *arg_size    = 1;
   if (data_type)   *data_type   = KINT;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _primitive_pos_print
|       Purpose: print the primitive position attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_primitive_pos_print(kobject obj, int assoc, int attrib, kaddr clientData,
                     kfile *outfile)
{
   kdms_print_attribute(obj, NULL, _INTERNAL_PRIMITIVE_POSITION, outfile);
   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KGEOM_NUMBER_VERTICES
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: _number_vertices_get
|       Purpose: get the number of vertices attribute
|    Written By: Steve Kubica
|          Date: Mar 07, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_number_vertices_get(kobject obj, int assoc, int attr, kaddr clientData, 
		     kva_list *list)
{
   kgeom_manifest *manifest = kgeom_get_manifest(obj); 
   int pos;

   if (manifest == NULL)
      kgeom_fail_error(KGEOM_ENO_PRIMITIVE_LIST);

   if (!kdms_get_attribute(obj, NULL, _INTERNAL_PRIMITIVE_POSITION, &pos))
      return FALSE;

   if (pos >= manifest->num_primitives)
      return FALSE;

   *(kva_arg(*list, int *)) = manifest->primitives[pos]->number;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: _number_vertices_set
|       Purpose: set the number of vertices attribute
|    Written By: Steve Kubica
|          Date: Mar 07, 1994
------------------------------------------------------------*/
/* ARGSUSED */
static int
_number_vertices_set(kobject obj, int assoc, int attr, kaddr clientData, 
		     kva_list *list)
{
   kgeom_manifest *manifest = kgeom_get_manifest(obj); 
   int pos;

   if (manifest == NULL)
      kgeom_fail_error(KGEOM_ENO_PRIMITIVE_LIST);

   if (!kdms_get_attribute(obj, NULL, _INTERNAL_PRIMITIVE_POSITION, &pos))
      return FALSE;

   if (pos >= manifest->num_primitives)
      return FALSE;

   manifest->primitives[pos]->number = (int) kva_arg(*list, int);;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: _number_vertices_match
|       Purpose: match the number of vertices attribute
|    Written By: Steve Kubica
|          Date: Mar 07, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_number_vertices_match(kobject obj1, kobject obj2, int assoc, int attrib,
		       kaddr clientData1, kaddr clientData2)
{
   /* -- save this for later //SK -- */
   return FALSE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _number_vertices_copy
|       Purpose: copy the number of vertices attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_number_vertices_copy(kobject obj1, kobject obj2, int assoc, int attrib,
		      kaddr clientData1, kaddr clientData2)
{
   /* -- save this for later //SK -- */
   return FALSE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _number_vertices_query
|       Purpose: query the number of vertices attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_number_vertices_query(kobject obj, int assoc, int attrib, kaddr clientData,
		       int *num_args, int *arg_size, int *data_type, 
		       int *persistence)
{
   /* these are fixed quantities - simply fill them out */
   if (num_args)    *num_args    = 1;
   if (persistence) *persistence = TRUE;
   if (arg_size)    *arg_size    = 1;
   if (data_type)   *data_type   = KINT;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _number_vertices_print
|       Purpose: print the number of vertices attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_number_vertices_print(kobject obj, int assoc, int attrib, kaddr clientData,
		       kfile *outfile)
{
   kgeom_manifest *manifest = kgeom_get_manifest(obj); 
   int pos;

   if (manifest == NULL)
      kgeom_fail_error(KGEOM_ENO_PRIMITIVE_LIST);

   if (!kdms_get_attribute(obj, NULL, _INTERNAL_PRIMITIVE_POSITION, &pos))
      return FALSE;

   if (pos >= manifest->num_primitives)
      return FALSE;

   if (outfile)
      kfprintf(outfile, "%d", manifest->primitives[pos]->number);

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KGEOM_LINE_TYPE  -  KGEOM_MARKER_TYPE
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: _type_attr_get
|       Purpose: get the line and marker type attributes
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_type_attr_get(kobject obj, int assoc, int attr, kaddr clientData, 
	       kva_list *list)
{
   kgeom_manifest *manifest = kgeom_get_manifest(obj); 
   char *attribute = ktoken_to_string(attr);
   int   pos;

   if (manifest == NULL)
      kgeom_fail_error(KGEOM_ENO_PRIMITIVE_LIST);

   /* -- make sure that the primitive is valid for this attribute -- */
   if (kstrcmp(attribute, KGEOM_LINE_TYPE) == 0)
   {
      /* -- line type is only valid on these primitives -- */
      if (assoc != KGEOM_POLYLINE_DISJOINT  &&
	  assoc != KGEOM_POLYLINE_CONNECTED &&
	  assoc != KGEOM_POLYGON            &&
	  assoc != KGEOM_RECTANGLES         &&
	  assoc != KGEOM_ELLIPSES)
	 return FALSE;
   }
   else if (kstrcmp(attribute, KGEOM_MARKER_TYPE) == 0)
   {
      /* -- marker type is only valid on these primitives -- */
      if (assoc != KGEOM_MARKERS)
	 return FALSE;
   }

   if (!kdms_get_attribute(obj, NULL, _INTERNAL_PRIMITIVE_POSITION, &pos))
      return FALSE;

   if (pos >= manifest->num_primitives)
      return FALSE;

   *(kva_arg(*list, int *)) = manifest->primitives[pos]->type;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: _type_attr_set
|       Purpose: set the line and marker type attributes
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
------------------------------------------------------------*/
/* ARGSUSED */
static int
_type_attr_set(kobject obj, int assoc, int attr, kaddr clientData, 
	       kva_list *list)
{
   kgeom_manifest *manifest = kgeom_get_manifest(obj); 
   char *attribute = ktoken_to_string(attr);
   int pos;

   if (manifest == NULL)
      kgeom_fail_error(KGEOM_ENO_PRIMITIVE_LIST);

   /* -- make sure that the primitive is valid for this attribute -- */
   if (kstrcmp(attribute, KGEOM_LINE_TYPE) == 0)
   {
      /* -- line type is only valid on these primitives -- */
      if (assoc != KGEOM_POLYLINE_DISJOINT  &&
	  assoc != KGEOM_POLYLINE_CONNECTED &&
	  assoc != KGEOM_POLYGON            &&
	  assoc != KGEOM_RECTANGLES         &&
	  assoc != KGEOM_ELLIPSES)
	 return FALSE;
   }
   else if (kstrcmp(attribute, KGEOM_MARKER_TYPE) == 0)
   {
      /* -- marker type is only valid on these primitives -- */
      if (assoc != KGEOM_MARKERS)
	 return FALSE;
   }

   if (!kdms_get_attribute(obj, NULL, _INTERNAL_PRIMITIVE_POSITION, &pos))
      return FALSE;

   if (pos >= manifest->num_primitives)
      return FALSE;

   manifest->primitives[pos]->type = (int) kva_arg(*list, int);;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: _type_attr_match
|       Purpose: match the line and marker type attributes
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_type_attr_match(kobject obj1, kobject obj2, int assoc, int attrib,
		 kaddr clientData1, kaddr clientData2)
{
   /* -- save this for later //SK -- */
   return FALSE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _type_attr_copy
|       Purpose: copy the line and marker type attributes
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_type_attr_copy(kobject obj1, kobject obj2, int assoc, int attrib,
		kaddr clientData1, kaddr clientData2)
{
   /* -- save this for later //SK -- */
   return FALSE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _type_attr_query
|       Purpose: query the line and marker type attributes
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_type_attr_query(kobject obj, int assoc, int attrib, kaddr clientData,
		 int *num_args, int *arg_size, int *data_type, 
		 int *persistence)
{
   /* these are fixed quantities - simply fill them out */
   if (num_args)    *num_args    = 1;
   if (persistence) *persistence = TRUE;
   if (arg_size)    *arg_size    = 1;
   if (data_type)   *data_type   = KINT;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _type_attr_print
|       Purpose: print the line and marker type attributes
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_type_attr_print(kobject obj, int assoc, int attrib, kaddr clientData,
		 kfile *outfile)
{
   kgeom_manifest *manifest = kgeom_get_manifest(obj); 
   char *attribute = ktoken_to_string(attrib);
   int   pos;

   if (manifest == NULL)
      kgeom_fail_error(KGEOM_ENO_PRIMITIVE_LIST);

   /* -- make sure that the primitive is valid for this attribute -- */
   if (kstrcmp(attribute, KGEOM_LINE_TYPE) == 0)
   {
      /* -- line type is only valid on these primitives -- */
      if (assoc != KGEOM_POLYLINE_DISJOINT  &&
	  assoc != KGEOM_POLYLINE_CONNECTED &&
	  assoc != KGEOM_POLYGON            &&
	  assoc != KGEOM_RECTANGLES         &&
	  assoc != KGEOM_ELLIPSES)
	 return FALSE;
   }
   else if (kstrcmp(attribute, KGEOM_MARKER_TYPE) == 0)
   {
      /* -- marker type is only valid on these primitives -- */
      if (assoc != KGEOM_MARKERS)
	 return FALSE;
   }

   if (!kdms_get_attribute(obj, NULL, _INTERNAL_PRIMITIVE_POSITION, &pos))
      return FALSE;

   if (pos >= manifest->num_primitives)
      return FALSE;

   if (outfile)
      kfprintf(outfile, "%d", manifest->primitives[pos]->type);

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KGEOM_LINE_WIDTH
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: _width_attr_get
|       Purpose: get the line width attribute
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_width_attr_get(kobject obj, int assoc, int attr, kaddr clientData, 
		kva_list *list)
{
   kgeom_manifest *manifest = kgeom_get_manifest(obj); 
   int   pos;

   if (manifest == NULL)
      kgeom_fail_error(KGEOM_ENO_PRIMITIVE_LIST);

   /* -- line width is only valid on these primitives -- */
   if (assoc != KGEOM_POLYLINE_DISJOINT  &&
       assoc != KGEOM_POLYLINE_CONNECTED &&
       assoc != KGEOM_POLYGON            &&
       assoc != KGEOM_RECTANGLES         &&
       assoc != KGEOM_ELLIPSES)
      return FALSE;

   if (!kdms_get_attribute(obj, NULL, _INTERNAL_PRIMITIVE_POSITION, &pos))
      return FALSE;

   if (pos >= manifest->num_primitives)
      return FALSE;

   *(kva_arg(*list, int *)) = manifest->primitives[pos]->width;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: _width_attr_set
|       Purpose: set the line width attribute
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
------------------------------------------------------------*/
/* ARGSUSED */
static int
_width_attr_set(kobject obj, int assoc, int attr, kaddr clientData, 
		kva_list *list)
{
   kgeom_manifest *manifest = kgeom_get_manifest(obj); 
   int pos;

   if (manifest == NULL)
      kgeom_fail_error(KGEOM_ENO_PRIMITIVE_LIST);

   /* -- line width is only valid on these primitives -- */
   if (assoc != KGEOM_POLYLINE_DISJOINT  &&
       assoc != KGEOM_POLYLINE_CONNECTED &&
       assoc != KGEOM_POLYGON            &&
       assoc != KGEOM_RECTANGLES         &&
       assoc != KGEOM_ELLIPSES)
      return FALSE;

   if (!kdms_get_attribute(obj, NULL, _INTERNAL_PRIMITIVE_POSITION, &pos))
      return FALSE;

   if (pos >= manifest->num_primitives)
      return FALSE;

   manifest->primitives[pos]->type = (int) kva_arg(*list, int);

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: _width_attr_match
|       Purpose: match the line width attribute
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_width_attr_match(kobject obj1, kobject obj2, int assoc, int attrib,
		  kaddr clientData1, kaddr clientData2)
{
   /* -- save this for later //SK -- */
   return FALSE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _width_attr_copy
|       Purpose: copy the line width attribute
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_width_attr_copy(kobject obj1, kobject obj2, int assoc, int attrib,
		 kaddr clientData1, kaddr clientData2)
{
   /* -- save this for later //SK -- */
   return FALSE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _width_attr_query
|       Purpose: query the line width attribute
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_width_attr_query(kobject obj, int assoc, int attrib, kaddr clientData,
		  int *num_args, int *arg_size, int *data_type, 
		  int *persistence)
{
   /* these are fixed quantities - simply fill them out */
   if (num_args)    *num_args    = 1;
   if (persistence) *persistence = TRUE;
   if (arg_size)    *arg_size    = 1;
   if (data_type)   *data_type   = KINT;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _width_attr_print
|       Purpose: print the line width attribute
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_width_attr_print(kobject obj, int assoc, int attrib, kaddr clientData,
		  kfile *outfile)
{
   kgeom_manifest *manifest = kgeom_get_manifest(obj); 
   int   pos;

   if (manifest == NULL)
      kgeom_fail_error(KGEOM_ENO_PRIMITIVE_LIST);

   /* -- line width is only valid on these primitives -- */
   if (assoc != KGEOM_POLYLINE_DISJOINT  &&
       assoc != KGEOM_POLYLINE_CONNECTED &&
       assoc != KGEOM_POLYGON            &&
       assoc != KGEOM_RECTANGLES         &&
       assoc != KGEOM_ELLIPSES)
      return FALSE;

   if (!kdms_get_attribute(obj, NULL, _INTERNAL_PRIMITIVE_POSITION, &pos))
      return FALSE;

   if (pos >= manifest->num_primitives)
      return FALSE;

   if (outfile)
      kfprintf(outfile, "%d", manifest->primitives[pos]->width);

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KGEOM_TEXTURE
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: _texture_get
|       Purpose: get the texture attribute
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_texture_get(kobject obj, int assoc, int attr, kaddr clientData, 
	     kva_list *list)
{
   kgeom_manifest *manifest = kgeom_get_manifest(obj); 
   char *attribute = ktoken_to_string(attr);
   int   pos;

   if (manifest == NULL)
      kgeom_fail_error(KGEOM_ENO_PRIMITIVE_LIST);

   /* -- texture is only valid on object or on these primitives -- */
   if (assoc != 0 &&
       assoc != KGEOM_TRIANGLES_DISJOINT &&
       assoc != KGEOM_TRIANGLES_CONNECTED)
      return FALSE;

   if (!kdms_get_attribute(obj, NULL, _INTERNAL_PRIMITIVE_POSITION, &pos))
      return FALSE;

   if (pos >= manifest->num_primitives)
      return FALSE;

   *(kva_arg(*list, char **)) = manifest->primitives[pos]->texture;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: _texture_set
|       Purpose: set the texture attribute
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
------------------------------------------------------------*/
/* ARGSUSED */
static int
_texture_set(kobject obj, int assoc, int attr, kaddr clientData, 
		kva_list *list)
{
   kgeom_manifest *manifest = kgeom_get_manifest(obj); 
   int pos;

   if (manifest == NULL)
      kgeom_fail_error(KGEOM_ENO_PRIMITIVE_LIST);

   /* -- texture is only valid on object or on these primitives -- */
   if (assoc != 0 &&
       assoc != KGEOM_TRIANGLES_DISJOINT &&
       assoc != KGEOM_TRIANGLES_CONNECTED)
      return FALSE;

   if (!kdms_get_attribute(obj, NULL, _INTERNAL_PRIMITIVE_POSITION, &pos))
      return FALSE;

   if (pos >= manifest->num_primitives)
      return FALSE;

   manifest->primitives[pos]->texture = 
      kstrdup((char *)kva_arg(*list, char *));

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: _texture_match
|       Purpose: match the line width attributes
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_texture_match(kobject obj1, kobject obj2, int assoc, int attrib,
	       kaddr clientData1, kaddr clientData2)
{
   /* -- save this for later //SK -- */
   return FALSE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _texture_copy
|       Purpose: copy the line width attributes
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_texture_copy(kobject obj1, kobject obj2, int assoc, int attrib,
		 kaddr clientData1, kaddr clientData2)
{
   /* -- save this for later //SK -- */
   return FALSE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _texture_query
|       Purpose: query the line width attributes
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_texture_query(kobject obj, int assoc, int attrib, kaddr clientData,
	       int *num_args, int *arg_size, int *data_type, 
	       int *persistence)
{
   /* -- save this for later //SK -- */
   return FALSE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _texture_print
|       Purpose: print the line width attributes
|    Written By: Steve Kubica
|          Date: Apr 3, 1994
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_texture_print(kobject obj, int assoc, int attrib, kaddr clientData,
	       kfile *outfile)
{
   kgeom_manifest *manifest = kgeom_get_manifest(obj); 
   int pos;

   if (manifest == NULL)
      kgeom_fail_error(KGEOM_ENO_PRIMITIVE_LIST);

   /* -- texture is only valid on object or on these primitives -- */
   if (assoc != 0 &&
       assoc != KGEOM_TRIANGLES_DISJOINT &&
       assoc != KGEOM_TRIANGLES_CONNECTED)
      return FALSE;

   if (!kdms_get_attribute(obj, NULL, _INTERNAL_PRIMITIVE_POSITION, &pos))
      return FALSE;

   if (pos >= manifest->num_primitives)
      return FALSE;

   if (outfile)
      kfprintf(outfile,"%s",  manifest->primitives[pos]->texture);

   return TRUE;
}

/*
 *    ==================================================================
 *    Initialization Routine
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: kgeom_define_attributes
|
|       Purpose: This routine makes the specific define attribute
|                calls for defining geometry services attributes.
|                These attributes are defined for the session.
|
|                Calls are made to other services who overlap
|                with geometry to ensure that the these attributes 
|                are also defined.
|
|         Input: none 
|
|        Output: none 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica & John Salas
|          Date: Dec 2, 1993 
| Modifications:
|
------------------------------------------------------------*/
void
kgeom_define_attributes(void)
{
   float zero[3] = {0.0, 0.0, 0.0};
   float one[3]  = {1.0, 1.0, 1.0};

   if (_kgeom_attributes_initialized)
      return;

   /* prevent this routine from being called again */
   _kgeom_attributes_initialized = TRUE;


   /*
    *    Data Type Attributes
    *   .....................................................................
    */

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KGEOM_LOCATION_DATA_TYPE, 
			       _INTERNAL_LOCATION_DATA_TYPE,
			       _data_type_get,  _data_type_set,
			       _data_type_match,  _data_type_copy,
			       _data_type_query,  _data_type_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KGEOM_COLOR_DATA_TYPE, 
			       _INTERNAL_COLOR_DATA_TYPE,
			       _data_type_get,  _data_type_set,
			       _data_type_match,  _data_type_copy,
			       _data_type_query,  _data_type_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KGEOM_RADIUS_DATA_TYPE, 
			       _INTERNAL_RADIUS_DATA_TYPE, 
			       _data_type_get,  _data_type_set,
			       _data_type_match,  _data_type_copy,
			       _data_type_query,  _data_type_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KGEOM_TEXTURE_COORD_DATA_TYPE,
			       _INTERNAL_TEXTURE_COORD_DATA_TYPE,
			       _data_type_get,  _data_type_set,
			       _data_type_match,  _data_type_copy,
			       _data_type_query,  _data_type_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KGEOM_NORMAL_DATA_TYPE,
			       _INTERNAL_NORMAL_DATA_TYPE,
			       _data_type_get,  _data_type_set,
			       _data_type_match,  _data_type_copy,
			       _data_type_query,  _data_type_print);
   /*
    *   General Object Level Attributes    
    *   .....................................................................
    */
   kdms_define_attribute(KDMS_OBJECT, KGEOM_AMBIENT, 
                         1, 1, KDOUBLE, TRUE, TRUE, 
			 KGEOM_AMBIENT_DEFAULT);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KGEOM_BOUNDING_BOX, NULL, 
			       _bounding_box_get,   _bounding_box_set, 
			       _bounding_box_match, _bounding_box_copy, 
			       _bounding_box_query, _bounding_box_print);

   kdms_define_quasi_attribute(KDMS_OBJECT,
			       KGEOM_CENTER, NULL, 
			       _center_get,   _center_set, 
			       _center_match, _center_copy, 
			       _center_query, _center_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KGEOM_COLOR, NULL, 
			       _color_get,   _color_set, 
			       _color_match, _color_copy, 
			       _color_query, _color_print);

   kdms_define_attribute(KDMS_OBJECT, KGEOM_DIFFUSE, 
                         1, 1, KDOUBLE, TRUE, TRUE,
			 KGEOM_DIFFUSE_DEFAULT);

   kdms_define_attribute(KDMS_OBJECT, KGEOM_LAYOUT,
                         1, 1, KINT, TRUE, TRUE,
			 KGEOM_LAYOUT_DEFAULT );

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KGEOM_LOCATION_SIZE, NULL,
			       _loc_size_get,   _loc_size_set,
			       _loc_size_match, _loc_size_copy,
			       _loc_size_query, _loc_size_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KGEOM_MAP, NULL, 
			       _map_get,   _map_set, 
			       _map_match, _map_copy, 
			       _map_query, _map_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KGEOM_MAP_LENGTH, NULL, 
			       _map_length_get,   _map_length_set, 
			       _map_length_match, _map_length_copy, 
			       _map_length_query, _map_length_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KGEOM_MATRIX, NULL, 
			       _matrix_get,   _matrix_set, 
			       _matrix_match, _matrix_copy, 
			       _matrix_query, _matrix_print);

   kdms_define_attribute(KDMS_OBJECT, KGEOM_METAL, 
                         1, 1, KDOUBLE, TRUE, TRUE,
			 KGEOM_METAL_DEFAULT);
 
   kdms_define_attribute(KDMS_OBJECT, KGEOM_MODELING_SPACE,
                         1, 1, KINT, TRUE, TRUE,
			 KGEOM_MODELING_SPACE_DEFAULT);

   kdms_define_attribute(KDMS_OBJECT, KGEOM_NAME,
                         1, 1, KSTRING, TRUE, TRUE,
			 NULL );

   kdms_define_attribute(KDMS_OBJECT, KGEOM_OPACITY, 
                         1, 1, KDOUBLE, TRUE, TRUE,
			 KGEOM_OPACITY_DEFAULT);

   kdms_define_attribute(KDMS_OBJECT, KGEOM_SPECULAR_COEFFICIENT, 
                         1, 1, KDOUBLE, TRUE, TRUE,
			 KGEOM_SPECULAR_COEFFICIENT_DEFAULT );

   kdms_define_attribute(KDMS_OBJECT, KGEOM_SPECULAR_EXPONENT, 
                         1, 1, KDOUBLE, TRUE, TRUE,
			 KGEOM_SPECULAR_EXPONENT_DEFAULT );

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KGEOM_TEXTURE_COORD_SIZE, NULL,
			       _tex_coord_size_get,   _tex_coord_size_set, 
			       _tex_coord_size_match, _tex_coord_size_copy, 
			       _tex_coord_size_query, _tex_coord_size_print);

   kdms_define_attribute(KDMS_OBJECT, KGEOM_VISIBLE, 
                         1, 1, KINT, TRUE, TRUE,
			 KGEOM_VISIBLE_DEFAULT );



   /*
    *   Primitive List Attributes
    *   .....................................................................
    */
   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KGEOM_PRIMITIVE_LIST, NULL, 
			       _primitive_list_get,   _primitive_list_set,
			       _primitive_list_match, _primitive_list_copy,
			       _primitive_list_query, _primitive_list_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			   KGEOM_NUMBER_PRIMITIVES, NULL, 
			   _number_primitives_get,   _number_primitives_set, 
		           _number_primitives_match, _number_primitives_copy, 
			   _number_primitives_query, _number_primitives_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KGEOM_PRIMITIVE_POSITION, NULL, 
			       _primitive_pos_get,   _primitive_pos_set, 
			       _primitive_pos_match, _primitive_pos_copy,
			       _primitive_pos_query, _primitive_pos_print);

   kdms_define_attribute(KDMS_OBJECT, _INTERNAL_PRIMITIVE_POSITION,  
                         1, 1, KINT, FALSE, FALSE,
			 KGEOM_PRIMITIVE_POSITION_DEFAULT);

   kdms_define_attribute(KDMS_OBJECT, _INTERNAL_GEOM_INC_STATE, 
                         1, 1, KINT, FALSE, FALSE,
			 KGEOM_INC_STATE_OK);

   /*
    *   Per Primitive Attributes
    *   .....................................................................
    */
   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KGEOM_NUMBER_VERTICES, NULL, 
			       _number_vertices_get,   _number_vertices_set, 
			       _number_vertices_match, _number_vertices_copy,
			       _number_vertices_query, _number_vertices_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KGEOM_LINE_TYPE, NULL, 
			       _type_attr_get,   _type_attr_set, 
			       _type_attr_match, _type_attr_copy,
			       _type_attr_query, _type_attr_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KGEOM_LINE_WIDTH, NULL, 
			       _width_attr_get,   _width_attr_set, 
			       _width_attr_match, _width_attr_copy,
			       _width_attr_query, _width_attr_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KGEOM_TEXTURE, NULL, 
			       _texture_get,   _texture_set, 
			       _texture_match, _texture_copy,
			       _texture_query, _texture_print);
   /*
    *    Common Application Services Attributes  
    *   .....................................................................
    */
   kaps_init();


   /*
    *    Color Attributes  
    *   .....................................................................
    */
   kcolor_init();


   /*
    * It is possible to generate errnos with duplicate attribute
    * definitions.  To prevent the user from seeing this, we'll
    * re-zero the errno.
    */
   errno = 0;
       
   return;
}
