 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>           Image Automatic Coloring Utilities
   >>>>
   >>>>   Static:
   >>>>                 initialize_maps()
   >>>>                 data_min_max()
   >>>>                 data_stats()
   >>>>                 restore_original()
   >>>>                 color_map332()
   >>>>                 color_cie_diagram()
   >>>>                 color_rgb_distance()
   >>>>                 color_rgb_triangle()
   >>>>                 color_rgb_cube()
   >>>>                 color_greyscale()
   >>>>                 color_strequalize()
   >>>>                 color_invert()
   >>>>                 color_reverse()
   >>>>                 color_row_rotate()
   >>>>                 color_chain_rotate()
   >>>>                 color_rgb_rotate()
   >>>>                 color_rainbow()
   >>>>                 color_greycode()
   >>>>                 color_disjoint()
   >>>>                 color_hls_spiral()
   >>>>                 color_rgb_spiral()
   >>>>                 color_hls_rings()
   >>>>                 color_hsv_rings()
   >>>>                 color_rgb_random()
   >>>>                 color_density_slice()
   >>>>                 color_rgbfilter()
   >>>>                 color_col_rotate()
   >>>>                 color_rgbswap()
   >>>>                 color_stddev()
   >>>>                 color_sa_pseudo()
   >>>>
   >>>>  Private:
   >>>>                 kmap_map_autocolor()
   >>>>                 kmap_map_operation()
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include <dataserv.h>
#include "colorspace.h"

#define ORIGINAL_SEGMENT  "OriginalColormap"

/*
 * a static array of available map_autocolor colormaps.
 */
char *kcolor_map_autocolor_list[] = { 
	"Original", 
	"3-3-2",
	"CIE Diagram",
	"Color Equalize",
        "Color Stretch", 
        "Density Slice", 
	"Disjoint", 
	"Greycode", 
	"Greyscale", 
	"HLS Rings", 
	"HLS Spiral",
        "HSV Rings", 
	"RGB Cube", 
	"RGB Distance", 
	"RGB Spiral", 
	"RGB Triangle", 
        "Rainbow",
	"Random",
	"SA Pseudo",
	"Standard Deviation"};

int kcolor_map_autocolor_num = knumber(kcolor_map_autocolor_list);

char *kcolor_map_operation_list[] = {
	"Blue Filter",
	"Chain Left",
	"Chain Right",
	"Green Filter",
	"Invert Original",
	"Invert",
	"Red Filter",
	"Reverse",
	"Rotate Blue Left",
	"Rotate Blue Right",
	"Rotate Col Left",
	"Rotate Col Right",
	"Rotate Green Left",
	"Rotate Green Right",
	"Rotate Red Left",
	"Rotate Red Right",
	"Rotate Row Left",
	"Rotate Row Right",
	"Swap Blue/Red",
	"Swap Green/Blue",
	"Swap Red/Green"};


int kcolor_map_operation_num = knumber(kcolor_map_operation_list);

/*-----------------------------------------------------------
|
|  Routine Name: initialize_maps - 
|
|       Purpose: initialize the object's maps...
|    Written By: Mark Young
|          Date: Jun 11, 1993
| Modifications:
|
------------------------------------------------------------*/

static int initialize_maps(
   kobject object)
{
	long minval = 0, maxval = 0;
	int height, i, j, w, h, d, t, e, type, num_rgns, num_pts;

	kaddr data = NULL;
	char  *bdata;
	short *sdata;
	int   *idata;
	long  *ldata;
	unsigned short *usdata;
	unsigned int   *uidata;
	unsigned long  *uldata;
	unsigned char  *ubdata;


	/*
	 *  Make sure that the colormap exists...
	 */
	if (kpds_query_map(object) == TRUE)
	   return(TRUE);

	kpds_get_attribute(object, KPDS_VALUE_DATA_TYPE, &type);
	if (type == KSHORT || type == KUSHORT ||
	    type == KINT   || type == KUINT ||
	    type == KLONG  || type == KULONG ||
	    type == KBYTE  || type == KUBYTE)
	{
	   kpds_get_attribute(object, KPDS_VALUE_OPTIMAL_REGION_SIZE,
			&w,&h,&d,&t,&e, &num_rgns);

	   kpds_set_attribute(object, KPDS_VALUE_REGION_SIZE, w, h, d, t, e);

	   num_pts = w*h*d*t*e;
	   for (i = 0; i < num_rgns; i++)
	   {
              data = kpds_get_data(object, KPDS_VALUE_REGION, data);
	      switch (type)
	      {
	         case KUSHORT:
	              usdata = (unsigned short *) data;
	              for (j = 0; j < num_pts; j++)
			  maxval = kmax(maxval, (long) usdata[j]);
	              break;

	         case KUINT:
	              uidata = (unsigned int *) data;
	              for (j = 0; j < num_pts; j++)
			  maxval = kmax(maxval, (long) uidata[j]);
	              break;

	         case KULONG:
	              uldata = (unsigned long *) data;
	              for (j = 0; j < num_pts; j++)
			  maxval = kmax(maxval, (long) uldata[j]);
	              break;

	         case KUBYTE:
	              ubdata = (unsigned char *) data;
	              for (j = 0; j < num_pts; j++)
			  maxval = kmax(maxval, (long) ubdata[j]);
	              break;

	         case KSHORT:
	              sdata = (short *) data;
	              for (j = 0; j < num_pts; j++)
		      {
		         minval = kmin(minval, sdata[j]);
		         maxval = kmax(maxval, sdata[j]);
	              }
	              break;

	         case KINT:
	              idata = (int *) data;
	              for (j = 0; j < num_pts; j++)
		      {
		         minval = kmin(minval, idata[j]);
		         maxval = kmax(maxval, idata[j]);
	              }
	              break;

	         case KLONG:
	              ldata = (long *) data;
	              for (j = 0; j < num_pts; j++)
		      {
		         minval = kmin(minval, ldata[j]);
		         maxval = kmax(maxval, ldata[j]);
	              }
	              break;

	         case KBYTE:
	              bdata = (char *) data;
	              for (j = 0; j < num_pts; j++)
		      {
		         minval = kmin(minval, bdata[j]);
		         maxval = kmax(maxval, bdata[j]);
	              }
	              break;
	      }
           }
	   kfree(data);
	   height = (maxval - minval) + 1;
	}
	else if (type == KBIT)
	{
	   height = 2;
	   type = KUBYTE;
	}
	else
	{
	   height = 256;
	   type = KUBYTE;
	}
	kpds_create_map(object);
	kpds_set_attribute(object, KPDS_MAP_SIZE, 3, height, 1, 1, 1);
	kpds_set_attribute(object, KPDS_MAP_DATA_TYPE, type);
	kpds_set_attribute(object, KPDS_COUPLING, KUNCOUPLED);
	kdms_update_references(object, KDMS_SEGMENT_MAP);

	/*
	 *  Return FALSE to indicate that the maps aren't initialized...
	 */
	return(FALSE);
}

/*-----------------------------------------------------------
|
|  Routine Name: data_min_max - 
|
|       Purpose: find the minimum and maximum values for the
|		 incoming values array.  The min & max values
|		 are for those indexes that have a histogram
|		 count of greater than zero.
|
|    Written By: Mark Young
|          Date: Jun 11, 1993
| Modifications:
|
------------------------------------------------------------*/

static void data_min_max(
   double	 *values,
   int		 ncolors,
   double	 *min,
   double	 *max)
{
	int i;

	*min = *max = values[0];
	for (i = 1; i < ncolors; i++)
	{
	   if (*min > values[i])
	      *min = values[i];
	   else if (*max < values[i])
	      *max = values[i];
	}
	if (*min == *max) *max = *min+1;
}

/*-----------------------------------------------------------
|
|  Routine Name: data_stats - 
|
|       Purpose: find the standard deviation and mean for a
|		 range of data.
|
|    Written By: Danielle Argiro & Mark Young
|          Date: Jun 15, 1993
| Modifications:
|
------------------------------------------------------------*/
#if 0
/* ARGSUSED */
static void data_stats(
   double *values,
   long	  *histogram,
   int	  ncolors,
   double *stddev,
   double *mean,
   double *variance,
   double *rms)
{
	int   i;
	long  count;
	double sum, sqr;

	for (i = 0, count = 0, sum = sqr = 0.0; i < ncolors; i++)
	{
	   sum   += values[i];
	   count += histogram[i];
	   sqr   += ksqr(values[i]) * histogram[i];
	}
}
#endif

/*-----------------------------------------------------------
|
|  Routine Name: restore_original - save or restore the original colormap into
|				  __original_colormap segment.
|
|       Purpose: This routine saves or restore the original colormap from or
|		 to the current colormap to or from the saved colormap.
|
|    Written By: Mark Young
|          Date: Jan 1, 1994
| Modifications: Converted from khoros 1.0 and changed to use data services (MY)
|
------------------------------------------------------------*/

static void restore_original(
   kobject image)
{
	if (!kdms_query_segment(image, ORIGINAL_SEGMENT))
	{
	   kdms_copy_segment_data(image, KDMS_SEGMENT_MAP,
				  image, ORIGINAL_SEGMENT);
	}
	else
	{
	   kdms_copy_segment_data(image, ORIGINAL_SEGMENT,
				  image, KDMS_SEGMENT_MAP);
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: color_map332 - 
|
|       Purpose: auto color this puppy using a 332 scheme.
|
|    Written By: Wes Bethel
|          Date: Apr 22, 1994
| Modifications:
|
------------------------------------------------------------*/

static unsigned char level8[8] =
{
    0,36,72,108,144,181,217,255
};
static unsigned char level4[4] =
{
    0,85,170,255
};

/* ARGSUSED */
static void color_map332(
   kobject image,
   double   *red,
   double   *green,
   double   *blue,
   int     ncolors)
{
	int i;

	for (i = 0; i < ncolors; i++)
	{
	   red[i]   = level8[(i >> 5) % 8];
	   green[i] = level8[(i >> 2) % 8];
	   blue[i]  = level4[i % 4];
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: color_cie_diagram - color according to CIE
|
|       Purpose: auto colors according to the CIE chromaticity
|		 diagram.  The routine starts at blue then varies
|		 the RB primaries until red is reached.  We then
|		 vary the RG primaries until green is reached.  On
|		 the last segment the GB primaries are varied until
|		 we come back to blue (our original starting place).
|
|    Written By: Mark Young & Tom Sauer
|          Date: Jun 10, 1993
| Modifications: Converted from khoros 1.0 and changed to use data services (MY)
|
------------------------------------------------------------*/
/* ARGSUSED */
static void color_cie_diagram(
   kobject image,
   double   *red,
   double   *green,
   double   *blue,
   int     ncolors)
{
	int   i, j,
	      num_per_range,	/* number to color per range*/
	      num_ranges = 3;	/* there is three sides to the diagram */
	double factor;		/* saturation factor used in coloring */

	num_per_range = ncolors/num_ranges;
	if (num_per_range < 2)
	{
	   kerror("kappserv", "color_cie_diagram", "Sorry! This image \
contain '%d' colors.  A minimum of 6 colors is needed for this auto coloring \
algorithm.", ncolors);
	   return;
	}
	factor = MAX_INTEN/((double) num_per_range -1);

	/*
	 *  Start at blue and vary RB primary until red is reached.  This
	 *  is done by increasing the red value while decreasing blue.
	 */
	i = j = 0;
	while (i < ncolors && j < num_per_range)
	{
	   red[i]   = j * factor;
	   green[i] = 0;
	   blue[i]  = MAX_INTEN - j * factor;
	   i++; j++;
	}

	/*
	 *  Start at red and vary RG primary until green is reached.  This
	 *  is done by increasing the green value while decreasing red.
	 */
	j = 0;
	while (i < ncolors && j < num_per_range)
	{
	   red[i]   = MAX_INTEN - j * factor;
	   green[i] = j * factor;
	   blue[i]  = 0;
	   i++; j++;
	}

	/*
	 *  Start at green and vary GB primary until blue is reached (our
	 *  original starting point).  This is done by increasing the blue
	 *  value while decreasing green.  Since this is the last leg we
	 *  include any remainder points from ncolors/3. 
	 */
	num_per_range += (ncolors - (num_per_range * num_ranges));
      	factor = MAX_INTEN/((double) num_per_range -1);

	j = 0;
	while (i < ncolors && j < num_per_range)
	{
	   red[i]   = 0;
	   green[i] = MAX_INTEN - j * factor;
	   blue[i]  = j * factor;
	   i++; j++;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: color_rgb_distance - color according to rgb distance
|					   major
|
|       Purpose: auto colors according to rgb distance mesaure.  The
|		 RGB is a data dependent colormap operation which
|		 assigns the red and green components of the colormap
|		 according to characteristics of the image's histogram,
|		 and the blue component according to pixel intensities.
|		 The red primary components are determined by calculating
|		 a delta variance for each cell of the original colormap,
|		 and the green components according to a delta mean calcu-
|		 lation for each cell.  The method of assigning colors is
|		 reviewed below:
|
|			a) Calculate mean pixel value of the histogram
|			b) Calculate the average variance of the histogram
|			c) Assigning the Red colormap:
|
|    Written By: Mark Young & Tom Sauer
|          Date: Jun 10, 1993
| Modifications: Converted from khoros 1.0 and changed to use data services (MY)
|
------------------------------------------------------------*/

static void color_rgb_distance(
   kobject image,
   double   *red,
   double   *green,
   double   *blue,
   int     ncolors)
{
	int	i, size, a, b;
	double	*variance, *mean, *intensity, *histogram = NULL;
	double   tvar, tmean, rms, sum, value, imin, vmin, mmin, max,
		vfact, mfact, ifact;


	if (histogram == NULL)
	   return;

	sum = rms = 0.0;
	for (i = 0; i < ncolors; i++)
	{
	   value = histogram[i] * i;
	   sum  += value;
	   rms  += value * value;
	}

	/*
	 * size  = image->row_size * image->col_size;
         */
        kpds_get_attribute(image, KPDS_VALUE_SIZE, &a, &b, NULL, NULL, NULL);
	size = a * b;
	intensity = (double *) kmalloc(ncolors * sizeof(double));
	mean = (double *) kmalloc(ncolors * sizeof(double));
	variance = (double *) kmalloc(ncolors * sizeof(double));

	tmean = sum/size;
	tvar  = (rms/size) - (tmean * tmean);

	for (i = 0; i < ncolors; i++)
	{
	   intensity[i] = i;
	   mean[i]      = (histogram[i] * i) - tmean;
	   variance[i]  = (histogram[i] * i * histogram[i] * i) - tvar;
	}

	data_min_max(variance, ncolors, &vmin, &max);
	vfact = MAX_INTEN/(max - vmin);
	vmin  = fabs((double) vmin);

	data_min_max(mean, ncolors, &mmin, &max);
	mfact = MAX_INTEN/(max - mmin);
	mmin  = fabs((double) mmin);

	data_min_max(intensity, ncolors, &imin, &max);
	ifact = MAX_INTEN/(max - imin);
	imin  = fabs((double) imin);

	for (i = 0; i < ncolors; i++)
	{
	   red[i]   = (mmin + mean[i]) * mfact;
	   blue[i]  = (imin + intensity[i]) * ifact;
	   green[i] = (vmin + variance[i]) * vfact;
	}

	/*
	 * not an optimal solution, but it'll work...
	 */
	kfree(intensity);
	kfree(mean);
	kfree(variance);
}

/*-----------------------------------------------------------
|
|  Routine Name: color_rgb_triangle - color according to rgb triangle
|
|       Purpose: 
|		We will try to follow the edges of the color triangle.
|		This is done by setting a value of one color to maximum
|		intensity, setting another to zero, then letting the other
|		color range from zero to maximum intensity.  Then set the
|		ranging color to zero, and let the other color range.  The
|		repeat setting other colors to max.  Hence we get:
|
|		Range1 - red (max), 	green (0), 	blue (range)
|		Range2 - red (max), 	green (range), 	blue (0)
|		Range3 - red (range),	green (max),	blue (0)
|		Range4 - red (0),	green (max),	blue(range)
|		Range5 - red (range),	green (0),	blue(max)
|		Range6 - red (0),	green (range),	blue(0)	
|
|		Hence our factor is MAX/(ncolors/6)
|		and to make it look better, when each color saturated we go
|		from MAX to low to MAX.
|
|		Lastly, Range 6 is larger as since ncolors/6 may not be an
|		even number, it has ncolor/6 + the remainer so that we paint
|		a total of ncolor colors
|
|		Note: this is clumsy and is not as pleasing as it should be
|		      Perhaps using the heat color scale would be cleaner.
|
|    Written By: Eric Engquist & Mark Young
|          Date: Jun 10, 1993
| Modifications: Converted from khoros 1.0 and changed to use data services (MY)
|
------------------------------------------------------------*/
/* ARGSUSED */
static void color_rgb_triangle(
   kobject image,
   double   *red,
   double   *green,
   double   *blue,
   int     ncolors)
{
	int	i, j,		/* famous loop variables */
		num_per_range,	/* number to color per range*/
		num_ranges = 6;	/* Each plane (R, G or B has 2 ranges */
	double factor;		/* saturation factor used in auto_color */

	num_per_range = ncolors/num_ranges;	/* ncolors per range */
	if (num_per_range < 2)
	{
	   kerror("kappserv", "color_rgb_triangle", "Sorry! This image \
contain '%d' colors.  A minimum of 12 colors is needed for this auto coloring \
algorithm.", ncolors);
	   return;
	}
      	factor = MAX_INTEN/((double) num_per_range -1);

	/*
	 * Color the first range
	 * Paint in increasing intensity
	 */
	i = j = 0;
	while (i < ncolors && j < num_per_range)
	{
	   red[i]   = MAX_INTEN;
	   green[i] = 0;
	   blue[i]  = j * factor;
	   i++; j++;
	}

	/*
	 * Color the second range
	 * Paint in decreasing intensity
	 */
	j = 0;
	while (i < ncolors && j < num_per_range)
	{
	   red[i]   = MAX_INTEN;
	   green[i] = MAX_INTEN - j * factor;
	   blue[i]  = 0;
	   i++; j++;
	}

	/*
	 * Color the third range
	 * Paint in increasing intensity
	 */
	j = 0;
	while (i < ncolors && j < num_per_range)
	{
	    red[i]   = j * factor;
	    green[i] = MAX_INTEN;
	    blue[i]  = 0;
	    i++; j++;
	}

	/*
	 * color the fourth range
	 * Paint in decreasing intensity
	 */
	j = 0;
	while (i < ncolors && j < num_per_range)
	{
	    red[i]   = 0;
	    green[i] = MAX_INTEN;
	    blue[i]  = MAX_INTEN - j * factor;
	    i++; j++;
	}

	/*
	 * color the fifth range
	 * Paint in increasing intensity
	 */
	j = 0;
	while (i < ncolors && j < num_per_range)
	{
	    red[i]   = j * factor;
	    green[i] = 0;
	    blue[i]  = MAX_INTEN;
	    i++; j++;
	}

	/*
	 * Now do the last range in decreasing intensity
	 * The size of this range is ncolors/6 + any
	 * remainer as ncolors/6 does not always have
	 * an even remainer
	 */
	num_per_range += (ncolors - (num_per_range * num_ranges));
      	factor = MAX_INTEN/((double) num_per_range -1);
	j = 0;
	while (i < ncolors && j < num_per_range)
	{
	   red[i]   = 0;
	   green[i] = MAX_INTEN - j * factor;
	   blue[i]  = MAX_INTEN;
	   i++; j++;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: color_rgb_cube - color according to rgb cube
|
|       Purpose: We will try to follow the edges of the RGB color cube.
|		 This is done by setting a value of one color to maximum
|		 intensity, setting another to zero, then letting the other
|		 color range from zero to maximum intensity.  Then set the
|		 ranging color to zero, and let the other color range.  The
|		 repeat setting other colors to max.  Hence we get:
|
|		 Range1 - red (range), 	green (min), 	blue (max)
|		 Range2 - red (max), 	green (min), 	blue (range)
|		 Range3 - red (max),	green (range),	blue (min)
|		 Range4 - red (range),	green (max),	blue(min)
|		 Range5 - red (min),	green (max),	blue(range)
|		 Range6 - red (min),	green (range),	blue(max)	
|
|		 Hence our factor is MAX/(ncolors/6)
|		 and to make it look better, when each color saturated we go
|		 from MAX to low to MAX.
|
|		 Lastly, Range 6 is larger as since ncolors/6 may not be an
|		 even number, it has ncolor/6 + the remainer so that we paint
|		 a total of ncolor colors
|
|    Written By: Mark Young
|          Date: Jun 10, 1993
| Modifications: Converted from khoros 1.0 and changed to use data services (MY)
|
------------------------------------------------------------*/
/* ARGSUSED */
static void color_rgb_cube(
   kobject image,
   double   *red,
   double   *green,
   double   *blue,
   int     ncolors)
{
	int	i, j,		/* famous loop variables */
		num_per_range,	/* number to color per range*/
		num_ranges = 6;	/* Each plane (R, G or B has 2 ranges */
	double factor;		/* saturation factor used in auto_color */

	num_per_range = ncolors/num_ranges;	/* ncolors per range */
	if (num_per_range < 2)
	{
	   kerror("kappserv", "color_rgb_cube", "Sorry! This image \
contain '%d' colors.  A minimum of 12 colors is needed for this auto coloring \
algorithm.", ncolors);
	   return;
	}
      	factor = MAX_INTEN/((double) num_per_range -1);

	/*
	 * Start at blue and go to magenta.
	 * Color the first range (start at blue and increase red)
	 */
	i = j = 0;
	while (i < ncolors && j < num_per_range)
	{
	   red[i]   = j * factor;
	   green[i] = 0;
	   blue[i]  = MAX_INTEN;
	   i++; j++;
	}

	/*
	 * Start at megenta and go to red.
	 * Color the second range (set red at max and decrease blue)
	 */
	j = 0;
	while (i < ncolors && j < num_per_range)
	{
	   red[i]   = MAX_INTEN;
	   green[i] = 0;
	   blue[i]  = MAX_INTEN - j * factor;
	   i++; j++;
	}

	/*
	 * Start at red and go to yellow.
	 * Color the third range (set red at max and increase green)
	 */
	j = 0;
	while (i < ncolors && j < num_per_range)
	{
	   red[i]   = MAX_INTEN;
	   green[i] = j * factor;
	   blue[i]  = 0;
	   i++; j++;
	}

	/*
	 * Start at yellow and go to green.
	 * Color the fourth range (set green at max and decrease red)
	 */
	j = 0;
	while (i < ncolors && j < num_per_range)
	{
	   red[i]   = MAX_INTEN - j * factor;
	   green[i] = MAX_INTEN;
	   blue[i]  = 0;
	   i++; j++;
	}

	/*
	 * Start at green and go to cyan.
	 * Color the fifth range (set green at max and increase blue)
	 */
	j = 0;
	while (i < ncolors && j < num_per_range)
	{
	   red[i]   = 0;
	   green[i] = MAX_INTEN;
	   blue[i]  = j * factor;
	   i++; j++;
	}

	/*
	 * Start at cyan and go to blue.
	 * Color the last range (set blue at max and decrease green)
	 *
	 * Now do the last range in decreasing intensity
	 * The size of this range is ncolors/6 + any
	 * remainer as ncolors/6 does not always have
	 * an even remainer
	 */
	num_per_range += (ncolors - (num_per_range * num_ranges));
      	factor = MAX_INTEN/((double) num_per_range -1);
	j = 0;
	while (i < ncolors && j < num_per_range)
	{
	   red[i]   = 0;
	   green[i] = MAX_INTEN - j * factor;
	   blue[i]  = MAX_INTEN;
	   i++; j++;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: color_greyscale - color according to linear greyscale
|
|       Purpose: This routine colors an image using a linear
|		 greyscale.  It starts at 0 and increases the
|		 RGB values to max intensity (MAX_INTEN) for the
|		 screen.  The increasing factor is computed by
|		 the following formula:
|
|				MAX_INTEN / (ncolors -1)
|
|		 This gives us a black beginning value and
|		 a white end value.
|
|    Written By: Mark Young
|          Date: Jun 10, 1993
| Modifications: Converted from khoros 1.0 and changed to use data services (MY)
|
------------------------------------------------------------*/
/* ARGSUSED */
static void color_greyscale(
   kobject image,
   double   *red,
   double   *green,
   double   *blue,
   int     ncolors)
{
	int   i;
	double factor;

	/*
	 *  make sure that the colors are linearized between black (0,0,0)
	 *  and white (MAX, MAX, MAX).
	 */
	if (ncolors == 1) 
	    factor = MAX_INTEN;
	else factor = MAX_INTEN / ((double) ncolors -1);

	for (i = 0; i < ncolors; i++)
	{
	   red[i]   =
	   green[i] = 
	   blue[i]  = i * factor;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: color_strequalize - color according to RGB stretch/equalize
|
|       Purpose: This routine colors an image using a linear
|		 scale for RGB.  It starts at 0 and increases the
|		 RGB values to max intensity (MAX_INTEN) for the
|		 each component of RED, GREEN, BLUE.  The increasing
|		 factor is computed by the following formula:
|
|		 This gives us a black beginning value and
|		 a white end value.
|
|    Written By: Mark Young
|          Date: Jun 10, 1993
| Modifications: Converted from khoros 1.0 and changed to use data services (MY)
|
------------------------------------------------------------*/
/* ARGSUSED */
static void color_strequalize(
   kobject image,
   double   *red,
   double   *green,
   double   *blue,
   int     ncolors,
   int	   type)
{
	int   i;
	double rmin, rmax, gmin, gmax, bmin, bmax;


	/*
	 *  make sure that the colors are linearized between black (0,0,0)
	 *  and white (MAX, MAX, MAX).
	 */
	data_min_max(red,   ncolors, &rmin, &rmax);
	data_min_max(green, ncolors, &gmin, &gmax);
	data_min_max(blue,  ncolors, &bmin, &bmax);
	if (type == KEQUALIZE)
	{
	   rmin = gmin = bmin = kmin3(rmin, gmin, bmin);
	   rmax = gmax = bmax = kmax3(rmax, gmax, bmax);
	}

	for (i = 0; i < ncolors; i++)
	{
	   red[i]   = (red[i] / (rmax - rmin)) * MAX_INTEN + rmin;
	   green[i] = (green[i] / (gmax - gmin)) * MAX_INTEN + gmin;
	   blue[i]  = (blue[i] / (bmax - bmin)) * MAX_INTEN + bmin;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: color_invert - invert the current color values
|
|       Purpose: This routine inverts an image using the current
|		 colors.  It takes each of three color values
|		 in the xcolor array and subtracts it from the
|		 max intensity (MAX_INTEN).  This is done for
|		 the first ncolors.
|
|    Written By: Mark Young
|          Date: Jun 10, 1993
| Modifications: Converted from khoros 1.0 and changed to use data services (MY)
|
------------------------------------------------------------*/
/* ARGSUSED */
static void color_invert(
   kobject image,
   double   *red,
   double   *green,
   double   *blue,
   int     ncolors)
{
	int   i;

	for (i = 0; i < ncolors; i++)
	{
	   red[i]   = MAX_INTEN - red[i];
	   green[i] = MAX_INTEN - green[i];
	   blue[i]  = MAX_INTEN - blue[i];
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: color_reverse - reverse the current color values
|
|       Purpose: This routine reverse an image using the current
|		 colors.  It takes each of three color values
|		 in the color arrays and swaps it from the
|		 (ncolor - ith) entry.  This is done for the
|		 first ncolors.
|
|    Written By: Mark Young
|          Date: Jun 10, 1993
| Modifications: Converted from khoros 1.0 and changed to use data services (MY)
|
------------------------------------------------------------*/
/* ARGSUSED */
static void color_reverse(
   kobject image,
   double   *red,
   double   *green,
   double   *blue,
   int     ncolors)
{
	int   i;
	double red_tmp, green_tmp, blue_tmp;

	for (i = 0; i < ncolors/2; i++)
	{
	   red_tmp   = red[i];
	   green_tmp = green[i];
	   blue_tmp  = blue[i];

	   red[i]   = red[ncolors - (i+1)];
	   green[i] = green[ncolors - (i+1)];
	   blue[i]  = blue[ncolors - (i+1)];

	   red[ncolors - (i+1)]   = red_tmp;
	   green[ncolors - (i+1)] = green_tmp;
	   blue[ncolors - (i+1)]  = blue_tmp;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: color_row_rotate - rotate left/right the current color values
|
|       Purpose: This routine rotates left an image using the current
|		 colors.  It takes each of three color values
|		 in the color arrays and shifts it left from the
|		 (ith + 1) entry.  This is done for the first ncolors.
|
|    Written By: Mark Young
|          Date: Jun 10, 1993
| Modifications: Converted from khoros 1.0 and changed to use data services (MY)
|
------------------------------------------------------------*/
/* ARGSUSED */
static void color_row_rotate(
   kobject image,
   double   *red,
   double   *green,
   double   *blue,
   int     ncolors,
   int	   type)
{
	int   i;
	double red_tmp, green_tmp, blue_tmp;


	if (type == KROW_ROTLEFT)
	{
	   red_tmp   = red[0];
	   green_tmp = green[0];
	   blue_tmp  = blue[0];
	   for (i = 1; i < ncolors; i++)
	   {
	      red[i-1]   = red[i];
	      green[i-1] = green[i];
	      blue[i-1]  = blue[i];
	   }
	   red[ncolors-1]   = red_tmp;
	   green[ncolors-1] = green_tmp;
	   blue[ncolors-1]  = blue_tmp;
	}
	else
	{
	   red_tmp   = red[ncolors-1];
	   green_tmp = green[ncolors-1];
	   blue_tmp  = blue[ncolors-1];
	   for (i = ncolors-2; i >= 0; i--)
	   {
	      red[i+1]   = red[i];
	      green[i+1] = green[i];
	      blue[i+1]  = blue[i];
	   }
	   red[0]   = red_tmp;
	   green[0] = green_tmp;
	   blue[0]  = blue_tmp;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: color_rgb_rotate - rotate left/right the current rgb values
|
|       Purpose: This routine rotates left an image using the current
|		 colors.  It takes each of three color values
|		 in the color arrays and shifts it left from the
|		 (ith + 1) entry.  This is done for the first ncolors.
|
|    Written By: Mark Young, Jeremy Worley, Steve Kubica
|          Date: Jun 04, 1994
| Modifications: 
|
------------------------------------------------------------*/
/* ARGSUSED */
static void color_rgb_rotate(
   kobject image,
   double   *red,
   double   *green,
   double   *blue,
   int     ncolors,
   int	   type)
{
	int   i;
	double red_tmp, green_tmp, blue_tmp;


	if (type == KRED_ROTLEFT || type == KGREEN_ROTLEFT ||
	    type == KBLUE_ROTLEFT)
	{
	   red_tmp   = red[0];
	   green_tmp = green[0];
	   blue_tmp  = blue[0];
	   for (i = 1; i < ncolors; i++)
	   {
	      if (type == KRED_ROTLEFT)   red[i-1]   = red[i];
	      if (type == KGREEN_ROTLEFT) green[i-1] = green[i];
	      if (type == KBLUE_ROTLEFT)  blue[i-1]  = blue[i];
	   }
	   if (type == KRED_ROTLEFT)   red[ncolors-1]   = red_tmp;
	   if (type == KGREEN_ROTLEFT) green[ncolors-1] = green_tmp;
	   if (type == KBLUE_ROTLEFT)  blue[ncolors-1]  = blue_tmp;
	}
	else
	{
	   red_tmp   = red[ncolors-1];
	   green_tmp = green[ncolors-1];
	   blue_tmp  = blue[ncolors-1];
	   for (i = ncolors-2; i >= 0; i--)
	   {
	      if (type == KRED_ROTRIGHT)   red[i+1]   = red[i];
	      if (type == KGREEN_ROTRIGHT) green[i+1] = green[i];
	      if (type == KBLUE_ROTRIGHT)  blue[i+1]  = blue[i];
	   }
	   if (type == KRED_ROTRIGHT)   red[0]   = red_tmp;
	   if (type == KGREEN_ROTRIGHT) green[0] = green_tmp;
	   if (type == KBLUE_ROTRIGHT)  blue[0]  = blue_tmp;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: color_chain_rotate - rotate left/right the current color values
|
|       Purpose: This routine rotates left or right as a chain.
|		 An image using the current colors.  It takes each
|		 of three color values in the color arrays and shifts
|		 it left from the (ith + 1) entry.  Where the last element
|		 of the red array, is the first element of the green array,
|		 and the last element of green, is the first to the blue.
|		 Finally the final element of the blue is the first element
|		 of the red array.  This is done for the first ncolors.
|
|    Written By: Mark Young, Steve Kubica, Jeremy Worley
|          Date: Jun 04, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void color_chain_rotate(
   kobject image,
   double   *red,
   double   *green,
   double   *blue,
   int     ncolors,
   int	   type)
{
	int   i;
	double red_tmp, green_tmp, blue_tmp;


	if (type == KCHAIN_ROTLEFT)
	{
	   red_tmp   = red[0];
	   green_tmp = green[0];
	   blue_tmp  = blue[0];
	   for (i = 1; i < ncolors; i++)
	   {
	      red[i-1]   = red[i];
	      green[i-1] = green[i];
	      blue[i-1]  = blue[i];
	   }
	   red[ncolors-1]   = green_tmp;
	   green[ncolors-1] = blue_tmp;
	   blue[ncolors-1]  = red_tmp;
	}
	else
	{
	   red_tmp   = red[ncolors-1];
	   green_tmp = green[ncolors-1];
	   blue_tmp  = blue[ncolors-1];
	   for (i = ncolors-2; i >= 0; i--)
	   {
	      red[i+1]   = red[i];
	      green[i+1] = green[i];
	      blue[i+1]  = blue[i];
	   }
	   red[0]   = blue_tmp;
	   green[0] = red_tmp;
	   blue[0]  = green_tmp;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: color_rainbow - color according to rainbow
|
|       Purpose: This routine colors an image using rainbow
|  		 colors.  The idea is from Philip R. Thompson
|		 (phils@athena.mit.edu).  In his xim3c program
|		 he used a technique of setting the saturation and
|		 value to max, and then varing the hue for the number
|		 of desired steps.  For each (hsv) he would then convert
|		 it to (rgb) for the desired rainbow affect.
|
|		 This is done for the first ncolors which is what we
|		 use as our step.
|
|    Written By: Mark Young
|          Date: Jun 10, 1993
| Modifications: Converted from khoros 1.0 and changed to use data services (MY)
|
------------------------------------------------------------*/
/* ARGSUSED */
static void color_rainbow(
   kobject image,
   double   *red,
   double   *green,
   double   *blue,
   int     ncolors)
{
	int      j;
	double    hue, saturation, value;

	saturation = value = 1.0;
	for (j = 0; j < ncolors; j++)
	{
	   hue = ((double) j)/ncolors;
	   HSV_to_RGB(hue, saturation, value, red[j], green[j], blue[j]);
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: color_greycode - color according to greycode
|
|       Purpose: This routine colors an image using greycode
|  		 colors.  The idea is from  XXXX because of
|		 YYYY.
|
|    Written By: Jeremy Worley
|          Date: Jul 02, 1993
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void color_greycode(
   kobject image,
   double   *red,
   double   *green,
   double   *blue,
   int     ncolors)
{
	int   size;
	double *temp;

	/*
	 *  Create temporary space for interpolating
	 */
	size = ncolors + ncolors/8;
	temp = (double *) kmalloc(size * sizeof(double));

	/*
	 *  Set and stretch the red ramp by calling kdata_sample() to
	 *  bi-linear interpolate the red ramp.
	 */
	red[0] = red[3] = red[4] = red[7] = 0;
	red[1] = red[2] = red[5] = red[6] = MAX_INTEN;
	kdata_sample((char *) red, 8, size, KDOUBLE, KFIRST_ORDER,
		KREAL, 0.0, 0.0, 0.0, 0.0, (char *) temp);
	kmemcpy(red, temp, ncolors * sizeof(double));

	/*
	 *  Set and stretch the red ramp by calling kdata_sample() to
	 *  bi-linear interpolate the red ramp.
	 */
	green[0] = green[1] = green[6] = green[7] = 0;
	green[2] = green[3] = green[4] = green[5] = MAX_INTEN;
	kdata_sample((char *) green, 8, size, KDOUBLE, KFIRST_ORDER,
		KREAL, 0.0, 0.0, 0.0, 0.0, (char *) temp);
	kmemcpy(green, temp, ncolors * sizeof(double));

	/*
	 *  Set and stretch the red ramp by calling kdata_sample() to
	 *  bi-linear interpolate the red ramp.
	 */
	blue[0] = blue[1] = blue[2] = blue[3] = 0;
	blue[4] = blue[5] = blue[6] = blue[7] = MAX_INTEN;
	kdata_sample((char *) blue, 8, size, KDOUBLE, KFIRST_ORDER,
		KREAL, 0.0, 0.0, 0.0, 0.0, (char *) temp);
	kmemcpy(blue, temp, ncolors * sizeof(double));

	/*
	 *  Free the temporary space
	 */
	kfree(temp);
}

/*-----------------------------------------------------------
|
|  Routine Name: color_disjoint - color according to greycode
|
|       Purpose: This routine colors an image using greycode
|  		 colors.  The idea is from  XXXX because of
|		 YYYY.
|
|    Written By: Jeremy Worley
|          Date: Jul 02, 1993
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void color_disjoint(
   kobject image,
   double   *red,
   double   *green,
   double   *blue,
   int     ncolors)
{
	int   size;
	double *temp;

	/*
	 *  Create temporary space for interpolating
	 */
	size = ncolors + ncolors/8;
	temp = (double *) kmalloc(size * sizeof(double));

	/*
	 *  Set and stretch the red ramp by calling kdata_sample() to
	 *  bi-linear interpolate the red ramp.
	 */
	red[0] = red[1] = red[4] = red[5] = 0;
	red[2] = red[3] = red[6] = red[7] = MAX_INTEN;
	kdata_sample((char *) red, 8, size, KDOUBLE, KFIRST_ORDER,
		KREAL, 0.0, 0.0, 0.0, 0.0, (char *) temp);
	kmemcpy(red, temp, ncolors * sizeof(double));

	/*
	 *  Set and stretch the red ramp by calling kdata_sample() to
	 *  bi-linear interpolate the red ramp.
	 */
	green[0] = green[3] = green[5] = green[6] = 0;
	green[1] = green[2] = green[4] = green[7] = MAX_INTEN;
	kdata_sample((char *) green, 8, size, KDOUBLE, KFIRST_ORDER,
		KREAL, 0.0, 0.0, 0.0, 0.0, (char *) temp);
	kmemcpy(green, temp, ncolors * sizeof(double));

	/*
	 *  Set and stretch the red ramp by calling kdata_sample() to
	 *  bi-linear interpolate the red ramp.
	 */
	blue[0] = blue[2] = blue[4] = blue[6] = 0;
	blue[1] = blue[3] = blue[5] = blue[7] = MAX_INTEN;
	kdata_sample((char *) blue, 8, size, KDOUBLE, KFIRST_ORDER,
		KREAL, 0.0, 0.0, 0.0, 0.0, (char *) temp);
	kmemcpy(blue, temp, ncolors * sizeof(double));

	/*
	 *  Free the temporary space
	 */
	kfree(temp);
}

/*-----------------------------------------------------------
|
|  Routine Name: color_hls_spiral - color according to hls spiral
|
|       Purpose: This routine colors an image using a spiral
|		 thru the rgb color cube.  The idea is done by
|		 converting HSL to RGB.  This is a variant of
|		 color_rainbow() routine.
|
|    Written By: Mark Young
|          Date: Jun 10, 1993
| Modifications: Converted from khoros 1.0 and changed to use data services (MY)
|
------------------------------------------------------------*/
/* ARGSUSED */
static void color_hls_spiral(
   kobject image,
   double   *red,
   double   *green,
   double   *blue,
   int     ncolors)
{
	int      j;
	double    hue, light, saturation;

	for (j = 0; j < ncolors; j++)
	{
	   hue   = ((double) j)/ncolors;
	   light = ((double) j)/ncolors;

	   saturation = ((double) 2*j)/ncolors;
	   if (saturation > 1.0)
	      saturation = 2.0 - saturation;

	   HLS_to_RGB(hue, light, saturation, red[j], green[j], blue[j]);
	}
}


/*-----------------------------------------------------------
|
|  Routine Name: color_rgb_spiral- - color according to rgb spiral
|
|       Purpose: This routine colors an image using a spiral
|		 thru the rgb color cube.
|
|    Written By: Jeremy Worley & Mark Young
|          Date: Jul 02, 1993
| Modifications: 
|
------------------------------------------------------------*/
/* ARGSUSED */
static void color_rgb_spiral(
   kobject image,
   double   *red,
   double   *green,
   double   *blue,
   int     ncolors)
{
	int   size;
	double *temp;

	/*
	 *  Create temporary space for interpolating
	 */
	size = ncolors + ncolors/8;
	temp = (double *) kmalloc(size * sizeof(double));

	/*
	 *  Set and stretch the red ramp by calling kdata_sample() to
	 *  bi-linear interpolate the red ramp.
	 */
	red[0] = red[3] = red[4] = red[5] = 0;
	red[1] = red[2] = red[6] = red[7] = MAX_INTEN;
	kdata_sample((char *) red, 8, size, KDOUBLE, KFIRST_ORDER, KREAL,
		0.0, 0.0, 0.0, 0.0, (char *) temp);
	kmemcpy(red, temp, ncolors * sizeof(double));

	/*
	 *  Set and stretch the red ramp by calling kdata_sample() to
	 *  bi-linear interpolate the red ramp.
	 */
	green[0] = green[1] = green[5] = green[6] = 0;
	green[2] = green[3] = green[4] = green[7] = MAX_INTEN;
	kdata_sample((char *) green, 8, size, KDOUBLE, KFIRST_ORDER, KREAL,
		0.0, 0.0, 0.0, 0.0, (char *) temp);
	kmemcpy(green, temp, ncolors * sizeof(double));

	/*
	 *  Set and stretch the red ramp by calling kdata_sample() to
	 *  bi-linear interpolate the red ramp.
	 */
	blue[0] = blue[1] = blue[2] = blue[3] = 0;
	blue[4] = blue[5] = blue[6] = blue[7] = MAX_INTEN;
	kdata_sample((char *) blue, 8, size, KDOUBLE, KFIRST_ORDER, KREAL,
		0.0, 0.0, 0.0, 0.0, (char *) temp);
	kmemcpy(blue, temp, ncolors * sizeof(double));

	/*
	 *  Free the temporary space
	 */
	kfree(temp);
}


/*-----------------------------------------------------------
|
|  Routine Name: color_rgb_rings - color according to rgb rings
|
|       Purpose: This routine colors an image by dividing into
|		 different rings.  Each ring is then colored
|		 by varying the hue around the ring (0.0 - 1.0).
|		 The number of points in the ring varies according
|		 to the number of points in the array.  And the number
|		 of points in each ring varies according to which
|		 ring.  The number of rings is based on the 
|
|    Written By: Mark Young
|          Date: Jun 10, 1993
| Modifications: Converted from khoros 1.0 and changed to use data services (MY)
|
------------------------------------------------------------*/
/* ARGSUSED */
static void color_hls_rings(
   kobject image,
   double   *red,
   double   *green,
   double   *blue,
   int     ncolors)
{
	int      i, j, num, iter;
	double    hue, light, saturation, sat, intv;


	num = klog((double) ncolors)/klog(2.0) - 1.0;
	if (num > 0)
	{
	   intv = 1.0/num;
	   iter = ncolors/num;
	}
	else
	{
	   intv = 0.5;
	   iter = 1.0;
	}
	light = intv;
	saturation = 2*intv;

	for (i = j = 0; i < ncolors; i++, j++)
	{
	   j %= iter;
	   hue = ((double) j)/iter;
	   if (saturation > 1.0)
	      sat = 2.0 - saturation;
	   else
	      sat = saturation;

	   HLS_to_RGB(hue, light, sat, red[i], green[i], blue[i]);
	   light      += intv;
	   saturation += 2*intv;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: color_hsv_rings - color according to hsv rings
|
|       Purpose: This routine colors an image by dividing into
|		 different rings.  Each ring is then colored
|		 by varying the hue around the ring (0.0 - 1.0).
|		 The number of points in the ring varies according
|		 to the number of points in the array.  And the number
|		 of points in each ring varies according to which
|		 ring.  The number of rings is based on the 
|
|    Written By: Mark Young
|          Date: Jun 10, 1993
| Modifications: Converted from khoros 1.0 and changed to use data services (MY)
|
------------------------------------------------------------*/
/* ARGSUSED */
static void color_hsv_rings(
   kobject image,
   double   *red,
   double   *green,
   double   *blue,
   int     ncolors)
{
	int      j, i, num, iter;
	double    hue, value, saturation, intv;


	num = klog((double) ncolors)/klog(2.0) - 1.0;
	if (num > 0)
	{
	   intv = 1.0/num;
	   iter = ncolors/num;
	}
	else
	{
	   intv = 0.5;
	   iter = 1.0;
	}

	saturation = value = intv;
	for (i = j = 0; i < ncolors; i++, j++)
	{
	   j %= iter;
	   hue = ((double) j)/iter;
	   HSV_to_RGB(hue, value, saturation, red[i], green[i], blue[i]);
	   value      += intv;
	   saturation += intv;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: color_rgb_random - color according to random coloring
|
|       Purpose: colors the image using a random function.  Each
|		 red, green, blue component is assigned a unifor
|		 aussian random number between 0 and MAX_INTEN
|
|    Written By: Mark Young
|          Date: Jun 10, 1993
| Modifications: Converted from khoros 1.0 and changed to use data services (MY)
|
------------------------------------------------------------*/
/* ARGSUSED */
static void color_rgb_random(
   kobject image,
   double   *red,
   double   *green,
   double   *blue,
   int     ncolors)
{
	int      i;

	for (i = 0; i < ncolors; i++)
	{
	   red[i]   = kurng() * MAX_INTEN;
	   green[i] = kurng() * MAX_INTEN;
	   blue[i]  = kurng() * MAX_INTEN;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: color_density_slice - color according to density
|
|       Purpose: This routine colors an image by density slice.
|		 Density Slice is a data dependent colormapping
|		 algorithm which assigns colors according to the
|		 distribution of the image's histogram.  When the
|		 histogram of the image is computed, the maximum
|		 number of pixels per cell is found.  This is used
|		 to determine the bin size for the density slicing
|		 operation.  Once the bin sizes have been determined,
|		 the histogram is searched for cells which fall into
|		 each bin, and these pixel values are mapped to colors
|		 which vary from green to blue to red (follows the first
|		 two legs of the CIE diagram triangle).  Blue represents
|		 the cell with the fewest number of pixels in the image,
|		 and red represents the cell with the highest number of pixels.
|
|		    Note: A colormapping which covers the spectrum from violet
|		          to red will be implemented in the future.
|
|		 A simple example of density slicing is given below.
|		 In the example histogram, there are 10 possible pixel
|		 values, or cells. The density slice will divide the
|		 histogram into 3 slices.  Cells which occupy enough
|		 pixels in the image to fall into bin 3 will be assigned
|		 red, those that fall into bin 2 will be green and those
|		 that fall into bin 1 will be blue.  Therefore, in this
|		 example, pixels 3 and 8 will be red, pixels 0, 4, 5,
|		 and 7 will be green, and pixels 1 and 6 will be blue.
|
|    Written By: Mark Young
|          Date: Jun 10, 1993
| Modifications: Converted from khoros 1.0 and changed to use data services (MY)
|
------------------------------------------------------------*/
/* ARGSUSED */
static void color_density_slice(
   kobject image,
   double   *red,
   double   *green,
   double   *blue,
   int     ncolors)
{
	int	 i, j, num;
	double	 hfact, value, max;
	unsigned long temp, *histogram = NULL;

	int	 num_per_range,		/* number to color per range  */
		 num_ranges = 6;	/* r-o-y-g-b-i-v  (6 ranges) */
	double	 factor;		/* saturation factor used in coloring */

	if (histogram == NULL)
	   return;

	num_per_range = ncolors/num_ranges;
	if (num_per_range < 2)
	{
	   kerror("kappserv", "color_density_slice", "Sorry! This image \
contain '%d' colors.  A minimum of 4 colors is needed for this auto coloring \
algorithm.", ncolors);
	   return;
	}
	factor = MAX_INTEN/((double) num_per_range -1);

	/*
	 *  Sort the histogram into ascending order.  The index array
	 *  is used to tell us which color the histogram came from.
	 */
	for (i = 0; i < ncolors -1; i++)
	{
	   num = i;
	   for (j = i+1; j < ncolors; j++)
	   {
	      if (histogram[j] < histogram[num]) num = j;
	   }

	   if (i != num)
	   {
	      temp = histogram[i];
	      histogram[i] = histogram[num];
	      histogram[num] = temp;
	   }
	}
	max   = histogram[ncolors -1];
	hfact = max/((double) ncolors -1);

	/*
	 *  Do the first range..
	 */
	for (i = j = 0, value = hfact; i < ncolors && value < max/2; i++)
	{
	   while (value < histogram[i])
	   {
	      value += hfact;
	      j++;
	   }
	   red[i]   = 0;
	   green[i] = j * factor;
	   blue[i]  = MAX_INTEN - j * factor;
	}

	/*
	 *  Do the second range..
	 */
	for (j = 0; i < ncolors; i++)
	{
	   while (value < histogram[i])
	   {
	      value += hfact;
	      j++;
	   }
	   red[i]   = j * factor;
	   green[i] = MAX_INTEN - j * factor;
	   blue[i]  = 0;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: color_rgbfilter - filter (RGB) current color values
|
|       Purpose: This routine will filter (by masking) either the
|		 red, green, or blue color data.
|
|    Written By: Mark Young
|          Date: Jun 10, 1993
| Modifications: Converted from khoros 1.0 and changed to use data services (MY)
|
------------------------------------------------------------*/
/* ARGSUSED */
static void color_rgbfilter(
   kobject image,
   double   *red,
   double   *green,
   double   *blue,
   int     ncolors,
   int	   type)
{
	if (type == KRED_FILTER)
	   kmemset((char *) red, 0, sizeof(double) * ncolors);
	else if (type == KGREEN_FILTER)
	   kmemset((char *) green, 0, sizeof(double) * ncolors);
	else if (type == KBLUE_FILTER)
	   kmemset((char *) blue, 0, sizeof(double) * ncolors);
}

/*-----------------------------------------------------------
|
|  Routine Name: color_col_rotate - column rotate left/right
|				    the current color values
|
|       Purpose: This routine rotates left an image using the current
|		 colors.  It takes each of three color values
|		 in the color arrays and shifts it left from the
|		 (ith + 1) entry.  This is done for the first ncolors.
|
|    Written By: Mark Young
|          Date: Jun 10, 1993
| Modifications: Converted from khoros 1.0 and changed to use data services (MY)
|
------------------------------------------------------------*/
/* ARGSUSED */
static void color_col_rotate(
   kobject image,
   double   *red,
   double   *green,
   double   *blue,
   int     ncolors,
   int	   type)
{
	int   i;
	double *swap1, *swap2, *swap3, tmp;


	if (type == KCOL_ROTLEFT)
	{
	   swap1 = red; swap2 = green; swap3 = blue;
	}
	else
	{
	   swap1 = blue; swap2 = green; swap3 = red;
	}

	for (i = 0; i < ncolors; i++)
	{
	   tmp = swap3[i];
	   swap3[i] = swap2[i];
	   swap2[i] = swap1[i];
	   swap1[i] = tmp;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: color_rgbswap - swap (RGB) current color values
|
|       Purpose: This routine will filter (by masking) either the
|		 red, green, or blue color data.
|
|    Written By: Mark Young
|          Date: Jun 10, 1993
| Modifications: Converted from khoros 1.0 and changed to use data services (MY)
|
------------------------------------------------------------*/
/* ARGSUSED */
static void color_rgbswap(
   kobject image,
   double   *red,
   double   *green,
   double   *blue,
   int     ncolors,
   int	   type)
{
	int   i;
	double *swap1, *swap2, tmp;


	if (type == KSWAP_REDGREEN)
	{
	   swap1 = red; swap2 = green;
	}
	else if (type == KSWAP_REDBLUE)
	{
	   swap1 = red; swap2 = blue;
	}
	else if (type == KSWAP_GREENBLUE)
	{
	   swap1 = green; swap2 = blue;
	}
	else return;

	for (i = 0; i < ncolors; i++, swap1++, swap2++)
	{
	   tmp = *swap1;
	   *swap1 = *swap2;
	   *swap2 = tmp;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: color_stddev - normalize data by their standard
|				deviation
|
|       Purpose: This routine normalizes the data according to their
|		 standard deviation.
|
|    Written By: Mark Young
|          Date: Jun 10, 1993
| Modifications: Converted from khoros 1.0 and changed to use data services (MY)
|
------------------------------------------------------------*/
/* ARGSUSED */
static void color_stddev(
   kobject image,
   double   *red,
   double   *green,
   double   *blue,
   int     ncolors)
{
}

/*-----------------------------------------------------------
|
|  Routine Name: color_sa_pseudo - color 
|
|       Purpose: This a pseudo color based on an article in Scientific
|		 American.  The pseudocolor is designed to convert a grey
|		 scale image and convert it to color image of the same
|		 intensity.  The algorithm takes the intensity and maps
|		 that to corresponding red, green, blue that has been
|		 choosen more for it's aesthetics than it's quantative
|		 value.  The article was discovered by Joe Fogler who
|		 typed in the and added a few corrections to make the
|		 pseudo color even more aestically pleasing.  The raw
|		 numbers were given to us by Joe,  which we put into a
|		 table and wrote the actual color_sa_pseudo routine.
|
|    Written By: Mark Young
| Values Supplied By: Original author unknown & Joe Fogler
|          Date: Jun 10, 1993
| Modifications: Converted from khoros 1.0 and changed to use data services (MY)
|
------------------------------------------------------------*/

typedef struct
{
	int num;
	unsigned char red, green, blue;
} SAElem;

static SAElem elems[] = {
	{ 3, 0, 0, 51 },
	{ 5, 0, 0, 86 },
	{ 5, 0, 0, 102 },
	{ 5, 0, 0, 117 },
	{ 5, 0, 0, 127 },
	{ 4, 0, 0, 137 },
	{ 5, 0, 0, 148 },
	{ 5, 0, 0, 158 },
	{ 5, 0, 0, 163 },
	{ 5, 0, 14, 168 },
	{ 5, 0, 28, 168 },
	{ 5, 0, 42, 168 },
	{ 5, 0, 56, 168 },
	{ 5, 0, 79, 158 },
	{ 5, 0, 94, 153 },
	{ 5, 0, 109, 148 },
	{ 4, 0, 119, 143 },
	{ 5, 0, 126, 139 },
	{ 5, 0, 132, 132 },
	{ 5, 0, 137, 126 },
	{ 5, 0, 143, 119 },
	{ 5, 0, 148, 96 },
	{ 5, 0, 153, 64 },
	{ 5, 0, 158, 32 },
	{ 5, 0, 163, 0 },
	{ 5, 109, 163, 0 },
	{ 4, 153, 158, 0 },
	{ 5, 173, 161, 0 },
	{ 5, 189, 157, 0 },
	{ 5, 203, 156, 1 },
	{ 5, 213, 156, 1 },
	{ 5, 223, 156, 1 },
	{ 5, 234, 156, 1 },
	{ 5, 234, 164, 1 },
	{ 5, 234, 172, 1 },
	{ 5, 234, 176, 1 },
	{ 4, 234, 176, 1 },
	{ 5, 234, 191, 1 },
	{ 5, 228, 198, 1 },
	{ 5, 228, 205, 1 },
	{ 5, 228, 209, 1 },
	{ 5, 228, 213, 1 },
	{ 5, 228, 217, 1 },
	{ 5, 228, 224, 1 },
	{ 5, 228, 228, 1 },
	{ 5, 234, 234, 52 },
	{ 5, 236, 236, 147 },
	{ 4, 241, 241, 157 },
	{ 5, 246, 246, 173 },
	{ 5, 250, 250, 184 },
	{ 5, 252, 252, 197 },
	{ 5, 254, 254, 216 },
	{ 3, 255, 255, 240 },
};

/* ARGSUSED */
static void color_sa_pseudo(
   kobject image,
   double   *red,
   double   *green,
   double   *blue,
   int     ncolors)
{
	int	i, j, l, num;


	num = knumber(elems);
	for (i = 0, l = 0; i < num && l < ncolors; i++)
	{
	   for (j = 0; j < elems[i].num && l < ncolors; j++)
	   {
	      red[l]   = elems[i].red;
	      green[l] = elems[i].green;
	      blue[l]  = elems[i].blue;
	      l++;
	   }
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: kmap_map_autocolor - image service routine for performing
|				  automatic coloring
|
|       Purpose: This should routine performs automatic coloring of
|		 images.  The current images supported are:
|
|			KRGB_CUBE       
|			KRGB_TRIANGLE   
|			KRANDOM     
|			KRGB_SPIRAL     
|			KHLS_SPIRAL     
|			KHSV_RINGS      
|			KHLS_RINGS      
|			KRGB_DISTANCE   
|			KCIE_DIAGRAM    
|			KDENSITY_SLICE  
|			KGREYSCALE      
|			KEQUALIZE       
|			KSTRETCH        
|			KSTDDEV         
|			KSA_PSEUDO      
|			KRAINBOW        
|			KDISJOINT       
|			KGREYCODE       
|			KORIGINAL       
|			KMAP332         
|
|         Input: object - object to color
|                type   - the map_autocolor type
|    Written By: Mark Young
|          Date: Jun 10, 1993
| Modifications:
|
------------------------------------------------------------*/

void kmap_map_autocolor(
   kobject original,
   int     type)
{
	kobject object;
	double *maps, *red, *green, *blue;
	int    i, j, width, height, num_maps, available;


	/*
	 *  Check to see if we have a copy of the original colormap..
	 */
	if (type != KORIGINAL && !kdms_query_segment(original,ORIGINAL_SEGMENT))
	   kmap_map_autocolor(original, KORIGINAL);

	/*
	 *  Go ahead and make a referenced copy since we need to set some
	 *  attributes.
	 */
	if ((object = kdms_reference(original)) == NULL)
	{
	   kerror("kappserv", "kmap_map_autocolor", "Unable to make a copy \
of the input object in which to map_autocolor");
	   return;
	}

	/*
	 *  Make sure that the colormap exists...
	 */
	available = initialize_maps(object);

	/* 
	 * get number of rows (width), cols (height), maps (maps)
	 */
	kpds_get_attribute(object, KPDS_MAP_SIZE, &width, &height, &num_maps,
			NULL, NULL);

	/* 
	 * set the map data to be double
	 */
	kpds_set_attributes(object,
			KPDS_MAP_DATA_TYPE, KDOUBLE,
			KPDS_MAP_REGION_SIZE, 3, height, 1, 1, 1,
			NULL);
	if (type == KORIGINAL || type == KINVERT_ORIG)
	   restore_original(original);

	/*
	 *  Allocate the colormap information.
	 */
	red   = (double *) kmalloc(height * sizeof(double));
	green = (double *) kmalloc(height * sizeof(double));
	blue  = (double *) kmalloc(height * sizeof(double));
	for (i = 0, maps = NULL; i < num_maps; i++)
	{
	   kpds_set_attribute(object, KPDS_MAP_POSITION, 0, 0, i, 0, 0);
	   maps = (double *) kpds_get_data(object, KPDS_MAP_REGION, maps);
	   for (j = 0; j < height; j++)
	   {
	      red[j]   = (!available) ? j : maps[j*3];
	      green[j] = (!available) ? j : maps[j*3 + 1];
	      blue[j]  = (!available) ? j : maps[j*3 + 2];
	   }

	   /*
	    *  Find out which type of automatic coloring they wish to
	    *  perform.
	    */
	   switch(type)
	   {
	      case KRGB_CUBE:
		   color_rgb_cube(object, red, green, blue, height);
		   break;

	      case KRANDOM:
		   color_rgb_random(object, red, green, blue, height);
		   break;

	      case KRGB_TRIANGLE:
		   color_rgb_triangle(object, red, green, blue, height);
		   break;

	      case KRGB_SPIRAL:
		   color_rgb_spiral(object, red, green, blue, height);
		   break;

	      case KHLS_SPIRAL:
		   color_hls_spiral(object, red, green, blue, height);
		   break;

	      case KHSV_RINGS:
		   color_hsv_rings(object, red, green, blue, height);
		   break;

	      case KHLS_RINGS:
		   color_hls_rings(object, red, green, blue, height);
		   break;

	      case KRGB_DISTANCE:
		   color_rgb_distance(object, red, green, blue, height);
		   break;

	      case KCIE_DIAGRAM:
		   color_cie_diagram(object, red, green, blue, height);
		   break;

	      case KDENSITY_SLICE:
		   color_density_slice(object, red, green, blue, height);
		   break;

	      case KGREYSCALE:
		   color_greyscale(object, red, green, blue, height);
		   break;

	      case KSTRETCH:
	      case KEQUALIZE:
		   color_strequalize(object, red, green, blue, height, type);
		   break;

	      case KSTDDEV:
		   color_stddev(object, red, green, blue, height);
		   break;

	      case KSA_PSEUDO:
		   color_sa_pseudo(object, red, green, blue, height);
		   break;

	      case KRAINBOW:
		   color_rainbow(object, red, green, blue, height);
		   break;

	      case KGREYCODE:
		   color_greycode(object, red, green, blue, height);
		   break;

	      case KDISJOINT:
		   color_disjoint(object, red, green, blue, height);
		   break;

	      case KMAP332:
		   color_map332(object, red, green, blue, height);
		   break;

	      case KORIGINAL:
		   break;

	      default:
		   return;
	   }

	   for (j = 0; j < height; j++)
	   {
	      maps[j*3]     = red[j];
	      maps[j*3 + 1] = green[j];
	      maps[j*3 + 2] = blue[j];
	   }
	   kpds_set_attribute(object, KPDS_MAP_POSITION, 0, 0, i, 0, 0);
	   kpds_put_data(object, KPDS_MAP_REGION, maps);
	}
	kfree(red);
	kfree(green);
	kfree(blue);
	kfree(maps);
	kpds_close_object(object);
}


/*-----------------------------------------------------------
|
|  Routine Name: kmap_map_operation - image service routine for performing
|				            automatic colormap operations
|
|       Purpose: This should routine performs automatic coloring of
|		 images.  The current images supported are:
|
|			KINVERT         
|			KINVERT_ORIG    
|			KRANDOM         
|			KREVERSE        
|			KROW_ROTLEFT    
|			KROW_ROTRIGHT   
|			KCOL_ROTLEFT    
|			KCOL_ROTRIGHT   
|			KSWAP_REDGREEN  
|			KSWAP_REDBLUE   
|			KSWAP_GREENBLUE 
|			KRED_FILTER     
|			KGREEN_FILTER   
|			KBLUE_FILTER    
|			KMAP332         
|
|         Input: object - object to color
|                type   - the map_autocolor type
|    Written By: Mark Young
|          Date: Jun 18, 1994
| Modifications:
|
------------------------------------------------------------*/

void kmap_map_operation(
   kobject original,
   int     type)
{
	kobject object;
	double  *maps, *red, *green, *blue;
	int     i, j, width, height, num_maps, available;


	/*
	 *  Check to see if we have a copy of the original colormap..
	 */
	if (type != KORIGINAL && !kdms_query_segment(original,ORIGINAL_SEGMENT))
	   kmap_map_autocolor(original, KORIGINAL);

	/*
	 *  Go ahead and make a referenced copy since we need to set some
	 *  attributes.
	 */
	if ((object = kdms_reference(original)) == NULL)
	{
	   kerror("kappserv", "kmap_map_autocolor", "Unable to make a copy \
of the input object in which to map_autocolor");
	   return;
	}

	/*
	 *  Make sure that the colormap exists...
	 */
	available = initialize_maps(object);

	/* 
	 * get number of rows (width), cols (height), maps (maps)
	 */
	kpds_get_attribute(object, KPDS_MAP_SIZE, &width, &height, &num_maps,
			NULL, NULL);

	/* 
	 * set the map data to be double
	 */
	kpds_set_attributes(object,
			KPDS_MAP_DATA_TYPE, KDOUBLE,
			KPDS_MAP_REGION_SIZE, 3, height, 1, 1, 1,
			NULL);
	if (type == KORIGINAL || type == KINVERT_ORIG)
	   restore_original(original);

	/*
	 *  Allocate the colormap information.
	 */
	red   = (double *) kmalloc(height * sizeof(double));
	green = (double *) kmalloc(height * sizeof(double));
	blue  = (double *) kmalloc(height * sizeof(double));
	for (i = 0, maps = NULL; i < num_maps; i++)
	{
	   kpds_set_attribute(object, KPDS_MAP_POSITION, 0, 0, i, 0, 0);
	   maps = (double *) kpds_get_data(object, KPDS_MAP_REGION, maps);
	   for (j = 0; j < height; j++)
	   {
	      red[j]   = (!available) ? j : maps[j*3];
	      green[j] = (!available) ? j : maps[j*3 + 1];
	      blue[j]  = (!available) ? j : maps[j*3 + 2];
	   }

	   /*
	    *  Find out which type of automatic colormap operation they wish to
	    *  perform.
	    */
	   switch(type)
	   {
	      case KINVERT:
		   color_invert(object, red, green, blue, height);
		   break;

	      case KRANDOM:
		   color_rgb_random(object, red, green, blue, height);
		   break;

	      case KREVERSE:
		   color_reverse(object, red, green, blue, height);
		   break;

	      case KROW_ROTLEFT:
	      case KROW_ROTRIGHT:
		   color_row_rotate(object, red, green, blue, height, type);
		   break;

	      case KCOL_ROTLEFT:
	      case KCOL_ROTRIGHT:
		   color_col_rotate(object, red, green, blue, height, type);
		   break;

	      case KCHAIN_ROTLEFT:
	      case KCHAIN_ROTRIGHT:
		   color_chain_rotate(object, red, green, blue, height, type);
		   break;

	      case KRED_ROTLEFT:
	      case KRED_ROTRIGHT:
	      case KGREEN_ROTLEFT:
	      case KGREEN_ROTRIGHT:
	      case KBLUE_ROTLEFT:
	      case KBLUE_ROTRIGHT:
		   color_rgb_rotate(object, red, green, blue, height, type);
		   break;

	      case KSWAP_REDGREEN:
	      case KSWAP_REDBLUE:
	      case KSWAP_GREENBLUE:
		   color_rgbswap(object, red, green, blue, height, type);
		   break;

	      case KRED_FILTER:
	      case KGREEN_FILTER:
	      case KBLUE_FILTER:
		   color_rgbfilter(object, red, green, blue, height, type);
		   break;

	      case KORIGINAL:
		   break;

	      default:
		   return;
	   }

	   for (j = 0; j < height; j++)
	   {
	      maps[j*3]     = red[j];
	      maps[j*3 + 1] = green[j];
	      maps[j*3 + 2] = blue[j];
	   }
	   kpds_set_attribute(object, KPDS_MAP_POSITION, 0, 0, i, 0, 0);
	   kpds_put_data(object, KPDS_MAP_REGION, maps);
	}
	kfree(red);
	kfree(green);
	kfree(blue);
	kfree(maps);
	kpds_close_object(object);
}
