#! /bin/sh 

echo "running test1.sh for knoise ... process ($$)"

tmpdir=${TMPDIR:-/tmp}
# knoise can generate 5 types of noises. Internally it uses 5 
# processing types. This comment explains some characteristics 
# of each noise type which might be used to infer whether the
# knoise program is working correctly. 
# For any type of noise in which the mean.variance is specified
# the statistics on the output should indicate roughly the 
# specified value. In this testsuite a margin of +-2% is tolerated
# between the generated and specified values. 
# The addition of noise to an existing object can be reliably tested
# only if the object is of type Float or Double so that any effects of
# downcasting the generated noise data are eliminated. The mean is 
# additive i.e. the mean of the output is the sum of the mean of the
# input and specified mean. Again some tolerance should be allowed
# between the observed and expected values.
# The uniform noise generator offers other ways to test itself.
# First off the output should be between the minimeanm and maximeanm
# values specified. For the case when the input type is int
# and max/min both lying between 2 consecutive integers the output
# should contain just one constant integer value.
 
# ------------------------------------------------------------------------
# report test_num test_obj cntrl_obj test_descr tol
#
# The report function uses ksegcmp to compare the test output data object
# ($2) against the correct result ($3), and reports success or failure of
# the test.  If the result of ksegcmp is false, status is set to 1.
# ------------------------------------------------------------------------
report()
{
test_num="$1"
test_obj="$2"
cntrl_obj="$3"
test_descr="$4"
tol="$5"

ksegcmp -i1 "$test_obj" -i2 "$cntrl_obj" -tol $tol -sat 1 -s -rt 1 -all
if [ $? = 0 ]
then
   echo "Passed:  Test  $test_num --> "$test_descr""
else
   echo "FAILED:  Test  $test_num --> "$test_descr""
   status=1
fi
}

#
#  Initialize status to be success (ie. 0).  Upon failure, status is
#  initialized to 1.
#
status=0

kstats -mean -i image:head -o $tmpdir/mean_head.$$

# ------------------------------------------------------------------------
#  set path for test data
# ------------------------------------------------------------------------
tbpath=`kecho -tb DATAMANIP -echo path`
dpath="$tbpath/data/testdata"

# ---------------------------------------------------------------------------
#  Test 1: Adding gaussian noise to floating data
# --------------------------------------------------------------------------
test_num=1
descr="Adding Gaussian Noise - Floating pt. data(image:head)"
tolerance=2
knoise -add -i image:head -o $tmpdir/jnk1.$$ -gauss -gmean 100.0 -gvar 100.0
# Mean on junk1 shld. be 98 or 102 greater than image:head's mean
karith2 -i1 $tmpdir/mean_head.$$ -real 100.0 -add -o $tmpdir/mean_hi.$$
kstats -mean -i $tmpdir/jnk1.$$ -o $tmpdir/mean1.$$
report $test_num "$tmpdir/mean1.$$" "$tmpdir/mean_hi.$$" "$descr" $tolerance
/bin/rm -f $tmpdir/mean_hi.$$ $tmpdir/jnk1.$$ $tmpdir/mean1.$$
/bin/rm -f $tmpdir/mean_head.$$


# ---------------------------------------------------------------------------
#  Test 2: Replacing float data with exponential noise
# ---------------------------------------------------------------------------
test_num="2a"
descr="Replace float data with exponential noise - testing mean (2% tol)"
kgconst -wsize 512 -hsize 512 -o $tmpdir/const.$$ -real 0 -type "float"
knoise -replace -i $tmpdir/const.$$ -o $tmpdir/noisy.$$ -exp -evar 50.0
kstats -i $tmpdir/noisy.$$ -mean -o $tmpdir/mean.$$
kstats -i $tmpdir/noisy.$$ -var -o $tmpdir/var.$$

cat <<EOF > $tmpdir/ascii.$$
7.07107
EOF
kasc2val -i1 $tmpdir/ascii.$$ -wsize 1 -hsize 1 -type double -o $tmpdir/ans.$$
report "$test_num" "$tmpdir/mean.$$" "$tmpdir/ans.$$" "$descr" "0.14"

test_num="2b"
descr="Replace float data with exponential noise - testing variance (2% tol)"
cat <<EOF > $tmpdir/ascii.$$
50.0
EOF
kasc2val -i1 $tmpdir/ascii.$$ -wsize 1 -hsize 1 -type double -o $tmpdir/ans.$$
report "$test_num" "$tmpdir/var.$$" "$tmpdir/ans.$$" "$descr" "1.0"

/bin/rm -f  $tmpdir/const.$$ $tmpdir/noisy.$$ $tmpdir/mean.$$ $tmpdir/var.$$
/bin/rm -f  $tmpdir/ascii.$$ $tmpdir/ans.$$

# ---------------------------------------------------------------------------
#  Test 3: Replacing float data with uniform noise
# ---------------------------------------------------------------------------
kgconst -wsize 128 -hsize 64 -dsize 2 -o $tmpdir/const.$$ -real 1313 -type "float"
knoise -replace -i $tmpdir/const.$$ -o $tmpdir/jnk1.$$ -unif -umin 100.0 -umax 100.999 
kstats -i $tmpdir/jnk1.$$ -mean -o $tmpdir/mean.$$
kstats -i $tmpdir/jnk1.$$ -var -o $tmpdir/var.$$
kstats -i $tmpdir/jnk1.$$ -minval -maxval -o $tmpdir/minmax.$$

test_num=3a
descr="Replace float data with uniform noise - test mean (2% tol)"
cat <<EOF > $tmpdir/ascii.$$
100.5
EOF
kasc2val -i1 $tmpdir/ascii.$$ -wsize 1 -hsize 1 -type double -o $tmpdir/ans.$$
report "$test_num" "$tmpdir/mean.$$" "$tmpdir/ans.$$" "$descr" "2.0"
# echo "------ mean should be 150 + - 2% -------"
# kprdata -i $tmpdir/mean.$$ -val

test_num="3b"
descr="Replace integer data with uniform noise - test variance (2% tol)"
cat <<EOF > $tmpdir/ascii.$$
0.0833
EOF
kasc2val -i1 $tmpdir/ascii.$$ -wsize 1 -hsize 1 -type double -o $tmpdir/ans.$$
report "$test_num" "$tmpdir/var.$$" "$tmpdir/ans.$$" "$descr" "0.0016"
# echo "------ variance should be 150 + - 2% -------"
# kprdata -i $tmpdir/var.$$ -val

/bin/rm -f $tmpdir/jnk1.$$
/bin/rm -f $tmpdir/mean.$$ $tmpdir/var.$$
/bin/rm -f  $tmpdir/ascii.$$ $tmpdir/ans.$$
/bin/rm -f $tmpdir/const.$$ $tmpdir/minmax.$$


# ---------------------------------------------------------------------------
#  Test 4: Replace float data with poisson noise
# ---------------------------------------------------------------------------
test_num="4a"
descr="Replace float data with poisson noise - testing mean (2% tol)"
kgconst -wsize 128 -hsize 64 -esize 2 -o $tmpdir/const.$$ -real 0 -type "float"
knoise -replace -i $tmpdir/const.$$ -o $tmpdir/noisy.$$ -poiss -ptime  1.0 -pvar 150.0
kstats -i $tmpdir/noisy.$$ -mean -o $tmpdir/mean.$$
kstats -i $tmpdir/noisy.$$ -var -o $tmpdir/var.$$

cat <<EOF > $tmpdir/ascii.$$
150
EOF
kasc2val -i1 $tmpdir/ascii.$$ -wsize 1 -hsize 1 -type double -o $tmpdir/ans.$$
report "$test_num" "$tmpdir/mean.$$" "$tmpdir/ans.$$" "$descr" "3.0"

test_num="4b"
descr="Replace float data with poisson noise - testing variance (2% tol)"
cat <<EOF > $tmpdir/ascii.$$
150.0
EOF
kasc2val -i1 $tmpdir/ascii.$$ -wsize 1 -hsize 1 -type double -o $tmpdir/ans.$$
report "$test_num" "$tmpdir/var.$$" "$tmpdir/ans.$$" "$descr" "3.0"

# echo "------ mean should be 150 + - 2% -------"
# kprdata -i $tmpdir/mean.$$ -val
# echo "------ variance should be 150 + - 2% -------"
# kprdata -i $tmpdir/var.$$ -val
/bin/rm -f  $tmpdir/const.$$ $tmpdir/noisy.$$ 
/bin/rm -f $tmpdir/mean.$$ $tmpdir/var.$$
/bin/rm -f  $tmpdir/ascii.$$ $tmpdir/ans.$$
# echo "         Test 4 may fail on certain architectures - this is due "
# echo "         to the current method used to generate the poisson noise"

# ---------------------------------------------------------------------------
#  Test 5: Replacing floating data with rayleigh noise
# ---------------------------------------------------------------------------
test_num=5a
descr="Replace floating data with rayleigh noise - test mean (2% tol)"
knoise -replace -i image:head -o $tmpdir/jnk1.$$ -ray -rvar 10.0 
kstats -i $tmpdir/jnk1.$$ -mean -o $tmpdir/mean.$$
kstats -i $tmpdir/jnk1.$$ -var -o $tmpdir/var.$$
cat <<EOF > $tmpdir/ascii.$$
6.05
EOF
kasc2val -i1 $tmpdir/ascii.$$ -wsize 1 -hsize 1 -type double -o $tmpdir/ans.$$
report "$test_num" "$tmpdir/mean.$$" "$tmpdir/ans.$$" "$descr" "0.12"

test_num="5b"
descr="Replace floating data with rayleigh noise - test variance (2% tol)"
cat <<EOF > $tmpdir/ascii.$$
10
EOF
kasc2val -i1 $tmpdir/ascii.$$ -wsize 1 -hsize 1 -type double -o $tmpdir/ans.$$
report "$test_num" "$tmpdir/var.$$" "$tmpdir/ans.$$" "$descr" "0.2"

/bin/rm -f $tmpdir/mean.$$ $tmpdir/var.$$
/bin/rm -f  $tmpdir/ascii.$$ $tmpdir/ans.$$
/bin/rm -f $tmpdir/jnk1.$$ 

exit $status
