 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for kresample
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lkresample
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lkresample - resample the value data of an object
* 
*       Purpose: lkresample is the library routine that actually
*                resamples the object data.  lkresample magnifies 
*		 or reduces the source data object (i) by using 
*		 the selected interpolation method (interpolate).  
*		 (At this time, only only pixel replication is 
*		 supported.)  Magnification factors greater than 1 
*		 magnify the object, and magnification factors 
*		 between 0 and 1 reduce the object.
*
*		Pixel replication and subsampling are done on a floating 
*		point indexing basis, where the data point index is 
*		obtained by truncating the cumulative floating point index.  
*		Therefore, when the calculated replication factor or 
*		the subsampling step size is non-integer, the resulting 
*		integer step size varies during the process.
*
*		 Magnification Factors: individual magnification 
*		 factors are specified for each dimension by the 
*		 wfactor, hfactor, dfactor, tfactor, and efactor
*		 parameters, and must be values greater than zero.
*
*		 Center of Action: The magnification or reduction 
*		 is done about a center point that can be specified 
*		 to be anywhere in the object (even outside of the 
*		 object).  The center of action coordinates are 
*		 specified for each dimension by the wcenter, hcenter, 
*		 dcenter, tcenter, and ecenter parameters.
*
*		 Destination Object Size: The resultant destination
*		 object size will be the same as the source object 
*		 size if the resize flag is FALSE, or it will be 
*		 resized to fit the magnified or reduced data set 
*		 if resize flag is TRUE.
*
*		 If resize is FALSE, and the source object is reduced 
*		 (magnification factors < 1.0) then unknown pixels 
*		 are set to zero.  Likewise, if the source object is 
*		 magnified (magnification factor > 1.0) then input 
*		 data that would be mapped outside of the destination 
*		 object are lost.
*
*		 Data Type: The data type of the destination object's 
*		 value data is the same as the source object's value 
*		 data type.  
*
*		 If there is map and value data in the source object, 
*		 and zero order interpolation (pixel replication) is 
*		 used, the map is transferred to the destination object.  
*		 If any interpolation algorithm is used, the value data 
*		 is mapped before the operation can begin and the
*		 destination will not have map data.  This program fails 
*		 if the source object lacks both map data and value data.
*
*		 This program fails if there is mask data and the operation 
*		 uses anything other than zero order interpolation.  In 
*		 other words, if value data is being replicated or 
*		 subsample on grid, then the mask can be correspondingly 
*		 replicated or subsampled.  (Bilinear interpolation can not
*		 be performed on mask data.)
*
*		 If explicit location or time data are available, the 
*		 sampling algorithm can not resample these data yet, and 
*		 will fail.  This will be implemented later ....
*
*         Input: i - the input data object to be resampled
*                wfactor - the width magnification
*                hfactor - the height magnification
*                dfactor - the depth magnification
*                tfactor - the time magnification
*                efactor - the element magnification
*                wcenter - the width center position
*                hcenter - the height center position
*                dcenter - the depth center position
*                tcenter - the time center position
*                ecenter - the element center position
*                resize  - reset the dimensions to the magnified size
*                interpolate - the interpolation type
*
*        Output: o - the output data object to be resampled
*       Returns: TRUE (1) on success, FALSE (0) on failure
*
*  Restrictions: 
*    Written By: Donna Koechner, Scott Wilson, & Mark Young
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 13 Dec 1993 - Scott Wilson - Converted from lkresample()
****************************************************************/
/* -library_def */
int lkresample(
	kobject i,
	double wfactor, 
	double hfactor, 
	double dfactor, 
	double tfactor, 
	double efactor,
	int wcenter, 
	int hcenter, 
	int dcenter, 
	int tcenter, 
	int ecenter,
	int resize,
	int interpolate,
	kobject o)
/* -library_def_end */

/* -library_code */
{
        kaddr data=NULL, mask=NULL;
        int  w,h,d,t,e;
        int  wnew,hnew,dnew,tnew,enew;
        int  wcpos,hcpos,dcpos,tcpos,ecpos;
	int  index=0,regw,regh,regd,regt,rege,num_rgns;
        int  map_exists=0, mask_exists=0;
	int  grid;
        char *prog="lkresample";
        char *lib="kdatamanip";
 
	/* 
	 *  Verify that interpolate is a legal type
	 */
        if ( (interpolate != KPAD) &&
             (interpolate != KZERO_ORDER) &&
             (interpolate != KFIRST_ORDER) )
	{
	   kerror(lib, prog, "Unrecognized interpolation type (%d).", 
		  interpolate);
	   return(FALSE);
	}
        if (interpolate == KFIRST_ORDER)
	{
	   kerror(lib, prog, "First order interpolation not supported yet...");
	   return(FALSE);
	}
	
	/*
	 *  Check if the source object has location or time data.  We
	 *  currently cannot operate on objects with rectilinear or
	 *  curvilinear location data or time data, so if either
	 *  exist, return. 
	 */ 
	if (kpds_query_location(i))
	{
	   if (!kpds_get_attribute(i, KPDS_LOCATION_GRID, &grid))
	   {
	      kerror(lib, prog, "Unable to retrieve location grid attribute.");
	      return(FALSE);
	   }
	   if (grid != KUNIFORM && (grid == KRECTILINEAR || 
				    grid == KCURVILINEAR))
	   {
	      kerror(lib, prog, 
		     "Cannot operate on object with explicit rectilinear "
		     "or curvilinear location data.");
	      return(FALSE);
	   }
	}
	if (kpds_query_time(i))
	{
           kerror(lib, prog, 
		  "Cannot operate on object with explicit time data.");
           return(FALSE);
	}

        /*
         *  Check to make sure that there's a value data segment
         *  from which to resample.
         */
        if (!kpds_query_value(i))
        {
           kerror(lib, prog, "No value data available for input object.");
           return(FALSE);
        }

	/*
	 *  Check if the source object has map and mask data.
	 */
	 mask_exists = kpds_query_mask(i);
	 map_exists  = kpds_query_map(i);

	/*
	 *  If the interpolation is first order, mask data can be replicated.
	 *  If the interpolation type is higher than first order, and a mask
	 *  exists, cannot operate on the object.  
	 */
	if (mask_exists)
	{
	   if ((interpolate==KPAD) || (interpolate==KZERO_ORDER))
	   {
	      if (!kpds_query_mask(o))
	      {
	         if (!kpds_create_mask(o))
	         {
                    kerror(lib, prog, "Failed to create destination mask.");
                    return(FALSE);
	         }
	      }
	   }
	   else
	   {
              kerror(lib, prog, "%s %s", "Cannot perform higher than zero"
		"order interpolation on data object with a validity mask.");
              return(FALSE);
	   }
	}

        /* 
	 *  Reference the source object right away, before anything can
         *  happen to it. 
	 */

        if ((i = kpds_reference_object(i)) == KOBJECT_INVALID)
        {
           kerror(lib, prog, "Failed to reference source object.");
           return(FALSE);
        }
 
	/*
	 *  If the interpolation is first order, map data can be transferred
	 *  directly to the destination object.  If the interpolation type 
	 *  is higher than first order, and a map exists, the value data 
	 *  must be mapped through the map before processing.
	 */
	if (map_exists)
	{
	   if ((interpolate == KPAD) || (interpolate == KZERO_ORDER))
	   {
	      if (!kpds_query_map(o))
	      {
	         if (!kpds_create_map(o))
                 {
		    kerror(lib, prog, "Failed to create destination map.");
		    kpds_close_object(i);
                    return(FALSE);
		 }
	      }
	   }
	   else
	   {
	      if (!kpds_set_attribute(i, KPDS_MAPPING_MODE, KMAPPED))
	      {
		 kerror(lib, prog, "Unable to set mapping mode on source.");
                 kpds_close_object(i);
                 return(FALSE);
	      }
	      if (kpds_query_map(o))
		 kpds_destroy_map(o);
	      map_exists = FALSE;
	   }
	}
 
        /* 
	 *  Find out how big the input object is. 
	 */
        if (!kpds_get_attribute(i, KPDS_VALUE_SIZE, &w, &h, &d, &t, &e))
	{
	   kerror(lib, prog, "Unable to get size of source object.");
	   kpds_close_object(i);
	   return(FALSE);
	}

	/*
	 *  Propagate source datatypes to destination object.
	 */
        if (!kpds_copy_attribute(i, o, KPDS_VALUE_DATA_TYPE))
	{
	   kerror(lib, prog, 
		"Unable to copy source value datatype to destination.");
	   kpds_close_object(i);
	   return(FALSE);
	}
	if (mask_exists)
	{
           if (!kpds_copy_attribute(i, o, KPDS_MASK_DATA_TYPE))
	   {
	      kerror(lib, prog, 
		   "Unable to copy source mask datatype to destination.");
	      kpds_close_object(i);
	      return(FALSE);
	   }
	}
	
        if (!resize)
        {  
	   /*  
	    *  Doing a normal magnify and letting things spill over - ouput 
	    *  size equals input size 
            *  
	    *  If the dimension center has been set to "auto" (-1) then the
            *  center of action is simply the middle of the dimension. 
            *  Otherwise just set the COA to the specified place. These are 
            *  in the original image space.   
	    */
           if (wcenter == -1) wcenter=w/2;
           if (hcenter == -1) hcenter=h/2;
           if (dcenter == -1) dcenter=d/2;
           if (tcenter == -1) tcenter=t/2;
           if (ecenter == -1) ecenter=e/2;

           /* 
	    *  Find the center in the magnified image space. 
	    */
	   wcenter *= wfactor;       
	   hcenter *= hfactor;       
	   dcenter *= dfactor;       
	   tcenter *= tfactor;       
	   ecenter *= efactor;       

           /*  
	    *  Set the corner position of the VALUE REGION to be one half 
            *  of the size of the unmagnified dimension "back" from the
            *  COA *in the magnified image space*. 
	    */
           wcpos = wcenter - w/2;
           hcpos = hcenter - h/2;
           dcpos = dcenter - d/2;
           tcpos = tcenter - t/2;
           ecpos = ecenter - e/2;

           /*  
	    *  Compute the magnified dimension sizes - add 0.5 to round
	    *  rather than truncate.
	    */
           wnew = (w * wfactor) + 0.5;
           hnew = (h * hfactor) + 0.5;
           dnew = (d * dfactor) + 0.5;
           tnew = (t * tfactor) + 0.5;
           enew = (e * efactor) + 0.5;

	   /*
	    *  Get the optimal region size from the original data size. 
	    */
	   if (! (kpds_get_attribute(i, KPDS_VALUE_OPTIMAL_REGION_SIZE, 
			&regw, &regh, &regd, &regt, &rege, &num_rgns)))
	   {
	      kerror(lib, prog, "Unable to get source optimal region size.");
	      kpds_close_object(i);
	      return(FALSE);
	   }

	   if (wnew < 1) wnew = 1;
	   if (hnew < 1) hnew = 1;
	   if (dnew < 1) dnew = 1;
	   if (tnew < 1) tnew = 1;
	   if (enew < 1) enew = 1;
           /*
	    *  Tell the *presentation* of the input object that we want to
            *  see an image with the new, magnified dimensions.  
	    */
           if (! (kpds_set_attribute(i,
		  KPDS_VALUE_SIZE, wnew, hnew, dnew, tnew, enew)))
	   {
	      kerror(lib, prog, "Unable to set source object size attribute.");
	      kpds_close_object(i);
	      return(FALSE);
	   }

           /*
	    *  Set up the output object to the correct final size. 
	    */
           if (! (kpds_set_attribute(o, KPDS_VALUE_SIZE, w, h, d, t, e)))
           {
              kerror(lib,prog,"Unable to set value size on destination object");
              kpds_close_object(i);
              return(FALSE);
           }
        }

        else
        { /* Reset the dimensions to fit the resampled data */

           /* 
	    *  Compute the magnified dimension sizes 
	    */
           wnew = (w * wfactor) + 0.5;
           hnew = (h * hfactor) + 0.5;
           dnew = (d * dfactor) + 0.5;
           tnew = (t * tfactor) + 0.5;
           enew = (e * efactor) + 0.5;

           if (wnew < 1) 
	   {
	      wnew = 1;
	      kinfo(KSTANDARD, "%s %s", "Computed width dimension is less",
		    "than 1, resetting to minimum of 1.");
	   }
           if (hnew < 1) 
	   {
	      hnew = 1;
	      kinfo(KSTANDARD, "%s %s", "Computed height dimension is less",
		    "than 1, resetting to minimum of 1.");
	   }
           if (dnew < 1) 
	   {
	      dnew = 1;
	      kinfo(KSTANDARD, "%s %s", "Computed depth dimension is less",
		    "than 1, resetting to minimum of 1.");
	   }
           if (tnew < 1) 
	   {
	      tnew = 1;
	      kinfo(KSTANDARD, "%s %s", "Computed time dimension is less",
		    "than 1, resetting to minimum of 1.");
	   }
           if (enew < 1) 
	   {
	      enew = 1;
	      kinfo(KSTANDARD, "%s %s", "Computed elements dimension is less",
		    "than 1, resetting to minimum of 1.");
	   }

	   /* 
	    *  Since COA is not necessary, set the offset positions to 0 
	    */
           wcpos = hcpos = dcpos = tcpos = ecpos = 0;

           /* 
	    *  Tell the *presentation* of the input object that we want to
            *  see an image with the new, magnified dimensions.
	    */
           if (! (kpds_set_attribute(i,
		  KPDS_VALUE_SIZE, wnew, hnew, dnew, tnew, enew)))
	   {
	      kerror(lib, prog, "Unable to set source object size attribute.");
	      kpds_close_object(i);
	      return(FALSE);
	   }

	   /* 
	    *  Get the optimal region size of the source data object.
	    */
	   if (! (kpds_get_attribute(i, KPDS_VALUE_OPTIMAL_REGION_SIZE, 
		  &regw, &regh, &regd, &regt, &rege, &num_rgns)))
	   {
	      kerror(lib, prog, "Unable to get source optimal region size.");
	      kpds_close_object(i);
	      return(FALSE);
	   }

           /*
            *  Tell the *real* output object that is should be the
            *  magnified size, etc.   If mask exists, its size should be
            *  automatically set by this call.
            */
           if (! (kpds_set_attribute(o,
		  KPDS_VALUE_SIZE, wnew, hnew, dnew, tnew, enew)))
	   {
	      kerror(lib, prog, "Unable to set destination object size.");
	      kpds_close_object(i);
	      return(FALSE);
	   }
        }

        /* 
	 *  Set *presentation* attributes of the input object.
         *  Setting the offset will cause data to be extracted from the 
	 *  area around the COA, when applicable.  
	 */
        if (! (kpds_set_attributes(i,
               KPDS_VALUE_OFFSET, wcpos, hcpos, dcpos, tcpos, ecpos,
               KPDS_VALUE_REGION_SIZE, regw, regh, regd, regt, rege,
               KPDS_VALUE_INTERPOLATE, interpolate, 
	       NULL)))
	{
	   kerror(lib, prog, "Unable to set attributes on source object");
	   kpds_close_object(i);
	   return(FALSE);
	}
        if (mask_exists)
        {
           if (! (kpds_set_attributes(i,
                  KPDS_MASK_OFFSET, wcpos, hcpos, dcpos, tcpos, ecpos,
         	  KPDS_MASK_REGION_SIZE, regw, regh, regd, regt, rege,
                  KPDS_MASK_INTERPOLATE,interpolate, 
		  NULL)))
           {
              kerror(lib, prog, "Unable to set attributes on source mask");
              kpds_close_object(i);
              return(FALSE);
           }
        }

        /*
         *  Reference the output object after everything has been set up,
         *  Then set attributes that should not be transferred to original
         *  destination object (no side effects).
         */
        if ((o = kpds_reference_object(o)) == KOBJECT_INVALID)
        {
           kerror(lib,prog,"Unable reference the destination object");
           kpds_close_object(i);
           return(FALSE);
        }

        if (! (kpds_set_attributes(o, 
	       KPDS_VALUE_OFFSET, 0, 0, 0, 0, 0,
	       KPDS_VALUE_REGION_SIZE, regw, regh, regd, regt, rege,
	       NULL)))
        {
           kerror(lib,prog,"Unable set destination object attributes.");
           kpds_close_object(i);
           kpds_close_object(o);
           return(FALSE);
        }
	if (mask_exists)
	{
	   if (! (kpds_set_attributes(o, 
		  KPDS_MASK_OFFSET, 0, 0, 0, 0, 0,
	          KPDS_MASK_REGION_SIZE, regw, regh, regd, regt, rege,
	          NULL)))
           {
              kerror(lib,prog,"Unable set destination mask attributes.");
              kpds_close_object(i);
              kpds_close_object(o);
              return(FALSE);
           }
	}

        /* 
	 *  Grab the region (letting kpds do all of the actual interpolation
	 *  and scaling) and just write the data to the output object. 
	 */
	for (index=0; index<num_rgns; index++)
	{
           if ((data = kpds_get_data(i, KPDS_VALUE_REGION, data)) == NULL)
	   {
	      kerror(lib, prog, 
			"get value data call failed on region %d.", index);
	      kpds_close_object(i);
	      kpds_close_object(o);
	      return(FALSE);
	   }
	   if (!(kpds_put_data(o, KPDS_VALUE_REGION, data)))
	   {
	      kerror(lib, prog, 
			"put value data call failed on region %d.", index);
	      kpds_close_object(i);
	      kpds_close_object(o);
	      return(FALSE);
	   }

	   if (mask_exists)
	   {
	      if ((mask = kpds_get_data(i, KPDS_MASK_REGION, mask)) == NULL)
	      {
	         kerror(lib, prog, 
			   "get mask data call failed on region %d.", index);
	         kpds_close_object(i);
	         kpds_close_object(o);
	         return(FALSE);
	      }
	      if ( !(kpds_put_data(o, KPDS_MASK_REGION, mask)))
	      {
	         kerror(lib, prog, 
			   "put mask data call failed on region %d.", index);
	         kpds_close_object(i);
	         kpds_close_object(o);
	         return(FALSE);
	      }
	   }
	}
        kfree(data);
	if (mask)  kfree(mask);

	if (map_exists)
	{
	   if (!kpds_copy_map(i, o, TRUE))
	   {
	      kerror(lib, prog, "Unable to copy source map to destination.");
	      kpds_close_object(i);
	      kpds_close_object(o);
	      return(FALSE);
	   }
	}	
        (void) kpds_close_object(i);
        (void) kpds_close_object(o);
        return(TRUE);
}
/* -library_code_end */
