 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for kreal2cmplx
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lkreal2cmplx
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
#include "internals.h"
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lkreal2cmplx - create a complex object from real source object(s)
* 
*       Purpose: lkreal2cmplx generates a complex destination object, 
*		 dest_obj, containing complex data from the supplied 
*		 source object(s), real_obj and imag_obj.  The complex 
*		 data type (float or double) is determined by taking
*		 the highest source data type.  
*		 
*		 Single Source Object:
*		 If only one source object is valid, the other component 
*		 of the complex pair will be the real_val or imag_val
*		 passed in.  If the source object has map data, the
*		 operation will be performed on the map only.  For the 
*		 single source case, location, time, and mask data
*		 are not needed or modified.
*
*		 Two Source Objects:
*		 If both source objects are valid, and both contain 
*		 maps and no value data the operation is performed 
*		 directly on the map.  For the case of two sources 
*		 with no value data, location, time, and mask data 
*		 are not needed or modified.
*
*		 If either source object contains both map and value 
*		 data, then both objects must have value data, and the 
*		 data will be mapped before processing.
*
*		 If the source objects are the same size, then location
*		 and time data can be ignored.
*
*		 If the source objects are different sizes, then 
*		 location and time data are checked, and if they exist, 
*		 lkreal2cmplx will only continue if the grid type is 
*		 uniform.  The destination size will the maximum of both 
*		 source sizes.  Padding required by real_obj will be 
*		 set to real_val, and padding required by imag_obj will 
*		 be set to imag_val.
*
*		 If either source object has a validity mask, the 
*		 destination object will get a mask.  If both source
*		 objects have masks, they will be combined (logical
*		 AND) after processing.
*
*		 lkreal2cmplx will fail if neither source object is
*		 valid; if the source objects do not contain value
*		 and/or map data; if either source object is complex;
*		 or if the destination object is not valid.
*
*         Input: real_obj - object to be used as real component
*                imag_obj - object to be used as imaginary component
*                real_val - real pad value, and real component if 
*			    real_obj is NULL
*                imag_val - imaginary pad value, and imaginary component 
*			    if imag_obj is NULL
*
*        Output: dest_obj - resulting complex data object
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Donna Koechner
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int lkreal2cmplx(
    kobject real_obj,
    kobject imag_obj,
    double  real_val,
    double  imag_val,
    kobject dest_obj)
/* -library_def_end */

/* -library_code */
{
        char      *lib="kdatamanip", *prog="kreal2cmplx";
        int       real_obj_exists=TRUE,   imag_obj_exists=TRUE;
        int       real_map_exists=FALSE,  real_val_exists=FALSE;
        int       imag_map_exists=FALSE,  imag_val_exists=FALSE;
        int       real_mask_exists=FALSE, imag_mask_exists=FALSE;
        int       rtype, itype, dtype, mtype=KUBYTE;
        int       wr=1, hr=1, dr=1, tr=1, er=1;
        int       wi=1, hi=1, di=1, ti=1, ei=1;
        int       wd=1, hd=1, dd=1, td=1, ed=1;
        int       rgn_w=1, rgn_h=1, rgn_d=1, rgn_t=1, rgn_e=1, num_rgns=1;
        int       num_elems, grid, i, j;
	int	  convert_type;
        kobject   src_obj;
        double    *rdata = NULL, *idata = NULL;
        kdcomplex *cdata = NULL;
        char      *region, *opt_region, *region_size, *size, *complex_conv;
        char      *data_type, *position, *pad_value, *interp_type;


	if (real_obj == KOBJECT_INVALID)  real_obj_exists = FALSE;
	if (imag_obj == KOBJECT_INVALID)  imag_obj_exists = FALSE;

	/* make sure that at least one source object is valid */
	if ((!real_obj_exists) && (!imag_obj_exists))
	{
	   kerror(lib, prog,"Must pass in at least one valid source object");
	   return(FALSE);
	}

	/* verify that destination object is valid */
	if (dest_obj == KOBJECT_INVALID)
	{
	   kerror(lib, prog,"Destination object is not a valid data object");
	   return(FALSE);
	}
	
	if (real_obj_exists)
	{
	   real_val_exists = kpds_query_value(real_obj);
	   real_map_exists = kpds_query_map(real_obj);
	   /* make sure that there is value and/or map data */
	   if ((!real_val_exists) && (!real_map_exists))
	   {
	      kerror(lib, prog,"real_obj must have value and/or map data");
	      return(FALSE);
	   }
	}
	if (imag_obj_exists)
	{
	   imag_val_exists = kpds_query_value(imag_obj);
	   imag_map_exists = kpds_query_map(imag_obj);
	   /* make sure that there is value and/or map data */
	   if ((!imag_val_exists) && (!imag_map_exists))
	   {
	      kerror(lib, prog,"imag_obj must have value and/or map data");
	      return(FALSE);
	   }
	}

	/* source objects must not be complex */
	if (real_map_exists)
	{
	   if (!kpds_get_attribute(real_obj, KPDS_MAP_DATA_TYPE, &rtype))
	   {
	      kerror(lib, prog,"unable to get real_obj map data type");
	      return(FALSE);
	   }
	}
	else if (real_val_exists)
	{
	   if (!kpds_get_attribute(real_obj, KPDS_VALUE_DATA_TYPE, &rtype))
	   {
	      kerror(lib, prog,"unable to get real_obj value data type");
	      return(FALSE);
	   }
	}
	if (imag_map_exists)
	{
	   if (!kpds_get_attribute(imag_obj, KPDS_MAP_DATA_TYPE, &itype))
	   {
	      kerror(lib, prog,"unable to get imag_obj map data type");
	      return(FALSE);
	   }
	}
	else if (imag_val_exists)
	{
	   if (!kpds_get_attribute(imag_obj, KPDS_VALUE_DATA_TYPE, &itype))
	   {
	      kerror(lib, prog,"unable to get imag_obj value data type");
	      return(FALSE);
	   }
	}
	if ( (rtype == KCOMPLEX) || (rtype == KDCOMPLEX) || 
	     (itype == KCOMPLEX) || (itype == KDCOMPLEX) )
	{
	   kerror(lib, prog,"source object cannot be complex");
	   return(FALSE);
	}

	/* two source objects */
	if ((real_obj_exists) && (imag_obj_exists))
	{
	   /* if either object has value, both must have value */
	   if ( ((real_val_exists) && (!imag_val_exists)) ||
	        ((imag_val_exists) && (!real_val_exists)) )
	   {
	      kerror(lib, prog,"if one source has value, both must have it");
	      return(FALSE);
	   }

	   /* reference the source objects */
	   if ((real_obj = kpds_reference_object(real_obj)) == KOBJECT_INVALID)
	   {
	      kerror(lib, prog,"unable to create reference for real_obj");
	      return(FALSE);
	   }
	   if ((imag_obj = kpds_reference_object(imag_obj)) == KOBJECT_INVALID)
	   {
	      kerror(lib, prog,"unable to create reference for imag_obj");
	      kpds_close_object(real_obj);
	      return(FALSE);
	   }

	   /* if only map data on both objects, operate directly on map */
	   if ( ((real_map_exists) && (!real_val_exists)) && 
		((imag_map_exists) && (!imag_val_exists)) )
	   {
              region       = KPDS_MAP_REGION;
              opt_region   = KPDS_MAP_OPTIMAL_REGION_SIZE;
              region_size  = KPDS_MAP_REGION_SIZE;
              size         = KPDS_MAP_SIZE;
              data_type    = KPDS_MAP_DATA_TYPE;
              position     = KPDS_MAP_POSITION;
              pad_value    = KPDS_MAP_PAD_VALUE;
              interp_type  = KPDS_MAP_INTERPOLATE;

	      /* if destination does not contain a map, create it */
	      if ((!kpds_query_map(dest_obj)) && (!kpds_create_map(dest_obj)))
	      {
	         kerror(lib, prog,"unable to create dest_obj map");
	         kpds_close_object(real_obj);
	         kpds_close_object(imag_obj);
	         return(FALSE);
	      }
	   }
	   else 
	   {
	      /* if destination does not contain value data, create it */
	      if ((!kpds_query_value(dest_obj)) && 
		  (!kpds_create_value(dest_obj)))
	      {
	         kerror(lib, prog,"unable to create dest_obj value data");
	         kpds_close_object(real_obj);
	         kpds_close_object(imag_obj);
	         return(FALSE);
	      }
	      /* since value data exists, need to map data if map exists */
	      if (real_map_exists)
	      {
	         if (!kpds_set_attribute(real_obj, KPDS_MAPPING_MODE, KMAPPED))
	   	 {
	      	    kerror(lib, prog,"unable to set mapping mode on real_obj");
	      	    kpds_close_object(real_obj);  kpds_close_object(imag_obj);
	      	    return(FALSE);
	   	 }
		 real_map_exists = FALSE;
	      }
	      if (imag_map_exists)
	      {
	         if (!kpds_set_attribute(imag_obj, KPDS_MAPPING_MODE, KMAPPED))
	   	 {
	      	    kerror(lib, prog,"unable to set mapping mode on imag_obj");
	      	    kpds_close_object(real_obj);  kpds_close_object(imag_obj);
	      	    return(FALSE);
	   	 }
		 imag_map_exists = FALSE;
	      }
	      if ((kpds_query_map(dest_obj)) && (!kpds_destroy_map(dest_obj)))
	      {
	      	 kerror(lib, prog,"unable to destroy dest_obj map");
	      	 kpds_close_object(real_obj);  kpds_close_object(imag_obj);
	      	 return(FALSE);
	      }
              region       = KPDS_VALUE_REGION;
              opt_region   = KPDS_VALUE_OPTIMAL_REGION_SIZE;
              region_size  = KPDS_VALUE_REGION_SIZE;
              size         = KPDS_VALUE_SIZE;
              data_type    = KPDS_VALUE_DATA_TYPE;
              position     = KPDS_VALUE_POSITION;
              pad_value    = KPDS_VALUE_PAD_VALUE;
              interp_type  = KPDS_VALUE_INTERPOLATE;

	      /* if either source object has a mask, create destination mask */
	      real_mask_exists = kpds_query_mask(real_obj);
	      imag_mask_exists = kpds_query_mask(imag_obj);
	      if (real_mask_exists || imag_mask_exists)
	      {
		 if (!kpds_query_mask(dest_obj))
		 {
	            if (!kpds_create_mask(dest_obj))
	            {
	      	       kerror(lib, prog,"unable to create dest_obj mask");
	      	       kpds_close_object(real_obj); kpds_close_object(imag_obj);
	      	       return(FALSE);
	            }
	            if (!kpds_set_attribute(dest_obj,KPDS_MASK_DATA_TYPE,mtype))
	            {
	      	       kerror(lib,prog,"unable to set dest_obj mask data type");
	      	       kpds_close_object(real_obj); kpds_close_object(imag_obj);
	      	       return(FALSE);
	            }
	         }
	      }
	   }

	   /* set dest_obj type to be the higher of source types (complex) */
	   dtype = kdatatype_cast_output(rtype, itype);
	   if (dtype == KDOUBLE)   dtype = KDCOMPLEX;
 	   else			   dtype = KCOMPLEX;

	   /* set destination size to maximum of both source sizes */
	   if (!kpds_get_attribute(real_obj, size, &wr, &hr, &dr, &tr, &er))
	   {
	      kerror(lib, prog,"unable to get real_obj data size");
	      kpds_close_object(real_obj);  kpds_close_object(imag_obj);
	      return(FALSE);
	   }
	   if (!kpds_get_attribute(imag_obj, size, &wi, &hi, &di, &ti, &ei))
	   {
	      kerror(lib, prog,"unable to get imag_obj data size");
	      kpds_close_object(real_obj);  kpds_close_object(imag_obj);
	      return(FALSE);
	   }
	   wd = kmax(wr, wi);
	   hd = kmax(hr, hi);
	   dd = kmax(dr, di);
	   td = kmax(tr, ti);
	   ed = kmax(er, ei);

  	   /* if objects are different sizes, check out location and time */
	   /* if location exists, only continue if grid is uniform */
	   if ((wd!=wr) || (hd!=hr) || (dd!=dr) || (td!=tr) || (ed!=er))
	   {
	      if (real_val_exists)
	      {
                 if (kpds_query_location(real_obj))
        	 {
                    if (!kpds_get_attribute(real_obj,KPDS_LOCATION_GRID,&grid))
           	    {
              	       kerror(lib, prog, 
		       "Unable to retrieve real_obj location grid attribute");
	      	       kpds_close_object(real_obj); kpds_close_object(imag_obj);
              	       return(FALSE);
           	    }
           	    if (grid != KUNIFORM && (grid == KRECTILINEAR ||
                                    grid == KCURVILINEAR))
           	    {
              	       kerror(lib, prog,
                     	    "Real object cannot contain explicit "
                     	    "rectilinear or curvilinear location data.");
	      	       kpds_close_object(real_obj); kpds_close_object(imag_obj);
              	       return(FALSE);
           	    }
        	 }
 
                 if (kpds_query_location(imag_obj))
        	 {
                    if (!kpds_get_attribute(imag_obj,KPDS_LOCATION_GRID,&grid))
           	    {
              	       kerror(lib, prog, 
		       "Unable to retrieve imag_obj location grid attribute");
	      	       kpds_close_object(real_obj); kpds_close_object(imag_obj);
              	       return(FALSE);
           	    }
           	    if (grid != KUNIFORM && (grid == KRECTILINEAR ||
                                    grid == KCURVILINEAR))
           	    {
              	       kerror(lib, prog,
                     	    "Imaginary object cannot contain explicit "
                     	    "rectilinear or curvilinear location data.");
	      	       kpds_close_object(real_obj); kpds_close_object(imag_obj);
              	       return(FALSE);
           	    }
        	 }
 
                 /* Objects may not have time data */
                 if (kpds_query_time(real_obj))
                 {
                    kerror(lib, prog,
                      "Real object cannot contain explicit time data. ");
	      	    kpds_close_object(real_obj);  kpds_close_object(imag_obj);
                    return(FALSE);
                 }
                 if (kpds_query_time(imag_obj))
                 {
                    kerror(lib, prog,
                       "Imaginary object cannot contain explicit time data. ");
	      	    kpds_close_object(real_obj);  kpds_close_object(imag_obj);
                    return(FALSE);
                 }

	      }
	      /* set source final sizes & pad value (real_val, imag_val) */
	      if (!kpds_set_attributes(real_obj, 
				size, wd, hd, dd, td, ed,
				interp_type, KPAD,
				pad_value, real_val, imag_val, NULL))
              {
                 kerror(lib, prog, "Unable to set real_obj attributes");
	      	 kpds_close_object(real_obj);  kpds_close_object(imag_obj);
                 return(FALSE);
              }
	      if (!kpds_set_attributes(imag_obj, 
				size, wd, hd, dd, td, ed,
				interp_type, KPAD,
				pad_value, real_val, imag_val, NULL))
              {
                 kerror(lib, prog, "Unable to set imag_obj attributes");
	      	 kpds_close_object(real_obj);  kpds_close_object(imag_obj);
                 return(FALSE);
              }
	   }

	   /* temporarily set real_obj datatype to complex to get 
	      correct optimal region size */
	   if (!kpds_set_attribute(real_obj, data_type, dtype))
           {
              kerror(lib, prog, "Unable to set real_obj data type");
	      kpds_close_object(real_obj);  kpds_close_object(imag_obj);
              return(FALSE);
           }

	   /* get optimal region size */
	   if (!kpds_get_attribute(real_obj, opt_region, 
			&rgn_w, &rgn_h, &rgn_d, &rgn_t, &rgn_e, &num_rgns))
           {
              kerror(lib, prog, "Unable to get optimal region size");
	      kpds_close_object(real_obj);  kpds_close_object(imag_obj);
              return(FALSE);
           }

	   /* set up source objects for getting data */
	   if (!kpds_set_attributes(real_obj,
			data_type, KDOUBLE,
			region_size, rgn_w, rgn_h, rgn_d, rgn_t, rgn_e, NULL))
           {
              kerror(lib, prog, "Unable to set real_obj attriubutes");
	      kpds_close_object(real_obj);  kpds_close_object(imag_obj);
              return(FALSE);
           }
	   if (!kpds_set_attributes(imag_obj,
			data_type, KDOUBLE,
			region_size, rgn_w, rgn_h, rgn_d, rgn_t, rgn_e, NULL))
           {
              kerror(lib, prog, "Unable to set imag_obj attriubutes");
	      kpds_close_object(real_obj);  kpds_close_object(imag_obj);
              return(FALSE);
           }
			
	   /* set up destination object */
	   if (!kpds_set_attributes(dest_obj, data_type, dtype,
			size, wd, hd, dd, td, ed, 
			position, 0, 0, 0, 0, 0,
			region_size, rgn_w, rgn_h, rgn_d, rgn_t, rgn_e, NULL))
           {
              kerror(lib, prog, "Unable to set dest_obj attriubutes");
	      kpds_close_object(real_obj);  kpds_close_object(imag_obj);
              return(FALSE);
           }

	   /* reference destination object, and set type to double complex */
	   if ((dest_obj = kpds_reference_object(dest_obj)) == KOBJECT_INVALID)
           {
              kerror(lib, prog, "Unable create reference for dest_obj");
	      kpds_close_object(real_obj);  kpds_close_object(imag_obj);
              return(FALSE);
           }
	   if (!kpds_set_attribute(dest_obj, data_type, KDCOMPLEX))
           {
              kerror(lib, prog, "Unable to set dest_obj data type");
	      kpds_close_object(real_obj);  kpds_close_object(imag_obj);
	      kpds_close_object(dest_obj);
              return(FALSE);
           }

	   num_elems = rgn_w * rgn_h * rgn_d * rgn_t * rgn_e;
	   /* allocate the complex data array */
	   if ((cdata = kmalloc(num_elems * sizeof(kdcomplex))) == NULL)
           {
              kerror(lib, prog, "Unable to allocate complex array");
	      kpds_close_object(real_obj);  kpds_close_object(imag_obj);
	      kpds_close_object(dest_obj);
              return(FALSE);
           }

	   for (i=0; i<num_rgns; i++)
	   {
	      /* get real and imaginary data from source objects */
	      if ((rdata = kpds_get_data(real_obj, region, rdata)) == NULL)
              {
                 kerror(lib, prog, "Unable to get real data, region %d",i);
	         kpds_close_object(real_obj);  kpds_close_object(imag_obj);
	         kpds_close_object(dest_obj);
		 kfree(rdata);  kfree(idata);  kfree(cdata);
                 return(FALSE);
	      }
	      if ((idata = kpds_get_data(imag_obj, region, idata)) == NULL)
              {
                 kerror(lib, prog, "Unable to get imaginary data, region %d",i);
	         kpds_close_object(real_obj);  kpds_close_object(imag_obj);
	         kpds_close_object(dest_obj);
		 kfree(rdata);  kfree(idata);  kfree(cdata);
                 return(FALSE);
	      }

	      /* FINALLY!  process arrays of data */
	      for (j=0; j<num_elems; j++)
	      {
	         cdata[j] = kdccomp(rdata[j], idata[j]);
	      }

	      if (!kpds_put_data(dest_obj, region, cdata))
              {
                 kerror(lib, prog, "Unable to put complex data, region %d",i);
	         kpds_close_object(real_obj);  kpds_close_object(imag_obj);
	         kpds_close_object(dest_obj);
		 kfree(rdata);  kfree(idata);  kfree(cdata);
                 return(FALSE);
	      }
           }
	
	   /* free arrays */
	   kfree(rdata);  kfree(idata);  kfree(cdata);

	   /* if either has a mask, combine the masks after processing */
	   if (real_mask_exists || imag_mask_exists)
	   {
	      if (!kdmanip_mask_ops(dest_obj, 
			KDMANIP_MASK_AND, 2, real_obj, imag_obj))
              {
                 kerror(lib, prog, "Unable to combine source masks");
	         kpds_close_object(real_obj);  kpds_close_object(imag_obj);
	         kpds_close_object(dest_obj);
                 return(FALSE);
	      }
	   }
	   kpds_close_object(real_obj);
	   kpds_close_object(imag_obj);
	   kpds_close_object(dest_obj);

	} /* end of "two source objects" */
	

	
	else /* only one source object */
	{
	   /* reference the source object */
	   if (real_obj_exists)
	   {
	      if ((src_obj=kpds_reference_object(real_obj)) == KOBJECT_INVALID)
              {
                 kerror(lib, prog, "Unable to reference real_obj");
                 return(FALSE);
	      }
	      convert_type = KREAL;
	   }
	   else
	   {
	      if ((src_obj=kpds_reference_object(imag_obj)) == KOBJECT_INVALID)
              {
                 kerror(lib, prog, "Unable to reference imag_obj");
                 return(FALSE);
	      }
	      convert_type = KIMAGINARY;
	   }
	   
	   /* operate directly on map if it exists */
	   if ( (real_map_exists) || (imag_map_exists) )
	   {
              data_type    = KPDS_MAP_DATA_TYPE;
              pad_value    = KPDS_MAP_PAD_VALUE;
              interp_type  = KPDS_MAP_INTERPOLATE;
	      complex_conv = KPDS_MAP_COMPLEX_CONVERT;
              /* if destination does not contain a map, create it */
              if ((!kpds_query_map(dest_obj)) && (!kpds_create_map(dest_obj)))
              {
                 kerror(lib, prog,"unable to create dest_obj map");
                 kpds_close_object(real_obj);
                 kpds_close_object(imag_obj);
                 return(FALSE);
              }
	   }
	   else
	   {
              data_type    = KPDS_VALUE_DATA_TYPE;
              pad_value    = KPDS_VALUE_PAD_VALUE;
              interp_type  = KPDS_VALUE_INTERPOLATE;
	      complex_conv = KPDS_VALUE_COMPLEX_CONVERT;
              /* if destination does not contain value data, create it */
              if ((!kpds_query_value(dest_obj)) && 
		  (!kpds_create_value(dest_obj)))
              {
                 kerror(lib, prog,"unable to create dest_obj value data");
                 kpds_close_object(real_obj);
                 kpds_close_object(imag_obj);
                 return(FALSE);
              }

	   }

	   /* determine destination data type */
	   if (real_obj_exists)   dtype = rtype;
	   else			  dtype = itype;
	   if (dtype == KDOUBLE)  dtype = KDCOMPLEX;
	   else			  dtype = KCOMPLEX;

	   /* set padding value and complex conversion on dest_obj, and copy */
	   if (!kpds_set_attributes(dest_obj, data_type, dtype,
				interp_type, KPAD,
				pad_value, real_val, imag_val,
				complex_conv, convert_type, NULL))
           {
              kerror(lib, prog, "Unable to set dest_obj attributes");
	      kpds_close_object(src_obj);  
              return(FALSE);
	   }

	   /* set padding value and complex conversion on src_obj, and copy */
	   if (!kpds_set_attributes(src_obj, data_type, dtype,
				interp_type, KPAD,
				pad_value, real_val, imag_val,
				complex_conv, convert_type, NULL))
           {
              kerror(lib, prog, "Unable to set src_obj attributes");
	      kpds_close_object(src_obj);  
              return(FALSE);
	   }
	   if ( (real_map_exists) || (imag_map_exists) )
	   {
	      if (!kpds_copy_map_data(src_obj, dest_obj, TRUE))
              {
                 kerror(lib, prog, "Unable to copy src_obj mask to dest_obj");
	         kpds_close_object(src_obj);  
                 return(FALSE);
	      }
	   }
	   else
	   {
	      if (!kpds_copy_value_data(src_obj, dest_obj, TRUE))
              {
                 kerror(lib, prog, "Unable to copy src_obj value to dest_obj");
	         kpds_close_object(src_obj);  
                 return(FALSE);
	      }
	   }
	   kpds_close_object(src_obj);

	} /* end of "only one source object" */

	return TRUE;
}
/* -library_code_end */
