 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for kkmeans
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lkkmeans
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lkkmeans - Perform K-means clustering
* 
*       Purpose: lkmeans accepts an input data object containing vectors 
*                of equal size and performs the K-means clustering algorithm on
*                the vectors.  The length of each vector is determined by the 
*                elements (E) dimension of the input data object.
*
*                The K-means algorithm is based on minimization of the sum 
*                of the squared distances from all points in a cluster to a 
*                cluster center.  The user chooses K initial cluster centers 
*                and the input vectors are iteratively distributed among the 
*                K cluster domains.  New cluster centers are computed from 
*                these results, such that the sum of the squared distances 
*                from all points in a cluster to the new cluster center is 
*                minimized.
*
*                Although the K-means algorithm does not really converge (in a 
*                continuous space), it may converge in a discrete space or a
*                practical upper limit can be chosen for convergence. The user 
*                has the option of specifying the maximum number of iterations 
*                using the n argument.
*
*                There are two ways to specify the initial cluster centers. If 
*                the in_obj2 argument is non-NULL, then the cluster centers are 
*                read from that object. The vectors are assumed to be 
*                stored along the E direction. Only the first K centers (as 
*                specified by the -k argument) will be read. If the in_obj2
*                argument is NULL, then the first K vectors in the in_obj1 
*                object will be used as the initial cluster centers.
*
*                It should be noted that it is possible to specify an initial 
*                cluster that lies at a sufficient distance from all input 
*                vectors that it will have no vectors assigned to it during a 
*                pass of the K-means algorithm. If this happens, lkkmeans 
*                will reinitialize the value of that cluster to the mean value 
*                of a moving pair of the existing cluster centers, thus avoiding
*                degeneracy.
*
*                If map_flag and spectrum_flag are false, the out_obj1
*                object will contain a value segment specifying the cluster 
*                number to which each input vector was assigned. If map_flag is
*                true, then a map segment will also be generated. The final 
*                cluster centers will be stored row by row in the map. The 
*                values in the value segment can be interpreted as "pointing" 
*                to a particular row in the map where the associated cluster 
*                for that input vector can be found.
*
*                If spectrum_flag is true, then the out_obj1 output object will
*                contain a special map segment (regardless of map_flag) with 
*                additional information required for use with the spectrum 
*                program in the most general sense. Here, not only the cluster 
*                centers are stored, but so are the number of vectors 
*                associated with each cluster and the packed upper triangle
*                of the covariance matrix for each cluster. See the spectrum
*                manual for additional information on how this data is used 
*                and the additional capabilities that become available when 
*                the extra data supplied by setting spectrum_flag to true.
*
*                If out_obj2 is non-NULL, the associated data object will
*                contain the cluster centers (mean vectors), stored row by row 
*                in the value segment.  The dimensions of the value segment 
*                will be WxHx1x1x1 where W is the number of elements in each 
*                mean vector and W is the number of clusters.
*
*                If out_obj3 is non-NULL, the associated data object will
*                contain the cluster variances, stored row by row in the value 
*                segment.  The dimensions of the value segment will be 
*                WxHx1x1x1 where W is the number of elements in each vector of 
*                variances and W is the number of clusters.
*
*                If out_obj4 is non-NULL, the associated data object will
*                contain the cluster membership counts, stored row by row in 
*                the value segment.  The dimensions of the value segment will
*                be 1xHx1x1x1 where H is the number of clusters. The membership
*                counts simply state the number of vectors that were present 
*                in the input object that were assigned to each of the final
*                cluster centers.
*
*                If statsfile is non-NULL, the an ASCII file will be written to
*                the associated file descriptor which will contain statistics
*                obtained during the execution of lkkmeans. The output includes
*                the following information:
*                  Total Number of K-means Iterations,
*                  Total Number of Clusters,
*                  Number of Vectors Per Cluster,
*                  Cluster Center Values,
*                  Cluster Center Variance Values,
*                  Trace of Covariance Matrix.
*
*               Results obtained by the K-means algorithm can be influenced by 
*               the number and choice of initial cluster centers and the 
*               geometrical properties of the data.
*
*               For the out_obj1, out_obj2, and out_obj3 output objects, the 
*               data will be stored as type KDOUBLE.  For the out_obj4 output 
*               object, the data will be stored as type KINT. For the out_obj1 
*               output object, the value data will be stored as type KSHORT 
*               and all map data as type KDOUBLE.
*
*               lkkmeans was converted from the K1.5 lkmeans routine, which 
*               was written by Tom Sauer and Charlie Gage, with assistance and 
*               ideas from Dr. Don Hush, University of New Mexico, Dept. of 
*               EECE. Significant modifications were made to the algorithm by 
*               Scott Wilson during conversion to K2.
*
*         Input: in_obj1 - input object for clustering
*                in_obj2 - input object for initial cluster centers (if non-NULL)
*                n - maximum number of iterations to be performed
*                k - number of clusters to generate
*                map_flag - (1 : generate map on out_obj1 containing the cluster centers, 0 : do not generate map on out_obj1)
*                spectrum_flag - (1 : generate special map on out_obj1 containing information required by spectrum [overrides map_flag setting] , 0 : do not generate spectrum compatible map on out_obj1)
*
*        Output: out_obj1 - output object to contain cluster numbers in value segment
*                out_obj2 - if non-NULL, contains cluster centers stored row by row in the value segment
*                out_obj3 - if non-NULL, contains cluster variances stored row by row in the value segment
*                out_obj4 - if non-NULL, contains cluster vector count stored row by row in the value segment
*                statsfile - if non-NULL, an ASCII record of statistics is written to the associated file
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: Complex data types are not supported.
*    Written By: Scott Wilson (stolen from the K1.5 vkmeans program, which was
written by Tom Sauer and Charlie Gage, with assistance from
Dr. Don Hush, UNM/EECE Dept.)
*          Date: Apr 15, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int lkkmeans(kobject in_obj1,
             kobject in_obj2,
             int n,
             int k,
             int map_flag,
             int spectrum_flag,
             kobject out_obj1,
             kobject out_obj2,
             kobject out_obj3,
             kobject out_obj4,
             kfile *statsfile
)
/* -library_def_end */

/* -library_code */
{

        char *lib = "kdatamanip", *rtn = "lkkmeans";
        klist *objlist=NULL;
        int w1,h1,d1,t1,e1;         /* Size of input data object */
        int w2,h2,d2,t2,e2;         /* Size of cluster center output object */
        int w3,h3,d3,t3,e3;         /* Size of cluster variance output object */
        int wr,hr,dr,tr,er;         /* Size of value segment region */
        int itype,otype;            /* Data type of input and output */
        short int ccnum;
        int replace;
        int *vc;
        int converged,count;
        short int *cc_num;
        double sum,min;
        double *vec,*cm,*cv;
        int i,c,itmp2,itmp3;
        register int j,l,m;
        register double *vec_buf,*cc;
        register int itmp;
        register double tmp;
        double *cvm,*u;
        double trace;

        /* Make sure we have valid required objects */
        if (in_obj1 == KOBJECT_INVALID)
          {
            kerror(lib, rtn, "Bogus input object");
            return(FALSE);
          }
        if (out_obj1 == KOBJECT_INVALID)
          {
            kerror(lib, rtn, "Bogus output object");
            return(FALSE);
          }

        /* Check out the required input object */
        /* Make sure there is a value segment */
        if (!kpds_query_value(in_obj1))
          {
            kerror(lib,rtn,"No value data in source object to operate on.");
            return(FALSE);
          }
        /* Make sure there is no location segment */
        if (kpds_query_location(in_obj1))
          {
            kerror(lib,rtn,"Object with location segment not supported.");
            return(FALSE);
          }
        /* Reference the input object to avoid side effects, then add the
           reference to the list of goodies to be autmatically free'd on
           error. */
        KCALL((in_obj1 = kpds_reference_object(in_obj1)));
        objlist = klist_add(objlist,in_obj1,"KOBJECT");
        /* If there's a map, pull the value data through it */
        KCALL(kpds_set_attribute(in_obj1,KPDS_MAPPING_MODE, KMAPPED));
        /* See how big the data is */
        KCALL(kpds_get_attribute(in_obj1,KPDS_VALUE_SIZE,&w1,&h1,&d1,&t1,&e1));

        if (k > w1*d1*h1*t1)
          {
            kerror(lib,rtn,"More clusters requested than vectors in object!.");
            (void)klist_free(objlist,(kfunc_void)lkcall_free); 
            return(FALSE);
          }

        /* See if we have the optional cluster center input object available */
        if (in_obj2 != KOBJECT_INVALID)
          {
            /* Make sure there is a value segment */
            if (!kpds_query_value(in_obj2))
              {
                kerror(lib,rtn,"No value data in source object 2.");
                (void)klist_free(objlist,(kfunc_void)lkcall_free);
                return(FALSE);
              }
            /* Make sure there is no location segment */
            if (kpds_query_location(in_obj2))
              {
                kerror(lib,rtn,"Object with location segment not supported.");
                (void)klist_free(objlist,(kfunc_void)lkcall_free);
                return(FALSE);
              }
            /* Reference the input object to avoid side effects, then add the
               reference to the list of goodies to be autmatically free'd on
               error. */
            KCALL((in_obj2 = kpds_reference_object(in_obj2)));
            objlist = klist_add(objlist,in_obj2,"KOBJECT");
            /* If there's a map, pull the value data through it */
            KCALL(kpds_set_attribute(in_obj2,KPDS_MAPPING_MODE, KMAPPED));
            /* See how big the data is */
            KCALL(kpds_get_attribute(in_obj2,KPDS_VALUE_SIZE,
                                             &w2,&h2,&d2,&t2,&e2));
            KCALL(kpds_set_attribute(in_obj2,KPDS_VALUE_DATA_TYPE,KDOUBLE));
          }

        /* Set up the required output */
        /* Set the output data type */
        KCALL(kpds_get_attribute(in_obj1,KPDS_VALUE_DATA_TYPE,&itype));
        if (itype == KCOMPLEX || itype == KDCOMPLEX) otype = KDCOMPLEX;
        else otype = KDOUBLE;
        if (otype == KDCOMPLEX)
          {
            kerror(lib,rtn,"Complex data not yet supported.");
            (void)klist_free(objlist,(kfunc_void)lkcall_free);
            return(FALSE);
          }
        KCALL(kpds_set_attribute(in_obj1,KPDS_VALUE_DATA_TYPE,KDOUBLE));
        KCALL(kpds_set_attribute(out_obj1,KPDS_VALUE_DATA_TYPE,KSHORT));
        KCALL(kpds_set_attribute(out_obj1,KPDS_VALUE_SIZE,w1,h1,d1,t1,1));

        /* Allocate an array for storing the cluster centers, stored as rows */
        KCALL(!((cc=(double *)kmalloc(k*e1*sizeof(double)))== NULL));
        objlist = klist_add(objlist,cc,"KMALLOC");
        kbzero(cc,k*e1*sizeof(double));
 
        /* Allocate an array for storing the cluster means, stored as rows */
        KCALL(!((cm=(double *)kmalloc(k*e1*sizeof(double)))== NULL));
        objlist = klist_add(objlist,cm,"KMALLOC");
        kbzero(cm,k*e1*sizeof(double));

        /* Allocate an array for storing the cluster variances, stored as
           rows */
        KCALL(!((cv=(double *)kmalloc(k*e1*sizeof(double)))== NULL));
        objlist = klist_add(objlist,cv,"KMALLOC");
        kbzero(cv,k*e1*sizeof(double));

        /* Allocate an array for storing the vector counts */
        KCALL(!((vc=(int *)kmalloc(k*sizeof(int)))== NULL));
        objlist = klist_add(objlist,vc,"KMALLOC");
        kbzero(vc,k*sizeof(int));

        /* Allocate vector buffer for reading in1_obj */
        KCALL(!((vec_buf=(double *)kmalloc(w1*e1*sizeof(double)))== NULL));
        objlist = klist_add(objlist,vec_buf,"KMALLOC");

        /* Allocate a single vector buffer for general use */
        KCALL(!((vec=(double *)kmalloc(e1*sizeof(double)))== NULL));
        objlist = klist_add(objlist,vec,"KMALLOC");

        /* Allocate an array for storing the cluster center association
           of each vector */
        KCALL(!((cc_num=(short int *)kmalloc(w1*h1*d1*t1*sizeof(short int)))== NULL));
        objlist = klist_add(objlist,cc_num,"KMALLOC");
        kmemset(cc_num,-1,w1*h1*d1*t1*sizeof(short int));

        /* Load the initial cluster centers with either the first k
           vectors in the input image or the first k vectors of the
           2nd input image if it is non-NULL */
        if (in_obj2 != KOBJECT_INVALID)
          {
            KCALL(kpds_set_attribute(in_obj2,KPDS_VALUE_POSITION,0,0,0,0,0));
            for (i=0; i<k; i++)
              {
                KCALL(kpds_get_data(in_obj2,KPDS_VALUE_VECTOR,(kaddr)vec));
                for (j=0; j<e1; j++) cc[i*e1+j] = vec[j];
              }  
          }
        else
          {
            KCALL(kpds_set_attribute(in_obj1,KPDS_VALUE_POSITION,0,0,0,0,0));
            for (i=0; i<k; i++)
              {
                KCALL(kpds_get_data(in_obj1,KPDS_VALUE_VECTOR,(kaddr)vec));
                for (j=0; j<e1; j++) cc[i*e1+j] = vec[j];
              }  
          }

        KCALL(kpds_set_attribute(in_obj1,KPDS_VALUE_REGION_SIZE,w1,1,1,1,e1));
                
        /* K-means iteration */
        replace = 0;
        count = 0;
        do
          {
            kinfo(KVERBOSE,"Iteration %d\n",count);
            /* Reinitialize the mean and variance accumulators */
            kbzero(cm,k*e1*sizeof(double));
            kbzero(cv,k*e1*sizeof(double));
            kbzero(vc,k*sizeof(int));

            /* Reset position to origin for input data */
            KCALL(kpds_set_attribute(in_obj1,KPDS_VALUE_POSITION,0,0,0,0,0));

            kinfo(KVERBOSE,"  Scanning vector set\n");
            converged = 1;
            for (i=0; i<h1*d1*t1; i++) /* Loop over all W vectors */
              {
                KCALL(kpds_get_data(in_obj1,KPDS_VALUE_REGION,(kaddr)vec_buf));
                for (j=0; j<w1; j++) /* Loop over the vectors in the W chunk */
                  {
                    /* Hunt down the closest cluster center */
                    min = KMAXFLOAT;
                    ccnum = -1;
                    for (l=0; l<k; l++) /* Which CC we're looking at */
                      {
                        sum = 0.0;
                        for (m=0; m<e1; m++) /* Elements of vector */
                          {
                            tmp = vec_buf[m*w1+j]-cc[l*e1+m];
                            sum += tmp*tmp;
                          }
                        if (sum < min)
                          {
                            min = sum;
                            ccnum = l;
                          }
                      }
                    /* Assign this vector to the closest cluster center */
                    itmp = i*w1+j;
                    if (cc_num[itmp] != ccnum) converged = 0;
                    cc_num[itmp] = ccnum;
                    /* Update the mean and variance accumulators for the
                       selected cluster center */
                    for (m=0; m<e1; m++) /* Elements of vector */
                      {
                        tmp = vec_buf[m*w1+j];
                        itmp = ccnum*e1+m;
                        cm[itmp] += tmp;
                        cv[itmp] += tmp*tmp;
                      }
                    vc[ccnum]++;
                  }
              }
            kinfo(KVERBOSE,"  Computing new cluster centers\n");
            /* Compute new cluster centers and variances */
            for (i=0; i<k; i++)
              {
                if (vc[i] > 0)
                  {
                    for (m=0; m<e1; m++)
                      {
                        itmp = i*e1+m;
                        cc[itmp] = cm[itmp] / vc[i];
                        cv[itmp] /= vc[i];
                        cv[itmp] -= cc[itmp]*cc[itmp];
                        cv[itmp] = sqrt(cv[itmp]);
                      }
                  }
                else
                  {
                    /* Degenerate case; got a cluster center that isn't close
                       to any vector! */
                    kinfo(KVERBOSE,"  Hit degenerate cluster center during iteration\n");
                    if (replace > k-2) replace = 0;
                    for (m=0; m<e1; m++)
                      {
                        itmp = i*e1+m;
                        itmp2 = replace*e1+m;
                        itmp3 = (replace+1)*e1+m;
                        cc[itmp] = (cc[itmp2]+cc[itmp3])/2.0;;
                        cv[itmp] = 0.0;
                        converged = 0;
                      }
                    replace++;
                  }
              }
            count++;
          }
        while (converged == 0 && count < n);
        kinfo(KVERBOSE,"converged is %d, count is %d\n",converged,count);

        /* Write the cluster number output (required) */
        KCALL(kpds_put_data(out_obj1,KPDS_VALUE_ALL,(kaddr)cc_num));
        if (map_flag == TRUE && spectrum_flag == FALSE)
          {
            if (!kpds_query_map(out_obj1)) KCALL(kpds_create_map(out_obj1));
            KCALL(kpds_set_attribute(out_obj1,KPDS_MAP_DATA_TYPE,KDOUBLE));
            KCALL(kpds_set_attribute(out_obj1,KPDS_MAP_SIZE,e1,k,1,1,1));
            KCALL(kpds_set_attribute(out_obj1,KPDS_MAP_POSITION,0,0,0,0,0));
            KCALL(kpds_put_data(out_obj1,KPDS_MAP_PLANE,(kaddr)cc));
          }
        else if (spectrum_flag == TRUE)
          {
            /* See the SPECTRUM manual for a description of the structure of
               the map segment when the extended functionality of SPECTRUM
               is desired */
            if (!kpds_query_map(out_obj1)) KCALL(kpds_create_map(out_obj1));
            KCALL(kpds_set_attribute(out_obj1,KPDS_MAP_SIZE,
                                              e1+1+(e1*(e1+1)/2),k,1,1,1));
            KCALL(kpds_set_attribute(out_obj1,KPDS_MAP_DATA_TYPE,KDOUBLE));
            /* Write the cluster centers */
            KCALL(kpds_set_attribute(out_obj1,KPDS_MAP_REGION_SIZE,e1,k,1,1,1));
            KCALL(kpds_set_attribute(out_obj1,KPDS_MAP_POSITION,0,0,0,0,0));
            KCALL(kpds_put_data(out_obj1,KPDS_MAP_REGION,(kaddr)cc));
            /* Write the counts; first transfer them to a KDOUBLE array
               so that they'll end up in the map in the proper binary format */
            for (i=0; i<k; i++)
              {
                cm[i] = vc[i];
              }
            KCALL(kpds_set_attribute(out_obj1,KPDS_MAP_REGION_SIZE,1,k,1,1,1));
            KCALL(kpds_set_attribute(out_obj1,KPDS_MAP_POSITION,e1,0,0,0,0));
            KCALL(kpds_put_data(out_obj1,KPDS_MAP_REGION,(kaddr)cm));

            /****************************************************************/

            /* Compute the covariance matrix for each cluster and
               store the upper triangle in packed form in the covariance
               matrix area of the map */

            /* Allocate space for the covariance matrix */
            KCALL(!((cvm=(double *)kmalloc(e1*e1*sizeof(double)))== NULL));
            objlist = klist_add(objlist,cvm,"KMALLOC");
            /* Allocate space for the packed upper triangle */
            KCALL(!((u=(double *)kmalloc((e1*(e1+1)/2)*sizeof(double)))==NULL));
            objlist = klist_add(objlist,u,"KMALLOC");

            /* Set up to grab slabs of vectors across the W direction */
            KCALL(kpds_set_attribute(in_obj1,KPDS_VALUE_REGION_SIZE,
                                             w1,1,1,1,e1));

            /* Step thru the entire set of clusters; compute the covariance
               matrix for each and slap it in the map */
            for (c=0; c<k; c++)
              {
                kbzero(cvm,e1*e1*sizeof(double));
                kbzero(cm,e1*sizeof(double));
                /* First compute the mean vector */
                KCALL(kpds_set_attribute(in_obj1,KPDS_VALUE_POSITION,
                                                 0,0,0,0,0));
                for (i=0; i<h1*d1*t1; i++)
                  {
                    KCALL(kpds_get_data(in_obj1,KPDS_VALUE_REGION,
                                                (kaddr)vec_buf));
                    for (j=0; j<w1; j++) /* Loop over vecs in the chunk */
                      {
                        if (cc_num[i*w1+j] == c)
                          {
                            for (m=0; m<e1; m++) /* Elements of vector */
                              {
                                cm[m] += vec_buf[m*w1+j];
                              }
                          }
                      }
                  }
                for (m=0; m<e1; m++) /* Elements of vector */
                  { 
                    /* Must check for zero divide because it's possible to
                       move a cluster into a region where it wont attract
                       any vectors! */
                    if (vc[c] != 0)
                      cm[m] /= vc[c];
                  }
                KCALL(kpds_set_attribute(in_obj1,KPDS_VALUE_POSITION,
                                                 0,0,0,0,0));
                for (i=0; i<h1*d1*t1; i++)
                  {
                    KCALL(kpds_get_data(in_obj1,KPDS_VALUE_REGION,
                                                (kaddr)vec_buf));
                    for (j=0; j<w1; j++) /* Loop over the vecs in the W chunk */
                      {
                        if (cc_num[i*w1+j] == c)
                          {
                            for (m=0; m<e1; m++) /* Elements of vector */
                              {
                                for (l=0; l<e1; l++) /* Elements of vector */
                                  {
                                    cvm[m*e1+l] += (vec_buf[m*w1+j]-cm[m])*
                                                   (vec_buf[l*w1+j]-cm[l]);
                                  }
                              }
                          }
                      }
                  }
                for (m=0; m<e1; m++)
                  {
                    for (l=0; l<e1; l++)
                      {
                        /* Must check for zero divide because it's possible to
                           move a cluster into a region where it wont attract
                           any vectors! */
                        if (vc[c] != 0)
                          cvm[m*e1+l] /= vc[c];
                      }
                  }
                /* Form packed upper triangle, including the diagonal */
                i = 0;
                for (m=0; m<e1; m++)
                  {
                    for (l=m; l<e1; l++)
                      {
                        u[i++] = cvm[m*e1+l];
                      }
                  }
                /* Write the covariance matrix */
                KCALL(kpds_set_attribute(out_obj1,KPDS_MAP_REGION_SIZE,
                                                  e1*(e1+1)/2,1,1,1,1));
                KCALL(kpds_set_attribute(out_obj1,KPDS_MAP_POSITION,
                                                  e1+1,c,0,0,0));
                KCALL(kpds_put_data(out_obj1,KPDS_MAP_REGION,(kaddr)u));
              }

            /**************************************************************/

            /* Create the special attributes that SPECTRUM needs to describe
               the map segment */
            KCALL(kpds_create_object_attr(out_obj1,
                                   "colorMapColnum",1,1,KINT,TRUE,TRUE));
            KCALL(kpds_set_attribute(out_obj1, "colorMapColnum", e1));
            KCALL(kpds_create_object_attr(out_obj1,
                                   "colorMapContents",1,1,KINT,TRUE,TRUE));
            KCALL(kpds_set_attribute(out_obj1, "colorMapContents", 0x5));
          }

        /* Write the cluster center output (optional) */
        if (out_obj2 != KOBJECT_INVALID)
          {
            if (!kpds_query_value(out_obj2)) KCALL(kpds_create_value(out_obj2));
            KCALL(kpds_set_attribute(out_obj2,KPDS_VALUE_DATA_TYPE,KDOUBLE));
            KCALL(kpds_set_attribute(out_obj2,KPDS_VALUE_SIZE,e1,k,1,1,1));
            KCALL(kpds_set_attribute(out_obj2,KPDS_VALUE_POSITION,0,0,0,0,0));
            KCALL(kpds_put_data(out_obj2,KPDS_VALUE_ALL,(kaddr)cc));
          }

        /* Write the cluster variance output (optional) */
        if (out_obj3 != KOBJECT_INVALID)
          {
            if (!kpds_query_value(out_obj3)) KCALL(kpds_create_value(out_obj3));
            KCALL(kpds_set_attribute(out_obj3,KPDS_VALUE_DATA_TYPE,KDOUBLE));
            KCALL(kpds_set_attribute(out_obj3,KPDS_VALUE_SIZE,e1,k,1,1,1));
            KCALL(kpds_set_attribute(out_obj3,KPDS_VALUE_POSITION,0,0,0,0,0));
            KCALL(kpds_put_data(out_obj3,KPDS_VALUE_ALL,(kaddr)cv));
          }

        /* Write the cluster membership count output (optional) */
        if (out_obj4 != KOBJECT_INVALID)
          {
            if (!kpds_query_value(out_obj4)) KCALL(kpds_create_value(out_obj4));
            KCALL(kpds_set_attribute(out_obj4,KPDS_VALUE_DATA_TYPE,KINT));
            KCALL(kpds_set_attribute(out_obj4,KPDS_VALUE_SIZE,1,k,1,1,1));
            KCALL(kpds_set_attribute(out_obj4,KPDS_VALUE_POSITION,0,0,0,0,0));
            KCALL(kpds_put_data(out_obj4,KPDS_VALUE_ALL,(kaddr)vc));
          }

        /* Write the statistics ASCII output (optional) */
        if (statsfile != NULL)
          {
            kfprintf(statsfile,"kkmeans Statistics\n");
            kfprintf(statsfile,"================\n");
            kfprintf(statsfile,"Total Number of K-MEANS Iterations: %d\n",count);
            kfprintf(statsfile,"Total Number of Clusters: %d\n",k);
            kfprintf(statsfile,"Number of Vectors Per Cluster\n");
            kfprintf(statsfile,"=============================\n");
            for (i=0; i<k; i++)
              (void) kfprintf(statsfile,"Cluster #%d = %d\n",i, vc[i]);
            kfprintf (statsfile, 
              "\nCluster Center Values:        (Vector Size: %d)\n",e1);
            kfprintf (statsfile, "=====================");
            for (i=0; i<k; i++)
              {
                j = 1;
                kfprintf (statsfile, "\n\nCluster #%d\n\n", i);
                for (l = 0; l < e1; l++)
                  {
                    kfprintf (statsfile, "%f  ", cc[i*e1+l]);
                    if (j++ % 5 == 0)
                      kfprintf (statsfile, "\n");
                  }
              }
            kfprintf (statsfile,
              "\n\nCluster Center Variance Values:        (Vector Size: %d)\n",e1);
            kfprintf (statsfile, "=====================");
            for (i=0; i<k; i++)
              {
                j = 1;
                kfprintf (statsfile, "\n\nCluster #%d\n\n", i);
                for (l = 0; l < e1; l++)
                  {
                    kfprintf (statsfile, "%f  ", cv[i*e1+l]);
                    if (j++ % 5 == 0)
                      kfprintf (statsfile, "\n");
                  }
              }
            kfprintf (statsfile,
              "\n\nTrace of Covariance Matrix:\n"
,e1);
            kfprintf (statsfile, "=====================");
            for (i=0; i<k; i++)
              {
                kfprintf (statsfile, "\n\nCluster #%d\t", i);
                trace = 0.0;
                for (l = 0; l < e1; l++)
                  {
                    trace += cv[i*e1+l];
                  }
                kfprintf (statsfile, "%f\n", trace);
              }
          }

        (void)klist_free(objlist,(kfunc_void)lkcall_free); 
	return TRUE;
}
/* -library_code_end */
