 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for kimpulse
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lkimpulse
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lkimpulse - generate impulse value data
* 
*       Purpose: This routine creates impulses at locations controlled
*                by the various parameters passed to it. The value
*                segment is created if one does not already exist
*                and is set to size w*h*d*t*e. The datatype of the
*                value segment is set according to the variable 
*                data_type. It can be either one of bit, byte, unsigned
*                byte, short, unsigned short, integer, unsigned integer,
*                long, unsigned long, float, double, complex and double
*                complex. The value assumed by the impulse generated can 
*                be specified by imp_rval and imp_ival which denote its
*                real and imaginary component respectively. The value 
*                assumed by the background can also be specified by the
*                back_rval and back_ival parameters which denote it's
*                real and imaginary components. Three parameters are
*                available to control the location of the impulses.
*                The spacing between two impulses, the starting point of 
*                the first impulse and the number of impulses can be 
*                specified for each of the five dimensions in the
*                object. 
*
*         Input: w       - The width of the object being created
*                h       - the height of the object being created
*                d       - the depth of the object being created
*                t       - the size of the objects time dimension
*                e       - the objects element dimension size
*                wsp     - the spacing between two impulses in width direction
*                hsp     - the spacing between two impulses in height direction
*                dsp     - the spacing between two impulses in depth direction
*                tsp     - the spacing between two impulses in time direction
*                esp     - the spacing between two impulses in elements dir
*                w_num   - Number of impulses along width
*                h_num   - Number of impulses along height
*                d_num   - Number of impulses along depth
*                t_num   - Number of impulses along time dimension
*                e_num   - Number of impulses along elements dimension 
*                woff    - Location of first impulse in the width direction
*                hoff    - Location of first impulse in the height direction
*                doff    - Location of first impulse in the depth direction
*                toff    - Location of first impulse in the time direction
*                eoff    - Location of first impulse in the elements direction
*                imp_rval- Real value assigned to the impulse
*                imp_ival- Imaginary value assigned to the impulse
*                back_rval- Real value assigned to the background
*                back_ival- Imaginary value assigned to the background
*                data_type- Data type of the output object
*
*        Output: dst_obj - output object containing impulse data
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Ashish Malhotra
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: If the value segment already exists, then all data
*                will be overwritten with the impulse data of the
*                specified datatype. The size of the object will
*                also be changed to the size requested in the 
*                function call.
* Modifications: 
****************************************************************/
/* -library_def */
int lkimpulse(int w, int h,
              int d, int t,
              int e, int wsp,
              int hsp, int dsp,
              int tsp, int esp,
              int w_num, int h_num,
              int d_num, int t_num,
              int e_num, int w_off,
              int h_off, int d_off,
              int t_off, int e_off,
              double imp_rval, double imp_ival,
              double back_rval, double back_ival,
              int data_type, kobject dst_obj)
/* -library_def_end */

/* -library_code */
{
   int i,j,k,l;
   int put_impulse = FALSE;
   int hcnt,dcnt,tcnt,ecnt;
   kaddr data = NULL, impl_data = NULL;
   char  *lib = "kdatamanip";
   char  *rtn = "lkimpulse";

   /*
    * A few sanity checks should be performed to ensure that the
    * arguments passed in are meaningful.  Its a good idea to
    * check the validity of the size arguments and the object.
    */
   if (dst_obj == NULL) 
   {
      kerror(lib,rtn,"The object argument must be a valid data object.");
      return (FALSE);
   }
  
   if (w<= 0 || h<= 0 || d<= 0 || t <= 0 || e<= 0)
   {
      kerror (lib,rtn,"The size arguments must all be greater than zero.");
      return (FALSE);
   }

   if (w_num<= 0 || h_num<= 0 || d_num<= 0 || t_num <= 0 || e_num<= 0)
   {
      kerror (lib,rtn,"The number of impulses in each direction %s",
              "must be greater than 0.");
      return (FALSE);
   }

   if (w_off >= w || h_off >= h || d_off >= d || t_off >= t || e_off >=e)
   {
      kerror (lib,rtn,"Offset for a dimension is greater than the size of %s",
              "the dimension.");
      return (FALSE);
   }

   /*
    * The existence of the value segment is checked because the output
    * object that is passed in may have been used previously.  If the
    * value segment does not exist, then create it.  If you don't
    * create it, future function calls that operate on the value
    * segment will fail.
    */
   if (!kpds_query_value(dst_obj))
      (void)kpds_create_value(dst_obj);

   if (!kpds_set_attributes(dst_obj,KPDS_VALUE_SIZE, w, h, d, t, e,
                            KPDS_VALUE_DATA_TYPE, data_type,NULL))
   {
      kerror (lib,rtn,"Attributes can not be set on the output %s",
              "object's value segment.");
      return(FALSE);
   }

   /*
    * Since all physical attributes have been set, we now reference 
    * the object to prevent any side effects.
    */
   if ((dst_obj = kpds_reference_object(dst_obj)) == KOBJECT_INVALID) 
   {
      kerror (lib,rtn,"A reference object can not be created %s",
              "for the output object.");
      return(FALSE);
   }

   /*
    * Allocate the right amount of space for arrays data and impl_data.
    * get_data returns the correct amount of malloc'ed space.
    */
   if ( ((data = kpds_get_data(dst_obj, KPDS_VALUE_LINE, NULL)) == NULL) ||
        ((impl_data = kpds_get_data(dst_obj, KPDS_VALUE_LINE, NULL)) == NULL) )
   {
      (void) kpds_close_object(dst_obj);
      kerror (lib,rtn,"Failure in getting data.");
      return(FALSE);
    }

   /*
    * Fill both arrays with the background value.
    */
   if (!kdata_fill(data,w,data_type,back_rval,back_ival) ||
       !kdata_fill(impl_data,w,data_type,back_rval,back_ival) )
   {
      (void) kpds_close_object(dst_obj);
      kerror (lib,rtn,"Cannot fill data array with specified values.");
      kfree(data);
      kfree(impl_data);
      return(FALSE);
   }

   /*
    * This routine puts data into the object using the LINE primitive. It
    * assumes that the index order of the object is w,h,d,t,e. There
    * are four levels of nested loops. In the innermost loop all the
    * lines are put. The line that is put is either a constant background
    * value or one containing impulses. The line containing impulses 
    * is initially created before entering the loop. The nested counted
    * upto the height,depth,time and elements of the data object with the 
    * inner most loop counting upto height. A count of the number of impulses
    * along each dir is maintained. An impulse line is put into
    * the output object if the h,d,t,e coordinates for that line 
    * match the offset,spacing and number of impulses spec for each dimension.
    */

   for (i=0; i<w_num; i++)
   {
      if (w_off+i*wsp < w)
         kdata_insert(impl_data,w_off+i*wsp,data_type,imp_rval,imp_ival);
   }

   kpds_set_attribute(dst_obj, KPDS_VALUE_POSITION, 0, 0, 0, 0, 0);

   ecnt = 0;
   for (i=0; i<e; i++)
   {
      int flag1,flag2,flag3,flag4;

      tcnt = 0;
      flag4 = FALSE;
      for (j=0; j<t; j++)
      {
         dcnt = 0;
         flag3 = FALSE;
         for (k=0; k<d; k++)
         {
            hcnt = 0;
            flag2 = FALSE;
            for (l=0; l<h; l++)
            {
               flag1 = FALSE;
               if (hcnt < h_num)
                  if (l == h_off + hcnt * hsp)
                  {
                     hcnt++;
                     flag1 = TRUE;
                  }

               if (!flag2 && dcnt < d_num)
                  if (k == d_off + dcnt * dsp)
                  {
                     dcnt++;
                     flag2 = TRUE;
                  }
                  
               
               if (!flag3 && tcnt < t_num)
                  if (j  == t_off + tcnt * tsp)
                  {
                     tcnt++;
                     flag3 = TRUE;
                  }
             
               if (!flag4 && ecnt < e_num)
                  if (i  == e_off + ecnt * esp)
                  {
                     ecnt++;
                     flag4 = TRUE; 
                  }

               if (flag1 && flag2 && flag3 && flag4)
               {
                  (void) kpds_put_data(dst_obj, KPDS_VALUE_LINE, impl_data);
               }
               else
               {
                  (void) kpds_put_data(dst_obj, KPDS_VALUE_LINE, data);
               }
            }
         }
      }
   }

   kfree(data);
   kfree(impl_data);

   (void) kpds_close_object(dst_obj);

   return TRUE;
}
/* -library_code_end */
