 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for khistops
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lkhistops
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
#include "internals.h"

#define KHISTOPS_EQUALIZE 1
#define KHISTOPS_STRETCH 2
#define KHISTOPS_INVERT 3
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lkhistops - perform histogram equalization and stretching
* 
*       Purpose: Perform histogram equalization and stretching with
*		 optional inversion. Equalization occurs using the
*                standard algorithm. 
*
*                Stretching is done between the
*                most negative data value (mapping to the most negative
*                value possible for the input data storage type) and the most
*                positive data value (mapping to the most positive value
*                possible for the input data storage type).
*
*                No span limits are applied, as was the case in the K1.5
*                vhstr routine.
*
*         Input: in_obj - object to be processed
*		 function - a string containing either "equalize" or "stretch"
*		 invertflag - if non-zero, invert greylevels after processing
*		 w h d t e - flags that, if set, indicate that the processing 
*			unit for independent histograms should include the 
*			given axis
*
*        Output: out_obj - output object
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: KBYTE, KUBYTE, KSHORT, KUSHORT data only
*    Written By: Scott Wilson
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 21-July-94 Scott Wilson - Fixed incorrect behavior
*		 caused by MAPPING_MODE being set after getting the
*		 size attributes of the input object.
****************************************************************/
/* -library_def */
int lkhistops(
  kobject in_obj,
  char *function,
  int invertflag,
  int w, int h, int d, int t, int e,
  kobject out_obj)
/* -library_def_end */

/* -library_code */
{
        char *lib = "kdatamanip", *rtn = "lkhistops";
	long i,k;
	int mostneg,khistopfunc,type;
	long hist[65536],map[65536];
	unsigned int wc,hc,dc,tc,ec; /* Dimensions of data set */
	unsigned int wu,hu,du,tu,eu; /* Dimensions of operation unit */
	unsigned int wr,hr,dr,tr,er,junk; /* Dimensions of optimal region */
        long maxcnt,lower,upper,n,map1[65536];
        int *b=NULL,*bmask=NULL;
	unsigned  numpts,numptsunit,numunits,numptsreg,numrgns,numu,numr;
	double scale,x;
	kobject in2_obj;
        unsigned map_exists,mask_exists;

        /* Make sure we have a valid object */
        if (in_obj == KOBJECT_INVALID)
        {
           kerror(lib, rtn, "Bogus input object");
           return(FALSE);
        }

        /* Make sure the operation is known */
	if (kstrcmp(function,"equalize")==0) khistopfunc=KHISTOPS_EQUALIZE;
	else if (kstrcmp(function,"stretch")==0) khistopfunc=KHISTOPS_STRETCH;
	else
	  {
            kerror(lib, rtn, "Unsupported operation requested.");
            return(FALSE);
          }

        /* Reference the input object to avoid side effects */
        if ((in_obj = kpds_reference_object(in_obj)) == KOBJECT_INVALID)
        {
           kerror(lib, rtn, "Failed to reference input object");
           return(FALSE);
        }

        mask_exists=kpds_query_mask(in_obj);
        map_exists =kpds_query_map(in_obj);

        /* If map exists, pull all the data thru the map */
        if (map_exists)
          {
            kpds_set_attribute(in_obj,KPDS_MAPPING_MODE,KMAPPED);
            kpds_destroy_map(out_obj);
          }

        /* See how big the data is */
        if (!kpds_get_attributes(in_obj,
                                 KPDS_VALUE_SIZE,&wc,&hc,&dc,&tc,&ec,
                                 KPDS_VALUE_DATA_TYPE,&type,NULL))
        {
           kerror(lib, rtn, "Unable to get attributes on input object.");
           (void) kpds_close_object(in_obj);
           return(FALSE);
        }

        /* Make sure the data types are amenable to processing */
        if (type != KUBYTE && type != KBYTE && type  != KUSHORT &&
            type != KSHORT)
        {
           kerror(lib, rtn,"Histogram operations available for \
 8 and 16 bit data only");
           (void) kpds_close_object(in_obj);
           return(FALSE);
        }

        /* Make sure we don't hit a machine that has 32 bit bytes! */
        if (((type == KUBYTE || type== KBYTE) && sizeof(char) != 1) ||
            ((type == KUSHORT || type== KSHORT) && sizeof(short) != 2))
        {
           kerror(lib, rtn,"Histogram operations allowed on data occupying \
 16 bits or less");
           (void) kpds_close_object(in_obj);
           return(FALSE);
        }

        /* Create the output value segment if it's not there already and
           set it's size and data type attributes.  */
        if (!kpds_query_value(out_obj))
        {
           if (!kpds_create_value(out_obj))
           {
              kerror(lib, rtn, "Unable to create destination value data.");
              (void) kpds_close_object(in_obj);
              return(FALSE);
           }
        }
        if (!kpds_set_attributes(out_obj,
                                 KPDS_VALUE_SIZE,wc,hc,dc,tc,ec,
                                 KPDS_VALUE_DATA_TYPE,type,NULL))
        {
           kerror(lib, rtn, "Unable to set output object attributes.");
           (void) kpds_close_object(in_obj);
           return(FALSE);
        }

        /* Constrain the input data to int type for reading */
        if (!kpds_set_attribute(in_obj, KPDS_VALUE_DATA_TYPE, KINT, NULL))
        {
           kerror(lib, rtn, "Unable to set input object datatype attribute.");
           (void) kpds_close_object(in_obj);
           (void) kpds_close_object(out_obj);
           return(FALSE);
        }

        /* Ref the output object to avoid side effects and proceed to
           constrain it to int for writing */
        if ((out_obj = kpds_reference_object(out_obj)) == KOBJECT_INVALID)
        {
           kerror(lib, rtn, "Failed to reference output object");
           (void) kpds_close_object(in_obj);
           return(FALSE);
        }
        if (!kpds_set_attribute(out_obj, KPDS_VALUE_DATA_TYPE, KINT, NULL))
        {
           kerror(lib, rtn, "Unable to set output object datatype attribute.");
           (void) kpds_close_object(in_obj);
           (void) kpds_close_object(out_obj);
           return(FALSE);
        }

        /* Figure out what the optimal region size is that will still let us
           process by the units we want to */
        if (!kpds_get_attributes(in_obj,
                         KPDS_VALUE_OPTIMAL_REGION_SIZE,&wr,&hr,&dr,&tr,&er,&junk,
                         NULL))
        {
           kerror(lib, rtn, "Unable to get optimal region attributes.");
           (void) kpds_close_object(in_obj);
           return(FALSE);
        }
        kinfo(KVERBOSE,"regn: wr:%d hr:%d dr:%d tr:%d er:%d\n",wr,hr,dr,tr,er);
        numpts = wc*hc*dc*tc*ec;
        numptsunit = 1;
        if (w) numptsunit*= wc; if (h) numptsunit*= hc; if (d) numptsunit*= dc;
        if (t) numptsunit*= tc; if (e) numptsunit*= ec;
        numunits = numpts/numptsunit;

        /* Set the in and out region sizes */
        wu = wc; hu= hc; du = dc; tu = tc; eu = ec;
        if (!w) wu=1; if (!h) hu=1; if (!d) du=1;
        if (!t) tu=1; if (!e) eu=1;

        /* Now figure out what the REAL optimal region size is */
        if (wu == 1) wr = 1; if (hu == 1) hr = 1; if (du == 1) dr = 1;
        if (tu == 1) tr = 1; if (eu == 1) er = 1;
        numptsreg = wr*hr*dr*tr*er;
        numrgns = numptsunit/numptsreg;

        /* Set the region sizes on the input and output objects */
        if (! (kpds_set_attribute(in_obj,
                                  KPDS_VALUE_REGION_SIZE,wr,hr,dr,tr,er)) ||
            ! (kpds_set_attribute(out_obj,
                                  KPDS_VALUE_REGION_SIZE,wr,hr,dr,tr,er)))
        {
           kerror(lib, rtn, "Unable to set object value region sizes.");
           (void) kpds_close_object(in_obj);
           (void) kpds_close_object(out_obj);
           return(FALSE);
        }

        /* If a mask is in place set the region size attributes appropriately */
	if (mask_exists)
          {
            if (!(kpds_set_attribute(in_obj,
                                      KPDS_MASK_REGION_SIZE,wr,hr,dr,tr,er)) ||
                !(kpds_set_attribute(out_obj,
                                      KPDS_MASK_REGION_SIZE,wr,hr,dr,tr,er)))
             {
               kerror(lib, rtn, "Unable to set object mask region sizes.");
               (void) kpds_close_object(in_obj);
               (void) kpds_close_object(out_obj);
               return(FALSE);
             }
          }

        kinfo(KVERBOSE,"data: wc:%d hc:%d dc:%d tc:%d ec:%d\n",wc,hc,dc,tc,ec);
        kinfo(KVERBOSE,"unit: wu:%d hu:%d du:%d tu:%d eu:%d\n",wu,hu,du,tu,eu);
        kinfo(KVERBOSE,"regn: wr:%d hr:%d dr:%d tr:%d er:%d\n",wr,hr,dr,tr,er);
	kinfo(KVERBOSE,"numpts:%d numptsunit:%d numunits:%d numptsreg:%d,numrgns:%d\n",
                       numpts,numptsunit,numunits,numptsreg,numrgns);

        /* Set up offset and range for the various data types */
        maxcnt=256;
        mostneg=0;
	if (type == KBYTE) { mostneg=-128; maxcnt=256; }
	else if (type == KUBYTE) { mostneg=0; maxcnt=256; }
	else if (type == KSHORT) { mostneg=-32768; maxcnt=65536; }
	else if (type == KUSHORT) { mostneg=0; maxcnt=65536; }

        for (numu=0; numu<numunits; numu++)
        {
           /* Zero the histo array */
	   (void)kbzero((kaddr)hist,maxcnt*(unsigned int)sizeof(long));

           /* Preserve the object state for writing later */
           if ((in2_obj = kpds_reference_object(in_obj)) == KOBJECT_INVALID)
           {
              kerror(lib, rtn, "Failed to reference input object");
              return(FALSE);
           }

	   kinfo(KVERBOSE,"reading unit #%d\n",numu);

           for (numr=0; numr<numrgns; numr++)
           {
             /* Grab a region of goodies from the input object */
	     b = (int *) kpds_get_data(in_obj,KPDS_VALUE_REGION, (kaddr)b );
             if (mask_exists)
               bmask=(int *)kpds_get_data(in_obj,KPDS_MASK_REGION,(kaddr)bmask);

             /* Form the histogram */
             if (!mask_exists)
               {
                 for (i=0; i<numptsreg; i++)
                   {
                     n = b[i]-mostneg;
                     hist[n]++;
                   }
               }
             else
               {
                 for (i=0; i<numptsreg; i++)
                   {
                     if (bmask[i] != 0)
                       {
                         n = b[i]-mostneg;
                         hist[n]++;
                       }
                   }
               }

           }

	   kinfo(KVERBOSE,"processing histo op for unit #%d\n",numu);

           switch (khistopfunc)
	   {
		case KHISTOPS_EQUALIZE:
           		/* Integrate to form equalization map */
	   		k=0;
	  	 	for (i=0; i<maxcnt; i++)
             		{
             		  k += hist[i];
               		  map[i] = k;
             		}

           		/* Scale the map back to 0..maxcnt-1 */
           		if (map[maxcnt-1] == 0)
            		 {
            		   kerror(lib, rtn, "Cumulative distribution is zero.");
             		  (void) kpds_close_object(in_obj);
            		   (void) kpds_close_object(out_obj);
            		   return(FALSE);
           		  }
	
           		scale=(double)(maxcnt-1)/(double)map[maxcnt-1];
           		for (i=0; i<maxcnt; i++)
                          {
                            x = map[i]*scale-mostneg;
                            map1[i] = ktrunc(x+0.5);
                          }
			break;
		case KHISTOPS_STRETCH:
			/* Locate upper and lower bounds */
			lower=-1; upper=-1;
			for (i=0; i<maxcnt; i++)
			  {
			    if (hist[i] != 0 && lower==-1) lower=i;
                            if (lower != -1) break;
                          }
                        for (i=maxcnt-1; i>=0; i--)
                          {
                            if (hist[i] != 0 && upper==-1) upper=i;
                            if (upper != -1) break;
                          }
                        if (lower == upper) upper=lower+1;
			for (i=lower; i<=upper; i++) 
                            map1[i] = ktrunc((double)mostneg+(double)(maxcnt-1)*
                                    (double)(i-lower)/(double)(upper-lower));
			break;
               default:
                  kerror(lib, rtn, "Failed to reference input object");
                  (void) kpds_close_object(in_obj);
                  (void) kpds_close_object(in2_obj);
                  (void) kpds_close_object(out_obj);
                  return(FALSE);
	   }

           /* Invert the map if invert after processing is requested */
	   if (invertflag) for (i=0; i<maxcnt; i++) map1[i]=(maxcnt-1)-map1[i];

	   kinfo(KVERBOSE,"mapping and writing unit #%d\n",numu);

           /* Remap the data */
           for (numr=0; numr<numrgns; numr++)
           {
	     b = (int *)kpds_get_data(in2_obj,KPDS_VALUE_REGION,(kaddr)b );
             if (mask_exists)
	       bmask = (int *)kpds_get_data(in2_obj,KPDS_MASK_REGION,(kaddr)bmask );

             if (!mask_exists)
               {
                 for (i=0; i<numptsreg; i++)
                   {
                     n=b[i]-mostneg;
                     b[i]=map1[n];
                   }
               }
             else
               {
                 for (i=0; i<numptsreg; i++)
                   {
                     if (bmask[i] != 0)
                       {
                         n=b[i]-mostneg;
                         b[i]=map1[n];
                       }
                   }
               }

             if (!kpds_put_data(out_obj, KPDS_VALUE_REGION, (kaddr) b))
               {
                 kerror(lib, rtn, "failed to put unit in output object");
                 (void) kpds_close_object(in_obj);
                 (void) kpds_close_object(out_obj);
                 return(FALSE);
               }
           }
           (void) kpds_close_object(in2_obj);
         }

        (void) kpds_close_object(in_obj);
        (void) kpds_close_object(out_obj);

	return(TRUE);
}
/* -library_code_end */
