 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for kgsin
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lkgsin
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lkgsin - generate sinusoidal value data
* 
*       Purpose: Library Routine for kgsin. lkgsin creates a 
*                value segment filled with sinusoidal values for the 
*                data out_obj. The value segment will be set to the 
*                the dimension of w_size * h_size * d_size * t_size
*                * e_size that were specified. The data type of the
*                value segment is specified by the datatype parameter,
*                and can be KBIT, KBYTE, KUBYTE, KSHORT, KUSHORT,
*                KINT, KUINT, KLONG, KULONG, KFLOAT, KDOUBLE, KCOMPLEX 
*                and KDCOMPLEX.  The value segment will contain 
*                either a sum of 5 sinusoids, one for each dimension
*                or the sine of the sum of the coordinates at each point
*                depending on the value of the parameter "sum". 
*
*         Input: w_size    - The width of the out_obj being created
*                h_size    - the height of the out_obj being created
*                d_size    - the depth of the out_obj being created
*                t_size    - the size of the out_objs time dimension
*                e_size    - the out_objs element dimension size
*                sum       - flag, if TRUE sine is calculated along 
*                            each dimension else sine is calculated
*                            on the sum of the coordinates
*                ampl      - pointer to amplitude of the sine being 
*                            generated. If sum is TRUE, an amplitude
*                            is needed for each dimension else only
*                            one amplitude value is needed.
*                wnum      - number of sine cycles along width
*                hnum      - number of sine cycles along height
*                dnum      - number of sine cycles along depth
*                tnum      - number of sine cycles along time
*                enum      - number of sine cycles along elements
*                w_phase   - phase offset along width
*                h_phase   - phase offset along height
*                d_phase   - phase offset along depth
*                t_phase   - phase offset along time
*                e_phase   - phase offset along elements
*                data_type - datatype of the output out_obj
*
*        Output: out_obj   - the output out_obj
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Ashish Malhotra
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int lkgsin(int w_size, int h_size,
           int d_size, int t_size,
           int e_size, int sum,
           double *ampl,double w_num,
           double h_num, double d_num, double t_num,
           double e_num, double w_phase,
           double h_phase, double d_phase,
           double t_phase, double e_phase,
           int data_type, kobject out_obj)
/* -library_def_end */

/* -library_code */
{
   int       i,j,k,l,m;
   double    *dbl_data = NULL;
   kdcomplex *cmplx_data = NULL;
   kaddr     *data = NULL;
   char      *lib = "kdatamanip";
   char      *rtn = "lkgsin";
   double    expr;
   int       error = FALSE;

   /*
    * Some basic checks on the args passed in.
    */

   if (out_obj == NULL)
   {
      kerror(lib,rtn,"The output object argument must be a valid data object.");
      return (FALSE);
   }

   if (w_size <= 0 || h_size <= 0 || d_size <= 0 || t_size <= 0 || e_size <= 0)
   {
      kerror (lib,rtn,"The size arguments must all be greater than zero.");
      return (FALSE);
   }

   /* 
    * Create value segment if none exists.
    */
   if (!kpds_query_value(out_obj))
      (void)kpds_create_value(out_obj);

   /*
    * Set the size and datatype on the output object.
    */

   if (!kpds_set_attributes(out_obj,KPDS_VALUE_SIZE,w_size,h_size,d_size,
                          t_size,e_size, KPDS_VALUE_DATA_TYPE, data_type, NULL))
   {
      kerror (lib,rtn,"Failed to set attributes on the output object's %s",
              "value segment");
      return FALSE;
   }
 
   /*
    * Since all attributes that the program needs to change are set,
    * reference the kobject so that all other attribute changes affect 
    * the presentation layer only.
    */

   if ((out_obj = kpds_reference_object(out_obj)) == KOBJECT_INVALID)
   {
      kerror (lib,rtn,"A reference object can not be created %s",
              "for the output object.");
      return FALSE;
   }

   /* 
    * Reset the position on the object and set the presentation 
    * type on the output object to be KDOUBLE or KCOMPLEX depending on the 
    * type of the physical layer.
    */

   kpds_set_attribute(out_obj, KPDS_VALUE_POSITION, 0, 0, 0, 0, 0);

   data_type = (data_type < KCOMPLEX) ? KDOUBLE : KDCOMPLEX;

   kpds_set_attribute(out_obj, KPDS_VALUE_DATA_TYPE, data_type);

   /* 
    * Get a line of data from the output object to allcate the right
    * amount of space for data. Depending on the presentation datatype
    * cast data to a kdcomplex * or a double *
    */

   if ((data = kpds_get_data(out_obj, KPDS_VALUE_LINE, NULL)) == NULL)
   {
      (void) kpds_close_object(out_obj);
      kerror (lib,rtn,"Failed to get data from output object.");
      return FALSE;
   }

   if (data_type == KDCOMPLEX)
      cmplx_data = (kdcomplex *) data;
   else
      dbl_data = (double *) data;

   /*
    * Depending in whether a sine of sums is requested (sum = 0) or a
    * sum of sines is requested (sum = 1) the array dbl_data or cmplx_data
    * is filled. Four levels of nested loops here which assume that the
    * index order on the referenced object is WIDTH,HEIGHT,DEPTH,TIME,ELEMENTS.
    */

   if (sum)
   {
      double angle1,angle2,angle3,angle4,expr2;

      for (i=0; i<e_size; i++)
      {
         angle4 = K2PI*e_num*i/e_size + e_phase*K2PI/360 ;
         for (j=0; j<t_size; j++)
         {
            angle3  = K2PI*t_num*j/t_size + t_phase*K2PI/360 ;
            for (k=0; k<d_size; k++)
            {
               angle2 = K2PI*d_num*k/d_size + d_phase*K2PI/360 ;
               for (l=0; l<h_size; l++)
               {
                  angle1 = K2PI*h_num*l/h_size + h_phase*K2PI/360 ;

                  expr = ampl[4]*ksin(angle4) + ampl[3]*ksin(angle3);
                  expr += ampl[2]*ksin(angle2) + ampl[1]*ksin(angle1);

                  if (dbl_data)
                  {
                     for (m=0; m<w_size; m++)
                     {
                         dbl_data[m] = expr + ampl[0]*ksin(K2PI*w_num*m/w_size +
                                              w_phase*K2PI/360);
                     }
                  }
                  else
                  {
                     expr2 = ampl[4]*kcos(angle4) + ampl[3]*kcos(angle3);
                     expr2 += ampl[2]*kcos(angle2) + ampl[1]*kcos(angle1);

                     for (m=0; m<w_size; m++)
                     {
                        expr += ampl[0]*ksin(K2PI*w_num*m/w_size+
                                w_phase*K2PI/360);
                        expr2 += ampl[0]*kcos(K2PI*w_num*m/w_size+
                                 w_phase*K2PI/360);

                        cmplx_data[m] = kdccomp(expr2,expr);
                     }
                  }

                  if (!kpds_put_data(out_obj,KPDS_VALUE_LINE,data))
                  {
                     kerror(lib,rtn,"Failed to put line into output object.");
                     error = TRUE;
                     break;
                  }
               }
               if (error) break;
            }
            if (error) break;
         }
         if (error) break;
      }
   }
   else
   {
      double angle1,angle2,angle3,angle4;

      for (i=0; i<e_size; i++)
      {
         angle4 = K2PI*e_num*i/e_size + e_phase*K2PI/360 ;
         for (j=0; j<t_size; j++)
         {
            angle3  = K2PI*t_num*j/t_size + t_phase*K2PI/360 ;
            for (k=0; k<d_size; k++)
            {
               angle2 = K2PI*d_num*k/d_size + d_phase*K2PI/360 ;
               for (l=0; l<h_size; l++)
               {
                  angle1 = K2PI*h_num*l/h_size + h_phase*K2PI/360 ;
                  if (dbl_data)
                  {
                     for (m=0; m<w_size; m++)
                     {
                        dbl_data[m] = ampl[0]* 
                                      ksin(angle1 + angle2 + angle3 + angle4 +
                                      K2PI*w_num*m/w_size + w_phase*K2PI/360);
                     }
                  }
                  else
                  {
                     for (m=0; m<w_size; m++)
                     {
                        double val1,val2;
                        val1 = ampl[0]*
                               ksin(angle1 + angle2 + angle3 + angle4 +
                                    K2PI*w_num*m/w_size + w_phase*K2PI/360);
                        val2 = ampl[0]*
                               kcos(angle1 + angle2 + angle3 + angle4 +
                                    K2PI*w_num*m/w_size + w_phase*K2PI/360);
                        cmplx_data[m] = kdccomp(val2,val1); 
                     }
                  }
                  
                  if (!kpds_put_data(out_obj,KPDS_VALUE_LINE,data))
                  {
                     kerror(lib,rtn,"Failed to put line into output object.");
                     error = TRUE;
                     break;
                  }
               }
               if (error) break;
            }
            if (error) break;
         }
         if (error) break;
      }
   }

   kfree(data);
   kpds_close_object(out_obj);

   if (error)
      return FALSE;
   else
      return TRUE;
}
/* -library_code_end */
