 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for kgnoise
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lkgnoise
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
/*
 * Operation structure declaration
 */
struct _gen_func
{
        char    *name;
        int     routine;
};


/*
 * Function Declaration list
 */
static struct _gen_func gen_func[] =
{
#define KGNOISE_GAUSS 1
    { "gauss", KGNOISE_GAUSS },
#define KGNOISE_RAYLEIGH 2
    { "ray",KGNOISE_RAYLEIGH},
#define KGNOISE_UNIFORM  3
    { "unif",KGNOISE_UNIFORM},
#define KGNOISE_POISSON  4
    { "poiss",KGNOISE_POISSON},
#define KGNOISE_EXP      5
    {"exp",KGNOISE_EXP      },
};
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lkgnoise - generate noise in value segment of data object
* 
*       Purpose: Library Routine for kgnoise.  lkgnoise creates   
*                noisy value segment for a data object. The value    
*                segment will be set to the dimension of width*
*                height * depth * time * elements that were specified.
*                The data type of the value segment is specified by
*                the datatype int and can be KBIT, KBYTE, KUBYTE, KSHORT, 
*                KUSHORT, KINT, KUINT, KLONG, KULONG, KFLOAT, KDOUBLE, 
*                KCOMPLEX and KDCOMPLEX.  The value segment will be set 
*                to the noisy data having the distribution specified by 
*                the noise_type string. The noise distribution can be 
*                Gaussian, Rayleigh, Uniform, Exponential or Poisson.
*                The parameters associated with a particular distribution
*                can be specifed by noise_apec1 and noise_spec2 (optional
*                depending on the distribution).
*
*         Input: width       - The width of the object being created
*                height      - the height of the object being created
*                depth       - the depth of the object being created
*                t           - the size of the objects time dimension
*                elements    - the objects element dimension size
*                datatype    - an int defining the datatype
*                noise_type  - a string defining the noise type to be generated
*                noise_spec1 - double specifying noise characteristic
*                noise_spec2 - double specifying noise characteristic
*
*        Output: object    - the output object
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Ashish Malhotra, Scott Wilson
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int lkgnoise(
             int width, 
	     int height,
             int depth, 
             int t,
             int elements,
             char *noise_type,
             int datatype,
             double noise_spec1,
             double noise_spec2,
             kobject object)
/* -library_def_end */

/* -library_code */
{
   int  i,routine,otype;
   kobject ref_object;
   int  num_lines, line_size;
   int  num, func;
   kdcomplex *cmplx_data = NULL;
   double    *real_data = NULL, *imag_data = NULL;

   if (object == KOBJECT_INVALID) 
   {
      kerror ("kdatamanip", "lkgnoise",
         "The object argument must be a valid data object.");
      return (FALSE);
   }

   if (width <= 0 || height <= 0 || depth <= 0 || t <= 0 || elements <= 0) 
   {
      kerror ("kdatamanip", "lkgnoise",
         "The size arguments must all be greater than zero.");
      return (FALSE);
   }

   /* See which noise generator to use. */
   for (num = 0, func = -1; num < knumber(gen_func); num++)
   {
      if (kstrcmp(noise_type,gen_func[num].name) == 0)
      {
         func = num;
         break;
      }
   }
   if (func == -1)
   {
      kerror("kdatamanip","kgnoise","Sorry, but the distribution '%s' %s",
      noise_type,"is not in the list of supported distributions at this time.");
      return(FALSE);
   }
   routine = (int) gen_func[func].routine;

   /* Create the value segment if it's not there already */
   if (!kpds_query_value(object)) (void)kpds_create_value(object);

   /* Set the size and data type of the value segment */
   if (!kpds_set_attributes(object,
           KPDS_VALUE_SIZE, width, height, depth, t, elements,
           KPDS_VALUE_DATA_TYPE, datatype,
           NULL)) 
   {
      kerror ("kdatamanip", "lkgnoise",
         "The attributes can not be set on the output object's value segment.");
      return(FALSE);
   }

   /* Reference the object to eliminate side effects */
   if ((ref_object = kpds_reference_object(object)) == KOBJECT_INVALID) 
   {
      kerror ("kdatamanip", "lkgnoise",
         "A reference object can not be created for the output object.");
      return(FALSE);
   }

   /* Insurance */
   kpds_set_attribute(ref_object, KPDS_VALUE_POSITION, 0, 0, 0, 0, 0);

   /* See what size the processing lines are */
   kpds_get_attribute(ref_object, KPDS_VALUE_LINE_INFO, &line_size, &num_lines);

   /* Set the data type for presentation */
   if ( (datatype == KCOMPLEX) || (datatype == KDCOMPLEX) ) otype = KDCOMPLEX;
   else otype = KDOUBLE;
   kpds_set_attribute(ref_object, KPDS_VALUE_DATA_TYPE, otype);

   /* Allocate space for the data buffers */
   if (otype == KDOUBLE) real_data=(double *)kmalloc(line_size*sizeof(double));
   else
     {
       cmplx_data = (kdcomplex *)kmalloc(line_size*sizeof(kdcomplex));
       real_data = (double *)kmalloc(line_size*sizeof(double));
       imag_data = (double *)kmalloc(line_size*sizeof(double));
     }
   if (real_data == NULL || 
       (otype == KDCOMPLEX && (cmplx_data==NULL || imag_data==NULL)))
     {
       kpds_close_object(ref_object);
       if (cmplx_data) kfree(cmplx_data);
       if (real_data) kfree(real_data);
       if (imag_data) kfree(imag_data);    
       kerror ("kdatamanip", "lkgnoise", "Cannot allocate data buffers.");
       return(FALSE);
     }

    /* Generate the numbers line by line and toss them out */
    for (i=0; i<num_lines; i++)
    {
        switch (routine)
        {
           case KGNOISE_GAUSS:
                   kgen_gauss(line_size,noise_spec1,noise_spec2,real_data);
                   if (otype == KDCOMPLEX)
                      kgen_gauss(line_size,noise_spec1,noise_spec2,imag_data);
                   break;
           case KGNOISE_RAYLEIGH:
                   kgen_rayleigh(line_size,noise_spec1,real_data);
                   if (otype == KDCOMPLEX)
                      kgen_rayleigh(line_size,noise_spec1,imag_data);
                   break;
           case KGNOISE_UNIFORM:
                   kgen_unif(line_size,noise_spec1,noise_spec2,real_data);
                   if (otype == KDCOMPLEX)
                      kgen_unif(line_size,noise_spec1,noise_spec2,imag_data);
                   break;
           case KGNOISE_POISSON:
                   kgen_poisson(line_size,noise_spec1,noise_spec2,real_data);
                   if (otype == KDCOMPLEX)
                      kgen_poisson(line_size,noise_spec1,noise_spec2,imag_data);
                   break;
           case KGNOISE_EXP :
                   kgen_expon(line_size,noise_spec1,real_data);
                   if (otype == KDCOMPLEX)
                      kgen_expon(line_size,noise_spec1,imag_data);
                   break;
        }

        if (otype == KDCOMPLEX)
        {
           kcombine_to_complex(&cmplx_data,line_size,real_data,imag_data);
           (void) kpds_put_data(ref_object, KPDS_VALUE_LINE, (kaddr)cmplx_data);
        }
        else
           (void) kpds_put_data(ref_object, KPDS_VALUE_LINE, (kaddr)real_data);

    }
    
    if (cmplx_data) kfree(cmplx_data);
    if (real_data) kfree(real_data);
    if (imag_data) kfree(imag_data);    

    kpds_close_object(ref_object);
    
    return TRUE;
}
/* -library_code_end */
