 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for kgate
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lkgate
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
void kgate_uchar    PROTO((long, unsigned char, int, unsigned char *, 
                           unsigned char *, unsigned char *, unsigned char *, 
			   unsigned char *));
void kgate_ulong    PROTO((long, unsigned long, int, unsigned long *, 
                           unsigned long *, unsigned char *, unsigned char *, 
			   unsigned long *));
void kgate_long     PROTO((long, long, int, long *, long *, 
			   unsigned char *, unsigned char *, long *));
void kgate_double   PROTO((long, double, int, double *, double *, 
			   unsigned char *, unsigned char *, double *));
void kgate_dcomplex PROTO((long, double, int, kdcomplex *, kdcomplex *, 
			   unsigned char *, unsigned char *, double *));
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lkgate - write data 1 or data 2 to destination depending on gate
* 
*       Purpose: lkgate takes two source objects and writes out
*                source object 1 or source object 2 to the destination
*                object depending on the value of the gating object.
*                A non-zero gating value implies that the values
*                taken on by source object 2 are propogated to the
*                destination object. Source object 1 values are
*                used if a zero gating value is found.
*
*		 If the source objects have map data, the data is
*		 mapped before the gating process.  If the destination
*		 object has a map, it will be destroyed.
*
*		 If any of the source objects have masks, the destination
*		 object will also have a map.  If none exists on the
*		 destination, one will be created.  The resulting mask
*		 will be the result of logically ANDing the source masks.
*
*         Input: src1_obj   - First source object
*                src2_obj   - Second source object
*                gate_obj   - Gating object which determines the output
*		 gate_val   - Value defining when to perform gating operation
*		 gate_logic - Gating logic: if gating element = gate_val, 
*			     (0) use src1_obj, (1) use src2_obj
*
*        Output: dest_obj   - Destination object which contains results
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Donna Koechner & Ashish Malhotra
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: *   
*   Declaration: int lkgate(
*                !   kobject src1_obj,
*                !   kobject src2_obj,
*                !   kobject gate_obj,
*                !   double  gate_val,
* 		 !   int     gate_logic,
*                !   kobject dest_obj)
****************************************************************/
/* -library_def */
int lkgate(
    kobject src1_obj,
    kobject src2_obj,
    kobject gate_obj,
    double  gate_val,
    int     gate_logic,
    kobject dest_obj)
/* -library_def_end */

/* -library_code */
{
        int  src1_map_exists=0, src1_val_exists=0, src1_mask_exists=0;
        int  src2_map_exists=0, src2_val_exists=0, src2_mask_exists=0;
        int  gate_map_exists=0, gate_val_exists=0, gate_mask_exists=0;
        char *region=NULL, *opt_region=NULL, *datasize=NULL, *datatype=NULL; 
	char *padtype=NULL, *interptype=NULL, *position=NULL, *region_size=NULL;
        int  w1,h1,d1,t1,e1, w2,d2,h2,t2,e2;
        int  type1, type2, gate_type, ptype;
        int  error_flag = FALSE, process_value = TRUE;
        int  num, i, num_rgns;
	int  grid;
        long num_pts;
        kaddr *data1=NULL, *data2=NULL, *gate_data=NULL;
        unsigned char *mask1=NULL, *mask2=NULL;
        char    *lib = "kdatamanip";
        char    *prog = "lkgate";

        
	/* Check that the objects passed into lkgate are all valid 
	 * data objects.
	 */

	if ((src1_obj == KOBJECT_INVALID) || (src2_obj == KOBJECT_INVALID) ||
	    (gate_obj == KOBJECT_INVALID) || (dest_obj == KOBJECT_INVALID))
	{
           kerror(lib, prog, "An invalid object was passed into lkgate.");
           return(FALSE);
	}

	/*
	 *  Check if the any source object has location or time data.  We
	 *  currently cannot operate on objects with rectilinear or
	 *  curvilinear location data or time data, so if either
	 *  exist, return. 
	 */ 
        if (kpds_query_location(src1_obj))
	{
	   if (!kpds_get_attribute(src1_obj, KPDS_LOCATION_GRID, &grid))
	   {
	      kerror(lib, prog, "Unable to retrieve location grid attribute.");
	      return(FALSE);
	   }
	   if (grid != KUNIFORM && (grid == KRECTILINEAR || 
				    grid == KCURVILINEAR))
	   {
	      kerror(lib, prog, 
		     "First source object can not contain explicit "
		     "rectilinear or curvilinear location data.");
	   }
	}
	if (kpds_query_location(src2_obj))
	{
	   if (!kpds_get_attribute(src2_obj, KPDS_LOCATION_GRID, &grid))
	   {
	      kerror(lib, prog, "Unable to retrieve location grid attribute.");
	      return(FALSE);
	   }
	   if (grid != KUNIFORM && (grid == KRECTILINEAR || 
				    grid == KCURVILINEAR))
	   {
	      kerror(lib, prog, 
		     "Second source object can not contain explicit "
		     "rectilinear or curvilinear location data.");
	   }
	}
	if (kpds_query_location(gate_obj))
	{
	   if (!kpds_get_attribute(gate_obj, KPDS_LOCATION_GRID, &grid))
	   {
	      kerror(lib, prog, "Unable to retrieve location grid attribute.");
	      return(FALSE);
	   }
	   if (grid != KUNIFORM && (grid == KRECTILINEAR || 
				    grid == KCURVILINEAR))
	   {
	      kerror(lib, prog, 
		     "Gate object can not contain explicit "
		     "rectilinear or curvilinear location data.");
	   }
	}
        if ( (kpds_query_time(src1_obj)) || (kpds_query_time(src2_obj)) || 
	     (kpds_query_time(gate_obj)) )
        {
           kerror(lib, prog,
                  "Source object(s) cannot contain time location data.");
           return FALSE;
        }

        src1_val_exists  = kpds_query_value(src1_obj);
        src1_map_exists  = kpds_query_map(src1_obj);
        src1_mask_exists = kpds_query_mask(src1_obj);
 
        src2_val_exists  = kpds_query_value(src2_obj);
        src2_map_exists  = kpds_query_map(src2_obj);
        src2_mask_exists = kpds_query_mask(src2_obj);
 
        gate_val_exists  = kpds_query_value(gate_obj);
        gate_map_exists  = kpds_query_map(gate_obj);
        gate_mask_exists = kpds_query_map(gate_obj);

	/* Check gating object.  If the gate has mask data, it must 
	 * also have value data.
	 */
        if (gate_mask_exists || !gate_val_exists)
        {
           kerror(lib, prog,
                  "Gating object missing value data, or contains mask data.");
           return (FALSE);
        }
 
	/* Source objects must have value and/or map data
	 */
        if (!src1_map_exists && !src1_val_exists)
        {
           kerror(lib, prog,
                  "No Map or Value data exist for first source object");
           return (FALSE);
        }
        if (!src2_map_exists && !src2_val_exists)
        {
           kerror(lib, prog,
                  "No Map or Value data exist for second source object");
           return (FALSE);
        }
 
	/* An object having only map data (no value data) cannot be gated 
	 * with an object that *does* have value data (with or without map).
	 */
        if ( (!src2_map_exists &&  src2_val_exists  &&
              !src1_val_exists &&  src1_map_exists) ||
             ( src2_map_exists && !src2_val_exists  &&
               src1_val_exists && !src1_map_exists) ||
             ( src2_map_exists && !src2_val_exists  &&
               src1_val_exists &&  src1_map_exists) ||
             ( src1_map_exists && !src1_val_exists  &&
               src2_map_exists &&  src2_val_exists) )
        {
           kerror(lib, prog,
                  "Source objects have illegal combination of segments.");
           return(FALSE);
        }


	/* Reference the source and gating objects.
	 */
        if ( (src1_obj = kpds_reference_object(src1_obj)) == KOBJECT_INVALID)
        {
           kerror(lib, prog, "Failed to reference source object 1.");
           return(FALSE);
        }
 
        if ( (src2_obj = kpds_reference_object(src2_obj)) == KOBJECT_INVALID)
        {
           (void) kpds_close_object(src1_obj);
           kerror(lib, prog, "Failed to reference source object 2.");
           return(FALSE);
        }
 
        if ( (gate_obj = kpds_reference_object(gate_obj)) == KOBJECT_INVALID)
        {
           (void) kpds_close_object(src1_obj);
           (void) kpds_close_object(src2_obj);
           kerror(lib, prog, "Failed to reference gating object.");
           return(FALSE);
        }

	/* If the source objects have value data, and either one has a map,
	 * set KPDS_MAPPING_MODE attribute on all objects to be KMAPPED.
         * (In the case that only map data exists, any masks that might 
	 * exist are ignored.) 
	 */
        if ((src1_val_exists && src2_val_exists &&
            (src1_map_exists || src2_map_exists)))
        {
	   if (src1_map_exists)
	   {
              if (!kpds_set_attribute(src1_obj, KPDS_MAPPING_MODE, KMAPPED))
	      {
                 kerror(lib, prog, "Unable to set mapping mode on object 1.");
                 error_flag = TRUE;
	      }
	      src1_map_exists = FALSE;
	   }
	   if (src2_map_exists)
	   {
              if (!kpds_set_attribute(src2_obj, KPDS_MAPPING_MODE, KMAPPED))
	      {
                 kerror(lib, prog, "Unable to set mapping mode on object 2.");
                 error_flag = TRUE;
	      }
	      src2_map_exists = FALSE;
	   }
	   if (gate_map_exists)
	   {
              if (!kpds_set_attribute(gate_obj, KPDS_MAPPING_MODE, KMAPPED))
	      {
                 kerror(lib, prog, "Unable to set gate object mapping mode.");
                 error_flag = TRUE;
	      }
	      gate_map_exists = FALSE;
	   }
           if (kpds_query_map(dest_obj))
	      if (!kpds_destroy_map(dest_obj))
	      {
                 kerror(lib, prog, "Unable to destroy destination map.");
                 error_flag = TRUE;
	      }
        }
	
	if (src1_val_exists || src2_val_exists)
	{
	   if (!kpds_query_value(dest_obj))
	      kpds_create_value(dest_obj);
	   if (src1_mask_exists || src2_mask_exists)
	   {
	      if (!kpds_query_mask(dest_obj))
                 kpds_create_mask(dest_obj);
	   }
	}
	if (src1_map_exists || src2_map_exists)
	{
	   if (!kpds_query_map(dest_obj))
	      kpds_create_map(dest_obj);
	}

	/* Gate data type cannot be complex */
	kpds_get_attribute(gate_obj, KPDS_VALUE_DATA_TYPE, &gate_type);
	if (gate_type == KCOMPLEX || gate_type == KDCOMPLEX)
	{
           kerror(lib, prog, "Gate object data type cannot be complex.");
           error_flag = TRUE;
	}
 
	/* Set up the attribute names and values to pass into 
	 * kpds_set_attribute calls.  If the source objects contain only
	 * map data, set the attribute names to be the map attribute
 	 * names.  Otherwise, the attribute names will be those of
	 * value segment.  
	 */
        if (src1_map_exists && !src1_val_exists &&
            src2_map_exists && !src2_val_exists)
        {
           process_value = FALSE;
           if ( !gate_map_exists)
           {
              kerror(lib, prog, "Gate object should be map data only.");
              error_flag = TRUE;
           }
           region      = KPDS_MAP_REGION;
           region_size = KPDS_MAP_REGION_SIZE;
           opt_region  = KPDS_MAP_OPTIMAL_REGION_SIZE;
           datasize    = KPDS_MAP_SIZE;
           datatype    = KPDS_MAP_DATA_TYPE;
           padtype     = KPDS_MAP_PAD_VALUE;
           interptype  = KPDS_MAP_INTERPOLATE;
	   position    = KPDS_MAP_POSITION;
        }
        else
        {
           process_value = TRUE;
           region      = KPDS_VALUE_REGION;
           region_size = KPDS_VALUE_REGION_SIZE;
           opt_region  = KPDS_VALUE_OPTIMAL_REGION_SIZE;
           datasize    = KPDS_VALUE_SIZE;
           datatype    = KPDS_VALUE_DATA_TYPE;
           padtype     = KPDS_VALUE_PAD_VALUE;
           interptype  = KPDS_VALUE_INTERPOLATE;
	   position    = KPDS_VALUE_POSITION;
        }
 
        if (!kpds_set_attributes(src1_obj, 
		position,   0,0,0,0,0,
                interptype, KPAD, 
		padtype, 0.0,0.0, NULL))
	{
           kerror(lib, prog, "Unable to set attributes on source object 1.");
           error_flag = TRUE;
	}
 
        if (!kpds_set_attributes(src2_obj,
                position, 0,0,0,0,0,
                interptype, KPAD,
                padtype, 0.0,0.0, NULL))
	{
           kerror(lib, prog, "Unable to set attributes on source object 2.");
           error_flag = TRUE;
	}
 
        if (!kpds_set_attributes(gate_obj,
                position, 0,0,0,0,0,
                interptype, KPAD,
                padtype, 0.0,0.0, NULL))
	{
           kerror(lib, prog, "Unable to set attributes on gate object.");
           error_flag = TRUE;
	}

        if ( (!kpds_get_attribute(src1_obj, datasize, &w1,&h1,&d1,&t1,&e1)) ||
	     (!kpds_get_attribute(src2_obj, datasize, &w2,&h2,&d2,&t2,&e2)))
	{
           kerror(lib, prog, "Unable to get data size of source object.");
           error_flag = TRUE;
	}
        w1 = kmax(w1, w2); h1 = kmax(h1, h2); d1 = kmax(d1, d2);
        t1 = kmax(t1, t2); e1 = kmax(e1, e2);
 
        if (!kpds_set_attribute(src1_obj, datasize, w1, h1, d1, t1, e1))
	{
           kerror(lib, prog, "Unable to set data size of source object 1.");
           error_flag = TRUE;
	}
        if (!kpds_set_attribute(src2_obj, datasize, w1, h1, d1, t1, e1))
	{
           kerror(lib, prog, "Unable to set data size of source object 2.");
           error_flag = TRUE;
	}
        if (!kpds_set_attribute(dest_obj, datasize, w1, h1, d1, t1, e1))
	{
           kerror(lib, prog, "Unable to set data size of destination object.");
           error_flag = TRUE;
	}
        if (!kpds_set_attribute(gate_obj, datasize, w1, h1, d1, t1, e1))
	{
           kerror(lib, prog, "Unable to set data size of gate object.");
           error_flag = TRUE;
	}

	if (process_value)
	{
	   if (src1_mask_exists)
	   {
	      if (!kpds_set_attributes(src1_obj, 
			KPDS_MASK_DATA_TYPE, KUBYTE,
			KPDS_MASK_INTERPOLATE, KPAD,
			KPDS_MASK_PAD_VALUE, 1.0,
			KPDS_MASK_POSITION, 0,0,0,0,0, NULL))
	      {
                 kerror(lib,prog,"Unable to set attributes on source 1 mask.");
                 error_flag = TRUE;
	      }
	   }
	   if (src2_mask_exists)
	   {
	      if (!kpds_set_attributes(src2_obj, 
			KPDS_MASK_DATA_TYPE, KUBYTE,
			KPDS_MASK_INTERPOLATE, KPAD,
			KPDS_MASK_PAD_VALUE, 1.0,
			KPDS_MASK_POSITION, 0,0,0,0,0, NULL))
	      {
                 kerror(lib,prog,"Unable to set attributes on source 2 mask.");
                 error_flag = TRUE;
	      }
	   }
	}

	/* Get data types of the source objects, and set the destination
	 * object to be the higher of the two.
	 */
        if ( (!kpds_get_attribute(src1_obj, datatype, &type1)) ||
             (!kpds_get_attribute(src2_obj, datatype, &type2)) )
	{
           kerror(lib,prog,"Unable to get datatype of source object.");
           error_flag = TRUE;
	}
 
        type1 = kdatatype_cast_output(type1, type2);
        if (!kpds_set_attribute(dest_obj, datatype, type1))
	{
           kerror(lib,prog,"Unable to set datatype of destination object.");
           error_flag = TRUE;
	}
 
	/* After setting all of the attributes that should be reflected
	 * in the pysicaly storage of the data, create a reference on 
	 * the destination object.
	 */
        if ((dest_obj = kpds_reference_object(dest_obj)) == KOBJECT_INVALID)
        {
           error_flag = TRUE;
           kerror(lib, prog, "Failed to reference destination object.");
        }
 
        if (error_flag)
        {
           (void) kpds_close_object(src1_obj);
           (void) kpds_close_object(src2_obj);
           (void) kpds_close_object(gate_obj);
           return (FALSE);
        }
 
	/* Decide which data type is best to process the data, and set all
	 * datatype attributes to that (except mask, which is always UBYTE).
	 * If the type comes back as double complex, set the gate data type
	 * to double.
	 */
        ptype = kdatatype_cast_process(type1, gate_type, 
			(KUBYTE | KLONG | KULONG | KDOUBLE | KDCOMPLEX));
	if (ptype == KDCOMPLEX)
	{
           if (!kpds_set_attribute(gate_obj, datatype, KDOUBLE))
           {
              kerror(lib, prog, "Failed to set datatype of gate object.");
              error_flag = TRUE;
           }
	}
	else
	{
           if (!kpds_set_attribute(gate_obj, datatype, ptype))
           {
              kerror(lib, prog, "Failed to set datatype of gate object.");
              error_flag = TRUE;
           }
	}

        if ( (!kpds_set_attribute(src1_obj, datatype, ptype)) ||
             (!kpds_set_attribute(src2_obj, datatype, ptype)) ||
             (!kpds_set_attribute(dest_obj, datatype, ptype)) )
        {
           kerror(lib, prog, "Failed to set data type on object.");
           error_flag = TRUE;
        }

	/* Get the optimal region size for processing (using the first source
	 * object as the master), and use this info to set the region size		 * attribute on all objects.
	 */
        if (!kpds_get_attribute(src1_obj, opt_region, &w2,&h2,&d2,&t2,&e2, 
		&num_rgns))
        {
           kerror(lib, prog, "Failed to get optimal region size.");
           error_flag = TRUE;
        }
	num_pts = w2*h2*d2*t2*e2;
	if ( (!kpds_set_attribute(src1_obj, region_size, w2,h2,d2,t2,e2)) ||
	     (!kpds_set_attribute(src2_obj, region_size, w2,h2,d2,t2,e2)) ||
	     (!kpds_set_attribute(gate_obj, region_size, w2,h2,d2,t2,e2)) ||
	     (!kpds_set_attribute(dest_obj, region_size, w2,h2,d2,t2,e2)) )
        {
           kerror(lib, prog, "Failed to set region size.");
           error_flag = TRUE;
        }
	if (process_value && src1_mask_exists)
	{
	   if (!kpds_set_attributes(src1_obj, 
			KPDS_MASK_REGION_SIZE, w2,h2,d2,t2,e2,
			KPDS_MASK_PAD_VALUE, 1.0, NULL))
           {
              kerror(lib, prog, "Failed to set mask attributes on source 1.");
              error_flag = TRUE;
           }
	}
	if (process_value && src2_mask_exists)
	{
	   if (!kpds_set_attributes(src2_obj, 
			KPDS_MASK_REGION_SIZE, w2,h2,d2,t2,e2,
			KPDS_MASK_PAD_VALUE, 1.0, NULL))
           {
              kerror(lib, prog, "Failed to set mask region size on source 2.");
              error_flag = TRUE;
           }
	}
	if (process_value && (src1_mask_exists || src2_mask_exists))
	{
	   if (!kpds_set_attributes(dest_obj, 
			KPDS_MASK_REGION_SIZE, w2,h2,d2,t2,e2,
	      		KPDS_MASK_DATA_TYPE, KUBYTE, NULL))
           {
              kerror(lib, prog, "Failed to set destination mask attributes.");
              error_flag = TRUE;
           }
	}
 
        if (!src1_mask_exists && src2_mask_exists)
	{
	   if ((mask1 = kmalloc(sizeof(unsigned char) * num_pts)) == NULL)
	   {
	      kerror(lib, prog, "Mask 1 memory allocation failed.");
              error_flag = TRUE;
	   }
	}

	if (!src2_mask_exists && src1_mask_exists)
	{
	   if ((mask2 = kmalloc(sizeof(unsigned char) * num_pts)) == NULL)
	   {
	      kerror(lib, prog, "Mask 2 memory allocation failed.");
              error_flag = TRUE;
	   }
	}

        if (error_flag)
        {
           (void) kpds_close_object(src1_obj);
           (void) kpds_close_object(src2_obj);
           (void) kpds_close_object(gate_obj);
           (void) kpds_close_object(dest_obj);
           return (FALSE);
        }
 
        for (num=0; num < num_rgns; num++)
        {
           data1 = kpds_get_data(src1_obj, region, data1);
           data2 = kpds_get_data(src2_obj, region, data2);
           gate_data = kpds_get_data(gate_obj, region, gate_data);
           
	   /* Since the gating functions that will be called below assume 
	    * that if one mask is passed in, both are, need to set up both
	    * mask1 and mask2 if either exist. 
 	    */
	   if (process_value)
	   {
              if (src1_mask_exists)
	      {
                 mask1 = (unsigned char *) kpds_get_data(src1_obj,
                             KPDS_MASK_REGION, (kaddr)mask1);
		 if (!src2_mask_exists)
		 {
		    for (i=0; i< num_pts; i++)
		       mask2[i] = 1;
		 }
	      }
	      if (src2_mask_exists)
	      {
                 mask2 = (unsigned char *) kpds_get_data(src2_obj,
                             KPDS_MASK_REGION, (kaddr)mask2);
		 if (!src1_mask_exists)
	         {
	            for (i=0; i< num_pts; i++)
	               mask1[i] = 1;
		 }
	      }
	   }

	   /* The gating functions put the results in data1 and mask1 (if 
	    * any mask exists) 
	    */
	   if (ptype == KUBYTE)
	   {
	      kgate_uchar(num_pts, (unsigned char)gate_val, gate_logic,
			  (unsigned char *) data1, (unsigned char *) data2, 
			  mask1, mask2, (unsigned char *) gate_data);
	   }

	   else if (ptype == KULONG)
	   {
	      kgate_ulong(num_pts, (unsigned long) gate_val, gate_logic,
			  (unsigned long *) data1, (unsigned long *) data2, 
			  mask1, mask2, (unsigned long *) gate_data);
	   }

	   else if (ptype == KLONG)
	   {
	      kgate_long(num_pts, (long) gate_val, gate_logic,
			 (long *) data1, (long *) data2, 
			 mask1, mask2, (long *) gate_data);
	   }

	   else if (ptype == KDOUBLE)
	   {
	      kgate_double(num_pts, gate_val, gate_logic,
			  (double *) data1, (double *) data2, 
			  mask1, mask2, (double *) gate_data);
	   }

	   else if (ptype == KDCOMPLEX)
	   {
	      kgate_dcomplex(num_pts, gate_val, gate_logic,
			    (kdcomplex *) data1, (kdcomplex *) data2, 
			    mask1, mask2, (double *) gate_data);
	   }

	   else 
	   {
	     /* unrecognized processing type - print error and get out */
              error_flag = TRUE;
	   }

           if (! kpds_put_data(dest_obj, region, (kaddr) data1) )
           {
              kerror(lib, prog,
                     "Failed to put %s data in output object.", region);
              error_flag = TRUE;
           }

           if (mask1)
           {
              if (! kpds_put_data(dest_obj, KPDS_MASK_REGION, (kaddr) mask1) )
              {
                 kerror(lib, prog, "Failed to put mask data in output object.");
                 error_flag = TRUE;
              }
           }
        }

        (void) kpds_close_object(src1_obj);
        (void) kpds_close_object(src2_obj);
        (void) kpds_close_object(dest_obj);
        (void) kpds_close_object(gate_obj);
      
        kfree(data1);
        kfree(data2);
        kfree(gate_data);
        if (mask1) kfree(mask1);
        if (mask2) kfree(mask2);

        if (error_flag)
           return FALSE;
        else
	   return TRUE;

}

/*-----------------------------------------------------------
|
|  Routine Name: kgate_uchar - gating for unsigned char data
|
|       Purpose: This function replaces values in the data1 and 
|		 mask1 arrays with those in the data2 and mask2
|		 arrays when the value in gate_data is equal to 
|		 gate_val, and gate_logic is true.  If gate_locic 
|		 is false, then data1 and mask1 values are replaced 
|		 when the gate_data is not equal to gate_val.
|                It is assumed that if either mask1 or mask2 is 
|                passed in, both masks are passed in.  
|
|         Input: numpts - number of points in data, gate, and mask arrays
|                gate_val - value defining when to perform gating operation
|		 gate_logic - gating logic: if gating element = gate_val, 
|		     	      (0) use src1_obj, (1) use src2_obj
|		 data1 - first data array
|		 data2 - second data array
|		 mask1 - mask associated with first data array
|		 mask2 - mask associated with second data array
|                gate_data - gating data array
|
|        Output: data1 - data results are stored in data1
|		 mask1 - if a mask exists, results are stored in mask1
|       Returns: nothing
|
|    Written By: Donna Koechner
|          Date: Thu Apr 28 1994
| Modifications:
|
------------------------------------------------------------*/

void kgate_uchar(
   long          numpts, 
   unsigned char gate_val, 
   int           gate_logic,
   unsigned char *data1, 
   unsigned char *data2, 
   unsigned char *mask1, 
   unsigned char *mask2, 
   unsigned char *gate_data )
{
        int j;

        /* gate_logic = 1:  if gating element = gate_val, use src2_obj */
	if (gate_logic == 1) 
	{
           for (j = 0; j < numpts; j++)
           {
	      if (gate_data[j] == gate_val)
	      {
	         data1[j] = data2[j];
		 if (mask1)
		    mask1[j] = mask2[j];
	      }
	   }
	}

        /* gate_logic = 0:  if gating element != gate_val, use src2_obj */
	else
	{
           for (j = 0; j < numpts; j++)
           {
	      if (gate_data[j] != gate_val)
	      {
	         data1[j] = data2[j];
		 if (mask1)
		    mask1[j] = mask2[j];
	      }
	   }
	}
}


/*-----------------------------------------------------------
|
|  Routine Name: kgate_ulong - gating for unsigned long data
|
|       Purpose: This function replaces values in the data1 and 
|		 mask1 arrays with those in the data2 and mask2
|		 arrays when the value in gate_data is equal to 
|		 gate_val, and gate_logic is true.  If gate_locic 
|		 is false, then data1 and mask1 values are replaced 
|		 when the gate_data is not equal to gate_val.
|                It is assumed that if either mask1 or mask2 is 
|                passed in, both masks are passed in.  
|
|         Input: numpts - number of points in data, gate, and mask arrays
|                gate_val - value defining when to perform gating operation
|		 gate_logic - gating logic: if gating element = gate_val, 
|		     	      (0) use src1_obj, (1) use src2_obj
|		 data1 - first data array
|		 data2 - second data array
|		 mask1 - mask associated with first data array
|		 mask2 - mask associated with second data array
|                gate_data - gating data array
|
|        Output: data1 - data results are stored in data1
|		 mask1 - if a mask exists, results are stored in mask1
|       Returns: nothing
|
|    Written By: Donna Koechner
|          Date: Thu Apr 28 1994
| Modifications:
|
------------------------------------------------------------*/

void kgate_ulong(
   long          numpts, 
   unsigned long gate_val, 
   int           gate_logic,
   unsigned long *data1, 
   unsigned long *data2, 
   unsigned char *mask1, 
   unsigned char *mask2, 
   unsigned long *gate_data)
{
        int j;

        /* gate_logic = 1:  if gating element = gate_val, use src2_obj */
	if (gate_logic == 1) 
	{
           for (j = 0; j < numpts; j++)
           {
	      if (gate_data[j] == gate_val)
	      {
	         data1[j] = data2[j];
		 if (mask1)
		    mask1[j] = mask2[j];
	      }
	   }
	}

        /* gate_logic = 0:  if gating element != gate_val, use src2_obj */
	else
	{
           for (j = 0; j < numpts; j++)
           {
	      if (gate_data[j] != gate_val)
	      {
	         data1[j] = data2[j];
		 if (mask1)
		    mask1[j] = mask2[j];
	      }
	   }
	}
}


/*-----------------------------------------------------------
|
|  Routine Name: kgate_long - gating for long data
|
|       Purpose: This function replaces values in the data1 and 
|		 mask1 arrays with those in the data2 and mask2
|		 arrays when the value in gate_data is equal to 
|		 gate_val, and gate_logic is true.  If gate_locic 
|		 is false, then data1 and mask1 values are replaced 
|		 when the gate_data is not equal to gate_val.
|                It is assumed that if either mask1 or mask2 is 
|                passed in, both masks are passed in.  
|
|         Input: numpts - number of points in data, gate, and mask arrays
|                gate_val - value defining when to perform gating operation
|		 gate_logic - gating logic: if gating element = gate_val, 
|		     	      (0) use src1_obj, (1) use src2_obj
|		 data1 - first data array
|		 data2 - second data array
|		 mask1 - mask associated with first data array
|		 mask2 - mask associated with second data array
|                gate_data - gating data array
|
|        Output: data1 - data results are stored in data1
|		 mask1 - if a mask exists, results are stored in mask1
|       Returns: nothing
|
|    Written By: Donna Koechner
|          Date: Thu Apr 28 1994
| Modifications:
|
------------------------------------------------------------*/

void kgate_long(
   long numpts, 
   long gate_val, 
   int  gate_logic,
   long *data1, 
   long *data2, 
   unsigned char *mask1, 
   unsigned char *mask2, 
   long *gate_data)
{
        int j;

        /* gate_logic = 1:  if gating element = gate_val, use src2_obj */
	if (gate_logic == 1) 
	{
           for (j = 0; j < numpts; j++)
           {
	      if (gate_data[j] == gate_val)
	      {
	         data1[j] = data2[j];
		 if (mask1)
		    mask1[j] = mask2[j];
	      }
	   }
	}

        /* gate_logic = 0:  if gating element != gate_val, use src2_obj */
	else
	{
           for (j = 0; j < numpts; j++)
           {
	      if (gate_data[j] != gate_val)
	      {
	         data1[j] = data2[j];
		 if (mask1)
		    mask1[j] = mask2[j];
	      }
	   }
	}
}


/*-----------------------------------------------------------
|
|  Routine Name: kgate_double - gating for double data
|
|       Purpose: This function replaces values in the data1 and 
|		 mask1 arrays with those in the data2 and mask2
|		 arrays when the value in gate_data is equal to 
|		 gate_val, and gate_logic is true.  If gate_locic 
|		 is false, then data1 and mask1 values are replaced 
|		 when the gate_data is not equal to gate_val.
|                It is assumed that if either mask1 or mask2 is 
|                passed in, both masks are passed in.  
|
|         Input: numpts - number of points in data, gate, and mask arrays
|                gate_val - value defining when to perform gating operation
|		 gate_logic - gating logic: if gating element = gate_val, 
|		     	      (0) use src1_obj, (1) use src2_obj
|		 data1 - first data array
|		 data2 - second data array
|		 mask1 - mask associated with first data array
|		 mask2 - mask associated with second data array
|                gate_data - gating data array
|
|        Output: data1 - data results are stored in data1
|		 mask1 - if a mask exists, results are stored in mask1
|       Returns: nothing
|
|    Written By: Donna Koechner
|          Date: Thu Apr 28 1994
| Modifications:
|
------------------------------------------------------------*/

void kgate_double(
   long   numpts, 
   double gate_val, 
   int    gate_logic,
   double *data1, 
   double *data2, 
   unsigned char *mask1, 
   unsigned char *mask2, 
   double *gate_data)
{
        int j;

        /* gate_logic = 1:  if gating element = gate_val, use src2_obj */
	if (gate_logic == 1) 
	{
           for (j = 0; j < numpts; j++)
           {
	      if (gate_data[j] == gate_val)
	      {
	         data1[j] = data2[j];
		 if (mask1)
		    mask1[j] = mask2[j];
	      }
	   }
	}

        /* gate_logic = 0:  if gating element != gate_val, use src2_obj */
	else
	{
           for (j = 0; j < numpts; j++)
           {
	      if (gate_data[j] != gate_val)
	      {
	         data1[j] = data2[j];
		 if (mask1)
		    mask1[j] = mask2[j];
	      }
	   }
	}
}


/*-----------------------------------------------------------
|
|  Routine Name: kgate_dcomplex - gating for double complex data
|
|       Purpose: This function replaces values in the data1 and 
|		 mask1 arrays with those in the data2 and mask2
|		 arrays when the value in gate_data is equal to 
|		 gate_val, and gate_logic is true.  If gate_locic 
|		 is false, then data1 and mask1 values are replaced 
|		 when the gate_data is not equal to gate_val.
|                It is assumed that if either mask1 or mask2 is 
|                passed in, both masks are passed in.  
|
|         Input: numpts - number of points in data, gate, and mask arrays
|                gate_val - value defining when to perform gating operation
|		 gate_logic - gating logic: if gating element = gate_val, 
|		     	      (0) use src1_obj, (1) use src2_obj
|		 data1 - first data array
|		 data2 - second data array
|		 mask1 - mask associated with first data array
|		 mask2 - mask associated with second data array
|                gate_data - gating data array
|
|        Output: data1 - data results are stored in data1
|		 mask1 - if a mask exists, results are stored in mask1
|       Returns: nothing
|
|    Written By: Donna Koechner
|          Date: Thu Apr 28 1994
| Modifications:
|
------------------------------------------------------------*/

void kgate_dcomplex(
   long          numpts, 
   double        gate_val, 
   int           gate_logic,
   kdcomplex     *data1, 
   kdcomplex     *data2, 
   unsigned char *mask1, 
   unsigned char *mask2, 
   double        *gate_data)
{
        int j;

        /* gate_logic = 1:  if gating element = gate_val, use src2_obj */
	if (gate_logic == 1) 
	{
           for (j = 0; j < numpts; j++)
           {
	      if (gate_data[j] == gate_val)
	      {
	         data1[j] = data2[j];
		 if (mask1)
		    mask1[j] = mask2[j];
	      }
	   }
	}

        /* gate_logic = 0:  if gating element != gate_val, use src2_obj */
	else
	{
           for (j = 0; j < numpts; j++)
           {
	      if (gate_data[j] != gate_val)
	      {
	         data1[j] = data2[j];
		 if (mask1)
		    mask1[j] = mask2[j];
	      }
	   }
	}
}
/* -library_code_end */
