/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Image Display Utility Routines
   >>>>
   >>>>  Private:
   >>>>   Static:
   >>>>			_escape
   >>>>			_delimit_str
   >>>>			_empty
   >>>>			_initstack
   >>>>			_push
   >>>>			_top_pop
   >>>>			_add_range
   >>>>			_get_match_str
   >>>>			_search_match
   >>>>			_match_parse
   >>>>			_analyze_key
   >>>>   Public:  
   >>>>			kparse_string_search
   >>>>			kparse_file_search
   >>>>			kparse_string_scan
   >>>>			kparse_file_scan
   >>>>			kparse_string_delimit
   >>>>			kparse_string_search_delimit
   >>>>			kparse_file_search_delimit
   >>>>			kparse_string_scan_delimit
   >>>>			kparse_file_scan_delimit
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

#define MAXCHR	128
#define CHRBIT	8
#define BITBLK	MAXCHR/CHRBIT
#define BLKIND	0170
#define BITIND	07

typedef /*unsigned*/ char CHAR;

static CHAR bittab[BITBLK];

#define isinset(y) 	(bittab[((y)&BLKIND)>>3] & (1<<((y)&BITIND)))

/*-----------------------------------------------------------
|
|  Routine Name: _escape - check to see if a character has an odd number of '\' 
in front of it
|
|       Purpose: This routine counts the number of '\'s in front of the
|                current character.  If the number is odd, TRUE is returned
|                otherwise FALSE is returned
|
|         Input: start   - the starting place of the string
|                current - the current location in the string
|
|        Output: 
|
|       Returns: TRUE (1) on an odd number of '\'s, FALSE (0) otherwise
|
|    Written By: Steven Jorgensen
|          Date: Nov 11, 1992 14:23
| Modifications:
|
------------------------------------------------------------*/
static int _escape(
   char *start,
   char *current)
  {
    int i = 0;
    if (current == start)
      return(FALSE);
    current--;
    while (*current == '\\' )
      {
        i++;
        if (current == start)
          break;
        current--;
      }
    return(((i%2) ? TRUE : FALSE));
  }

/*-----------------------------------------------------------
|
|  Routine Name: _delimit_str - make an array of strings by breaking
|                               a single string apart
|
|       Purpose: This routine breaks a single string into 1 or more
|                strings.  It decides where to break the strings
|                via a list of delimiters.  This is an internal
|                routine for kparse_string_delimit.
|
|         Input: data      - the string to break apart
|                delimiter - the string of delimiters
|		 mode      - the delimit mode
|
|        Output: array_cnt - the number of items in the array that was created
|
|       Returns: an array of strings created by delimiting the data string
|
|    Written By: Steven Jorgensen
|          Date: Sep 14, 1992 15:23
| Modifications:
|
------------------------------------------------------------*/
static char **_delimit_str(
   char *data,
   char *delimiter,
   int   mode,
   int  *array_cnt)
  {
    char **result = NULL, *start, *d;
    int    end = FALSE, i;

    *array_cnt = 0;
    if (kstrlen(data) == 0)
      return(NULL);
    start = data;
    for (i = 0; i < BITBLK; i++)
        bittab[i] = (char) 0;
    for (d = delimiter; d != NULL && *d != '\0'; d++)
	bittab[(((int)*d) & BLKIND) >> 3] |= 1 << (((int) *d) & BITIND);
    while (!end)
      {
        if ((isinset(*data) && _escape(start, data) == FALSE) || *data == '\0')
          {
	    end = (*data == '\0');
	    *data = '\0';
	    if ((mode & KDELIM_CLEAN) == KDELIM_CLEAN)
	      {
	        d = kstring_cleanup(start, NULL);
		if (d != NULL)
		  {
		    result = karray_insert(result, d, *array_cnt, KLIST_TAIL,
					   TRUE);
		    (*array_cnt)++;
		  }
		
	      }
	    else
	      {
		result = karray_insert(result, kstrdup(start), *array_cnt,
				KLIST_TAIL, TRUE);
		(*array_cnt)++;
	      }
		if (!end)
		{
		    start = data +1;
		    end = (*start == '\0');
		}
          }
        data++;
      }
    if (*array_cnt == 0)
      {
        kfree(result);
        return(NULL);
      }
    return(result);
  }

/************************************************************
*
*  Routine Name: kparse_string_search - match a search key in a data string
*
*       Purpose: This routine performs a regular expression search of an input
*		 data string for the first occurance of a specific search
*		 string.  The search string is specified as a unix regular
*		 expression string (see syntax listed below), and
*		 it returns a pointer to the portion of the string
*		 that follows the search string.  It also returns the
*		 exact format of the search string that the regular
*		 expression has matched.  The formatted return string is
*		 kmalloc'ed for you.
*
*         Input: data       - the data string to search through
*		 key        - the regular expression key to search for
*		 mode       - tells the parser which mode to work in:
*			      ! KIGNORE_CASE	Ignore case on a-z and A-Z
*			      ! KLITERAL		Use case information on
*			      ! 			a-z and A-Z
*
*        Output: key_format - the address of a pointer to hold the returned
*                             key that was matched.  Sufficient space for
*                             the returned string will be allocated if you
*                             pass in a valid pointer.  Note that if this
*                             parameter is passed in as NULL, this routine
*                             will ignore it, and the string that key matched
*                             will not be returned.
*		 status     - error status of the search.  It can be
*			      one of the following:
*			    ! KPARSE_OK		(parse ok, return data
*			    !				 valid)
*			    ! KPARSE_NOKEY		(couldn't find key)
*			    ! KPARSE_PARTKEY	(data ends on a partial
*			    !				 match)
*			    ! KPARSE_NULLKEY	(key was NULL)
*			    ! KPARSE_SYNTAXKEY	(key had an illegal syntax)
*       Returns: This function returns a character pointer set to the address
*		 of the character following the matched search key.  If an
*		 error occured, NULL is returned and status is set to the
*		 appropriate error status.
*
*  Restrictions: It does not support the following regular expression
*		 constructs:  or'ing '|',  grouping of regular expressions '()',
*		 match one or more times '+', or match n to m times '\\{n,m\\}'.
*		 Finally, the '\\number' and '\\(\\)' constructs have no meaning
*		 for these routines, so they are not supported either.
*
*		 Search keys and data strings should not contain the
*		 values '\\001', '\\002', '\\003', or '\\004',
*		 because these values are used as special search parameters
*		 by the parser.
*    Written By: Steven Jorgensen
*          Date: Aug 05, 1992 12:25
*      Verified:
*  Side Effects: This routine mallocs data and sets the value of key_format
*		 parameter.  Thus, the user should pass in an address of
*		 an unused character pointer.  The calling routine is
*		 responsible for freeing the space malloc'ed for
*		 the key_format parameter.
* Modifications:
*
*************************************************************/
char *kparse_string_search(
   char *data,
   char *key,
   int  mode,
   char **key_format,
   int  *status)
  {
    char *tmp1, *tmp2;
    if (key == NULL)
      {
	*status = KPARSE_NULLKEY;
	return(NULL);
      }
    if (mode == KIGNORE_CASE)
      kre_icomp(key);
    else
      kre_comp(key);
    *status = kre_status();
    if (*status != KPARSE_OK)
      return(NULL);
    kre_exec(data);
    *status = kre_status();
    if (*status != KPARSE_OK)
      return(NULL);
    kre_pos(0, (kaddr *) &tmp1, (kaddr *) &tmp2);
    if (key_format != NULL)
      {
	int i, l;
        i = tmp2-tmp1;
	l = kstrlen(key)-1;
        if (key[l] == '$' && *(tmp2-1) == '\n' &&
	    _escape(key, &key[l]) == FALSE)
	  i--;
	*key_format = kstring_ncopy(tmp1, i, NULL);
      }
    return(tmp2);
  }

/************************************************************
*
*  Routine Name: kparse_file_search - search a file for a specific key
*
*       Purpose: This routine is a Khoros Transport Interface to the string
*		 parser provided in the kparse_string_search routine.  It
*		 returns a malloc'ed copy of the key that was matched in the
*		 data file.  It also sets the current file position, with the
*		 transport call kfseek(), to the character following the
*		 matched key.
*
*         Input: file       - a pointer to the transport file opened for reading
*		 key        - The regular expression key to search for.
*		 mode       - tells the parser which mode to work in:
*			      ! KIGNORE_CASE	Ignore case on a-z and A-Z
*			      ! KLITERAL		Use case information on
*			      ! 			a-z and A-Z
*
*        Output: key_format - the address of a pointer to hold the returned
*                             key that was matched.  Sufficient space for
*                             the returned string will be allocated if you
*                             pass in a valid pointer.  Note that if this
*                             parameter is passed in as NULL, this routine
*                             will ignore it, and the string that key matched
*                             will not be returned.
*       Returns: return status of the search.  It can be one of the following:
*		 ! KPARSE_OK		(parse ok, return data valid),
*		 ! KPARSE_NOKEY		(couldn't find key)
*		 ! KPARSE_DATAERR	(something was wrong with the data)
*		 ! KPARSE_PARTKEY	(data ended with a partial match)
*		 ! KPARSE_NULLKEY	(key was NULL)
*		 ! KPARSE_SYNTAXKEY	(key had an illegal syntax)
*
*  Restrictions: It does not support the following regular expression
*		 constructs:  or'ing '|',  grouping of regular expressions '()',
*		 match one or more times '+', or match n to m times '\\{n,m\\}'.
*		 Finally, the '\\number' and '\\(\\)' constructs have no meaning
*		 for these routines, so they are not supported either.
*
*		 Search keys and the data file should not contain the
*		 values '\\001', '\\002', '\\003', or '\\004',
*		 because these values are used as special search parameters
*		 by the parser.
*    Written By: Steven Jorgensen
*          Date: Sep 15, 1992 20:37
*      Verified:
*  Side Effects: This routine mallocs data and sets the value of key_format
*		 parameter.  Thus, the user should pass in an address of
*		 an unused character pointer.  The calling routine is
*		 responsible for freeing the space malloc'ed for
*		 the key_format parameter.
* Modifications:
*
*************************************************************/
int
kparse_file_search(
   kfile *file,
   char  *key,
   int   mode,
   char  **key_format)
{
   char  buffer[(KLENGTH*10)+1];
   char  *bufptr;
   char  *cr;
   int   actual_read;
   int   done = FALSE;
   int   offset = 0;
   int   size;
   int   orig_pos = kftell(file);
   int   old_pos;
   int   tmp_pos;
   int   status = KPARSE_OK;
   int   herelast=FALSE;


    old_pos = orig_pos;
    if (key == NULL || kstrlen(key) == 0)
    {
	if (key_format != NULL)
	  *key_format = NULL;
	return(KPARSE_NULLKEY);
    }

    do
    {
	size = actual_read = kfread(buffer, 1, KLENGTH + offset, file);
	buffer[actual_read] = '\0';
	cr = (actual_read < (KLENGTH + offset)) ? &buffer[actual_read-1] :
	      					 kstrrchr(buffer,'\n');
	if (cr != NULL)
	  {
	    size = (cr + 1 - buffer);
	    *(cr+1) = '\0';
	    if (kfseek(file, -1 * (actual_read - size), 1) == -1)
	      kinfo(KSTANDARD, "kparse_file_search: kfseek failed");
	  }
	offset = 0;
	tmp_pos = kftell(file);
	bufptr = kparse_string_search(buffer, key, mode, key_format, &status);
	
	switch (status)
	  {
	    case KPARSE_OK:
	      done = TRUE;
	      if (kfseek(file, old_pos + (bufptr - buffer), 0) == -1)
		kinfo(KSTANDARD, "kparse_file_search: kfseek failed");
	      herelast = FALSE;
	    break;

	    case KPARSE_NOKEY:
	      old_pos = tmp_pos;
	      kinfo(KDEBUG, "%s(): %s",
                    "kparse_file_search", "Key not found in this buffer");
	      herelast = FALSE;
	    break;

	    case KPARSE_PARTKEY:
	      offset = KLENGTH;

	      if (kfseek(file, old_pos, 0) == -1)
		kinfo(KSTANDARD, "kparse_file_search: kfseek failed");
	      if (herelast == TRUE)
		{
	          if (kfseek(file, old_pos + size, 0) == -1)
		    kinfo(KSTANDARD, "kparse_file_search: kfseek failed");
		  old_pos = old_pos + size;
		  offset = 0;
	          herelast = FALSE;
		}
	      else
		herelast = TRUE;
	    break;

	    case KPARSE_NULLKEY:
	      errno = KNULL_PARAMETER;
	      kerror( "kutils", "kparse_file_search",
		 "Key parameter is NULL (this shouldn't happen)" );
	      return(KPARSE_NULLKEY);
	    /*NOTREACHED*/

	    case KPARSE_SYNTAXKEY:
	      return(KPARSE_SYNTAXKEY);
	    /*NOTREACHED*/

	    default:
	      errno = KINTERNAL;
	      kerror("kutils", "kparse_file_search",
	             "Unknown status from kparse_string_search "
                     "(this shouldn't happen)" );
	      return(KPARSE_DATAERR);
	    /*NOTREACHED*/
	  }
    } while((actual_read == KLENGTH || actual_read == 2*KLENGTH) && !done);
    return (status);
  }

/************************************************************
*
*  Routine Name: kparse_string_scan - scan a data string for a specific section
*
*       Purpose: This routine finds a section of data, marked by
*		 begin and end keys, out of a data string. When the begin
*		 and end keys are matched, this routine allocates a string big
*		 enough to hold the data between the keys, and then copies the
*		 data into that space.  A pointer to this string is then
*		 returned by this routine.  This routine makes calls
*		 to kparse_string_search() which handles the regular
*		 expression searching for the begin and end match keys.
*
*         Input: data        - the data string to search through
*		 key1        - the regular expression begin key to search for
*		 key2        - the regular expression end key to search for
*		 mode        - tells the parser which mode to work in:
*			       ! KIGNORE_CASE	Ignore case on a-z and A-Z
*			       ! KLITERAL		Use case information on
*			       ! 			a-z and A-Z
*
*        Output: key1_format - the address of a pointer to hold the returned
*                              begin key that was matched.  Sufficient space for
*                              the returned string will be allocated if you
*                              pass in a valid pointer.  Note that if this
*                              parameter is passed in as NULL, this routine
*                              will ignore it, and the string that key1 matched
*                              will not be returned.
*                key2_format - the address of a pointer to hold the returned
*                              end key that was matched.  Sufficient space for
*			       the returned string will be allocated if you
*			       pass in a valid pointer.  Note that if this
*			       parameter is passed in as NULL, this routine
*			       will ignore it, and the string that key2 matched
*			       will not be returned.
*		 status      - error status of the search.  It can be one of
*			       the following:
*			       ! KPARSE_OK		(parse ok, return data
*			       !				 valid),
*			       ! KPARSE_NOKEY		(couldn't find begin key)
*			       ! KPARSE_NOEND		(couldn't find end key)
*			       ! KPARSE_DATAERR	(data string was invalid)
*			       ! KPARSE_PARTKEY	(data ended with partial
*			       !				 match)
*			       ! KPARSE_PARTEND	(same as above, but on
*			       !				 end key)
*			       ! KPARSE_NULLKEY	(key was NULL)
*			       ! KPARSE_NULLEND	(end key was NULL)
*			       ! KPARSE_SYNTAXKEY	(key had an illegal
*			       !				 syntax)
*			       ! KPARSE_SYNTAXEND	(end key had an illegal
*			       !				 syntax)
*       Returns: This routine returns a pointer to a malloc'ed string
*		 containing the text between the two matched keys.  
*		 If an error occured during the search, it will return NULL,
*		 and the error status is set appropriately.
*
*  Restrictions: It does not support the following regular expression
*                constructs:  or'ing '|',  grouping of regular expressions '()',
*                match one or more times '+', or match n to m times '\\{n,m\\}'.
*                Finally, the '\\number' and '\\(\\)' constructs have no meaning
*                for these routines, so they are not supported either.
*    Written By: Steven Jorgensen
*          Date: Aug 06, 1992 18:38
*      Verified:
*  Side Effects: This routine mallocs data and sets the value of the key1_format
*                and key2_format parameters.  Thus, the user should pass in
*		 addresses of an unused character pointers for them.  The
*		 calling routine is responsible for freeing the space
*		 malloc'ed for the key1_format and key2_format parameters.
*
*		 This routine mallocs the space for the return string;
*		 and hence, is responsible for freeing the that space
*		 via kfree() when they are done with it.
* Modifications:
*
*************************************************************/
char *kparse_string_scan(
   char *data,
   char *key1,
   char *key2,
   int  mode,
   char **key1_format,
   char **key2_format,
   int  *status)
  {
    char *result1, *result2, *ret_str, *tmpstr;
    int  offset;

    result1 = kparse_string_search(data, key1, mode, key1_format, status);
    if (*status != KPARSE_OK)
      return(NULL);
    result2 = kparse_string_search(result1, key2, mode, &tmpstr, status);
    if (*status != KPARSE_OK)
      {
	if (key1_format != NULL)
	  kfree(*key1_format);
	*status = (*status == KPARSE_NOKEY) ? KPARSE_NOEND : *status;
	*status = (*status == KPARSE_PARTKEY) ? KPARSE_PARTEND : *status;
	*status = (*status == KPARSE_NULLKEY) ? KPARSE_NULLEND : *status;
	*status = (*status == KPARSE_SYNTAXKEY) ? KPARSE_SYNTAXEND : *status;
        return(NULL);
      }
    offset = (key2[kstrlen(key2)-1] == '$') ? 1 : 0;
    offset = (*(result2-1) == '\n') ? offset : 0;
    ret_str = kstring_ncopy(result1,
			  (result2 - result1 - kstrlen(tmpstr) - offset), NULL);
    if (key2_format != NULL)
      *key2_format = tmpstr;
    else
      kfree(tmpstr);
    return(ret_str);
  }

/************************************************************
*
*  Routine Name: kparse_file_scan - scan a Khoros Data Transport Stream for a
*				    specific section of data
*
*       Purpose: This routine finds a section of data, marked by
*		 begin and end keys, out of a Khoros Data Transport
*		 Stream that was opened for input.  When
*		 the begin and end keys are matched, this routine allocates
*		 a string big enough to hold the data between the keys, and
*		 then copies the data into that space.  A pointer to this
*		 string is then returned.  This routine makes
*		 calls to kparse_file_search() which handles the regular
*                expression parsing for the begin and end match keys.
*		 On a successful search, the current position in the
*		 Khoros Data Transport Stream will be set to the
*		 character directly following the last character matched
*		 by the end key.
*		
*
*         Input: file        - a pointer to an open file pointer
*		 key1        - the regular expression begin key to search for.
*			       If this key is the #define KPARSE_BOF, it
*			       will use a file offset of 0 for the returned
*			       data's starting point.
*		 key2        - the regular expression end key to search for.
*			       If this key is the #define KPARSE_EOF, it
*			       will set the file offset to the end of the file
*			       for the returned data's ending point.
*		 mode        - tells the parser which mode to work in:
*			       ! KIGNORE_CASE	Ignore case on a-z and A-Z
*			       ! KLITERAL		Use case information on
*			       !			a-z and A-Z
*
*        Output: key1_format - the address of a pointer to hold the returned
*			       begin key that was matched.  If key1 was
*			       KPARSE_BOF, this address will set to NULL.
*                              Sufficient space for the returned string
*                              will be allocated if you pass in a valid 
*                              pointer.  Note that if this parameter is
*                              passed in as NULL, this routine will ignore it,
*                              and the string that key1 matched will not be
*                              returned.
*		 key2_format - the address of a pointer to hold the returned
*			       end key that was matched.  If key2 was
*			       KPARSE_EOF, this address will be set to NULL.
*			       Sufficient space for the returned string
*			       will be allocated if you pass in a valid 
*                              pointer.  Note that if this parameter is
*                              passed in as NULL, this routine will ignore it,
*			       and the string that key2 matched will not be
*			       returned. 
*		 status      - error status of the search.  It can be one of
*			       the following:
*			       ! KPARSE_OK		(parse ok, return data
*			       !				 valid),
*			       ! KPARSE_NOKEY		(couldn't find begin key)
*			       ! KPARSE_NOEND		(couldn't find end key)
*			       ! KPARSE_DATAERR	(data string was invalid)
*			       ! KPARSE_PARTKEY	(data ended with partial
*			       !				 match)
*			       ! KPARSE_PARTEND	(same as above, but on
*			       ! 				 end key)
*			       ! KPARSE_NULLKEY	(key was NULL)
*			       ! KPARSE_NULLEND	(end key was NULL)
*			       ! KPARSE_SYNTAXKEY	(key had an illegal
*			       !				 syntax)
*			       ! KPARSE_SYNTAXEND	(end key had an illegal
*			       !				 syntax)
*       Returns: This routine returns a pointer to a malloc'ed string
*                containing the text between the two matched keys.
*                If an error occured during the search, it will return NULL,
*                and the error status is set appropriately.  This current
*		 file position is set to the 
*
*  Restrictions: It does not support the following regular expression
*                constructs:  or'ing '|',  grouping of regular expressions '()',
*                match one or more times '+', or match n to m times '\\{n,m\\}'.
*                Finally, the '\\number' and '\\(\\)' constructs have no meaning
*                for these routines, so they are not supported either.
*    Written By: Steven Jorgensen
*          Date: Aug 06, 1992 18:38
*      Verified:
*  Side Effects: This routine mallocs data and sets the value of the key1_format
*                and key2_format parameters.  Thus, the user should pass in
*                addresses of an unused character pointers for them.  The
*                calling routine is responsible for freeing the space
*                malloc'ed for the key1_format and key2_format parameters.
*
*                This routine mallocs the space for the return string;
*                and hence, is responsible for freeing the that space
*                via kfree() when they are done with it.
* Modifications:
*
*************************************************************/
char *kparse_file_scan(
   kfile *file,
   char  *key1,
   char  *key2,
   int   mode,
   char  **key1_format,
   char  **key2_format,
   int   *status)
  {
    char *ret_str, *tmpstr = NULL;
    int result1, result2, count;

    if (kstrcmp(KPARSE_BOF, key2) == 0)
      {
	*status = KPARSE_NOKEY;
	return(NULL);
      }
    if (kstrcmp(KPARSE_EOF, key1) == 0)
      {
	*status = KPARSE_NOEND;
	return(NULL);
      }
    if (kstrcmp(KPARSE_BOF, key1) == 0)
      {
	if (key1_format != NULL)
	  *key1_format = NULL;
	krewind(file);
      }
    else
      {
	*status = kparse_file_search(file, key1, mode, key1_format);
	if (*status != KPARSE_OK)
	  return(NULL);
      }
    result1 = kftell(file);
    if (kstrcmp(KPARSE_EOF, key2) == 0)
      {
	*status = KPARSE_OK;
	if (kfseek(file, 0, 2) == -1)
	  {
	    *status = KPARSE_NOEND;
	    return(NULL);
	  }
      }
    else
      {
	*status = kparse_file_search(file, key2, mode, &tmpstr);
	if (*status != KPARSE_OK)
	  {
	    if (key1_format != NULL)
	      kfree(*key1_format);
	    *status = (*status == KPARSE_NOKEY) ? KPARSE_NOEND : *status;
	    *status = (*status == KPARSE_PARTKEY) ? KPARSE_PARTEND : *status;
	    *status = (*status == KPARSE_NULLKEY) ? KPARSE_NULLEND : *status;
	    *status = (*status == KPARSE_SYNTAXKEY) ? KPARSE_SYNTAXEND :*status;
            return(NULL);
          }
      }
    result2 = kftell(file);
    if (key2[kstrlen(key2)-1] == '$')
      result2--;
    if (kfseek(file, result1, 0) == -1)
      {
	kfree(tmpstr);
	*status = KPARSE_DATAERR;
	return(NULL);
      }
    if ((ret_str=kmalloc(result2 - result1 - kstrlen(tmpstr)+1)) == NULL)
      {
	kfree(tmpstr);
	kerror("kutils", "kparse_file_scan", "Couldn't malloc result");
	*status = KPARSE_DATAERR;
	return(NULL);
      }
    ret_str[result2 - result1 - kstrlen(tmpstr)] = '\0';
    count = kfread(ret_str, 1, result2 - result1 - kstrlen(tmpstr), file);
    if (count < result2 - result1 - kstrlen(tmpstr))
      {
	kfree(tmpstr);
	errno = KINTERNAL;
	kerror("kutils", "kparse_file_scan",
	       "Didn't read enough input (this shouldn't happen)");
	*status = KPARSE_DATAERR;
	return(NULL);
      }
    if (kfseek(file, result2, 0) == -1)
      {
	kfree(tmpstr);
	*status = KPARSE_DATAERR;
	return(NULL);
      }
    if (key2_format != NULL )
      *key2_format = tmpstr;
    else
      kfree(tmpstr);
    return(ret_str);
  }

/************************************************************
*
*  Routine Name: kparse_string_delimit - break a string into an array of strings
*					 based on some set of delimiters.
*
*       Purpose: This routine parses the input data string according to a set
*		 of single character delimiters.  These delimiters can be
*		 escaped by a '\\' inside of the data string if the delimiter
*		 must appear as an item; hence, you cannot use a '\\' as a
*		 delimiter.  The new array is created via calls to
*		 the karray_add() library call, and a pointer to the new
*		 array is returned back to the calling routine
*
*         Input: data      - the data string to delimit
*		 delimiter - the list of delimiters to use
*		 mode      - the delimit mode
*			     ! KLITERAL     - delimit on delimiters, leave
*                            !                extra whitespace, and leave
*                            !                NULL entries in the array
*                            !                if two delimiters are next
*                            !                to each other.
*			     ! KDELIM_CLEAN - eliminate whitespace on
*			     !                strings, and ignore two
*			     !                delimiters next to each other
*
*        Output: array_cnt - the number of items in the list
*
*       Returns: This routine returns a pointer to an array of items we just
*		 broke apart from the input data string.
*
*  Restrictions: 
*    Written By: Steven Jorgensen
*          Date: Nov 11, 1992 14:10
*      Verified:
*  Side Effects: This routine creates a new array of strings, and the
*		 calling routine is responsible for freeing the space
*		 allocated while creating the array via a call to
*		 karray_free().
* Modifications:
*
*************************************************************/
char **kparse_string_delimit(
   char *data,
   char *delimiter,
   int   mode,
   int  *array_cnt)
  {
    char *d, **ret;

    *array_cnt = 0;
    if (data == NULL)
      {
	kinfo(KSYSLIB, "kparse_string_delimit: Data string is NULL");
	return(NULL);
      }
    if (delimiter == NULL)
      {
	kinfo(KSYSLIB, "kparse_string_delimit: Delimiter string is NULL");
	return(NULL);
      }
    d = kstrdup(data);
    ret = _delimit_str(d, delimiter, mode, array_cnt);
    kfree(d);
    return(ret);
  }
/************************************************************
*
*  Routine Name: kparse_string_search_delimit - break up a line data into
*						an array of strings based on
*						some set of delimiters
*
*       Purpose: This routine looks for a user specified key with a
*		 call to kparse_string_search(); then, it
*		 parses the rest of the line, up to a '\\n', according to the
*		 user's list of delimiters by calling kparse_string_delimit().
*		 The user can specify a set of continuation characters.
*		 Continuation characters must appear as the last character
*		 of the line.  Line continuation can be chained together
*		 at the end of each consecutive line you want this routine
*		 to parse as a single line.
*
*         Input: data       - input string to search and delimit.
*		 key        - regular expression key to search for.
*		 mode	    - tells the parser which mode to work in:
*                             ! KIGNORE_CASE	Ignore case on a-z and A-Z
*                             ! KLITERAL		Use case information on
*			      ! 			a-z and A-Z
*		 delimiters - a string containing the delimiter characters.
*		 cont       - a string containing the continuation characters.
*
*        Output: key_format - the address of a pointer to hold the returned
*                             key that was matched.  Sufficient space for
*                             the returned string will be allocated if you
*                             pass in a valid pointer.  Note that if this
*                             parameter is passed in as NULL, this routine
*                             will ignore it, and the string that key matched
*                             will not be returned.
*		 array_cnt  - returns the number of items in the array, -1
*			      on error
*                status     - error status of the search.  It can be
*                             one of the following:
*                           ! KPARSE_OK		(parse ok, return data
*                           !				 valid)
*                           ! KPARSE_NOKEY		(couldn't find key)
*                           ! KPARSE_PARTKEY	(data ends on a partial
*                           !				 match)
*                           ! KPARSE_NULLKEY	(key was NULL)
*                           ! KPARSE_SYNTAXKEY	(key had an illegal syntax)
*
*       Returns: This routine returns a pointer to an array of items we just
*                broke apart from the input data string.  NULL is returned
*		 when the kparse_string_search() or kparse_string_delimit()
*		 fails.
*
*  Restrictions: It does not support the following regular expression
*                constructs:  or'ing '|',  grouping of regular expressions '()',
*                match one or more times '+', or match n to m times '\\{n,m\\}'.
*                Finally, the '\\number' and '\\(\\)' constructs have no meaning
*                for these routines, so they are not supported either.
*
*                Search keys and data strings should not contain the
*		 values '\\001', '\\002', '\\003', or '\\004',
*		 because these values are used as special search parameters
*		 by the parser.
*    Written By: Steven Jorgensen
*          Date: Sep 10, 1992 17:48
*      Verified:
*  Side Effects: This routine mallocs data and sets the value of key_format
*                parameter.  Thus, the user should pass in an address of
*                an unused character pointer.  The calling routine is
*                responsible for freeing the space malloc'ed for
*                the key_format parameter.
*
*                This routine creates a new array of strings, and the
*		 calling routine is responsible for freeing the space
*		 allocated while creating the array via a call to
*		 karray_free().
* Modifications:
*
*************************************************************/
char **kparse_string_search_delimit(
   char *data,
   char *key,
   int  mode,
   char *delimiters,
   char *cont,
   char **key_format,
   int  *array_cnt,
   int  *status)
  {
    char *result, *cr, **ret_str;
    int len, i,j, done = FALSE;

    result = kparse_string_search(data, key, mode, key_format, status);
    if (*status != KPARSE_OK)
      return(NULL);
    if ((cr = kstrchr(result, '\n')) != NULL)
      if (kstrchr(cont, *(cr-1)) != NULL)
	{
	  while ( done == FALSE )
	    if ((cr = kstrchr(cr + 1 , '\n')) == NULL)
	      done = TRUE;
	    else if (kstrchr(cont, *(cr-1)) == NULL)
	      done = TRUE;
	}
    len = (cr == NULL) ? kstrlen(result) : cr - result;
    if ((cr = kmalloc(len * sizeof(char)+1)) == NULL)
      {
	kerror("kutils", "kparse_string_search_delimit", NULL);
	*status = KPARSE_DATAERR;
	return(NULL);
      }
    for (i = 0, j = 0; i < len; i++)
      {
	if (result[i + 1] != '\n' || kstrchr(cont, result[i]) == NULL)
	  cr[j++] = result[i];
      }
    cr[j] = '\0';
    ret_str = _delimit_str(cr, delimiters, mode, array_cnt);
    kfree(cr);
    return(ret_str);
  }

/************************************************************
*
*  Routine Name: kparse_file_search_delimit - break up a line data into
*                                             an array of strings based on
*                                             some set of delimiters
*
*       Purpose: This routine looks for a user specified key with a
*                call to kparse_file_search(); then, it reads in the
*                rest of the line into a string, up to a '\\n'.  Once the
*		 line is read in, it breaks up the line according to the
*                user's list of delimiters by calling kparse_string_delimit().
*                The user can specify a set of continuation characters.
*                Continuation characters must appear as the last character
*                of the line.  Line continuation can be chained together
*                at the end of each consecutive line you want this routine
*                to parse as a single line.
*
*
*         Input: file       - a pointer to the transport file opened for reading
*                key        - The regular expression key to search for.
*                mode       - tells the parser which mode to work in:
*                             ! KIGNORE_CASE	Ignore case on a-z and A-Z
*                             ! KLITERAL		Use case information on
*                             ! 			a-z and A-Z
*		 delimiters - a string containing the delimiter characters.
*		 cont       - a string containing the continuation characters.
*
*        Output: key_format - the address of a pointer to hold the returned
*                             key that was matched.  Sufficient space for
*                             the returned string will be allocated if you
*                             pass in a valid pointer.  Note that if this
*                             parameter is passed in as NULL, this routine
*                             will ignore it, and the string that key matched
*                             will not be returned.
*		 array_cnt  - returns the number of items in the array, -1
*			      on error
*                status     - return status of the search.  It can be
*                             one of the following:
*                             KPARSE_OK        (parse ok, return data valid),
*                             KPARSE_NOKEY     (couldn't find key)
*                             KPARSE_PARTKEY   (data ended with a partial match)
*                             KPARSE_NULLKEY   (key was NULL)
*                             KPARSE_SYNTAXKEY (key had an illegal syntax)
*                             KPARSE_DATAERR   (routine had a a data error)
*
*       Returns: This routine returns a pointer to an array of items we just
*                broke apart from the input data string.  NULL is returned
*		 when the kparse_file_search() or kparse_string_delimit()
*		 fails.
*
*  Restrictions: It does not support the following regular expression
*                constructs:  or'ing '|',  grouping of regular expressions '()',
*                match one or more times '+', or match n to m times '\\{n,m\\}'.
*                Finally, the '\\number' and '\\(\\)' constructs have no meaning
*                for these routines, so they are not supported either.
*
*                Search keys and the data file should not contain the
*		 values '\\001', '\\002', '\\003', or '\\004',
*		 because these values are used as special search parameters
*		 by the parser.
*    Written By: Steven Jorgensen
*          Date: Sep 14, 1992 17:05
*      Verified:
*  Side Effects: This routine mallocs data and sets the value of key_format
*                parameter.  Thus, the user should pass in an address of
*                an unused character pointer.  The calling routine is
*                responsible for freeing the space malloc'ed for
*                the key_format parameter.
*
*                This routine creates a new array of strings, and the
*                calling routine is responsible for freeing the space
*                allocated while creating the array via a call to
*                karray_free().
* Modifications:
*
*************************************************************/
char **kparse_file_search_delimit(
   kfile *file,
   char  *key,
   int   mode,
   char  *delimiters,
   char  *cont,
   char  **key_format,
   int   *array_cnt,
   int   *status)
  {
    char *result, *cr, **ret_str, last, cur;
    int len, i,j;

    *array_cnt = 0;
    *status = kparse_file_search(file, key, mode, key_format);
    if (*status != KPARSE_OK)
      return(NULL);
    last = '\0';
    i = kftell(file);
    while ((cur = kfgetc(file)) != EOF)
      {
	if (cur == '\n')
	  if (last == '\0' || kstrchr(cont, last) == NULL)
	    break;
	last = cur;
      }
    j = kftell(file);
    len = j - i + 1;
    if (len < 2)
      return(NULL);
    if ((cr = kmalloc((len * sizeof(char))+1)) == NULL)
      {
	kerror("kutils", "kparse_file_search_delimit",NULL);
	*status = KPARSE_DATAERR;
	return(NULL);
      }
    if ((result = kmalloc(len * sizeof(char))) == NULL)
      {
	kerror("kutils", "kparse_file_search_delimit",NULL);
	*status = KPARSE_DATAERR;
	return(NULL);
      }
    kfseek(file, i, 0);
    if (kfread(result, sizeof(char), len, file) == -1)
      {
	kerror("kutils", "kparse_file_search_delimit", "Fread error");
	return(NULL);
      }
    result[j-i-1] = '\0';
    if (j != kftell(file))
      kinfo(KDEBUG, "Bug, read didn't end at correct place\n");
    for (i = 0, j = 0; i < len; i++)
      {
	if (i == len - 1 ||
	    result[i + 1] != '\n' ||
	    kstrchr(cont, result[i]) == NULL)
	  cr[j++] = result[i];
      }
    cr[j] = '\0';
    ret_str = _delimit_str(cr, delimiters, mode, array_cnt);
    kfree(cr);
    kfree(result);
    return(ret_str);
  }

/************************************************************
*
*  Routine Name: kparse_string_scan_delimit - Break a string into an array of
*					      strings
*
*       Purpose: This routine looks in a data string for an area of text
*		 between two user specified match keys, then it delimits
*		 the section of text in to an array of smaller strings
*		 based on a set of character delimiters.  Delimiters
*		 can be escaped by a '\\' if they need to appear in the
*		 text section.  This routine is a combination of the
*		 calls kparse_string_scan() and kparse_string_delimit().
*		 This routine cleans up the entries in the array via
*		 a kstring_cleanup() call.
*
*         Input: data        - the data string to search through
*		 key1        - the regular expression begin key to search for
*		 key2        - the regular expression end key to search for
*		 mode        - tells the parser which mode to work in:
*			       KIGNORE_CASE (match reguardless of case),
*			       KLITERAL (match with case).
*		 delimiters  - a string containing the delimiter characters.
*
*        Output: key1_format - the address of a pointer to hold the returned
*			       begin key that was matched.  If key1 was
*			       KPARSE_BOF, this address will set to NULL.
*                              Sufficient space for the returned string
*                              will be allocated if you pass in a valid 
*                              pointer.  Note that if this parameter is
*                              passed in as NULL, this routine will ignore it,
*                              and the string that key1 matched will not be
*                              returned.
*		 key2_format - the address of a pointer to hold the returned
*			       end key that was matched.  If key2 was
*			       KPARSE_EOF, this address will be set to NULL.
*			       Sufficient space for the returned string
*			       will be allocated if you pass in a valid 
*                              pointer.  Note that if this parameter is
*                              passed in as NULL, this routine will ignore it,
*			       and the string that key2 matched will not be
*			       returned. 
*		 array_cnt   - returns the number of items in the array, -1
*			       on error
*		 status      - error status of the search.  It can be one of
*			       the following:
*			       ! KPARSE_OK		(parse ok, return data
*			       !				 valid),
*			       ! KPARSE_NOKEY		(couldn't find begin key)
*			       ! KPARSE_NOEND		(couldn't find end key)
*			       ! KPARSE_DATAERR	(data string was invalid)
*			       ! KPARSE_PARTKEY	(data ended with partial
*			       !				 match)
*			       ! KPARSE_PARTEND	(same as above, but on
*			       !				 end key)
*			       ! KPARSE_NULLKEY	(key was NULL)
*			       ! KPARSE_NULLEND	(end key was NULL)
*			       ! KPARSE_SYNTAXKEY	(key had an illegal
*			       !				 syntax)
*			       ! KPARSE_SYNTAXEND	(end key had an illegal
*			       !				 syntax)
*
*       Returns: This routine returns a pointer to an array of items that were
*		 just broke apart from the input data string.  NULL is returned
*		 when the kparse_string_scan() or kparse_string_delimit()
*		 fails.
*
*  Restrictions: It does not support the following regular expression
*                constructs:  or'ing '|',  grouping of regular expressions '()',
*                match one or more times '+', or match n to m times '\\{n,m\\}'.
*                Finally, the '\\number' and '\\(\\)' constructs have no meaning
*                for these routines, so they are not supported either.
*
*                Search keys and the data file should not contain the
*		 values '\\001', '\\002', '\\003', or '\\004',
*		 because these values are used as special search parameters
*		 by the parser.
*    Written By: Steven Jorgensen
*          Date: Sep 28, 1992 15:58
*      Verified:
*  Side Effects: This routine mallocs data and sets the value of the key1_format
*                and key2_format parameters.  Thus, the user should pass in
*		 addresses of an unused character pointers for them.  The
*		 calling routine is responsible for freeing the space
*		 malloc'ed for the key1_format and key2_format parameters.
*
*		 This routine mallocs the space for the return string;
*		 and hence, is responsible for freeing the that space
*		 via kfree() when they are done with it.
*
*                This routine creates a new array of strings, and the
*                calling routine is responsible for freeing the space
*                allocated while creating the array via a call to
*                karray_free().
* Modifications:
*
*************************************************************/
char **kparse_string_scan_delimit(
   char *data,
   char *key1,
   char *key2,
   int  mode,
   char *delimiters,
   char **key1_format,
   char **key2_format,
   int  *array_cnt,
   int  *status)
  {
    char **ret_array;
    char *info;

    info = kparse_string_scan(data, key1, key2, mode, key1_format, key2_format,
			      status);
    if (*status != KPARSE_OK)
      {
	*array_cnt = -1;
	return(NULL);
      }
    ret_array = _delimit_str(info, delimiters, mode, array_cnt);
    kfree(info);
    return(ret_array);
  }

/************************************************************
*
*  Routine Name: kparse_file_scan_delimit - break a section of a Khoros
*					    Data Transport Stream into
*					    an array of strings
*
*       Purpose: This routine looks for an area of text between two user
*		 specified keys in a Khoros Data Transport Stream; then,
*		 copies the section into a string, and parses it 
*		 according to the user's set of delimiters.  Delimiters
*		 can be escaped by a '\\' character if a delimiter must
*		 appear in the text.  This routine is a combination of
*		 the routines kparse_file_scan() and kparse_string_delimit().
*		 This routine cleans up the strings to be returned via the
*		 kstring_cleanup() routine.
*
*         Input: file        - a pointer to an open file pointer
*		 key1        - the regular expression begin key to search for
*			       if this key is the #define KPARSE_BOF, it
*			       will make file offset 0 the returned data
*			       starting point.
*		 key2        - the regular expression end key to search for
*			       if this key is the #define KPARSE_EOF, it
*			       will make file offset at the end of the file
*			       the returned data ending point.
*		 mode        - tells the parser which mode to work in:
*			       ! KIGNORE_CASE	Ignore case on a-z and A-Z
*			       ! KLITERAL		Use case information on
*			       ! 			a-z and A-Z
*		 delimiters  - a string containing the delimiter characters.
*
*        Output: key1_format - the address of a pointer to hold the returned
*                              begin key that was matched.  If key1 was
*                              KPARSE_BOF, this address will set to NULL.
*                              Sufficient space for the returned string
*                              will be allocated if you pass in a valid
*                              pointer.  Note that if this parameter is
*                              passed in as NULL, this routine will ignore it,
*                              and the string that key1 matched will not be
*                              returned.
*                key2_format - the address of a pointer to hold the returned
*                              end key that was matched.  If key2 was
*                              KPARSE_EOF, this address will be set to NULL.
*                              Sufficient space for the returned string
*                              will be allocated if you pass in a valid
*                              pointer.  Note that if this parameter is
*                              passed in as NULL, this routine will ignore it,
*                              and the string that key2 matched will not be
*                              returned.
*		 array_cnt   - returns the number of items in the array, -1
*			       on error
*		 status      - error status of the search.  It can be one of
*			       the following:
*			       ! KPARSE_OK		(parse ok, return data
*			       !				 valid),
*			       ! KPARSE_NOKEY		(couldn't find begin key)
*			       ! KPARSE_NOEND		(couldn't find end key)
*			       ! KPARSE_DATAERR	(data string was invalid)
*			       ! KPARSE_PARTKEY	(data ended with partial
*			       !				 match)
*			       ! KPARSE_PARTEND	(same as above, but on
*			       !				 end key)
*			       ! KPARSE_NULLKEY	(key was NULL)
*			       ! KPARSE_NULLEND	(end key was NULL)
*			       ! KPARSE_SYNTAXKEY	(key had an illegal
*			       !				 syntax)
*			       ! KPARSE_SYNTAXEND	(end key had an illegal
*			       !				 syntax)
*
*       Returns: an array of of strings that were delimited.
*
*  Restrictions: 
*    Written By: Steven Jorgensen
*          Date: Nov 11, 1992 15:47
*      Verified:
*  Side Effects: This routine mallocs data and sets the value of the key1_format
*                and key2_format parameters.  Thus, the user should pass in
*		 addresses of an unused character pointers for them.  The
*		 calling routine is responsible for freeing the space
*		 malloc'ed for the key1_format and key2_format parameters.
*
*		 This routine mallocs the space for the return string;
*		 and hence, is responsible for freeing the that space
*		 via kfree() when they are done with it.
*
*                This routine creates a new array of strings, and the
*                calling routine is responsible for freeing the space
*                allocated while creating the array via a call to
*                karray_free().
* Modifications:
*
*************************************************************/
char **kparse_file_scan_delimit(
   kfile *file,
   char  *key1,
   char  *key2,
   int   mode,
   char  *delimiters,
   char  **key1_format,
   char  **key2_format,
   int   *array_cnt,
   int   *status)
  {
    char **ret_array;
    char *info;

    info = kparse_file_scan(file, key1, key2, mode, key1_format, key2_format,
			    status);
    if (*status != KPARSE_OK)
      {
	*array_cnt = -1;
	return(NULL);
      }
    ret_array = _delimit_str(info, delimiters, mode, array_cnt);
    kfree(info);
    return(ret_array);
  }
