/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Memory Allocation Routines
   >>>>
   >>>>  Private:
   >>>>             _kfree()
   >>>>   Static:
   >>>>   Public:
   >>>>             kmalloc()
   >>>>             kcalloc()
   >>>>             krealloc()
   >>>>             kdupalloc()
   >>>>
   >>>>		    kmemcpy()
   >>>>		    kmemccpy()
   >>>>		    kmemcmp()
   >>>>		    kmemset()
   >>>>		    kmemmove()
   >>>>		    kmemchr()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


/*-----------------------------------------------------------
|
|  Routine Name: _kfree - free allocated memory
|
|       Purpose: This routine calls free() to free previously allocated data.
|		 However, it also checks to make sure the pointer argument
|		 is not null.
|
|         Input: ptr - A pointer to memory to be freed.
|        Output: 
|       Returns: nothing (void)
|
|    Written By: Nick Ruprecht & Steven Jorgensen
|          Date: Jul 11, 1993
| Modifications: Changed header to reflect void return instead of
|		 TRUE and FALSE. SRW 20-July-94
|
------------------------------------------------------------*/
void _kfree(
   kaddr ptr)
{
	if (ptr) free(ptr);
}

/************************************************************
*
*  Routine Name: kmalloc - allocate a contiguous piece of memory
*
*       Purpose: Calls malloc() to allocate the requested data.  However,
*		 it checks to make sure the size parameter is greater
*		 then zero.  If the size paramter is 0, it returns NULL.
*
*         Input: size - number of bytes to allocate
*       Returns: a pointer to malloc'ed data on success, or NULL on size
*		 0 or malloc failure
*    Written By: Mark Young
*          Date: Jul 11, 1993
* Modifications: Converted from kmalloc in Khoros 1.0 (SJ)
*************************************************************/

kaddr kmalloc(
   unsigned size)
{
	return(size == 0 ? NULL : malloc(size));
}


/************************************************************
*
*  Routine Name: kcalloc - allocate memory and initialize it
*
*       Purpose: This macro calls calloc() to allocate the requested data.
*		 However, it checks nelem and elsize to make sure both
*		 are greater than 0.  If either parameter is 0, then NULL
*		 is returned.
*
*         Input: nelem  - number of elements to calloc
*		 elsize - number of bytes in an element
*
*        Output: none
*       Returns: The address of the allocated memory.  NULL is returned
*		 if the allocation fails, or if either nelem or elsize
*		 are 0
*    Written By: Mark Young
*          Date: Jul 11, 1993
* Modifications: Converted from kcalloc in Khoros 1.0 (SJ)
*************************************************************/

kaddr kcalloc(
   unsigned nelem,
   unsigned elsize)
{
	return(nelem == 0 || elsize == 0 ? NULL : calloc(nelem, elsize));
}

/************************************************************
*
*  Routine Name: krealloc - re-allocate a piece of memory to a new size
*       Purpose: The macro calls realloc() to re-allocate the block of
*		 memory with size bytes.  The realloc will automatically
*		 copy the data from the first memory block to the newly
*		 allocated one.
*         Input: ptr  - the pointer to reallocate
*                size - the number of characters to reallocate to.
*       Returns: The address of the allocated memory.  NULL is returned
*                if the allocation fails, or if size is 0.  If ptr is
*		 NULL, malloc will be called instead of realloc.
*    Written By: Mark Young
*          Date: Jul 11, 1993
*************************************************************/

kaddr krealloc(
   kaddr    ptr,
   unsigned size)
{
	return(size == 0 ? NULL : (!ptr ? malloc(size) : realloc(ptr, size)));
}

/************************************************************
*
*  Routine Name: kdupalloc - duplicates a piece of memory
*
*       Purpose: The routine allocates new memory and then copies the contents
*		 from the ptr to the new memory.  This routine is really a
*		 convience routine that uses kmalloc() and kmemcpy() in which
*		 to make a duplicate. 
*
*         Input: src  - the source pointer to duplicate
*                size - the number of characters to duplicate from the source.
*       Returns: The newly duplicated memory or NULL upon failure.
*    Written By: Mark Young
*          Date: Aug 14, 1994
*************************************************************/

kaddr kdupalloc(
   kaddr    src,
   unsigned size)
{
	kaddr dest;

	if (src == NULL || (dest = kmalloc(size)) == NULL)
	   return(NULL);

	if (kmemcpy(dest, src, size) == NULL)
	   kfree(dest);

	return(dest);
}

/************************************************************
*
*  Routine Name: kmemcpy - copies bytes from src to dest
*
*       Purpose: This routine copies num bytes of memory from src
*		 to dest in a similar manner as memcpy.  However,
*		 this routine checks that the src, dest, and num values
*		 are not NULL or 0.
*         Input: src  - the source pointer to copy "num" bytes from
*		 num  - the number of bytes to be copied
*        Output: dest - the destination pointer to copy the bytes to
*       Returns: dest on a successful memory copy.  If dest or src are
*		 NULL, NULL is returned.  If num is 0, NULL is returned.
*    Written By: Mark Young
*          Date: Jul 06, 1993
*************************************************************/

kaddr kmemcpy(
   kaddr    dest,
   kaddr    src,
   unsigned num)
{
	return((dest && src && num > 0) ?
		(kaddr) memcpy(dest, src, num) : NULL);
}


/************************************************************
*
*  Routine Name: kmemccpy - restricted copy of  bytes from src to dest
*
*       Purpose: This function is the same as the system call memccpy().
*	         Except kmemccpy() will make sure that the src and dest
*		 pointers are not NULL before calling memccpy().
*
*	         memccpy() copies the bytes in the "src" character
*		 array to the "dest" array.  The number of bytes
*		 to be copied is determined by the "c" character value
*		 and the parameter "num".  Bytes will be copied from "src"
*		 to "dest" until the "c" character value is encountered or
*		 num bytes have been copied.  If "c" is encountered and
*		 copied before "num" bytes then the "dest" array is returned.
*		 Otherwise if "num" bytes are copied then NULL is returned
*		 to indicate the "c" was not encountered.
*
*         Input: src  - the source pointer to copy "num" bytes from
*		 c    - the character value to terminate copy
*		 num  - the number of bytes to be copied
*        Output: dest - the destination pointer to copy the bytes to
*       Returns: returns dest if "c" was encountered, NULL otherwise
*    Written By: Mark Young
*          Date: Jul 06, 1993
*************************************************************/

kaddr kmemccpy(
   kaddr dest,
   kaddr src,
   int   c,
   unsigned num)
{
	return((dest && src && num > 0) ?
		(kaddr) memccpy(dest, src, c, num) : NULL);
}


/************************************************************
*  Routine Name: kmemcmp - compare bytes from src1 and src2
*       Purpose: This function is the same as the system call memcmp().
*	         Except kmemcmp() will make sure that the src1 and src2
*		 pointers are not NULL before calling memcmp().
*
*	         memcmp() compares its arguments, looking at the first
*		 num bytes, and returns an integer less than, equal to,
*		 or greater than 0, according as src1 is lexicographically
*		 less than, equal to, or greater than src2.
*         Input: src1 - the first source pointer to compare from
*		 src2 - the second source pointer to compare from
*		 num  - the number of bytes to be compared
*       Returns:  0 if up to num characters in src1 equals the corresponding
*		  characters in src2.  The positive difference of the
*		  first character that differs between src1 and src2 if
*		  src1 is lexicographically greater than src2.  The negative
*		  difference of the first character that differs, if src2 is
*		  lexicographically greater than src1.
*    Written By: Mark Young
*          Date: Jul 06, 1993
*************************************************************/

int kmemcmp(
   kaddr    src1, 
   kaddr    src2,
   unsigned num)
{
	return((src1 && src2) ? (int) memcmp(src1, src2, num) :
	       (!src1 && !src2) ? 0 : (!src1) ? -1 : 1);
}


/************************************************************
*
*  Routine Name: kmemset - initialize bytes in dest to the character
*			   value 'c'
*       Purpose: This function is the same as the system call memset().
*	         Except kmemset() will make sure that the dest and the
*		 the number of bytes to be initialized are greater than
*		 0.
*
*	         memset() initializes the bytes in the "dest" character
*		 array to the character value "c".  The number of bytes
*		 to be set is determined by the parameter "num".
*
*         Input: c    - the character value to set in dest
*		 num  - the number of bytes to be initialized
*	 Output: dest - the destination pointer to be initialized
*       Returns: returns dest on success, NULL upon failure
*    Written By: Mark Young
*          Date: Jul 06, 1993
*************************************************************/

kaddr kmemset(
   kaddr    dest, 
   int      c,
   unsigned num)
{
	return((dest && num > 0) ? (kaddr) memset(dest, c, num) : NULL);
}

#if defined(__sun__) && defined(__GNUC__) && !defined(SVR4)
void *memmove(void *s1, void *s2, int n)
{
	bcopy(s2,s1,n);
	return(s1);
}
#endif

/************************************************************
*
*  Routine Name: kmemmove - copy a block of memory to another block
*
*       Purpose: This function is the same as the system call memmove().
*	         Except kmemmove() will make sure that the source and the
*		 destination are not NULL.
*
*		 memmove() copies n bytes from memory areas src to dest.
*		 Copying  between objects that overlap will take place
*		 correctly.  It returns dest.
*         Input: src  - the source pointer to move the data from
*		 num  - the number of bytes to be moved
*        Output: dest - the destination pointer to move the data to
*       Returns: returns the dest pointer on success, NULL otherwise
*    Written By: Mark Young
*          Date: Jul 06, 1993
*************************************************************/

kaddr kmemmove(
   kaddr    src, 
   kaddr    dest, 
   unsigned num)
{
	return((src && dest && num > 0) ? (kaddr) memmove(src,dest,num) : NULL);
}

/************************************************************
*
*  Routine Name: kmemchr - find the first occurence of 'c' in an character array
*       Purpose: This function is the same as the system call memchr().
*	         Except kmemchr() will make sure that the source and the
*		 the number of bytes to be searched are greater than 0.
*
*	         memchr() searches the bytes in the "src" character
*		 array for the character value "c".  The number of bytes
*		 to be searched is determined by the parameter "num".
*		 Upon loction of "c" in the source array, the pointer pointing
*		 to "c" within the source array is returned.
*		 If "c" does not occur within the first "num" bytes then
*		 NULL is returned.
*         Input: src - the source pointer to be searched
*		 c   - the character value to be searched for within src
*		 num - the number of bytes to be searched
*       Returns: pointer in which "c" occurs, NULL upon failure
*    Written By: Mark Young
*          Date: Jul 06, 1993
*************************************************************/

kaddr kmemchr(
   kaddr    src, 
   int      c,
   unsigned num)
{
	return((src && num > 0) ? (kaddr) memchr(src, c, num) : NULL);
}
