/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>         General Khoros STDIO Transport Routines
   >>>>
   >>>>   Static:
   >>>>			kdoprint()
   >>>>  Private:
   >>>>
   >>>>   Public:
   >>>>			 kscanf()
   >>>>			 kfscanf()
   >>>>			 ksscanf()
   >>>>			 kvscanf()
   >>>>			 kvsscanf()
   >>>>			 kvfscanf()
   >>>>
   >>>>			 kprintf()
   >>>>			 kfprintf()
   >>>>			 ksprintf()
   >>>>			 kvprintf()
   >>>>			 kvsprintf()
   >>>>			 kvfprintf()
   >>>>
   >>>>			 kfopen()
   >>>>			 kfdopen()
   >>>>			 kfreopen()
   >>>>			 ktmpfile()
   >>>>			 kfclose()
   >>>>			 kfread()
   >>>>			 kfwrite()
   >>>>			 kfputc()
   >>>>			 kfgetc()
   >>>>			 kungetc()
   >>>>			 kfputs()
   >>>>			 kfgets()
   >>>>			 kfseek()
   >>>>			 kftell()
   >>>>			 kfdup()
   >>>>			 kfdup2()
   >>>>			 krewind()
   >>>>			 kfflush()
   >>>>			 kfeof()
   >>>>			 kclearerr()
   >>>>			 ksetbuffer()
   >>>>			 kgetbuffer()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


/*-----------------------------------------------------------
|
|  Routine Name: kdoprint - 
|
|       Purpose: This is a bogus version of doprint
|		 
|
|         Input: file   - kfile structure
|		 string - if the kfile structure is NULL, then use the string
|		 format - the scanner format
|		 args   - the variable arguments
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young and Jeremy Worley
|          Date: Oct 15, 1992
| Modifications:
|
------------------------------------------------------------*/

static int kdoprint(
   kfile    *file,
   char     *string,
   char     *format,
   kva_list args)
{
	int     num = 0;
	static  char *temp = NULL;


	if (file != NULL)
	{
	   if (!temp) temp = kmalloc(KLENGTH*1000);

	   /*
	    *  Call vsprintf() to do the actual formating of the arguments
	    *  into the temporary character array.  If the number of arguments
	    *  printed does not -1 then write the string to the transport.  The
	    *  return value of kfprintf() represents
            *  the number of values actually written to the transport device.
	    */
	   if (vsprintf((char *) temp, format, args) == 0)
	      return(EOF);

	   /*
	    *  Write the printed string to our output transport
	    */
	   num = kstrlen(temp);
           if (kfwrite(temp, num, 1, file) != 1)
	       return(EOF);

	   /*
	    *  Flush stdout and kstderr, since these descriptors are never
	    *  buffered
	    */
	   if (file == kstdout || file == kstderr)
	      kfflush(file);
	}
	else
	{
	   if (vsprintf((char *) string, format, args) != 0)
	      num = kstrlen(string);
	}
	kva_end(args);
	return(num);
}


/************************************************************
*
*  Routine Name: kscanf - scan kstdin and format the input into one or
*                         more arguments of the type specified
*       Purpose: This function is a replacement for the system "scanf"
*                call;  however, kscanf() uses data transport mechanisms
*                to scan the input, in order to support distributed computing.
*
*                The kscanf() function reads characters from kstdin,
*                interprets them according to the format specified, and stores
*                the result(s) in the input argument(s) specified by
*		 a variable argument list.
*                In addition, the kscanf() routine guarantees that the
*                integer returned will be the number of arguments correctly
*                scanned, and that scanning will never continue past the
*                first error.  If there are insufficient arguments for the
*                format, the behavior is undefined;  if the format is exhausted
*                while arguments remain, the excess arguments are ignored.
*         Input: format  - the format in which to interpret the input
*        Output: kvalist - variable number of arguments into which to format
*                          the input coming from kstdin.   The format string
*                          determines the type of pointer(s) to be provided
*                          as the argument(s).
*                          Note that each argument must be a pointer, and that 
*                          sufficient space to accomodate the output must be 
*                          provided for strings.
*       Returns: The number of arguments successfully scanned into the
*                input argument(s).
*    Written By: Mark Young
*          Date: Oct 15, 1992
*************************************************************/

int kscanf(
   char *format,
   kvalist)
{
	int num = 0;
	kva_list args;


	/*
	 *  Attach the "args" to the variable argument list.
	 */
	kva_start(args, format);

	/*
	 *  Call kdoscan() to do the actual scanning of the arguments.
	 *  The variable arguments are filled and the number of characters
	 *  written
	 */
	num = kdoscan(kstdin, NULL, format, args);

	kva_end(args);
	return(num);
}

/************************************************************
*
*  Routine Name: kfscanf - scan file input and format it into one 
*                          or more arguments of the type specified
*
*       Purpose: This function is a replacement for the system "fscanf"
*                call;  however, kfscanf() uses data transport mechanisms
*                to scan the input, in order to support distributed computing.
*
*                The kfscanf() function reads characters from an input
*		 transport, interprets them according to the format specified,
*		 and stores the result in the input argument(s) specified
*		 by the variable argument list.
*                In addition, the kfscanf() routine guarantees that the 
*                integer returned will be the number of fields correctly 
*                scanned, and that scanning will never continue past the 
*                first error.  If there are insufficient arguments for the
*                format, the behavior is undefined;  if the format is exhausted
*                while arguments remain, the excess arguments are ignored.
*
*         Input: file    - stream to open input file
*		 format  - the format in which to interpret the input
*        Output: kvalist - variable number of arguments into which to format
*                          the input from the file stream.  The format string
*                          determines the type of pointer(s) to be provided
*                          as the argument(s).  Note that each argument must
*                          be a pointer, and that sufficient space to accomodate
*                          the output must be provided for strings.
*       Returns: The number of arguments successfully scanned into the
*                input argument(s).
*    Written By: Mark Young
*          Date: Oct 15, 1992
*************************************************************/

int kfscanf(
   kfile  *file,
   char   *format,
   kvalist)
{
	int num = 0;
	kva_list args;


	/*
	 *  Attach the "args" to the variable argument list.
	 */
	kva_start(args, format);

	/*
	 *  Call kdoscan() to do the actual scanning of the arguments.
	 *  The variable arguments are filled and the number of characters
	 *  written
	 */
	num = kdoscan(file, NULL, format, args);

	kva_end(args);
	return(num);
}

/************************************************************
*
*  Routine Name: ksscanf - scan input string and format it into 
*                          one or more arguments of the type specified.
*       Purpose: This function is a replacement for the system "sscanf"
*		 call;  however, ksscanf() uses data transport mechanisms
*                to scan the input, in order to support distributed computing. 
*
*                The ksscanf() function reads character data input, interprets 
*                them according to the format specified, and stores the result 
*                in the input arguments(s) specified by the variable argument
*		 list.  In addition, the ksscanf() routine guarantees that the 
*                integer returned will be the number of fields correctly 
*                scanned, and that scanning will never continue past the 
*                first error.  If there are insufficient arguments for the
*                format, the behavior is undefined;  if the format is exhausted
*                while arguments remain, the excess arguments are ignored.
*         Input: string  - the input string to format
*                format  - the format in which to interpret the input 
*        Output: kvalist - variable number of arguments into which to format the
*                          input string. The format string determines the type 
*                          of pointer(s) to be provided as the argument(s).
*                          Note that each argument must be a pointer, and that 
*                          sufficient space to accomodate the output must be 
*                          provided for strings.
*       Returns: The number of characters successfully formatted into the
*                input argument(s).
*    Written By: Mark Young
*          Date: Oct 15, 1992
*************************************************************/

int ksscanf(
   char *string,
   char *format,
   kvalist)
{
	int    num;
	kva_list args;


	/*
	 *  Attach the "args" to the variable argument list.
	 */
	kva_start(args, format);

	num = kdoscan(NULL, string, format, args);
	kva_end(args);
	return(num);
}

/************************************************************
*
*  Routine Name: kvscanf - scan formatted kstdin input of variable arguments
*       Purpose: This function is a replacement for the system "vscanf"
*		 call.  The only difference is that kvscanf() uses the
*		 transport routines to scan the string.  The text will
*		 be scanned in from kstdin.
*         Input: format - the format in which to the arguments will be
*        Output: args   - an argument list which are used as the
*			  the inputs to the format command.
*       Returns: number of match arguments found from kstdin 
*    Written By: Mark Young
*          Date: Oct 15, 1992
*************************************************************/

int kvscanf(
   char     *format,
   kva_list args)
{
	int num = 0;


	/*
	 *  Call kdoscan() to do the actual scanning of the arguments.
	 *  The variable arguments are filled and the number of characters
	 *  written
	 */
	num = kdoscan(kstdin, NULL, format, args);
	return(num);
}

/************************************************************
*
*  Routine Name: kvfscanf - scan formatted kfile input of variable arguments
*       Purpose: This function is a replacement for the system "vfscanf"
*		 call.  The only difference is that kvfscanf() uses the
*		 transport routines to scan the text.  The text will
*		 be scanned from the kfile transport.
*         Input: file   - kfile structure
*		 format - the format in which to the arguments will be
*        Output: args   - an argument list which are used as the
*			  the inputs to the format command.
*       Returns: number of matched arguments found in the kfile transport
*    Written By: Mark Young
*          Date: Oct 15, 1992
*************************************************************/

int kvfscanf(
   kfile    *file,
   char     *format,
   kva_list args)
{
	int num = 0;


	/*
	 *  Call kdoscan() to do the actual scanning of the arguments.
	 *  The variable arguments are filled and the number of characters
	 *  written
	 */
	num = kdoscan(file, NULL, format, args);
	return(num);
}

/************************************************************
*
*  Routine Name: kvsscanf - scan formatted string of a variable argument list
*       Purpose: This function is a replacement for the system "vsscanf"
*		 call.  The only difference is that kvsscanf() uses the
*		 transport routines scanning routines.  The text will
*		 be scanned to the supplied string.
*         Input: string - the output string to format the arguments into
*		 format - the format in which to the arguments will be
*        Output: args   - variable argument list which are used as the
*			  the inputs to the format command.
*       Returns: number of match arguments found in the supplied string
*    Written By: Mark Young
*          Date: Oct 15, 1992
*************************************************************/

int kvsscanf(
   char     *string,
   char     *format,
   kva_list args)
{
	int    num;

        /*
         *  Call kdoscan() to do the actual scanning of the arguments.
         *  The variable arguments are filled and the number of characters
         *  written
         */
	num = kdoscan(NULL, string, format, args);
	return(num);
}

/************************************************************
*
*  Routine Name: kprintf - print formatted output to kstdout
*       Purpose: This function is a replacement for the system "printf"
*		 call; however, kprintf() uses data transport mechanisms
*                to print the text, in order to support distributed computing.
*
*                The kprintf() function converts, formats, and writes
*                its value parameters as specified by the format parameter
*                to kstdout. If there are insufficient values for the format, 
*                the behavior is undefined;  if the format is exhausted while 
*                values remain, the excess values are ignored.
*         Input: format  - the format in which to print the values 
*                kvalist - variable number of values to format and write
*                          to kstdout.  The format string determines the data
*                          type of the value(s) to be provided.
*       Returns: The number of characters written to stdout
*    Written By: Mark Young
*          Date: Oct 15, 1992
*************************************************************/

int kprintf(
   char *format,
   kvalist)
{
	int num = 0;
	kva_list args;


	/*
	 *  Attach the "args" to the variable argument list.
	 */
	kva_start(args, format);

	/*
	 *  Call kdoprint() to do the actual printing of the arguments.
	 *  The variable arguments are filled and the number of characters
	 *  written
	 */
	num = kdoprint(kstdout, NULL, format, args);

	kva_end(args);
	return(num);
}

/************************************************************
*
*  Routine Name: kfprintf - print one or more arguments in the format
*			    specified to an output file stream
*       Purpose: This function is a replacement for the system "fprintf"
*                call;  however, kfprint() uses data transport mechanisms
*                to print the output, in order to support distributed computing.
*
*                The kfprintf() function converts, formats, and writes
*                its value parameters as specified by the format parameter
*                to the output stream given. If there are insufficient 
*                values for the format, the behavior is undefined;  
*                if the format is exhausted while values remain, the excess 
*                values are ignored.
*         Input: file    - stream to open output file
*		 format  - the format in which to print the values
*                kvalist - variable number of values to format and write
*                          to the output file stream.  The format string
*                          determines the data type of the value(s) to be 
*                          provided.
*       Returns: The number of characters written to the output file.
*    Written By: Mark Young
*          Date: Oct 15, 1992
*************************************************************/

int kfprintf(
   kfile  *file,
   char   *format,
   kvalist)
{
	int num = 0;
	kva_list args;


	/*
	 *  Attach the "args" to the variable argument list.
	 */
	kva_start(args, format);

	/*
	 *  Call kdoprint() to do the actual printing of the arguments.
	 *  The variable arguments are filled and the number of characters
	 *  written
	 */
	num = kdoprint(file, NULL, format, args);

	kva_end(args);
	return(num);
}

/************************************************************
*
*  Routine Name: ksprintf - print one or more arguments in the format
*                           specified to an output string.
*       Purpose: This function is a replacement for the system "sprintf"
*                call;  however, ksprint() uses data transport mechanisms
*                to print the output string, in order to support distributed 
*                computing.
*
*                The ksprintf() function converts, formats, and writes
*                its value parameters as specified by the format parameter
*                to the output string given. If there are insufficient
*                values for the format, the behavior is undefined;
*                if the format is exhausted while values remain, the excess
*                values are ignored.
*         Input: string  - the output string in which to format the values 
*		 format  - the format in which to print the values 
*                kvalist - variable number of values to format and write
*                          to the output file string.  The format string
*                          determines the data type of the value(s) to be 
*                          provided.
*       Returns: The number characters written to the output string.
*    Written By: Mark Young
*          Date: Oct 15, 1992
*************************************************************/

int ksprintf(
   char *string,
   char *format,
   kvalist)
{
	int  num;
	kva_list args;


	/*
	 *  Attach the "args" to the variable argument list.
	 */
	kva_start(args, format);

	/*
	 *  Call kdoprint() to do the actual printing of the arguments.
	 *  The variable arguments are filled and the number of characters
	 *  written
	 */
	num = kdoprint(NULL, string, format, args);
	string[num] = '\0';
	kva_end(args);
	return(num);
}

/************************************************************
*
*  Routine Name: kvprintf - print formatted kstdout output of variable
*			    arguments
*       Purpose: This function is a replacement for the system "vprintf"
*		 call.  The only difference is that kvprintf() uses the
*		 transport routines to print the string.  The text will
*		 be printed to kstdout.
*         Input: format - the format in which to the arguments will be
*                args   - an argument list which are used as the
*			  the inputs to the format command.
*       Returns: The number of characters written to kstdout.
*    Written By: Mark Young
*          Date: Oct 15, 1992
*************************************************************/

int kvprintf(
   char     *format,
   kva_list args)
{
	int num = 0;


	/*
	 *  Call kdoprint() to do the actual printing of the arguments.
	 *  The variable arguments are filled and the number of characters
	 *  written
	 */
	num = kdoprint(kstdout, NULL, format, args);
	return(num);
}

/************************************************************
*
*  Routine Name: kvfprintf - print formatted kfile output of variable
*			     arguments
*       Purpose: This function is a replacement for the system "vfprintf"
*		 call.  The only difference is that kvfprintf() uses the
*		 transport routines to print the text.  The text will
*		 be printed to the kfile transport.
*         Input: file   - kfile structure
*		 format - the format in which to the arguments will be
*                args   - an argument list which are used as the
*			  the inputs to the format command.
*       Returns: The number of characters written to kfile transport.
*    Written By: Mark Young
*          Date: Oct 15, 1992
*************************************************************/

int kvfprintf(
   kfile    *file,
   char     *format,
   kva_list args)
{
	int num = 0;


	/*
	 *  Call kdoprint() to do the actual printing of the arguments.
	 *  The variable arguments are filled and the number of characters
	 *  written
	 */
	num = kdoprint(file, NULL, format, args);
	return(num);
}

/************************************************************
*
*  Routine Name: kvsprintf - print formatted string output of variable
*			     arguments list
*       Purpose: This function is a replacement for the system "vsprintf"
*		 call.  The only difference is that kvsprintf() uses the
*		 transport routines printing routines.  The text will
*		 be printed to the supplied string.
*         Input: string - the output string to format the arguments into
*		 format - the format in which to the arguments will be
*                args   - variable argument list which are used as the
*			  the inputs to the format command.
*       Returns: The number of characters written to the string
*    Written By: Mark Young
*          Date: Oct 15, 1992
*************************************************************/

int kvsprintf(
   char     *string,
   char     *format,
   kva_list args)
{
	int    num;


	/*
	 *  Call kdoprint() to do the actual printing of the arguments.
	 *  The variable arguments are filled and the number of characters
	 *  written
	 */
	num = kdoprint(NULL, string, format, args);
	string[num] = '\0';
	return(num);
}

/**************************************************************
*  
*  Routine Name: kfopen - open a data transport stream
*       Purpose: This function is a replacement for the system "fopen"
*     	         call;  kfopen() supports the opening of any khoros 
*                data transport mechanism.
*
*     Identifiers for Khoros data transport mechanisms include the following:
*     ! 'file'	 (standard Unix file; local transport / permanent storage)
*     ! 'pipe'   (standard pipes; local transport / no permanent storage)
*     ! 'mmap'   (virtual memory; local transport / permanent storage)
*     ! 'shm'    (shared memory; local transport / permanent storage)
*     ! 'socket' (sockets; local or remote transport / no permanent storage)
*     ! 'stream' (named streams/pipes;  local transport / no permanent storage)
*
*     The token is the type of identifier for that transport.  For a file, 
*     it is simply the filename.  For shared memory, it is the shared memory 
*     key.  For a pipe, it is the input & output file descriptors, as in 
*     "pipe=[3,4]".  For a socket, it is the number of the socket, as in 
*     "socket=5430".
*
*     Together, the identifier and the token identify the path to the
*     data transport to be opened.  The path must be in the form of an 
*     identifier and a token, separated by an "=" equal indicator, as in:
*     ! "{identifier}={transport token}"
*
*     For example, the path parameter for opening shared memory might be:
*     ! "shm=1032"
*     The path parameter for opening a file might be:
*     ! "file=/usr/tmp/vadd1235"
*     When opening a file as the data transport mechanism, 
*     the identifier and the "=" sign may be omitted, as in:
*     ! "./foobar"
*         Input: path - the path specifying the data transport to be 
*                       opened; see above.
*	         type - how the data transport is to be opened.
*  	                !"r"  - Open file for only reading
*  	   		!"w"  - Open file for only writing
*  	   		!"a"  - Append file for only writing
*  	   		!"A"  - Append file for writing; non over-write
*  	   		!"r+" - Open file for both reading & writing
*  	   		!"w+" - Open file for both reading & writing
*  	   		!"a+" - Append file for only writing
*  	   		!"A+" - Append file for writing; non over-write
*       Returns: The kfile pointer representing the open stream 
*                to the data transport on success, NULL on failure
*    Written By: Mark Young  
*          Date: Jul 08, 1992
* Modifications:
**************************************************************/

kfile *kfopen(
   char *path,
   char *type)
{
	kfile *file;
	int   flags;


	flags = ktype_to_flags(type);
	if ((file = kfile_open(path, flags, 0666)) == NULL)
	{
	   return(NULL);
	}
	return(file);
}

/**************************************************************
*  
*  Routine Name: kfdopen - open an existing transport descriptor as a
*			   data transport stream
*       Purpose: This function is a replacement for the system "fdopen"
*     	         call;  kfdopen() supports all data transport mechanisms, 
*                not just unix files.
*
*	         The kfdopen() function associates a data transport stream
*                with an existing transport descriptor previously obtained 
*                from a call to kopen(), kdup(), kdup2(), or kpipe().  These
*                functions open data transports, but do not return pointers
*                to kfile structures; many of the khoros I/O functions require
*                pointers to the kfile structure.  Note that the type of
*                khoros data transport must agree with the mode of the open
*                data transport.
*         Input: id   - the transport identifier opened earlier
*		 type - how the data transport is to be opened.
*                      !"r"  - Open file for only reading
*                      !"w"  - Open file for only writing
*                      !"a"  - Append file for only writing
*                      !"A"  - Append file for writing; non over-write
*                      !"r+" - Open file for both reading & writing
*                      !"w+" - Open file for both reading & writing
*                      !"a+" - Append file for only writing
*                      !"A+" - Append file for writing; non over-write
*       Returns: The kfile pointer representing the open stream 
*                to the data transport on success, NULL on failure
*    Written By: Mark Young  
*          Date: Jul 08, 1992
* Modifications:
**************************************************************/

kfile *kfdopen(
   int  id,
   char *type)
{
	kfile *file;
	int   flags;


	if ((file = kfile_checkid(id)) == NULL)
	   return(NULL);

#if 0
	flags = ktype_to_flags(type);
	if ((file->flags & flags) != flags)
	   return(NULL);
#endif

#if 0
	kflags_to_type(file->flags, old_type);
	if (kstrcmp(type, old_type) == 0)
	   return(NULL);
#endif

	return(file);
}

/************************************************************
*
*  Routine Name: kfreopen - re-open a data transport stream
*       Purpose: This function is a replacement for the UNIX system
*		 call "freopen";  kfreopen() supports all khoros data 
*                transport mechanisms, not just unix files.
*
*		 kfreopen() will open a specified file on for an existing
*		 stream.  The existings stream is closed before the new
*		 filename is opened.  This function is typically used to
*		 open a specified file as one of the predefined streams;
*		 such as standard input, standard output, or standard error.
*
*                The kfreopen() function substitutes the specified data 
*                transport in place of the open data transport stream. 
*                The original stream is closed regardless of whether the 
*                kfreopen() function succeeds with the new data transport. 
*                The kfreopen() function is typically used to attach the 
*                preopened data transport streams associated with kstdin, 
*                kstdout, and kstderr to other data transport mechanisms.
*
*     Identifiers for Khoros data transport mechanisms include the following:
*     ! 'file'   (standard Unix file; local transport / permanent storage)
*     ! 'pipe'   (standard pipes; local transport / no permanent storage)
*     ! 'mmap'   (virtual memory; local transport / permanent storage)
*     ! 'shm'    (shared memory; local transport / permanent storage)
*     ! 'socket' (sockets; local or remote transport / no permanent storage)
*     ! 'stream' (named streams/pipes;  local transport / no permanent storage)
*
*     The token is the type of identifier for that transport.  For a file,
*     it is simply the filename.  For shared memory, it is the shared memory
*     key.  For a pipe, it is the input & output file descriptors, as in
*     "pipe=[3,4]".  For a socket, it is the number of the socket, as in
*     "socket=5430".
*
*     Together, the identifier and the token identify the path to the
*     data transport to be opened.  The path must be in the form of an
*     identifier and a token, separated by an "=" equal indicator, as in:
*     ! "{identifier}={transport token}"
*
*     For example, the path parameter for opening shared memory might be:
*     ! "shm=1032"
*     The path parameter for opening a file might be:
*     ! "file=/usr/tmp/vadd1235"
*     When opening a file as the data transport mechanism,
*     the identifier and the "=" sign may be omitted, as in:
*     ! "./foobar"
*
*         Input: path - the path specifying the data transport to be
*                       re-opened; see above.
*		 type - how the data transport is to be re-opened.
*                      !"r"  - Open file for only reading
*                      !"w"  - Open file for only writing
*                      !"a"  - Append file for only writing
*                      !"A"  - Append file for writing; non over-write
*                      !"r+" - Open file for both reading & writing
*                      !"w+" - Open file for both reading & writing
*                      !"a+" - Append file for only writing
*                      !"A+" - Append file for writing; non over-write
*		 file - the existing data transport stream to be reopened
*			for the specified path.
*       Returns: The kfile pointer representing the open stream 
*                to the re-opened data transport on success, NULL on failure
*    Written By: Mark Young
*          Date: Apr 28, 1993 12:06
*************************************************************/

kfile *kfreopen(
   char  *path,
   char  *type,
   kfile *file)
{
	char *otype;
	int  oflags;


	/*
	 *  Save the old flags and reopen the file with the new type
	 */
	if ((file = kfile_checkfile(file)) == NULL)
	{
	   errno = EBADF;
	   return(NULL);
	}
	otype  = file->type;
	oflags = file->flags;
	file->type  = type;
	file->flags = ktype_to_flags(type);

	if (kfile_rename(file, path) == -1)
	{
	   file->flags = oflags;
	   file->type  = otype;
	   return(NULL);
	}
	file->type = kstrdup(file->type);
	kfree(otype);
	return(file);
}

/************************************************************
*
*  Routine Name: ktmpfile - create a temporary data transport stream
*       Purpose: This function is a replacement for the UNIX system
*		 call "ktmpfile";  ktmpfile() supports all khoros data 
*                transports, not just unix files.  The data transport 
*                will automatically be deleted when closed.
*       Returns: The kfile pointer representing the open stream 
*                to the temporary transport on success, NULL on failure
*    Written By: Mark Young
*          Date: May 11, 1993 12:21
*************************************************************/

kfile *ktmpfile(void)
{
	kfile *file;
	char  *filename = kprog_get_program();


	/*
	 *  Create temporary file and open it for reading and writing
	 */
	if (kstrncmp(filename, "Unknown", kstrlen("Unknown")) == 0)
	   filename = "kfileXXXXX";

	if ((filename = ktempnam(NULL, filename)) == NULL ||
	    (file = kfopen(filename, "w+")) == NULL)
	{
	   if (filename) kunlink(filename);
	   kfree(filename);
	   return(NULL);
	}
	kfile_setstate(file, KFILE_TEMP);
	kfree(filename);
	return(file);
}

/**************************************************************
*  
*  Routine Name: kfclose - close a data transport stream
*       Purpose: This function is a replacement for the system "fclose"
*     	         call;  kfclose() supports all khoros data transport
*                mechanisms, not just Unix files.
*         Input: file - the kfile pointer representing the open 
*                       stream to a khoros data transport mechanism
*                       that was opened earlier using kfopen().
*       Returns: 0 on success, -1 on failure.
*    Written By: Mark Young  
*          Date: Jul 08, 1992
* Modifications:
**************************************************************/

int kfclose(
   kfile *file)
{
	/*
	 *  Make sure the entry is within our table.  If not then it
	 *  has probably been closed already.
	 */
	if ((file = kfile_checkfile(file)) == NULL)
	{
	   errno = EBADF;
	   return(-1);
	}
	return(kfile_close(file));
}

/**************************************************************
*  
*  Routine Name: kfread - read from a data transport stream
*       Purpose: This function is a replacement for the system "fread"
*       	 call;  kfread() supports all khoros data transport
*                mechanisms, not just Unix files.
*  
*	         The routine will read 'nitem' items, each of 'size' 
*                bytes from the data transport stream associated 
*                with 'file' into the memory location accessed by 'ptr'.  
*         Input: ptr    - a pointer to allocated space into which
*                         to read the data 
*	         size   - the size, in bytes, of each item
* 	         nitems - the number of items to be read
*	         file   - the data transport transport to be read from;
*                         must have been opened earlier using kfopen().
*       Returns: The number of items read on success; 0 when end-of-file 
*                is encountered or an error occurs.
*    Written By: Mark Young
*          Date: Jul 08, 1992
* Modifications:
**************************************************************/

int kfread(
   kaddr  ptr,
   int    size,
   int    nitems,
   kfile  *file)
{
	int	num;


	/*
	 *  Make sure the entry is within our table.  If not then it
	 *  has probably been closed already.
	 */
	if ((file = kfile_checkfile(file)) == NULL)
	{
	   errno = EBADF;
	   return(-1);
	}

	/*
	 *  Sanity check to make sure we aren't writing 0 bytes
	 */
	if (size <= 0 || nitems <= 0)
	   return(0);

	/*
	 *  Call the "read()" routine to read from the transport
	 */
	num = kfile_read(file, ptr, size*nitems);
	return(num/size);
}

/**************************************************************
*  
*  Routine Name: kfwrite - write to a data transport stream
*       Purpose: This function is a replacement for the system "fwrite"
*                call;  kwrite() supports all khoros data transport
*                mechanisms, not just Unix files.
*
*	         The routine will write 'nitem' items, each of 'size' 
*                bytes to the data transport stream associated 
*                with 'file' from the memory location accessed by 'ptr'.  
*         Input: ptr    - a pointer to allocated space from which
*                         to write the data 
*	         size   - the size, in bytes, of each item
* 	         nitems - the number of items to be written
*	         file   - the data transport transport to be written to;
*                         must have been opened earlier using kfopen().
*       Returns: The number of items written on success, 0 if
*	         an error is encountered.
*    Written By: Mark Young  
*          Date: Jul 08, 1992
* Modifications:
**************************************************************/

int kfwrite(
   kaddr  ptr,
   int    size,
   int    nitems,
   kfile  *file)
{
	int	num;


	/*
	 *  Make sure the entry is within our table.  If not then it
	 *  has probably been closed already.
	 */
	if ((file = kfile_checkfile(file)) == NULL)
	{
	   errno = EBADF;
	   return(-1);
	}

	/*
	 *  Sanity check to make sure we aren't writing 0 bytes
	 */
	if (size <= 0 || nitems <= 0)
	   return(0);

	/*
	 *  Call the "write()" routine to write to the transport
	 */
	num = kfile_write(file, ptr, size*nitems);
	return(num/size);
}

/**************************************************************
*  
*  Routine Name: kfputc - put a character onto the data transport stream
*       Purpose: This function is a replacement for the system "fputc"
*                call; kfputc() supports all khoros data transport
*                mechanisms, not just Unix files.
*
*	         The routine will write a single character to the
*	         data transport stream.  If the character could not be
*	         written, EOF is returned; otherwise, the character
*	         that is written is returned.
*         Input: character - the character to be written
*	         file      - the kfile pointer associated with the open
*                            stream to the data transport
*       Returns: The character written on success; EOF on failure
*    Written By: Mark Young  
*          Date: Jul 08, 1992
* Modifications:
**************************************************************/

int kfputc(
   int    character,
   kfile  *file)
{
	int	 num;
	unsigned char ptr[1]; 


	/*
	 *  Make sure the entry is within our table.  If not then it
	 *  has probably been closed already.
	 */
	if ((file = kfile_checkfile(file)) == NULL)
	{
	   errno = EBADF;
	   return(-1);
	}

	/*
	 *  Call the "write()" routine to write to the transport
	 */
	ptr[0] = (unsigned char) character;
	num    = kfile_write(file, (kaddr) ptr, 1);
	return((num == 1) ? character : EOF);
}

/**************************************************************
*  
*  Routine Name: kfputs - put a string onto the data transport stream 
*       Purpose: This function is a replacement for the system "fputs"
*                call; kfputs() supports all khoros data transport
*                mechanisms, not just Unix files.
*
*                The routine will write a NULL-terminated string to the data 
*                transport output stream.  If the string could not be
*                written, EOF is returned; otherwise, the number of
*                characters that were written are returned.
*         Input: buffer - the string to be written
*                file   - the kfile pointer associated with the open stream 
*                         to the data transport, opened earlier with kfopen().
*       Returns: The number of characters written on success, 
*                EOF if an error is encountered.
*    Written By: Mark Young  
*          Date: Jul 08, 1992
* Modifications:
**************************************************************/

int kfputs(
   char  *buffer,
   kfile *file)
{
	int	 num;


	/*
	 *  Make sure the entry is within our table.  If not then it
	 *  has probably been closed already.
	 */
	if ((file = kfile_checkfile(file)) == NULL)
	{
	   errno = EBADF;
	   return(-1);
	}

	/*
	 *  Call the "write()" routine to write to the transport
	 */
	num = kstrlen(buffer);
	if (kfile_write(file, (kaddr) buffer, num) != num)
	   return(EOF);

	return(num);
}

/**************************************************************
*  
*  Routine Name: kfgetc - get a character from the data transport stream
*       Purpose: This function is a replacement for the system "fgetc()"
*                routine;  kfgetc() supports all khoros data transport
*                mechanisms, not just Unix files.
*
*	         The routine gets a single character from the data 
*                transport stream, and moves the data transport pointer 
*                ahead by one character.  If the character could not 
*                be read, EOF is returned; otherwise, the character 
*                that is read is returned.
*         Input: file - the kfile pointer associated with the open stream
*                       to the data transport, opened earlier with kfopen().
*       Returns: The character read on success, EOF if an error is encountered.
*    Written By: Mark Young  
*          Date: Jul 08, 1992
* Modifications:
**************************************************************/

int kfgetc(
   kfile *file)
{
	unsigned char character; 


	/*
	 *  Make sure the entry is within our table.  If not then it
	 *  has probably been closed already.
	 */
	if ((file = kfile_checkfile(file)) == NULL)
	{
	   errno = EBADF;
	   return(-1);
	}

	if (file->num_bytes > file->offset)
	   character = file->buf[file->offset++], file->position++;
	else if (kfile_read(file, (kaddr) &character, 1) <= 0)
	   return(EOF);

	return((int) character);
}

/**************************************************************
*  
*  Routine Name: kungetc - push a character back onto the data transport stream
*       Purpose: This function is a replacement for the system ungetc()
*	         routine;  kungetc() supports all khoros data transport
*                mechanisms, not just Unix files.
*
*	         The routine will put a single character back 
*                onto the specified data transport stream, and moves the
*                data transport pointer back one character.  If the
*	         character could not be put back, EOF is returned; 
*                otherwise, the character that is put back is returned.
*         Input: character - the character to be put back onto the stream
*	         file      - the kfile pointer associated with the open stream
*                            to the data transport, opened earlier with
*			     kfopen().
*       Returns: The character that was put back on success, EOF if an error
*		 is encountered.
*    Written By: Mark Young  
*          Date: Aug 19, 1992
**************************************************************/

int kungetc(
   int    character,
   kfile *file)
{
	unsigned char *buf;

	/*
	 *  Make sure the entry is within our table.  If not then it
	 *  has probably been closed already.
	 */
	if ((file = kfile_checkfile(file)) == NULL)
	{
	   errno = EBADF;
	   return(EOF);
	}

	/*
	 *  Call the "ungetc()" routine to ungetc from the transport
	 */
	if (file->offset == 0 || !file->buf)
	   return(EOF);

	/*
	 *  Return the character put back on the stream
	 */
	file->offset--;
	file->position--;
	buf = (unsigned char *) file->buf;
	if (buf[file->offset] != character)
	   buf[file->offset] = character;

	return(character);
}

/**************************************************************
*  
*  Routine Name: kfgets - get a string from a data transport stream
*       Purpose: This function is a replacement for the system "fgets"
*	         call;  kfgets() supports all khoros data transport
*                mechanisms, not just Unix files.
*  
*	         The routine reads a string of characters from the 
*                specified data transport stream, and moves the data
*                transport pointer ahead by that number of characters.  
*                kfgets() reads characters
*	         from the stream into the array pointed to by 'buffer'
*	         until 'num'-1 characters are read, or until a '\n' 
*                (newline character) is read (and added to the string), 
*                or until EOF is encountered. The string is terminated
*                with a NULL character.  If the string could not be read, 
*                NULL is returned; otherwise, the string is returned.  
*         Input: num    - maximum number of characters to read (the
*                         size of the character array).
*                file   - the kfile pointer associated with the open stream
*                         to the data transport, opened earlier with kfopen()
*        Output: buffer - pointer to allocated character array into which
*                         to read the string.
*       Returns: Returns the string read on success, 
*                NULL if EOF is encountered before any characters have
*                been read, or if an error occurs.
*    Written By: Mark Young  
*          Date: Jul 08, 1992
* Modifications:
**************************************************************/

char *kfgets(
   char  *buffer,
   int    num,
   kfile *file)
{
	int  i = num;
	char *buf = buffer;


	/*
	 *  Make sure the entry is within our table.  If not then it
	 *  has probably been closed already.
	 */
	if ((file = kfile_checkfile(file)) == NULL)
	{
	   errno = EBADF;
	   return(NULL);
	}

	/*
	 *  while i is greater than 0 then we have space in which to read
	 *  into the character "buffer".
	 */
	while (i == -1 || --i > 0)
	{
	    if (file->num_bytes > file->offset)
	       *buf = file->buf[file->offset++], file->position++;
	    else if (kfile_read(file, buf, 1) <= 0)
	       break;
 
	    if ((int) *buf++ == '\n')
	       break;
	}
	*buf = '\0';

	/*
	 *  Check why we terminated and return the appropriate value
	 */
	if (buffer == buf)
	   return(NULL);
	else
	   return(buffer);
}

/**************************************************************
*  
*  Routine Name: kfseek - set position in a data transport stream
*       Purpose: This function is a replacement for the system "fseek"
*                call;  kfseek() supports all khoros data transport
*                mechanisms, not just Unix files.
*
*                This function sets the position of the next input or
*                output operation on the data transport stream by changing
*		 the location of the data transport pointer.
*         Input: file   - the kfile pointer associated with the open stream
*                         to the data transport, opened earlier with kfopen()
*  		 offset - offset specifying new position of file pointer
*  		 whence - controls how the offset will be applied:
*                         ! 0 - sets the pointer to the absolute 
*                         !     value of the offset parameter
*                         ! 1 - sets the pointer to its current location plus 
*                         !     the value of the offset parameter
*                         ! 2 - sets the pointer to the size of the file plus
*                         !     the value of the offset parameter
*       Returns: 0 on success, -1 on failure
*    Written By: Mark Young  
*          Date: Jul 08, 1992
* Modifications:
**************************************************************/

int kfseek(
   kfile *file,
   long  offset,
   int   whence)
{
	/*
	 *  Make sure the entry is within our table.  If not then it
	 *  has probably been closed already.
	 */
	if ((file = kfile_checkfile(file)) == NULL)
	{
	   errno = EBADF;
	   return(-1);
	}

	/*
	 *  Call the "kfile_seek()" routine to seek on the transport
	 */
	return(kfile_seek(file, offset, whence));
}

/**************************************************************
*  
*  Routine Name: kftell - report current position in the data transport stream
*       Purpose: This function is a replacement for the system "ftell"
*                call;  kftell() supports all khoros data transport
*                mechanisms, not just Unix files.
*
*                This function is used to check the current position
*	         of the data transport pointer within the data transport stream.
*         Input: file - the kfile pointer associated with the open stream
*                       to the data transport, opened earlier with kfopen()
*       Returns: 0 on success, -1 on failure
*    Written By: Mark Young  
*          Date: Jul 08, 1992
* Modifications:
**************************************************************/

int kftell(
   kfile *file)
{
	/*
	 *  Get the file entry associated with the id
	 */
	if ((file = kfile_checkfile(file)) == NULL)
	   return(FALSE);

	/*
	 *  Call the "kfile_tell()" routine to tell on the transport
	 */
	return(kfile_tell(file));
}

/************************************************************
*
*  Routine Name: kfdup - duplicate an existing data transport stream
*       Purpose: Duplicates an existing data transport stream.  
*                The "file" is entry in the process's transport
*		 descriptor table.  The new entry returned kfdup() will
*		 be the lowest table entry.  
*         Input: file - the kfile pointer associated with the open stream
*                       to the data transport to be duplicated, 
*                       opened earlier with kfopen()
*       Returns: The newly duplicated stream on success, NULL upon failure
*    Written By: Mark Young
*          Date: Apr 28, 1993 22:13
*************************************************************/

kfile *kfdup(
   kfile *file)
{
	int newid;


	/*
	 *  Get the file entry associated with the id
	 */
	if ((kfile_checkfile(file)) == NULL)
	   return(NULL);

	newid = kfile_dup(kfileno(file), -1);
	return(kfile_checkid(newid));
}

/************************************************************
*
*  Routine Name: kfdup2 - duplicate an existing data transport into a
*			  specific stream
*       Purpose: kfdup2() is used to duplicate an existing khoros transport
*		 stream.  The "file1" is an entry in the process's transport
*		 descriptor table.  The newly dupedd entry will be stored in the
*		 transport table entry specified by "file2".  If file2 is
*		 open then it will be closed (kfclose) before the dup of
*		 file1 is put into file2.
*         Input: file1 - the existing khoros transport stream to be dup'ed
*		 file2 - the stream to be used for the newly dup'ed stream
*       Returns: the newly dup'ed stream on success, or NULL upon failure
*    Written By: Mark Young
*          Date: Apr 28, 1993 22:13
*************************************************************/

kfile *kfdup2(
   kfile *file1,
   kfile *file2)
{
	int newid;


	/*
	 *  Get the file entry associated with the id
	 */
	if ((kfile_checkfile(file1)) == NULL)
	   return(NULL);

	(void) kfclose(file2);
	newid = kfile_dup(kfileno(file1), kfileno(file1));
	return(kfile_checkid(newid));
}

/**************************************************************
*  
*  Routine Name: krewind - rewind a data transport stream to the beginning
*       Purpose: This function is a replacement for the system "rewind"
*                call;  krewind() supports all khoros data transport
*                mechanisms, not just Unix files.
*
*                This function is used to re-position the data transport 
*                pointer to the beginning of the stream. It is equivalent to:
*  	         ! (void) kfseek(file, 0L, 0);
*  
*	         The behavior of the function will differ depending on 
*                the data transport being used. For a file, it calls fseek() 
*                to reposition the file pointer; for shared memory segments,
*                it simply resets the internal offset pointer.
*         Input: file - the kfile pointer associated with the open stream
*                       to the data transport, opened earlier with kfopen()
*    Written By: Mark Young  
*          Date: Jul 08, 1992
* Modifications:
**************************************************************/

void krewind(
   kfile *file)
{
	/*
	 *  Make sure the entry is within our table.  If not then it
	 *  has probably been closed already.
	 */
	if ((file = kfile_checkfile(file)) == NULL)
	{
	   errno = EBADF;
	   return;
	}

	/*
	 *  Call the "kfile_seek()" routine to seek on the transport
	 */
	(void) kfile_seek(file, 0L, 0);
}

/************************************************************
*
*  Routine Name: kfflush - flush buffered output of a data transport stream
*       Purpose: This function is a replacement for the system "fflush"
*                call;  kfflush() supports all khoros data transport
*                mechanisms, not just Unix files.
*
*                This routine is used to flush any buffered output;
*                any data in the buffer of the output stream is written
*                to the data transport stream specified.
*         Input: file - the kfile pointer associated with the open stream
*                       to the data transport, opened earlier with kfopen()
*	Returns: 0 on success, EOF if the file was not open for writing,
*                or if a write error occurred.
*    Written By: Mark Young
*          Date: Aug 19, 1992
*************************************************************/

int kfflush(
   kfile *file)
{
	int num, num_bytes;
	TransportInformation *routines;


	/*
	 *  Make sure the entry is within our table.  If not then it
	 *  has probably been closed already.
	 */
	if ((file = kfile_checkfile(file)) == NULL)
	{
	   errno = EBADF;
	   return(EOF);
	}

	/*
	 *  Make sure that we are trying to flush a writeable transport
	 */
	if (!kfile_iswrite(file))
	   return(EOF);

	/*
	 *  If the offset is less than or equal to 0 then the buffer is
	 *  empty.
	 */
	if (file->num_bytes == 0)
	   return(0);

	/*
	 *  Call the "write" routine to write the remaining buffer to the
	 *  transport
	 */
	num_bytes = file->num_bytes;
	file->offset = file->num_bytes = 0;

	routines = file->routines;
	num = routines->write(file, file->buf, num_bytes);
	if (num != num_bytes)
	{
	   kerror("ktransport", "kfflush", "Only able to flush %d out of %d \
bytes.", num, num_bytes);
	   file->offset    = num;
	   file->num_bytes = num_bytes;
	   return(EOF);
	}
	return(0);
}

/************************************************************
*
*  Routine Name: kfeof - check if a data transport stream is at EOF
*       Purpose: This function is a replacement for the system "feof"
*                call;  kfeof() supports all khoros data transport
*                mechanisms, not just Unix files.
*
*                This routines test to see if the data transport
*                pointer of a stream open for reading has reached 
*                the end of file.
*         Input: file - the kfile pointer associated with the open stream
*                       to the data transport, opened earlier with kfopen()
*       Returns: A non-zero value if EOF has been reached on the
*                specified data transport stream; otherwise, zero is
*		 returned.
*    Written By: Mark Young and Steve Kubica
*          Date: Aug 20, 1992
*************************************************************/

int kfeof(
   kfile *file)
{
	/*
	 *  Make sure the entry is within our table.  If not then it
	 *  has probably been closed already.
	 */
	if ((file = kfile_checkfile(file)) == NULL)
	{
	   errno = EBADF;
	   return(1);
	}

	/*
	 *  The file state is set to EOF when end of file is encountered
	 */
	if (kfile_iseof(file))
	   return(1);
	else
	   return(0);
}

/************************************************************
*
*  Routine Name: kclearerr - clear the EOF/error flags of a data transport
*			     stream
*       Purpose: This function is a replacement for the system "clearerr"
*                call;  kclearerr() supports all khoros data transport
*                mechanisms, not just Unix files.
*               
*                The function resets the EOF and error flags for the
*                data transport stream specified. Note that these flags
*                are cleared originally when the data transport is
*                opened, or when the data transport stream is re-wound
*                using krewind().  The EOF flag is also cleared by calls
*                to kfseek().
*         Input: file - the kfile pointer associated with the open stream
*                       to the data transport, opened earlier with kfopen()
*    Written By: Mark Young and Steve Kubica
*          Date: Aug 20, 1992
*************************************************************/

void kclearerr(
   kfile *file)
{
	/*
	 *  Make sure the entry is within our table.  If not then it
	 *  has probably been closed already.
	 */
	if ((file = kfile_checkfile(file)) == NULL)
	{
	   errno = EBADF;
	   return;
	}

	/*
	 *  The file state is only set when EOF is encountered
	 */
	kfile_clrstate(file, KFILE_EOF);
}

/************************************************************
*
*  Routine Name: ksetbuffer - set the I/O buffer of a data transport stream
*       Purpose: ksetbuffer(), an alternate form of setbuf(), can be used
*		 after a stream has been opened but before it is read or
*		 written.  It uses the character array buf whose size is
*		 specified by the bufsiz argument instead of an automatically
*		 fixed size of BUFSIZ.  If buf is NULL, then no buffer is
*		 is used.
*         Input: file   - the kfile pointer associated with the open stream
*                         to the data transport, opened earlier with kfopen()
*		 buf    - the new I/O buffer or NULL if no buffer is desired
*		 bufsiz - the size of the new buffer
*    Written By: Mark Young
*          Date: Aug 20, 1992
*************************************************************/

void ksetbuffer(
   kfile  *file,
   char   *buf,
   int    bufsiz)
{
	/*
	 *  Make sure the entry is within our table.  If not then it
	 *  has probably been closed already.
	 */
	if ((file = kfile_checkfile(file)) == NULL)
	{
	   errno = EBADF;
	   return;
	}

	/*
	 *  flush the existing buffer or copy it into the new buf if it
	 *  is a read buffer...
	 */
	if (kfile_iswrite(file))
	   (void) kfflush(file);

	/*
	 *  The buf is freed if this is my buffer and was allocated in
	 *  kfile_open() and was malloc'ed.
	 */
	if (kfile_ismybuf(file))
	{
	   kfree(file->buf);
	   kfile_clrstate(file, KFILE_MYBUF);
	}
	file->buf    = buf;
	file->bufsiz = bufsiz;

	file->offset    = 0;
	if (kfile_isread(file))
	   file->num_bytes = bufsiz;
	else
	   file->num_bytes = 0;
}

/************************************************************
*
*  Routine Name: kgetbuffer - get the current data transport stream's
*			      input/output buffer and it's size
*
*       Purpose: kgetbuffer(), the opposite from ksetbuffer(), can be used
*		 after a stream buffer has been initialized.  It returns
*		 the initialized stream buffer.  The character array buf
*		 whose size is returned as the bufsiz argument, unless NULL.
*         Input: file - the kfile transport to be written which was
*			opened earlier with kfopen().
*        Output: bufsiz - return the size of the new buffer if not NULL
*       Returns: The externally inialized buf or NULL if the
*		 the transport is non-buffered or if the buffer is the
*		 internal buffer.
*    Written By: Mark Young
*          Date: Aug 20, 1992
*************************************************************/

char  *kgetbuffer(
   kfile *file,
   int   *bufsiz)
{
	/*
	 *  Make sure the entry is within our table.  If not then it
	 *  has probably been closed already.
	 */
	if ((file = kfile_checkfile(file)) == NULL)
	{
	   errno = EBADF;
	   return(NULL);
	}

	/*
	 *  The buf is returned if it's not my buf.  I can't give anyone
	 *  access to our internal buffers.
	 */
	if (kfile_ismybuf(file))
	   return(NULL);

	if (bufsiz) *bufsiz = file->bufsiz;
	return(file->buf);
}
