/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Transport Machine Independent Reading Utilities
   >>>>
   >>>>  Private:
   >>>>             read_convert
   >>>>   Public:
   >>>>             kread_bit()
   >>>>             kread_byte()
   >>>>             kread_short()
   >>>>             kread_int()
   >>>>             kread_long()
   >>>>             kread_float()
   >>>>             kread_double()
   >>>>             kread_ubyte()
   >>>>             kread_ushort()
   >>>>             kread_uint()
   >>>>             kread_ulong()
   >>>>             kread_complex()
   >>>>             kread_dcomplex()
   >>>>             kread_string()
   >>>>	       	    kread_generic()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"	


/*-----------------------------------------------------------
|
|  Routine Name: read_convert - private read and conversion routine
|
|       Purpose: The read_convert routine is the grunt routine that
|		 does the actual reading of data as well calls the
|		 appropriate conversion routine. 
|
|         Input: kfile - the device in which we will read the data from
|                ptr   - the data array in which we will be storing the
|			 converted data
|                src_esize  - the source element size
|                dest_esize - the destination element size
|                num        - the number of elements to be read and converted
|                converter  - the routine that converts the data for that
|			      routine.
|
|        Output: None
|       Returns: number of converted elements on success, -1 otherwise
|
|    Written By: Mark Young  
|          Date: Jun 14, 1992
| Modifications:
|
------------------------------------------------------------*/

static int read_convert(
   kfile         *file,
   kaddr         ptr,
   int           src_esize,
   int           dest_esize,
   int           num,
   unsigned char *(*converter)())
{
	int	 count = 0;
	kaddr	 tmp = ptr;
	long	 src_mach, dest_mach;
	
	static   int size = 0;
	static   kaddr data = NULL;

	src_mach  = file->machtype;
	dest_mach = kmach_type(NULL);
	if (src_esize != dest_esize)
	{
	   if (size < num*src_esize)
	   {
	      size = num*src_esize;
	      data = (kaddr) krealloc(data, size);
	   }
	   tmp = data;
	}

	/*
	 *  Read the data
	 */
	count = kfread(tmp, src_esize, num, file);
	if (count > 0 && src_mach != dest_mach)
	{
	   if (!converter(tmp, ptr, src_mach, dest_mach, count))
	      return(0);
	}
	return(count);
}


/************************************************************
*  
*    Routine Name: kread_bit - read an array of bits
*         Purpose: This module is used to read an array of bits.
*		   The data will need to contain enough memory to
*		   store "num" bits.
*  
*		   (note: bits are machine independent and therefore
*		    require no conversion.  This routine is included to
*		    complete the set of read utilities)
*           Input: id   - the file id to be read which was opened earlier
*		          with kopen().
*		   num  - the number of bits to be read in the data array.
*	   Output: data - the char array in which the data will be stored.
*         Returns: the number of bits read
*      Written By: Mark Young  
*            Date: Jun 14, 1992
*************************************************************/

int kread_bit(
   int  id,
   unsigned char *data,
   int  num)
{
	kfile   *file;
	int	val, src, dest;


        /*
         *  Get the file entry associated with the id
         */
        if ((file = kfile_checkid(id)) == NULL)
        {
           errno = EBADF;
           return(-1);
        }

	/*
	 *  Get the source and destination element sizes for signed bytes for
	 *  the local (destination) and remote (source) machine types.
	 */
	src  = kmach_sizeof(file->machtype, KUBYTE);
	dest = kmach_sizeof(kmach_type(NULL), KUBYTE);

	/*
	 *  Call read_convert to do the actual work of reading and converting
	 *  the data to the local machine type.
	 */
	val = (num+7)/8;
	val = read_convert(file, (kaddr) data, src, dest, val, convert_byte);
	return(kmin(val*8, num));
}

/************************************************************
*  
*    Routine Name: kread_byte - read an array of signed bytes
*         Purpose: This module is used to read an array of signed
*		   bytes.  The data will need to contain enough
*		   memory to store "num" signed bytes.
*  
*		   (note: bytes are machine independent and therefore
*		    require no conversion.  This routine is included to
*		    complete the set of read utilities)
*  
*           Input: id   - the file id to be read which was opened earlier
*		          with kopen().
*		   num  - the number of signed bytes to be read in the
*		  	  data array.
*	   Output: data - the char array in which the data will be stored.
*         Returns: the number of signed bytes read
*      Written By: Mark Young  
*            Date: Jun 14, 1992
*************************************************************/

int kread_byte(
   int  id,
   char *data,
   int  num)
{
	kfile   *file;
	int	src, dest;


        /*
         *  Get the file entry associated with the id
         */
        if ((file = kfile_checkid(id)) == NULL)
        {
           errno = EBADF;
           return(-1);
        }

	/*
	 *  Get the source and destination element sizes for signed bytes for
	 *  the local (destination) and remote (source) machine types.
	 */
	src  = kmach_sizeof(file->machtype, KBYTE);
	dest = kmach_sizeof(kmach_type(NULL), KBYTE);

	/*
	 *  Call read_convert to do the actual work of reading and converting
	 *  the data to the local machine type.
	 */
	return(read_convert(file,(kaddr) data, src, dest, num, convert_byte));
}

/************************************************************
*  
*    Routine Name: kread_short - read an array of signed shorts
*         Purpose: This module is used to read an array of signed
*		   shorts.  The data will need to contain enough
*		   memory to store "num" signed shorts.
*           Input: id   - the file id to be read which was opened earlier
*		        with kopen().
*		   num  - the number of signed shorts to be read in the
*		  	data array.
*	   Output: data - the short array in which the data will be stored.
*         Returns: the number of short integers read
*      Written By: Mark Young  
*            Date: Jun 11, 1992
*************************************************************/

int kread_short(
   int  id,
   short *data,
   int  num)
{
	kfile   *file;
	int	src, dest;


        /*
         *  Get the file entry associated with the id
         */
        if ((file = kfile_checkid(id)) == NULL)
        {
           errno = EBADF;
           return(-1);
        }

	/*
	 *  Get the source and destination element sizes for signed bytes for
	 *  the local (destination) and remote (source) machine types.
	 */
	src  = kmach_sizeof(file->machtype, KSHORT);
	dest = kmach_sizeof(kmach_type(NULL), KSHORT);

	/*
	 *  Call read_convert to do the actual work of reading and converting
	 *  the data to the local machine type.
	 */
	return(read_convert(file,(kaddr) data, src, dest, num, convert_short));
}

/************************************************************
*  
*    Routine Name: kread_int - read an array of signed ints
*         Purpose: This module is used to read an array of signed
*		   ints.  The data will need to contain enough
*		   memory to store "num" signed ints.
*           Input: id   - the file id to be read which was opened earlier
*		        with kopen().
*		   num  - the number of signed integers to be read in the
*		  	  data array.
*	   Output: data - the integer array in which the data will be stored.
*         Returns: the number of integers read
*      Written By: Mark Young  
*            Date: Jun 11, 1992
*************************************************************/

int kread_int(
   int  id,
   int *data,
   int  num)
{
	kfile   *file;
	int	src, dest;


        /*
         *  Get the file entry associated with the id
         */
        if ((file = kfile_checkid(id)) == NULL)
        {
           errno = EBADF;
           return(-1);
        }

	/*
	 *  Get the source and destination element sizes for signed bytes for
	 *  the local (destination) and remote (source) machine types.
	 */
	src  = kmach_sizeof(file->machtype, KINT);
	dest = kmach_sizeof(kmach_type(NULL), KINT);

	/*
	 *  Call read_convert to do the actual work of reading and converting
	 *  the data to the local machine type.
	 */
	return(read_convert(file,(kaddr) data, src, dest, num, convert_int));
}

/************************************************************
*  
*    Routine Name: kread_long - read an array of signed longs
*         Purpose: This module is used to read an array of signed
*		   longs.  The data will need to contain enough
*		   memory to store "num" signed longs.
*           Input: id   - the file id to be read which was opened earlier
*		        with kopen().
*		   num  - the number of signed longs to be read in the
*		  	data array.
*	   Output: data - the long array in which the data will be stored.
*         Returns: the number of longs read
*      Written By: Mark Young  
*            Date: Jun 11, 1992
*************************************************************/

int kread_long(
   int  id,
   long *data,
   int  num)
{
	kfile   *file;
	int	src, dest;


        /*
         *  Get the file entry associated with the id
         */
        if ((file = kfile_checkid(id)) == NULL)
        {
           errno = EBADF;
           return(-1);
        }

	/*
	 *  Get the source and destination element sizes for signed bytes for
	 *  the local (destination) and remote (source) machine types.
	 */
	src  = kmach_sizeof(file->machtype, KLONG);
	dest = kmach_sizeof(kmach_type(NULL), KLONG);

	/*
	 *  Call read_convert to do the actual work of reading and converting
	 *  the data to the local machine type.
	 */
	return(read_convert(file,(kaddr) data, src, dest, num, convert_long));
}

/************************************************************
*  
*    Routine Name: kread_float - read an array of floats
*         Purpose: This module is used to read an array of floats.
*		   The data will need to contain enough memory to store
*		   "num" floats.
*           Input: id   - the file id to be read which was opened earlier
*		        with kopen().
*		   num  - the number of floats to be read in the
*		  	data array.
*	   Output: data - the float array in which the data will be stored.
*         Returns: the number of floats read
*      Written By: Mark Young  
*            Date: Jun 11, 1992
*************************************************************/

int kread_float(
   int  id,
   float *data,
   int  num)
{
	kfile   *file;
	int	src, dest;


        /*
         *  Get the file entry associated with the id
         */
        if ((file = kfile_checkid(id)) == NULL)
        {
           errno = EBADF;
           return(-1);
        }

	/*
	 *  Get the source and destination element sizes for floats for
	 *  the local (destination) and remote (source) machine types.
	 */
	src  = kmach_sizeof(file->machtype, KFLOAT);
	dest = kmach_sizeof(kmach_type(NULL), KFLOAT);

	/*
	 *  Call read_convert to do the actual work of reading and converting
	 *  the data to the local machine type.
	 */
	return(read_convert(file,(kaddr) data, src, dest, num, convert_float));
}

/************************************************************
*  
*    Routine Name: kread_double - read an array of doubles
*         Purpose: This module is used to read an array of doubles.
*		   The data will need to contain enough memory to store
*		   "num" doubles.
*           Input: id   - the file id to be read which was opened earlier
*		          with kopen().
*		   num  - the number of doubles to be read in the data array.
*	   Output: data - the double array in which the data will be stored.
*         Returns: the number of doubles read
*      Written By: Mark Young  
*            Date: Jun 11, 1992
*************************************************************/

int kread_double(
   int  id,
   double *data,
   int  num)
{
	kfile   *file;
	int	src, dest;


        /*
         *  Get the file entry associated with the id
         */
        if ((file = kfile_checkid(id)) == NULL)
        {
           errno = EBADF;
           return(-1);
        }

	/*
	 *  Get the source and destination element sizes for doubles for
	 *  the local (destination) and remote (source) machine types.
	 */
	src  = kmach_sizeof(file->machtype, KDOUBLE);
	dest = kmach_sizeof(kmach_type(NULL), KDOUBLE);

	/*
	 *  Call read_convert to do the actual work of reading and converting
	 *  the data to the local machine type.
	 */
	return(read_convert(file,(kaddr) data, src, dest, num, convert_double));
}

/************************************************************
*  
*    Routine Name: kread_ubyte - read an array of unsigned bytes
*  
*         Purpose: This module is used to read an array of unsigned
*		   bytes.  The data will need to contain enough
*		   memory to store "num" unsigned bytes.
*  
*		   (note: bytes are machine independent and therefore
*		    require no conversion.  This routine is included to
*		    complete the set of read utilities)
*           Input: id   - the file id to be read which was opened earlier
*		          with kopen().
*		   num  - the number of unsigned bytes to be read in the
*		  	  data array.
*	   Output: data - the unsigned char array in which the data will be
*			  stored.
*         Returns: the number of bytes read
*      Written By: Mark Young  
*            Date: Jun 11, 1992
*************************************************************/

int kread_ubyte(
   int  id,
   unsigned char *data,
   int  num)
{
	kfile   *file;
	int	src, dest;


        /*
         *  Get the file entry associated with the id
         */
        if ((file = kfile_checkid(id)) == NULL)
        {
           errno = EBADF;
           return(-1);
        }

	/*
	 *  Get the source and destination element sizes for unsigned bytes
	 *  for the local (destination) and remote (source) machine types.
	 */
	src  = kmach_sizeof(file->machtype, KUBYTE);
	dest = kmach_sizeof(kmach_type(NULL), KUBYTE);

	/*
	 *  Call read_convert to do the actual work of reading and converting
	 *  the data to the local machine type.
	 */
	return(read_convert(file,(kaddr) data, src, dest, num, convert_ubyte));
}

/************************************************************
*  
*    Routine Name: kread_ushort - read an array of unsigned shorts
*         Purpose: This module is used to read an array of unsigned
*		   shorts.  The data will need to contain enough
*		   memory to store "num" unsigned shorts.
*           Input: id   - the file id to be read which was opened earlier
*		          with kopen().
*		   num  - the number of unsigned shorts to be read in the
*		  	  data array.
*	   Output: data - the unsigned short array in which the data will be
*			  stored.
*         Returns: the number of shorts read
*      Written By: Mark Young  
*            Date: Jun 11, 1992
*************************************************************/

int kread_ushort(
   int  id,
   unsigned short *data,
   int  num)
{
	kfile   *file;
	int	src, dest;


        /*
         *  Get the file entry associated with the id
         */
        if ((file = kfile_checkid(id)) == NULL)
        {
           errno = EBADF;
           return(-1);
        }

	/*
	 *  Get the source and destination element sizes for unsigned shorts
	 *  for the local (destination) and remote (source) machine types.
	 */
	src  = kmach_sizeof(file->machtype, KUSHORT);
	dest = kmach_sizeof(kmach_type(NULL), KUSHORT);

	/*
	 *  Call read_convert to do the actual work of reading and converting
	 *  the data to the local machine type.
	 */
	return(read_convert(file,(kaddr) data, src, dest, num, convert_ushort));
}

/************************************************************
*  
*    Routine Name: kread_uint - read an array of unsigned ints
*         Purpose: This module is used to read an array of unsigned
*		   ints.  The data will need to contain enough
*		   memory to store "num" unsigned ints.
*           Input: id   - the file id to be read which was opened earlier
*		          with kopen().
*		   num  - the number of unsigned ints to be read in the
*		  	  data array.
*	   Output: data - the unsigned integer array in which the data will be
*			  stored.
*         Returns: the number of integers read
*      Written By: Mark Young  
*            Date: Jun 11, 1992
*************************************************************/

int kread_uint(
   int  id,
   unsigned int *data,
   int  num)
{
	kfile   *file;
	int	src, dest;


        /*
         *  Get the file entry associated with the id
         */
        if ((file = kfile_checkid(id)) == NULL)
        {
           errno = EBADF;
           return(-1);
        }

	/*
	 *  Get the source and destination element sizes for unsigned ints
	 *  for the local (destination) and remote (source) machine types.
	 */
	src  = kmach_sizeof(file->machtype, KUINT);
	dest = kmach_sizeof(kmach_type(NULL), KUINT);

	/*
	 *  Call read_convert to do the actual work of reading and converting
	 *  the data to the local machine type.
	 */
	return(read_convert(file,(kaddr) data, src, dest, num, convert_uint));
}

/************************************************************
*  
*    Routine Name: kread_ulong - read an array of unsigned longs
*         Purpose: This module is used to read an array of unsigned
*		   longs.  The data will need to contain enough
*		   memory to store "num" unsigned longs.
*           Input: id   - the file id to be read which was opened earlier
*		          with kopen().
*		   num  - the number of unsigned longs to be read in the
*		  	  data array.
*	   Output: data - the unsigned long array in which the data will be
*			  stored.
*         Returns: the number of longs read
*      Written By: Mark Young  
*            Date: Jun 11, 1992
*************************************************************/

int kread_ulong(
   int  id,
   unsigned long *data,
   int  num)
{
	kfile   *file;
	int	src, dest;


        /*
         *  Get the file entry associated with the id
         */
        if ((file = kfile_checkid(id)) == NULL)
        {
           errno = EBADF;
           return(-1);
        }

	/*
	 *  Get the source and destination element sizes for unsigned longs
	 *  for the local (destination) and remote (source) machine types.
	 */
	src  = kmach_sizeof(file->machtype, KULONG);
	dest = kmach_sizeof(kmach_type(NULL), KULONG);

	/*
	 *  Call read_convert to do the actual work of reading and converting
	 *  the data to the local machine type.
	 */
	return(read_convert(file,(kaddr) data, src, dest, num, convert_ulong));
}

/************************************************************
*  
*    Routine Name: kread_complex - read an array of complex
*         Purpose: This module is used to read an array of complex
*		   numbers.  The data will need to contain enough
*		   memory to store "num" complex structures.
*           Input: id   - the file id to be read which was opened earlier
*		          with kopen().
*		   num  - the number of complex to be read in the data array.
*	   Output: data - the kcomplex array in which the data will be stored.
*         Returns: the number of complex read
*      Written By: Mark Young  
*            Date: Jul 20, 1992 16:32
*************************************************************/

int kread_complex(
   int  id,
   kcomplex *data,
   int  num)
{
	kfile   *file;
	int	src, dest;


        /*
         *  Get the file entry associated with the id
         */
        if ((file = kfile_checkid(id)) == NULL)
        {
           errno = EBADF;
           return(-1);
        }

	/*
	 *  Get the source and destination element sizes for unsigned longs
	 *  for the local (destination) and remote (source) machine types.
	 */
	src  = kmach_sizeof(file->machtype, KCOMPLEX);
	dest = kmach_sizeof(kmach_type(NULL), KCOMPLEX);

	/*
	 *  Call read_convert to do the actual work of reading and converting
	 *  the data to the local machine type.
	 */
	return(read_convert(file,(kaddr) data, src, dest, num,
				convert_complex));
}

/************************************************************
*  
*    Routine Name: kread_dcomplex - read an array of double complex
*         Purpose: This module is used to read an array of double complex
*		   numbers.  The data will need to contain enough
*		   memory to store "num" double complex structures.
*           Input: id   - the file id to be read which was opened earlier
*		          with kopen().
*		   num  - the number of double complex to be read in the
*		  	  data array.
*	   Output: data - the kdcomplex array in which the data will be stored.
*         Returns: the number of double complex read
*      Written By: Mark Young  
*            Date: Jul 20, 1992 16:32
*************************************************************/

int kread_dcomplex(
   int  id,
   kdcomplex *data,
   int  num)
{
	kfile   *file;
	int	src, dest;


        /*
         *  Get the file entry associated with the id
         */
        if ((file = kfile_checkid(id)) == NULL)
        {
           errno = EBADF;
           return(-1);
        }

	/*
	 *  Get the source and destination element sizes for unsigned longs
	 *  for the local (destination) and remote (source) machine types.
	 */
	src  = kmach_sizeof(file->machtype, KDCOMPLEX);
	dest = kmach_sizeof(kmach_type(NULL), KDCOMPLEX);

	/*
	 *  Call read_convert to do the actual work of reading and converting
	 *  the data to the local machine type.
	 */
	return(read_convert(file,(kaddr) data, src, dest, num,
			convert_dcomplex));
}

/************************************************************
*  
*    Routine Name: kread_string - read an array of strings
*         Purpose: This module is used to read an array of strings.  The
*		   data will need to contain enough memory to store "num"
*		   string pointers.  The  memory that the strings
*		   are stored in will be allocated by kread_string.
*
*		   Where "kstring" is really a typedef for a character
*		   pointer "char *".  So an array of kstring is really
*		   an array of character pointers.
*           Input: id   - the file id to be read which was opened earlier
*		         with kopen().
*		   num  - the number of string to be read in the
*		  	  data array.
*	   Output: data - the kstring array in which the data will be stored.
*         Returns: the number of strings read
*      Written By: Mark Young  
*            Date: Dec 13, 1993
*    Side Effects: the strings that are returned should be freed by the
*		   programmer
*************************************************************/

int kread_string(
   int  id,
   kstring *data,
   int  num)
{
	kfile   *file;
	int	i, j, character, done;

	static int  size = 0;
	static char *temp = NULL;


        /*
         *  Get the file entry associated with the id
         */
        if ((file = kfile_checkid(id)) == NULL)
        {
           errno = EBADF;
           return(-1);
        }

	for (i = 0; i < num; i++)
	{
	   done = FALSE; j = 0;
	   do
	   {
	      if (j == size)
	      {
	         size += KLENGTH;
		 temp = krealloc(temp, size);
	      }

	      character = kfgetc(file);
	      if (character == '\0' || character == -1)
	      {
	         temp[j] = '\0';
		 done = TRUE;
	      }
	      else
	         temp[j++] = character;

	   } while (!done);
	   data[i] = kstrdup(temp);
	}
	return(i);
}

/************************************************************
*  
*    Routine Name: kread_generic - read in any data type.
*         Purpose: This module is used to read an array of 
*      	       	   data from a transport.  The data will need 
*      	       	   to contain enough memory to store "num" 
*      	       	   data points of the specified type.
*           Input: id   - the file id to be read which was
*      	       	       	  opened earlier with kopen().
*		   num  - the number of data points to be
*      	       	       	  read in the data array.
*      	       	   type - data type.
*	   Output: data - the kaddr array in which the data will be stored.
*         Returns: the number of points read
*      Written By: Jeremy Worley, Mark Young
*            Date: Sep 04, 1992
*************************************************************/

int kread_generic(
   int  id,
   kaddr data,
   int  num,
   int  type)
{
	kfile   *file;
	int	src, dest;
        unsigned char *(*converter)();

        /*
         *  Get the file entry associated with the id
         */
        if ((file = kfile_checkid(id)) == NULL)
        {
           errno = EBADF;
           return(-1);
        }

	/*
	 *  Get the source and destination element sizes for unsigned longs
	 *  for the local (destination) and remote (source) machine types.
	 */
	src  = kmach_sizeof(file->machtype, type);
	dest = kmach_sizeof(kmach_type(NULL), type);

	/*
	 *  Set the correct converter according to the specified data
	 *  type.
	 */
        switch(type){
       	   case KBYTE: 	   converter = convert_byte;     break;
       	   case KUBYTE:	   converter = convert_ubyte;    break;
       	   case KSHORT:	   converter = convert_short;    break;
       	   case KUSHORT:   converter = convert_ushort;   break;
       	   case KLONG: 	   converter = convert_long;     break;
       	   case KULONG:	   converter = convert_ulong;    break;
       	   case KINT:  	   converter = convert_int;      break;
       	   case KUINT: 	   converter = convert_uint;     break;
       	   case KFLOAT:	   converter = convert_float;    break;
       	   case KDOUBLE:   converter = convert_double;   break;
       	   case KCOMPLEX:  converter = convert_complex;  break;
       	   case KDCOMPLEX: converter = convert_dcomplex; break;
       	   case KBIT: 	   return(kread_bit(id, data, num));
       	   case KSTRING:   return(kread_string(id, data, num));
       	   default:    	   return(-1);
       	}

	/*
	 *  Call read_convert to do the actual work of reading and converting
	 *  the data to the local machine type.
	 */
	return(read_convert(file,(kaddr) data, src, dest, num, converter));
}

/************************************************************
*  
*    Routine Name: kfread_generic - read in any data type.
*         Purpose: This module is used to read an array of 
*      	       	   data from a transport.  The data will need 
*      	       	   to contain enough memory to store "num" 
*      	       	   data points of the specified type.
*           Input: file - the file to be read which was 
*      	       	       	  opened earlier with kfopen().
*		   num  - the number of data points to be 
*      	       	       	  read in the data array.
*      	       	   type - data type.
*	   Output: data - the kaddr array in which the data will be stored.
*         Returns: the number of points read
*      Written By: Jeremy Worley, Mark Young
*            Date: Sep 04, 1992 12:36
*************************************************************/

int kfread_generic(
   kfile *file,
   kaddr data,
   int  num,
   int  type)
{
	return(kread_generic(kfileno(file), data, num, type));
}
