/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Stream (Named Pipe) Transport Routines
   >>>>
   >>>>  Private:
   >>>>			stream_tempnam
   >>>>			stream_open
   >>>>			stream_close
   >>>>			stream_read
   >>>>			stream_write
   >>>>			stream_access
   >>>>			stream_unlink
   >>>>			stream_lock
   >>>>			stream_descriptors
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


#if !defined(KSOCKET_DEF)
TransportInformation stream_transport[] = {NULL};
#else

static int stream_tempnam     PROTO((char *, char *, char *));
static int stream_open	      PROTO((char *, int, int, kfile *));
static int stream_close	      PROTO((kfile *));
static int stream_read	      PROTO((kfile *, char *, int));
static int stream_write	      PROTO((kfile *, char *, int));
static int stream_access      PROTO((char *, char *, int));
static int stream_unlink      PROTO((char *, char *));
static int stream_lock	      PROTO((kfile *, int));
static int stream_descriptors PROTO((kfile *, int *, int *));

TransportInformation stream_transport[] =
{
    {
	"Standard Stream (FIFO)",
	"stream",
	FALSE,
	stream_tempnam,
	stream_open,
	stream_close,
	stream_read,
	stream_write,
	NULL,
	stream_access,
	stream_unlink,
	stream_lock,
	stream_descriptors,
    }
};

/*
 *  Internal resource structures for "Standard Stream (FIFO)" transport
 */
typedef struct
{
	int	fid;
} ResourceStruct;



/*-------------------------------------------------------------------*
|  
|		  Stream Routines
|  
--------------------------------------------------------------------*/


/*-------------------------------------------------------------
|  
|  Routine Name: stream_tempnam
|  
|       Purpose: This function initializes a tempnam for a "stream"
|	         type transport.  Given a request with a filename of
|	         the following syntax:
|  
|		  	"stream=XXXXXXXXXX"
|  
|         Input:  identifier - the transport identifier
|		  template   - the filename template to be modeled
|        Output:  result     - the newly created transport name
|   Called From:  internal routine called from ktempnam()
|    Written By:  Mark Young
|          Date:  Jul 08, 1992 16:09
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int stream_tempnam(
   char *identifier,
   char *template,
   char *result)
{
	char	*temp, filename[KLENGTH];


	(void) kstrcpy(filename, template);
	if ((temp = ktmpnam(filename)) == NULL)
	   return(-1);

	/*
	 *  Create the FIFO stream if it doesn't already exist.
	 */
	if (mkfifo(temp, 0666) < 0)
	{
	   if (errno != EEXIST)
	   {
	      kinfo(KSYSLIB, "stream_tempnam:  Unable to create temporary \
stream file '%s'", temp);
	      return(-1);
	   }
	}

	if (identifier == NULL)
	   (void) sprintf(result, "stream=%s", temp);
	else
	   (void) sprintf(result,"%s=%s", identifier, temp);

	kfree(temp);
	return(0);
}

/*-------------------------------------------------------------
|  
|  Routine Name: stream_open
|  
|       Purpose: This function opens a "stream" connection.  It is the
|	         internal driver to open a stream, which is called by
|	         the user when they call kopen().  kopen() calls when
|	         it gets an open request with a filename of the following
|	         syntax:
|  
|		  	"stream=filename"
|  
|         Input: 
|        Output:  returns 0 or -1 if an error occurs
|   Called From:  internal routine called from kopen()
|    Written By:  Mark Young
|          Date:  Jul 08, 1992 16:09
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int stream_open(
   char  *path,
   int   flags,
   int   mode,
   kfile *file)
{
	int	     fid, stream_mode;
	ResourceStruct *resources;


	/*
	 *  Create the FIFO stream if it doesn't already exist.
	 */
	if (!(flags & KOPEN_CREAT))
	   stream_mode = 0666;
	else
	   stream_mode = mode & 0777;

	if (mkfifo(path, stream_mode) < 0)
	{
	   if (errno != EEXIST)
	   {
	      kinfo(KSYSLIB, "stream_open: Create Failure for stream file \
'%s'", path);
	      return(-1);
	   }
	}

	if ((fid = open(path, flags, mode)) < 0)
	{
	   kinfo(KSYSLIB, "stream_open: Open failure for stream file '%s'",
			path);
	   return(-1);
	}

	if ((resources = (ResourceStruct *) kcalloc(1,
			sizeof(ResourceStruct))) == NULL)
	{
	   kinfo(KSYSLIB, "stream_open: Not enough memory to kcalloc (%d) \
bytes for the khoros stream structure.\n", sizeof(ResourceStruct));
	   return(-1);
	}

	resources->fid = fid;
	file->resources = (kaddr) resources;
	return(0);
}

/*-------------------------------------------------------------
|  
|  Routine Name: stream_close
|  
|       Purpose: This function closes a "stream".  It is the internal driver
|	         to close a data stream, which is called by the user
|	         when they call kclose().  kclose() calls the "stream" internal
|	         drivers by the nature of the transport, which is dictated
|	         by the initial kopen().
|  
|         Input:  file - the kfile structure.
|        Output:  returns whether we were able to close the stream
|   Called From:  internal routine called from kclose()
|    Written By:  Mark Young
|          Date:  Jul 08, 1992 16:09
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int stream_close(
   kfile *file)
{
	int	     fid, status;
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	/*
	 *  Simply call close() since "kfile" is just a regular file
	 *  descriptor.
	 */
	fid = resources->fid;
	status = close(fid);
	return(status);
}

/*-------------------------------------------------------------
|  
|  Routine Name: stream_read
|  
|       Purpose: This function reads a "stream".  It is the internal driver
|	         to read data from a stream, which is called by the user
|	         when they call kread().  kread() calls the "stream" internal
|	         drivers by the nature of the transport, which is dictated
|	         by the kopen().
|  
|         Input:  file   - the kfile structure.
|		  ptr    - the pointer to store the data into.
|		  nbytes - the number of bytes to read.
|        Output:  returns the number of bytes read from the file
|   Called From:  internal routine called from kread()
|    Written By:  Mark Young
|          Date:  Jul 08, 1992 16:09
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int stream_read(
   kfile *file,
   char  *ptr,
   int   nbytes)
{
	int	     fid, count;
	int	     numread = 0;
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	/*
	 *  Simply call read() since "kfile" is just a regular file
	 *  descriptor.
	 */
	fid = resources->fid;
	do
	{
	   count = kmin(nbytes - numread, DefaultPageSize);
	   count = read(fid, (char *) (ptr + numread), count);
	   numread += count;
	}
	while (numread < nbytes && count > 0);

	return(numread);
}

/*-------------------------------------------------------------
|  
|  Routine Name: stream_write
|  
|       Purpose: This function writes to a "stream".  It is the internal driver
|	         to write data from the supplied data array to a stream, which
|	         is called by the user when they call kwrite().  kwrite() calls
|	         the "stream" internal drivers by the nature of the transport,
|	         which is dictated by the kopen().
|  
|         Input:  file   - the kfile structure.
|		  ptr    - the pointer to write the data from.
|		  nbytes - the number of bytes to read.
|        Output:  returns the number of bytes written to the kfile
|   Called From:  internal routine called from kread()
|    Written By:  Mark Young
|          Date:  Jul 08, 1992 16:09
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int stream_write(
   kfile *file,
   char  *ptr,
   int   nbytes)
{
	int	     fid, count;
	int	     numwrite = 0;
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	/*
	 *  Simply call write() since "kfile" is just a regular file
	 *  descriptor.
	 */
	fid = resources->fid;
	do
	{
	    count = kmin(nbytes - numwrite, DefaultPageSize);
	    count = write(fid, (char *) (ptr + numwrite), count);
	    numwrite += count;
	} while (numwrite < nbytes && count > 0);

	return(numwrite);
}

/*-------------------------------------------------------------
|  
|  Routine Name: stream_access
|  
|       Purpose: This function is used to do see if a file is accessable.
|	         This is called when the user calls kaccess().  kaccess()
|	         calls the "stream" internal drivers by the nature of the
|	         transport, which is dictated by the ktransport_identifier().
|  
|         Input:  path     - the initial path.
|		  filename - the filename that they wish to access
|		  mode     - the access mode information
|        Output:  returns the result
|   Called From:  internal routine called from kaccess()
|    Written By:  Mark Young
|          Date:  Jul 08, 1992 16:09
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int stream_access(
   char *path,
   char *filename,
   int  mode)
{
	int	   status;

	/*
	 *  Simply call access() since "filename" is just a stream file
	 */
	status = access(filename, mode);
	return(status);
}

/*-------------------------------------------------------------
|  
|  Routine Name: stream_unlink
|  
|       Purpose: This function initializes a tempnam for a "stream"
|	         type transport.  Given a request with a filename of
|	         the following syntax:
|  
|		  	"stream=XXXXXXXXXX"
|  
|         Input: 
|        Output:  returns 0 or -1 depending whether we
|		  sucessfully created the template
|   Called From:  internal routine called from kunlink()
|    Written By:  Mark Young
|          Date:  Jul 08, 1992 16:09
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int stream_unlink(
   char *identifier,
   char *filename)
{
	int	status;


	status = unlink(filename);
	return(status);
}


/*-------------------------------------------------------------
|  
|  Routine Name: stream_lock
|  
|       Purpose: This function locks a "stream" transport.  Given
|	         a request with a filename of the following syntax:
|
|		  	"stream=filename"
|
|         Input: 
|        Output:  returns 0 or -1 depending whether we
|		  sucessfully created the template
|   Called From:  internal routine called from kflock()
|    Written By:  Mark Young
|          Date:  Jul 08, 1992 16:09
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int stream_lock(
   kfile *file,
   int   operation)
{
	int	status;
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	status = flock(resources->fid, operation);
	return(status);
}

/*-------------------------------------------------------------
|  
|  Routine Name: stream_descriptors
|  
|       Purpose: This function shouldn't exist, but there is occasion
|	         when the user wants to know the actual file descriptor
|	         associated with the file descriptor.
|  
|         Input: file structure
|        Output:  inum - the input file descriptor
|   Called From:  internal routine called from kdescriptors()
|    Written By:  Mark Young
|          Date:  Jul 08, 1992 16:09
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int stream_descriptors(
   kfile *file,
   int   *inum,
   int   *onum)
{
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	if (inum != NULL) *inum = resources->fid;
	if (onum != NULL) *onum = resources->fid;
	return(0);
}

#endif  /* KSTREAM_DEF */
/* don't add after the endif */
