/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>              Data Manipulation Utilities
   >>>>
   >>>>  Private:
   >>>>   Static:
   >>>>   Public:      
   >>>>		kdata_arith2_ubyte
   >>>>		kdata_arith2_ulong
   >>>>		kdata_arith2_long
   >>>>		kdata_arith2_double
   >>>>		kdata_arith2_dcomplex
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

/************************************************************
*
*  Routine Name: kdata_arith2_ubyte - perform 2-input unsigned byte arithmetic 
*
*       Purpose: This function performs the operation defined by "mode"
*                using data in array "data1" as the first operand, and 
*                data in array "data2" as the second operand.  Valid
*                operations are KADD, KSUB, KSUBFROM, KMULT, KDIV, 
*                KDIVINTO, KABSDIFF, KMODULO, KMINIMUM, and KMAXIMUM.
*
*                This function assumes that both input arrays, data1 and 
*                data2, are valid, and that if either mask1 or mask2 is 
*                passed in, then both masks have been passed in.  
*
*                Potential division by zero is checked, and the maximum
*                unsigned byte (255) returned instead.
*
*         Input: mode   - operation to be performed
*                numpts - number of points in data1, mask1, data2 and mask2
*                data1  - first data array and first operand
*                data2  - second data array and second operand
*                mask1  - mask associated with first data array
*                mask2  - mask associated with second data array
*
*        Output: data1 - data results are stored in data1
*                mask1 - if a mask exists, results are stored in mask1
*       Returns: nothing
*
*  Restrictions: 
*    Written By: Mark Young & Donna Koechner
*          Date: Nov 1993
*      Verified:
*  Side Effects: None
* Modifications: May  2 1994 - Removed gating input option.
*                Added functionality for KSUBFROM and KDIVINTO cases.
*		 Feb 14 1995 - Changed name from karith2_ubyte to
*		 kdata_arith2_ubyte, moved to BOOTSTRAP::kmath, and 
*		 changed status from private to public.
*
*************************************************************/

void kdata_arith2_ubyte(
   int           mode,
   int           numpts,
   unsigned char *data1,
   unsigned char *data2,
   unsigned char *mask1,
   unsigned char *mask2 )
{
	int j;

	switch (mode)
	{
	   case KADD:
		if (!mask1 && !mask2)
		{
		   for (j = 0; j < numpts; j++)
                      data1[j] += data2[j];
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
			    data1[j] += data2[j];
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
		break;
   
           case KSUB:
		if (!mask1 && !mask2)
		{
		   for (j = 0; j < numpts; j++)
                      data1[j] -= data2[j];
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
			    data1[j] -= data2[j];
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
		break;

           case KSUBFROM:
		if (!mask1 && !mask2)
		{
		   for (j = 0; j < numpts; j++)
                      data1[j] = data2[j] - data1[j];
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
			    data1[j] = data2[j] - data1[j];
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
		break;

           case KMULT:
		if (!mask1 && !mask2)
		{
		   for (j = 0; j < numpts; j++)
                      data1[j] *= data2[j];
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
			    data1[j] *= data2[j];
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
		break;

	   case KDIV:
		if (!mask1 && !mask2)
		{
		   for (j = 0; j < numpts; j++)
                      data1[j] = 
			 (data2[j] != 0 ? data1[j]/data2[j] : 255);
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
		            data1[j] = (data2[j] != 0 ? data1[j]/data2[j] : 255);
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
		break;

	   case KDIVINTO:
		if (!mask1 && !mask2)
		{
		   for (j = 0; j < numpts; j++)
                      data1[j] = 
			 (data1[j] != 0 ? data2[j]/data1[j] : 255);
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
		            data1[j] = (data1[j] != 0 ? data2[j]/data1[j] : 255);
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
		break;

           case KABSDIFF:
		if (!mask1 && !mask2)
		{
		   for (j = 0; j < numpts; j++)
                      data1[j] = kabs((int) data1[j] - (int) data2[j]);
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
		            data1[j] = kabs((int) data1[j] - (int) data2[j]);
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
                break;

           case KMODULO:
		if (!mask1 && !mask2)
		{
		   for (j = 0; j < numpts; j++)
                      data1[j] = (data2[j] != 0 ? data1[j] % data2[j] : 0);
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
		            data1[j] = (data2[j] != 0 ? data1[j] % data2[j] : 0);
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
		break;

   	   case KMINIMUM:
		if (!mask1 && !mask2)
		{
		   for (j = 0; j < numpts; j++)
                      data1[j] = kmin(data1[j], data2[j]);
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
		            data1[j] = kmin(data1[j], data2[j]);
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
		break;

   	   case KMAXIMUM:
		if (!mask1 && !mask2)
		{
		   for (j = 0; j < numpts; j++)
                      data1[j] = kmax(data1[j], data2[j]);
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
		            data1[j] = kmax(data1[j], data2[j]);
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
   		break;

	   default:
		break;
	}
}


/************************************************************
*
*  Routine Name: kdata_arith2_ulong - perform 2-input unsigned long arithmetic 
*
*       Purpose: This function performs the operation defined by "mode"
*                using data in array "data1" as the first operand, and 
*                data in array "data2" as the second operand.  Valid
*                operations are KADD, KSUB, KSUBFROM, KMULT, KDIV, 
*                KDIVINTO, KABSDIFF, KMODULO, KMINIMUM, and KMAXIMUM.
*
*                This function assumes that both input arrays, data1 and 
*                data2, are valid, and that if either mask1 or mask2 is 
*                passed in, then both masks have been passed in.  
*
*                Potential division by zero is checked, and the maximum
*                unsigned long value [2**(sizeof(unsigned long)) -1]
*		 is returned instead.
*
*         Input: mode   - operation to be performed
*                numpts - number of points in data1, mask1, data2 and mask2
*                data1  - first data array and first operand
*                data2  - second data array and second operand
*                mask1  - mask associated with first data array
*                mask2  - mask associated with second data array
*
*        Output: data1 - data results are stored in data1
*                mask1 - if a mask exists, results are stored in mask1
*       Returns: nothing
*
*  Restrictions: 
*    Written By: Mark Young & Donna Koechner
*          Date: Nov 1993
*      Verified:
*  Side Effects: None
* Modifications: May  2 1994 - Removed gating input option.
*                Added functionality for KSUBFROM and KDIVINTO cases.
*		 Feb 14 1995 - Changed name from karith2_ulong to
*		 kdata_arith2_ulong, moved to BOOTSTRAP::kmath, and 
*		 changed status from private to public.
*
*************************************************************/

void kdata_arith2_ulong(
   int           mode,
   int           numpts,
   unsigned long *data1,
   unsigned long *data2,
   unsigned char *mask1,
   unsigned char *mask2 )
{
	int j;
	unsigned long  maxval;
	
	maxval = (unsigned long) kpow(2.0, (double)(sizeof(unsigned long))) - 1;

	switch (mode)
	{
	   case KADD:
		if (!mask1 && !mask2)
		{
		   for (j = 0; j < numpts; j++)
                      data1[j] += data2[j];
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
			    data1[j] += data2[j];
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
		break;
   
           case KSUB:
		if (!mask1 && !mask2)
		{
		   for (j = 0; j < numpts; j++)
                      data1[j] -= data2[j];
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
			    data1[j] -= data2[j];
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
		break;

           case KSUBFROM:
		if (!mask1 && !mask2)
		{
		   for (j = 0; j < numpts; j++)
                      data1[j] = data2[j] - data1[j];
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
			    data1[j] = data2[j] - data1[j];
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
		break;

           case KMULT:
		if (!mask1 && !mask2)
		{
		   for (j = 0; j < numpts; j++)
                      data1[j] *= data2[j];
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
			    data1[j] *= data2[j];
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
		break;

	   case KDIV:
		if (!mask1 && !mask2)
		{
		   for (j = 0; j < numpts; j++)
                      data1[j] = 
			 (data2[j] != 0 ? data1[j]/data2[j] : maxval);
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
		            data1[j] = (data2[j] != 0 ? data1[j]/data2[j] : maxval);
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
		break;

	   case KDIVINTO:
		if (!mask1 && !mask2)
		{
		   for (j = 0; j < numpts; j++)
                      data1[j] = 
			 (data1[j] != 0 ? data2[j]/data1[j] : maxval);
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
		            data1[j] = (data1[j] != 0 ? data2[j]/data1[j] : maxval);
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
		break;

           case KABSDIFF:
		if (!mask1 && !mask2)
		{
		   for (j = 0; j < numpts; j++)
                      data1[j] = kabs((int) data1[j] - (int) data2[j]);
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
		            data1[j] = kabs((int) data1[j] - (int) data2[j]);
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
                break;

           case KMODULO:
		if (!mask1 && !mask2)
		{
		   for (j = 0; j < numpts; j++)
                      data1[j] = (data2[j] != 0 ? data1[j] % data2[j] : 0);
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
		            data1[j] = (data2[j] != 0 ? data1[j] % data2[j] : 0);
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
		break;

   	   case KMINIMUM:
		if (!mask1 && !mask2)
		{
		   for (j = 0; j < numpts; j++)
                      data1[j] = kmin(data1[j], data2[j]);
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
		            data1[j] = kmin(data1[j], data2[j]);
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
		break;

   	   case KMAXIMUM:
		if (!mask1 && !mask2)
		{
		   for (j = 0; j < numpts; j++)
                      data1[j] = kmax(data1[j], data2[j]);
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
		            data1[j] = kmax(data1[j], data2[j]);
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
   		break;

	   default:
		break;
	}
}


/************************************************************
*
*  Routine Name: kdata_arith2_long - perform 2-input long arithmetic 
*
*       Purpose: This function performs the operation defined by "mode"
*                using data in array "data1" as the first operand, and 
*		 data in array "data2", if it is valid, as the second 
*		 operand.  If "data2" is not valid, the operation will 
*		 be performed using the value "real" as the second operand.  
*                Valid operations are KADD, KSUB, KSUBFROM, KMULT, KDIV, 
*                KDIVINTO, KABSDIFF, KMODULO, KMINIMUM, and KMAXIMUM.
*
*                This function assumes that both input arrays, data1 and 
*                data2, are valid, and that if either mask1 or mask2 is 
*                passed in, then both masks have been passed in.  If just 
*		 data1 is valid, only mask1 is expected.
*
*		 Potential division by zero is checked, and KMAXLINT is 
*		 returned instead.
*
*         Input: mode   - operation to be performed
*                numpts - number of points in data1, mask1, data2 and mask2
*                data1  - first data array and first operand
*                data2  - second data array and second operand
*                real   - second operand if data2 array is not valid
*                mask1  - mask associated with first data array
*                mask2  - mask associated with second data array
*
*        Output: data1 - data results are stored in data1
*                mask1 - if a mask exists, results are stored in mask1
*       Returns: nothing
*
*  Restrictions: 
*    Written By: Mark Young & Donna Koechner
*          Date: Nov 1993
*      Verified:
*  Side Effects: None
* Modifications: May  2 1994 - Removed gating input option.
*                Added functionality for KSUBFROM and KDIVINTO cases.
*		 Feb 14 1995 - Changed name from karith2_long to
*		 kdata_arith2_long, moved to BOOTSTRAP::kmath, and 
*		 changed status from private to public.
*
*************************************************************/

void kdata_arith2_long(
   int           mode,
   int           numpts,
   long          *data1,
   long          *data2,
   long          real,
   unsigned char *mask1,
   unsigned char *mask2 )
{
	int j;
	long temp;

	switch (mode)
	{
	   case KADD:
		if (!mask1 && !mask2)
		{
		   if (!data2)
		      for (j = 0; j < numpts; j++)
                         data1[j] += real;
		   else
		      for (j = 0; j < numpts; j++)
                         data1[j] += data2[j];
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
		      if (!mask2 || mask2[j])  /* data2 valid */
		      {
		         if (!mask1 || mask1[j])  /* data1 valid */
			    data1[j] += (!data2 ? real : data2[j]);
			 else  /* data1 invalid */
			    data1[j] = (!data2 ? data1[j] : data2[j]);
		      }
		      else  /* data2 invalid */
		      mask1[j] = 0;
		   }
		}
		break;
   
           case KSUB:
		if (!mask1 && !mask2)
		{
		   if (!data2)
		      for (j = 0; j < numpts; j++)
                         data1[j] -= real;
		   else
		      for (j = 0; j < numpts; j++)
                         data1[j] -= data2[j];
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
		            data1[j] -= (!data2 ? real : data2[j]);
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
		break;

           case KSUBFROM:
                if (!mask1 && !mask2)
                {
                   if (!data2)
                      for (j = 0; j < numpts; j++)
                         data1[j] = real - data1[j];
                   else
                      for (j = 0; j < numpts; j++)
                         data1[j] = data2[j] - data1[j];
                }
                else
                {
                   for (j = 0; j < numpts; j++)
                   {
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
                            data1[j] = (!data2 ? real-data1[j] : data2[j]-data1[j]);
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
                   }
                }
                break;


           case KMULT:
		if (!mask1 && !mask2)
		{
		   if (!data2)
		      for (j = 0; j < numpts; j++)
                         data1[j] *= real;
		   else
		      for (j = 0; j < numpts; j++)
                         data1[j] *= data2[j];
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
			    data1[j] *= (!data2 ? real : data2[j]);
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
		break;

	   case KDIV:
		if (!mask1 && !mask2)
		{
		   if (!data2)
		      for (j = 0; j < numpts; j++)
                         data1[j] = (real != 0.0 ? data1[j]/real : KMAXLINT);
		   else
		      for (j = 0; j < numpts; j++)
                         data1[j] = (data2[j] != 0.0 ? data1[j]/data2[j] : KMAXLINT);
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
		      temp = (!data2 ? real : data2[j]);
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
		            data1[j] = (temp != 0.0 ? data1[j]/temp : KMAXLINT);
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
		break;

           case KDIVINTO:
                if (!mask1 && !mask2)
                {
                   if (!data2)
                      for (j = 0; j < numpts; j++)
                         data1[j] = (data1[j] != 0.0 ? real/data1[j] : KMAXLINT);
                   else
                      for (j = 0; j < numpts; j++)
                         data1[j] = (data1[j] != 0.0 ? data2[j]/data1[j] : KMAXLINT);
                }
                else
                {
                   for (j = 0; j < numpts; j++)
                   {
                      temp = (!data2 ? real : data2[j]);
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
                            data1[j] = (data1[j] != 0.0 ? temp/data1[j] : KMAXLINT);
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
                   }
                }
                break;


           case KABSDIFF:
		if (!mask1 && !mask2)
		{
		   if (!data2)
		      for (j = 0; j < numpts; j++)
                         data1[j] = kabs(data1[j] - real);
		   else
		      for (j = 0; j < numpts; j++)
                         data1[j] = kabs(data1[j] - data2[j]);
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
                            data1[j] = kabs(data1[j] - (!data2 ? real: data2[j]));
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
                break;


           case KMODULO:
		if (!mask1 && !mask2)
		{
		   if (!data2)
		      for (j = 0; j < numpts; j++)
                         data1[j] = (real != 0 ? data1[j] % real : 0);
		   else
		      for (j = 0; j < numpts; j++)
                         data1[j] = (data2[j] != 0 ? data1[j] % data2[j] : 0);
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
		      temp = (!data2[j] ? real : data2[j]);
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
                            data1[j] = (temp != 0 ? data1[j] % temp : 0);
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
                break;

   	   case KMINIMUM:
		if (!mask1 && !mask2)
		{
		   if (!data2)
		      for (j = 0; j < numpts; j++)
                         data1[j] = kmin(data1[j], real);
		   else
		      for (j = 0; j < numpts; j++)
                         data1[j] = kmin(data1[j], data2[j]);
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
                            data1[j] = kmin(data1[j], (!data2 ? real: data2[j]));
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
   		break;

   	   case KMAXIMUM:
		if (!mask1 && !mask2)
		{
		   if (!data2)
		      for (j = 0; j < numpts; j++)
                         data1[j] = kmax(data1[j], real);
		   else
		      for (j = 0; j < numpts; j++)
                         data1[j] = kmax(data1[j], data2[j]);
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
                            data1[j] = kmax(data1[j], (!data2 ? real: data2[j]));
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
   		break;

	   default:
		break;
	}
}


/************************************************************
*
*  Routine Name: kdata_arith2_double - perform 2-input double precision arithmetic 
*
*       Purpose: This function performs the operation defined by "mode"
*                using data in array "data1" as the first operand, and 
*		 data in array "data2", if it is valid, as the second 
*		 operand.  If "data2" is not valid, the operation will 
*		 be performed using the value "real" as the second operand.  
*                Valid operations are KADD, KSUB, KSUBFROM, KMULT, KDIV, 
*                KDIVINTO, KABSDIFF, KMODULO, KMINIMUM, and KMAXIMUM.
*
*                This function assumes that both input arrays, data1 and 
*                data2, are valid, and that if either mask1 or mask2 is 
*                passed in, then both masks have been passed in.  If just 
*		 data1 is valid, only mask1 is expected.
*
*		 Potential division by zero is checked, and KMAXFLOAT is 
*		 returned instead.
*
*         Input: mode   - operation to be performed
*                numpts - number of points in data1, mask1, data2 and mask2
*                data1  - first data array and first operand
*                data2  - second data array and second operand
*                real   - second operand if data2 array is not valid
*                mask1  - mask associated with first data array
*                mask2  - mask associated with second data array
*
*        Output: data1 - data results are stored in data1
*                mask1 - if a mask exists, results are stored in mask1
*       Returns: nothing
*
*  Restrictions: 
*    Written By: Mark Young & Donna Koechner
*          Date: Nov 1993
*      Verified:
*  Side Effects: None
* Modifications: May  2 1994 - Removed gating input option.
*                Added functionality for KSUBFROM and KDIVINTO cases.
*		 Feb 14 1995 - Changed name from karith2_double to
*		 kdata_arith2_double, moved to BOOTSTRAP::kmath, and 
*		 changed status from private to public.
*
*************************************************************/

void kdata_arith2_double(
   int           mode,
   int           numpts,
   double        *data1,
   double        *data2,
   double        real,
   unsigned char *mask1,
   unsigned char *mask2 )
{
	int j;
	double temp;

	switch (mode)
	{
	   case KADD:
		if (!mask1 && !mask2)
		{
		   if (!data2)
		      for (j = 0; j < numpts; j++)
                         data1[j] += real;
		   else
		      for (j = 0; j < numpts; j++)
                         data1[j] += data2[j];
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
		      if (!mask2 || mask2[j])  /* data2 valid */
		      {
		         if (!mask1 || mask1[j])  /* data1 valid */
			    data1[j] += (!data2 ? real : data2[j]);
			 else  /* data1 invalid */
			    data1[j] = (!data2 ? data1[j] : data2[j]);
		      }
		      else  /* data2 invalid */
		      mask1[j] = 0;
		   }
		}
		break;
   
           case KSUB:
		if (!mask1 && !mask2)
		{
		   if (!data2)
		      for (j = 0; j < numpts; j++)
                         data1[j] -= real;
		   else
		      for (j = 0; j < numpts; j++)
                         data1[j] -= data2[j];
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
		            data1[j] -= (!data2 ? real : data2[j]);
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
		break;

           case KSUBFROM:
                if (!mask1 && !mask2)
                {
                   if (!data2)
                      for (j = 0; j < numpts; j++)
                         data1[j] = real - data1[j];
                   else
                      for (j = 0; j < numpts; j++)
                         data1[j] = data2[j] - data1[j];
                }
                else
                {
                   for (j = 0; j < numpts; j++)
                   {
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
                            data1[j] = (!data2 ? real-data1[j] : data2[j]-data1[j]);
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
                   }
                }
                break;


           case KMULT:
		if (!mask1 && !mask2)
		{
		   if (!data2)
		      for (j = 0; j < numpts; j++)
                         data1[j] *= real;
		   else
		      for (j = 0; j < numpts; j++)
                         data1[j] *= data2[j];
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
			    data1[j] *= (!data2 ? real : data2[j]);
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
		break;

	   case KDIV:
		if (!mask1 && !mask2)
		{
		   if (!data2)
		      for (j = 0; j < numpts; j++)
                         data1[j] = (real != 0.0 ? data1[j]/real : KMAXFLOAT);
		   else
		      for (j = 0; j < numpts; j++)
                         data1[j] = (data2[j] != 0.0 ? data1[j]/data2[j] : KMAXFLOAT);
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
		      temp = (!data2 ? real : data2[j]);
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
		            data1[j] = (temp != 0.0 ? data1[j]/temp : KMAXFLOAT);
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
		break;

           case KDIVINTO:
                if (!mask1 && !mask2)
                {
                   if (!data2)
                      for (j = 0; j < numpts; j++)
                         data1[j] = (data1[j] != 0.0 ? real/data1[j] : KMAXFLOAT);
                   else
                      for (j = 0; j < numpts; j++)
                         data1[j] = (data1[j] != 0.0 ? data2[j]/data1[j] : KMAXFLOAT);
                }
                else
                {
                   for (j = 0; j < numpts; j++)
                   {
                      temp = (!data2 ? real : data2[j]);
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
                            data1[j] = (data1[j] != 0.0 ? temp/data1[j] : KMAXFLOAT);
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
                   }
                }
                break;

           case KPOW:
		if (!mask1 && !mask2)
		{
		   if (!data2)
		      for (j = 0; j < numpts; j++)
                         data1[j] = kpow(data1[j], real);
		   else
		      for (j = 0; j < numpts; j++)
                         data1[j] = kpow(data1[j], data2[j]);
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
                            data1[j] = kpow(data1[j], (!data2 ? real: data2[j]));
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
		break;

           case KABSDIFF:
		if (!mask1 && !mask2)
		{
		   if (!data2)
		      for (j = 0; j < numpts; j++)
                         data1[j] = kabs(data1[j] - real);
		   else
		      for (j = 0; j < numpts; j++)
                         data1[j] = kabs(data1[j] - data2[j]);
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
                            data1[j] = kabs(data1[j] - (!data2 ? real: data2[j]));
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
                break;

           case KMODULO:
		if (!mask1 && !mask2)
		{
		   if (!data2)
		      for (j = 0; j < numpts; j++)
                         data1[j] = kfmod(data1[j], real);
		   else
		      for (j = 0; j < numpts; j++)
                         data1[j] = kfmod(data1[j], data2[j]);
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
                            data1[j] = kfmod(data1[j], (!data2 ? real: data2[j]));
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
                break;

           case KATAN2:
		if (!mask1 && !mask2)
		{
		   if (!data2)
		   {
		      for (j = 0; j < numpts; j++)
		      {
			 temp = (real != 0.0 ? data1[j]/real : KMAXFLOAT);
                         data1[j] = katan(temp);
		      }
		   }
		   else
		   {
		      for (j = 0; j < numpts; j++)
		      {
			 temp=(data2[j] != 0.0 ? data1[j]/data2[j] : KMAXFLOAT);
                         data1[j] = katan(temp);
		      }
		   }
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
		         {
                            data1[j] += (!data2 ? real : data2[j]);
			    temp = (!data2 ? real : data2[j]);
			    temp = (temp != 0.0 ? data1[j]/temp : KMAXFLOAT);
                            data1[j] = katan(temp);
			 }
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
                break;
   
   	   case KHYPOT:
		if (!mask1 && !mask2)
		{
		   if (!data2)
		      for (j = 0; j < numpts; j++)
                          data1[j] = khypot(data1[j], real);
		   else
		      for (j = 0; j < numpts; j++)
                          data1[j] = khypot(data1[j], data2[j]);
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
                            data1[j] = khypot(data1[j], (!data2 ? real: data2[j]));
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
   		break;

   	   case KMINIMUM:
		if (!mask1 && !mask2)
		{
		   if (!data2)
		      for (j = 0; j < numpts; j++)
                         data1[j] = kmin(data1[j], real);
		   else
		      for (j = 0; j < numpts; j++)
                         data1[j] = kmin(data1[j], data2[j]);
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
                            data1[j] = kmin(data1[j], (!data2 ? real: data2[j]));
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
   		break;

   	   case KMAXIMUM:
		if (!mask1 && !mask2)
		{
		   if (!data2)
		      for (j = 0; j < numpts; j++)
                         data1[j] = kmax(data1[j], real);
		   else
		      for (j = 0; j < numpts; j++)
                         data1[j] = kmax(data1[j], data2[j]);
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
                            data1[j] = kmax(data1[j], (!data2 ? real: data2[j]));
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
   		break;

   	   case KLDEXP:
		if (!mask1 && !mask2)
		{
		   if (!data2)
		      for (j = 0; j < numpts; j++)
                         data1[j] = kldexp(data1[j], (int) real);
		   else
		      for (j = 0; j < numpts; j++)
                         data1[j] = kldexp(data1[j], (int) data2[j]);
		}
		else
		{
		   for (j = 0; j < numpts; j++)
		   {
                      if (!mask2 || mask2[j])  /* data2 valid */
                      {
                         if (!mask1 || mask1[j])  /* data1 valid */
                            data1[j] = kldexp(data1[j], (!data2 ? (int) real: (int) data2[j]));
                         else  /* data1 invalid */
                            data1[j] = (!data2 ? data1[j] : data2[j]);
                      }
                      else  /* data2 invalid */
                         mask1[j] = 0;
		   }
		}
   		break;


	   default:
		break;
	}
}


/************************************************************
*
*  Routine Name: kdata_arith2_dcomplex - perform 2-input double complex arithmetic 
*
|		 
|
*       Purpose: This function performs the operation defined by "mode"
*                using data in array "data1" as the first operand, and 
*		 data in array "data2", if it is valid, as the second 
*		 operand.  If "data2" is not valid, the operation will 
*		 be performed using the value "complex" as the second 
*                operand.  Valid operations are KADD, KSUB, KSUBFROM, 
*		 KMULT, KDIV, KDIVINTO, and KPOW.
*
*                This function assumes that both input arrays, data1 and 
*                data2, are valid, and that if either mask1 or mask2 is 
*                passed in, then both masks have been passed in.  If just 
*		 data1 is valid, only mask1 is expected.
*
*		 Potential division by zero is checked, and KMAXFLOAT is 
*		 returned instead.
*
*         Input: mode   - operation to be performed
*                numpts - number of points in data1, mask1, data2 and mask2
*                data1  - first data array and first operand
*                data2  - second data array and second operand
*                complex - second operand if data2 array is not valid
*                mask1  - mask associated with first data array
*                mask2  - mask associated with second data array
*
*        Output: data1 - data results are stored in data1
*                mask1 - if a mask exists, results are stored in mask1
*       Returns: nothing
*
*  Restrictions: 
*    Written By: Mark Young & Donna Koechner
*          Date: Nov 1993
*      Verified:
*  Side Effects: None
* Modifications: May  2 1994 - Removed gating input option.
*                Added functionality for KSUBFROM and KDIVINTO cases.
*		 Feb 14 1995 - Changed name from karith2_dcomplex to
*		 kdata_arith2_dcomplex, moved to BOOTSTRAP::kmath, and 
*		 changed status from private to public.
*		 March 14, 1995 - added double complex power function
*
*************************************************************/

void kdata_arith2_dcomplex(
   int           mode,
   int 	         numpts,
   kdcomplex     *data1,
   kdcomplex     *data2,
   kdcomplex     complex,
   unsigned char *mask1,
   unsigned char *mask2 )
{
	int j;

	switch (mode)
	{
	   case KADD:
		for (j = 0; j < numpts; j++)
		{
                   if (!mask2 || mask2[j])  /* data2 valid */
                   {
                      if (!mask1 || mask1[j])  /* data1 valid */
		         data1[j] = kdcadd(data1[j], (!data2 ? complex :data2[j]));
                      else  /* data1 invalid */
                         data1[j] = (!data2 ? data1[j] : data2[j]);
                   }
                   else  /* data2 invalid */
                      mask1[j] = 0;
		}
		break;
   
           case KSUB:
		for (j = 0; j < numpts; j++)
		{
                   if (!mask2 || mask2[j])  /* data2 valid */
                   {
                      if (!mask1 || mask1[j])  /* data1 valid */
		         data1[j] = kdcsub(data1[j], (!data2 ? complex :data2[j]));
                      else  /* data1 invalid */
                         data1[j] = (!data2 ? data1[j] : data2[j]);
                   }
                   else  /* data2 invalid */
                      mask1[j] = 0;
		}
		break;

           case KSUBFROM:
		for (j = 0; j < numpts; j++)
		{
                   if (!mask2 || mask2[j])  /* data2 valid */
                   {
                      if (!mask1 || mask1[j])  /* data1 valid */
		         data1[j] = kdcsub((!data2 ? complex :data2[j]), data1[j]);
                      else  /* data1 invalid */
                         data1[j] = (!data2 ? data1[j] : data2[j]);
                   }
                   else  /* data2 invalid */
                      mask1[j] = 0;
		}
		break;

           case KMULT:
		for (j = 0; j < numpts; j++)
		{
                   if (!mask2 || mask2[j])  /* data2 valid */
                   {
                      if (!mask1 || mask1[j])  /* data1 valid */
		         data1[j] = kdcmult(data1[j],(!data2 ? complex :data2[j]));
                      else  /* data1 invalid */
                         data1[j] = (!data2 ? data1[j] : data2[j]);
                   }
                   else  /* data2 invalid */
                      mask1[j] = 0;
		}
		break;

	   case KDIV:
		for (j = 0; j < numpts; j++)
		{
                   if (!mask2 || mask2[j])  /* data2 valid */
                   {
                      if (!mask1 || mask1[j])  /* data1 valid */
		         data1[j] = kdcdiv(data1[j],(!data2 ? complex :data2[j]));
                      else  /* data1 invalid */
                         data1[j] = (!data2 ? data1[j] : data2[j]);
                   }
                   else  /* data2 invalid */
                      mask1[j] = 0;
		}
		break;

	   case KDIVINTO:
		for (j = 0; j < numpts; j++)
		{
                   if (!mask2 || mask2[j])  /* data2 valid */
                   {
                      if (!mask1 || mask1[j])  /* data1 valid */
		         data1[j] = kdcdiv((!data2 ? complex :data2[j]), data1[j]);
                      else  /* data1 invalid */
                         data1[j] = (!data2 ? data1[j] : data2[j]);
                   }
                   else  /* data2 invalid */
                      mask1[j] = 0;
		}
		break;

	   case KPOW:
		for (j = 0; j < numpts; j++)
		{
                   if (!mask2 || mask2[j])  /* data2 valid */
                   {
                      if (!mask1 || mask1[j])  /* data1 valid */
                         data1[j] = kdcpow(data1[j], (!data2 ? complex: data2[j]));
                      else  /* data1 invalid */
                         data1[j] = (!data2 ? data1[j] : data2[j]);
                   }
                   else  /* data2 invalid */
                      mask1[j] = 0;
		}
		break;
	   default:
		break;
	}
}

