/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>		       Routines to Print UIS                  <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>                kvf_print_UIS()                        <<<<
   >>>>                kvf_print_names()                      <<<<
   >>>>                kvf_print_subform()                    <<<<
   >>>>                kvf_print_pane()                       <<<<
   >>>>   Static:                                             <<<<
   >>>>                kvf_print_selections()                 <<<<
   >>>>                kvf_print_UIS_line()                   <<<<
   >>>>                kvf_print_indent()                     <<<<
   >>>>                kvf_print_selection_names()            <<<<
   >>>>                kvf_print_group()                      <<<<
   >>>>                                                       <<<<
   >>>>                kvf_print_subformfile()                <<<<
   >>>>                kvf_print_panefile()                   <<<<
   >>>>                kvf_retrieve_subform_filename()        <<<<
   >>>>                kvf_retrieve_pane_filename()           <<<<
   >>>>   Public:                                             <<<<
   >>>>                kvf_print_UIS_file()                   <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


static void kvf_print_selections       PROTO((kfile *, kselection *, int));
static kselection *kvf_sort_selections PROTO((kselection *, int));
static void kvf_print_group            PROTO((kfile *, kselection *, int));
static void kvf_print_UIS_line         PROTO((kfile *, char *));
static void kvf_print_indent           PROTO((kfile *, int));
static void kvf_print_selection_names  PROTO((kselection *));

#define SORT_ON_Y 1
#define SORT_ON_X 2

/************************************************************
*
*  Routine Name: kvf_print_UIS_file - print GUI to UIS file
*
*       Purpose: Prints the current GUI, exactly as it appears on 
*                the screen at the time that this routine is called, 
*                to a corresponding UIS file.  Prompts to over-write if
*                the filename specified already exists, and pops up an 
*                info message when the UIS file has been output.
*
*         Input: form     - pointer to the form tree associated with
*                           the GUI to be written to a UIS file.
*                filename - name of the file in which to print UIS.
*                force    - TRUE if forcing over-write (suppress prompting)
*                make_backups - TRUE if making backups of overwritten files
*
*        Output: None.
*	Returns: TRUE on success, FALSE on failure
*  Restrictions:
*    Written By: Danielle Argiro
*          Date: Jul 16, 1992 10:39
*      Verified: 
*  Side Effects:
* Modifications:
*
*************************************************************/

int kvf_print_UIS_file(
   kform *form,
   char  *filename,
   int   force,
   int   make_backups)
{
        kfile *file;
        char  temp[KLENGTH];

        if (form == NULL) return(FALSE);

        if (!force)
        {
            if (!(koverwrite(KSTANDARD, filename)))
                return(FALSE);
        }

        if (make_backups)
        {
            sprintf(temp, "%s.old", filename);
            kcopyfile(filename, temp);
        }

        if ((file = kfopen(filename, "w")) == NULL)
        {
            kerror("kforms", "kvf_print_UIS",
                   "Cannot write to output UIS file %s", filename);
            return(FALSE);
        }

        kvf_print_UIS(file, form, force, make_backups);
        kfclose(file);

        if (!force) 
	    kinfo(KSTANDARD,  "UIS written to file %s", filename);
        return(TRUE);
}

/*------------------------------------------------------------
|
|  Routine Name: kvf_print_UIS
|
|       Purpose: Prints out the current UIS as specified
|                by the form tree to the open file specified.
|
|         Input: file         - open stream of output file.
|                form         - pointer to the form tree
|                force        - TRUE if forcing over-write (suppress prompting)
|                make_backups - TRUE if making backups of overwritten files
|        Output:
|	Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: Oct 03, 1992 
| Modifications:
|
------------------------------------------------------------*/

int kvf_print_UIS(
   kfile *file,
   kform *form,
   int   force,
   int   make_backups)
{
	kfile      *subform_file;
	ksubform   *subform;
	int        indent_num;
	char       temp[KLENGTH]; 
	char       *subform_filename;

	/* print -F line */
	indent_num = 0;
	if (form->control_comment != NULL)
	    kfprintf(file, "%s", form->control_comment);
	kfprintf(file, "%s\n", form->control_line);
	indent_num += 2;

	/* 
	 * print out -S to -E Master Form def (if one) 
 	 */
	if (form->master != NULL)
	{
	    if (form->master->control_comment != NULL)
	    {
	        kvf_print_indent(file, indent_num);
	        kfprintf(file, "%s", form->master->control_comment); 
	    } 
	    kvf_print_indent(file, indent_num);
	    kfprintf(file, "%s\n", form->master->control_line);
	    indent_num +=2;
	    subform = form->master->subform_list;

	    /*
	     *  print selections on master
	     */
	    form->master->sel_list = 
		kvf_sort_selections(form->master->sel_list, SORT_ON_X);
	    form->master->sel_list = 
		kvf_sort_selections(form->master->sel_list, SORT_ON_Y);
	    kvf_print_selections(file, form->master->sel_list, indent_num);

	    /*
	     *  print subform item if not part of SubMenu def
	     *  (if it is, it will be printed in kvf_print_selections)
	     */
	    while (subform != NULL)
	    {
		if (subform->back_submenu == NULL)
		{
		    if (subform->comment != NULL)
		    {
	                kvf_print_indent(file, indent_num);
	                kfprintf(file, "%s", subform->comment);
		    }
	            kvf_print_indent(file, indent_num);
		    kvf_print_UIS_line(file, subform->line);
		}
		subform = subform->next;
	    }

	    indent_num -= 2;
	    kvf_print_indent(file, indent_num);
	    kfprintf(file, "-E\n");
	}


	/*
	 *  Go thru all subforms, printing UIS definitions 
	 */
	if (form->master != NULL)
	    subform = form->master->subform_list;
	else subform = form->subform;

	while (subform != NULL)
	{
	    subform_filename = kvf_retrieve_subform_filename(subform);
	    if (subform_filename != NULL)
	    {
		/* print the -k line to the main UIS file */
		kvf_print_indent(file, indent_num);
		kfprintf(file, "-k %s\n", subform_filename);

	        if ((force) || (koverwrite(KSTANDARD, subform_filename)))
	        {
		    if (make_backups)
		    {
			    ksprintf(temp, "%s.old", subform_filename);
			    kcopyfile(subform_filename, temp);
		    }
		    subform_file = kfopen(subform_filename, "w");
                    if (subform_file == NULL)
                    {
                            kerror("kforms", "kvf_print_UIS",
                                   "Cannot write to output UIS file %s", 
			  	   subform_filename);
                            return(FALSE);
                    }
	            kvf_print_subform(subform_file, subform, indent_num, 
			 	          TRUE, force, make_backups);
		    kfclose(subform_file);
		}
		kfree(subform_filename);
	    }
	    else kvf_print_subform(file, subform, indent_num, 
				    TRUE, force, make_backups);
	    subform = subform->next;
	}
	indent_num -= 2;
	kvf_print_indent(file, indent_num);
	kfprintf(file, "-E\n");  /* end UIS def */
	return(TRUE);
}


/*-----------------------------------------------------------
|
i  Routine Name: kvf_print_subform
|
|       Purpose: Prints out the UIS lines associated with a subform;
|                may print to a separate UIS file specified by a
|                -k line, or into the same file as the rest of it.
|
|         Input: file       - open stream of output file (main UIS or -k file).
|                subform    - pointer to the subform
|                indent_num - number of spaces to indent into file
|                interpret  - TRUE if control_comment is to be interpreted
|                             for inclusion of -k line (if FALSE, 
|                             control_comment will not be printed: Bug)
|                force      - TRUE if forcing over-write (suppress prompting)
|                make_backups - TRUE if making backups of overwritten files
|        Output: 
|	Returns: 
|    Written By: Danielle Argiro
|          Date: May 4, 1992
| Modifications:
|
-------------------------------------------------------------*/
void kvf_print_subform(
    kfile    *file,
    ksubform *subform,
    int      indent_num,
    int      interpret,
    int      force,
    int      make_backups)
{
	kguide   *guide;
	kcontrol *pane;
	kfile    *pane_file;
	char     temp[KLENGTH]; 
	char     *tmp1, *tmp2, *pane_filename;
	int      status, separate_file = FALSE;

	/*  print out -M line */
	if ((subform->control_comment != NULL) && (interpret))
	{
	   /*
            *  the 'secret' comment line "@-k {filename}" needs to be
	    *  taken out, any valid comments following it printed out,
            *  and the "@-F ..." UIS line printed w/o the "@"
            */
	    if (kstrncmp(subform->control_comment, "@-k", 3) == 0)
	    {
		/* tmp1 points to whatever follows "@-k {filename}\n" */
		tmp1 = kparse_string_search(subform->control_comment,
			        "^@-k\\s+.*$", KLITERAL, NULL, &status);
		separate_file = TRUE;

		/* tmp2 points 3 letters into the "@-F" string */ 
		tmp2 = kparse_string_search(tmp1, "^@-F", KLITERAL, NULL, 
					    &status);

	        /* temp should contain the valid comment, if one */
		kstring_ncopy(tmp1, tmp2 - tmp1 - 3, temp); 
		kfprintf(file, "%s", temp);
                tmp2--; tmp2--;

	        /* print the -F line to the file */
		kfprintf(file, "%s", tmp2);
	    }
	    else 
	    {
	        kvf_print_indent(file, indent_num);
	        kfprintf(file, "%s", subform->control_comment);
	    }
	}
	kvf_print_indent(file, indent_num);
	kfprintf(file, "%s\n", subform->control_line);
	indent_num +=2;

	/* 
	 * print out [-G to -E] Guidepane definition (if one) 
         */
	if (subform->guidepane != NULL)
	{
	    /* print out -G line */
	    if (subform->guidepane->control_comment != NULL)
	    {
	        kvf_print_indent(file, indent_num);
	        kfprintf(file, "%s", subform->guidepane->control_comment);
	    }
	    kvf_print_indent(file, indent_num);
	    kfprintf(file, "%s\n", subform->guidepane->control_line);
	    indent_num += 2;

	    /* print any other selections on guide pane */
	    subform->guidepane->sel_list = 
		kvf_sort_selections(subform->guidepane->sel_list, SORT_ON_X);
	    subform->guidepane->sel_list = 
		kvf_sort_selections(subform->guidepane->sel_list, SORT_ON_Y);

	    kvf_print_selections(file, subform->guidepane->sel_list, 
				 indent_num);

	    /*
	     *  print guide button if not part of SubMenu def
	     *  (if it is, it will be printed in kvf_print_selections)
	     */
	    guide = subform->guidepane->guide_list;
	    while (guide != NULL) 
	    {
	        if (guide->back_submenu == NULL)
		{
		    if (guide->comment != NULL)
		    {
	                kvf_print_indent(file, indent_num);
	            	kfprintf(file, "%s", guide->comment);
		    }
	            kvf_print_indent(file, indent_num);
		    kvf_print_UIS_line(file, guide->line);
		}
		guide = guide->next;
	    }

	    /* end the Guidepane definition */
	    indent_num -= 2;
	    kvf_print_indent(file, indent_num);
	    kfprintf(file, "-E\n");

	    guide = subform->guidepane->guide_list;
	}
	else guide = subform->guide;

	/*
	 *  Go thru all panes, printing UIS definitions 
	 */
	while (guide != NULL) 
	{
	    if (guide->pane != NULL)
	    {
		pane = guide->pane;
		pane_filename =  kvf_retrieve_pane_filename(pane);
	        if (pane_filename != NULL)
		{
                    /* print the -p line to the main UIS file */
                    kvf_print_indent(file, indent_num);
                    kfprintf(file, "-p %s\n", pane_filename);

		    if ((force) || (koverwrite(KSTANDARD, pane_filename)))
                    {
                        if (make_backups)
                        {
                            sprintf(temp, "%s.old", pane_filename);
                            kcopyfile(pane_filename, temp);
                        }
			pane_file = kfopen(pane_filename, "w");
                        if (pane_file == NULL)
                        {
                            kerror("kforms", "kvf_print_UIS", 
                                   "Cannot write to output UIS file %s",
                                   pane_filename);
                            return;
                        }
                        kvf_print_pane(pane_file, pane, indent_num, TRUE);
                        kfclose(pane_file);
		    }
		    kfree(pane_filename);
		}
                else kvf_print_pane(file, pane, indent_num, TRUE);
	    }
	    guide = guide->next;
	}
	indent_num -= 2;
	kvf_print_indent(file, indent_num);
	kfprintf(file, "-E\n");  /* end subform def */

	if (separate_file)
	    kfprintf(file, "-E\n");  /* end form def */
}


/*-----------------------------------------------------------
|
|  Routine Name: kvf_print_pane
|
|       Purpose: Prints out the UIS lines associated with a pane;
|                may print to a separate UIS file specified by a
|                -p line, or into the same file as the rest of it.
|
|         Input: file       - open stream of output file (main UIS or -p file).
|                pane       - pointer to the pane
|                indent_num - number of spaces to indent into file
|                interpret  - TRUE if control_comment is to be interpreted
|                             for inclusion of -p line (if FALSE, 
|                             control_comment will not be printed: Bug)
|        Output: 
|	Returns: 
|
|        Output: none
|
|    Written By: Danielle Argiro
|          Date: May 4, 1992
| Modifications:
|
-------------------------------------------------------------*/
void kvf_print_pane(
    kfile    *file,
    kcontrol *pane,
    int      indent_num,
    int      interpret)
{
	char       *tmp1, *tmp2;
	char       temp[KLENGTH];
	int        status, separate_file = FALSE;

	if ((pane->control_comment != NULL) && (interpret))
        {
           /*
            *  the 'secret' comment line "@-p {filename}" needs to be
            *  taken out, any valid comments following it printed out,
            *  and the "@-F ..." and "@-M ..." UIS lines printed w/o the "@"
            */
            if (kstrncmp(pane->control_comment, "@-p", 3) == 0)
            {
                /* tmp1 points to whatever follows "@-p {filename}\n" */
                tmp1 = kparse_string_search(pane->control_comment,
                                "^@-p\\s+.*$", KLITERAL, NULL, &status);
                separate_file = TRUE;

                /* tmp2 points 3 letters into the "@-F" string */
                tmp2 = kparse_string_search(tmp1, "^@-F", KLITERAL, NULL,
                                            &status);

                /* temp should contain the valid comment, if one */
                kstring_ncopy(tmp1, tmp2 - tmp1 - 3, temp);
                kfprintf(file, "%s", temp);

                tmp2--; tmp2--;
                /* tmp1 points 3 letters into the "@-M" string */
                tmp1 = kparse_string_search(tmp2, "^@-M", KLITERAL, NULL,
                                            &status);

		/* temp should contain the -F UIS line*/
                kstring_ncopy(tmp2, tmp1 - tmp2 - 3, temp);
                kfprintf(file, "%s", temp);

		tmp1--; tmp1--;
                /* print the -M line to the file */
                kfprintf(file, "  %s", tmp1);
            }
            else
	    {
                kvf_print_indent(file, indent_num);
                kfprintf(file, "%s", pane->control_comment);
	    }
        }
        kvf_print_indent(file, indent_num);
        kvf_print_UIS_line(file, pane->control_line);
        indent_num += 2;
	pane->sel_list = kvf_sort_selections(pane->sel_list, SORT_ON_X);
	pane->sel_list = kvf_sort_selections(pane->sel_list, SORT_ON_Y);
        kvf_print_selections(file, pane->sel_list, indent_num);
        indent_num -= 2;
        kvf_print_indent(file, indent_num);
        kfprintf(file, "-E\n");  /* end pane def */

	if (separate_file)
	{
	    indent_num -= 2;
            kvf_print_indent(file, indent_num);
            kfprintf(file, "-E\n");  /* end subform def */
	    indent_num -= 2;
            kvf_print_indent(file, indent_num);
            kfprintf(file, "-E\n");  /* end form def */
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: kvf_print_selections
|
|       Purpose: Prints out the UIS lines associated with selections
|                in the specified selection list.
|
|         Input: file       - open stream to output file
|                selection  - header of selection list
|                indent_num - number of spaces to indent UIS lines
|
|        Output: none
|
|    Written By: Danielle Argiro
|          Date: May 4, 1992
| Modifications:
|
-------------------------------------------------------------*/

static void kvf_print_selections(
   kfile      *file,
   kselection *selection,
   int        indent_num)
{
	kselection *toggle;
	kform_struct    *kformstruct;

	/*
	 *  Go thru all selections, printing UIS lines 
	 */
	while (selection != NULL) 
	{
	    
	    /*
	     *  print out UIS lines for selections - if part of SubMenu
             *  definition, line will be printed out later.
	     */
	    if (selection->back_submenu == NULL)
	    {
	        if (selection->comment != NULL)
	        {
	            kvf_print_indent(file, indent_num);
	            kfprintf(file, "%s", selection->comment);
	        }
	        kvf_print_indent(file, indent_num);
	        kvf_print_UIS_line(file, selection->line);
	    }

	    /*
	     *  print out SubMenu def if applicable (-D line written out above)
	     */
	    if (selection->type == KUIS_STARTSUBMENU) 
	    { 
            	kformstruct = selection->submenu_next;
		indent_num += 2;
            	while (kformstruct != NULL)
            	{
               	    if (kformstruct->type == KSELECTION)
		    {
	    		if (kformstruct->Selptr->comment != NULL)
	    		{
	        	    kvf_print_indent(file, indent_num);
	                    kfprintf(file, "%s", kformstruct->Selptr->comment);
	    	        }
	                kvf_print_indent(file, indent_num);
	                kvf_print_UIS_line(file, kformstruct->Selptr->line);
		    }
                    else if (kformstruct->type == KSUBFORM)
		    {
	    		if (kformstruct->Subformptr->comment != NULL)
	    		{
	        	    kvf_print_indent(file, indent_num);
	                    kfprintf(file, "%s", kformstruct->Subformptr->comment);
	    	        }
	                kvf_print_indent(file, indent_num);
	                kvf_print_UIS_line(file, kformstruct->Subformptr->line);
		    }
                    else if (kformstruct->type == KGUIDE)
		    {
	    		if (kformstruct->Guideptr->comment != NULL)
	    		{
	        	    kvf_print_indent(file, indent_num);
	                    kfprintf(file, "%s", kformstruct->Guideptr->comment);
	    	        }
	                kvf_print_indent(file, indent_num);
	                kvf_print_UIS_line(file, kformstruct->Guideptr->line);
		    }
                    kformstruct = kformstruct->next;
                }
		indent_num -= 2;
	        kvf_print_indent(file, indent_num);
                kfprintf(file, "-E\n");
            }

	    /*
	     *  print out -T to -E toggle def if applicable 
	     */
	    else if (selection->toggle_next != NULL)
	    {
	    	indent_num += 2;
		toggle = selection->toggle_next;
		while(toggle != NULL)
		{
		    if (toggle->comment != NULL)
		    {
	                kvf_print_indent(file, indent_num);
	                kfprintf(file, "%s", toggle->comment);
		    }
	            kvf_print_indent(file, indent_num);
	            kvf_print_UIS_line(file, toggle->line);
		    toggle = toggle->next;
		}
		indent_num -= 2;
	        kvf_print_indent(file, indent_num);
		kfprintf(file, "-E\n");
	    }

	    /*
	     *  print out group definition if applicable 
	     */
	    else if (selection->group_next != NULL)
	    {
	 	indent_num += 2;
	        kvf_print_group(file, selection->group_next, indent_num);
	        indent_num -= 2;
 	        kvf_print_indent(file, indent_num);
		kfprintf(file, "-E\n");
	    }
	    selection = selection->next;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: kvf_print_group
|
|       Purpose: Prints out the UIS lines associated with a 
|                Mutually Exclusive or Mutually Inclusive group.
|
|         Input: file       - open stream to output file
|                group      - header of group list
|                indent_num - number of spaces to indent UIS lines
|
|        Output: none
|    Written By: Danielle Argiro
|          Date: May 4, 1992
| Modifications:
|
---------------------------------------------------------------*/

static void kvf_print_group(
   kfile      *file,
   kselection *group,
   int        indent_num)
{
	kselection *toggle;

	while (group != NULL)
	{
	    /* print group member */
	    if (group->comment != NULL)
	    {
	        kvf_print_indent(file, indent_num);
	        kfprintf(file, "%s", group->comment);
	    }
	    kvf_print_indent(file, indent_num);
	    kvf_print_UIS_line(file, group->line);

	    /* print toggle w/in group */
	    if (group->toggle_next != NULL)
            {
                indent_num += 2;
                toggle = group->toggle_next;
                while(toggle != NULL)
                {
                    if (toggle->comment != NULL)
                    {
                        kvf_print_indent(file, indent_num);
                        kfprintf(file, "%s", toggle->comment);
                    }
                    kvf_print_indent(file, indent_num);
                    kvf_print_UIS_line(file, toggle->line);
                    toggle = toggle->next;
                }
                indent_num -= 2;
                kvf_print_indent(file, indent_num);
                kfprintf(file, "-E\n");
            }

	    /* print group w/in group */
	    if (group->group_next != NULL)
	    {
		indent_num += 2;
		kvf_print_group(file, group->group_next, indent_num);
		indent_num -= 2;
		kvf_print_indent(file, indent_num);
                kfprintf(file, "-E\n");
	    }
	    group = group->next;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: kvf_print_UIS_line
|
|       Purpose: Prints out a single UIS line
|
|         Input: file - open stream to output file
|                line - UIS line to print
|        Output: none
|    Written By: Danielle Argiro
|          Date: May 4, 1992
| Modifications:
|
--------------------------------------------------------------*/

static void kvf_print_UIS_line(
   kfile *file,
   char  *line)
{
	Line_Info line_info;
	char *tmp_line = NULL;
	
	kvf_clear_line_info(&line_info);
	kvf_gen_parse(line, &line_info);
	line_info.write = TRUE;
	if ((line_info.typeflag != KUIS_GUIDEBUTTON) &&
	    (line_info.typeflag != KUIS_SUBFORMBUTTON))
	    line_info.selected = FALSE;
	kvf_gen_deparse(&line_info, &tmp_line);
	kvf_free_line_info_strings(&line_info);
	kfprintf(file, "%s\n", tmp_line);
	kfree(tmp_line);
}

/*-----------------------------------------------------------
|
|  Routine Name: kvf_print_indent
|
|       Purpose: Prints out the specified number of spaces
|                for proper indentation in the UIS file
|
|         Input: file - open stream to output file
|                num  - number of spaces to indent
|        Output: none
|    Written By: Danielle Argiro
|          Date: May 4, 1992
| Modifications:
|
--------------------------------------------------------------*/
static void kvf_print_indent(
   kfile *file,
   int   num)
{
	int i;
	for (i = 0; i < num; i++) kfprintf(file, " ");
}


/*------------------------------------------------------------
|
|  Routine Name: kvf_print_names
|
|       Purpose: Prints out the unique names associated with each 
|                item in the GUI; used for debugging purposes.
|
|         Input: form - pointer to the form tree
|        Output: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: May 1, 1992
| Modifications:
|
-------------------------------------------------------------*/

void kvf_print_names(
   kform *form)
{
	ksubform  *subform;
	kguide     *guide;

	kfprintf(kstderr, "form full name is : %s\n", 
		 ktoken_to_string(form->var_token));

	if (form->master != NULL)
	{
	   subform = form->master->subform_list;
	   kvf_print_selection_names(form->master->sel_list);
	}
	else
	   subform = form->subform;

	while (subform != NULL)
	{
	    /* 
	     * print subform names
	     */
	    kfprintf(kstderr, "subform full name is: %s\n", 
		     ktoken_to_string(subform->name_token));

	    if  (subform->guidepane != NULL)
	    {
	        guide = subform->guide;
	        kvf_print_selection_names(subform->guidepane->sel_list);
	    }
	    else
	        guide = subform->guide;
		
	    while (guide != NULL) 
	    {
	        if (guide->pane != NULL)
	        {
			
	            kfprintf(kstderr, "pane full name is: %s\n", 
			     ktoken_to_string(guide->pane->name_token));
		    kvf_print_selection_names(guide->pane->sel_list); 
	        }
	        guide = guide->next;
	    }
	    subform = subform->next;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: kvf_print_selection_names
|
|       Purpose: Prints out the names associated with each object 
|                in a selection list.
|
|         Input: selection - header of the selection list
|        Output: none
|
|    Written By: Danielle Argiro
|          Date: May 4, 1992
| Modifications:
|
-------------------------------------------------------------*/

static void kvf_print_selection_names(
   kselection *selection)
{
	kselection *toggle;

	/*
	 *  Go thru all selections, clearing UIS lines 
	 */
	while (selection != NULL) 
	{
	    kfprintf(kstderr, "selection full name is: %s\n", 
		     ktoken_to_string(selection->name_token));

	    /*
	     *  -T to -E toggle def if applicable 
	     */
	    if (selection->toggle_next != NULL)
	    {
	         toggle = selection->toggle_next;
	         while(toggle != NULL)
	         {
		     kfprintf(kstderr, "toggle full name is: %s\n", 
			      ktoken_to_string(toggle->name_token));
		     toggle = toggle->next;
		 }
	    }
	    selection = selection->next;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: kvf_sort_selections
|
|       Purpose: Sorts the selections in a selection list based
|                on their geometry - first on y, then on x.
|
|         Input: sel_list  - header of selection list
|                sort_type - 1: Sort on Y values
|                sort_type - 2: Sort on X values
|        Output: none
|       Returns: none
|    Written By: Danielle Argiro
|          Date: Oct 21, 1993
| Modifications:
|
-------------------------------------------------------------*/

static kselection *kvf_sort_selections(
    kselection *sel_list,
    int         sort_type)
{
	float      least_y, least_x;
	Line_Info  line_info;
	kselection *tmpsel1, *tmpsel2, *prev;
	kselection *save_before, *save_after, *move_sel;
	if (sel_list == NULL) return(NULL);

	kvf_clear_line_info(&line_info);
	tmpsel1     = sel_list;
	save_after  = NULL;
	save_before = NULL;
	prev        = NULL;
	while (tmpsel1 != NULL)
	{
	    tmpsel2     = tmpsel1;
	    move_sel    = tmpsel1;
	    least_y     = KMAXFLOAT;
	    least_x     = KMAXFLOAT;

	    while (tmpsel2 != NULL)
	    {
	        kvf_gen_parse(tmpsel2->line, &line_info);
	        if ((sort_type == SORT_ON_Y) && 
                    (line_info.y < least_y)  &&
		    (tmpsel2->back_submenu == NULL))
	        {
		    move_sel    = tmpsel2;
		    save_before = prev;
		    least_y     = line_info.y;
	        }
		else if ((sort_type == SORT_ON_X) &&
                         (line_info.x < least_x)  &&
		         (tmpsel2->back_submenu == NULL))
                {
                    move_sel    = tmpsel2;
                    save_before = prev;
                    least_x     = line_info.x;
                }
	        kvf_free_line_info_strings(&line_info);
	        prev = tmpsel2;
	        tmpsel2 = tmpsel2->next;
	    }
    
	    if (save_before != NULL)
	        save_before->next = move_sel->next;
	    else sel_list = move_sel->next; 
    
	    if (save_after != NULL) 
	    {
	        move_sel->next = save_after->next;
	        save_after->next = move_sel;
            }
	    else if (move_sel != sel_list)
	    {
	        move_sel->next = sel_list;
		sel_list = move_sel;
	    }
	    save_after  = move_sel;
	    save_before = move_sel;
	    prev        = move_sel;

	    if (move_sel->group_next != NULL)
		move_sel->group_next = 
			kvf_sort_selections(move_sel->group_next, sort_type);
	    if (move_sel->toggle_next != NULL)
                move_sel->toggle_next = 
                        kvf_sort_selections(move_sel->toggle_next, sort_type);

	    tmpsel1     = move_sel->next;
	}
	return(sel_list);
}

/*-----------------------------------------------------------
|
|  Routine Name: kvf_print_subformfile
|
|       Purpose: Prints out a subform (*.sub) UIS file, given
|                a filename and a subform
|
|         Input: subform_filename - name of *.sub UIS file to write out
|                subform          - subform to write to UIS file
|                force            - TRUE to force over-write, FALSE to prompt
|        Output: none
|       Returns: none
|    Written By: Danielle Argiro
|          Date: Feb 28, 1994
| Modifications:
|
-------------------------------------------------------------*/

int kvf_print_subformfile(
    ksubform *subform,
    int      force)
{
        kfile    *file;
	char     tbpath[KLENGTH];
	char     *subformpath = NULL;

	if (!(kvf_get_subform_file(subform->back_kformstruct, 
				   KVF_SUBFORM_FILE, &subformpath)))
            return(FALSE);

        if (subformpath == NULL) return(FALSE);

        tbpath[0] = '\0';
        (void) ktbpath(subformpath, tbpath);
        if (kstrlen(tbpath) != 0)
        {
            if ((file = kfopen(tbpath, "w")) == NULL)
            {
                kerror("kforms", "kvf_print_subformfile",
                       "Cannot write to output UIS file %s", tbpath);
                return(FALSE);
            }
        }
        else
        {
            sprintf(tbpath, "%s", subformpath);
            if ((file = kfopen(subformpath, "w")) == NULL)
            {
                kerror("kforms", "kvf_print_subformfile",
                       "Cannot write to output UIS file %s", subformpath);
                return(FALSE);
            }
        }

        kfprintf(file, "%s\n", subform->back_form->control_line);

        kvf_print_subform(file, subform, 2, FALSE, force, FALSE);

        kfprintf(file, "-E\n"); /* end form def */

        kfclose(file);

        kinfo(KSTANDARD, "UIS for subform written to %s", tbpath);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: kvf_print_panefile
|
|       Purpose: Prints out a pane (*.pane) UIS file, given
|                a filename and a pane
|
|         Input: pane_filename - name of *.pane UIS file to write out
|                pane          - pane to write to UIS file
|                force         - TRUE to force over-write
|        Output: none
|       Returns: none
|    Written By: Danielle Argiro
|          Date: Feb 28, 1994
| Modifications:
|
-------------------------------------------------------------*/

int kvf_print_panefile(
    kcontrol *pane,
    int      force)
{
	kfile        *file;
	char         *panepath = NULL;

	if (!(kvf_get_pane_file(pane->back_kformstruct, 
				KVF_PANE_FILE, &panepath)))
	    return(FALSE);

	if (panepath == NULL) return(FALSE);
	
        if ((force) || (koverwrite(KSTANDARD, panepath)))
	{
	    file = kfopen(panepath, "w");
            if (file == NULL)
            {
                kerror("kforms", "kvf_print_panefile",
                        "Cannot write to output UIS file %s", panepath);
                return(FALSE);
            }
	}
	else return(TRUE);

        kfprintf(file, "%s\n", pane->back_form->control_line);
        kfprintf(file, "  %s\n", pane->back_subform->control_line);

        kvf_print_pane(file, pane, 4, FALSE);

        kfprintf(file, "  -E\n"); /* end subform def */
        kfprintf(file, "-E\n"); /* end form def */

        kfclose(file);

        kinfo(KSTANDARD, "UIS for pane written to %s", panepath);
	return(TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: kvf_retrieve_subform_filename
|
|       Purpose: If there is a *.subform file associated with this subform,
|                the comment of the subform will read, "@-k {filename}"
|                where filename is the path to the *.subform file.  This
|                routine sees if the subform comment has this entry, and
|                if so, returns the path to the *.subform file as it was
|                stored.  Returns NULL otherwise.
|
|         Input: subform - pointer to subform for which filename is to be 
|                retrieved
|        Output: none
|       Returns: the path to the *.subform file if there is a *.subform file
|                associated with the subform and stored properly in the subform 
|                comment, NULL otherwise
|    Written By: Danielle Argiro
|          Date: June 10, 1994
| Modifications:
|
-------------------------------------------------------------*/

char *kvf_retrieve_subform_filename(
    ksubform *subform)
{
	char subform_filename[KLENGTH];
	int  status;

	/*
	 *  if there is a *.subform file associated with the subform,
         *  that information will be stored in the subform comment.
	 */
	if (subform->control_comment == NULL)
	    return(NULL);

	/*
	 *  if there is a *.subform file associated with the subform,
         *  the subform comment will read "@-k {filename}"  where filename
         *  is the path to the *.subform file
         */
	subform_filename[0] = '\0';
	if (kstrncmp(subform->control_comment, "@-k", 3) == 0)
        {
            /* pull the filename out of '@-k ~user/path.../filename */
                    (void) kparse_string_search(subform->control_comment,
                                "^@-k\\s+(.*)$", KLITERAL, NULL, &status);
            kre_subs("\\1", subform_filename);
	}

	if (kstrlen(subform_filename) > 0)
	    return(kstrdup(subform_filename));
	else return(NULL);

}


/*-----------------------------------------------------------
|
|  Routine Name: kvf_retrieve_pane_filename
|
|       Purpose: If there is a *.pane file associated with this pane,
|                the comment of the pane will read, "@-p {filename}"
|                where filename is the path to the *.pane file.  This
|                routine sees if the pane comment has this entry, and
|                if so, returns the path to the *.pane file as it was
|                stored.  Returns NULL otherwise.
|
|         Input: pane - pointer to pane for which filename is to be
|                retrieved
|        Output: none
|       Returns: the path to the *.pane file if there is a *.pane file
|                associated with the pane and stored properly in the pane 
|                comment, NULL otherwise
|    Written By: Danielle Argiro
|          Date: June 10, 1994
| Modifications:
|
-------------------------------------------------------------*/

char *kvf_retrieve_pane_filename(
    kcontrol *pane)
{
        char pane_filename[KLENGTH];
        int  status;

        /*
         *  if there is a *.pane file associated with the pane,
         *  that information will be stored in the pane comment.
         */
        if (pane->control_comment == NULL)
            return(NULL);

        /*
         *  if there is a *.pane file associated with the pane,
         *  the pane comment will read "@-k {filename}"  where filename
         *  is the path to the *.pane file
         */
        pane_filename[0] = '\0';
        if (kstrncmp(pane->control_comment, "@-p", 3) == 0)
        {
            /* pull the filename out of '@-p ~user/path.../filename */
                    (void) kparse_string_search(pane->control_comment,
                                "^@-p\\s+(.*)$", KLITERAL, NULL, &status);
            kre_subs("\\1", pane_filename);
        }

        if (kstrlen(pane_filename) > 0)
            return(kstrdup(pane_filename));
        else return(NULL);
}
