/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>                kform_struct routines                  <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>                kvf_create_struct_from_formname()      <<<<
   >>>>                kvf_create_struct_from_subformname()   <<<<
   >>>>                kvf_create_struct_from_guidename()     <<<<
   >>>>                kvf_create_struct_from_panename()      <<<<
   >>>>                kvf_create_struct_from_selname()       <<<<
   >>>>                                                       <<<<
   >>>>                kvf_create_formstruct()                <<<<
   >>>>                kvf_link_formstruct()                  <<<<
   >>>>                kvf_unlink_formstruct()                <<<<
   >>>>                                                       <<<<
   >>>>                kvf_get_selection_kformstructs()       <<<<
   >>>>                kvf_get_var_selection_kformstruct()    <<<<
   >>>>   Static:                                             <<<<
   >>>>                get_selection_kformstruct()            <<<<
   >>>>   Public:                                             <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


static kform_struct *get_selection_kformstruct PROTO((kselection *, int));

/*------------------------------------------------------------
|
|  Routine Name: kvf_create_struct_from_formname
|
|       Purpose: Looks at the root of the form tree for the given name 
|
|         Input: form - pointer to the form tree
|                name - name of desired form
|
|        Output: none
|       Returns: If successful, returns a generic kformstruct containing 
|                the form with the name given. Returns NULL on failure.
|          Date: Mar 04, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/

kform_struct *kvf_create_struct_from_formname(
   kform *form,
   char  *name)
{
	kform_struct *kformstruct = NULL;
	int          name_token;

	name_token = kstring_to_token(name);
	
	if (form->var_token == name_token)
            kformstruct = kvf_create_struct((kaddr) form,
                                           KUIS_STARTFORM, KFORM);

	return(kformstruct);
}

/*------------------------------------------------------------
|
|  Routine Name: kvf_create_struct_from_mastername
|
|       Purpose: Looks at the master of the form tree for the given name
|
|         Input: form - pointer to the form tree
|                name - name of desired form
|
|        Output: none
|       Returns: If successful, returns a generic kformstruct containing
|                the form with the name given. Returns NULL on failure.
|          Date: Mar 04, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/

kform_struct *kvf_create_struct_from_mastername(
   kform *form,
   char  *name)
{
        kform_struct *kformstruct = NULL;
        int          name_token;

        name_token = kstring_to_token(name);

	if (form->master == NULL)
	    return(NULL);

        if (form->master->var_token == name_token)
            kformstruct = kvf_create_struct((kaddr) form->master,
                                           KUIS_STARTMASTER, KMASTER);
        return(kformstruct);
}


/*------------------------------------------------------------
|
|  Routine Name: kvf_create_struct_from_subformname
|
|       Purpose: Searches thru all subforms in the form tree
|                looking for a subform with the given name 
|
|         Input: form - pointer to the form tree
|                name - name of desired subform
|
|        Output: none
|       Returns: If successful, returns a generic kformstruct containing 
|                the subform with the name given. Returns NULL on failure.
|
|          Date: Mar 04, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/

kform_struct *kvf_create_struct_from_subformname(
   kform *form,
   char  *name)
{
	kform_struct   *kformstruct = NULL;
	ksubform *subform;
	int          name_token;

	name_token = kstring_to_token(name);

	if (form->master != NULL)
	    subform = form->master->subform_list;
	else subform = form->subform;
	
	while (subform != NULL)
        {
	     if (subform->name_token == name_token)
	     {
                 kformstruct = kvf_create_struct((kaddr) subform,
                                               subform->type, KSUBFORM);
		 return(kformstruct);
	     }
	     subform = subform->next;
	}
	return(kformstruct);
}

/*-----------------------------------------------------------
|
|  Routine Name: kvf_create_struct_from_guidename
|
|       Purpose: Searches thru the guides on a subform
|                looking for a guide with the given name 
|
|         Input: subform - pointer to the subform branch
|                name - name of desired guide
|        Output: none
|       Returns: If successful, returns a generic kformstruct containing 
|                the guide with the name given. Returns NULL on failure.
|          Date: Mar 04, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/

kform_struct *kvf_create_struct_from_guidename(
   ksubform *subform,
   char     *name)
{
	kform_struct *kformstruct = NULL;
	kguide  *guide;
	int        name_token;

	name_token = kstring_to_token(name);

	if (subform->guidepane != NULL)
	    guide = subform->guidepane->guide_list;
	else guide = subform->guide;
	
	while (guide != NULL)
        {
	     if (guide->name_token == name_token)
	     {
                 kformstruct = kvf_create_struct((kaddr) guide,
                                               KUIS_GUIDEBUTTON, KGUIDE);
		 return(kformstruct);
	     }
	     guide = guide->next;
	}
	return(kformstruct);
}


/*-----------------------------------------------------------
|
|  Routine Name: kvf_create_struct_from_panename
|
|       Purpose: Searches thru all panes on a subform
|                looking for a panes with the given name 
|
|         Input: subform - pointer to the subform branch
|                name - name of desired pane
|
|        Output: none
|       Returns: If successful, returns a generic kformstruct containing 
|                the pane with the name given. Returns NULL on failure.
|          Date: Mar 04, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/

kform_struct *kvf_create_struct_from_panename(
   ksubform *subform,
   char     *name)
{
	kform_struct *kformstruct = NULL;
	kguide  *guide;
	int        name_token;

	name_token = kstring_to_token(name);

	if (subform->guidepane != NULL)
	    guide = subform->guidepane->guide_list;
	else guide = subform->guide;
	
	while (guide != NULL)
        {
	     if (guide->pane != NULL)
	     {
	         if (guide->pane->name_token == name_token)
		 {
                     kformstruct = kvf_create_struct((kaddr) guide->pane,
                                               KUIS_STARTPANE, KPANE);
	             return(kformstruct);
		 }
	     }
	     guide = guide->next;
	}
	return(kformstruct);
}

/*-----------------------------------------------------------
|
|  Routine Name: kvf_create_struct_from_selname
|
|       Purpose: Searches thru all selections on a control panel
|                looking for a selection with the given name 
|
|         Input: sel_list - header of selection list
|                name     - name of desired selection
|
|        Output: none
|       Returns: If successful, returns a generic kformstruct containing 
|                the selection with the name given. Returns NULL on failure.
|          Date: Mar 04, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/

kform_struct *kvf_create_struct_from_selname(
   kselection *sel_list,
   char       *name)
{
	kform_struct    *kformstruct = NULL;
	kselection *selection;
	int           name_token;

	name_token = kstring_to_token(name);

	selection = sel_list;
	
	while (selection != NULL)
        {
	     if (selection->name_token == name_token)
	     {
                 kformstruct = kvf_create_struct((kaddr) selection,
                                               selection->type, KSELECTION);
		 return(kformstruct);
	     }
	     if (selection->group_next != NULL)
	     {
		kformstruct = 
		    kvf_create_struct_from_selname(selection->group_next, name);
		if (kformstruct != NULL) return(kformstruct);
	     }
	     selection = selection->next;
	}
        return(kformstruct);
}

/*-----------------------------------------------------------
|
|  Routine Name: kvf_create_struct
|
|       Purpose: Creates a generic kforms structure based
|                on a pointer to a specific kforms structure,
|                the typeflag of the UIS line associated with
|                that structure, and the type of the specific
|                kforms structure contained.
|
|         Input: ptr  - ptr to form, subform, guide, pane, or selection
|                flag - typeflag of UIS line assoc. w/ structure
|                type - indicates type of structure provided
|
|        Output: none
|       Returns: The generic kforms structure representing
|                the form tree node passed in
|          Date: Mar 04, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/

kform_struct *kvf_create_struct(
   kaddr ptr,
   int   flag,
   int   type)
{
        kform_struct *kformstruct;

        if ((kformstruct = (kform_struct *)
			kcalloc(1, sizeof(kform_struct))) == NULL)
	{
	    kerror("kforms", "kvf_create_struct",
		    "Unable to allocate kformstruct");
	    return(NULL);
	}
        kformstruct->type = type;
        kformstruct->flag = flag;
        kformstruct->next = NULL;
        kformstruct->callback_list = NULL;
	if (ptr == NULL)
	{
	    errno = KINTERNAL;
	    kerror("kforms", "kvf_create_struct",
		   "WARNING! Creating a kformstruct w/ null pointer");
	}
        kformstruct->Formptr = (kform *) ptr;
        return(kformstruct);
}

/*-----------------------------------------------------------
|
|  Routine Name: kvf_link_formstruct
|
|       Purpose: Links a generic kforms_struct into
|                the global list of such structures that are
|                used as client_data for the various callbacks.
|                This list is maintained so that all the structures
|                may be freed when the form is destroyed.
|
|                NOTE: DON'T! link kformstucts that are used
|                      to thread together the list of submenu items!
|                      the submenu item list will get corrupted!
|
|         Input: kformstruct - ptr to the kformstruct to be added into the list
|        Output: none
|       Returns: nothing
|          Date: Jul 20, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/

void kvf_link_formstruct(
   kform_struct *kformstruct)
{
	if (kformstruct_hdr == NULL)
	    kformstruct_hdr = kformstruct;
	else if (kformstruct_hdr->next == NULL)
	{
	    kformstruct_hdr->next = kformstruct;
	    kformstruct_tail = kformstruct_hdr->next;
	}
	else
	{
	    kformstruct_tail->next = kformstruct;
	    kformstruct_tail = kformstruct_tail->next;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: kvf_unlink_formstruct
|
|       Purpose: Unlinks a generic kforms_struct out of 
|                the global list of such structures that are
|                used as client_data for the various callbacks.
|                A kformstruct that has been linked into the global
|                list must be unlinked before it can be freed.
|
|         Input: kformstruct - ptr to the kformstruct to be freed from the list
|        Output: none
|       Returns: 
|          Date: Oct 20, 1993
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/
void kvf_unlink_formstruct(
    kform_struct *kformstruct)
{
	kform_struct *kformstructptr;

	if (kformstruct_hdr == kformstruct)
	    kformstruct_hdr = kformstruct->next;
	else	 
	{
	    kformstructptr = kformstruct_hdr;
	    while ((kformstructptr != NULL) && 
		   (kformstructptr->next != kformstruct))
		kformstructptr = kformstructptr->next;
	    if (kformstructptr != NULL)
	        kformstructptr->next = kformstructptr->next->next;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: kvf_get_selection_kformstructs
|
|       Purpose: For use by cantata, returns an array of kformstructs
|                representing the backplanes of the selections in 
|                the selection list of the pane passed in.
| 
|         Input: pane - pointer to the pane structure
|                flag - KUIS_INPUTFILE, KUIS_OUTPUTFILE, KUIS_OPTIONS, or NONE
|
|                       KUIS_INPUTFILE:  returned array will have only inputs
|                                        (includes stdin selections)
|                       KUIS_OUTPUTFILE: returned array will have only outputs
|                                        (includes stdout selections)
|                       KUIS_OPTIONS:    returned array will have only the
|                                        one options button from the pane
|                       NONE:            output array will have all selections
|
|        Output: size - size of array
|       Returns: The array of kformstructs on success, NULL on failure
|    Written By: Danielle Argiro
|          Date: March 25, 1994
| Modifications:
|
------------------------------------------------------------*/
kform_struct **kvf_get_selection_kformstructs(
   kcontrol *pane,
   int      flag,
   int      *elem_num)
{
	int          size = 0;
	kselection   *selection, *group, *subgroup;
	kform_struct **kformstruct_array = NULL;
	kform_struct *kformstruct;

	selection = pane->sel_list;
	while (selection != NULL)
	{
	    kformstruct = get_selection_kformstruct(selection, flag);
	    if (kformstruct != NULL)
	    {
	        size++;
	        kformstruct_array = (kform_struct **) 
	    		    	     krealloc(kformstruct_array,
				    	      size*sizeof(kform_struct *));
	        if (kformstruct_array == NULL)
	        {
	    	    kerror("xvforms", "kvf_get_selection_kformstructs",
		           "Unable to allocate array for kformstructs");
		    return(NULL);
	        }
	        kformstruct_array[size-1] = selection->back_kformstruct;
	    }
	    if (selection->group_next != NULL)
	    {
		group = selection->group_next;
		while (group != NULL)
		{
	            kformstruct = get_selection_kformstruct(group, flag);
		    if (kformstruct != NULL)
		    {
                        size++;
	                kformstruct_array = (kform_struct **) 
			 	             krealloc(kformstruct_array,
					              size*sizeof(kform_struct *));
                        if (kformstruct_array == NULL)
                        {
                	    kerror("xvforms", "kvf_get_selection_kformstructs",
                                  "Unable to allocate array for kformstructs");
                            return(NULL);
                        }
                        kformstruct_array[size-1] = group->back_kformstruct;
	            }
		    if (group->group_next != NULL)
		    {
			subgroup = group->group_next;
			while (subgroup != NULL)
                        {
			    kformstruct = get_selection_kformstruct(subgroup, flag);
			    if (kformstruct != NULL)
		            {
			        size++;
                                kformstruct_array = (kform_struct **)
                                                     krealloc(kformstruct_array,
                                                     size*sizeof(kform_struct *));
                                if (kformstruct_array == NULL)
                                {
                                    kerror("xvforms", "kvf_get_selection_kformstructs",
                                           "Unable to allocate array for kformstructs");
                                    return(NULL);
                                }
                                kformstruct_array[size-1] = subgroup->back_kformstruct;
			    }
			    subgroup = subgroup->next;
		 	}
		    }
		    group = group->next;
		}
	    }
            selection = selection->next; 
	}
	*elem_num = size;
	return(kformstruct_array);
}

/*-----------------------------------------------------------
|
|  Routine Name: get_selection_kformstruct
|
|       Purpose: Based on the flag passed in, returns the
|                kformstruct associated with the selection if
|                the selection is of the type specified by the
|                flag;  otherwise, returns NULL.
|
|         Input: selection - ptr to the selection in question
|                flag      - flag determining type of selection wanted
|        Output: none
|       Returns: the kformstruct associated with the selection if
|                the selection matches the flag, NULL otherwise
|          Date: March 25, 1994
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/

static kform_struct *get_selection_kformstruct(
    kselection *selection,
    int        flag)
{
        int           toggletype;
        kform_struct *kformstruct = NULL;

        /*
         * if we've got an inputfile and are looking for inputfiles,
         * if we've got an outputfile and are looking for outputfiles,
         * if we've got a routine button and are looking for routine buttons,
         * or if we're looking for just anything, add selection backplane
         * into the return array.
         */
        if (  ((flag == KUIS_INPUTFILE) &&
               ((selection->type == KUIS_INPUTFILE) ||
		(selection->type == KUIS_STDIN))) ||

               ((flag == KUIS_OUTPUTFILE) &&
                ((selection->type == KUIS_OUTPUTFILE) ||
	 	 (selection->type == KUIS_STDOUT))) ||

               ((flag == KUIS_ROUTINE) &&
                (selection->type == KUIS_ROUTINE)) ||

               ((flag == NONE) &&
               ((selection->type != KUIS_MUTEXCL) &&
                (selection->type != KUIS_MUTINCL) &&
                (selection->type != KUIS_GROUP)))     )
                kformstruct = selection->back_kformstruct;

        /*
         * if we've got an options button and are looking for options buttons,
         * return the menu button (not the menu, considered the backplane)
         */
        else if ((flag == KUIS_OPTIONS) &&
                (selection->type == KUIS_STARTSUBMENU) &&
                (selection->var_token == kstring_to_token("_gui_options")))
                kformstruct = selection->back_kformstruct;

        /*
         * if we've got a inputfile toggle and are looking for inputfiles,
         * or if we've got an outputfile toggle and are looking for outputfiles,
         * or if we're looking for just anything, add toggle backplane
         * into the return array.
         */
        else if (selection->toggle_next != NULL)
        {
            toggletype = kvf_get_line_type(selection->toggle_next->line);
            if ( ((toggletype == KUIS_INPUTFILE) &&
                  (flag == KUIS_INPUTFILE))  ||
                 ((toggletype == KUIS_OUTPUTFILE) &&
                  (flag == KUIS_OUTPUTFILE))  ||
                  (flag == NONE)  )
                   kformstruct = selection->back_kformstruct;
        }

        return(kformstruct);
}

/*-----------------------------------------------------------
|
|  Routine Name: kvf_get_selection_kformstructs
|
|       Purpose: For use by cantata, returns the kformstruct
|                associated with the selection identified by
|                the variable passed in.
|
|         Input: form     - pointer to the form tree
|                variable - variable identifying desired selection
|        Output: none
|       Returns: The kformstruct of the selection on success, NULL on failure
|    Written By: Danielle Argiro
|          Date: March 25, 1994
| Modifications:
|
------------------------------------------------------------*/

kform_struct *kvf_get_var_selection_kformstruct(
    kform *form,
    char  *variable)
{
	kselection *selection;

	selection = kvf_variable_sel_search(form, variable);
	if (selection != NULL)
	    return(selection->back_kformstruct);
	return(NULL);
}
