/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>		Attribute Actions For Float Selections        <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>	     		kvf_get_float_gt0_sel()	              <<<<
   >>>>	     		kvf_set_float_gt0_sel()	              <<<<
   >>>>	     		kvf_get_float_ge0_sel()	              <<<<
   >>>>	     		kvf_set_float_ge0_sel()	              <<<<
   >>>>	     		kvf_get_float_lt0_sel()	              <<<<
   >>>>	     		kvf_set_float_lt0_sel()	              <<<<
   >>>>	     		kvf_get_float_le0_sel()	              <<<<
   >>>>	     		kvf_set_float_le0_sel()	              <<<<
   >>>>	     		kvf_get_float_upper()		      <<<<
   >>>>	     		kvf_set_float_upper()		      <<<<
   >>>>	     		kvf_get_float_lower()		      <<<<
   >>>>	     		kvf_set_float_lower()		      <<<<
   >>>>	     		kvf_get_float_def()		      <<<<
   >>>>	     		kvf_set_float_def()		      <<<<
   >>>>	     		kvf_get_float_val()		      <<<<
   >>>>	     		kvf_set_float_val()		      <<<<
   >>>>	     		kvf_change_floatbounds()	      <<<<
   >>>>                                                       <<<<
   >>>>   Static:                                             <<<<
   >>>>   Public:                                             <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<  */

#include "internals.h"


/*-----------------------------------------------------------
|
|       Routine: kvf_get_float_gt0_sel
|       Purpose: Gets the boolean TRUE or FALSE depending on whether
|                or not the bounds of an float selection are > 0
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_FLOAT_GT0_SEL ("kvf_float_gt0_sel")
|        Output: calldata    - passes back TRUE if bounds > 0 set
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_float_gt0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int       *gt0;
	Line_Info line_info;

        /* set the pointer to be returned */
        gt0 = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	if ((line_info.upper_float == line_info.lower_float) &&
	    (line_info.lower_float == 2.0))
  	    *gt0 = TRUE;
	else *gt0 = FALSE;

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_float_gt0_sel
|       Purpose: Sets the bounds of an float selection to > 0
|                according to the value of the "float_gt0" 
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_FLOAT_GT0_SEL ("kvf_float_gt0_sel")
|                calldata    - provides value of TRUE if bounds > 0 are set
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_float_gt0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int   *gt0; 
	float float_def;

	/* value to which KVF_FLOAT_GT0_SEL is to be set */
        gt0  = (int *) calldata;

	if (*gt0 != TRUE) return(FALSE);

	if (!(kvf_change_floatbounds(kformstruct, "gt0")))
	    return(FALSE);

	kvf_get_attribute(kformstruct, KVF_FLOAT_DEF, &float_def);
	if (float_def <= 0.0)
	    kvf_set_attribute(kformstruct, KVF_FLOAT_DEF, 1.0);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_float_ge0_sel
|       Purpose: Gets the boolean TRUE or FALSE depending on whether
|                or not the bounds of an float selection are >= 0
|
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_FLOAT_GE0_SEL ("kvf_float_ge0_sel")
|        Output: calldata    - passes back TRUE if bounds are set >= 0
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_float_ge0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int       *ge0;
	Line_Info line_info;

        /* set the pointer to be returned */
        ge0 = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	if ((line_info.lower_float == line_info.upper_float) &&
	    (line_info.lower_float == 1))
  	    *ge0 = TRUE;
	else *ge0 = FALSE;

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_float_ge0_sel
|       Purpose: Sets the bounds of an float selection to >= 0
|                according to the value of the "float_ge0" 
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_FLOAT_GE0_SEL ("kvf_float_ge0_sel")
|                calldata    - provides boolean value of TRUE if bounds are >= 0
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_float_ge0_sel(
   kform_struct *kformstruct,
   char        *attribute,
   kaddr       calldata)
{
	int   *ge0; 
	float float_def;

	/* value to which KVF_FLOAT_GE0_SEL is to be set */
        ge0  = (int *) calldata;

	if (*ge0 != TRUE) return(FALSE);

	if (!(kvf_change_floatbounds(kformstruct, "ge0")))
	    return(FALSE);

	kvf_get_attribute(kformstruct, KVF_FLOAT_DEF, &float_def);
	if (float_def < 0)
	    kvf_set_attribute(kformstruct, KVF_FLOAT_DEF, 0.0);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_float_lt0_sel
|       Purpose: Gets the boolean TRUE or FALSE depending on whether
|                or not the bounds of an float selection are < 0
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_FLOAT_LT0_SEL ("kvf_float_lt0_sel")
|        Output: calldata    - passes back TRUE if bounds are set to < 0
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_float_lt0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int       *lt0;
	Line_Info line_info;

        /* set the pointer to be returned */
        lt0 = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	if ((line_info.lower_float == line_info.lower_float) &&
	    (line_info.lower_float == -2.0))
  	    *lt0 = TRUE;
	else *lt0 = FALSE;

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_float_lt0_sel
|       Purpose: Sets the bounds of an float selection to < 0
|                according to the value of the "float_lt0" 
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_FLOAT_LT0_SEL ("kvf_float_lt0_sel")
|                calldata    - provides boolean value of TRUE if bounds are < 0
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_float_lt0_sel(
   kform_struct *kformstruct,
   char        *attribute,
   kaddr       calldata)
{
	int   *lt0; 
	float float_def;

	/* value to which KVF_FLOAT_LT0_SEL is to be set */
        lt0  = (int *) calldata;

	if (*lt0 != TRUE) return(FALSE);
 
	if (!(kvf_change_floatbounds(kformstruct, "lt0")))
	    return(FALSE);

	kvf_get_attribute(kformstruct, KVF_FLOAT_DEF, &float_def);
	if (float_def >= 0)
	    kvf_set_attribute(kformstruct, KVF_FLOAT_DEF, -1.0);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_float_le0_sel
|       Purpose: Gets the boolean TRUE or FALSE depending on whether
|                or not the bounds of an float selection are <= 0
|
|         Input: kformstruct    - backplane of the selection
|                attribute - KVF_FLOAT_LE0_SEL ("kvf_float_le0_sel")
|        Output: calldata  - passes back TRUE if bounds are set to <= 0
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_float_le0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int       *le0;
	Line_Info line_info;

        /* set the pointer to be returned */
        le0 = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	if ((line_info.lower_float == line_info.lower_float) &&
	    (line_info.lower_float == -1.0))
  	    *le0 = TRUE;
	else *le0 = FALSE;

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_float_le0_sel
|       Purpose: Sets the bounds of an float selection to <= 0
|                according to the value of the "float_le0" 
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_FLOAT_LE0_SEL ("kvf_float_le0_sel")
|                calldata    - provides boolean value of TRUE if bounds are <= 0
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_float_le0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int   *le0; 
	float float_def;

	/* value to which KVF_FLOAT_LE0_SEL is to be set */
        le0  = (int *) calldata;

	if (*le0 != TRUE) return(FALSE);

	if (!(kvf_change_floatbounds(kformstruct, "le0")))
	    return(FALSE);

	kvf_get_attribute(kformstruct, KVF_FLOAT_DEF, &float_def);
	if (float_def > 0)
	    kvf_set_attribute(kformstruct, KVF_FLOAT_DEF, 0.0);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_float_upper
|       Purpose: Gets the value of the upper bound for an float,
|                assuming that "kvf_float_upper_sel" is TRUE
|
|         Input: kformstruct    - backplane of the selection
|                attribute - KVF_FLOAT_UPPER ("kvf_float_upper")
|        Output: calldata  - passes back float upper bound value
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_float_upper(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	float     *upper;
	Line_Info line_info;

        /* set the pointer to be returned */
        upper = (float *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

  	*upper = line_info.upper_float;

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_float_upper
|       Purpose: Sets the value of the upper bound for an float,
|                assuming that "kvf_float_upper_sel" is TRUE
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_FLOAT_UPPER ("kvf_float_upper")
|                calldata    - provides float upper bound value
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_float_upper(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int   prec;
	float *upper; 
	float float_def;
	char  temp[KLENGTH];
	char  scale[KLENGTH];

	/* value to which KVF_FLOAT_UPPER is to be set */
        upper  = (float *) calldata;

	kvf_get_attribute(kformstruct, KVF_FLOAT_PREC, &prec);
	if (prec == 0)
	    sprintf(scale, "upper = %%g");
	else if (prec == -1)
	    sprintf(scale, "upper = %%f");
	else sprintf(scale, "upper = %%.%df", prec);
	sprintf(temp, scale, *upper);

	if (!(kvf_change_floatbounds(kformstruct, temp)))
	    return(FALSE);

	if (!(kvf_get_attribute(kformstruct, KVF_FLOAT_DEF, &float_def)))
            return(FALSE);

	if (float_def > *upper)
	{
	     float_def = *upper;
	     if (!(kvf_set_attribute(kformstruct, KVF_FLOAT_DEF, float_def)))
            	return(FALSE);
	}
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_float_lower
|       Purpose: Gets the value of the lower bound for an float,
|                assuming that "kvf_float_lower_sel" is TRUE
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_FLOAT_LOWER ("kvf_float_lower")
|        Output: calldata    - passes back float lower bound value
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_float_lower(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	float     *lower;
	Line_Info line_info;

        /* set the pointer to be returned */
        lower = (float *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

  	*lower = line_info.lower_float;

	kvf_free_line_info_strings(&line_info);
	return(TRUE);

}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_float_lower
|       Purpose: Sets the value of the lower bound for an float,
|                assuming that "kvf_float_lower_sel" is TRUE
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_FLOAT_LOWER ("kvf_float_lower")
|                calldata    - provides float lower bound value
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_float_lower(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int   prec;
	float *lower; 
	float float_def;
	char  temp[KLENGTH];
	char  scale[KLENGTH];

	/* value to which KVF_FLOAT_GE0_SEL is to be set */
        lower  = (float *) calldata;

	kvf_get_attribute(kformstruct, KVF_FLOAT_PREC, &prec);
	if (prec == 0)
	    sprintf(scale, "lower = %%g");
	else if (prec == -1)
	    sprintf(scale, "lower = %%f");
	else sprintf(scale, "lower = %%.%df", prec);
	sprintf(temp, scale, *lower);

	if (!(kvf_change_floatbounds(kformstruct, temp)))
	    return(FALSE);

	if (!(kvf_get_attribute(kformstruct, KVF_FLOAT_DEF, &float_def)))
	    return(FALSE);

	if (float_def < *lower)
	{
	     float_def = *lower;
	     if (!(kvf_set_attribute(kformstruct, KVF_FLOAT_DEF, float_def)))
            	return(FALSE);
	}
	return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: kvf_get_float_def
|       Purpose: Gets the value of the default for an float
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_FLOAT_DEF ("kvf_float_def")
|        Output: calldata    - passes back float default value
|
|          Date: March 15, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_float_def(
   kform_struct *kformstruct,
   char     *attribute,
   kaddr    calldata)
{
	float     *float_def;
	Line_Info line_info;

        /* set the pointer to be returned */
        float_def = (float *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

  	*float_def = line_info.float_def;

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_float_def
|       Purpose: Sets the value of the default for an float
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_FLOAT_DEF ("kvf_float_def")
|                calldata    - provides float default value
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_float_def(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	char       temp[KLENGTH];
	float      *float_def;
	Line_Info  line_info;
	kselection *selection;

	/* value to which KVF_FLOAT_DEF is to be set */
        float_def  = (float *) calldata;

	if (!(kvf_set_attribute(kformstruct, KVF_FLOAT_VAL, *float_def)))
	    return(FALSE);

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	line_info.float_def = *float_def;
	kvf_gen_deparse_kformstruct(kformstruct, &line_info);

	if (!(kvf_set_attribute(kformstruct, KVF_FLOAT_VAL, *float_def)))
            return(FALSE);

	selection = kformstruct->Selptr;
	if (selection->back_toggle != NULL)
	{
	    sprintf(temp, "%f", *float_def);
	    kvf_set_attribute(kformstruct, KVF_TITLE, temp);
	}

	return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: kvf_get_float_val
|       Purpose: Gets the value of the float
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_FLOAT_VAL ("kvf_float_val")
|        Output: calldata    - passes back float value
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_float_val(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	float     *float_val;
	Line_Info line_info;

        /* set the pointer to be returned */
        float_val = (float *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

  	*float_val = line_info.float_val;

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_float_val
|       Purpose: Sets the value of an float
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_FLOAT_VAL ("kvf_float_val")
|                calldata    - provides float value
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_float_val(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	float      *float_val;
	Line_Info  line_info;

	/* value to which KVF_FLOAT_VAL is to be set */
        float_val  = (float *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);


        if (!(kvf_check_double((double) *float_val, 
			       (double) line_info.upper_float,
			       (double) line_info.lower_float, 
				line_info.variable, line_info.title)))
        {
            kvf_free_line_info_strings(&line_info);
            return(FALSE);
        }

	line_info.float_val = *float_val;

	kvf_gen_deparse_kformstruct(kformstruct, &line_info);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: kvf_change_floatbounds
|
|       Purpose: Changes the bounds for a float selection
|
|         Input: kformstruct - generic kform_struct pointer
|		 string    - string rep of float bounds, may be one of:
|			     "gt0"        : bounds >  0
|			     "ge0"        : bounds >= 0
|			     "lt0"        : bounds <  0
|			     "le0"        : bounds <= 0
|                            "lower = %f" : lower bound 
|                            "upper = %f" : upper bound 
|			     "nobounds"   : no bounds 
|
|        Output: Returns TRUE on success, FALSE on Failure
|          Date: March 20, 1994 
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/

int kvf_change_floatbounds(
   kform_struct *kformstruct,
   char         *string)
{
	kselection *selection;
	Line_Info  line_info;
	float      lower, upper;
	int        status, bounds = FALSE;

	kvf_clear_line_info(&line_info);
	selection = kformstruct->Selptr;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	if (kstrcmp(string, "gt0") == 0)
	{
	   line_info.lower_float = 2.0;
	   line_info.upper_float = 2.0;
	}
	else if (kstrcmp(string, "ge0") == 0)
        {
           line_info.lower_float = 1.0;
           line_info.upper_float = 1.0;
        }
	else if (kstrcmp(string, "lt0") == 0)
        {
           line_info.lower_float = -2.0;
           line_info.upper_float = -2.0;
        }
	else if (kstrcmp(string, "le0") == 0)
        {
           line_info.lower_float = -1.0;
           line_info.upper_float = -1.0;
        }
	else if (kstrcmp(string, "nobounds") == 0)
        {
           line_info.lower_float = 0.0;
           line_info.upper_float = 0.0;
        }
	else if (kstrcmp(string, "bounds") == 0)
        {
	   if (line_info.lower_float == line_info.upper_float)
	   {
              line_info.lower_float = -100.0;
              line_info.upper_float =  100.0;
	   }
	   bounds = TRUE;
        }
	else 
	{
	   status = ksscanf(string, "lower = %f", &lower);
	   if (status != 1)
	   {
	       status = ksscanf(string, "upper = %f", &upper);
	       if (status != 1)
	       {
	 	   kerror("kforms", "kvf_change_float_bounds",
		          "Invalid string provided for KVF_FLOATBOUNDS_CHNG");
		   return(FALSE);
	       }
	       else line_info.upper_float = upper;
	   }
	   else line_info.lower_float = lower;

           if (line_info.lower_float != line_info.upper_float)
	       bounds = TRUE;
	}

	if (line_info.upper_float < line_info.lower_float)
	{
	    kerror("kforms", "kvf_change_float_bounds",
	  	   "Attempt to set float upper bound < lower bound");
	    return(FALSE);
	}
	kvf_gen_deparse(&line_info, &selection->line);

	if (bounds)
	{
	    if (line_info.float_val < line_info.lower_float)
	        line_info.float_val = line_info.lower_float;

    	    if (line_info.float_val > line_info.upper_float)
	        line_info.float_val = line_info.upper_float;

	    kvf_set_attribute(kformstruct, KVF_FLOAT_DEF,
			      line_info.float_val);
	}

	kvf_free_line_info_strings(&line_info);

	return(TRUE);
}

