 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Public Khoros Expression Routines
   >>>>
   >>>>  Private:
   >>>>
   >>>>   Public:
   >>>>             kexpr_evaluate_generic()
   >>>>             kexpr_compile_function()
   >>>>             kexpr_execute_function()
   >>>>             kexpr_parent_set()
   >>>>             kexpr_parent_unset()
   >>>>             kexpr_variables_list()
   >>>>             kexpr_variables_print()
   >>>>             kexpr_variables_copy()
   >>>>             kexpr_variables_clear()
   >>>>             kexpr_variable_delete()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include "y.tab.h"


/************************************************************
*
*  Routine Name: kexpr_evaluate_generic - evaluate generic for desired data type
*
*       Purpose: This routine evaluates the input string and returns
*		 a generic value of the expression according to the
*		 desired data type.  See the kexpr_evaluate_{data type}
*		 for specific information about evaluation of an expression
*		 for the desired data type.
*
*         Input: id     - the variable identifier.
*                string - the string to be evaluated.
*                type   - the data type for the value pointer.  Valid data
*			  types are: KBYTE, KUBYTE, KSHORT, KUSHORT, KINT,
*			  KUINT, KLONG, KULONG, KFLOAT, KDOUBLE, KCOMPLEX,
*			  KDCOMPLEX, KSTRING.
*
*        Output: value  - if no error occurred then the value of
*			  the expression is stored and True returned.
*		 error  - if an error occurred the error message is stored in
*			  the error string array and False returned.  The
*			  error string array must be at least a 1024 string
*			  array that is allocated by the calling routine.  If
*			  the error string array has not been allocated by the
*			  calling routine (error is passed in as NULL) then the
*			  error message is output with the kerror facility.
*
*	Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Mark Young  
*          Date: Thu Jun 25 1992
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int kexpr_evaluate_generic(
   long  id,
   char  *string,
   int   type,
   kaddr value,
   char  *error)
{
	int   status;


	if (string != NULL)
	{
	   if (kstrchr(string, '$') == NULL)
	      kexpr_set_value();
	   else
	      kexpr_set_string();

	   status = kexpr_eval_expression(id, string, error);
	}
	else
	   status = FALSE;

	if (value == NULL)
	{
	   return(status);
	}
	else if (FinalValue.type == STRING)
	{
	   kstrcpy((kstring) value, FinalValue.String);
	   return(status);
	}

	switch (type)
	{
           case KBYTE:
		*((char *) value) = (char) FinalValue.Value;
		break;

           case KUBYTE:
		*((unsigned char *) value) = (unsigned char) FinalValue.Value;
		break;

           case KSHORT:
		*((short *) value) = (short) FinalValue.Value;
		break;

           case KUSHORT:
		*((unsigned short *) value) = (unsigned short) 
				FinalValue.Value;
		break;

           case KLONG:
		*((long *) value) = (long) FinalValue.Value;
		break;

           case KULONG:
		*((unsigned long *) value) = (unsigned char) FinalValue.Value;
		break;

           case KINT:
		*((int *) value) = (int) FinalValue.Value;
		break;

           case KUINT:
		*((unsigned int *) value) = (unsigned int) FinalValue.Value;
		break;

           case KFLOAT:
		*((float *) value) = (float) FinalValue.Value;
		break;

           case KDOUBLE:
		*((double *) value) = (double) FinalValue.Value;
		break;

           case KCOMPLEX:
		break;

           case KDCOMPLEX:
		break;

           case KSTRING:
		ksprintf((kstring) value, "%g", FinalValue.Value);
		break;

           default:
		return(FALSE);
	}
	return(status);
}

/************************************************************
*
*  Routine Name: kexpr_compile_function - compile a function
*
*       Purpose: This routine compiles a function into it's corresponding
*		 symbol list.  The symbol list pointer is passed back and
*		 can be used in conjunction with the kexpr_execute_function()
*		 routine.
*         Input: id     - the variable identifier.
*                string - the string to be evaluated.
*        Output: error  - if an error occurred the error message is stored in
*			  the error string array and False returned.  The
*			  error string array must be at least a 1024 string
*			  array that is allocated by the calling routine.  If
*			  the error string array has not been allocated by the
*			  calling routine (error is passed in as NULL) then the
*			  error message is output with the kerror facility.
*
*	Returns: TRUE (1) on success, FALSE (0) otherwise
*  Restrictions: 
*    Written By: Mark Young  
*          Date: Dec 31, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

kaddr kexpr_compile_function(
   long  id,
   char  *string,
   char  *error)
{
	int        i;
	Symbol	   *symbol;
	SymbolList *symlist = NULL, *arglist = NULL;


	if (string == NULL)
	   return(NULL);

	kexpr_set_value();
	if (!kexpr_eval_expression(id, string, error))
	   return(NULL);

	/*
	 *  Since we have the function (specified by symbol), go ahead and
	 *  add it to the function list.
	 */
	symbol = CompiledSymbol;
	symlist = kexpr_add_function(symlist, symbol, symbol->num);

	/*
	 *  Next we build our expression list.
	 */
	for (i = 0; i < (int) symbol->num; i++)
           arglist = kexpr_add_number(arglist, 0.0);

        symlist = kexpr_add_symlist(symlist, arglist);
	return(symlist);
}

/************************************************************
*
*  Routine Name: kexpr_execute_function - execute a previous compiled function
*
*       Purpose: This routine executes a previous compiled function in order
*		 to avoid the overhead of repeated calls to the yyparse().
*         Input: function - the variable identifier.
*                type     - the data type for the value pointer.
*        Output: value    - pointer to the data pointer.
*	Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Mark Young  
*          Date: Dec 31, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int kexpr_execute_function(
   kaddr function,
   int   type,
   kaddr value,
   char  *error,
   kvalist)
{
	kva_list   list;
	double     number;
	Symbol     *symbol;
	int        i, status;
	SymbolList *symlist, *arglist;


	/*
	 *  Sanity check...
	 */
	if (!function)
	   return(FALSE);

	/*
	 *  Get the function (which is really a symbol list pointer).
	 */
	symlist = (SymbolList *) function;
	symbol =  symlist->symbol;
	arglist = symlist->next;

	kva_start(list, error);
	for (i = 0; i < (int) symbol->num; i++)
	{
	   number = kva_arg(list, double);
	   arglist->symbol->Value = number;
	   arglist = arglist->next;
	}
	string_array = NULL;
	kexpr_error_string = error;
	status = kexpr_eval_symlist(symlist, TRUE, &FinalValue);

	if (value == NULL)
	   return(status);

	if (type == KDOUBLE)
	{
	   *((double *) value) = (double) FinalValue.Value;
	   return(status);
	}

	switch (type)
	{
           case KBYTE:
		*((char *) value) = (char) FinalValue.Value;
		break;

           case KUBYTE:
		*((unsigned char *) value) = (unsigned char) FinalValue.Value;
		break;

           case KSHORT:
		*((short *) value) = (short) FinalValue.Value;
		break;

           case KUSHORT:
		*((unsigned short *) value) = (unsigned short) 
				FinalValue.Value;
		break;

           case KLONG:
		*((long *) value) = (long) FinalValue.Value;
		break;

           case KULONG:
		*((unsigned long *) value) = (unsigned char) FinalValue.Value;
		break;

           case KINT:
		*((int *) value) = (int) FinalValue.Value;
		break;

           case KUINT:
		*((unsigned int *) value) = (unsigned int) FinalValue.Value;
		break;

           case KFLOAT:
		*((float *) value) = (float) FinalValue.Value;
		break;

           case KDOUBLE:
		*((double *) value) = (double) FinalValue.Value;
		break;

           case KCOMPLEX:
		break;

           case KDCOMPLEX:
		break;

           case KSTRING:
		ksprintf((kstring) value, "%g", FinalValue.Value);
		break;

           default:
		return(FALSE);
	}
	return(status);
}

/************************************************************
*
*  Routine Name: kexpr_free_function - frees up a previous compiled function
*
*       Purpose: This routine frees the resources associated with a previous
*		 compiled function in order to avoid the overhead of repeated
*		 calls to the kexpr_compile_function().
*         Input: function - the variable identifier.
*                type     - the data type for the value pointer.
*        Output: value    - pointer to the data pointer.
*	Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Mark Young  
*          Date: Dec 31, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int kexpr_free_function(
   kaddr function)
{
	kexpr_free_symlist((SymbolList *) function);
	return(TRUE);
}

/************************************************************
*
*  Routine Name: kexpr_parent_set() - set the parent variable list associated
*                                     with an id
*
*       Purpose: This routine sets the child alternate variable list
*		 specified by the parent id.  This variable list
*		 will be searched when a variable is not found in
*		 the current variable list.  If the parent's variable
*		 list "parent" is set then the procedure is repeated
*		 until either the variable is found or no more
*		 variable lists are available to search.  In short, this
*		 routine provides a method of expanding the scope
*		 of variables used.
*
*        Input:  id   - the id variable (for the child) in which to set the
*			parent variable list.
*		 parent - the id variable for the parent.
*
*       Output:  
*	Returns: TRUE (1) if the child's parent variable list was set to that
*		 of the parent, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young  
*          Date: Thu Jun 25 1992
*      Verified:
*  Side Effects:
* Modifications:
*
************************************************************/

int kexpr_parent_set(
   long id,
   long parent)
{
	Varlist *varlist, *parlist;


	/*
	 *  Get the child's variable list.
	 */
	if ((varlist = kexpr_get_varlist(id)) == NULL)
	{
	   kfprintf(kstderr,"Unable to get list for child id (%d)\n", id);
	   return(FALSE);
	}

	/*
	 *  Get the parents's variable list.
	 */
	if ((parlist = kexpr_get_varlist(parent)) == NULL)
	{
	   kfprintf(kstderr,"Unable to get list for parent id (%d)\n", parent);
	   return(FALSE);
	}

	/*
	 *  Set the child's parent list to be that of the parent.
	 */
	varlist->parent = parlist;
	return(TRUE);
}

/************************************************************
*
*  Routine Name: kexpr_parent_unset() -  unset the parent variable list
*                                        associated with an id
*
*       Purpose: This routine unsets the parent variable list previously
*		 specified by kexpr_parent_set().  The result of a call
*		 to this routine is that if a variable is not found in the
*		 variable list associated with the kexpr identifier, then
*		 no other variable list will be checked.  In short, this
*		 routine provides a method of forcing variables used to
*		 be "local".
*
*        Input:  id   - the id variable (for the child) in which
*			to unset the parent variable list.
*	Returns: TRUE (1) if the child's parent list was successfully set to
*		 NULL, FALSE (0) otherwise.
*
*  Restrictions:
*    Written By: Mark Young  
*          Date: Thu Jun 25 1992
*      Verified:
*  Side Effects:
* Modifications:
*
************************************************************/

int kexpr_parent_unset(
   long id)
{
	Varlist *varlist;


	/*
	 *  Get the child's variable list.
	 */
	if ((varlist = kexpr_get_varlist(id)) == NULL)
	{
	   kfprintf(kstderr,"Unable to get list for child id (%d)\n", id);
	   return(FALSE);
	}

	/*
	 *  Set the child's parent list to be NULL
	 */
	varlist->parent = NULL;
	return(TRUE);
}

/************************************************************
*
*  Routine Name: kexpr_variables_list() - list variables associated with an id
*
*       Purpose: This routines extracts the current list of variables for a
*		 particular id variable and returns a pointer to the list of
*		 variables.
*
*        Input:  id   - the id variable whose list of current variables will
*			be extracted.
*		 mode - list mode of what to be included in the list of
*			variables.
*	Output:	 num  - number of variables in list or NULL if not needed
*      Returns:  the list of variables or NULL upon failure
*
*  Restrictions:
*    Written By: Mark Young  
*          Date: Dec 13, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
************************************************************/

char **kexpr_variables_list(
   long id,
   int  mode,
   int  *num)
{
	Varlist    *varlist;
	Symbol     *symbol;
	SymbolList *symlist;

	double value;
	int    size = 0;
	char   temp[KLENGTH], **list = NULL;


	/*
	 * Check to see if it is a current variable list.
	 */
	if ((varlist = kexpr_get_varlist(id)) != NULL)
	{
	   symlist = varlist->symlist;
	   while (symlist != NULL)
	   {
	      symbol = symlist->symbol;
	      if (symbol->type == UNDEFINED)
	      {
		 symlist = symlist->next;
		 continue;
	      }

	      if (mode == 1)
	      {
		 kexpr_evaluate_generic(id, symbol->name, KDOUBLE, &value,NULL);
	         ksprintf(temp, "%s = %g", symbol->name, value);
	      }
	      else if (mode == 0)
	         ksprintf(temp, "%s", symbol->name);

	      list = karray_add(list, kstrdup(temp), size++);
	      symlist = symlist->next;
	   }
	}

	if (num) *num = size;
	return(list);
}

/************************************************************
*
*  Routine Name: kexpr_variables_print() - prints variables associated with
*                                         an id
*
*       Purpose: This routine prints the current list of variables
*		 for a particular id into a file that has been opened
*		 for writing.
*
*        Input:  id   - the list of id from which we will list 
*			current variables.
*		 file - the kfile pointer in which we will print
*			this information to
*
*       Output:  returns the TRUE or FALSE if we were able
*		 to list the workspace.
*
*	Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young  
*          Date: Thu Jun 25 1992
*      Verified:
*  Side Effects:
* Modifications:
*
************************************************************/

int kexpr_variables_print(
   long id,
   kfile *file)
{
	Varlist    *varlist;
	Symbol     *symbol;
	SymbolList *symlist;


	/*
	 * Check to see if it is a current variable list.
	 */
	kfprintf(file,"\n\n");
	if ((varlist = kexpr_get_varlist(id)) != NULL)
	{
	   symlist = varlist->symlist;
	   if (symlist == NULL)
	   {
	      kfprintf(file,"\n# No variables currently declared. #\n\n");
	      return(TRUE);
	   }

	   kfprintf(file,"#\n# list of currently declared variables\n#\n\n");
	   while (symlist != NULL)
	   {
	      symbol = symlist->symbol;
	      kexpr_print_symbol(file, symbol, FALSE);
	      symlist = symlist->next;
	   }
	   kfprintf(file,"\n");
	}
	else
	{
	   kfprintf(file,"\n\n# Error! Unable to retrieve expressions. #\n\n");
	   return(FALSE);
	}
	return(TRUE);
}

/************************************************************
*
*  Routine Name: kexpr_variables_copy() - copies variables from one id to
*                                         another
*
*       Purpose: This routine copies the current list of variables
*		 from one particular id to another.
*
*        Input:  from   - the id variable whose list of current viariables
*			  will be copied the id.
*		 to     - the id variable to recieve the copied list of
*			  variables.
*
*       Output:
*	Returns: returns TRUE (1) if the list of variables copied successfully,
*		 or FALSE (0) if the list of variables was not copied.
*
*  Restrictions:
*    Written By: Mark Young  
*          Date: Thu Jun 25 1992
*      Verified:
*  Side Effects:
* Modifications:
*
************************************************************/


int kexpr_variables_copy(
   long from,
   long to)
{
/*
	Varlist	   *fromlist, *tolist;
	Symbol	   *symbol;
	SymbolList *symlist;


	if ((fromlist = kexpr_get_varlist(from)) == NULL)
	{
	}

	if ((tolist = kexpr_get_varlist(to)) == NULL)
	{
	}
 */
	return(TRUE);
}

/************************************************************
*
*  Routine Name: kexpr_variables_clear() - deletes variables associated with
*                                          an id
*
*       Purpose: This routine deletes the current list of variables
*		 for a particular id.  The id is also removed.
*
*	  Input: id - the id the variable to be to be cleared.
*	 Output:
*	Returns: returns TRUE (1) if the list of variables were cleared
*		 successfully, or FALSE (0) if the list of variables were not
*		 cleared.
*  Restrictions:
*    Written By: Mark Young  
*          Date: Thu Jun 25 1992
*      Verified:
*  Side Effects:
* Modifications:
*
************************************************************/

int kexpr_variables_clear(
   long id)
{
	return(kexpr_delete_varlist(id));
}

/************************************************************
*
*  Routine Name: kexpr_variable_delete() - delete a variable associated
*                                          with an id
*
*       Purpose: This routine deletes a variable from the current list of
*	         variables for a particular id.  kexpr_variable_delete()
*		 deletes the specified variable from the id's variable list.
*		 a particular id.
*
*         Input: id - the id variable whose list of variables will have one
*		      variable deleted.
*		 variable - the variable to delete from the list of variables.
*        Output:
*	Returns: returns TRUE (1) if the variable was deleted from the list
*		 of variables or FALSE (0) if the variable was not deleted from
*		 the list of variables or was not in the list of variables.
*
*  Restrictions:
*    Written By: Mark Young  
*          Date: Dec 13, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
************************************************************/

int kexpr_variable_delete(
   long id,
   char *variable)
{
	Symbol  *symbol;
	Varlist *varlist;

	/*
	 * Check to see if it is in the id varlist list.
	 */
	if ((varlist = kexpr_get_varlist(id)) == NULL)
	   return(FALSE);

	/*
	 * Check to see if it is in the id varlist list.
	 */
	if ((symbol = kexpr_get_symbol(varlist, variable, FALSE)) == NULL)
	   return(FALSE);

	return(kexpr_delete_symbol(id, symbol));
}
