/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>          Routines to generate prompt strings for      <<<<
   >>>>                    Interactive Prompting              <<<<
   >>>>                                                       <<<<
   >>>>   Private:                                            <<<<
   >>>>            kgen_create_pstr                           <<<<
   >>>>    Static:                                            <<<<
   >>>>            print_toggle_choices_to_pstr               <<<<
   >>>>            print_list_choices_to_pstr                 <<<<
   >>>>            print_bounds_to_pstr                       <<<<
   >>>>    Public:                                            <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

static void print_toggle_choices_to_pstr PROTO((char *, kselection *));
static void print_list_choices_to_pstr   PROTO((char *, kselection *));
static void print_bounds_to_pstr         PROTO((char *, double, double));


/*-----------------------------------------------------------
|
|       Routine: kgen_create_pstr
|
|       Purpose: This routine creates a single prompt string
|                for a command line argument
|
|         Input:
|        Output: none
|       Returns: The prompt string on success, NULL on failure
|    Written By: Danielle Argiro
|          Date: March 29, 1994
| Modifications: 
|
------------------------------------------------------------*/

char *kgen_create_pstr(
    kselection *selection)
{
	char   buffer[KLENGTH];
	char   temp[KLENGTH];
	char   *variable;
	char   *description;
	int    optional, logic_def;
	int    int_lower, int_upper, int_def;
	float  flt_lower, flt_upper, flt_def;
	double dbl_lower, dbl_upper, dbl_def;
	char   *file_def, *string_def;
	

	if ((selection->type != KUIS_LOGICAL)     &&
	    (selection->type != KUIS_FLAG)        &&
	    (selection->type != KUIS_INTEGER)     &&
	    (selection->type != KUIS_FLOAT)       &&
	    (selection->type != KUIS_DOUBLE)      &&
	    (selection->type != KUIS_STRING)      &&
	    (selection->type != KUIS_STRINGLIST)  &&
	    (selection->type != KUIS_INPUTFILE)   &&
	    (selection->type != KUIS_OUTPUTFILE)  &&
	    (selection->type != KUIS_LIST)        &&
	    (selection->type != KUIS_DISPLAYLIST) &&
	    (selection->type != KUIS_TOGGLE)      &&
	    (selection->type != KUIS_CYCLE))
	    return(NULL);

	kvf_get_attributes(selection->back_kformstruct,
			   KVF_DESCRIPTION, &description,
			   KVF_OPTIONAL,    &optional,
			   NULL);
	if (description == NULL)
	    description = kstrdup("no description");
	variable = ktoken_to_string(selection->var_token);
	buffer[0] = '\0';

	/*
	 *	prompt string for logical arguments
	 */
	if (selection->type == KUIS_LOGICAL)
	{
	    /* print the base of the prompt string */
	    sprintf(buffer, "Enter: (%s) %s {logical, (y/n)",
	            variable, description);

	    /* tack on appropriate default info if argument is optional */
	    if (optional)
	    {
		kvf_get_attribute(selection->back_kformstruct,
				  KVF_LOGIC_DEF, &logic_def);
	        if (logic_def)
	     	    (void) kstring_cat(buffer, " [yes] ", buffer);
		else (void) kstring_cat(buffer, " [no] ", buffer);
	    }
	}
	
	/*
	 *	prompt string for flag arguments
	 */
	if (selection->type == KUIS_FLAG)
	{
	    /* print the base of the prompt string */
	    sprintf(buffer, "Enter: (%s) %s {flag, (y/n) [n]",
	            variable, description);
	}

	/*
	 *	prompt string for integer arguments
	 */
	else if (selection->type == KUIS_INTEGER)
	{
	    /* print the base of the prompt string */
	    sprintf(buffer, "Enter: (%s) %s {integer",
	            variable, description);

	    /* cat the bounds into the prompt string */
	    kvf_get_attributes(selection->back_kformstruct,
			       KVF_INT_LOWER, &int_lower,
			       KVF_INT_UPPER, &int_upper,
			       NULL);
	    print_bounds_to_pstr(buffer, (double) int_lower,
                                 (double) int_upper);

	    /* tack on appropriate default info if argument is optional */
	    if (optional)
	    {
		kvf_get_attribute(selection->back_kformstruct,
                                  KVF_INT_DEF, &int_def);
		sprintf(temp, " [%d] ", int_def);
  	        (void) kstring_cat(buffer, temp, buffer);
	    }
	}

        /*
         *      prompt string for float arguments
         */
        else if (selection->type == KUIS_FLOAT)
        {
	    /* print the base of the prompt string */
	    sprintf(buffer, "Enter: (%s) %s {float",
	            variable, description);

            /* cat the bounds into the prompt string */
            kvf_get_attributes(selection->back_kformstruct,
                               KVF_FLOAT_LOWER, &flt_lower,
                               KVF_FLOAT_UPPER, &flt_upper,
                               NULL);
            print_bounds_to_pstr(buffer, (double) flt_lower,
                                 (double) flt_upper);

            /* tack on appropriate default info if argument is optional */
            if (optional)
            {
                kvf_get_attribute(selection->back_kformstruct,
                                  KVF_FLOAT_DEF, &flt_def);
                sprintf(temp, " [%g] ", flt_def);
                (void) kstring_cat(buffer, temp, buffer);
            }
        }

        /*
         *      prompt string for integer arguments
         */
        else if (selection->type == KUIS_DOUBLE)
        {
	    /* print the base of the prompt string */
	    sprintf(buffer, "Enter: (%s) %s {double",
	            variable, description);

            /* cat the bounds into the prompt string */
            kvf_get_attributes(selection->back_kformstruct,
                               KVF_DOUBLE_LOWER, &dbl_lower,
                               KVF_DOUBLE_UPPER, &dbl_upper,
                               NULL);
            print_bounds_to_pstr(buffer, dbl_lower, dbl_upper);

            /* tack on appropriate default info if argument is optional */
            if (optional)
            {
                kvf_get_attribute(selection->back_kformstruct,
                                  KVF_DOUBLE_DEF, &dbl_def);
                sprintf(temp, " [%g] ", dbl_def);
                (void) kstring_cat(buffer, temp, buffer);
            }
        }


	/*
	 *	prompt string for string arguments
	 */
	else if ((selection->type == KUIS_STRING) ||
	         (selection->type == KUIS_STRINGLIST))
	{
	    /* print the base of the prompt string */
	    sprintf(buffer, "Enter: (%s) %s {string",
	            variable, description);

	    if (optional)
            {
	  	kvf_get_attribute(selection->back_kformstruct,
				  KVF_STRING_DEF, &string_def);
		if (string_def != NULL)
		{
		    sprintf(temp, " , ['%s'] ", string_def);
	      	    (void) kstring_cat(buffer, temp, buffer);
		}
	        else (void) kstring_cat(buffer, " [ ] ", buffer);
	    }
	}

        /*
         *      prompt string for inputfile & outputfile arguments
         */
        else if ((selection->type == KUIS_INPUTFILE) ||
                 (selection->type == KUIS_OUTPUTFILE))
        {
	    /* print the base of the prompt string */
	    sprintf(buffer, "Enter: (%s) %s {%s",
	            variable, description, 
	    	    kgen_ascii_datatype(selection->type)); 

            if (optional)
            {
                kvf_get_attribute(selection->back_kformstruct,
                                  KVF_FILE_DEF, &file_def);
                if (file_def !=  NULL)
		{
                    sprintf(temp, " , ['%s'] ", file_def);
                    (void) kstring_cat(buffer, temp, buffer);
		}
		else (void) kstring_cat(buffer, " [ ] ", buffer);
            }   
        }

	/*
         *      prompt string for list & cycle arguments
         */
        else if ((selection->type == KUIS_LIST)        ||
                 (selection->type == KUIS_DISPLAYLIST) ||
		 (selection->type == KUIS_CYCLE))
        {
            /* print the base of the prompt string */
            sprintf(buffer, "Enter: (%s) %s --",
                    variable, description);

	    print_list_choices_to_pstr(buffer, selection);

            if (optional)
            {
                kvf_get_attribute(selection->back_kformstruct,
                                   KVF_LIST_VAL, &int_def);
                sprintf(temp, " , [%d] ", int_def);
                (void) kstring_cat(buffer, temp, buffer);
            }
        }

	/*
         *      prompt string for toggle arguments
         */
        else if (selection->type == KUIS_TOGGLE)
        {
            /* print the base of the prompt string */
            sprintf(buffer, "Enter: (%s) %s --",
                    variable, description);

            print_toggle_choices_to_pstr(buffer, selection);

            if (optional)
            {
                kvf_get_attribute(selection->back_kformstruct,
                                  KVF_TOGGLE_VAL, &string_def);
                if (string_def != NULL)
                {
                    sprintf(temp, " , [%s] ", string_def);
                    (void) kstring_cat(buffer, temp, buffer);
                }
                else (void) kstring_cat(buffer, " [ ] ", buffer);
            }
        }
	
	


        /* finish off prompt string (they all end like this) */
        kstring_cat(buffer, "}: ", buffer);

	return(kstrdup(buffer));
}

/*-----------------------------------------------------------
|
|       Routine: print_list_choices_to_pstr
|
|       Purpose: For list, cycle, and displaylist arguments,
|                appends possible values in appropriate notation
|                to the prompt string passed in the buffer.
|
|         Input: buffer     - buffer in which prompt string is being created
|                selection - selection representing argument
|        Output: none
|       Returns: none
|    Written By: Danielle Argiro
|          Date: March 29, 1994
| Modifications:
|
------------------------------------------------------------*/
static void print_list_choices_to_pstr(
    char       *buffer,
    kselection *selection)
{
	int  i, list_size, list_start;
	char **val_labels;
	char temp[KLENGTH];

	kvf_get_attributes(selection->back_kformstruct,
			   KVF_LIST_SIZE,  &list_size,
			   KVF_LIST_START, &list_start,
			   NULL);

	for (i = 0; i < list_size; i++)
	{
	    kvf_get_attribute(selection->back_kformstruct,
			      KVF_VAL_LABELS, &val_labels);
	    ksprintf(temp, "\n\t\t%d (\"%s\")", 
		     list_start + i, val_labels[i]);
	    (void) kstring_cat(buffer, temp, buffer);
	}

}

/*-----------------------------------------------------------
|
|       Routine: print_toggle_choices_to_pstr
|
|       Purpose: For toggle arguments, appends possible values in 
|                appropriate notation to the prompt string passed in the buffer.
|
|         Input: buffer     - buffer in which prompt string is being created
|                selection - selection representing argument
|        Output: none
|       Returns: none
|    Written By: Danielle Argiro
|          Date: March 30, 1994
| Modifications:
|
------------------------------------------------------------*/
static void print_toggle_choices_to_pstr(
    char       *buffer,
    kselection *selection)
{
	int  i, toggle_size;
        char **toggle_contents;
        char **descriptions;
        char temp[KLENGTH];

	kvf_get_attributes(selection->back_kformstruct, 
			   KVF_TOGGLE_SIZE,          &toggle_size,
			   KVF_TOGGLE_CONTENTS,      &toggle_contents,
			   KVF_TOGGLE_CONTENTS_DESC, &descriptions,
			   NULL);
        for (i = 0; i < toggle_size; i++)
        {
	    if (descriptions[i] != NULL)
                ksprintf(temp, "\n\t\t%s (%s)",
                         toggle_contents[i], descriptions[i]);
	    else ksprintf(temp, "\n\t\t%s (no description)",
                         toggle_contents[i]);
            (void) kstring_cat(buffer, temp, buffer);
        }

}


/*-----------------------------------------------------------
|
|       Routine: print_bounds_to_pstr
|
|       Purpose: For integer, float, and double arguments, 
|                looks at bounds, and appends appropriate notation
|                to the prompt string passed in the buffer.
|                Cast up to double; %g will provide truncated notation.
|
|         Input: buffer - buffer in which prompt string is being created
|                lower  - lower bound
|                upper  - upper bound
|        Output: none
|       Returns: none
|    Written By: Danielle Argiro
|          Date: March 29, 1994
| Modifications:
|
------------------------------------------------------------*/

static void print_bounds_to_pstr(
    char   *buffer,
    double lower,
    double upper)
{
	char temp[KLENGTH];

	if (lower == upper)
	{
            if (lower == -2.0)
	        (void) kstring_cat(buffer, ", < 0", buffer);

            else if (lower == -1.0)
	        (void) kstring_cat(buffer, ", <= 0", buffer);

            else if (lower == 1.0)
	        (void) kstring_cat(buffer, ", >= 0", buffer);

            else if (lower == 2.0)
	        (void) kstring_cat(buffer, ", > 0", buffer);

            else (void) kstring_cat(buffer, ", unbounded", buffer);
	}
	else
	{
	    sprintf(temp, " , %g to %g", lower, upper); 
	    (void) kstring_cat(buffer, temp, buffer);
	}
}

