/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>              Routines to Generate *.h file            <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>                kgen_clui_generate_hfile()             <<<<
   >>>>   Static:                                             <<<<
   >>>>                print_struct_fields()                  <<<<
   >>>>                print_single_fields() 	              <<<<
   >>>>                print_toggle_fields()                  <<<<
   >>>>                print_list_fields()                    <<<<
   >>>>                print_group_fields()                   <<<<
   >>>>   Public:                                             <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

static void print_struct_fields PROTO((kobject, kfile *));
static void print_single_fields PROTO((kobject, kfile *, kselection *));
static void print_toggle_fields PROTO((kfile *, kselection *));
static void print_list_fields   PROTO((kfile *, kselection *));
static void print_group_fields  PROTO((kobject, kfile *, kselection *, int, int));


/*-----------------------------------------------------------
|
|  Routine Name: kgen_clui_generate_hfile
|
|      Purpose:  This routine creates the C structure that provides the
|		 link between the command line user interface and the
|		 application program, in the *.h file
|
|        Input: program   - program object being created
|               prog_spec - internal representation of PS file
|               opath     - path to program object
|               tb_name   - name of toolbox
|	        oname     - name of program object being generated
|               hybrid    - TRUE if this is a hybrid xvroutine
|
|        Output: Returns TRUE on success, FALSE on failure, 
|                2 if *.prog needs to be re-gen'ed because it still has
|                the outdated -include_additions tag.
|
|    Written By: Danielle Argiro
|          Date: Jul 23, 1992 10:37
| Modifications: Converted from Khoros 1.0 (DA)
|
-------------------------------------------------------------*/

int kgen_clui_generate_hfile(
   kobject   program,
   char    **prog_spec,
   char     *opath,
   char     *tb_name,
   char     *oname,
   int      hybrid)
{
        kstring   routine      = "kgen_clui_generate_hfile()";
	kobject   toolbox      = NULL;
	kobject   file_object  = NULL;
	int       force_flag   = FALSE;
	int       isnewfile    = FALSE;
	int       object_type;
	int       status;
        kfile    *file;
	kstring   lname;
	kstring   filepath;
        kstring   copyright;
        char      temp[KLENGTH], dirname[KLENGTH], toolbox_name[KLENGTH]; 

        /*
         * get file object representing *.h page
         */
        if (!kcms_get_attributes(program,
				 KCMS_PARENT,          &toolbox,
				 KCMS_CMOBJ_GEN_INCL,  &file_object,
				 KCMS_CMOBJ_TYPE,      &object_type,
				 KCMS_END)
	    || !kcms_query_bit(program, KCMS_CMOBJ_FLAGS, KCMS_BIT_CMOBJ_FORCE,
			       &force_flag))
	    return(FALSE);

        /*
         * 1st time *.h page has been created - create file object, so that 
         * kcms registers the *.h file in the kcms database
         */
	if (file_object == NULL)
        {
	    ksprintf(temp, "%s/src/%s.h", opath, oname);
            file_object = kcms_create_fileobj(program, temp, NULL,
                                               KCMS_FOBJ_TYPE_SRC,
                                               KCMS_FOBJ_SUBTYPE_INCL,
                                               KCMS_FOBJ_GEN_INCL,
                                               KCMS_FOBJ_ACCESS_RDWR);
            if (file_object == NULL)
                return(FALSE);
            isnewfile = TRUE;
            filepath = temp;
        }

	/*
         * the *.h file has already been created & added to kcms database;
         * simply get the path to the *.h page.
         */
	else if (!kcms_get_attribute(file_object, KCMS_PATH, &filepath))
            return(FALSE);

	/*
         *  if somehow the database GHOST_HEADER key got deleted from the kcms
         *  database, but the man1 file really is there, don't want to
         *  clobber it!
         */
        if ((isnewfile) && (kaccess(filepath, R_OK) == 0))
        {
            kerror(KCODEGEN, "kgen_clui_generate",
                   "WARNING!  CMS database appears to have been corrupted; cms dbm key GHOST_HEADER is missing, but the %s.h file does exist.  Regeneration of *.h file will NOT take place. Adding GHOST_HEADER cms key to database; next run of ghostwriter should be OK.", oname);
	    return(FALSE);
        }

        /*
         *  see if *.h file already exists.  If so, prompt to over-write,
         *  return TRUE if the answer is NO
         */
        if (!force_flag &&
	    koverwrite(kget_notify(), filepath) == FALSE)
            return(TRUE);
   
        /*
         *  create & open *.h file
         */
	if (kaccess(filepath, R_OK) != 0)
	   isnewfile = TRUE;
        if ((file = kfopen(filepath, "w"))== NULL)
        {
	     kerror(KCODEGEN, "kgen_clui_generate_hfile", 
		    "Could not create file '%s'", filepath);
             return(FALSE);
        }
   
       /*
        * begin *.h file with RCS header
        */
        kgen_rcs_hhdr(file);
   
       /*
        *  follow RCS header with Khoros copyright
        */
        copyright = kcms_get_copyright(toolbox, KCMS_TB_COPYRIGHT_LONG,
				       KCMS_LANG_C);

        if (copyright != NULL) 
            kfprintf(file, "%s\n", copyright);
	kfree(copyright);
        
   
       /*
        *  print the include file header 
        */
	ksprintf(temp, "Include file for %s", oname);
	kgen_inclfile_hdr(file, temp, prog_spec[AUTHORS], NULL);

       /*
        *  print ifdef's
        */
        kfprintf(file, "#ifndef _%s_h_\n", oname);
        kfprintf(file, "#define _%s_h_\n\n", oname);
   
       /*
        *  add in -include_includes from .prog file
        */ 
	kgen_inclfile_incl_comm(file);

       /*
        *  add in #includes
        */
	(void) kstring_lower(tb_name, toolbox_name);
        if (object_type == KCMS_XVROUTINE && !hybrid)
	{
	    kfprintf(file, "#include <%s.h>\n", toolbox_name);
	    kfprintf(file, "#include <xvinclude.h>\n");
	    kdirname(filepath, dirname);
	    kfprintf(file, "#include \"form_info.h\"\n");
	    ksprintf(temp, "%s/form_info.h", dirname);
	    if (kaccess(temp, R_OK) != 0)
		kcreat(temp, 0666);
	}
        else if (hybrid)
	{
	    kfprintf(file, "#include <%s.h>\n", toolbox_name);
	    kfprintf(file, "#include <xvinclude.h>\n");
	}
	else
	{
	    kfprintf(file, "#include <%s.h>\n", toolbox_name);
	}
   
	/*
	 *  add in -include_includes
	 */
	kfprintf(file, "\n");
	kcms_get_attribute(program, KCMS_CMOBJ_GEN_LIBNAME, &lname);
        kfprintf(file,  "/* -include_includes */\n");
	if (kstrlen(prog_spec[INCLUDE_INCLUDES]) > 0) 
             kfprintf(file, "%s",prog_spec[INCLUDE_INCLUDES]);
	else if (lname != NULL)
	     kfprintf(file, "#include <%s/%s.h>\n", lname, lname);
        kfprintf(file,  "/* -include_includes_end */\n\n\n");
   
	/*
	 * add in -include_defines from .prog file
	 */
	kgen_inclfile_def_comm(file);

        kfprintf(file,  "/* -include_defines */\n");
	if (kstrlen(prog_spec[INCLUDE_DEFINES]) > 0) 
             kfprintf(file, "%s", prog_spec[INCLUDE_DEFINES]);
        kfprintf(file,  "/* -include_defines_end */\n\n");


	/*
	 *  get the file object associated with the *.pane file, so we
         *  can get the path to the *.pane file and put the #define PANEPATH
         *  line in the *.h file that is used in the main to pass to 
         *  kgen_initialize()
	 */
	if (!kcms_get_attribute(program, KCMS_CMOBJ_UIS_PANE, &file_object)
	    || file_object == NULL
	    || !kcms_get_attribute(file_object, KCMS_PATH, &filepath))
	{
	   return(FALSE);
	}
	kfprintf(file, "#define PANEPATH \"%s\"\n", filepath);


       /*
        *  generate C structure for command-line user interface
        */ 
	kgen_inclfile_typedef_comm(file);

        if ((kgen_arg_total > 0) || (object_type == KCMS_XVROUTINE))
        {
           kfprintf(file, "typedef struct _clui_info_struct {\n\n", oname);
           print_struct_fields(program, file);
           kfprintf(file, "} clui_info_struct;\n\n");
        }
	kfprintf(file,  "/* -include_typedefs */\n");
	if (kstrlen(prog_spec[INCLUDE_TYPEDEFS]) > 0) 
             kfprintf(file, "%s",prog_spec[INCLUDE_TYPEDEFS]);
	kfprintf(file,  "/* -include_typedefs_end */\n\n\n");

        
       /*
        *  put in necessary global variables
        */
	kgen_inclfile_global_comm(file);
	if ((kgen_arg_total > 0) || (object_type == KCMS_XVROUTINE))
            kfprintf(file, "extern clui_info_struct *clui_info;\n");
	if (object_type == KCMS_XVROUTINE && !hybrid)
	    kfprintf(file, "extern gui_info_struct *gui_info;\n\n");

	kfprintf(file,  "/* -include_variables */\n");
	if (kstrlen(prog_spec[INCLUDE_VARIABLES]) > 0) 
             kfprintf(file, "%s", prog_spec[INCLUDE_VARIABLES]);
	kfprintf(file,  "/* -include_variables_end */\n\n\n");
   
       /*
        *  add in -include_macros from .prog file
        */ 
	kgen_inclfile_macro_comm(file);

        kfprintf(file,  "/* -include_macros */\n");
	if (kstrlen(prog_spec[INCLUDE_MACROS]) > 0) 
             kfprintf(file, "%s",prog_spec[INCLUDE_MACROS]);
        kfprintf(file,  "/* -include_macros_end */\n\n");

	/*
	 *  add in -include_routine_defs
 	 */
	kgen_inclfile_routine_comm(file);
	kfprintf(file,  "void main PROTO((int, char **, char **));\n");
	kfprintf(file,  "void %s_get_args PROTO((kform *));\n", oname);
	kfprintf(file,  "void %s_usage_additions PROTO((void));\n", oname);
	kfprintf(file,  "void %s_free_args PROTO((int, kaddr));\n\n", oname);

        kfprintf(file,  "/* -include_routines */\n");
	if (kstrlen(prog_spec[INCLUDE_ROUTINES]) > 0) 
             kfprintf(file, "%s",prog_spec[INCLUDE_ROUTINES]);
        kfprintf(file,  "/* -include_routines_end */\n\n");

   
       /*
 	*  NOTE: -include_additions are outdated as of Khoros 2.0.
        *         they are supported only for backwards compatibility.
        *         add in -include_additions from .prog file if they exist,
        *         but print warning message.
        */ 
	status = TRUE;
	if (kstrlen(prog_spec[INCLUDE_ADDITIONS]) > 0) 
	{
            kfprintf(file,  "/* -include_additions */\n");
            kfprintf(file, "%s",prog_spec[INCLUDE_ADDITIONS]);
            kfprintf(file,  "/* -include_additions_end */\n\n");
	    status = 2;
	}
   
       /*
        *  end include file
        */
        kfprintf(file, "#endif\n");
   
        kfclose(file);
        kannounce(KCODEGEN, routine, "done generating %s.h", oname);
	kinfo(KHOSTILE, "your %s.h file is merely so much trash.", oname);
        return(status);
}

/*-----------------------------------------------------------
|
|  Routine Name: print_struct_fields
|
|       Purpose: Prints the necessary fields to the CLUI Information
|                structure generated in the *.h file
|
|         Input: file - open stream to *.h file
|
|        Output: none
|    Written By: Danielle Argiro
|          Date: Sept 2, 1993
| Modifications: Converted from Khoros 1.0 (DA)
|
-------------------------------------------------------------*/

static void print_struct_fields(
   kobject   object,
   kfile    *file)
{
        int         indx;
	kobject     form_file = NULL;
	int         object_type;
        kselection *selection;
   

	/* this should really be returning FALSE */
	if (!kcms_get_attributes(object,
				 KCMS_CMOBJ_TYPE,     &object_type,
				 KCMS_CMOBJ_UIS_FORM, &form_file,
				 KCMS_END))
	   return;

       /*
        * print out fields corresponding to required arguments
        */
	indx = 0;
        while (indx < kgen_req_num)
	{
	   selection = kgen_req_sels[indx++];
	   if ((selection->type == KUIS_MUTEXCL) ||
	       (selection->type == KUIS_MUTINCL) ||
	       (selection->type == KUIS_GROUP))
                print_group_fields(object, file, selection, TRUE, FALSE);

	   else if (selection->type == KUIS_TOGGLE)
                print_toggle_fields(file, selection);

	   else if ((selection->type == KUIS_LIST)  ||
		    (selection->type == KUIS_CYCLE) ||
		    (selection->type == KUIS_DISPLAYLIST))
                print_list_fields(file, selection);

	   else print_single_fields(object, file, selection);
	}
   
       /*
        * print out fields corresponding to optional arguments
        */
	if (object_type == KCMS_XVROUTINE && form_file != NULL)
	{
    	    kfprintf(file, "/*\n *  Alternate UIS file (optional infile)\n */\n");
            kfprintf(file, "char\t*form_file;\t/* Alternate UIS file FILENAME */\n");
            kfprintf(file, "int\t form_flag;\t/* Alternate UIS file FLAG */\n\n");
	    kfprintf(file, "/*\n *  (x,y) position for automatic GUI placement */\n\n");
            kfprintf(file, "int\t x_int;\t\t/* x position INT */\n");
            kfprintf(file, "int\t x_flag;\t/* x position FLAG */\n\n");
            kfprintf(file, "int\t y_int;\t\t/* y position INT */\n");
            kfprintf(file, "int\t y_flag;\t/* y position FLAG */\n\n");

	}

        indx = 0;
        while (indx < kgen_opt_num)
	{
	   selection = kgen_opt_sels[indx++];

           if ((selection->type == KUIS_MUTEXCL) ||
               (selection->type == KUIS_MUTINCL) ||
               (selection->type == KUIS_GROUP))
                print_group_fields(object, file, selection, FALSE, FALSE);

	   else if (selection->type == KUIS_TOGGLE)
                print_toggle_fields(file, selection);

	   else if ((selection->type == KUIS_LIST)  ||
		    (selection->type == KUIS_CYCLE) ||
		    (selection->type == KUIS_DISPLAYLIST))
                print_list_fields(file, selection);

	   else print_single_fields(object, file, selection);
	}
}


/*-----------------------------------------------------------
|
|  Routine Name: print_single_fields
|
|       Purpose: Prints a single field to the CLUI Information
|                structure being generated in the .h file
|
|         Input: object    - The software object being generated.
|                file      - open stream to *.h page.
|		 selection - selection representing argument.
|
|        Output: none
|    Written By: Danielle Argiro
|          Date: March 30, 1994 
| Modifications: 
|
-------------------------------------------------------------*/

static void print_single_fields(
   kobject     object,
   kfile      *file,
   kselection *selection)
{
        int      optional;
	kobject  form_file;
	int      object_type;
	char    *req_str, *variable, *description;
    

	/* this should really be returning FALSE */
	if (!kcms_get_attributes(object,
				 KCMS_CMOBJ_TYPE,     &object_type,
				 KCMS_CMOBJ_UIS_FORM, &form_file,
				 KCMS_END))
	   return;

 	kvf_get_attributes(selection->back_kformstruct,
			   KVF_DESCRIPTION, &description,
			   KVF_OPTIONAL,    &optional,
			   NULL);
	if (description == NULL) 
	    description = kstrdup("no description");
	variable = ktoken_to_string(selection->var_token); 

        if (optional)
           req_str = kstrdup("optional");
        else req_str = kstrdup("required");
    

        switch (selection->type) {
    	    
             case KUIS_INPUTFILE:
    	          kfprintf(file, "/*\n *  %s (%s infile)\n */\n",
    			  description, req_str);
           	  kfprintf(file, "char\t*%s_file;\t/* %s FILENAME */\n", 
    		          variable, description);
           	  kfprintf(file, "int\t %s_flag;\t/* %s FLAG */\n\n", 
    		          variable, description);
    	          break;
    
             case KUIS_OUTPUTFILE:
    	          kfprintf(file, "/*\n *  %s (%s outfile)\n */\n",
    			  description, req_str);
           	  kfprintf(file, "char\t*%s_file;\t/* %s FILENAME */\n", 
    		          variable, description);
           	  kfprintf(file, "int\t %s_flag;\t/* %s FLAG */\n\n", 
    		          variable, description);
    	          break;
    
             case KUIS_FLOAT:
    	          kfprintf(file, "/*\n *  %s (%s float)\n */\n",
    			  description, req_str);
    	          kfprintf(file, "float\t %s_float;\t/* %s FLOAT */\n",
                          variable, description);
                  kfprintf(file, "int\t %s_flag;   \t/* %s FLAG */\n\n",
                          variable, description);
                  break;
    
             case KUIS_DOUBLE:
    	          kfprintf(file, "/*\n *  %s (%s double)\n */\n",
    			  description, req_str);
    	          kfprintf(file, "double\t %s_double;\t/* %s DOUBLE */\n",
                          variable, description);
                  kfprintf(file, "int\t %s_flag;   \t/* %s FLAG */\n\n",
                          variable, description);
                  break;
    
             case KUIS_INTEGER:
		  if (((kstrcmp(variable, "x") == 0) ||
                       (kstrcmp(variable, "y") == 0)) &&
		      (object_type == KCMS_XVROUTINE) &&
		     form_file != NULL)
                     break;

    	          kfprintf(file, "/*\n *  %s (%s integer)\n */\n",
    			  description, req_str);
    	          kfprintf(file, "int\t %s_int; \t/* %s INT */\n",
                          variable, description);
                  kfprintf(file, "int\t %s_flag;\t/* %s FLAG */\n\n",
                          variable, description);
                  break;
    		
             case KUIS_STRING:
             case KUIS_STRINGLIST:
    	          kfprintf(file, "/*\n *  %s (%s string)\n */\n",
    			  description, req_str);
    	          kfprintf(file, "char\t*%s_string;\t/* %s STRING */\n",
                          variable, description);
                  kfprintf(file, "int\t %s_flag;   \t/* %s FLAG */\n\n",
                          variable, description);
                  break;
    
             case KUIS_LOGICAL:
    	          kfprintf(file, "/*\n *  %s (%s logical)\n */\n",
    			  description, req_str);
    	          kfprintf(file, "int\t %s_logic;\t/* %s LOGIC */\n",
                          variable, description);
                  kfprintf(file, "int\t %s_flag;  \t/* %s FLAG */\n\n",
                          variable, description);
                  break;
    
             case KUIS_FLAG:
    	          kfprintf(file, "/*\n *  %s (%s flag)\n */\n",
    			  description, req_str);
                  kfprintf(file, "int\t %s_flag;  \t/* %s FLAG */\n\n",
                          variable, description);
                  break;

        }
	kfree(req_str);
}
		

/*-----------------------------------------------------------
|
|  Routine Name: print_toggle_fields
|
|       Purpose: Prints the toggle fields to the CLUI Information
|                structure being generated in the *.h file.  
|
|         Input: file      - open stream to *.h page
|		 selection - selection corresponding to toggle (-T header)
|
|        Output: none
|    Written By: Danielle Argiro
|          Date: March 30, 1994 
| Modifications: 
|
-------------------------------------------------------------*/

static void print_toggle_fields(
   kfile      *file,
   kselection *selection)
{
	int  optional, toggle_type;
	char *req_str, *description, *variable;

	kvf_get_attributes(selection->back_kformstruct,
			   KVF_OPTIONAL,    &optional,
			   KVF_TOGGLE_TYPE, &toggle_type,
			   KVF_DESCRIPTION, &description,
			   NULL);

	if (description == NULL) 
	    description = kstrdup("no description");
	variable = ktoken_to_string(selection->var_token);
	if (optional) 
	    req_str = kstrdup("Optional");
	else req_str = kstrdup("Required");

	switch(toggle_type) {

            case KUIS_INPUTFILE:
	         kfprintf(file, "/*\n * %s (%s %s toggle)\n", description, 
			 req_str, kgen_ascii_datatype(toggle_type));
                 kgen_clui_print_toggle_vals(file, selection, 
				             KGEN_CLUI_INFO_STRUCT, FALSE);
	         kfprintf(file, " */\n");
                 kfprintf(file,"char\t*%s_toggle;\t\t/* %s INFILE TOGGLE */\n",
                               variable, description);
       	         kfprintf(file, "int\t %s_flag;\t\t/* %s FLAG */\n\n", 
	                  variable, description);
	         break;

            case KUIS_OUTPUTFILE:
	         kfprintf(file, "/*\n * %s (%s %s toggle)\n", description, 
			 req_str, kgen_ascii_datatype(toggle_type));
                 kgen_clui_print_toggle_vals(file, selection, 
					     KGEN_CLUI_INFO_STRUCT, FALSE);
	         kfprintf(file, " */\n");
                 kfprintf(file,"char\t*%s_toggle;\t\t/* %s OUTFILE TOGGLE */\n",
                               variable, description);
       	         kfprintf(file, "int\t %s_flag;\t\t/* %s FLAG */\n\n", 
	                  variable, description);
	         break;

	    case KUIS_INTEGER:
	         kfprintf(file, "/*\n * %s (%s %s toggle)\n", description, 
			 req_str, kgen_ascii_datatype(toggle_type));
                 kgen_clui_print_toggle_vals(file, selection,
				             KGEN_CLUI_INFO_STRUCT, FALSE);
	         kfprintf(file, " */\n");

                 kfprintf(file, "int\t %s_toggle;\t\t/* %s INT TOGGLE */\n",
                               variable, description);
       	         kfprintf(file, "int\t %s_flag;\t\t/* %s FLAG */\n\n", 
	                  variable, description);
	         break;

	    case KUIS_FLOAT:
	         kfprintf(file, "/*\n * %s (%s %s toggle)\n", description, 
			 req_str, kgen_ascii_datatype(toggle_type));
                 kgen_clui_print_toggle_vals(file, selection, 
					     KGEN_CLUI_INFO_STRUCT, FALSE);
	         kfprintf(file, " */\n");
                 kfprintf(file,"float\t %s_toggle;\t\t/* %s FLOAT TOGGLE */\n",
                               variable, description);
       	         kfprintf(file, "int\t %s_flag;\t\t/* %s FLAG */\n\n", 
	                  variable, description);
	         break;

	    case KUIS_DOUBLE:
	         kfprintf(file, "/*\n * %s (%s %s toggle)\n", description, 
			 req_str, kgen_ascii_datatype(toggle_type));
                 kgen_clui_print_toggle_vals(file, selection, 
					     KGEN_CLUI_INFO_STRUCT, FALSE);
	         kfprintf(file, " */\n");
                 kfprintf(file,"double\t %s_toggle;\t\t/* %s DOUBLE TOGGLE */\n",
                               variable, description);
       	         kfprintf(file, "int\t %s_flag;\t\t/* %s FLAG */\n\n", 
	                  variable, description);
	         break;

	    case KUIS_STRING:
	         kfprintf(file, "/*\n * %s (%s %s toggle)\n", description, 
			 req_str, kgen_ascii_datatype(toggle_type));
                 kgen_clui_print_toggle_vals(file, selection, 
					     KGEN_CLUI_INFO_STRUCT, FALSE);
	         kfprintf(file, " */\n");
                 kfprintf(file,"char\t*%s_toggle;\t\t/* %s STRING TOGGLE */\n",
                               variable, description);
       	         kfprintf(file, "int\t %s_flag;\t\t/* %s FLAG */\n\n", 
	                  variable, description);
	         break;

	    case KUIS_LOGICAL:
	         kfprintf(file, "/*\n * %s (%s %s toggle)\n", description, 
			 req_str, kgen_ascii_datatype(toggle_type));
                 kgen_clui_print_toggle_vals(file, selection, 
					     KGEN_CLUI_INFO_STRUCT, FALSE);
	         kfprintf(file, " */\n");
                 kfprintf(file, "int\t %s_toggle;\t\t/* %s LOGICAL TOGGLE */\n",
                               variable, description);
       	         kfprintf(file, "int\t %s_flag;\t\t/* %s FLAG */\n\n", 
	                  variable, description);
	         break;

	    case KUIS_FLAG:
	         kfprintf(file, "/*\n * %s (%s %s toggle)\n", description, 
			 req_str, kgen_ascii_datatype(toggle_type));
                 kgen_clui_print_toggle_vals(file, selection, 
					     KGEN_CLUI_INFO_STRUCT, FALSE);
	         kfprintf(file, " */\n");
                 kfprintf(file, "int\t %s_toggle;\t\t/* %s FLAG TOGGLE */\n",
                               variable, description);
       	         kfprintf(file, "int\t %s_flag;\t\t/* %s FLAG */\n\n", 
	                  variable, description);
	         break;

	}
	kfree(req_str);
}

/*-----------------------------------------------------------
|
|  Routine Name: print_list_fields
|
|       Purpose: Prints the list fields to the C structure being 
|                generated in the .h file. 
|
|         Input: file      - open stream to *.h file
|		 selection - selection corresponding to list argument
|
|        Output: none
|    Written By: Danielle Argiro
|          Date: March 31, 1994 
| Modifications: 
|
-------------------------------------------------------------*/

static void print_list_fields(
   kfile      *file,
   kselection *selection)
{
	int  optional;
	char *req_str, *description, *variable;

	kvf_get_attributes(selection->back_kformstruct,
			   KVF_OPTIONAL,    &optional,
			   KVF_DESCRIPTION, &description,
			   NULL);

	if (description == NULL) 
	    description = kstrdup("no description");
	variable = ktoken_to_string(selection->var_token);
	if (optional) 
	    req_str = kstrdup("Optional");
	else req_str = kstrdup("Required");

	switch(selection->type) {

             case KUIS_LIST:
             case KUIS_DISPLAYLIST:
    	          kfprintf(file, "/*\n *  %s (%s list)\n */\n",
    			  description, req_str);
    	          kfprintf(file, "int\t %s_list;\t/* %s LIST INDEX */\n",
                          variable, description);
                  kfprintf(file, "char\t *%s_label;\t/* %s LIST LABEL */\n",
                          variable, description);
       	          kfprintf(file, "int\t %s_flag;\t/* %s FLAG */\n\n", 
	                   variable, description);
                  break;

             case KUIS_CYCLE:
    	          kfprintf(file, "/*\n *  %s (%s cycle)\n */\n",
    			  description, req_str);
    	          kfprintf(file, "int\t %s_cycle;\t/* %s CYCLE INDEX */\n",
                          variable, description);
                  kfprintf(file, "char\t *%s_label;\t/* %s CYCLE LABEL */\n",
                          variable, description);
       	          kfprintf(file, "int\t %s_flag;\t/* %s FLAG */\n\n", 
	                   variable, description);
                  break;
	}
	kfree(req_str);
}


/*-----------------------------------------------------------
|
|  Routine Name: print_group_fields
|
|       Purpose: Prints the fields associated with a required
|                or optional ME/MI group to the C structure being 
|                generated in the .h file
|
|         Input: file        - open stream to *.h page
|                group_start - 1st member in the ME/MI group
|        Output: none
|    Written By: Danielle Argiro
|          Date: Sept 2, 1993
| Modifications: 
|
-------------------------------------------------------------*/

static void print_group_fields(
    kobject     object,
    kfile      *file,
    kselection *group_start,
    int        required,
    int        nested)
{
	kselection *groupmember;
	char       type_temp[KLENGTH], req_temp[KLENGTH];

        if (group_start->type == KUIS_MUTEXCL)
	    ksprintf(type_temp, "ME");
	else if (group_start->type == KUIS_MUTINCL)
	    ksprintf(type_temp, "MI");
	else ksprintf(type_temp, " ");
	
	if (required)
	     ksprintf(req_temp, "required");
	else ksprintf(req_temp, "optional");

        if (nested)
	    kfprintf(file, "\t/*---- nested %s %s group ----*/\n",
		     req_temp, type_temp);
	else kfprintf(file, "/*----- %s %s group ----*/\n",
		     req_temp, type_temp);

        groupmember = group_start->group_next;
        while (groupmember != NULL)
        {
	    if (groupmember->type == KUIS_BLANK)
	    {
                groupmember = groupmember->next;
	 	continue;
	    }
            if ((groupmember->type == KUIS_MUTEXCL) ||
               (groupmember->type == KUIS_MUTINCL) ||
               (groupmember->type == KUIS_GROUP))
                print_group_fields(object, file, groupmember, FALSE, TRUE);

            else if (groupmember->type == KUIS_TOGGLE)
                print_toggle_fields(file, groupmember);

            else if ((groupmember->type == KUIS_LIST)  ||
                    (groupmember->type == KUIS_CYCLE) ||
                    (groupmember->type == KUIS_DISPLAYLIST))
                print_list_fields(file, groupmember);

	    else print_single_fields(object, file, groupmember);

            groupmember = groupmember->next;
        }

        if (nested)
	    kfprintf(file, "\t/*-- nested %s %s group end --*/\n\n",
                      req_temp, type_temp);
	else kfprintf(file, "/*-- %s %s group end ---*/\n\n",
		      req_temp, type_temp);
}
