/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>  Routines for command line argument error checking    <<<<
   >>>>                                                       <<<<
   >>>>   Private:                                            <<<<
   >>>>              kgen_clui_cmdline()                      <<<<
   >>>>              kgen_clui_cmdline_mut_excl()             <<<<
   >>>>              kgen_clui_cmdline_mut_incl()             <<<<
   >>>>              kgen_clui_cmdline_loose_group()          <<<<
   >>>>    Static:                                            <<<<
   >>>>    Public:                                            <<<<
   >>>>              kgen_interpret_clui_usage		      <<<<
   >>>>              kgen_interpret_clui_args		      <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

/*-----------------------------------------------------------
|
|  Routine Name: kgen_clui_cmdline
|
|       Purpose: This routine makes sure that the user has
|                provided values for required arguments on the command
|                line when interactive prompting is NOT being used.
|
|         Input: tbname    - name of toolbox in which program is located
|                oname     - name of program running
|                usage_additions - usage_additions routine of program running
|                selection - selection corresponding to program argument
|                present   - returns TRUE if selection is present on cmd
|                            line or in answer file, FALSE otherwise.
|        Output: none
|       Returns:
|    Written By: Danielle Argiro
|          Date: March 30, 1994
| Modifications:
|
------------------------------------------------------------*/

int kgen_clui_cmdline(
   char        *tbname,
   char        *oname,
   void       (*usage_additions)(void),
   kselection  *selection,
   int         *present)
{
	int    optional;
	kform  *form;
	char   *variable;

	form = selection->back_form;

	switch(selection->type)
	{
	    case KUIS_INPUTFILE:
	    case KUIS_OUTPUTFILE:
	    case KUIS_INTEGER:
	    case KUIS_FLOAT:
	    case KUIS_DOUBLE:
	    case KUIS_STRING:
	    case KUIS_STRINGLIST:
	    case KUIS_LOGICAL:
	    case KUIS_FLAG:
	    case KUIS_TOGGLE:
	    case KUIS_LIST:
	    case KUIS_CYCLE:
	    case KUIS_DISPLAYLIST:
		 /*
	          *  if the user provided any arguments & values on the command
                  *  line, those would have been already stored in the form
                  *  tree representing the pane, during an earlier call to
                  *  kvf_modify_form_from_cmd.  Here, if a selection's modified
                  *  field is set to TRUE, that means that it was on the command
                  *  line, and kvf_modify_form_from_cmd picked it up and stored
                  *  it properly.  just need to be sure that all required 
                  *  arguments have been specified.
	          */
	         kvf_get_attributes(selection->back_kformstruct, 
			            KVF_MODIFIED, present,
			            KVF_OPTIONAL, &optional,
				    NULL);

	         variable = ktoken_to_string(selection->var_token);

		 /* KVF_MODIFIED set to -1 indicates invalid input */
		 if (*present == -1)
		 {
                     kfprintf(kstderr, "\nError: invalid input specified for argument '%s'\n\n", 
			          variable);
	             kexit(KEXIT_FAILURE);
		 }
	         if ((*present != TRUE) && (!optional))
                 {
	
		     /* KVF_MODIFIED set to 0 indicates lack of input */
		     kfprintf(kstderr, "\nError: %s is a required argument\n\n",
			      variable);
		
		     /* print usage & exit */
	             kgen_interpret_clui_usage(kstderr, form, tbname, oname,
					       usage_additions);
	             kexit(KEXIT_FAILURE);
                 }
	         break;
	
	    case KUIS_MUTEXCL:
		 if (!kgen_clui_cmdline_mut_excl(tbname, oname, usage_additions,
					         selection, present))
		     return(FALSE);
	         break;

            case KUIS_MUTINCL:
		 if (!kgen_clui_cmdline_mut_incl(tbname, oname, usage_additions,
                                                 selection, present))
		     return(FALSE);
	         break;

            case KUIS_GROUP:
		 if (!kgen_clui_cmdline_loose(tbname, oname, usage_additions,
                                               selection, present))
		     return(FALSE);
	         break;

	}
	return(TRUE);
    
} /* end kgen_clui_cmdline */

/*-----------------------------------------------------------
|
|       Routine: kgen_clui_cmdline_mut_excl
|
|       Purpose: This routine makes sure that the user has properly
|                provided values for ME group arguments on the command line.
|                If the ME group is required, they must have provided
|                one and only one value.  If the ME group is optional,
|                they may have provided one or no values.
|
|         Input: tbname    - name of toolbox in which program is located
|                oname     - name of program running
|                usage_additions - usage_additions routine of program running
|                group_start - header of the group list
|                oname       - name of the program being generated
|
|        Output: present     - TRUE if one and only one value
|                              was entered for required ME group;
|                              TRUE if one or zero values was
|                              entered for optional ME group;
|                              FALSE otherwise
|
|       Returns: TRUE on success, FALSE on error
|    Written By: Danielle Argiro
|          Date: March 31, 1994
| Modifications: 
|
------------------------------------------------------------*/
int kgen_clui_cmdline_mut_excl(
   char       *tbname,
   char       *oname,
   void      (*usage_additions)(void),
   kselection *group_start,
   int        *present)
{
	kselection *groupmember;
        int        optional;
	int        presence_flag, me_count;

	/*
         *  print code to get values for all members of ME group
         */
        me_count = 0;
	presence_flag = *present = FALSE;
	kvf_get_attribute(group_start->back_kformstruct,
			  KVF_OPTIONAL, &optional);
	groupmember = group_start->group_next;
        while (groupmember != NULL)
        {
	    if (groupmember->type == KUIS_BLANK)
	    {
		groupmember = groupmember->next;
		continue;
	    }
 
            /* ME group has illegal ME group nested inside */
            if (groupmember->type == KUIS_MUTEXCL)
            {
                kerror(KCODEGEN, "kgen_clui_cmdline_mut_excl",
                       "Nesting of MI groups within MI groups not supported");
                return(FALSE);
            }

            /* ME group has MI group nested inside */
            else if (groupmember->type == KUIS_MUTINCL)
            {
                if (!kgen_clui_cmdline_mut_incl(tbname, oname, usage_additions,
					         groupmember, &presence_flag))
                    return(FALSE);

                if (presence_flag) me_count++;
            }

            /* ME group has loose group nested inside */
            else if (groupmember->type == KUIS_GROUP)
            {
                if (!kgen_clui_cmdline_loose(tbname, oname, usage_additions,
                                             groupmember, &presence_flag))
                    return(FALSE);

                if (presence_flag) me_count++;
            }

            /* ME group has "regular" argument inside */
            else
            {
		kvf_get_attribute(groupmember->back_kformstruct,
                                  KVF_MODIFIED, &presence_flag);
                if (presence_flag) me_count++;
	    }
            groupmember = groupmember->next;	
        }

	/*
         *  if the ME group is required, must make sure that one 
         *  and only one value was provided for all members of group
         */
        if ((!optional) && (me_count == 0))
	{
            kfprintf(kstderr, "Error: You MUST specify one of :\n");
	    kgen_print_group_choices(group_start, NULL);
	    kgen_interpret_clui_usage(kstderr, group_start->back_form, 
				      tbname, oname, usage_additions);
	    kexit(KEXIT_FAILURE);
        }
	else if (me_count > 1)
	{
            kfprintf(kstderr, "Error: You may specify ONLY ONE of :\n");
	    kgen_print_group_choices(group_start, NULL);
	    kgen_interpret_clui_usage(kstderr, group_start->back_form, 
				      tbname, oname, usage_additions);
	    kexit(KEXIT_FAILURE);
	}
	else if (me_count == 1) *present = TRUE;
	return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: kgen_clui_cmdline_mut_incl
|
|       Purpose: This routine makes sure that the user has properly
|                provided values for MI group arguments on the command line.
|                The user must have provided values for all or none of the 
|                arguments in the group.
|
|         Input: tbname    - name of toolbox in which program is located
|                oname     - name of program running
|                usage_additions - usage_additions routine of program running
|                group_start - header of the group list
|                oname       - name of the program being generated
|
|        Output: presence_flag - TRUE if all or no values were entered
|                                FALSE otherwise
|       Returns: TRUE on success, FALSE on error
|    Written By: Danielle Argiro
|          Date: March 31, 1994
| Modifications:
|
------------------------------------------------------------*/

int kgen_clui_cmdline_mut_incl(
   char       *tbname,
   char       *oname,
   void      (*usage_additions)(void),
   kselection *group_start,
   int        *present)
{
        int        mi_count, group_size, presence_flag;
        kselection *groupmember;

	/*
	 *  print code to get values for all members of MI group
	 */
        mi_count = 0;
	presence_flag = *present = FALSE;
        group_size = kgen_get_group_size(group_start);

	groupmember = group_start->group_next;
        while (groupmember != NULL)
        {
	    if (groupmember->type == KUIS_BLANK)
	    {
		groupmember = groupmember->next;
		continue;
	    }

	    /* MI group has illegal MI group nested inside */
            if (groupmember->type == KUIS_MUTINCL)
            {
                kerror(KCODEGEN, "kgen_clui_cmdline_mut_incl",
                       "Nesting of MI groups within MI groups not supported");
                return(FALSE);
            }

            /* MI group has ME group nested inside */
            else if (groupmember->type == KUIS_MUTEXCL)
            {
                if (!kgen_clui_cmdline_mut_excl(tbname, oname, usage_additions,
				                groupmember, &presence_flag))
                    return(FALSE);

                if (presence_flag) mi_count++;
            }

            /* MI group has loose group nested inside */
            else if (groupmember->type == KUIS_GROUP)
            {
                if (!kgen_clui_cmdline_loose(tbname, oname, usage_additions,
				             groupmember, &presence_flag))
		    return(FALSE);

                if (presence_flag) mi_count++;
            }

            /* MI group has "regular" argument inside */
            else
            {
		kvf_get_attribute(groupmember->back_kformstruct,
                                  KVF_MODIFIED, &presence_flag);
                if (presence_flag) mi_count++;
            }
            groupmember = groupmember->next;	
        }

	/*
	 *  make sure values were provided for all or no members of group
	 */
	if ((mi_count != 0) && (mi_count != group_size))
	{
            kfprintf(kstderr, "Error: You MUST specify ALL or NONE of:\n");
	    kgen_print_group_choices(group_start, NULL);
            kgen_interpret_clui_usage(kstderr, group_start->back_form, 
				      tbname, oname, usage_additions);
	    kexit(KEXIT_FAILURE);
	}
	else if (mi_count == group_size) *present = TRUE;
	return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: kgen_clui_cmdline_loose
|
|       Purpose: This routine makes sure that the user has properly
|                provided values for loose group arguments on the command line.
|                The user must have provided values for at least one of the
|                arguments in the group.
|
|         Input: tbname    - name of toolbox in which program is located
|                oname     - name of program running
|                usage_additions - usage_additions routine of program running
|                group_start - header of the group list
|                oname       - name of the program being generated
|
|        Output: present     - TRUE if at least one value was entered
|                              FALSE otherwise
|       Returns: TRUE on success, FALSE on error
|    Written By: Danielle Argiro
|          Date: March 31, 1994
| Modifications:
|
------------------------------------------------------------*/

int kgen_clui_cmdline_loose(
   char       *tbname,
   char       *oname,
   void      (*usage_additions)(void),
   kselection *group_start,
   int        *present)
{
        int        gp_count, optional;
	int        presence_flag;
        kselection *groupmember;

	/*
	 *  print code to get values for all members of MI group
	 */
        gp_count = 0;
	presence_flag = *present = FALSE;

	groupmember = group_start->group_next;
        while (groupmember != NULL)
        {
	    if (groupmember->type == KUIS_BLANK)
	    {
		groupmember = groupmember->next;
		continue;
	    }

            /* loose group has illegal loose group nested inside */
            if (groupmember->type == KUIS_GROUP)
            {
                kerror(KCODEGEN, "kgen_clui_cmdline_loose",
                       "Nesting of MI groups within MI groups not supported");
                return(FALSE);
            }

            /* loose group has ME group nested inside */
            else if (groupmember->type == KUIS_MUTEXCL)
            {
                if (!kgen_clui_cmdline_mut_excl(tbname, oname, usage_additions,
						 groupmember, &presence_flag))
                    return(FALSE);

                if (presence_flag) gp_count++;
            }

            /* loose group has MI group nested inside */
            else if (groupmember->type == KUIS_GROUP)
            {
                if (!kgen_clui_cmdline_mut_incl(tbname, oname, usage_additions,
						 groupmember, &presence_flag))
                    return(FALSE);

                if (presence_flag) gp_count++;
            }

            /* loose group has "regular" argument inside */
            else
            {
		kvf_get_attribute(groupmember->back_kformstruct,
                                  KVF_MODIFIED, &presence_flag);
                if (presence_flag) gp_count++;
            }

            groupmember = groupmember->next;	
        }

        /*
         *  make sure values were provided for all or no members of group
         */
	kvf_get_attribute(group_start->back_kformstruct,
			  KVF_OPTIONAL, &optional);

        if ((gp_count == 0) && (!optional))
        {
            kfprintf(kstderr, "Error: You MUST specify AT LEAST ONE of:\n");
            kgen_print_group_choices(group_start, NULL);
            kgen_interpret_clui_usage(kstderr, group_start->back_form, 
				      tbname, oname, usage_additions);
	    kexit(KEXIT_FAILURE);
        }
	else if (gp_count > 0) *present = TRUE;

	return(TRUE);
}


/************************************************************
*
*  Routine Name: kgen_interpret_clui_usage - interpret usage for software object
*
*       Purpose: Given a software object, interprets the usage for the 
*                program from its *.pane file, and prints the usage to 
*                the file given.  It might be kstderr (for -U on cmd line),
*                the file associated with the man page (for kman), or the
*                file associated with the help page (for online help).
*
*         Input: file   - file to which to print usage 
*                form   - the form tree associated with 
*                         the program objects's *.pane file
*                tbname - name of toolbox object in which program object exists
*                oname  - name of program object
*                usage_additions - usage_additions() routine for program
*                                  (for kroutines & xvroutines) NULL otherwise
*        Output: 
*       Returns: TRUE if successful, FALSE otherwise.
*    Written By: Danielle Argiro 
*          Date: March 18, 1994
*
*************************************************************/

int kgen_interpret_clui_usage(
    kfile      *file, 
    kform      *form,
    char       *tbname,
    char       *oname,
    void      (*usage_additions)(void))
{
	char    *prog_desc = NULL;
	kobject toolbox = NULL, program = NULL;

	/*
	 *  sanity check
	 */
	if (form == NULL)
	{
	    kerror(KCODEGEN, "kgen_interpret_clui_usage",
		   "Can't interpret syntax on NULL pane!");
	    return(FALSE);
	}

	/*
	 *  get information off the pane: set global variables such as
         *  kgen_req_sel, kgen_opt_sels printing usage.
	 */
        if (!(kgen_clui_get_pane_info(form)))
        {
            kerror(KCODEGEN, "kgen_interpret_clui_usage", 
		  "Unable to obtain UIS information from *.pane form tree\n");
            kexit(KEXIT_FAILURE);
        }

	kfprintf(file, "Usage for %s: ", oname);

	/*
	 *  a kroutine or xvroutine will have a usage_additions routine, in 
         *  which an fprintf stmt to print the short description is hardwired.
	 */
	if (usage_additions != NULL)
            usage_additions();

	/*
	 *  a pane object has no usage_additions routine (or any code of its 
         *  own) so we have to rely on kcms to get the short program description
         *  from the -M line of the *.pane UIS file.  prone to failure, but
         *  give it a try anyway, and print if we find anything.
	 */
	else 
	{
	    toolbox = kcms_open_toolbox(tbname);
	    if (toolbox != NULL)
               program = kcms_open_cmobj(toolbox, oname);
	    if (program != NULL)
	        kcms_get_attribute(program, KCMS_CMOBJ_SHORT_DESC, &prog_desc);
	    if (prog_desc != NULL)
		kfprintf(file, "%s", prog_desc);
	}

	kfprintf(file, "\n%% %s\n", oname);

	/*
	 * kgen_clui_print_usage() does the real work of printing usage.
	 */
        kgen_clui_print_usage(file);

	return(TRUE);
}

/************************************************************
*
*  Routine Name: kgen_interpret_clui_args - gets arguments for software object off command line, from interactive prompting, or from answer file
*
*       Purpose: Given a software object, interprets the arguments for the
*                program from its *.pane file, and gets those arguments
*                either from the command line, from the user with interactive
*                prompting, or from an answer file.
*
*         Input: tbname    - name of toolbox in which software object exists
*                oname     - name of software object
*                usage_additions - usage_additions() routine for program
*                                  (for kroutines & xvroutines) NULL otherwise
*                form      - the form tree associated with 
*                            the program objects's *.pane file
*                prompting - TRUE if we are doing interactive
*                            prompting as requested by [-P].
*        Output:
*       Returns: TRUE if successful, FALSE otherwise.
*    Written By: Danielle Argiro
*          Date: March 30, 1994
*
*************************************************************/

int kgen_interpret_clui_args(
    char       *tbname,
    char       *oname,
    void      (*usage_additions)(void),
    kform      *form,
    int        prompting)
{
	int indx = 0;
	int dummy;

        /*
         *  sanity check
         */
        if (form == NULL)
        {
            kerror(KCODEGEN, "kgen_interpret_clui_args",
                   "Can't interpret syntax on NULL pane!");
            return(FALSE);
        }

        if (!(kgen_clui_get_pane_info(form)))
        {
            kerror(KCODEGEN, "kgen_interpret_clui_args",
                  "Unable to obtain UIS information from *.pane form tree\n");
            kexit(KEXIT_FAILURE);
        }

	/*UPDATE: huh? */
        if (oname == NULL)
        {
            kinfo(KSTANDARD, "Warning: Unknown program name");
            oname = kstrdup(" ");
        }

	if (prompting)
	{
	    indx = 0;
	    while (indx < kgen_req_num)
                kgen_clui_prompt(kgen_req_sels[indx++], &dummy);

	    indx = 0;
	    while (indx < kgen_opt_num)
                kgen_clui_prompt(kgen_opt_sels[indx++], &dummy);
	}
	else
	{
	    indx = 0;
	    while (indx < kgen_req_num)
	    {
                if (!(kgen_clui_cmdline(tbname, oname, usage_additions,
				        kgen_req_sels[indx++], &dummy)))
		    return(FALSE);
	    }
	    indx = 0;
            while (indx < kgen_opt_num)
            {
                if (!(kgen_clui_cmdline(tbname, oname, usage_additions,
				        kgen_opt_sels[indx++], &dummy)))
                    return(FALSE);
            }
	}

        return(TRUE);
}
