/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            File Title
   >>>>
   >>>>  Static:
   >>>>  Private:
   >>>>  Public:
   >>>>             generate_cmobj()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

static int _kcms_copy_template PROTO((kobject, kstring, kstring, int));

typedef struct _InfoFobj
{
   int    attribute;
   int    subtype;
   char  *suffix;
   char  *template;
}
InfoFobj;

InfoFobj info_details[] =
{
   {KCMS_CMOBJ_TODO, KCMS_FOBJ_SUBTYPE_TODO, "todo", "todo"},
#if 0
   /*--------------------------------------------------------------------
   |	for 2.0.2, these files will be supported, but not created for
   |	new software objects.  In 2.1 the done file will not be supported
   |	at all.
   --------------------------------------------------------------------*/
   {KCMS_CMOBJ_BUGS, KCMS_FOBJ_SUBTYPE_BUGS, "bugs", "bugs"},
   {KCMS_CMOBJ_DONE, KCMS_FOBJ_SUBTYPE_DONE, "done", "done"},
#endif
   {KCMS_CMOBJ_CHANGELOG,
    KCMS_FOBJ_SUBTYPE_CHANGELOG, "changelog", "changelog"},
   {-1, -1, NULL, NULL}
};

#define KCMS_TEMPLATE_SRC	\
		"$BOOTSTRAP/objects/library/kcms/templates/src/lkroutine.c"

/*-----------------------------------------------------------
| Routine Name:	_kcms_cmobj_destroy_return - call kcms_destroy, return NULL
|
| Purpose:	This routine is a helper routine that removes the newly
|		created object if an error occurs.
|
| Input:	cmobj - the object to remove
| Output:	
| Returns:	NULL so that the call to this routine can be put in a
|		return() statement to clean up after itself on an error
|
| Written By:	Steven Jorgensen
| Date:		Sep 29, 1993
------------------------------------------------------------*/
static kobject
_kcms_cmobj_destroy_return(
			     kobject cmobj)
{
   int save_errno;

   save_errno = errno;
   kcms_set_bit(cmobj, KCMS_CMOBJ_FLAGS, KCMS_BIT_CMOBJ_FORCE);
   kcms_destroy(cmobj);
   errno = save_errno;
   return NULL;
}

/*-----------------------------------------------------------
|
| Routine Name:	_kcms_cmobj_generate - call the code to generate an object
|
| Purpose:	This routine is a helper for kcms_cmobj_generate.  It
|		actually creates the object, and sets all the attributes
|		requested by the calling routine.  This routine does
|		not check the validity of these parameters, it
|		assumes the calling routine has already done it.
|
| Input:	toolbox     - toolbox object to create a new object in
|		oname       - name of object to create
|		otype       - type of new object
|		bname       - binary name of object to create
|		ltype       - langage dependancies
|		lname       - library object name to associate with object
|		cat         - category of the object
|		subcat      - sub-category of the object
|		description - short description of the new object
|		form  	    - path to template form file
|		pane        - path to template pane file
|		old_oname   - object name in template uis files to replace
|		old_bname   - binary name in template uis files to replace
|
|		NOTE - on all the strings above except oname, a NULL
|		indicates that the parameter should not be set.
|
| Output:	
|
| Returns:	the new cmobj on success, or NULL on error
|
| Written By:	Steve Jorgensen and Neil Bowers
| Date:		sometime in 1993
------------------------------------------------------------*/
static kobject
_kcms_cmobj_generate(
   kobject  toolbox,
   kstring  oname,
   int      otype,
   kstring  bname,
   int      ltype,
   kstring  lname,
   kstring  cat,
   kstring  subcat,
   kstring  description,
   kstring  form,
   kstring  pane,
   int      incantata,
   kstring  tb_name,
   kstring  old_oname,
   kstring  old_bname)
{
   kstring  routine = "Generate Object";
   kstring  opath  = NULL;
   kstring  lopath = NULL;
   kstring  tbname = NULL;
   kobject  cmobj  = NULL;
   kobject  libobj = NULL;
   kobject  fobj   = NULL;
   kstring  short_copyright;
   kstring  long_copyright;
   int      cpy_lang;
   char     tbpath[KLENGTH];
   char     template_path[KLENGTH];
   char     lsrcpath[KLENGTH];
   char     temp[KLENGTH];
   kstring  typestring;
   kstring  suffix;
   int      subtype;
   int      i;


   typestring = kcms_attr_int2string(KCMS_CMOBJ_TYPE, otype);
   if ((cmobj = kcms_create_cmobj(toolbox, oname, otype)) == NULL)
      return NULL;
   if (bname != NULL && !kcms_set_attribute(cmobj, KCMS_CMOBJ_BNAME, bname))
      return _kcms_cmobj_destroy_return(cmobj);
   if (!kcms_set_attribute(cmobj, KCMS_CMOBJ_LANGTYPE, ltype))
      return _kcms_cmobj_destroy_return(cmobj);

   if (!kcms_get_attribute(cmobj, KCMS_PATH, &opath))
      return _kcms_cmobj_destroy_return(cmobj);
   if (!kcms_get_attribute(toolbox, KCMS_NAME, &tbname))
      return _kcms_cmobj_destroy_return(cmobj);
   (void)ksprintf(tbpath, "$%s", tbname);

   /*-- not fatal if we can't set the short description ---------------*/
   (void)kcms_set_attribute(cmobj, KCMS_CMOBJ_SHORT_DESC, description);

   cpy_lang = (ltype == KCMS_LANG_NONE ? KCMS_LANG_C : ltype);
   short_copyright = kcms_get_copyright(toolbox, KCMS_TB_COPYRIGHT_SHORT,
					cpy_lang);
   long_copyright = kcms_get_copyright(toolbox, KCMS_TB_COPYRIGHT_LONG,
					cpy_lang);

   /*-- if an associated library was specified, does it exist? --*/
   if (lname != NULL)
   {
      if ((libobj = kcms_open_cmobj(toolbox, lname)) == NULL)
      {
	 if (!kprompt(KSTANDARD, "Create", "Abort", TRUE,
		      "\nOperation: generate software object\n\n"
		      "\tToolbox: %s\n"
		      "\tObject:  %s\n\n"
		      "You specified that the kroutine should have "
		      "an associated library routine, in library %s.\n"
		      "This library does not exist in toolbox `%s'.\n"
		      "Do you want to create the library in the toolbox?",
		      tbname, oname, lname, tbname))
	    return _kcms_cmobj_destroy_return(cmobj);

	 libobj = kcms_generate_cmobj(toolbox, lname,
				      KCMS_LIBRARY, lname, NULL, NULL, NULL,
				      NULL, KCMS_NOLANG, FALSE, FALSE, FALSE,
				      NULL);
	 if (libobj == NULL)
	    return _kcms_cmobj_destroy_return(cmobj);
      }

      if (kcms_get_attribute(libobj, KCMS_PATH, &lopath))
      {
	 /* put a template l*.c in the library */
	 ksprintf(lsrcpath, "%s/src/l%s.c", lopath, oname);
	 if (!ksedfile(KCMS_TEMPLATE_SRC, lsrcpath, FALSE,
		       KFILE_UPDATE, NULL,
		       "#object-name#",       oname,
		       "#library-name#",      lname,
		       "#short-description#", description,
		       "#short-copyright#",   short_copyright,
		       "#long-copyright#",    long_copyright,
		       NULL))
	    kerror(KCMS, routine, "Unable to create template l*.c");
	 else
	 {
	    fobj = kcms_create_fileobj(libobj, lsrcpath,
				       NULL,
				       KCMS_FOBJ_TYPE_SRC,
				       KCMS_FOBJ_SUBTYPE_C,
				       KCMS_FOBJ_GEN_NONE,
				       KCMS_FOBJ_ACCESS_RDWR);
	    kcms_set_attribute(libobj, KCMS_CMOBJ_UPDATE_DB,
			       (KCMS_UPDATE_SYNC | KCMS_UPDATE_REGEN));
	    if (fobj == NULL)
	       kerror(KCMS, routine,
		      "Unable to create %s.",
		      "file object for l*.c");
	 }
      }
      kcms_sync(toolbox);
      if (libobj != NULL)
	 kcms_close(libobj);
   }

   /*-- if new object is a library, give it template includes ---------*/
   if (otype == KCMS_LIBRARY)
   {
      ksprintf(temp, "%s/src/internals.h", opath);
      _kcms_copy_template(cmobj,
		    "$BOOTSTRAP/objects/library/kcms/templates/src/internals.h",
			  temp, cpy_lang);

      ksprintf(temp, "$%s/include/%s/%s.h", tbname, oname, oname);
      _kcms_copy_template(cmobj,
		      "$BOOTSTRAP/objects/library/kcms/templates/src/library.h",
			  temp, cpy_lang);
   }
   else if (otype == KCMS_SCRIPT)
   {
      switch (ltype)
      {
	 case KCMS_CSH:
	    suffix = "csh";
	    subtype = KCMS_FOBJ_SUBTYPE_CSH;
	    break;
	 case KCMS_SH:
	    suffix = "sh";
	    subtype = KCMS_FOBJ_SUBTYPE_SH;
	    break;
	 case KCMS_KSH:
	    suffix = "ksh";
	    subtype = KCMS_FOBJ_SUBTYPE_SH;
	    break;
	 case KCMS_PERL:
	    suffix = "pl";
	    subtype = KCMS_FOBJ_SUBTYPE_PERL;
	    break;
	 default:
	    subtype = KCMS_FOBJ_SUBTYPE_SH;
	    suffix = "sh";
      }
      ksprintf(template_path, "$BOOTSTRAP/objects/library/kcms/"
	       "templates/src/template.%s", suffix);
      ksprintf(temp, "%s/src/%s.%s", opath, oname, suffix);
      _kcms_copy_template(cmobj, template_path, temp, cpy_lang);
      if (kcms_create_fileobj(cmobj, temp, NULL,
			      KCMS_FOBJ_TYPE_SCRIPT, subtype,
			      KCMS_FOBJ_GEN_NONE,
			      KCMS_FOBJ_ACCESS_RDWR) == NULL)
      {
	 return _kcms_cmobj_destroy_return(cmobj);
      }
   }

   if (form)
   {
      ksprintf(temp, "%s/uis/%s.form", opath, oname);
      if (!ksedfile(form, temp, FALSE, KFILE_UPDATE, NULL,
		    tb_name, tbpath,
		    old_oname, oname,
		    old_bname, bname,
		    "#category#", cat,
		    "#subcategory#", subcat,
		    "#short-description#", description,
		    "#object-type#", typestring, NULL))
	 return _kcms_cmobj_destroy_return(cmobj);
      if (kcms_create_fileobj(cmobj, temp, NULL,
			      KCMS_FOBJ_TYPE_UIS, KCMS_FOBJ_SUBTYPE_FORM,
			 KCMS_FOBJ_GEN_NONE, KCMS_FOBJ_ACCESS_RDWR) == NULL)
	 return _kcms_cmobj_destroy_return(cmobj);
   }
   if (pane)
   {
      ksprintf(temp, "%s/uis/%s.pane", opath, oname);
      if (!ksedfile(pane, temp, FALSE, KFILE_UPDATE, NULL,
		    tb_name, tbpath,
		    old_oname, oname,
		    old_bname, bname,
		    "#category#", cat,
		    "#subcategory#", subcat,
		    "#short-description#", description,
		    "#object-type#", typestring, NULL))
	 return _kcms_cmobj_destroy_return(cmobj);
      if (kcms_create_fileobj(cmobj, temp, NULL,
			      KCMS_FOBJ_TYPE_UIS, KCMS_FOBJ_SUBTYPE_PANE,
			 KCMS_FOBJ_GEN_NONE, KCMS_FOBJ_ACCESS_RDWR) == NULL)
	 return _kcms_cmobj_destroy_return(cmobj);
   }

   if (otype == KCMS_PANE)
   {
      kstrcpy(template_path, "$BOOTSTRAP/objects/library/kcms/"
	      "templates/info/pane.todo");
      if (kaccess(template_path, R_OK) == -1)
	 kwarn(KCMS, routine, "Could not read template "
	       "for the new object's todo file:\n\n"
	       "\t%s\n\n"
	       "The object will not have a "
	       "todo file.\n", template_path);
      else
      {
	 ksprintf(temp, "%s/info/%s.todo", opath, oname);
	 if (!ksedfile(template_path, temp, FALSE,
		       KFILE_UPDATE, NULL,
		       "#toolbox-name#", tbname,
		       "#object-name#", oname,
		       "#short-description#", description,
		       "#object-type#", typestring, NULL))
	    kwarn(KCMS, routine, "Could not copy template "
		  "todo file into the new "
		  "object -- no todo file was "
		  "created.\n");
	 else
	    /*UPDATE: does this give error messages? */
	    (void)kcms_create_fileobj(cmobj, temp, NULL,
				      KCMS_FOBJ_TYPE_INFO,
				      KCMS_FOBJ_SUBTYPE_TODO,
				      KCMS_FOBJ_GEN_NONE,
				      KCMS_FOBJ_ACCESS_RDWR);
      }
   }
   else
   {
      for (i = 0; info_details[i].attribute != -1; i++)
      {
	 ksprintf(template_path, "$BOOTSTRAP/objects/library/kcms/"
		  "templates/info/%s",
		  info_details[i].template);
	 if (kaccess(template_path, R_OK) == -1)
	    kwarn(KCMS, routine, "Could not read template "
		  "for the new object's %s file:\n\n"
		  "\t%s\n\n"
		  "The object will not have a "
		  "%s file.\n",
		  info_details[i].suffix,
		  template_path,
		  info_details[i].suffix);
	 else
	 {
	    ksprintf(temp, "%s/info/%s.%s", opath, oname,
		     info_details[i].suffix);
	    if (!ksedfile(template_path, temp, FALSE, KFILE_UPDATE, NULL,
			  "#toolbox-name#", tbname,
			  "#object-name#", oname,
			  "#short-description#", description,
			  "#object-type#", typestring, NULL))
	       kwarn(KCMS, routine,
		     "Could not copy template "
		     "%s file into the new "
		     "object -- no %s file was "
		     "created.\n",
		     info_details[i].suffix,
		     info_details[i].suffix);
	    else
	       kcms_create_fileobj(cmobj, temp, NULL,
				   KCMS_FOBJ_TYPE_INFO,
				   info_details[i].subtype,
				   KCMS_FOBJ_GEN_NONE,
				   KCMS_FOBJ_ACCESS_RDWR);
	 }
      }
   }

   if (lname && !kcms_set_attribute(cmobj, KCMS_CMOBJ_GEN_LIBNAME, lname))
      return _kcms_cmobj_destroy_return(cmobj);
   if (kstrlen(cat) && !kcms_set_attribute(cmobj, KCMS_CMOBJ_CATEGORY, cat))
      kerror(KCMS, routine, "Warning: unable to set object Category.");
   if (kstrlen(subcat) > 0 && !kcms_set_attribute(cmobj,
					    KCMS_CMOBJ_SUBCATEGORY, subcat))
      kerror(KCMS, routine,
	     "Warning: unable to set object Subcategory.");
   kcms_set_attribute(cmobj, KCMS_CMOBJ_UPDATE_DB, (KCMS_UPDATE_SYNC
						    | KCMS_UPDATE_CACHE
						    | KCMS_UPDATE_REGEN
						    | KCMS_UPDATE_NEW));

   kcms_set_attribute(cmobj, KCMS_CMOBJ_INCANTATA, incantata);

   kcms_sync(toolbox);
   return cmobj;
}

/************************************************************
* Routine Name:	kcms_generate_cmobj - create and initialize a software object
*
* Purpose:	This routine is a high-level interface to
*		\f(CW\s-2kcms_create_cmobj()\s+2\fP
*		that will create an empty software, and then set some
*		attributes about the cmobj.  These attributes include
*		bname, l*.c file, adding a template pane or form,
*		language dependancies, category, and sub-category information.
*		The description variable contains the short description of
*		the new object.  Note the first three parameters MUST be
*		valid input.  If they are invalid values, this routine will
*		return an error.  The parameters starting with [bname] are
*		optional, and an value of FALSE or NULL depending on the type
*		implies that the option is not set and a default value is
*		used.  Also, if the template_object is not NULL, default
*		values for the optional parameters will be taken from the
*		values that are set in the template.
*
* Input:	toolbox         - The object reference for the parent toolbox.
*		oname           - The name for the new software object.
*		otype           - The type of software object to generate.
*				  This can be one of the following values:
*				  !\f(CW\s-1
*				  !   KCMS_KROUTINE
*				  !   KCMS_XVROUTINE
*				  !   KCMS_LIBRARY
*				  !   KCMS_PANE
*				  !   KCMS_SCRIPT
*				  !\s+1\fP
*		bname           - The binary name of new object.
*				  If this argument is NULL,
*				  then the binary name will be the same as
*				  the object name.
*		cat             - A category string for new object.
*		subcat          - A subcategory string for new object.
*		description     - A short description of the new object.
*		lname           - The name of a library to associate with
*				  this object.  If a library is specified,
*				  then a file l<object-name>.c will be created
*				  in the specified library object,
*				  with a template function, l<object-name>().
*		ltype           - The programming language used for
*				  this object.
*		form            - A boolean flag which specifies whether
*				  a form UIS file should be created for the
*				  new object.  This file defines the GUI for
*				  xvroutines.
*		pane            - A boolean flag which specifies whether
*				  a pane UIS file should be created for the
*				  new object.  This file defines the command
*				  line user interface for the object.
*		ci              - A boolean flag which specifies whether
*				  the object should be installed (visible)
*				  in cantata.
*		template_object - An existing software object to take
*				  default values from.
*
* Returns:	TRUE (1) on success, FALSE (0) otherwise
*
* Written By:	Steven Jorgensen and Neil Bowers
* Date:		
*************************************************************/
kobject
kcms_generate_cmobj(
   kobject  toolbox,
   kstring  oname,
   int      otype,
   kstring  bname,
   kstring  cat,
   kstring  subcat,
   kstring  description,
   kstring  lname,
   int      ltype,
   int      form,
   int      pane,
   int      ci,
   kobject  template_object)
{
   kstring  routine   = "Generate Object";
   kobject  tmp_obj   = NULL;
   kobject  fobj;
   int      gen_otype = otype;
   int      gen_ltype = KCMS_NOLANG;
   int      gen_pane  = FALSE;
   int      gen_form  = FALSE;
   int      gen_ci    = FALSE;
   char     pane_path[KLENGTH];
   char     form_path[KLENGTH];
   char     tb_name[KLENGTH];
   char     old_oname[KLENGTH];
   char     old_bname[KLENGTH];
   kstring  tbname     = NULL;
   kstring  gen_cat    = NULL;
   kstring  gen_subcat = NULL;
   kstring  gen_lname  = NULL;
   kstring  gen_oname  = oname;
   kstring  gen_bname  = (bname ? bname : oname);
   kstring  tmp;


   if (!kcms_legal_identifier(gen_oname, KOBJ_CMSOBJ))
   {
      kerror(KCMS, routine, "Invalid object name -- %s%s",
	     "object names must start with a letter, followed ",
	     "by letters, digits or _");
      return NULL;
   }

   if (gen_otype == KCMS_MANUAL || gen_otype == KCMS_MINDEX ||
       gen_otype == KCMS_MGLOSSARY)
   {
      kerror(NULL, "kcms_generate_cmobj()",
	     "This object type is unsupported");
      return NULL;
   }

   if (!kcms_get_attribute(toolbox, KCMS_NAME, &tbname))
      return NULL;

   /*-- category and subcategory specified for the object? ------------*/
   if (cat != NULL)
      gen_cat = kstring_cleanup(cat, NULL);
   if (subcat != NULL)
      gen_subcat = kstring_cleanup(subcat, NULL);

   kstrcpy(tb_name, "$TOOLBOX");
   kstrcpy(old_oname, "#object-name#");
   kstrcpy(old_bname, "#binary-name#");
   kstrcpy(pane_path, "$BOOTSTRAP/objects/library/kcms/"
	   "templates/uis/template.pane");
   kstrcpy(form_path,
	   "$BOOTSTRAP/objects/library/kcms/templates/uis/template.form");

   /*-- if template object passed, suck appropriate values from it --*/
   if (template_object)
   {
      if (!kcms_get_attribute(template_object, KCMS_PARENT, &tmp_obj))
	 return NULL;
      tmp = NULL;
      if (!kcms_get_attribute(tmp_obj, KCMS_NAME, &tmp))
	 return NULL;
      ksprintf(tb_name, "$%s", tmp);
      if (!kcms_get_attribute(template_object, KCMS_CMOBJ_LANGTYPE,
			      &gen_ltype))
	 return NULL;
      tmp = NULL;
      if (!kcms_get_attribute(template_object, KCMS_NAME, &tmp))
	 return NULL;
      kstrcpy(old_oname, tmp);
      tmp = NULL;
      if (!kcms_get_attribute(template_object, KCMS_CMOBJ_BNAME, &tmp))
	 return NULL;
      kstrcpy(old_bname, tmp);
      tmp = NULL;
      if (!kcms_get_attribute(template_object, KCMS_CMOBJ_UIS_PANE, &fobj)
	  || !kcms_get_attribute(fobj, KCMS_PATH, &tmp))
	 return NULL;
      if (tmp != NULL)
      {
	 gen_pane = TRUE;
	 kstrcpy(pane_path, tmp);
      }
      tmp = NULL;
      if (!kcms_get_attribute(template_object, KCMS_CMOBJ_UIS_FORM, &fobj)
	  || !kcms_get_attribute(fobj, KCMS_PATH, &tmp))
	 return NULL;
      if (tmp != NULL)
      {
	 gen_form = TRUE;
	 kstrcpy(form_path, tmp);
      }

      /*-- category and subcategory -----------------------------------*/
      if (gen_cat == NULL && kcms_get_attribute(template_object,
						KCMS_CMOBJ_CATEGORY, &tmp))
      {
	 gen_cat = kstring_cleanup(tmp, NULL);
      }
      if (gen_subcat == NULL && kcms_get_attribute(template_object,
					      KCMS_CMOBJ_SUBCATEGORY, &tmp))
      {
	 gen_subcat = kstring_cleanup(tmp, NULL);
      }

      /*-- does template object have an associated library? -----------*/
      if (!kcms_get_attribute(template_object, KCMS_CMOBJ_GEN_LIBNAME,
			      &gen_lname))
	 return NULL;
      if (!kcms_get_attribute(template_object, KCMS_CMOBJ_INCANTATA, &gen_ci))
	 return NULL;
   }
   gen_form = (gen_form ? gen_form : form);
   gen_pane = (gen_pane ? gen_pane : pane);

   if (ltype != KCMS_LANG_NONE)
      gen_ltype = ltype;
   if (ci)
      gen_ci = ci;
   if (lname != NULL)
      gen_lname = lname;

   /*-- if the new object has in-cantata==TRUE, must have category ----*/
   /*-- and subcategory specified -------------------------------------*/
   if (gen_ci && (gen_cat == NULL || gen_subcat == NULL))
   {
      kerror(KCMS, routine,
	     "Error generating object \"%s\" in toolbox \"%s\"\n"
	     "You must specify both a Category and Subcategory "
	     "for objects which are installed in cantata.",
	     oname, tbname);
      return NULL;
   }

   switch (gen_otype)
   {
	 /* FALLTHROUGH */
      case KCMS_XVROUTINE:
      case KCMS_KROUTINE:
	 gen_pane = TRUE;
	 break;

	 /* FALLTHROUGH */
      case KCMS_PANE:
	 gen_pane = TRUE;
	 if (gen_form)
	 {
	    kwarn(KCMS, routine, "A pane object can not have a .form file.\n"
		  "No template .form will be created.");
	    gen_form = FALSE;
	 }

      case KCMS_EXAMPLE:
      case KCMS_XEXAMPLE:
      case KCMS_TESTSUITE:
      case KCMS_LIBRARY:
	 break;

      case KCMS_SCRIPT:
	 if (gen_ltype == KCMS_LANG_NONE)
	 {
	    kwarn(KCMS, routine,
		  "A script object must have a language type of [csh], [sh], "
		  "[ksh], or [perl].  Assuming [sh].");
	    gen_ltype = KCMS_SH;
	 }
	 break;
   }

   return _kcms_cmobj_generate(toolbox, gen_oname, gen_otype, gen_bname,
			       gen_ltype, gen_lname,
			       gen_cat, gen_subcat,
			       description,
			       (gen_form ? form_path : NULL),
			       (gen_pane ? pane_path : NULL),
			       gen_ci,
			       tb_name, old_oname, old_bname);
}

/*-----------------------------------------------------------
| Routine Name:	_kcms_copy_template - copy template file into object
|
| Purpose:	This function copies a template file into the specified
|		object, performing substitutions for standard KCMS
|		meta-strings, object name, object type, toolbox name etc
|
| Input:	object	- the software object we're copying into.
|		from	- path of template file.
|		to	- path to copy the template file to.
|
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Neil Bowers
| Date:		1-dec-93
------------------------------------------------------------*/
static int
_kcms_copy_template(
   kobject  object,
   kstring  from,
   kstring  to,
   int      language)
{
   kstring  routine = "_kcms_copy_template()";
   kobject  toolbox;
   kstring  oname;
   kstring  tbname;
   kstring  opath;
   kstring  typestring;
   kstring  short_description;
   kstring  short_copyright;
   kstring  long_copyright;
   char     tblower[KLENGTH];
   char     tbcap[KLENGTH];
   int      otype;


   if (!kcms_get_attributes(object,
			    KCMS_NAME,             &oname,
			    KCMS_PATH,             &opath,
			    KCMS_CMOBJ_PROGTYPE,   &otype,
			    KCMS_PARENT,           &toolbox,
			    KCMS_CMOBJ_SHORT_DESC, &short_description,
			    KCMS_END)
       || !kcms_get_attribute(toolbox, KCMS_NAME, &tbname))
   {
      return FALSE;
   }

   if ((typestring = kcms_attr_int2string(KCMS_CMOBJ_TYPE, otype)) == NULL)
      return FALSE;

   if (kstring_lower(tbname, tblower) == NULL
       || kstring_capitalize(tbname, tbcap) == NULL)
   {
      kerror(KCMS, routine, "Could not generate case variants of "
	     "toolbox name for toolbox %s.", tbname);
      return FALSE;
   }

   short_copyright = kcms_get_copyright(toolbox, KCMS_TB_COPYRIGHT_SHORT,
					language);
   long_copyright = kcms_get_copyright(toolbox, KCMS_TB_COPYRIGHT_LONG,
					language);

   return ksedfile(from, to, FALSE, KFILE_UPDATE, NULL,
		   "#object-name#",       oname,
		   "#object-type#",       typestring,
		   "#object-path#",       opath,
		   "#TOOLBOX-NAME#",      tbname,
		   "#Toolbox-Name#",      tbcap,
		   "#toolbox-name#",      tblower,
		   "#short-description#", short_description,
		   "#short-copyright#",   short_copyright,
		   "#long-copyright#",    long_copyright,
		   NULL);
}
