/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Khoros Program library object routines
   >>>>
   >>>>   Static:
   >>>>             _kcms_cmobj_del_subobj_wrk()
   >>>>             _kcms_cmobj_add_or_mod()
   >>>>             _kcms_cmobj_add_subobj()
   >>>>             _kcms_cmobj_del_subobj()
   >>>>             _kcms_cmobj_mod_subobj()
   >>>>             _kcms_cmobj_filenames_to_fileobjs()
   >>>>             _kcms_cmobj_open_err()
   >>>>             _kcms_cmobj_filename_to_fileobj()
   >>>>             _kcms_cmobj_get_doc()
   >>>>  Private:
   >>>>             kcms_cmobj_initialize()
   >>>>             kcms_cmobj_open()
   >>>>             kcms_cmobj_complete_open()
   >>>>             kcms_cmobj_close()
   >>>>             kcms_cmobj_duplicate()
   >>>>             kcms_cmobj_destroy()
   >>>>
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include "cmobjP.h"
#include "databaseP.h"

static char *illegal_object_names[] =
{
   "toolbox",
   "object",
   "usage"
};

static int _kcms_cmobj_add_subobj     PROTO((kobject, kobject));
static int _kcms_cmobj_del_subobj     PROTO((kobject, kobject));
static int _kcms_cmobj_mod_subobj     PROTO((kobject, kobject));
static int _kcms_cmobj_del_subobj_wrk PROTO((kobject, kobject, kobject *));
static klist *_kcms_cmobj_filenames_to_fileobjs PROTO((kobject, char *,
					               char **, int, int,
                                                       int, int, int));
static kobject _kcms_cmobj_open_err   PROTO((kobject, int, char *, kvalist));
static kobject _kcms_cmobj_filename_to_fileobj PROTO((kobject, kdbm *, char *,
					              klist **, int, int,
                                                      int, int));
static int _kcms_cmobj_get_doc     PROTO((kobject));
static int _kcms_duplicate_library PROTO((kobject, kobject, int));
static int _kcms_update_man_db     PROTO((kdbm *, kobject, int));

extern kerrlist kcms_errors[];
extern int kcms_numerrors;

/************************************************************
* Routine Name:	kcms_create_cmobj - create a new program or library object
*
* Purpose:	This routine creates the necessary files (on disk) for
*		a template software object.
*		An object reference (of type \f(CW\s-1kobject\s+1\fP)
*		is returned for the new object,
*		which can then be configured by setting object attributes.
*
* Input:	toolbox - The toolbox object which the new object
*		          should be created in.
*		oname   - The name of the new program object.
*		otype   - The type of object to create.
*			  Kcms currently supports five types of software
*			  object:
*			  !\f(CW\s-1
*			  !   KCMS_KROUTINE
*			  !   KCMS_XVROUTINE
*			  !   KCMS_LIBRARY
*			  !   KCMS_PANE
*			  !   KCMS_SCRIPT
*			  !\s+1\fP
*
* Returns:	An object reference for the newly created software object,
*		or NULL if the object could not be created.
*		In the latter case,
*		\f(CW\s-1errno\s+1\fP is set to signify the error.
* Written By:	Steven Jorgensen and Neil Bowers
* Date:		2-apr-93
*************************************************************/
kobject
kcms_create_cmobj(
   kobject  toolbox,
   kstring  oname,
   int      otype)
{
   kstring  routine = "kcms_create_cmobj()";
   kobject  object;
   kobject  cmobj;
   char     opath[KLENGTH];
   char     temp[KLENGTH];
   char     path_bits[3][KLENGTH];
   kstring  tbname;
   kstring  typestring;
   kstring  datestamp;
   int      i;


   if (!kcms_legal_object(toolbox, routine))
      return NULL;

   if (oname == NULL)
   {
      kerror(KCMS, routine, "Program name is NULL");
      errno = KCMS_NULLNAME;
      return NULL;
   }

   if (!kcms_legal_identifier(oname, KOBJ_CMSOBJ))
   {
      kerror(KCMS, routine,
	     "\nInvalid object name --\n"
	     "Object names must start with a letter, followed "
	     "by letters, digits or _");
      return NULL;
   }
   for (i = 0; i < sizeof(illegal_object_names) / sizeof(char *); i++)
      if (!kstrcasecmp(illegal_object_names[i], oname))
      {
	 kerror(KCMS, routine, "\nInvalid object name --\n"
		"You can not have an object called \"%s\".\n", oname);
	 return NULL;
      }

   typestring = kcms_attr_int2string(KCMS_CMOBJ_TYPE, otype);
   if (!kcms_get_attribute(toolbox, KCMS_NAME, &tbname))
      return NULL;

   /*-- can we write the object directory structure? ------------------*/
   kstrcpy(path_bits[0], "objects");
   kstrcpy(path_bits[1], typestring);
   kstrcpy(path_bits[2], oname);
   ksprintf(opath, "$%s", tbname);
   for (i = 0; i < 3; i++)
   {
      if (kstring_3cat(opath, "/", path_bits[i], opath) == NULL)
      {
	 kerror(KCMS, routine, "Unable to generate object path (%s)\n"
		"The new software object will not be created.\n",
		opath);
	 return NULL;
      }

      /*-- can't create directory -------------------------------------*/
      if (kmkdir(opath, 0755) == -1 && errno != EEXIST)
      {
	 kerror(KCMS, routine,
		"Unable to create directory\n\n"
		"\tPath: %s\n\n"
		"Object will not be created\n",
		opath);
	 return NULL;
      }
   }

   /*-- is there already an object of the same name in the toolbox? ---*/
   if ((object = kcms_open_cmobj(toolbox, oname)) != NULL)
   {
      kerror(KCMS, routine, "An object %s already exists in toolbox %s\n",
	     oname, tbname);
      errno = KCMS_OBJALREADYEXIST;
      kcms_close(object);
      return NULL;
   }

   if ((cmobj = kcms_cmobj_initialize(toolbox, oname, typestring)) == NULL)
      return NULL;

   ksprintf(temp, "%s/db/cms", opath);
   cmobj->database = kcms_open_fileobj(cmobj, temp, NULL,
				       KCMS_FOBJ_TYPE_DBM,
				       KCMS_FOBJ_SUBTYPE_CM,
				       KCMS_FOBJ_GEN_NONE,
				       KCMS_FOBJ_ACCESS_RDWR);

   cmobj->noaccess = FALSE;

   if (!kmake_dir(cmobj->topsrc, 0777) && errno != EEXIST)
      return _kcms_cmobj_open_err(cmobj, KCMS_PERMISSION,
		      "kcms_cmobj_create: cannot create top level src dir");

   /*UPDATE: Are these needed?  already done in initialize? -- */

   cmobj->fullyopen   = TRUE;
   cmobj->lang_type   = KCMS_NOLANG;
   cmobj->levels      = 3;
   cmobj->code_type   = KCMS_ANSI;
   cmobj->version     = kstrdup(KCMS_DEFAULT_SW_VERSION);

   if (!_kcms_cmobj_get_doc(cmobj))
      return _kcms_cmobj_open_err(cmobj, errno,
		     "kcms_cmobj_open: Error getting the doc file objects");

   if (cmobj->parent->add_sub_obj(toolbox, cmobj) == FALSE)
      return _kcms_cmobj_open_err(cmobj, KCMS_TOOLBOXUPDATEFAIL, NULL);

   /*-- set creation, modification and generation dates ---------------*/
   datestamp = kcms_get_date();
   cmobj->times.creation     = kstrdup(datestamp);
   cmobj->times.modification = kstrdup(datestamp);
   cmobj->times.generation   = kstrdup(datestamp);
   kfree(datestamp);

   kcms_set_attribute(cmobj, KCMS_CMOBJ_UPDATE_DB,
		      (KCMS_UPDATE_SYNC
		       | KCMS_UPDATE_REGEN
		       | KCMS_UPDATE_NEW));
   return cmobj;
}

/************************************************************
* Routine Name:	kcms_open_cmobj - open an existing software object
*
* Purpose:	This routine reads and parses the configuration
*		database used to keep track of the files and information
*		for a software object.  This information is stored
*		in a `kobject' object, which is returned to the caller.
*		This object is used to reference the software object
*		in all configuration management operations.
*
* Input:	toolbox	- A valid toolbox object.
*		oname	- A string containing the name of the software
*			  object to open.
*
* Returns:	A kobject initialized for the specified software object,
*		or NULL on error.
*		In the latter case,
*		\f(CW\s-1errno\s+1\fP is set to signify the error.
*
* Written By:	Steven Jorgensen and Neil Bowers
* Date:		2-apr-93
*************************************************************/
kobject
kcms_open_cmobj(
   kobject  toolbox,
   kstring  oname)
{
   kstring   routine      = "kcms_open_cmobj()";
   kobject   object;
   klist    *object_list  = NULL;
   klist    *list;
   kstring   opath;
   char      dbpath[KLENGTH];
   int       token;


   if (oname == NULL)
   {
      kinfo(KSYSLIB, "%s: %s\n", routine, "name parameter is NULL");
      errno = KCMS_NULLNAME;
      return NULL;
   }
   if (!kcms_legal_object(toolbox, routine))
      return NULL;

   if (!kcms_get_attribute(toolbox, KCMS_TB_SOFTWARE_OBJECTS, &object_list))
      return NULL;

   token = kstring_to_token(oname);

   if ((list = klist_locate(object_list, (kaddr)token)) == NULL)
      return NULL;

   object = (kobject)klist_clientdata(list);

   if (object == NULL)
   {
      errno = KCMS_EINTERNAL;
      kerror(KCMS, routine, "NULL object retrieved from toolbox.\n"
	     "Please report this as a bug.");
      return NULL;
   }

   if (!kcms_get_attribute(object, KCMS_PATH, &opath))
      return NULL;

   ksprintf(dbpath, "%s/db/cms", opath);
   if (kaccess(dbpath, R_OK) == -1)
   {
      errno = KCMS_ENODB;
      return NULL;
   }

   return object;
}

/*-----------------------------------------------------------
| Routine Name:	kcms_cmobj_initialize - initialize the software object
|
| Purpose:	This routine will initialize the CMS program
|		and library object.
|
| Input:	oname - string containing the name of the program to
|		initialize
|		opath - string containing the top level src path
|		toolbox - a kobject pointing to the toolbox structure this
|		program is a part of
|
| Returns:	new software object on success, NULL otherwise
|
| Written By:	Tom Sauer & Steven Jorgensen
| Date:		Oct 23, 1992 14:47
------------------------------------------------------------*/
kobject
kcms_cmobj_initialize(
   kobject  toolbox,
   kstring  oname,
   kstring  typestring)
{
   kstring     routine         = "Initialize Object";
   kobject     cmobj           = NULL;
   kstring     tbname;
   char        opath[KLENGTH];
   static int  errs_init       = 0;
   int         otype;


   if (!errs_init)
   {
      kcms_init_errors();
      errs_init = 1;
   }

   if (oname == NULL)
   {
      kinfo(KSYSLIB,"%s: function called with 	NULL object name\n", routine);
      errno = KCMS_NULLNAME;
   }
   else if ((otype = kcms_attr_string2int(KCMS_CMOBJ_PROGTYPE, typestring))
	    < KCMS_KROUTINE)
   {
      kerror(KCMS,routine,"Function called with invalid object type (%s).",
	     typestring);
   }
   else if (kcms_legal_object(toolbox, routine))
   {

      /* Malloc the cmobj structure. */
      if ((cmobj = (kobject) kcalloc(1, sizeof(struct _kobject))) == NULL)
      {
	 errno = KCMS_CANNOTMALLOCOBJ;
	 kerror(KCMS, routine, "Cannot malloc software object structure.");
      }
      else
      {
	 kcms_get_attribute(toolbox, KCMS_NAME, &tbname);
	 ksprintf(opath, "$%s/objects/%s/%s", tbname, typestring, oname);

	 /* Initalize the Objec	t */
	 cmobj->oname_tkn     = kstring_to_token(oname);
	 cmobj->oname         = ktoken_to_string(cmobj->oname_tkn);
	 cmobj->bname         = NULL;
	 cmobj->opath         = kstrdup(opath);
	 cmobj->type          = KOBJ_CMSOBJ;
	 cmobj->parent        = toolbox;
	 cmobj->flags         = (unsigned long)0;
	 cmobj->ci_installed  = FALSE;
	 cmobj->levels        = 3;
	 cmobj->attributes    = (AttrTbl *) kcms_cmobj_attr_tbl();
	 cmobj->attribute_cnt = KCMS_CMOBJ_MAX;

	 cmobj->add_sub_obj   = _kcms_cmobj_add_subobj;
	 cmobj->del_sub_obj   = _kcms_cmobj_del_subobj;
	 cmobj->mod_sub_obj   = _kcms_cmobj_mod_subobj;
	 cmobj->makefile      = NULL;
	 cmobj->imakefile     = NULL;
	 cmobj->app_defaults  = NULL;
	 cmobj->workspace     = NULL;

	 /*-- initialize object information strings --*/
	 cmobj->category      = NULL;
	 cmobj->subcategory   = NULL;
	 cmobj->short_description = NULL;
	 cmobj->poc_name      = NULL;
	 cmobj->poc_email     = NULL;
	 cmobj->keywords      = NULL;
	 cmobj->filename      = NULL;
	 cmobj->icon_name     = NULL;
	 cmobj->library       = NULL;

	 /*-- file objects, initialized to NULL --*/
	 cmobj->pane          = NULL;
	 cmobj->form          = NULL;
	 cmobj->database      = NULL;

	 cmobj->allconfig     = NULL;
	 cmobj->allfiles      = NULL;

	 cmobj->todo          = NULL;
	 cmobj->bugs          = NULL;
	 cmobj->done          = NULL;
	 cmobj->changelog     = NULL;
	 cmobj->allinfo       = NULL;
	 cmobj->generate_executable = FALSE;
	 cmobj->miscuis       = NULL;	/* misc. UIS files */

	 cmobj->help_files    = NULL;

	 cmobj->prog_type     = otype;
	 cmobj->lang_type     = KCMS_NOLANG;
	 cmobj->topsrc        = kstring_cat(cmobj->opath, "/src", NULL);
	 cmobj->update        = KCMS_UPDATE_NONE;
	 cmobj->fullyopen     = FALSE;
	 cmobj->header_list   = NULL;

	 cmobj->callbacks     = NULL;

	 cmobj->assoc_tbname  = NULL;
	 cmobj->assoc_oname   = NULL;
	 cmobj->assoc_args    = NULL;
      }
   }

   return cmobj;
}

/*-----------------------------------------------------------
| Routine Name:	_kcms_cmobj_del_subobj_wrk - delete a file object from the
|		cmobj
|
| Purpose:	This routine deletes the file object passed in from the
|		appropriate list in the cm object structure, and updates the
|		.cm file and Imakefile as appropriate.  It is a worker
|		routine for _kcms_cmobj_add_subobj, _kcms_cmobj_del_subobj,
|		and _kcms_cmobj_mod_subobj
|
| Input:	cmobj      - the cmobj kobject to delete a subobj from
|		fileobj    - the file kobject to delete from the cmobj
| Output:	oldfileobj - the fileobj that used to be in the list.
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Steven Jorgensen
| Date:		Jun 07, 1993 10:25
------------------------------------------------------------*/
static int
_kcms_cmobj_del_subobj_wrk(
   kobject   cmobj,
   kobject   fileobj,
   kobject  *oldfileobj)
{
   kstring   routine = "Delete Sub-Object";
   int       type;
   int       subtype;
   int       generated;
   int       token;
   kobject   fobj;
   klist    *list = NULL;
   klist    *all;


   if (!kcms_legal_object(cmobj, routine)
       || !kcms_legal_object(fileobj, routine))
      return FALSE;

   if (fileobj->type != KOBJ_CMSFILE)
   {
      errno = KCMS_OBJTYPEMISMATCH;
      return FALSE;
   }
   if (!kcms_get_attribute(fileobj, KCMS_FOBJ_TYPE, &type))
      return FALSE;
   if (!kcms_get_attribute(fileobj, KCMS_FOBJ_SUBTYPE, &subtype))
      return FALSE;
   if (!kcms_get_attribute(fileobj, KCMS_FOBJ_GENERATED, &generated))
      return FALSE;
   if (!kcms_get_attribute(fileobj, KCMS_FOBJ_FULLNAME_TKN, &token))
      return FALSE;

   kcms_get_attribute(cmobj, KCMS_CMOBJ_ALL_FILES, &all);
   list = klist_locate(all, (kaddr) token);
   if (list == NULL)
   {
      if (oldfileobj != NULL)
	 *oldfileobj = NULL;
      return TRUE;
   }
   else
      fobj = (kobject) klist_clientdata(list);
   cmobj->misc_files = klist_delete(cmobj->misc_files, (kaddr)token);
   cmobj->y_files = klist_delete(cmobj->y_files, (kaddr)token);
   cmobj->l_files = klist_delete(cmobj->l_files, (kaddr)token);
   cmobj->c_files = klist_delete(cmobj->c_files, (kaddr)token);
   cmobj->f_files = klist_delete(cmobj->f_files, (kaddr)token);
   cmobj->cplus_files = klist_delete(cmobj->cplus_files, (kaddr)token);
   cmobj->inc_files = klist_delete(cmobj->inc_files, (kaddr)token);
   cmobj->script_files = klist_delete(cmobj->script_files, (kaddr)token);
   cmobj->libman_files = klist_delete(cmobj->libman_files, (kaddr)token);
   cmobj->help_files = klist_delete(cmobj->help_files, (kaddr)token);
   cmobj->miscuis = klist_delete(cmobj->miscuis, (kaddr)token);
   cmobj->app_defaults = klist_delete(cmobj->app_defaults, (kaddr)token);

   if (cmobj->pane == fobj)
      cmobj->pane = NULL;
   if (cmobj->form == fobj)
      cmobj->form = NULL;
   if (cmobj->cfile == fobj)
      cmobj->cfile = NULL;
   if (cmobj->hfile == fobj)
      cmobj->hfile = NULL;
   if (cmobj->lfile == fobj)
      cmobj->lfile = NULL;
   if (cmobj->man1file == fobj)
      cmobj->man1file = NULL;
   if (cmobj->man3file == fobj)
      cmobj->man3file = NULL;
   if (cmobj->helpfile == fobj)
      cmobj->helpfile = NULL;
   if (cmobj->scriptfile == fobj)
      cmobj->scriptfile = NULL;

   if (cmobj->database == fobj)
      cmobj->database = NULL;

   if (cmobj->imakefile == fobj)
      cmobj->imakefile = NULL;

   if (cmobj->makefile == fobj)
      cmobj->makefile = NULL;
   if (cmobj->workspace == fobj)
      cmobj->workspace = NULL;

   if (cmobj->todo == fobj)
      cmobj->todo = NULL;
   if (cmobj->bugs == fobj)
      cmobj->bugs = NULL;
   if (cmobj->done == fobj)
      cmobj->done = NULL;
   if (cmobj->changelog == fobj)
      cmobj->changelog = NULL;

   kcms_cmobj_free_comb_lists(cmobj);

   if (oldfileobj != NULL)
      *oldfileobj = fobj;

   if (type == KCMS_FOBJ_TYPE_MAN || type == KCMS_FOBJ_TYPE_HELP ||
       type == KCMS_FOBJ_TYPE_MANUAL ||
       type == KCMS_FOBJ_TYPE_INFO)
      kcms_set_attribute(cmobj, KCMS_CMOBJ_UPDATE_DB, KCMS_UPDATE_SYNC);
   else
      kcms_set_attribute(cmobj, KCMS_CMOBJ_UPDATE_DB,
			 (KCMS_UPDATE_SYNC | KCMS_UPDATE_REGEN));

   return TRUE;
}


/*-----------------------------------------------------------
| Routine Name:	_kcms_cmobj_add_or_mod - add a new object or modify an
|		existing sub-object in a list of objects
|
| Purpose:	This routine takes a file object and its fullpath token
|		and searchs for the entry in the list.  If the entry exists,
|		it will store the file object passed into the routine in the
|		client data field of the list element.  If no entry
|		for the token currently exists in the list, a new entry
|		will be added with the token as the identifier and the
|		file object as the client data.
|
| Input:	list    - a klist structure we want to modify or add to.
|		token   - the list identifier we want search for
|		fileobj - the file object to be stored
| Output:	
| Returns:	the updated list pointer.
|
| Written By:	Steven Jorgensen
| Date:		Aug 26, 1993
------------------------------------------------------------*/
static klist *
_kcms_cmobj_add_or_mod(
   klist    *list,
   int       token,
   kobject   fileobj)
{
   klist    *find;


   find = klist_locate(list, (kaddr) token);
   if (find != NULL && (fileobj != (kobject) klist_clientdata(find)))
   {
      kcms_close((kobject) klist_clientdata(find));
      find->client_data = (kaddr) fileobj;
      return list;
   }
   return klist_add(list, (kaddr) token, (kobject) fileobj);
}

/*-----------------------------------------------------------
| Routine Name:	_kcms_cmobj_add_subobj - add a file object to the cmobj
|
| Purpose:	This routine adds the file object passed in to the
|		appropriate list in the cmobj structure, and updates the cm
|		db file and Imakefile as appropriate.
|
| Input:	cmobj   - the cmobj kobject to add a subobj to
|		fileobj - the file kobject to add to the cmobj
| Output:	
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Steven Jorgensen
| Date:		Jun 07, 1993
------------------------------------------------------------*/
static int
_kcms_cmobj_add_subobj(
   kobject  cmobj,
   kobject  fileobj)
{
   kstring  routine = "Add Sub-Object";
   int      type;
   int      subtype;
   int      generated;
   int      token;


   if (!kcms_legal_object(cmobj, routine))
      return FALSE;

   if (cmobj->noaccess == TRUE)
      return TRUE;

   if (!cmobj->fullyopen)
      cmobj = kcms_cmobj_complete_open(cmobj);

   if (!kcms_legal_object(fileobj, routine))
      return FALSE;
   if (fileobj->type != KOBJ_CMSFILE)
   {
      errno = KCMS_OBJTYPEMISMATCH;
      return FALSE;
   }
   if (!kcms_get_attributes(fileobj,
			    KCMS_FOBJ_TYPE,         &type,
			    KCMS_FOBJ_SUBTYPE,      &subtype,
			    KCMS_FOBJ_GENERATED,    &generated,
			    KCMS_FOBJ_FULLNAME_TKN, &token,
			    KCMS_END))
      return FALSE;

   kcms_cmobj_free_comb_lists(cmobj);

   /* add to the fileobj check for special placement. */
   switch (type)
   {
      case KCMS_FOBJ_TYPE_MISC:
	 cmobj->misc_files = _kcms_cmobj_add_or_mod(cmobj->misc_files, token,
                                                    fileobj);
	 break;
      case KCMS_FOBJ_TYPE_SRC:
	 switch (subtype)
	 {
	    case KCMS_FOBJ_SUBTYPE_C:
	       if (generated != KCMS_FOBJ_GEN_LFILE)
		  cmobj->c_files = _kcms_cmobj_add_or_mod(cmobj->c_files,
                                                          token, fileobj);
	       break;
	    case KCMS_FOBJ_SUBTYPE_INCL:
	       cmobj->inc_files = _kcms_cmobj_add_or_mod(cmobj->inc_files,
                                                         token, fileobj);
	       break;
	    case KCMS_FOBJ_SUBTYPE_FORTRAN:
	       cmobj->f_files = _kcms_cmobj_add_or_mod(cmobj->f_files, token,
                                                       fileobj);
	       break;
	    case KCMS_FOBJ_SUBTYPE_CPLUSPLUS:
	       cmobj->cplus_files = _kcms_cmobj_add_or_mod(cmobj->cplus_files,
                                                           token, fileobj);
	       break;
	    case KCMS_FOBJ_SUBTYPE_YACC:
	       cmobj->y_files = _kcms_cmobj_add_or_mod(cmobj->y_files, token,
                                                       fileobj);
	       break;
	    case KCMS_FOBJ_SUBTYPE_LEX:
	       cmobj->l_files = _kcms_cmobj_add_or_mod(cmobj->l_files, token,
                                                       fileobj);
	       break;
	    default:
	       errno = KCMS_RANGEERR;
	       return FALSE;
	 }
	 break;
      case KCMS_FOBJ_TYPE_MAN:
	 if (subtype == KCMS_FOBJ_SUBTYPE_MAN1
             && generated == KCMS_FOBJ_GEN_MAN1)
	 {
	    if (cmobj->man1file != fileobj && cmobj->man1file != NULL)
	       kcms_close(cmobj->man1file);
	    cmobj->man1file = fileobj;
	 }

	 else if (subtype == KCMS_FOBJ_SUBTYPE_MAN3
                  && generated == KCMS_FOBJ_GEN_MAN3)
	 {
	    if (cmobj->man3file != fileobj && cmobj->man3file != NULL)
	       kcms_close(cmobj->man3file);
	    cmobj->man3file = fileobj;
	 }
	 else if (subtype == KCMS_FOBJ_SUBTYPE_MAN3)
	 {

	    cmobj->libman_files = _kcms_cmobj_add_or_mod(cmobj->libman_files,
                                                         token, fileobj);
	 }
	 else
	 {
	    errno = KCMS_RANGEERR;
	    return FALSE;
	 }
	 break;
      case KCMS_FOBJ_TYPE_HELP:
	 cmobj->help_files = _kcms_cmobj_add_or_mod(cmobj->help_files, token,
                                                    fileobj);
	 break;
      case KCMS_FOBJ_TYPE_UIS:
	 if (subtype == KCMS_FOBJ_SUBTYPE_FORM)
	 {
	    if (cmobj->form != fileobj &&
		cmobj->form != NULL)
	       kcms_close(cmobj->form);
	    cmobj->form = fileobj;
	 }
	 else if (subtype == KCMS_FOBJ_SUBTYPE_PANE)
	 {
	    if (cmobj->pane != fileobj &&
		cmobj->pane != NULL)
	       kcms_close(cmobj->pane);
	    cmobj->pane = fileobj;
	 }
	 else if (subtype == KCMS_FOBJ_SUBTYPE_MISC)
	 {
	    cmobj->miscuis = _kcms_cmobj_add_or_mod(cmobj->miscuis,
						    token, fileobj);
	 }
	 else
	 {
	    errno = KCMS_RANGEERR;
	    return FALSE;
	 }
	 break;
      case KCMS_FOBJ_TYPE_SCRIPT:
	 cmobj->script_files = _kcms_cmobj_add_or_mod(cmobj->script_files,
                                                      token, fileobj);
	 break;
      case KCMS_FOBJ_TYPE_DBM:
	 if (subtype == KCMS_FOBJ_SUBTYPE_CM)
	    cmobj->database = fileobj;
	 else if (subtype == KCMS_FOBJ_SUBTYPE_WORKSPACE)
	    cmobj->workspace = fileobj;
	 else
	 {
	    errno = KCMS_RANGEERR;
	    return FALSE;
	 }
	 break;

      case KCMS_FOBJ_TYPE_INFO:
	 if (subtype == KCMS_FOBJ_SUBTYPE_TODO)
	    cmobj->todo = fileobj;
	 else if (subtype == KCMS_FOBJ_SUBTYPE_BUGS)
	    cmobj->bugs = fileobj;
	 else if (subtype == KCMS_FOBJ_SUBTYPE_DONE)
	    cmobj->done = fileobj;
	 else if (subtype == KCMS_FOBJ_SUBTYPE_CHANGELOG)
	    cmobj->changelog = fileobj;
	 else
	 {
	    errno = KCMS_RANGEERR;
	    return FALSE;
	 }
	 break;

      case KCMS_FOBJ_TYPE_CONFIG:
	 if (subtype == KCMS_FOBJ_SUBTYPE_IMAKEFILE)
	    cmobj->imakefile = fileobj;
	 else if (subtype == KCMS_FOBJ_SUBTYPE_MAKEFILE)
	    cmobj->makefile = fileobj;
	 else if (subtype == KCMS_FOBJ_SUBTYPE_APP_DEFAULTS)
	    cmobj->app_defaults = _kcms_cmobj_add_or_mod(cmobj->app_defaults,
							 token, fileobj);
	 else
	 {
	    errno = KCMS_RANGEERR;
	    return FALSE;
	 }
	 break;

      default:
	 errno = KCMS_RANGEERR;
	 return FALSE;
   }

   switch (generated)
   {
      case KCMS_FOBJ_GEN_NONE:
      case KCMS_FOBJ_GEN_MAN1:
      case KCMS_FOBJ_GEN_MAN3:
      case KCMS_FOBJ_GEN_DOFILE:
      case KCMS_FOBJ_GEN_GWFILE:
      case KCMS_FOBJ_GEN_COND:
	 /* do nothing.. legal value, but either already
	    handled or not handled at all.. */
	 break;
      case KCMS_FOBJ_GEN_LFILE:
	 cmobj->lfile = fileobj;
	 break;
      case KCMS_FOBJ_GEN_MAIN:
	 cmobj->cfile = fileobj;
	 break;
      case KCMS_FOBJ_GEN_INCL:
	 cmobj->hfile = fileobj;
	 break;
      case KCMS_FOBJ_GEN_HELP:
	 cmobj->helpfile = fileobj;
	 break;
      case KCMS_FOBJ_GEN_SCRIPT:
	 cmobj->scriptfile = fileobj;
	 break;
      default:
	 errno = KCMS_RANGEERR;
	 return FALSE;
   }
   return TRUE;
}

/*-----------------------------------------------------------
| Routine Name:	_kcms_cmobj_del_subobj - delete a file object from the cmobj
|
| Purpose:	This routine deletes the file object passed in from the
|		appropriate list in the cmobj object
|		structure, and updates the .cm file and Imakefile as
|		appropriate.
|
| Input:	cmobj - the cmobj kobject to delete a subobj from
|		fileobj - the file kobject to delete from the cmobj
| Output:	
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Steven Jorgensen
| Date:		Jun 07, 1993 10:25
------------------------------------------------------------*/
static int
_kcms_cmobj_del_subobj(
   kobject  cmobj,
   kobject  fileobj)
{
   /*UPDATE---------------------------------------------------------
   |    work routine does all the work.. This routine exists to
   |    be the standard interface to the kobject function pointer.
   |    this seems kinda kludgey -- where else is the work
   |    routine called from? -neilb
   +-------------------------------------------------------------- */

   return _kcms_cmobj_del_subobj_wrk(cmobj, fileobj, NULL);
}

/*-----------------------------------------------------------
| Routine Name:	_kcms_cmobj_mod_subobj - inform the cmobj of a change to a
|		subobj
|
| Purpose:	This routine rearranges the placement of the file object
|		passed in to a new list or status as appropriate in the
|		cmobj structure.  It also updates the .cm
|		file and Imakefile if appropriate.
|
| Input:	cmobj - the cmobj kobject to modify a subobj in
|		fileobj - the file kobject that was modified
| Output:	
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Steven Jorgensen
| Date:		Jun 07, 1993 10:25
------------------------------------------------------------*/
static int
_kcms_cmobj_mod_subobj(
   kobject  cmobj,
   kobject  fileobj)
{
   /*UPDATE:
   | note, this should be a call to delete followed by the call to add.
   | However, add does the delete if necessary (see note in
   | _kcms_cmobj_add_subobj).  So, if add changes, this routine will
   | have to change too.
   +---------------------------------------------------------------- */

   return _kcms_cmobj_add_subobj(cmobj, fileobj);
}

/*-----------------------------------------------------------
| Routine Name:	_kcms_cmobj_filenames_to_fileobjs - convert an array of file
|		names to a list of file
|		objects
|
| Purpose:	This routine takes the file name of a part of cmobj
|		object, and converts it to a list of equivalent file object
|		structures.
|
| Input:	cmobj     - cmobj we're adding objects to
|		path      - path to prepend onto filename.
|		array     - array of files
|		array_cnt - number of files in list
|		type      - the file's type
|		subtype   - the file's subtype
|		generated - ghost or conductor special file mark
|		acc       - file access flags
| Output:	
| Returns:	A list of kobjects on success, NULL if cnt is 0 or on error
|
| Written By:	Steven Jorgensen
| Date:		May 12, 1993 14:48
------------------------------------------------------------*/
static klist *
_kcms_cmobj_filenames_to_fileobjs(
   kobject   cmobj,
   char     *path,
   char    **array,
   int       array_cnt,
   int       type,
   int       subtype,
   int       generated,
   int       acc)
{
   kobject   tmp;
   klist    *fileobjs = NULL;
   int       i;
   int       token;
   kstring   tmp1;


   if (array_cnt <= 0)
      return NULL;
   for (i = 0; i < array_cnt; i++)
   {
      if ((tmp = kcms_open_fileobj(cmobj, array[i], path, type,
				   subtype, generated, acc)) == NULL)
      {
	 klist_free(fileobjs, close_n_free);
	 errno = KCMS_CANNOTMALLOCOBJ;
	 return NULL;
      }

      /*UPDATE-----------------------------------------------------
      | The following section is a KLUDGE, this info should really
      | be stored and parsed in the DBM file, but it hasn't been
      | implemented as yet.  --Aug 26, 1993 (SJ)
      +---------------------------------------------------------- */
      kcms_get_attribute(tmp, KCMS_FOBJ_FULLNAME_TKN, &token);
      kcms_get_attribute(tmp, KCMS_NAME, &tmp1);
      if (!kstrcmp(tmp1, "form_info.h")
	  || !kstrcmp(tmp1, "form_info.c")
	  || !kstrcmp(tmp1, "form_drv.c")
	  || !kstrcmp(tmp1, "form_init.c"))
      {
	 cmobj->noaccess = TRUE;
	 kcms_set_attribute(tmp, KCMS_FOBJ_ACCESS,
			    KCMS_FOBJ_ACCESS_RDONLY);
	 kcms_set_attribute(tmp, KCMS_FOBJ_GENERATED,
			    KCMS_FOBJ_GEN_COND);
	 cmobj->noaccess = FALSE;
      }
      else if (!kstrcmp(tmp1, "usage.c"))
      {
	 cmobj->noaccess = TRUE;
	 kcms_set_attribute(tmp, KCMS_FOBJ_ACCESS,
			    KCMS_FOBJ_ACCESS_RDONLY);
	 kcms_set_attribute(tmp, KCMS_FOBJ_GENERATED,
			    KCMS_FOBJ_GEN_GWFILE);
	 cmobj->noaccess = FALSE;
      }
		/*-----------------------------------------------------------
		| End of KLUDGE
		| --Aug 26, 1993 (SJ)
		+----------------------------------------------------------*/
      fileobjs = klist_add(fileobjs, (kaddr) token, (kaddr) tmp);
   }
   return fileobjs;
}

/*-----------------------------------------------------------
| Routine Name:	_kcms_cmobj_open_err - reset tb variables, set errno, and
|		print err on a cm parsing error
|
| Purpose:	This routine resets the toolbox sub-object name to the
|		name provided, the errno variable to the value
|		provided, and prints a VERBOSE error message if
|		mesg isn't NULL.  This routine exists to make
|		kcms_cmobj_open's error checking sane and short.
|
| Input:	cmobj   - cmobj that needs to be freed
|		merrno  - value to set errno to
|		mesg    - a message that is sent to kinfo in VERBOSE mode
|		kvalist - the list of arguments to make mesg a formatted
|		message (as in kinfo).
| Output:	
| Returns:	NULL all the time so you can put this call in the
|		return statement of the calling routine.
| Written By:	Steven Jorgensen
| Date:		May 23, 1993 17:47
------------------------------------------------------------*/
static kobject
_kcms_cmobj_open_err(
   kobject   cmobj,
   int       merrno,
   char     *mesg,
   kvalist)
{
   kva_list args;


   kva_start(args, mesg);
   if (mesg != NULL)
      kvinfo(KSYSLIB, mesg, args);

   kcms_cmobj_free(cmobj);
   errno = merrno;
   kva_end(args);
   return NULL;
}

/*-----------------------------------------------------------
| Routine Name:	_kcms_cmobj_filename_to_fileobj - get a key out of a database
|		make a file object
|
| Purpose:	Get a key out of the database, and set the appropriate file
|		object.  It will scan for the object in the list passed in if
|		it is not NULL, and just set attributes on the existing
|		file object.
|
| Input:	cmobj     - cmobj to set a find a set a sub object for.
|		database  - dbm pointer to get filename from
|		key       - dbm key to use as a search key
|		list      - klist to add the fileobj to
|		type      - file type of the fileobj
|		subtype   - file subtype of the fileobj
|		generated - file generated status for the fileobj
|		acc       - file access status for the fileobj
| Output:	
| Returns:	the file kobject that was opened on the parsed data, NULL
|		on error.
| Written By:	Steven Jorgensen
| Date:		May 23, 1993
------------------------------------------------------------*/
static kobject
_kcms_cmobj_filename_to_fileobj(
   kobject   cmobj,
   kdbm     *database,
   char     *key,
   klist   **list,
   int       type,
   int       subtype,
   int       generated,
   int       acc)
{
   kstring   tmp;
   int       token;
   klist    *tlist = NULL;
   kobject   fileobj;


   /*
    * BUG FIXED Nov 05, 1993 by SJ
    * since errno is now global, kutils setting of errno caused
    * bogus error condition that wasn't really an error.  This
    * errno setting is a kludge, but it'll work for now.. :)
    */
   errno = KCMS_OK;
   /*
    * end BUG FIX
    */

   tmp = kcms_db_get_key(database, key);
   if (tmp == NULL)
      return NULL;

   token = kstring_to_token(tmp);
   tlist = klist_locate(((list == NULL) ? NULL : *list), (kaddr) token);
   if (tlist == NULL)
   {
      fileobj = kcms_open_fileobj(cmobj, tmp, NULL, type, subtype,
				  generated, acc);
      if (list != NULL)
	 *list = klist_add(*list, (kaddr) token, (kaddr) fileobj);
   }
   else
   {
      fileobj = (kobject) klist_clientdata(tlist);
      kcms_set_attribute(fileobj, KCMS_FOBJ_GENERATED, generated);
      kcms_set_attribute(fileobj, KCMS_FOBJ_ACCESS, acc);
   }
   kfree(tmp);
   return fileobj;
}

/*-----------------------------------------------------------
| Routine Name:	_kcms_cmobj_get_doc - get the help and man pages for a cmobj
|
| Purpose:	This routine checks the standard doc directories for
|		a list of help and man pages, and adds them to the lists
|		of fileobjs for documentation.
|
| Input:	cmobj - the cmobj to get the information for
| Output:	
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Steven Jorgensen
| Date:		May 31, 1993 16:43
------------------------------------------------------------*/
static int
_kcms_cmobj_get_doc(
   kobject cmobj)
{
   kobject    fobj;
   char       path[KLENGTH];
   char     **llist;
   char      *tbname;
   int        lcnt;
   int        i;
   int        tkn;
   int        htkn = 0;


   if (cmobj->helpfile != NULL)
      kcms_get_attribute(cmobj->helpfile, KCMS_FOBJ_FULLNAME_TKN, &htkn);
   if (!kcms_get_attribute(cmobj->parent, KCMS_NAME, &tbname))
      return FALSE;
   ksprintf(path, "$%s/objects/%s/%s/help", tbname,
	    kcms_attr_int2string(KCMS_CMOBJ_TYPE, cmobj->prog_type),
	    cmobj->oname);
   llist = karray_dirlist(NULL, path, NULL, KRECURSE | KFILE | KPATH,
			  FALSE, &lcnt);

   /*UPDATE: this routine needs sorting out! -- */
   /* cmobj->help_files = NULL; */
   for (i = 0; i < lcnt; i++)
   {
      kcms_create_toplevel(cmobj->parent, llist[i], path);
      tkn = kstring_to_token(path);
      if (tkn != htkn)
	 fobj = kcms_open_fileobj(cmobj, path, NULL,
				  KCMS_FOBJ_TYPE_HELP,
				  KCMS_FOBJ_SUBTYPE_HELP,
				  KCMS_FOBJ_GEN_NONE,
				  KCMS_FOBJ_ACCESS_RDWR);
      else
	 fobj = cmobj->helpfile;
      cmobj->help_files = klist_add(cmobj->help_files, (kaddr) tkn,
				    (kaddr) fobj);
   }
   karray_free(llist, lcnt, NULL);
   cmobj->libman_files = NULL;
   if (cmobj->prog_type == KCMS_LIBRARY)
   {
      ksprintf(path, "$%s/objects/%s/%s/man", tbname,
	       kcms_attr_int2string(KCMS_CMOBJ_TYPE, cmobj->prog_type),
	       cmobj->oname);
      llist = karray_dirlist(NULL, path, NULL, KRECURSE | KFILE |
			     KPATH, FALSE, &lcnt);
      for (i = 0; i < lcnt; i++)
      {
	 kcms_create_toplevel(cmobj->parent, llist[i], path);
	 tkn = kstring_to_token(path);
	 fobj = kcms_open_fileobj(cmobj, path, NULL,
				  KCMS_FOBJ_TYPE_MAN,
				  KCMS_FOBJ_SUBTYPE_MAN3,
				  KCMS_FOBJ_GEN_NONE,
				  KCMS_FOBJ_ACCESS_RDWR);
	 cmobj->libman_files = klist_add(cmobj->libman_files,
					 (kaddr) tkn, (kaddr) fobj);
      }
      karray_free(llist, lcnt, NULL);
   }

   /*-- KROUTINE with associated library function ---------------------*/
   if (cmobj->prog_type == KCMS_KROUTINE && cmobj->library != NULL)
   {
      ksprintf(path, "$%s/objects/library/%s/man/l%s.3",
	       tbname, cmobj->library, cmobj->oname);
      tkn = kstring_to_token(path);
      fobj = kcms_open_fileobj(cmobj, path, NULL,
			       KCMS_FOBJ_TYPE_MAN,
			       KCMS_FOBJ_SUBTYPE_MAN3,
			       KCMS_FOBJ_GEN_LFILE,
			       KCMS_FOBJ_ACCESS_RDWR);
      cmobj->libman_files = klist_add(cmobj->libman_files,
				      (kaddr)tkn, (kaddr)fobj);
   }

   return TRUE;
}

/*-----------------------------------------------------------
| Routine Name:	kcms_cmobj_complete_open - completely open a cmobj
|
| Purpose:	This routine reads in the configuration files
|		for an existing program and fills out a kobject
|		structure that is returned to the user.
|
| Input:	cmobj  -	a kobject containing a partially open kcms
|		software object.
| Output:	
| Returns:	a kobject containing all the path, ghost, and state
|		information.  Or NULL on error.
|
| Written By:	Steven Jorgensen and Neil Bowers
| Date:		Apr 26, 1993 16:56
------------------------------------------------------------*/
kobject
kcms_cmobj_complete_open(
   kobject cmobj)
{
   kstring    routine = "Complete Object Open";
   kobject    fobj;
   kstring    oname;
   char       temp[KLENGTH];
   kstring    imakename;
   kstring    cmname;
   kstring   *list;
   kstring    tmp;
   int        count;
   int        cnt;
   int        token;
   int        update_immediate = FALSE;
   kdbm      *database;
   kobject    tmpobj;
   kobject    toolbox;
   klist     *tlist;
   kstring    tbname;
   kstring    opath;
   int        i;


   errno = KCMS_OK;

   if (!kcms_legal_object(cmobj, routine))
      return NULL;

   if (cmobj->fullyopen)
      return cmobj;

   kcms_get_attribute(cmobj, KCMS_PARENT, &toolbox);
   kcms_get_attribute(cmobj, KCMS_PATH, &opath);
   kcms_get_attribute(cmobj, KCMS_NAME, &oname);

   cmobj->noaccess = TRUE;
   if (!kcms_get_attribute(toolbox, KCMS_NAME, &tbname))
      return NULL;
   ksprintf(temp, "%s/db/cms", opath);
   cmobj->database = kcms_open_fileobj(cmobj, temp, NULL,
				       KCMS_FOBJ_TYPE_DBM,
				       KCMS_FOBJ_SUBTYPE_CM,
				       KCMS_FOBJ_GEN_NONE,
				       KCMS_FOBJ_ACCESS_RDWR);

   cmobj->noaccess = FALSE;
   cmobj->fullyopen = TRUE;


   /*--------------------------------------------------------------+
   | open the object database
   +--------------------------------------------------------------*/
   kcms_get_attribute(cmobj->database, KCMS_PATH, &cmname);

   if ((database = kdbm_open(cmname, O_RDONLY, 0666)) == NULL)
      return _kcms_cmobj_open_err(cmobj, KCMS_CMOPENERR,
		      "%s: Cannot open the .cm file '%s'", routine, cmname);

   /*-- determine type of software object -----------------------------*/
   if ((tmp = kcms_db_get_key(database, KCMS_KEY_CMOBJ_TYPE)) == NULL)
      return _kcms_cmobj_open_err(cmobj, KCMS_RANGEERR,
		       "%s: Cannot determine object type from the database",
				  routine);
   cmobj->prog_type = kcms_attr_string2int(KCMS_CMOBJ_PROGTYPE, tmp);
   kfree(tmp);

   /*-- get the code type: ANSI or KNR --*/
   tmp = kcms_db_get_key(database, KCMS_KEY_CMOBJ_STATE_CTYPE);
   cmobj->code_type = (tmp != NULL
		       ? kcms_attr_string2int(KCMS_CMOBJ_CODETYPE, tmp)
		       : KCMS_ANSI);
   kfree(tmp);

   /*-- object dates --------------------------------------------------*/
   kcms_db_read_dates(database,
		      &cmobj->times.creation,
		      &cmobj->times.modification,
		      &cmobj->times.generation);

   /*-- object's binary name ------------------------------------------*/
   cmobj->bname = kcms_db_get_key(database, KCMS_KEY_CMOBJ_BNAME);

   /*-- object version string -----------------------------------------*/
   cmobj->version = kcms_db_get_key(database, KCMS_KEY_VERSION);

   cmobj->short_description = kcms_db_get_key(database,
					      KCMS_KEY_SHORT_DESCRIPTION);

   /*-- object's icon name (used as cantata glyph name, eg) -----------*/
   cmobj->icon_name = kcms_db_get_key(database, KCMS_KEY_ICON_NAME);

   /*-- any keywords specified for the object? ------------------------*/
   if ((tmp = kcms_db_get_key(database, KCMS_KEY_KEYWORDS)) != NULL)
   {
      list = kparse_string_delimit(tmp, "\n", KDELIM_CLEAN, &count);
      for (i=0; i<count; i++)
      {
	 token = kstring_to_token(list[i]);
	 cmobj->keywords = klist_add(cmobj->keywords, (kaddr)token,
				     (kaddr)kstrdup(list[i]));
      }
      karray_free(list, count, NULL);
      kfree(tmp);
   }

   /*-- get the language type -----------------------------------------*/
   tmp = kcms_db_get_key(database, KCMS_KEY_CMOBJ_LANG);
   cmobj->lang_type = (tmp != NULL
		       ? kcms_attr_string2int(KCMS_CMOBJ_LANGTYPE, tmp)
		       : KCMS_NOLANG);
   kfree(tmp);

   /*-- is the object installed in cantata? ---------------------------*/
   tmp = kcms_db_get_key(database, KCMS_KEY_IN_CANTATA);
   cmobj->ci_installed = tmp ? kcms_decode_yesno(tmp) : FALSE;
   kfree(tmp);

   /*-- the generate executable flag is used for pane objects ---------*/
   if (cmobj->prog_type == KCMS_PANE)
   {
      kstring  *substrings;
      int       sub_count;


      tmp = kcms_db_get_key(database, KCMS_KEY_GENERATE_EXECUTABLE);
      cmobj->generate_executable = tmp ? kcms_decode_yesno(tmp) : FALSE;
      kfree(tmp);

      tmp = kcms_db_get_key(database, KCMS_KEY_ASSOCIATED_OBJECT);
      if (tmp != NULL)
      {
	 substrings = kparse_string_delimit(tmp, "\n", KDELIM_CLEAN,
					    &sub_count);
	 if (sub_count != 2 && sub_count != 3)
	    kerror(KCMS, routine, "Malformed database entry for associated "
		   "software object (under key %s) -- ignoring.",
		   KCMS_KEY_ASSOCIATED_OBJECT);
	 else
	 {
	    cmobj->assoc_tbname = kstrdup(substrings[0]);
	    cmobj->assoc_oname  = kstrdup(substrings[1]);
	    if (sub_count == 3 && kstrlen(substrings[2]) > 0)
	       cmobj->assoc_args = kstrdup(substrings[2]);
	 }
	 karray_free(substrings, sub_count, NULL);
	 kfree(tmp);
      }
   }

   /*-- get the point of contact entries for the object --*/
   cmobj->poc_email = kcms_db_get_key(database, KCMS_KEY_AUTHOR_EMAIL);
   cmobj->poc_name = kcms_db_get_key(database, KCMS_KEY_AUTHOR);
   cmobj->category = kcms_db_get_key(database, KCMS_KEY_CATEGORY);
   cmobj->subcategory = kcms_db_get_key(database, KCMS_KEY_SUBCATEGORY);

   /*-- get object's todo, bugs and done files --*/
   cmobj->todo = _kcms_cmobj_filename_to_fileobj(cmobj, database,
						 KCMS_KEY_TODO,
						 NULL, KCMS_FOBJ_TYPE_INFO,
						 KCMS_FOBJ_SUBTYPE_TODO,
						 KCMS_FOBJ_GEN_NONE,
						 KCMS_FOBJ_ACCESS_RDWR);

   cmobj->bugs = _kcms_cmobj_filename_to_fileobj(cmobj, database,
						 KCMS_KEY_BUGS,
						 NULL, KCMS_FOBJ_TYPE_INFO,
						 KCMS_FOBJ_SUBTYPE_BUGS,
						 KCMS_FOBJ_GEN_NONE,
						 KCMS_FOBJ_ACCESS_RDWR);

   cmobj->done = _kcms_cmobj_filename_to_fileobj(cmobj, database,
						 KCMS_KEY_DONE,
						 NULL, KCMS_FOBJ_TYPE_INFO,
						 KCMS_FOBJ_SUBTYPE_DONE,
						 KCMS_FOBJ_GEN_NONE,
						 KCMS_FOBJ_ACCESS_RDWR);

   cmobj->changelog = _kcms_cmobj_filename_to_fileobj(cmobj, database,
						      KCMS_KEY_CHANGELOG,
						  NULL, KCMS_FOBJ_TYPE_INFO,
						KCMS_FOBJ_SUBTYPE_CHANGELOG,
						      KCMS_FOBJ_GEN_NONE,
						      KCMS_FOBJ_ACCESS_RDWR);

   /*-- workspace file object --*/
   cmobj->workspace = _kcms_cmobj_filename_to_fileobj(cmobj, database,
						      KCMS_KEY_WORKSPACE,
						   NULL, KCMS_FOBJ_TYPE_DBM,
						KCMS_FOBJ_SUBTYPE_WORKSPACE,
						      KCMS_FOBJ_GEN_NONE,
						      KCMS_FOBJ_ACCESS_RDWR);

   /*-- get the YSRCS, list of yacc source files --*/
   tmp = kcms_db_get_key(database, KCMS_KEY_CMOBJ_YSRCS);
   list = kparse_string_delimit(tmp, "\n", KDELIM_CLEAN, &count);
   kfree(tmp);
   cmobj->y_files = _kcms_cmobj_filenames_to_fileobjs(cmobj,
						      NULL,
						      list, count,
						      KCMS_FOBJ_TYPE_SRC,
						      KCMS_FOBJ_SUBTYPE_YACC,
						      KCMS_FOBJ_GEN_NONE,
						      KCMS_FOBJ_ACCESS_RDWR);
   karray_free(list, count, NULL);
   if (cmobj->y_files == NULL && count != 0)
      return NULL;
   /*-- make sure there is an error msg --*/

   /*-- get the LSRCS, list of lex source files --*/
   tmp = kcms_db_get_key(database, KCMS_KEY_CMOBJ_LSRCS);
   list = kparse_string_delimit(tmp, "\n", KDELIM_CLEAN, &count);
   kfree(tmp);
   cmobj->l_files = _kcms_cmobj_filenames_to_fileobjs(cmobj,
						      NULL,
						      list, count,
						      KCMS_FOBJ_TYPE_SRC,
						      KCMS_FOBJ_SUBTYPE_LEX,
						      KCMS_FOBJ_GEN_NONE,
						      KCMS_FOBJ_ACCESS_RDWR);
   karray_free(list, count, NULL);
   if (cmobj->l_files == NULL && count != 0)
      return NULL;
   /*-- make sure there is an error msg --*/

   /*-- get the C++SRCS, list of C++ source files --*/
   tmp = kcms_db_get_key(database, KCMS_KEY_CMOBJ_CPLUSPLUS);
   list = kparse_string_delimit(tmp, "\n", KDELIM_CLEAN, &count);
   kfree(tmp);
   cmobj->cplus_files = _kcms_cmobj_filenames_to_fileobjs(cmobj,
							  NULL,
							  list, count,
							  KCMS_FOBJ_TYPE_SRC,
						KCMS_FOBJ_SUBTYPE_CPLUSPLUS,
							  KCMS_FOBJ_GEN_NONE,
						     KCMS_FOBJ_ACCESS_RDWR);
   karray_free(list, count, NULL);
   if (cmobj->cplus_files == NULL && count != 0)
      return NULL;
   /*-- make sure there is an error msg --*/

   /*-- get the LSRCS, list of C source files --*/
   tmp = kcms_db_get_key(database, KCMS_KEY_CMOBJ_SRCS);
   list = kparse_string_delimit(tmp, "\n", KDELIM_CLEAN, &count);
   kfree(tmp);
   cmobj->c_files = _kcms_cmobj_filenames_to_fileobjs(cmobj,
						      NULL, list, count,
						      KCMS_FOBJ_TYPE_SRC,
						      KCMS_FOBJ_SUBTYPE_C,
						      KCMS_FOBJ_GEN_NONE,
						      KCMS_FOBJ_ACCESS_RDWR);
   karray_free(list, count, NULL);
   if (cmobj->c_files == NULL && count != 0)
      return NULL;
   /*-- make sure there is an error msg --*/

   /*-- get the FSRCS, list of fortran source files --*/
   tmp = kcms_db_get_key(database, KCMS_KEY_CMOBJ_FSRCS);
   list = kparse_string_delimit(tmp, "\n", KDELIM_CLEAN, &count);
   kfree(tmp);
   cmobj->f_files = _kcms_cmobj_filenames_to_fileobjs(cmobj,
						      NULL,
						      list, count,
						      KCMS_FOBJ_TYPE_SRC,
						  KCMS_FOBJ_SUBTYPE_FORTRAN,
						      KCMS_FOBJ_GEN_NONE,
						      KCMS_FOBJ_ACCESS_RDWR);
   karray_free(list, count, NULL);
   if (cmobj->f_files == NULL && count != 0)
      return NULL;
   /*-- make sure there is an error msg --*/

   /*-- dependant library and l<oname>.c for ghostwriter --------------*/
   tmp = kcms_db_get_key(database, KCMS_KEY_CMOBJ_GHOST_LIB);
   if (tmp != NULL)
   {
      cmobj->library = tmp;
      tmpobj = kcms_open_cmobj(cmobj->parent, cmobj->library);
      if (tmpobj == NULL)
	 kerror(KCMS, routine, "Error, library object '%s' does not exist, "
		"so no path to the 'l%s.c' file is available.",
		cmobj->library, cmobj->oname);
      else
      {
	 kcms_get_attribute(tmpobj, KCMS_CMOBJ_TOPSRC, &tmp);
	 ksprintf(temp, "%s/l%s.c", tmp, cmobj->oname);
	 cmobj->lfile = kcms_open_fileobj(cmobj, temp, NULL,
				    KCMS_FOBJ_TYPE_SRC, KCMS_FOBJ_SUBTYPE_C,
					  KCMS_FOBJ_GEN_LFILE,
					  KCMS_FOBJ_ACCESS_RDWR);
      }
   }

   /*-- get the HEADERS, list of include files --*/
   tmp = kcms_db_get_key(database, KCMS_KEY_CMOBJ_HEADERS);
   list = kparse_string_delimit(tmp, "\n", KDELIM_CLEAN, &count);
   kfree(tmp);
   cmobj->inc_files = _kcms_cmobj_filenames_to_fileobjs(cmobj,
							NULL,
							list, count,
							KCMS_FOBJ_TYPE_SRC,
						     KCMS_FOBJ_SUBTYPE_INCL,
							KCMS_FOBJ_GEN_NONE,
						     KCMS_FOBJ_ACCESS_RDWR);
   karray_free(list, count, NULL);
   if (cmobj->inc_files == NULL && count != 0)
      return NULL;
   /*-- make sure there is an error msg --*/

   /*--------------------------------------------------------------------
   | if the object is a kroutine, and has associated library function
   | then add the library's public include file to the list of includes
   +-------------------------------------------------------------------*/

   if (cmobj->prog_type == KCMS_KROUTINE && cmobj->library != NULL)
   {
      ksprintf(temp, "$%s/include/%s/%s.h", tbname, cmobj->library,
	       cmobj->library);
      fobj = kcms_open_fileobj(cmobj, temp, NULL,
			       KCMS_FOBJ_TYPE_SRC, KCMS_FOBJ_SUBTYPE_INCL,
			       KCMS_FOBJ_GEN_LFILE, KCMS_FOBJ_ACCESS_RDWR);
      if (fobj != NULL)
      {
	 kcms_get_attribute(fobj, KCMS_FOBJ_FULLNAME_TKN, &token);
	 tlist = klist_add(NULL, (kaddr)token, (kaddr)fobj);
	 cmobj->inc_files = klist_merge(cmobj->inc_files, tlist);
      }
   }

   /*UPDATE: separate attributes for private and public includes -- */

   /*-- get object's public includes, only applies to library? --*/
   ksprintf(temp, "$%s/include/%s", tbname, cmobj->oname);
   list = karray_dirlist(NULL, temp, "\\.h$", KFILE | KLINK | KREAD, FALSE,
			 &count);
   tlist = _kcms_cmobj_filenames_to_fileobjs(cmobj, temp, list, count,
				 KCMS_FOBJ_TYPE_SRC, KCMS_FOBJ_SUBTYPE_INCL,
				 KCMS_FOBJ_GEN_NONE, KCMS_FOBJ_ACCESS_RDWR);
   cmobj->inc_files = klist_merge(cmobj->inc_files, tlist);
   karray_free(list, count, NULL);


   /*-- get the SCRIPTS, list of script files --*/
   if (cmobj->prog_type == KCMS_SCRIPT)
   {
      tmp = kcms_db_get_key(database, KCMS_KEY_CMOBJ_SCRIPTS);
      list = kparse_string_delimit(tmp, "\n", KDELIM_CLEAN, &count);
      kfree(tmp);

      /*UPDATE: shouldn't be hard-wired to SUBTYPE_SH */
      cmobj->script_files = _kcms_cmobj_filenames_to_fileobjs(cmobj,
							      NULL,
							      list, count,
						      KCMS_FOBJ_TYPE_SCRIPT,
						       KCMS_FOBJ_SUBTYPE_SH,
							 KCMS_FOBJ_GEN_NONE,
						     KCMS_FOBJ_ACCESS_RDWR);
      karray_free(list, count, NULL);
      /*-- make sure there is an error msg --*/
      if (cmobj->script_files == NULL && count != 0)
	 return NULL;
   }

   /*---------------------------------------------------------------
   |	get Imakefile information
   +--------------------------------------------------------------*/
   if (cmobj->prog_type != KCMS_PANE || cmobj->generate_executable)
   {
      cmobj->makefile = kcms_open_fileobj(cmobj, "Makefile",
					  cmobj->topsrc,
					  KCMS_FOBJ_TYPE_CONFIG,
					  KCMS_FOBJ_SUBTYPE_MAKEFILE,
					  KCMS_FOBJ_GEN_NONE,
					  KCMS_FOBJ_ACCESS_RDWR);

      cmobj->imakefile = kcms_open_fileobj(cmobj, "Imakefile",
					   cmobj->topsrc,
					   KCMS_FOBJ_TYPE_CONFIG,
					   KCMS_FOBJ_SUBTYPE_IMAKEFILE,
				 KCMS_FOBJ_GEN_NONE, KCMS_FOBJ_ACCESS_RDWR);

      kcms_get_attribute(cmobj->imakefile, KCMS_PATH, &imakename);
   }


   /*--------------------------------------------------------------
   |	object version string
   +-------------------------------------------------------------*/

   tmp = kcms_db_get_key(database, KCMS_KEY_CMS_VERSION);
   if (kstrcmp(tmp, CM_VERSION) != 0)
   {
      kerror(KCMS, routine, "KCMS Warning:  Current database version is "
	     "'%s', and your CM file for '%s' is outdated.  "
	     "Therefore the CM and Imakefile will be regenerated.",
	     CM_VERSION, oname);
      update_immediate = TRUE;
   }
   kfree(tmp);

   /*----------------------------------------------------------------
   |	get misc file path information
   +---------------------------------------------------------------*/
   tmp = kcms_db_get_key(database, KCMS_KEY_MISC_FILES);
   cnt = 0;
   list = kparse_string_delimit(tmp, "\n", KDELIM_CLEAN, &cnt);
   cmobj->misc_files = _kcms_cmobj_filenames_to_fileobjs(cmobj, NULL,
							 list, cnt,
							 KCMS_FOBJ_TYPE_MISC,
						     KCMS_FOBJ_SUBTYPE_NONE,
							 KCMS_FOBJ_GEN_NONE,
						     KCMS_FOBJ_ACCESS_RDWR);
   kfree(tmp);
   karray_free(list, cnt, NULL);
   if (cmobj->misc_files == NULL && cnt != 0)
      return _kcms_cmobj_open_err(cmobj, errno, "%s: %s.",
				  "kcms_cmobj_complete_open",
			    "Error create the misc file object list field");

   /* get misc_files from the cmobj */
   tmp = kcms_db_get_key(database, KCMS_KEY_APP_DEFAULTS);
   cnt = 0;
   list = kparse_string_delimit(tmp, "\n", KDELIM_CLEAN, &cnt);
   cmobj->app_defaults = _kcms_cmobj_filenames_to_fileobjs(cmobj, NULL,
							   list, cnt,
					       KCMS_FOBJ_TYPE_CONFIG,
					       KCMS_FOBJ_SUBTYPE_APP_DEFAULTS,
					       KCMS_FOBJ_GEN_NONE,
					       KCMS_FOBJ_ACCESS_RDWR);
   karray_free(list, cnt, NULL);
   kfree(tmp);
   if (cmobj->app_defaults == NULL && cnt != 0)
      return _kcms_cmobj_open_err(cmobj, errno, "%s: %s",
				  "kcms_cmobj_complete_open",
		      "Error creating app-defaults file object list field");

   /*-------------------------------------------------------------
   |	get UIS path information
   +------------------------------------------------------------*/
   if (cmobj->pane == NULL)
      cmobj->pane = _kcms_cmobj_filename_to_fileobj(cmobj, database,
						    KCMS_KEY_CMOBJ_UIS_PANE,
						    NULL, KCMS_FOBJ_TYPE_UIS,
						    KCMS_FOBJ_SUBTYPE_PANE,
						    KCMS_FOBJ_GEN_NONE,
						    KCMS_FOBJ_ACCESS_RDWR);
   if (kerrno_check(errno, kcms_errors, kcms_numerrors) && errno != KCMS_OK)
      return _kcms_cmobj_open_err(cmobj, errno, NULL);

   cmobj->form = _kcms_cmobj_filename_to_fileobj(cmobj, database,
						 KCMS_KEY_CMOBJ_UIS_FORM,
						 NULL, KCMS_FOBJ_TYPE_UIS,
						 KCMS_FOBJ_SUBTYPE_FORM,
						 KCMS_FOBJ_GEN_NONE,
						 KCMS_FOBJ_ACCESS_RDWR);
   if (kerrno_check(errno, kcms_errors, kcms_numerrors) && errno != KCMS_OK)
      return _kcms_cmobj_open_err(cmobj, errno, NULL);

   /*-- miscellaneous UIS files --*/
   tmp = kcms_db_get_key(database, KCMS_KEY_UIS_MISC);
   cnt = 0;
   list = kparse_string_delimit(tmp, "\n", KDELIM_CLEAN, &cnt);
   cmobj->miscuis = _kcms_cmobj_filenames_to_fileobjs(cmobj, NULL,
						      list, cnt,
						      KCMS_FOBJ_TYPE_UIS,
						      KCMS_FOBJ_SUBTYPE_MISC,
						      KCMS_FOBJ_GEN_NONE,
						      KCMS_FOBJ_ACCESS_RDWR);
   karray_free(list, cnt, NULL);
   kfree(tmp);
   if (cmobj->miscuis == NULL && cnt != 0)
      return _kcms_cmobj_open_err(cmobj, errno, "%s: %s",
				  "kcms_cmobj_complete_open",
			  "Error creating miscellaneous file object list.");

   /*----------------------------------------------------------------
   |	get code generation path information
   +---------------------------------------------------------------*/
   /* <oname>.c for ghostwriter */
   cmobj->cfile = _kcms_cmobj_filename_to_fileobj(cmobj, database,
						  KCMS_KEY_CMOBJ_GHOST_SRC,
						  &cmobj->c_files,
						  KCMS_FOBJ_TYPE_SRC,
						  KCMS_FOBJ_SUBTYPE_C,
						  KCMS_FOBJ_GEN_MAIN,
						  KCMS_FOBJ_ACCESS_RDWR);
   if (kerrno_check(errno, kcms_errors, kcms_numerrors) && errno != KCMS_OK)
      return _kcms_cmobj_open_err(cmobj, errno, NULL);

   /* <oname>.h for ghostwriter */
   cmobj->hfile = _kcms_cmobj_filename_to_fileobj(cmobj, database,
						KCMS_KEY_CMOBJ_GHOST_HEADER,
						  &cmobj->inc_files,
						  KCMS_FOBJ_TYPE_SRC,
						  KCMS_FOBJ_SUBTYPE_C,
						  KCMS_FOBJ_GEN_INCL,
						  KCMS_FOBJ_ACCESS_RDWR);
   if (kerrno_check(errno, kcms_errors, kcms_numerrors) && errno != KCMS_OK)
      return _kcms_cmobj_open_err(cmobj, errno, NULL);

   /* <oname>.sh for ghostwriter */
   cmobj->scriptfile = _kcms_cmobj_filename_to_fileobj(cmobj, database,
						KCMS_KEY_CMOBJ_GHOST_SCRIPT,
						       &cmobj->script_files,
						       KCMS_FOBJ_TYPE_SCRIPT,
						       KCMS_FOBJ_SUBTYPE_SH,
						       KCMS_FOBJ_GEN_SCRIPT,
						     KCMS_FOBJ_ACCESS_RDWR);
   if (kerrno_check(errno, kcms_errors, kcms_numerrors) && errno != KCMS_OK)
      return _kcms_cmobj_open_err(cmobj, errno, NULL);

   /* man1/<bname> for ghostwriter */
   cmobj->man1file = _kcms_cmobj_filename_to_fileobj(cmobj, database,
						  KCMS_KEY_CMOBJ_GHOST_MAN1,
						   NULL, KCMS_FOBJ_TYPE_MAN,
						     KCMS_FOBJ_SUBTYPE_MAN1,
						     KCMS_FOBJ_GEN_MAN1,
						     KCMS_FOBJ_ACCESS_RDWR);
   if (kerrno_check(errno, kcms_errors, kcms_numerrors) && errno != KCMS_OK)
      return _kcms_cmobj_open_err(cmobj, errno, NULL);

   /* man3/<bname> for ghostwriter */
   cmobj->man3file = _kcms_cmobj_filename_to_fileobj(cmobj, database,
						  KCMS_KEY_CMOBJ_GHOST_MAN3,
						   NULL, KCMS_FOBJ_TYPE_MAN,
						     KCMS_FOBJ_SUBTYPE_MAN3,
						     KCMS_FOBJ_GEN_MAN3,
						     KCMS_FOBJ_ACCESS_RDWR);
   if (kerrno_check(errno, kcms_errors, kcms_numerrors) && errno != KCMS_OK)
      return _kcms_cmobj_open_err(cmobj, errno, NULL);

   /* <oname>.hlp for ghostwriter */
   cmobj->helpfile = _kcms_cmobj_filename_to_fileobj(cmobj, database,
						  KCMS_KEY_CMOBJ_GHOST_HELP,
						     &cmobj->help_files,
						     KCMS_FOBJ_TYPE_HELP,
						     KCMS_FOBJ_SUBTYPE_HELP,
						     KCMS_FOBJ_GEN_HELP,
						     KCMS_FOBJ_ACCESS_RDWR);
   if (kerrno_check(errno, kcms_errors, kcms_numerrors) && errno != KCMS_OK)
      return _kcms_cmobj_open_err(cmobj, errno, NULL);

   if (_kcms_cmobj_get_doc(cmobj) == FALSE)
      return _kcms_cmobj_open_err(cmobj, errno, "%s: %s.",
				  "kcms_cmobj_complete_open:",
				  "Error getting the doc file objects");

   /*---------------------------------------------------------------
   |	get conductor info
   +---------------------------------------------------------------*/
   if (cmobj->form != NULL)
   {
      tmp = kcms_db_get_key(database, KCMS_KEY_CMOBJ_COND_LEVELS);
      cmobj->levels = (tmp == NULL ? 3 : atoi(tmp));
      if (cmobj->levels < 1 || cmobj->levels > 3)
	 cmobj->levels = 3;
      kfree(tmp);
   }

   kdbm_close(database);

   if (update_immediate && !kcms_cmobj_sync(cmobj))
      return _kcms_cmobj_open_err(cmobj, errno, "%s: %s.",
				  "kcms_cmobj_complete_open",
				  "Error updating the cmobj's database");
   return cmobj;
}

/*------------------------------------------------------------
| Routine Name:	kcms_cmobj_close - close a cmobj
|
| Purpose:	This routine just frees the program object structure
|		associated with an opened program or library object.
|
| Input:	cmobj - cmobj to close
| Output:	
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Steven Jorgensen
| Date:		Jan 07, 1993
------------------------------------------------------------*/
int
kcms_cmobj_close(
   kobject cmobj)
{
   errno = KCMS_OK;

   if (!kcms_legal_object(cmobj, "Close Object"))
      return FALSE;

   if (cmobj->update != KCMS_UPDATE_NONE)
      kcms_cmobj_sync(cmobj);
   kcms_sync_header_close();
   kcms_cmobj_free(cmobj);

   return TRUE;
}

static char *library_options[2] =
{
   "Create library, copy just the l*.c file over",
   "Copy the entire library as well"
};

/*------------------------------------------------------------
| Routine Name:	kcms_cmobj_duplicate - duplicate an existing software object.
|
| Purpose:	This routine generates a copy of an existing software object
|		in another toolbox.  This is the back-end for kcms_duplicate(),
|		when called with a software object.
|
| Input:	cmobj   - the software object being copied.
|		toolbox - the open toolbox object to copy the object into.
|
| Returns:	a pointer to the new (duplicate) software object,
|		or NULL on error
|
| Written By:	Steven Jorgensen and Neil Bowers
| Date:		Jan 24, 1993 00:37
------------------------------------------------------------*/
kobject
kcms_cmobj_duplicate(
   kobject  cmobj,
   kobject  toolbox)
{
   kstring     routine = "Duplicate Object";
   int         fgen;
   int         force_flag = FALSE;
   kobject     new_cmobj = NULL;
   kobject     fileobj1;
   kobject     fileobj2;
   char        temp[KLENGTH];
   kstring     fpath;
   klist      *flist;
   klist      *destn_files         = NULL;
   klist      *df_attr;
   static int  string_attributes[] =
               {
		  KCMS_CMOBJ_ICON_NAME,
		  KCMS_CMOBJ_AUTHOR,
		  KCMS_CMOBJ_AUTHOR_EMAIL,
		  KCMS_CMOBJ_CATEGORY,
		  KCMS_CMOBJ_SUBCATEGORY,
		  KCMS_CMOBJ_SHORT_DESC,
		  KCMS_CMOBJ_VERSION
	       };
   static int  int_attributes[] =
               {
		  KCMS_CMOBJ_CODETYPE,
		  KCMS_CMOBJ_LEVELS,
		  KCMS_CMOBJ_LANGTYPE,
		  KCMS_CMOBJ_INCANTATA
               };
   int       i;
   int       token;
   int       dest_token;
   int       dbToken;
   int       makeToken;
   kstring   otbname;
   kstring   newtbname;
   kdbm     *dbm;
   int       genexec;
   int       ftype;
   kbool     updating   = FALSE;	/* updating object in destination? */
   char      destpath[KLENGTH];
   kstring   relpath;
   kobject   fobj;
   kobject   libobj;
   int       libchoice;
   char      retstring[KLENGTH];
   int       src_int_value;
   int       dest_int_value;
   kstring   src_string_value;
   kstring   dest_string_value;


   if (!kcms_legal_object(cmobj, routine)
       || !kcms_legal_object(toolbox, routine))
      return NULL;

   if (!kcms_get_attribute(cmobj, KCMS_CMOBJ_ALL_FILES, &flist)
       || !kcms_get_indirect_attribute(cmobj, KCMS_PARENT, KCMS_NAME, &otbname)
       || !kcms_get_attribute(toolbox, KCMS_NAME, &newtbname))
      return NULL;

   if (!kstrcasecmp(newtbname, otbname))
   {
      kerror(KCMS, routine, "source and destination toolboxes are the same!");
      return NULL;
   }

   /*-- lkroutine check: how to deal with the library? ---------------*/
   if (cmobj->prog_type == KCMS_KROUTINE && cmobj->library != NULL)
   {
      if ((libobj = kcms_open_cmobj(toolbox, cmobj->library)) == NULL)
      {
	 kchoose(KSTANDARD, library_options, 2, 0, retstring, &libchoice,
		 "The object `%s' is marked as having an associated "
		 "library routine in library `%s'.\n"
		 "This library does not exist in toolbox `%s'.\n",
		 cmobj->oname, cmobj->library, newtbname);
	 if (libchoice == 0)
	    return NULL;
      }
      else
      {
	 kcms_close(libobj);
	 libchoice = 3;
      }
   }

   /*-- is there already a version of the object in destination? -----*/
   if ((new_cmobj = kcms_open_cmobj(toolbox, cmobj->oname)) != NULL)
   {
      if (new_cmobj->prog_type != cmobj->prog_type)
      {
	 if (!kprompt(KSTANDARD, NULL, NULL, 0,
		      "\nOperation: duplicate software object\n\n"
		      "\tToolbox:             %s\n"
		      "\tObject:              %s\n"
		      "\tDestination Toolbox: %s\n\n"
		      "The \"%s\" object in toolbox \"%s\" is not the "
		      "same type (%s) as the source object (%s).\n"
		      "Do you want to delete the object in the destination "
		      "toolbox?",
		      otbname, cmobj->oname,
		      cmobj->oname, newtbname,
		      kcms_attr_int2string(KCMS_CMOBJ_TYPE,
					   new_cmobj->prog_type),
		      kcms_attr_int2string(KCMS_CMOBJ_TYPE,
					   cmobj->prog_type)))
	 {
	    kcms_close(new_cmobj);
	    return NULL;
	 }
	 kcms_set_bit(new_cmobj, KCMS_CMOBJ_FLAGS, KCMS_BIT_CMOBJ_FORCE);
	 kcms_destroy(new_cmobj);
      }
      else
      {
	 updating = TRUE;

	 /*-- get list of files in destination, and make copy -----------*/
	 if (!kcms_get_attribute(new_cmobj, KCMS_CMOBJ_ALL_FILES, &df_attr))
	    return NULL;
	 destn_files = klist_copy(df_attr);
      }
   }

   if (!updating)
   {
      new_cmobj = kcms_create_cmobj(toolbox, cmobj->oname, cmobj->prog_type);
      if (new_cmobj == NULL)
         return NULL;
   }

   /*-- kroutine with associated library, duplicate the library? --*/
   if (cmobj->library != NULL)
   {
      if (!_kcms_duplicate_library(cmobj, toolbox, libchoice))
      {
	 /* kcms_destroy(new_cmobj); */
	 return NULL;
      }
      new_cmobj->library = kstrdup(cmobj->library);
   }

   kcms_get_attribute(cmobj->database, KCMS_FOBJ_FULLNAME_TKN, &dbToken);
   kcms_get_attribute(cmobj->makefile, KCMS_FOBJ_FULLNAME_TKN, &makeToken);

   /*-- we'll open the manpage dbm, ready to update if needs be ------*/
   ksprintf(temp, "$%s/repos/db/manpage", newtbname);
   dbm = kdbm_open(temp, (O_WRONLY | O_CREAT), 0666);

   for (; flist != NULL; flist = klist_next(flist))
   {
      token = (int)klist_identifier(flist);

      /*-- don't copy the Makefile or object database! ---------------*/
      if (token == dbToken || token == makeToken)
	 continue;
      fileobj1 = (kobject) klist_clientdata(flist);
      fpath = "(unknown)";
      if (!kcms_get_attributes(fileobj1,
                               KCMS_PATH,           &fpath,
			       KCMS_FOBJ_GENERATED, &fgen,
			       KCMS_FOBJ_TYPE,      &ftype,
			       KCMS_END)
          || (relpath = kstrchr(fpath, '/')) == NULL)
      {
         kwarn(KCMS, routine, "Could not get all attributes for file,\n"
	       "so file will not be copied with object:\n\n"
	       "\tFile: %s\n", fpath);
         continue;
      }

      /*-- don't copy if file is generated ---------------------------*/
      if (fgen == KCMS_FOBJ_GEN_LFILE)
	 continue;

      /*-- is the file actually there? -------------------------------*/
      if (kaccess(fpath, R_OK) == -1)
      {
	 kwarn(KCMS, routine,
	       "The following file does not exist in the source object:\n"
	       "\t%s\n"
	       "It will not appear in the new object.\n", fpath);
	 continue;
      }

      if (updating)
      {
         ksprintf(destpath, "$%s%s", newtbname, relpath);
         dest_token = kstring_to_token(destpath);

         /*-- does the file exist in the destination list of files? ---*/
         if (klist_locate(destn_files, (kaddr)dest_token) != NULL)
            destn_files = klist_delete(destn_files, (kaddr)dest_token);
         else
            kannounce(KCMS, routine, "adding %s", destpath);
      }

      fileobj2 = kcms_duplicate(fileobj1, new_cmobj);
      if (fileobj2 == NULL)
      {
	 int itmp = errno;
	 errno = itmp;
	 kerror(NULL, routine, "%s%s%s%s%s%s%s",
		"Cannot copy file '", fpath, "' in cmobj '",
		cmobj->oname, "'.  The copy will continue, ",
		"but please note that this file will not ",
		"exist in the copied cmobj.");
      }

      /*-- update manpage database in destination toolbox ------------*/
      if (dbm != NULL && ftype == KCMS_FOBJ_TYPE_MAN)
         _kcms_update_man_db(dbm, fileobj2, TRUE);
   }

   /*-- any other files left in destination list should be removed ---*/
   if (updating)
   {
      kcms_get_attribute(new_cmobj->database,
			 KCMS_FOBJ_FULLNAME_TKN, &dbToken);
      kcms_get_attribute(new_cmobj->makefile,
			 KCMS_FOBJ_FULLNAME_TKN, &makeToken);

      for (flist = destn_files; flist != NULL; flist = klist_next(flist))
      {
         fobj = (kobject)klist_clientdata(flist);
         token = (int)klist_identifier(flist);
         if (!kcms_get_attributes(fobj,
				 KCMS_PATH, &fpath,
				 KCMS_FOBJ_GENERATED, &fgen,
				 KCMS_END))
            continue;
   
         /*-- don't delete file if it is generated ----------------------*/
         if (token == dbToken || token == makeToken
	     || fgen == KCMS_FOBJ_GEN_LFILE)
	    continue;
   
         if ((kcms_query_bit(cmobj, KCMS_CMOBJ_FLAGS, KCMS_BIT_CMOBJ_FORCE,
			     &force_flag)
	      && force_flag)
	     || kprompt(KSTANDARD, NULL, NULL, 0,
			"\nOperation: duplicate software object\n\n"
			"\tToolbox:             %s\n"
			"\tObject:              %s\n"
			"\tDestination Toolbox: %s\n\n"
			"The destination object contains a file \"%s\",\n"
			"which does not appear in the source object.\n"
			"Do you want to remove file \"%s\" "
			"from destination object?",
			otbname, cmobj->oname, newtbname, fpath, fpath))
         {
            kannounce(KCMS, routine, "deleting %s", fpath);
            kcms_destroy(fobj);
         }
      }
   }

   if (dbm != NULL)
      kdbm_close(dbm);

   kcms_set_attribute(new_cmobj, KCMS_CMOBJ_GEN_LIBNAME, cmobj->library);
   kcms_set_bit(cmobj, KCMS_CMOBJ_FLAGS, KCMS_BIT_CMOBJ_FORCE);
   if (cmobj->bname != NULL && kstrcmp(cmobj->oname, cmobj->bname))
      kcms_set_attribute(new_cmobj, KCMS_CMOBJ_BNAME, cmobj->bname);

   /*-- propagate string attributes -----------------------------------*/
   for (i = 0; i < sizeof(string_attributes) / sizeof(int); i++)
      if (kcms_get_attribute(cmobj, string_attributes[i], &src_string_value)
	  && kcms_get_attribute(new_cmobj, string_attributes[i],
				&dest_string_value)
	  && kstrcmp(src_string_value, dest_string_value) != 0)
      {
	 kcms_set_attribute(new_cmobj, string_attributes[i], src_string_value);
      }

   /*-- propagate integer attributes ----------------------------------*/
   for (i = 0; i < sizeof(int_attributes) / sizeof(int); i++)
      if (kcms_get_attribute(cmobj, int_attributes[i], &src_int_value)
	  && kcms_get_attribute(new_cmobj, int_attributes[i], &dest_int_value)
	  && src_int_value != dest_int_value)
      {
	 kcms_set_attribute(new_cmobj, int_attributes[i], src_int_value);
      }

   if (cmobj->prog_type == KCMS_PANE)
   {
      kstring  src_tbname, src_oname, src_args;
      kstring  dest_tbname, dest_oname, dest_args;

      if (kcms_get_attribute(cmobj, KCMS_CMOBJ_GENERATE_EXECUTABLE, &genexec))
	 kcms_set_attribute(new_cmobj, KCMS_CMOBJ_GENERATE_EXECUTABLE, genexec);

      if (kcms_get_attribute(cmobj, KCMS_CMOBJ_ASSOCIATED_OBJECT,
			     &src_tbname, &src_oname, &src_args)
	  && kcms_get_attribute(new_cmobj, KCMS_CMOBJ_ASSOCIATED_OBJECT,
			     &dest_tbname, &dest_oname, &dest_args)
	  && (kstrcmp(src_tbname, dest_tbname) != 0
	      || kstrcmp(src_oname, dest_oname) != 0
	      || kstrcmp(src_args, dest_args) != 0))
	 kcms_set_attribute(new_cmobj, KCMS_CMOBJ_ASSOCIATED_OBJECT,
			    src_tbname, src_oname, src_args);
   }

   /*UPDATE: should set this via an attributes --*/
   kcms_set_bit(new_cmobj, KCMS_CMOBJ_FLAGS, KCMS_BIT_CMOBJ_FORCE);
   kcms_set_attribute(new_cmobj, KCMS_CMOBJ_UPDATE_DB,
		      (KCMS_UPDATE_SYNC | KCMS_UPDATE_REGEN |
		       KCMS_UPDATE_NEW | KCMS_UPDATE_CACHE));

   return new_cmobj;
}

/*-----------------------------------------------------------
| Routine Name:	kcms_cmobj_destroy - remove cmobj from its current toolbox
|
| Purpose:	This routine removes a software object from a toolbox.
|
| Input:	cmobj - the cmobj to remove
| Returns:	TRUE (1) on success, FALSE (0) otherwise
| Written By:	Steven Jorgensen and Neil Bowers
| Date:		Feb 07, 1993 21:43
------------------------------------------------------------*/
int
kcms_cmobj_destroy(
   kobject cmobj)
{
   kstring    routine = "Destroy Object";
   kobject    toolbox;
   kobject    libobj;
   kobject    fobj;
   kdbm      *dbm;
   kstring   *dlist = NULL;
   kstring    basename;
   kstring    tbpath;
   kstring    tbname;
   kstring    opath;
   char       temp[KLENGTH];
   char       path[KLENGTH];
   char       cwd[KLENGTH];
   char       lfile[KLENGTH];
   char       dirpath[KLENGTH];
   int        otype;
   int        count;
   int        incantata;
   klist     *clist;
   klist     *doc_list;
   kstring    oname = NULL;
   kstring    typestring;
   kdatum     key;
   int        force;
   int        i;
   kstring    tmp;


   if (kgetcwd(cwd, KLENGTH) == NULL)
   {
      kerror(KCMS, routine, "Could not determine the current directory.\n");
      return FALSE;
   }

   if (!kcms_legal_object(cmobj, routine))
      return FALSE;

   if (!kcms_get_attributes(cmobj,
			    KCMS_PATH,            &opath,
			    KCMS_PARENT,          &toolbox,
			    KCMS_NAME,            &oname,
			    KCMS_CMOBJ_TYPE,      &otype,
			    KCMS_CMOBJ_INCANTATA, &incantata,
			    KCMS_CMOBJ_ALL_DOC,   &doc_list,
			    KCMS_END)
       || !kcms_get_attributes(toolbox,
			       KCMS_NAME, &tbname,
			       KCMS_PATH, &tbpath,
			       KCMS_END))
      return FALSE;
   typestring = kcms_attr_int2string(KCMS_CMOBJ_TYPE, otype);

   if (kfullpath(opath, NULL, path) == NULL)
   {
      kerror(KCMS, routine, "Unable to resolve the full path of the "
	     "software object being deleted:\n\n"
	     "\tPath: %s\n\n"
	     "Deletion will not proceed.", opath);
      return FALSE;
   }

   if (!kstrncmp(path, cwd, kstrlen(path)))
   {
      kerror(KCMS, routine,
	     "Your current directory is within the object's "
	     "directory structure,\n"
	     "so the object cannot be removed.\n");
      return FALSE;
   }

   if (kcms_query_bit(cmobj, KCMS_CMOBJ_FLAGS, KCMS_BIT_CMOBJ_FORCE, &force)
       && !force
       && !kprompt(KSTANDARD, NULL, NULL, 0,
		"\nOperation: delete software object from toolbox\n\n"
		"\tToolbox: %s\n"
		"\tObject:  %s\n\n"
		"Do you want to continue with removal of object?",
		tbname, cmobj->oname))
   {
      kinfo(KSYSLIB, "%s: %s.\n", routine,
	    "destroy aborted at request of user");
      return TRUE;
   }

   /*-- we'll open the manpage dbm, ready to update if needs be ------*/
   if (doc_list != NULL)
   {
      ksprintf(temp, "$%s/repos/db/manpage", tbname);
      dbm = kdbm_open(temp, (O_WRONLY | O_CREAT), 0666);
      for (; doc_list != NULL; doc_list = klist_next(doc_list))
      {
	 fobj = (kobject)klist_clientdata(doc_list);
         _kcms_update_man_db(dbm, fobj, FALSE);
      }
      kdbm_close(dbm);
   }

   /*-- remove it from the objects cache? --*/
   ksprintf(path, "%s/repos/db/cache/objects", tbpath);
   if ((dbm = kdbm_open(path, (O_RDWR | O_CREAT), 0666)) == NULL)
      kwarn(KCMS, routine, "Could not open objects cache for toolbox "
	    "`%s'.\n"
	    "The objects cache should be at %s\n",
	    tbname, path);
   else
   {
      key.dptr = cmobj->oname;
      key.dsize = kstrlen(cmobj->oname) + 1;
      (void)kdbm_delete(dbm, key);
      kdbm_close(dbm);
   }

   /*-- kroutine with associated library, destroy the l*.c? --*/
   if (cmobj->library != NULL)
   {
      if (kprompt(KSTANDARD, NULL, NULL, 0,
		  "\nOperation: delete software object from toolbox\n\n"
		  "\tToolbox: %s\n"
		  "\tObject:  %s\n\n"
		  "Do you want to remove l%s.c from library (%s)?",
		  tbname, cmobj->oname, cmobj->oname, cmobj->library))
      {
	 libobj = kcms_open_cmobj(cmobj->parent, cmobj->library);
	 if (libobj == NULL)
	    kerror(KCMS, routine,
		   "Unable to open associated library"
		   "object (%s) for object %s in ",
		   "toolbox %s\n",
		   cmobj->library, cmobj->parent, tbname);
	 else
	 {
	    kcms_get_attribute(libobj, KCMS_CMOBJ_ALL_C,
			       &clist);
	    ksprintf(lfile, "l%s.c", cmobj->oname);
	    for (; clist != NULL; clist = klist_next(clist))
	    {
	       fobj = klist_clientdata(clist);
	       kcms_get_attribute(fobj, KCMS_NAME, &basename);
	       if (!kstrcasecmp(lfile, basename))
	       {
		  kcms_destroy(fobj);
		  break;
	       }
	    }

	    kcms_get_attribute(libobj, KCMS_CMOBJ_ALL_LIBMAN, &clist);
	    ksprintf(lfile, "l%s", cmobj->oname);
	    for (; clist != NULL; clist = klist_next(clist))
	    {
	       fobj = klist_clientdata(clist);
	       kcms_get_attribute(fobj, KCMS_NAME, &basename);
	       if (!kstrcasecmp(lfile, basename))
	       {
		  kcms_destroy(fobj);
		  break;
	       }
	    }

	    kcms_set_attribute(libobj, KCMS_CMOBJ_UPDATE_DB,
			       (KCMS_UPDATE_SYNC | KCMS_UPDATE_REGEN));

	    kcms_close(libobj);
	 }
      }
   }


   /*--------------------------------------------------------------------
   |	Delete any binaries associated with the object:
   |		(a) for libraries these will be archive files
   |		(b) for pane objects, this is the object name (script)
   |		(b) for other objects, the executable
   --------------------------------------------------------------------*/
   if (!kcms_get_attribute(cmobj, KCMS_CMOBJ_BNAME, &tmp))
   {
      kerror(KCMS, routine, "Operation aborted -- bname retrieve failed\n");
      return TRUE;
   }

   if (otype == KCMS_LIBRARY)
   {
      ksprintf(path, "$%s/lib", tbname);
      ksprintf(temp, "lib%s.", tmp);
      dlist = karray_dirlist(temp, path, NULL, KPATH | KFILE | KLINK,
			     FALSE, &count);
      if (count > 0 &&
	  (force ||
	   kprompt(KFORCE, NULL, NULL, 0,
		   "\nOperation: delete software object from toolbox\n\n"
		   "\tToolbox: %s\n"
		   "\tObject:  %s\n\n"
		   "Do you want to delete the library's archive files?",
		   tbname, cmobj->oname)))
      {
	 for (i = 0; i < count; i++)
	    kunlink(dlist[i]);
      }	   
      karray_free(dlist, count, NULL);
      dlist = NULL;
   }
   else if (otype == KCMS_PANE)
   {
      ksprintf(path, "$%s/bin/%s", tbname, cmobj->oname);
      if (kaccess(path, F_OK) != -1
	  && (force ||
	      kprompt(KFORCE, NULL, NULL, 0,
		      "\nOperation: delete software object from toolbox\n\n"
		      "\tTool	box: %s\n"
		      "\tObject:  %s\n\n"
		      "Do you want to delete the pane object's executable?",
		      tbname, cmobj->oname)))
      {
	 kunlink(path);
      }
   }
   else
   {
      ksprintf(path, "$%s/bin/%s", tbname, tmp);
      if (kaccess(path, F_OK) != -1
	  && (force ||
	      kprompt(KFORCE, NULL, NULL, 0,
		      "\nOperation: delete software object from toolbox\n\n"
		      "\tToolbox: %s\n"
		      "\tObject:  %s\n\n"
		      "Do you want to delete the object's executable?",
		      tbname, cmobj->oname)))
      {
	 kunlink(path);
      }
   }

   /*UPDATE------------------------------------------------------
   |    at this point we should check:
   |    a) are there any references out of the object?
   |    b) any references from other objects into us?
   +----------------------------------------------------------- */

   /*-- if it is a library, delete any public includes --*/
   if (otype == KCMS_LIBRARY)
   {
      ksprintf(path, "$%s/include/%s", tbname, oname);

      if (kaccess(path, R_OK) == 0 && !kremove_dir(path))
	 kerror(KCMS, routine,
		"Failed to remove public include directory.");
   }

   /*-- check whether object-type directory can be removed as well? --*/
   (void)ksprintf(dirpath, "$%s/objects/%s", tbname, typestring);
   if (kfullpath(dirpath, NULL, path) == NULL)
      return FALSE;

   dlist = karray_dirlist(dirpath, NULL, NULL, KDIR, FALSE, &count);
   if (count != 2 || (kstrcasecmp(dlist[1], oname) != 0 &&
		      kstrcasecmp(dlist[0], oname) != 0) ||
       !kstrncmp(path, cwd, kstrlen(path)))
      kstrcpy(dirpath, cmobj->opath);
   karray_free(dlist, count, NULL);

   if (!kremove_dir(dirpath))
      kerror(KCMS, routine, "\n"
	     "The object's directory structure could not "
	     "be completely deleted.\n"
	     "The most likely reason was that you were in "
	     "the directory you were trying to delete.\n");

   /*-- remove the program object from the toolbox's manpage dbm --*/
   ksprintf(temp, "$%s/repos/db/manpage", tbname);
   if ((dbm = kdbm_open(temp, O_WRONLY, 0666)) != NULL)
   {
      key.dptr = oname;
      key.dsize = kstrlen(key.dptr) + 1;
      (void)kdbm_delete(dbm, key);
      kdbm_close(dbm);
   }

   /*-- remake imake and make files --*/
   if (kdirname(dirpath, path) != NULL)
      kcms_dir_imakeandmake(path, tbname, 1);

   cmobj->parent->del_sub_obj(cmobj->parent, cmobj);
   kcms_cmobj_free(cmobj);
   return TRUE;
}

/*-----------------------------------------------------------
| Routine Name:	_kcms_duplicate_library - duplicate lroutine associated with
|		a kroutine
|
| Purpose:	This function is used to duplicate the lroutine associated
|		with a kroutine.  It is calle when you are duplicating a
|		kroutine, for example when copying to a new toolbox.
|		In this situation, you have to ensure the library and lroutine
|		exist in the destination toolbox as well.
|
| Input:	cmobj		- the software object being duplicated
|		toolbox	- the destination toolbox
|
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Neil Bowers
| Date:		17-nov-93
------------------------------------------------------------*/
static int
_kcms_duplicate_library(
   kobject  cmobj,
   kobject  toolbox,
   int      copy_choice)
{
   kstring   routine  = "Duplicate Associated Library";
   klist    *list     = NULL;
   kstring   tbname   = NULL;
   kstring   bname    = NULL;
   kstring   fobjpath = NULL;
   kstring   otbname  = NULL;
   kobject   oldlib;
   kobject   libobj;
   kobject   newfobj;
   kobject   fobj;
   char      curfpath[KLENGTH];
   kstring   cat;
   kstring   subcat;
   int       lang_type;
   int       result = FALSE;


   /*UPDATE: need to clean up the logic of this routine! --*/

   if (!kcms_get_attribute(toolbox, KCMS_NAME, &tbname))
      return FALSE;

   if ((oldlib = kcms_open_cmobj(cmobj->parent, cmobj->library)) == NULL)
   {
      kerror(KCMS, "Duplicate Associated Library",
	     "Could not open library `%s' in source toolbox, %s.",
	     cmobj->library, tbname);
      return FALSE;
   }

   switch (copy_choice)
   {
      case 1:
	 /*-- Create library, copy just the l*.c file over ------------*/
	 if (!kcms_get_attributes(oldlib,
				  KCMS_CMOBJ_BNAME,       &bname,
				  KCMS_CMOBJ_CATEGORY,    &cat,
				  KCMS_CMOBJ_SUBCATEGORY, &subcat,
				  KCMS_CMOBJ_LANGTYPE,    &lang_type,
				  KCMS_END))
	    return FALSE;

	 libobj = kcms_generate_cmobj(toolbox, cmobj->library,
				      KCMS_LIBRARY, bname, cat, subcat,
				      "a most excellent library",
				      NULL, lang_type, FALSE, FALSE, FALSE,
				      NULL);
	 if (libobj == NULL)
	    return FALSE;
	 break;

      case 2:
	 /*-- Copy the entire library as well -------------------------*/
	 if ((libobj = kcms_duplicate(oldlib, toolbox)) == NULL)
	 {
	    kerror(KCMS, routine, "Library copy failed :-(");
	    result = FALSE;
	 }
	 else
	 {
	    kcms_close(libobj);
	    result = TRUE;
	 }
	 kcms_close(oldlib);
	 break;

      case 3:
	 /*-- The library already exists in the destination toolbox ---*/
	 libobj = kcms_open_cmobj(toolbox, cmobj->library);
	 if (libobj == NULL)
	 {
	    kerror(KCMS, routine, "Couldn't open destination object.");
	    return FALSE;
	 }
	 break;

      default:
	 errno = KCMS_EINTERNAL;
	 kerror(KCMS, routine, "Invalid copy-library-option specified.");
	 return FALSE;
   }

   /*-- entire library should be copied, so quit at this point --------*/
   if (copy_choice == 2)
      return result;

   kcms_get_attribute(cmobj->parent, KCMS_NAME, &otbname);

   ksprintf(curfpath, "$%s/objects/library/%s/src/l%s.c",
	    otbname, cmobj->library, cmobj->oname);
   if (!kcms_get_attribute(oldlib, KCMS_CMOBJ_ALL_SRC, &list))
   {
      kerror(KCMS, routine, "Unable to get list of sources from old library.");

      /*UPDATE: should we do more on error? -- */
      return FALSE;
   }
   for (; list != NULL; list = klist_next(list))
   {
      fobj = (kobject) klist_clientdata(list);
      fobjpath = NULL;
      kcms_get_attribute(fobj, KCMS_PATH, &fobjpath);
      if (!kstrcasecmp(curfpath, fobjpath))
      {
	 newfobj = kcms_duplicate(fobj, libobj);
	 if (newfobj == NULL)
	    kerror(KCMS, routine, "newfobj failed");
      }
   }

   kcms_close(libobj);
   return TRUE;
}


/*-----------------------------------------------------------
| Routine Name:	_kcms_update_man_db - update toolbox manpage database
|
| Purpose:	This function updates the manpage database held for a toolbox.
|		It is invoked when a software object is added to, or deleted
|		from, a toolbox.
|		In the former case entries are added to the database for the
|		man pages associated with the object, and in the latter case
|		the entries are removed.
|
| Input:	database - the kdbm pointer for the toolbox manpage database.
|		file_object - a file object for a man page.
|		addref      - an integer which is TRUE if a reference
|			      should be added, and FALSE if the reference
|			      should be removed.
|
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Neil Bowers
| Date:		8-jan-1995
------------------------------------------------------------*/
static int
_kcms_update_man_db(
    kdbm     *database,
    kobject   file_object,
    int       addref)
{
   kstring   routine = "_kcms_update_man_db()";
   kstring   filepath;
   kdatum    key;
   kdatum    keydata;
   int       object_type;
   kstring   object_name;
   int       subtype;
   char      temp[KLENGTH];


   if (!kcms_get_attributes(file_object,
                            KCMS_PATH,           &filepath,
			    KCMS_FOBJ_SUBTYPE,   &subtype,
			    KCMS_END)
       || !kcms_get_indirect_attribute(file_object, KCMS_PARENT,
				       KCMS_CMOBJ_TYPE, &object_type)
       || !kcms_get_indirect_attribute(file_object, KCMS_PARENT,
				       KCMS_NAME, &object_name))
      return FALSE;

   if (addref)
   {
      keydata.dptr = kstrdup(filepath);
      keydata.dsize = kstrlen(filepath) + 1;
   }

   if (subtype == KCMS_FOBJ_SUBTYPE_MAN1)
   {
      key.dptr = kstrdup(object_name);
      key.dsize = kstrlen(object_name) + 1;
   }
   else if (subtype == KCMS_FOBJ_SUBTYPE_MAN3)
   {
      if (object_type == KCMS_KROUTINE)
         ksprintf(temp, "l%s", object_name);
      else
         /*UPDATE: check, should be lib */
      {
         int ii;

         kbasename(filepath, temp);
         ii = kstrlen(temp);
         while (ii > 0 && temp[ii] != '.')
            ii--;
         if (ii > 0 && temp[ii] == '.')
            temp[ii] = '\0';
      }
      key.dptr = kstrdup(temp);
      key.dsize = kstrlen(temp) + 1;
   }
   else
   {
      /*-- should never hit here --*/
      kerror(KCMS, routine, "Bogus file object, of type man!");

      /*UPDATE: this is dangerous! */
      return FALSE;
   }

   if (addref)
      return (kdbm_store(database, key, keydata, KDBM_REPLACE) == 0);

   return (kdbm_delete(database, key) == 0);
}
