/* BSE - Bedevilled Sound Engine
 * Copyright (C) 1998 Olaf Hoehmann and Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */
#include	"bsevalueblock.h"
#include	<string.h>



/* --- prototypes --- */
static void	bse_value_block_setup_i	(BseValueBlock  *block,
					 const gchar    *identifier,
					 guint           loop_begin,
					 guint           loop_end);


/* --- variables --- */
static GSList	*block_cache = NULL;


/* --- functions --- */
static gint
bse_value_block_matches (BseValueBlock *block,
			 const gchar   *identifier,
			 guint          n_values,
			 guint          loop_begin,
			 guint          loop_end)
{
  if (!identifier)
    {
      /* we need a block filled with zeros
       */
      return (block->identifier == NULL &&
	      block->n_values <= n_values);
    }
  else if (!block->identifier)
    {
      /* we got a block filled with zeros
       */
      return (identifier == NULL &&
	      block->n_values <= n_values);
    }

  /* we need an exact match for the values because of the trailing
   * mix_buffer portion at the end of the value block.
   */
  return (g_str_equal ((gpointer) block->identifier, (gpointer) identifier) &&
	  block->n_values == n_values &&
	  block->loop_begin == loop_begin &&
	  block->loop_end == loop_end);
}

BseValueBlock*
bse_value_block_get (const gchar   *identifier,
		     guint          n_values,
		     guint          loop_begin,
		     guint          loop_end)
{
  GSList *slist;

  if (!identifier)
    loop_begin = loop_end = 0;

  for (slist = block_cache; slist; slist = slist->next)
    if (bse_value_block_matches (slist->data, identifier, n_values, loop_begin, loop_end))
      return slist->data;

  return NULL;
}

BseValueBlock*
bse_value_block_get_closest (const gchar   *identifier,
			     guint          n_values,
			     guint          loop_begin,
			     guint          loop_end)
{
  BseValueBlock *match;
  BseValueBlock *match_n_values;
  BseValueBlock *match_loop;
  GSList *slist;

  if (!identifier)
    loop_begin = loop_end = 0;

  match = NULL;
  match_n_values = NULL;
  match_loop = NULL;
  for (slist = block_cache; slist; slist = slist->next)
    {
      BseValueBlock *block;

      block = slist->data;
      
      if ((!identifier && !block->identifier) ||
	  (identifier && block->identifier &&
	   g_str_equal ((gpointer) identifier, (gpointer) block->identifier)))
	{
	  match = block;
	  if (!n_values)
	    break;
	  if (match->n_values == n_values)
	    {
	      match_n_values = match;
	      if (match->loop_begin == loop_begin &&
		  match->loop_end == loop_end)
		{
		  match_loop = match;
		  break;
		}
	    }
	}
    }

  if (match_loop)
    return match_loop;

  if (match_n_values)
    return match_n_values;

  return match;
}

void
bse_value_block_ref (BseValueBlock  *block)
{
  g_return_if_fail (block != NULL);
  g_return_if_fail (block->ref_count > 0);
  
  block->ref_count++;
}

void
bse_value_block_unref (BseValueBlock  *block)
{
  g_return_if_fail (block != NULL);
  g_return_if_fail (block->ref_count > 0);

  block->ref_count--;

  if (!block->ref_count)
    {
      block_cache = g_slist_remove (block_cache, block);
      g_free ((gpointer) block->identifier);
      block->identifier = NULL;
      g_dataset_destroy (block);
      bse_value_block_free (block);
    }
}

BseValueBlock*
bse_value_block_alloc (guint n_values)
{
  BseValueBlock *block;

  g_return_val_if_fail (n_values > 0, NULL);

  block = g_malloc0 (sizeof (BseValueBlock) +
		     sizeof (BseSampleValue) * (n_values + BSE_VALUE_BLOCK_N_PAD_VALUES));
  block->ref_count = 0;
  block->identifier = NULL;
  block->n_values = n_values;
  block->loop_begin = 0;
  block->loop_end = 0;
  block->values = (BseSampleValue*) (((guchar*) block) + sizeof (BseValueBlock));

  return block;
}

BseValueBlock*
bse_value_block_setup (BseValueBlock  *block_mem,
		       const gchar    *identifier,
		       guint           loop_begin,
		       guint           loop_end)
{
  guint n_values;

  g_return_val_if_fail (block_mem != NULL, NULL);
  g_return_val_if_fail (block_mem->ref_count == 0, NULL);
  if (!identifier)
    {
      bse_value_block_free (block_mem);
      g_return_val_if_fail (identifier != NULL, NULL);
    }
  if (loop_begin > loop_end)
    {
      bse_value_block_free (block_mem);
      g_return_val_if_fail (loop_begin <= loop_end, NULL);
    }
  n_values = block_mem->n_values;
  if (loop_end >= n_values)
    {
      bse_value_block_free (block_mem);
      g_return_val_if_fail (loop_end < n_values, NULL);
    }
  if (bse_value_block_get (identifier, n_values, loop_begin, loop_end))
    {
      g_warning ("BseValueBlock \"%s\" already exists.", identifier);
      bse_value_block_free (block_mem);
      return NULL;
    }

  bse_value_block_setup_i (block_mem, identifier, loop_begin, loop_end);

  return block_mem;
}

static void
bse_value_block_setup_i (BseValueBlock  *block,
			 const gchar    *identifier,
			 guint           loop_begin,
			 guint           loop_end)
{
  if (block->identifier)
    g_warning ("BseValueBlock: member `identifier' of unregistered block has a value");

  block->ref_count++;
  block->identifier = g_strdup (identifier);
  block->loop_begin = loop_begin;
  block->loop_end = loop_end;

  /* fill the last portion of the value block with either 0s,
   * or with the loop contents
   */
  if (!loop_end)
    {
      register BseSampleValue *v;
      register BseSampleValue *v_max;

      /* FIXME: fading required for click elimination!
       */

      v = block->values + block->n_values;
      v_max = block->values + block->n_values + BSE_VALUE_BLOCK_N_PAD_VALUES;
      while (v < v_max)
	*(v++) = 0;
    }
  else
    {
      register BseSampleValue *v;
      register BseSampleValue *v_max;
      register BseSampleValue *v_begin;
      register BseSampleValue *v_loop;
      register BseSampleValue *v_end;

      v_begin = block->values + loop_begin;
      v_loop = v_begin;
      v_end = block->values + loop_end;
      v_max = block->values + block->n_values + BSE_VALUE_BLOCK_N_PAD_VALUES;
      for (v = block->values + block->n_values; v < v_max; v++)
	{
	  *v = *(v_loop++);
	  if (v_loop >= v_end)
	    v_loop = v_begin;
	}
    }
  
  block_cache = g_slist_prepend (block_cache, block);
}

void
bse_value_block_free (BseValueBlock  *block_mem)
{
  g_return_if_fail (block_mem != NULL);
  g_return_if_fail (block_mem->ref_count == 0);

  if (block_mem->identifier)
    g_warning ("BseValueBlock: member `identifier' of unregistered block has a value");

  g_free (block_mem);
}

BseValueBlock*
bse_value_block_zero (guint n_values)
{
  BseValueBlock *block;

  g_return_val_if_fail (n_values > 0, NULL);

  block = bse_value_block_get (NULL, n_values, 0, 0);
  if (block)
    return block;

  block = bse_value_block_alloc (n_values);
  bse_value_block_setup_i (block, NULL, 0, 0);

  return block;
}

BseValueBlock*
bse_value_block_new (const gchar    *identifier,
		     guint           n_values,
		     BseSampleValue *values,
		     guint           loop_begin,
		     guint           loop_end)
{
  BseValueBlock *block;

  g_return_val_if_fail (identifier != NULL, NULL);
  g_return_val_if_fail (n_values > 0, NULL);
  g_return_val_if_fail (values != NULL, NULL);
  g_return_val_if_fail (loop_begin <= loop_end, NULL);
  g_return_val_if_fail (loop_begin < n_values, NULL);
  
  if (bse_value_block_get (identifier, n_values, loop_begin, loop_end))
    {
      g_warning ("BseValueBlock \"%s\" already exists.", identifier);
      return NULL;
    }

  block = bse_value_block_alloc (n_values);
  g_memmove (block->values, values, sizeof (values[0]) * n_values);
  bse_value_block_setup_i (block, identifier, loop_begin, loop_end);

  return block;
}
