/* BSE - Bedevilled Sound Engine
 * Copyright (C) 1998 Olaf Hoehmann and Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */
#include	"bsestream.h"



/* --- functions --- */
void
bse_stream_ref (BseStream       *stream)
{
  g_return_if_fail (stream != NULL);
  g_return_if_fail (stream->ref_count > 0);
  
  stream->ref_count++;
}

void
bse_stream_unref (BseStream       *stream)
{
  g_return_if_fail (stream != NULL);
  g_return_if_fail (stream->ref_count > 0);
  
  stream->ref_count--;
  
  if (!stream->ref_count)
    {
      if (stream->opened)
	{
	  if (!stream->suspended)
	    bse_stream_suspend (stream);
	  
	  stream->close (stream);
	}
      
      if (stream->private)
	g_warning ("BseStream: private portion still allocated during free");

      g_dataset_destroy (stream);
      g_free (stream);
    }
}

BseErrorType
bse_stream_open (BseStream       *stream,
		 const gchar     *name,
		 gboolean         read_access,
		 gboolean         write_access)
{
  BseErrorType error;

  g_return_val_if_fail (stream != NULL, BSE_ERROR_STREAM_INTERNAL);
  g_return_val_if_fail (name != NULL, BSE_ERROR_STREAM_INTERNAL);
  g_return_val_if_fail (stream->opened == FALSE, BSE_ERROR_STREAM_INTERNAL);
  if (read_access)
    g_return_val_if_fail (stream->readable == TRUE, BSE_ERROR_STREAM_INTERNAL);
  if (write_access)
    g_return_val_if_fail (stream->writable == TRUE, BSE_ERROR_STREAM_INTERNAL);
  g_return_val_if_fail (!stream->name, BSE_ERROR_STREAM_INTERNAL);

  stream->name = g_strdup (name);
  stream->suspended = TRUE;

  error = stream->open (stream, read_access, write_access);

  if (error != BSE_ERROR_NONE)
    {
      g_free (stream->name);
      stream->name = NULL;
      stream->opened = FALSE;
    }
  else
    stream->opened = TRUE;

  return error;
}

BseErrorType
bse_stream_start (BseStream *stream)
{
  BseErrorType error;

  g_return_val_if_fail (stream != NULL, BSE_ERROR_STREAM_INTERNAL);
  g_return_val_if_fail (stream->opened == TRUE, BSE_ERROR_STREAM_INTERNAL);

  if (!stream->suspended)
    return BSE_ERROR_NONE;

  error = stream->start (stream);
  if (error != BSE_ERROR_NONE)
    stream->suspended = TRUE;
  else
    stream->suspended = FALSE;

  return error;
}

BseErrorType
bse_stream_suspend (BseStream *stream)
{
  BseErrorType error;

  g_return_val_if_fail (stream != NULL, BSE_ERROR_STREAM_INTERNAL);
  g_return_val_if_fail (stream->opened == TRUE, BSE_ERROR_STREAM_INTERNAL);

  if (stream->suspended)
    return BSE_ERROR_NONE;

  error = stream->suspend (stream);

  stream->suspended = TRUE;

  return error;
}

gboolean
bse_stream_would_block (BseStream *stream,
			guint	   n_values)
{
  gboolean would_block;

  g_return_val_if_fail (stream != NULL, BSE_ERROR_STREAM_INTERNAL);
  g_return_val_if_fail (stream->opened == TRUE, BSE_ERROR_STREAM_INTERNAL);

  if (stream->suspended)
    would_block = FALSE;
  else
    would_block = stream->would_block (stream, n_values);

  return would_block;
}

BseErrorType
bse_stream_read (BseStream	*stream,
		 guint           n_values,
		 BseSampleValue *values)
{
  BseErrorType error;

  g_return_val_if_fail (stream != NULL, BSE_ERROR_STREAM_INTERNAL);
  g_return_val_if_fail (n_values > 0, BSE_ERROR_STREAM_INTERNAL);
  g_return_val_if_fail (values != NULL, BSE_ERROR_STREAM_INTERNAL);
  g_return_val_if_fail (stream->opened == TRUE, BSE_ERROR_STREAM_INTERNAL);

  if (stream->suspended)
    return BSE_ERROR_STREAM_SUSPENDED;

  error = stream->read (stream, n_values, values);

  return error;
}

BseErrorType
bse_stream_write (BseStream	 *stream,
		  guint           n_values,
		  BseSampleValue *values)
{
  BseErrorType error;

  g_return_val_if_fail (stream != NULL, BSE_ERROR_STREAM_INTERNAL);
  g_return_val_if_fail (n_values > 0, BSE_ERROR_STREAM_INTERNAL);
  g_return_val_if_fail (values != NULL, BSE_ERROR_STREAM_INTERNAL);
  g_return_val_if_fail (stream->opened == TRUE, BSE_ERROR_STREAM_INTERNAL);

  if (stream->suspended)
    return BSE_ERROR_STREAM_SUSPENDED;

  error = stream->write (stream, n_values, values);

  return error;
}

BseErrorType
bse_stream_set_attribs (BseStream	*stream,
			BseStreamAttribMask mask,
			BseStreamAttribs *attribs)
{
  BseErrorType error;

  g_return_val_if_fail (stream != NULL, BSE_ERROR_STREAM_INTERNAL);
  g_return_val_if_fail (attribs != NULL, BSE_ERROR_STREAM_INTERNAL);
  g_return_val_if_fail (stream->opened == TRUE, BSE_ERROR_STREAM_INTERNAL);

  mask &= BSE_SA_MASK;

  if (mask)
    error = stream->set_attribs (stream, mask, attribs);
  else
    error = BSE_ERROR_NONE;

  return error;
}
