/* BSE - Bedevilled Sound Engine
 * Copyright (C) 1998 Olaf Hoehmann and Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */
#include	"bsesong.h"
#include	"bseeffects.h"
#include	"bsepattern.h"
#include	"bsemem.h"
#include	"bseprivate.h"
#include	"bsesequencer.h"
#include	"bseio.h"
#include	<time.h>


/* --- signals --- */
enum
{
  SIGNAL_CHANGED,
  SIGNAL_PATTERN_CHANGED,
  SIGNAL_INSTRUMENT_CHANGED,
  SIGNAL_NOTE_CHANGED,
  SIGNAL_LAST
};
typedef	void (*SignalChanged)		(GtkObject	*object,
					 gulong		 m_time,
					 gpointer	 func_data);
typedef	void (*SignalPatternChanged)	(GtkObject	*object,
					 BsePattern	*pattern,
					 gpointer	 func_data);
typedef	void (*SignalNoteChanged)	(GtkObject	*object,
					 BsePattern	*pattern,
					 guint		 channel,
					 guint		 row,
					 gpointer	 func_data);
static	void song_handle_signal_changed	(BseSong	*song,
					 gulong		 m_time);
static	void marshal_signal_changed	(GtkObject	*object,
					 GtkSignalFunc	 func,
					 gpointer	 func_data,
					 GtkArg		*args)
{
  register SignalChanged sfunc = (SignalChanged) func;
  (* sfunc) (object, GTK_VALUE_ULONG (args[0]), func_data);
}
static	void marshal_signal_pattern_changed	(GtkObject	*object,
						 GtkSignalFunc	 func,
						 gpointer	 func_data,
						 GtkArg		*args)
{
  register SignalPatternChanged sfunc = (SignalPatternChanged) func;
  (* sfunc) (object, GTK_VALUE_POINTER (args[0]), func_data);
}
#define	marshal_signal_instrument_changed	marshal_signal_pattern_changed
static	void marshal_signal_note_changed	(GtkObject	*object,
						 GtkSignalFunc	 func,
						 gpointer	 func_data,
						 GtkArg		*args)
{
  register SignalNoteChanged sfunc = (SignalNoteChanged) func;
  (* sfunc) (object, GTK_VALUE_POINTER (args[0]),
	     GTK_VALUE_UINT (args[1]), GTK_VALUE_UINT (args[2]), func_data);
}



/* --- prototypes --- */
static void	song_init			(BseSong	*song);
static void	song_class_init			(BseSongClass	*class);
static void	song_finalize			(GtkObject	*object);


/* --- variables --- */
static GtkObjectClass	*parent_class = NULL;
static guint		 song_signals[SIGNAL_LAST] = { 0 };
static GHashTable	*bse_song_hash_table = NULL;


/* --- functions --- */
GtkType
bse_song_get_type (void)
{
  static GtkType song_type = 0;

  if (!song_type)
    {
      GtkTypeInfo song_info =
      {
	"BseSong",
	sizeof (BseSong),
	sizeof (BseSongClass),
	(GtkClassInitFunc) song_class_init,
	(GtkObjectInitFunc) song_init,
	(GtkArgSetFunc) NULL,
	(GtkArgGetFunc) NULL,
      };

      song_type = gtk_type_unique (GTK_TYPE_OBJECT, &song_info);
      gtk_type_set_chunk_alloc (song_type, 4);
    }

  return song_type;
}

static void
song_init (BseSong	*song)
{
  /* initial refcount is 1
   */
  gtk_object_ref (GTK_OBJECT (song));
  gtk_object_sink (GTK_OBJECT (song));

  song->name = NULL;
  song->blurb = NULL;
  song->author = NULL;
  song->copyright = NULL;
  song->bse_version = NULL;
  song->creation_time = 0;
  song->modification_time = 0;
  song->last_modification_time = 0;
  song->bpm = bse_default_bpm;
  song->master_volume = BSE_DFL_MASTER_VOLUME;

  song->pattern_length = BSE_DFL_PATTERN_LENGTH;
  song->n_channels = 0;

  song->instruments = NULL;
  song->patterns = NULL;
  song->effect_processors = NULL;
  song->lfos = NULL;
  
  song->sequencer = NULL;

  song->pattern_list_length = 0;
  song->pattern_list = NULL;
}

static void
song_class_init (BseSongClass	*class)
{
  GtkObjectClass *object_class;

  object_class = (GtkObjectClass*) class;
  parent_class = gtk_type_class (GTK_TYPE_OBJECT);
  
  song_signals[SIGNAL_CHANGED] =
    gtk_signal_new ("changed",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (BseSongClass, changed),
		    marshal_signal_changed,
		    GTK_TYPE_NONE, 1,
		    GTK_TYPE_ULONG);
  song_signals[SIGNAL_PATTERN_CHANGED] =
    gtk_signal_new ("pattern-changed",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (BseSongClass, pattern_changed),
		    marshal_signal_pattern_changed,
		    GTK_TYPE_NONE, 1,
		    GTK_TYPE_POINTER);
  song_signals[SIGNAL_INSTRUMENT_CHANGED] =
    gtk_signal_new ("instrument-changed",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (BseSongClass, instrument_changed),
		    marshal_signal_instrument_changed,
		    GTK_TYPE_NONE, 1,
		    GTK_TYPE_POINTER);
  song_signals[SIGNAL_NOTE_CHANGED] =
    gtk_signal_new ("note-changed",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (BseSongClass, note_changed),
		    marshal_signal_note_changed,
		    GTK_TYPE_NONE, 3,
		    GTK_TYPE_POINTER,
		    GTK_TYPE_UINT,
		    GTK_TYPE_UINT);
  gtk_object_class_add_signals (object_class, song_signals, SIGNAL_LAST);

  object_class->finalize = song_finalize;

  class->changed = song_handle_signal_changed;
  class->pattern_changed = NULL;
  class->instrument_changed = bse_instrument_handle_changed;
  class->note_changed = NULL;
}

BseSong*
bse_song_new (const gchar    *name,
	      const gchar    *author,
	      const gchar    *copyright,
	      guint	     n_channels)
{
  register BseSong *song;
  
  g_return_val_if_fail (name != NULL, NULL);
  g_return_val_if_fail (n_channels >= BSE_MIN_CHANNELS && n_channels <= BSE_MAX_CHANNELS, NULL);
  
  if (bse_song_lookup (name))
    {
      g_warning ("song name \"%s\" already exists", name);
      return NULL;
    }
  
  song = gtk_type_new (BSE_TYPE_SONG);
  
  song->name = g_strdup (name);
  song->author = g_strdup (author);
  song->copyright = g_strdup (copyright);
  song->bse_version = g_strdup (bse_version);
  song->creation_time = time (NULL);
  song->modification_time = song->creation_time;
  song->last_modification_time = song->modification_time;
  song->bpm = bse_default_bpm;
  song->master_volume = BSE_DFL_MASTER_VOLUME;

  song->pattern_length = BSE_DFL_PATTERN_LENGTH;
  song->n_channels = n_channels;

  g_hash_table_insert (bse_song_hash_table, song->name, song);

  bse_song_changed (song);
  
  return song;
}

BseSong*
bse_song_lookup (const gchar *song_name)
{
  g_return_val_if_fail (song_name != NULL, NULL);
  
  if (!bse_song_hash_table)
    bse_song_hash_table = g_hash_table_new (g_str_hash, g_str_equal);
  
  return g_hash_table_lookup (bse_song_hash_table, (gchar*) song_name);
}

static void
song_finalize (GtkObject *object)
{
  BseSong* song;
  GList *list;
  GList *first;

  song = BSE_SONG (object);
  
  g_hash_table_remove (bse_song_hash_table, song->name);
  
  first = song->instruments;
  song->instruments = NULL;
  for (list = first; list; list = list->next)
    bse_instrument_delete (list->data);
  g_list_free (first);
  
  /* FIXME: free pattern_list */
  
  first = song->patterns;
  song->patterns = NULL;
  for (list = first; list; list = list->next)
    bse_pattern_delete (list->data);
  g_list_free (first);
  
  /* FIXME: free effect_processors, lfos */
  
  g_dataset_destroy (song);
  
  g_free (song->name);
  g_free (song->blurb);
  g_free (song->author);
  g_free (song->copyright);
  g_free (song->bse_version);

  GTK_OBJECT_CLASS (parent_class)->finalize (object);
}

void
bse_song_set_pattern_length (BseSong *song,
			     guint    pattern_length)
{
  g_return_if_fail (song != NULL);
  g_return_if_fail (song->patterns == NULL);
  g_return_if_fail (pattern_length >= BSE_MIN_PATTERN_LENGTH);
  g_return_if_fail (pattern_length <= BSE_MAX_PATTERN_LENGTH);

  song->pattern_length = pattern_length;

  bse_song_changed (song);
}

BsePattern*
bse_song_add_pattern (BseSong	     *song)
{
  BsePattern *pattern;

  g_return_val_if_fail (song != NULL, NULL);

  pattern = bse_pattern_new (song);

  song->patterns = g_list_append (song->patterns, pattern);

  bse_song_changed (song);
  
  return pattern;
}

BsePattern*
bse_song_get_pattern (BseSong		     *song,
		      guint		      pattern_guid)
{
  GList *list;

  g_return_val_if_fail (song != NULL, NULL);
  g_return_val_if_fail (pattern_guid > 0, NULL);

  list = g_list_nth (song->patterns, pattern_guid - 1);
  if (list)
    return list->data;
  return NULL;
}

BsePattern*
bse_song_get_pattern_from_list (BseSong	       *song,
				guint		pattern_index)
{
  GList *list;

  g_return_val_if_fail (song != NULL, NULL);

  /* FIXME: test implementation */

  list = g_list_nth (song->patterns, pattern_index);

  if (list)
    return list->data;

  return NULL;
}

BseInstrument*
bse_song_sample_instrument_new (BseSong		       *song,
				BseSample	       *sample)
{
  BseInstrument *instrument;
  
  g_return_val_if_fail (song != NULL, NULL);
  g_return_val_if_fail (sample != NULL, NULL);
  
  instrument = bse_instrument_type_sample_new (song, sample);

  song->instruments = g_list_append (song->instruments, instrument);

  bse_song_changed (song);
  
  return instrument;
}

BseInstrument*
bse_song_get_instrument (BseSong		*song,
			 guint			 instrument_guid)
{
  GList *list;

  g_return_val_if_fail (song != NULL, NULL);
  g_return_val_if_fail (instrument_guid > 0, NULL);

  list = g_list_nth (song->instruments, instrument_guid - 1);
  if (list)
    return list->data;
  return NULL;
}

gboolean
bse_song_is_playing (BseSong	    *song)
{
  g_return_val_if_fail (song != NULL, FALSE);

  return song->sequencer != NULL;
}

void
bse_song_set_bpm (BseSong		 *song,
		  guint			  bpm)
{
  g_return_if_fail (song != NULL);

  song->bpm = CLAMP (bpm, BSE_MIN_BPM, BSE_MAX_BPM);

  if (song->sequencer)
    bse_sequencer_recalc (song);

  bse_song_changed (song);
}

void
bse_song_set_volume (BseSong		    *song,
		     guint		     master_volume)
{
  g_return_if_fail (song != NULL);

  song->master_volume = CLAMP (master_volume, BSE_MIN_VOLUME, BSE_MAX_VOLUME);

  if (song->sequencer)
    bse_sequencer_recalc (song);

  bse_song_changed (song);
}

void
bse_song_set_creation (BseSong		      *song,
		       gulong		       c_time)
{
  g_return_if_fail (song != NULL);
  g_return_if_fail (c_time >= 631148400);

  song->creation_time = c_time;

  bse_song_set_modification (song, c_time);
}

void
bse_song_set_modification (BseSong		  *song,
			   gulong		   m_time)
{
  g_return_if_fail (song != NULL);
  g_return_if_fail (m_time >= song->creation_time);

  song->modification_time = m_time;

  gtk_signal_emit (GTK_OBJECT (song), song_signals[SIGNAL_CHANGED], m_time);
}

static void
bse_song_ht_foreach (gpointer key,
		     gpointer value,
		     gpointer user_data)
{
  GList **list_p;

  list_p = user_data;
  *list_p = g_list_prepend (*list_p, value);
}

GList*
bse_song_list_all (void)
{
  GList *list;

  list = NULL;
  g_hash_table_foreach (bse_song_hash_table, bse_song_ht_foreach, &list);

  return list;
}

void
bse_song_set_name (BseSong	  *song,
		   const gchar	  *name)
{
  g_return_if_fail (song != NULL);

  g_free (song->name);
  song->name = g_strdup (name);

  bse_song_changed (song);
}

gchar*
bse_song_get_name (BseSong	  *song)
{
  g_return_val_if_fail (song != NULL, NULL);

  return song->name;
}

void
bse_song_set_blurb (BseSong	   *song,
		    const gchar	   *blurb)
{
  g_return_if_fail (song != NULL);
  
  g_free (song->blurb);
  song->blurb = g_strdup (blurb);
  
  bse_song_changed (song);
}

gchar*
bse_song_get_blurb (BseSong	   *song)
{
  g_return_val_if_fail (song != NULL, NULL);
  
  return song->blurb;
}

void
bse_song_set_author (BseSong	    *song,
		     const gchar    *author)
{
  g_return_if_fail (song != NULL);
  
  g_free (song->author);
  song->author = g_strdup (author);
  
  bse_song_changed (song);
}

gchar*
bse_song_get_author (BseSong	    *song)
{
  g_return_val_if_fail (song != NULL, NULL);
  
  return song->author;
}

void
bse_song_set_copyright (BseSong	       *song,
			const gchar    *copyright)
{
  g_return_if_fail (song != NULL);
  
  g_free (song->copyright);
  song->copyright = g_strdup (copyright);
  
  bse_song_changed (song);
}

gchar*
bse_song_get_copyright (BseSong	       *song)
{
  g_return_val_if_fail (song != NULL, NULL);
  
  return song->copyright;
}

void
bse_song_changed (BseSong *song)
{
  gulong t;

  g_return_if_fail (song != NULL);

  t = time (NULL);

  gtk_signal_emit (GTK_OBJECT (song), song_signals[SIGNAL_CHANGED], t);
}

void
bse_song_pattern_changed (BseSong	 *song,
			  BsePattern	 *pattern)
{
  g_return_if_fail (song != NULL);
  g_return_if_fail (pattern != NULL);
  g_return_if_fail (pattern->song == song);

  gtk_signal_emit (GTK_OBJECT (song), song_signals[SIGNAL_PATTERN_CHANGED], pattern);

  bse_song_changed (song);
}

void
bse_song_instrument_changed (BseSong	    *song,
			     BseInstrument  *instrument)
{
  g_return_if_fail (song != NULL);
  g_return_if_fail (instrument != NULL);
  g_return_if_fail (instrument->song == song);

  gtk_signal_emit (GTK_OBJECT (song), song_signals[SIGNAL_INSTRUMENT_CHANGED], instrument);

  bse_song_changed (song);
}

void
bse_song_note_changed (BseSong	      *song,
		       BsePattern     *pattern,
		       guint	       channel,
		       guint	       row)
{
  g_return_if_fail (song != NULL);
  g_return_if_fail (pattern != NULL);
  g_return_if_fail (pattern->song == song);

  gtk_signal_emit (GTK_OBJECT (song), song_signals[SIGNAL_NOTE_CHANGED], pattern, channel, row);

  bse_song_changed (song);
}

static void
song_handle_signal_changed (BseSong *song,
			    gulong   m_time)
{
  song->last_modification_time = MAX (song->modification_time, m_time);
}

void
bse_song_reload_instrument_samples (BseSong                *song,
				    BseSampleLookupCB      cb_func,
				    gpointer               cb_data)
{
  GList *list;

  g_return_if_fail (song != NULL);

  for (list = song->instruments; list; list = list->next)
    {
      BseInstrument *instrument;

      instrument = list->data;
      if (instrument->type == BSE_INSTRUMENT_SAMPLE &&
	  instrument->deferred_sample_name)
	{
	  BseSample *sample;

	  sample = bse_sample_lookup (instrument->deferred_sample_name);
	  if (sample && sample != instrument->sample)
	    bse_instrument_set_sample (instrument, sample);
	  else if (!sample && cb_func)
	    {
	      BseIoData *io_data;

	      io_data = cb_func (cb_data,
				 instrument->deferred_sample_name,
				 instrument->deferred_sample_path);
	      sample = bse_sample_lookup (instrument->deferred_sample_name);
	      if (!sample)
		{
		  gchar *path;

		  if (io_data)
		    bse_io_data_destroy (io_data);
		  io_data = NULL;

		  path = bse_sample_lookup_path (instrument->deferred_sample_name);
		  if (path && !g_str_equal (path, instrument->deferred_sample_path))
		    io_data = cb_func (cb_data,
				       instrument->deferred_sample_name,
				       path);
		  sample = bse_sample_lookup (instrument->deferred_sample_name);
		}
	      if (sample && sample != instrument->sample)
		bse_instrument_set_sample (instrument, sample);
	      if (io_data)
		bse_io_data_destroy (io_data);
	    }
	  if (sample)
	    {
	      g_free (instrument->deferred_sample_name);
	      instrument->deferred_sample_name = NULL;
	      g_free (instrument->deferred_sample_path);
	      instrument->deferred_sample_path = NULL;
	    }
	}
    }
}
