/* types.h -- global data values.
 *
 ! Copyright (C) 1990-1992 by Matthew Clegg.  All Rights Reserved
 ! 
 ! OKbridge is made available as a free service to the Internet.
 ! Accordingly, the following restrictions are placed on its use:
 ! 
 ! 1.  OKbridge may not be modified in any way without the explicit 
 !     permission of Matthew Clegg.  
 ! 
 ! 2.  OKbridge may not be used in any way for commercial advantage.
 !     It may not be placed on for-profit networks or on for-profit
 !     computer systems.  It may not be bundled as part of a package
 !     or service provided by a for-profit organization.
 ! 
 ! If you have questions about restrictions on the use of OKbridge,
 ! write to mclegg@cs.ucsd.edu.
 ! 
 ! DISCLAIMER:  The user of OKbridge accepts full responsibility for any
 ! damage which may be caused by OKbridge.
 *
 * This file defines the common data structures shared by all of the
 * modules of okbridge.
 *
 */

#define TYPES_INCLUDED
 
/* The following constants define the various game modes which we may be in.
 *  
 * The normal mode of operation is CLUB mode.  However, the program
 * also supports a PRACTICE mode and a FORMAL mode.  The features of
 * these modes are as follows:
 *
 * PRACTICE mode:
 *   1. During bidding, the players who are sitting may bid.  When it
 *      is time for a bid to be made from an unoccupied seat, the program
 *      automatically supplies a bid of "PASS".
 *   2. During play, the cards are revealed to everyone, and anyone may
 *      specify the next card to be played.
 *   3. A hand may be ended with the /claim command.  Confirmation is not
 *      requested for claims.
 *
 * FORMAL mode:
 *   1. Alerts are not shown to the partner of the person making the alert.
 *   2. Talk messages are not transmitted to partner.
 *   3. The dummy is not displayed to anyone until the opening lead has
 *      been made.
 *   4. The bidding cannot be reviewed after play has begun.
 *
 * VUGRAPH mode:
 *   1. "Human" players cannot sit in any of the seats.
 *   2. The /north, /east, /south, and /west commands can take additional
 *      optional parameters which specify a command to be executed for that
 *      seat (i.e., /north name xxx) to set north's name to xxx.
 */ 

#define CLUB_PLAYING_MODE      0
#define PRACTICE_PLAYING_MODE  1
#define FORMAL_PLAYING_MODE    2
#define VUGRAPH_PLAYING_MODE   3

/* The following constants define the possible states which the program
 * can be in.  The current program state is stored in the variable
 * game_mode.  The meaning of the states are:
 *  STARTUP:  This is the initial state of the program, and it is
 *            re-entered at the beginning of each hand.  STARTUP mode
 *            is terminated by a READY message, which should only be
 *            issued when the players have taken their seats.
 *  DEAL:     In this mode, we receive the board to be played.  DEAL
 *            mode is terminated by a BEGIN message.
 *  BIDDING:  This state is entered when we are in the bidding phase.
 *  PLAYING:  This state is entered when we are in the playing phase.
 *  SCORING:  This state is entered when we are displaying the scores
 *            for this hand.
 */
 
#define STARTUP_MODE	0
#define DEALING_MODE    1
#define BIDDING_MODE	2
#define PLAYING_MODE	3
#define SCORING_MODE    4
#define SKIP_MODE       5


/* The following constants define the various reasons for a reset
 * (non-local branch to the top of the program).
 */

#define RESET_FULL        1  /* A full reset has been delivered. */
#define RESET_CONNECT     2  /* We wish to connect to a new server. */
#define RESET_SERVE       3  /* We are opening a new table. */
#define RESET_DISCONNECT  4  /* We have been disconnected from our server. */
#define RESET_LURKER      5  /* We wish to become a lurker */

/* The following constants define the locations on the screen where
 * user input may occur.
 */
 
#define PLAY_ROW	16
#define PLAY_COL	1
#define PLAY_LENGTH	30
 
#define TALK_ROW	17
#define TALK_COL	1
#define TALK_LENGTH	75

/* The following constants define the types of messages which may be
 * displayed in the talk window:
 */

#define COMMENT_PRIVATE 0
#define COMMENT_FORMAL  1
#define COMMENT_PUBLIC  2
#define COMMENT_SPEC    3
 
/* The following constants are used to identify the sources of messages.
 * The possible sources are
 *   NORTH   -- the player sitting in the north seat (always the server).
 *   EAST    -- east
 *   SOUTH   -- south
 *   WEST    -- west
 *   OBS     -- an observer who is not playing the game.
 *   SERVER  -- the person who controls the network connections and also
 *              who controls the flow of the game.
 */

#define PLAYER_NORTH    0
#define PLAYER_EAST     1
#define PLAYER_SOUTH    2
#define PLAYER_WEST     3
#define PLAYER_OBS      4
#define PLAYER_SERVER   5
#define PLAYER_TYPES    6

#define IS_OBSERVER(x)  ((x) == PLAYER_OBS)
#define IS_PLAYER(x)    ((0 <= (x)) && ((x) < PLAYER_OBS))

#define SIDE_NS         0
#define SIDE_EW         1

/* Each suit is assigned an index, and each rank is also assigned an index.
 * This implicitly defines an index for each card, which is defined
 * as 13 * SUIT + RANK.
 */

/* The following constants identify the suits.  No Trump is treated as
 * a suit because it reduces the number of special cases which must be
 * handled in the bidding and the playing.
 */
 
#define SUIT_CLUBS	0
#define SUIT_DIAMONDS	1
#define SUIT_HEARTS	2
#define SUIT_SPADES	3
#define SUIT_NOTRUMP	4
#define MAJOR(suit)     (((suit) == SUIT_HEARTS) || ((suit) == SUIT_SPADES))
#define MINOR(suit)     (((suit) == SUIT_CLUBS)  || ((suit) == SUIT_DIAMONDS))

/* The ranks of the cards. */
#define RANK_TWO	0
#define RANK_THREE	1
#define RANK_FOUR	2
#define RANK_FIVE	3
#define RANK_SIX	4
#define RANK_SEVEN	5
#define RANK_EIGHT	6
#define RANK_NINE	7
#define RANK_TEN	8
#define RANK_JACK	9
#define RANK_QUEEN	10
#define RANK_KING	11
#define RANK_ACE	12

/* The bids are indexed from 0 to 38.  The first three bids are the special
 * cases: PASS, DOUBLE and REDOUBLE.  The remaining bids are ordered as
 * they should be. 
 */
 
#define BID_PASS	0
#define BID_DOUBLE	1
#define BID_REDOUBLE	2
#define BID_SUIT	3

#define side_of(x) 	((x)%2)

#define ALERT_BY_BIDDER   0
#define ALERT_BY_PARTNER  1

/* The following definitions define a few commonly used data types: */

#define NAME_LENGTH     10
#define MESSAGE_LENGTH  120
#define BUFFER_LENGTH   100
#define FILENAME_LENGTH 60

typedef char name_buffer[NAME_LENGTH];
typedef char message_buffer[MESSAGE_LENGTH];
typedef char command_buffer[BUFFER_LENGTH];
typedef char filename_buffer[FILENAME_LENGTH];

/* We represent both a deck of cards and a hand by an array of 52 elements,
 * where the array is indexed by card indices.  That is, to each card we
 * associate a unique integer in the range 0-51, and this "card index"
 * is used to reference an array representing a deal or a hand.  In the
 * case of a deal, the array contains thirteen elements which are 0,
 * thirteen which are 1, thirteen which are 2, and thirteen which are 3.
 * Thus an element of an array refers to the holder of that particular
 * card.  We represent a hand by an array where thirteen elements are 1
 * and the rest are 0.  Thus, an element of an array is TRUE exactly when
 * that particular card is in the hand.
 */

typedef char card_type;
typedef card_type *suit_type;
typedef card_type deal [52];
typedef card_type hand [52];
typedef int  bid_list [100];
 
/* int suit_of (int card_code); */
/* Returns the suit index associated to the card with the given card_code. */
#define suit_of(cc)	((cc)/13)
 
/* int rank_of (int card_code); */
/* Returns the rank index associated to the card with the given card_code. */
#define rank_of(cc)	((cc) % 13)

/* int card_code (int suit, int rank); */
/* Returns the index of the card having the given suit and rank. */
#define card_code(suit,rank)  (13 * (suit) + (rank)) 

/* int trumpsuit_of (int bid_code); */
/* Returns the trump suit determined by a given bid. */
#define trumpsuit_of(bc)	((bc) > 2 ?  ((bc) - 3) % 5: -1)
 
/* int contract_of (int bid_code); */
/* Returns the level of the contract associated to the given bid. */
#define level_of(bc)		((bc) > 2 ?  (((bc) - 3) / 5) + 1: 0)

#define ROTATION_LENGTH 16
  /* The number of entries in the deal/vulnerabilities rotation sequence. */

#ifdef _BRIDGE_
 
	/* Character constants: */

        char *major_revision_level = "1.8";  /* In theory, all copies of the
		 			        program at a given major 
						revision level should be 
						compatible. */
        char *minor_revision_level = "K";

	char *card_names [] =  {"C2", "C3", "C4", "C5", "C6", "C7", "C8",
				"C9", "CT", "CJ", "CQ", "CK", "CA",
				"D2", "D3", "D4", "D5", "D6", "D7", "D8",
				"D9", "DT", "DJ", "DQ", "DK", "DA",
				"H2", "H3", "H4", "H5", "H6", "H7", "H8",
				"H9", "HT", "HJ", "HQ", "HK", "HA",
				"S2", "S3", "S4", "S5", "S6", "S7", "S8",
				"S9", "ST", "SJ", "SQ", "SK", "SA",
				NULL};
	char *suit_names [] =  {"C", "D", "H", "S", "NT"};
	char *rank_names [] =  {"2", "3", "4", "5", "6", "7", "8", "9",
				"T", "J", "Q", "K", "A"};
	char *bid_names  [] =  {"P" , "X" , "XX",
				"1C", "1D", "1H", "1S", "1NT",
				"2C", "2D", "2H", "2S", "2NT",
				"3C", "3D", "3H", "3S", "3NT",
				"4C", "4D", "4H", "4S", "4NT",
				"5C", "5D", "5H", "5S", "5NT",
				"6C", "6D", "6H", "6S", "6NT",
				"7C", "7D", "7H", "7S", "7NT",
				NULL};
 
	char *seat_names [] ={"NORTH", "EAST", "SOUTH", "WEST", "OBS",
			      "MOD"};
 
	/* Information regarding the positions of the players: */
	int player_next [] = {1, 2, 3, 0};
	int player_prev [] = {3, 0, 1, 2};
	int player_partner [] = {2, 3, 0, 1};

        /* The rotation of deals and vulnerabilities: */

        int dealer_list [16] =
          {PLAYER_NORTH, PLAYER_EAST, PLAYER_SOUTH, PLAYER_WEST,
	   PLAYER_NORTH, PLAYER_EAST, PLAYER_SOUTH, PLAYER_WEST,
	   PLAYER_NORTH, PLAYER_EAST, PLAYER_SOUTH, PLAYER_WEST,
           PLAYER_NORTH, PLAYER_EAST, PLAYER_SOUTH, PLAYER_WEST};

        int ns_vulnerability_list [16] =
          {0, 1, 0, 1, 1, 0, 1, 0, 0, 1, 0, 1, 1, 0, 1, 0};

        int ew_vulnerability_list [16] =
          {0, 0, 1, 1, 0, 1, 1, 0, 1, 1, 0, 0, 1, 0, 0, 1};

        char *month_names [] =
          { "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", 
            "Sept", "Oct", "Nov", "Dec"};

        char *long_suit_names [] = 
          {"CLUB", "DIAMOND", "HEART", "SPADE", "NO TRUMP"};

#else
 
	/* Character constants: */
        extern char *major_revision_level, *minor_revision_level;
      	extern char *card_names[],
		    *suit_names[],
		    *rank_names[],
		    *bid_names[],
		    *seat_names[];

	/* Information regarding the positions of the players: */
	extern int  player_next [],
		    player_prev [],
		    player_partner [];
 
        extern int dealer_list [], ns_vulnerability_list [], 
          ew_vulnerability_list [];

        extern char *month_names [], *long_suit_names [];
#endif

