/* socket.h -- routines for establishing connections over the internet.
 *
 ! Copyright (C) 1990-1992 by Matthew Clegg.  All Rights Reserved
 ! 
 ! OKbridge is made available as a free service to the Internet.
 ! Accordingly, the following restrictions are placed on its use:
 ! 
 ! 1.  OKbridge may not be modified in any way without the explicit 
 !     permission of Matthew Clegg.  
 ! 
 ! 2.  OKbridge may not be used in any way for commercial advantage.
 !     It may not be placed on for-profit networks or on for-profit
 !     computer systems.  It may not be bundled as part of a package
 !     or service provided by a for-profit organization.
 ! 
 ! If you have questions about restrictions on the use of OKbridge,
 ! write to mclegg@cs.ucsd.edu.
 ! 
 ! DISCLAIMER:  The user of OKbridge accepts full responsibility for any
 ! damage which may be caused by OKbridge.
 *
 */

#define SOCKET_INCLUDED

extern int client_init ();
/* int client_init(char *host, int portnum, int retry_limit); */
/* Attempts to establish a connection with the host identified by the
 * string 'host' at port 'portnum'.  If the connection is established,
 * then returns the socket number.  Otherwise, returns -1 and places
 * an error message in socket_error.
 *
 * This routine was adapted from a routine of the same name
 * written by Jarkko Oikarinen of the University of Oulu as part
 * of the Internet Relay Chat package.
 * 
 */

extern int open_port ();
/* int open_port(int portnum); */
/* Opens a socket port for listening.  If successful, returns the
 * number of socket which has been established.  If unsuccessful,
 * returns -1 and places an error string in socket_error.
 *
 * This routine was adapted from a routine of the same name
 * written by Jarkko Oikarinen of the University of Oulu as part
 * of the Internet Relay Chat package.
 * 
 */

extern int fd_readln ();
/* int fd_readln (int fd, char *buf, int buflen) */
/* Reads characters from the socket fd until a newline character '\n'
 * is detected.  Copies up to buflen-1 characters into buf, and
 * terminates the string with a null byte.  Returns the number of
 * bytes read.  -1 is returned if the socket is closed or if an
 * exceptional condition has occurred on the socket. 
 */

extern int fd_writeln ();
/* int fd_writeln (int fd, char *buf) */
/* Writes the contents of the buffer to the file descriptor fd, terminating
 * it by a CR-LF pair.  Returns true if the write was completed successfully,
 * and 0 otherwise (in this latter case, it may be that either not all of
 * the data was written or that an error occurred.)
 */

extern int Check_for_data ();
/* int Check_for_data (int fd) */
/* Checks the socket descriptor fd to see if any incoming data has
 * arrived.  If yes, then returns 1.  If no, then returns 0.
 * If an error, returns -1 and stores the error message in socket_error.
 */


/* Non-blocking I/O routines.
 *
 * The following routines are meant for handling non-blocking I/O.
 * Non-blocking I/O is mediated through non-blocking buffers.
 * For each file descriptor, the user should allocate on buffer for reading
 * and another for writing.  The nbb routines present an interface to
 * the user where the basic unit of data is the message, which is
 * handled as a null-terminated string internally and which is sent
 * as a CR-LF terminated string over the net.  The nbb routines use the
 * system read() and write() routines to transmit messages.
 */

typedef struct Non_blocking_buffer_struct {
  char *first;    /* The first byte of the buffer. */
  char *in;       /* The first location which is available for transferring
		     data into the buffer. */
  char *out;      /* The next location to be transferred out of the buffer. */
  char *last;     /* The last byte + 1 of the buffer. */
  char *hwater;   /* When out reaches this point, we may recopy the buffer. */
  char *search;   /* The next location which should be examined for an end
		     of message marker. */
} Non_blocking_buffer;
  
/* 
 * ERROR CODES
 */

#define NBB_NODATA   -4   /* No message is available to be received. */
#define NBB_BLOCK    -3   /* The output buffer is full, and the write */
                          /* could only be performed by blocking. */
#define NBB_EOF      -2   /* The end-of-file was read on a file descriptor. */
#define NBB_SYSERR   -1   /* The operating system reports an error. */
                          /* In this case, the system variable errno contains
			     the error code returned by the OS. */

#define nbb_is_empty(nbb)   ((nbb)->in == (nbb)->out)

extern Non_blocking_buffer *nbb_allocate ();
/* Non_blocking_buffer *nbb_allocate (int buflen); */
/* Allocates a non-blocking buffer of size buflen.  Buflen should be at
   least twice the size of the maximum amount of data we expect to have
   in the buffer at one time, since the nbb routines may block with only
   a half-full buffer. 
*/

extern void nbb_deallocate ();
/* void nbb_deallocate (Non_blocking_buffer *nbb); */
/* Deallocates the buffer nbb. */

extern int nbb_message_available ();
/* int nbb_message_available (Non_blocking_buffer *nbb); */
/* Examines the buffer nbb to see if a completed message is available.
   If so, then returns true. 
*/

extern int nbb_getln ();
/* int nbb_getln (Non_blocking_buffer *nbb, char *msgbuf, int msglen); */
/* If a message is available in the buffer nbb, then copies the (first)
   message into the output buffer msgbuf.  If the message is of length
   msglen or greater, then the extra bytes are truncated.  Returns the
   number of bytes transferred.  Zero-terminates the buffer msgbuf.
   If no message is available, then returns NBB_NODATA.
*/

extern int nbb_fill ();
/* int nbb_fill (int filedes, Non_blocking_buffer *nbb); */
/* Performs a read on the file descriptor filedes, tranferring the
   data into the buffer nbb.  Returns 0 if data was successfully
   transferred into the buffer, or a negative error code (as described
   above) otherwise.
*/

extern int nbb_readln ();
/* int nbb_readln (int filedes, Non_blocking_buffer *nbb, char *msgbuf, 
                   int msglen); */
/* Performs a read on the file descriptor filedes, transferring the data
   into the buffer nbb.  Then attempts to transfer a message from nbb
   to the message buffer msgbuf.  Returns the length of the message
   returned, or a negative error code (as described above).
*/

extern int nbb_putln ();
/* int nbb_putln (Non_blocking_buffer *nbb, char *msgbuf); */
/* Transfers the characters from the message buffer msgbuf to the
   non_blocking buffer nbb.  If successful, returns 0.  If there is
   insufficient room in the buffer, returns NBB_BLOCK.
*/

extern int nbb_writeln ();
/* int nbb_writeln (int filedes, Non_blocking_buffer *nbb, char *msgbuf); */
/* Transfers the message in msgbuf to the Non_blocking_buffer nbb.  Appends
   a CR-LF pair to the transferred message.  Then writes the message to the
   file descriptor filedes.  If successful, returns 0.  If the system
   reports an error, then returns NBB_SYSERR.  If the nbb buffer does not have
   room for the message, returns NBB_BLOCK.  In this latter case, no data is
   transferred into the nbb buffer.
*/

extern int nbb_flush ();
/* int nbb_flush (int filedes, Non_blocking_buffer *nbb); */
/* Attempts to write any data in the buffer nbb to the file descriptor.
   Returns 0 if successful or NBB_SYSERR if the system reports an error.
*/

extern void nbb_clear ();
/* void nbb_clear (Non_blocking_buffer *nbb); */
/* Clears all of the pointers in the buffer nbb so that it is in its
   initial state.
*/
