/* socket.c -- routines for establishing socket connections
 *             over the Internet.
 *
 ! Copyright (C) 1990-1992 by Matthew Clegg.  All Rights Reserved
 ! 
 ! OKbridge is made available as a free service to the Internet.
 ! Accordingly, the following restrictions are placed on its use:
 ! 
 ! 1.  OKbridge may not be modified in any way without the explicit 
 !     permission of Matthew Clegg.  
 ! 
 ! 2.  OKbridge may not be used in any way for commercial advantage.
 !     It may not be placed on for-profit networks or on for-profit
 !     computer systems.  It may not be bundled as part of a package
 !     or service provided by a for-profit organization.
 ! 
 ! If you have questions about restrictions on the use of OKbridge,
 ! write to mclegg@cs.ucsd.edu.
 ! 
 ! DISCLAIMER:  The user of OKbridge accepts full responsibility for any
 ! damage which may be caused by OKbridge.
 *
 */

#include <ctype.h>
#include <sys/errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <stdio.h>
#include <string.h>
#include <sys/time.h>
#include <signal.h>
#ifdef AIX
#include <sys/select.h>
#include <time.h>
#include <unistd.h>
#endif

#include "fds.h"
#include "socket.h"

extern int errno;
extern char *sys_errlist[];

#ifdef blahblah
#ifndef AIX
extern void sleep (), close ();
extern int read (), write ();
extern int select ();
extern int socket ();
extern connect ();
extern setsockopt ();
extern bind ();
extern listen ();
extern malloc ();
extern free ();
#endif
#endif

#ifdef VMS
#define write socket_write
#define read socket_read
#endif

#ifdef AIX
#define SYSV_READ
#endif

#ifdef HPUX
#define SYSV_READ
#endif

#ifdef VMS
#define SYSV_READ
#endif

#ifdef CONVEX
#define SYSV_READ
#endif

extern int server_mode;

char socket_error [80];
  /* An error message buffer for recording socket errors. */

int client_init(host, portnum, retry_limit)
     char *host;
     int portnum;
     int retry_limit;
/* Attempts to establish a connection with the host identified by the
 * string 'host' at port 'portnum'.  If the connection is established,
 * then returns the socket number.  Otherwise, returns -1 and places
 * an error message in socket_error.
 *
 * This routine was adapted from a routine of the same name
 * written by Jarkko Oikarinen of the University of Oulu as part
 * of the Internet Relay Chat package.
 * 
 */
{
  int sock, notrys;
  static struct hostent *hp;
  static struct sockaddr_in server;
  int server_avail;


  notrys = 0;
  while (notrys < retry_limit) {
  /* FIX:  jtrim@duorion.cair.du.edu -- 3/4/89 
     and jto@tolsun.oulu.fi -- 3/7/89 */

    sock = socket(AF_INET, SOCK_STREAM, 0);

    if (sock < 0) {
      sprintf (socket_error, "error opening socket: %s",
	       sys_errlist [errno]);
      return (-1);
    }
    server.sin_family = AF_INET;
    
    /* MY FIX -- jtrim@duorion.cair.du.edu   (2/10/89) */
    if ( isdigit(*host))
      {
	server.sin_addr.s_addr = inet_addr(host);
      }
    else
      { 
	hp = gethostbyname(host);
	if (hp == 0) {
	  sprintf(socket_error, "%s: unknown host", host);
	  return (-1);
	}
	bcopy(hp->h_addr, &server.sin_addr, hp->h_length);
      }
    server.sin_port = htons(portnum);
    /* End Fix */
    
    server_avail = connect(sock, (struct sockaddr *) &server, sizeof(server));
    if (server_avail) {
            close (sock);
	    sprintf (socket_error, "connection error: %s", 
		     sys_errlist [errno]);
	    notrys++;
	    if (errno == EINTR) return (-1);
	    if (notrys < retry_limit)
	      sleep (5);
    } else
      return(sock);
  }
  sprintf (socket_error, "CAN'T SEEM TO CONNECT TO SERVER -- GIVING UP");
  return (-1);
}

int open_port(portnum)
     int portnum;
/* Opens a socket port for listening.  If successful, returns the
 * number of socket which has been established.  If unsuccessful,
 * returns -1 and places an error string in socket_error.
 *
 * This routine was adapted from a routine of the same name
 * written by Jarkko Oikarinen of the University of Oulu as part
 * of the Internet Relay Chat package.
 * 
 */
{
  int sock, i;
  static struct sockaddr_in server;
  /* At first, open a new socket */
  sock = socket(AF_INET, SOCK_STREAM, 0);
  if (sock < 0) {
    sprintf (socket_error, "error opening socket: %s",
	     sys_errlist[errno]);
    return(-1);
  }

  i = 1;
  if(setsockopt(sock,SOL_SOCKET,SO_REUSEADDR,(char *)&i,sizeof(i)) < 0) {
    sprintf (socket_error, "error setting REUSE option on socket: %s",
	     sys_errlist[errno]);
    return(-1);
  }
 
  /* Bind a port to listen for new connections */
  server.sin_family = AF_INET;
  server.sin_addr.s_addr = INADDR_ANY;
  server.sin_port = htons(portnum);
  if (bind(sock, (struct sockaddr *) (&server), sizeof(server))) {
    sprintf (socket_error, "error binding stream socket: %s",
	     sys_errlist[errno]);
    close (sock);
    return(-1);
  }

  if(listen(sock, 3)) {
    sprintf (socket_error, "error listening on socket: %s",
	     sys_errlist[errno]);
    close (sock);
    return (-1);
  }

  return(sock);
}

int fd_readln (fd, buf, buflen)
int fd; char *buf; int buflen;
/* Reads characters from the socket fd until a newline character \n
 * is detected.  Copies up to buflen-1 characters into buf, and
 * terminates the string with a null byte.  Returns the number of
 * bytes read.  -1 is returned if the socket is closed or if an
 * exceptional condition has occurred on the socket. 
 */
{
	int buflog, readlog;
	char chbuf[2];

	buf[0] = '\0';
	readlog = read (fd, chbuf, 1);
	if (readlog == 0)
	  sprintf (socket_error, "EOF on read from socket");
	else if (readlog < 0)
	  sprintf (socket_error, "socket read error: %s",
		   sys_errlist[errno]);
	if (readlog <= 0) return (-1);
	buflog = 0;
	while ((chbuf[0] != '\015') && (chbuf[0] != '\0')) {
		if (buflog < buflen-1)
			buf[buflog++] = chbuf[0];
		readlog = read (fd, chbuf, 1);
		if (readlog <= 0) {
		  buf[0] = '\0';
		  if (readlog < 0)
		    sprintf (socket_error, "socket read error: %s",
			     sys_errlist[errno]);
		  else
		    sprintf (socket_error, "EOF on read from socket");
		  return (readlog);
		}
	}
	if (chbuf[0] != '\0')
	  readlog = read (fd, chbuf, 1);
	buf[buflog] = '\0';
	return (buflog);
}

int fd_writeln (fd, buf)
     int fd; char *buf;
/* Writes the contents of the buffer to the file descriptor fd, terminating
   it by a newline character.  If the call is successful, then returns
   the file descriptor 1.  Otherwise, returns 0.
*/
{
  char output_buf[120];
  int n = strlen (buf);
  int cc;  /* condition code. */

  if (fd <= 0)
    return (0);

  bcopy (buf, output_buf, n);
  output_buf[n++] = '\015';
  output_buf[n++] = '\012';
  cc = write (fd, output_buf, n);

  return ((cc < n)? 0: 1);
}

int Check_for_data (fd)
     int fd;
/* Checks the socket descriptor fd to see if any incoming data has
   arrived.  If yes, then returns 1.  If no, then returns 0.
   If an error, returns -1 and stores the error message in socket_error.
*/
{
  int status;                 /* return code from Select call. */
  struct fd_set wait_set;     /* A set representing the connections that
				 have been established. */
  struct timeval tm;          /* A timelimit of zero for polling for new
				 connections. */

  FD_ZERO (&wait_set);
  FD_SET (fd, &wait_set);

  tm.tv_sec = 0;
  tm.tv_usec = 0;
  status = select (FD_SETSIZE, &wait_set, (fd_set *) 0, (fd_set *) 0, &tm);

  if (status < 0)
    sprintf (socket_error, "Error in select: %s", sys_errlist[errno]);

  return (status);

}

static void nbb_recopy (nbb)
     Non_blocking_buffer *nbb;
{
  int buflen;

  if (nbb_is_empty(nbb))
    nbb->search = nbb->in = nbb->out = nbb->first;
  else if (nbb->out > nbb->hwater) {
    buflen = nbb->in - nbb->out;
    bcopy (nbb->out, nbb->first, buflen);
    nbb->search = nbb->out = nbb->first;
    nbb->in  = nbb->first + buflen;
  }
}

Non_blocking_buffer *nbb_allocate (buflen)
     int buflen;
/* Allocates a non-blocking buffer of size buflen.  Buflen should be at
   least twice the size of the maximum amount of data we expect to have
   in the buffer at one time, since the nbb routines may block with only
   a half-full buffer. 
*/
{
  Non_blocking_buffer *nbb;
  
  nbb = (Non_blocking_buffer *) malloc (sizeof(Non_blocking_buffer));
  nbb->first = (char *) malloc (buflen);
  nbb->in = nbb->out = nbb->search = nbb->first;
  nbb->last  = nbb->first + buflen;
  nbb->hwater = nbb->first + buflen/2;
  return (nbb);
}

void nbb_deallocate (nbb)
     Non_blocking_buffer *nbb;
/* Deallocates the buffer nbb. */
{
  free (nbb->first);
  free (nbb);
}

int nbb_message_available (nbb)
     Non_blocking_buffer *nbb;
/* Examines the buffer nbb to see if a completed message is available.
   If so, then returns true. 
*/
{
  register char *ch = nbb->search;

  while ((ch < nbb->in) && (*ch != '\0') && (*ch != '\012'))
    ch++;

  if (ch >= nbb->in) {
    nbb->search = nbb->in;
    return (0);
  } else {
    nbb->search = ch;
    return (1);
  }

}


int nbb_getln (nbb, msgbuf, msglen)
     Non_blocking_buffer *nbb; char *msgbuf; int msglen;
/* If a message is available in the buffer nbb, then copies the (first)
   message into the output buffer msgbuf.  If the message is of length
   msglen or greater, then the extra bytes are truncated.  Returns the
   number of bytes transferred.  Zero-terminates the buffer msgbuf.
   If no message is available, then returns NBB_NODATA.
*/
{
  register char *ch = nbb->search;
  int n;

  while ((ch < nbb->in) && (*ch != '\0') && (*ch != '\012'))
    ch++;

  if (ch >= nbb->in) {
    nbb->search = nbb->in;
    return (NBB_NODATA);
  }

  n = ch - nbb->out;
  if (n > msglen - 1)
    n = msglen - 1;

  if (n > 0) {
    bcopy (nbb->out, msgbuf, n);
    if (msgbuf[n-1] == '\015') n--;
  }

  msgbuf[n] = '\0';

  nbb->search = nbb->out = ch + 1;
  return (n);
}

int nbb_fill (filedes, nbb)
     int filedes;  Non_blocking_buffer *nbb;
/* Performs a read on the file descriptor filedes, tranferring the
   data into the buffer nbb.  Returns 0 if data was successfully
   transferred into the buffer, or a negative error code (as described
   above) otherwise.
*/
{
#ifdef SYSV_READ
  int data_avail;  /* true if data is available on the socket. */
#endif
  int n;   /* number of bytes transferred. */

  if (nbb->in != nbb->first)
    nbb_recopy (nbb);

#ifdef SYSV_READ
  data_avail = Check_for_data (filedes);
  if (data_avail < 0)
    return (NBB_SYSERR);
  else if (data_avail == 0)
    return (NBB_NODATA);
#endif

  n = read (filedes, nbb->in, nbb->last - nbb->in);

  if (n == 0)
    return (NBB_EOF);
  else if ((n < 0) && (errno == EWOULDBLOCK))
    return (NBB_NODATA);
  else if (n < 0)
    return (NBB_SYSERR);

  nbb->in += n;
  return (0);
}

int nbb_readln (filedes, nbb, msgbuf, msglen)     
     int filedes; Non_blocking_buffer *nbb; char *msgbuf; int msglen;
/* Performs a read on the file descriptor filedes, transferring the data
   into the buffer nbb.  Then attempts to transfer a message from nbb
   to the message buffer msgbuf.  Returns the length of the message
   returned, or a negative error code (as described above).
*/
{
  int len = nbb_getln (nbb, msgbuf, msglen);
    /* Length of message transferred from buffer, or -1 if no transfer.*/
  int read_status;
    /* Number of bytes transferred by read () call, or -1. */

  if (len >= 0)
    return (len);

  read_status = nbb_fill (filedes, nbb);

  if (read_status)
    return (read_status);

  len = nbb_getln (nbb, msgbuf, msglen);

  /* If we read a half a buffer or more without finding a message, then
     we give up and return an error. */
  if ((len < 0) && (nbb->in+1 == nbb->last))
    return (NBB_EOF);
  else if (len < 0)
    return (NBB_NODATA);

  return (len);
}

int nbb_putln (nbb, msgbuf)
     Non_blocking_buffer *nbb; char *msgbuf;
/* Transfers the characters from the message buffer msgbuf to the
   non_blocking buffer nbb.  If successful, returns 0.  If there is
   insufficient room in the buffer, returns NBB_BLOCK.
*/
{
  int n = strlen (msgbuf);

  if (nbb->in != nbb->first)
    nbb_recopy (nbb);

  if (nbb->in + n + 2 >= nbb->last)
    return (NBB_BLOCK);

  if (n > 0)
    bcopy (msgbuf, nbb->in, n);
  nbb->in += n;
  *(nbb->in++) = '\015';
  *(nbb->in++) = '\012';
}

int nbb_writeln (filedes, nbb, msgbuf)
     int filedes; Non_blocking_buffer *nbb; char *msgbuf;
/* Transfers the message in msgbuf to the Non_blocking_buffer nbb.  Appends
   a CR-LF pair to the transferred message.  Then writes the message to the
   file descriptor filedes.  If successful, returns 0.  If the system
   reports an error, then returns NBB_SYSERR.  If the nbb buffer does not have
   room for the message, returns NBB_BLOCK.  In this latter case, no data is
   transferred into the nbb buffer.
*/
{
  int error_code = nbb_putln (nbb, msgbuf);
  if (error_code) return error_code;
  return(nbb_flush (filedes, nbb));
}

int nbb_flush (filedes, nbb)
     int filedes; Non_blocking_buffer *nbb;
/* Attempts to write any data in the buffer nbb to the file descriptor.
   Returns 0 if successful or NBB_SYSERR if the system reports an error.
*/
{
  int n;

  if(nbb_is_empty(nbb))
    return (0);

  n = write (filedes, nbb->out, nbb->in - nbb->out);

  if ((n == -1) && (errno == EWOULDBLOCK))
    return (0);
  else if (n == -1)
    return (NBB_SYSERR);

  nbb->out += n;
  return (0);
}

void nbb_clear (nbb)
     Non_blocking_buffer *nbb;
/* Clears all of the pointers in the buffer nbb so that it is in its
   initial state.
*/
{
  nbb->in = nbb->out = nbb->search = nbb->first;
}
