/* scoring.h -- scoring functions for the bridge program.
 *
 ! Copyright (C) 1990-1992 by Matthew Clegg.  All Rights Reserved
 ! 
 ! OKbridge is made available as a free service to the Internet.
 ! Accordingly, the following restrictions are placed on its use:
 ! 
 ! 1.  OKbridge may not be modified in any way without the explicit 
 !     permission of Matthew Clegg.  
 ! 
 ! 2.  OKbridge may not be used in any way for commercial advantage.
 !     It may not be placed on for-profit networks or on for-profit
 !     computer systems.  It may not be bundled as part of a package
 !     or service provided by a for-profit organization.
 ! 
 ! If you have questions about restrictions on the use of OKbridge,
 ! write to mclegg@cs.ucsd.edu.
 ! 
 ! DISCLAIMER:  The user of OKbridge accepts full responsibility for any
 ! damage which may be caused by OKbridge.
 *
 * This file defines the functions used for computing scores.
 * We provide functions for scoring according to the rules of
 * rubber bridge as well as according to the rules of Chicago style
 * bridge.  Instead of being passed parameters, these functions
 * obtain most of their information from the global variables
 * defined in globals.h.
 *
 * I would like to thank Tom Kronmiller for supplying the code
 * for scoring according to the Chicago rules and David Morrison
 * for supplying the MIMP scoring code.
 */

#define SCORING_INCLUDED

/* The following types define the scoring modes which are available: */

/* The following scoring modes are supported by okbridge:  */

#define RUBBER_SCORING      	0
#define DUPLICATE_SCORING   	1
#define IMP_SCORING         	2
#define MP_SCORING          	3
#define VP_SCORING		4  /* not implemented */

/* The following data structures are used to represent score records: */

typedef union Score_union {
  struct { int above_line;  /* above-the-line score for this hand */
           int below_line;  /* below-the-line score for this hand */
	 } rubber;
  struct { 
    int total;    	    /* the number of points awarded according to 
			       the duplicate scoring system */
    union {
      struct {
	int imp_total;  /* number of IMPs won, totaled over all tables */
	int imp_tables; /* number of tables which played this board */
      } imp;
      struct { 
	int mp_total;   /* number of MPs won, TIMES 2 */
	int mp_tables;  /* number of tables which played this board */
      } mp;
    } competitive;      /* competitive scoring info */
  } duplicate;
} Score;

extern int MIMP_scoring_vuln [];
extern int MIMP_scoring_nonvuln [];

/*  All of the routines in this module use the same set of parameters:
 *  
 *  vul     := a boolean flag which if true indicates that the declaring
 *             side was vulnerable.
 *  level   := the level of the contract.
 *  suit    := the trump suit (or SUIT_NOTRUMP).
 *  doubled := is 0 for an undoubled contract, 1 for a doubled contract,
 *             and 2 for a redoubled contract.
 *  made    := If the contract was made, then the number of tricks made
 *             minus 6.  Otherwise, the negative of the number of tricks set.
 *  hcp     := Number of highcard points held by the declaring side.
 */


extern int Rubber_score_above ();
/* int Rubber_score_above (vul, level, suit, doubled, made); */
/* Computes the above-the-line score for the current contract, assuming
   the contract was made. */

extern int Rubber_score_below ();
/* int Rubber_score_below (vul, level, suit, doubled, made); */
/* Computes the below-the-line score for the current contract,
 * assuming the contract was made. */

extern int Rubber_score_set ();
/* int Rubber_score_set (vul, level, suit, doubled, made); */
/* Computes the penalty score for the current contract assuming that
 * the contract was set.  
 */

extern int Chicago_score_made ();
/* int Chicago_score_made (vul, level, suit, doubled, made); */
/* Computes the score for the current contract under the Chicago scoring
 * system, assuming that it was made.
 *
 * Original version by Tom Kronmiller.
 */

extern int Chicago_score_set ();
/* int Chicago_score_set (vul, level, suit, doubled, made); */
/* Computes the score for the given contract under the Chicago scoring
 * system, assuming that it was set.
 *
 * Original version by Tom Kronmiller.
 */

extern int Duplicate_score_made ();
/* int Duplicate_score_made (vul, level, suit, doubled, made); */
/* Computes the score for the given contract under the rules of
 * duplicate scoring. 
 */

extern int Duplicate_score_set ();
/* int Duplicate_score_set (vul, level, suit, doubled, made); */
/* Computes the score for the given contract under the rules of
 * duplicate scoring, assuming that it was set.
 */

int IMP_rating ();
/* Returns the number of IMPs awarded for the given score difference. 
 * The score is computed according International Match Point Scale of
 * the ACBL (1988).
 */

extern int Simulated_IMP_score_made ();
/* int Simulated_IMP_score_made (vul, level, suit, doubled, made, hcp); */
/* Computes the simulated IMP score for the  contract assuming that it 
 * was made.
 *
 * The Simulated IMP scoring system awards a score which is based on how
 * well the declaring team but offset by their number of highcard points.
 * The duplicate score for the declaring team is computed and then 
 * converted to a corresponding number of IMP points.  From this is
 * subtracted the excess number of highcard points above twenty of
 * the declaring team.  If the declaring team has fewer than 20 hcp,
 * then the deficit is added to the IMP score.
 */

extern int Simulated_IMP_score_set ();
/* int Simulated_IMP_score_set (vul, level, suit, doubled, made, hcp); */
/* Computes the simulated IMP score for the given contract assuming that 
 * it was set.
 */

extern int MIMP_score_made ();
/* int MIMP_score_made (vul, level, suit, doubled, made, hcp); */
/* Computes the MIMP score for the given contract, assuming it was made. 
 * 
 * The MIMP system is similar to spirit to the simulated IMP system,
 * in that the number of points awarded is offset by the highcard holdings
 * of the declaring side.  For a detailed description of the MIMP system,
 * see the article by Gary Greene in the May/June 1990 issue of 
 * Bridge Today.
 */

extern int MIMP_score_set ();
/* int MIMP_score_set (vul, level, suit, doubled, made, hcp); */
/* Computes the MIMP score for the given contract, assuming it was set. */


/*************************************************************************/

/* The following routines compute scores using the Score_union 
   data type as defined above:  */

extern void Compute_base_scores ();
/* void Compute_scores 
   (int score_type, int vul, int level, int suit, int doubled, int made,
    Score *declarer, Score *defender);
*/
/* Based on the score_type and information on the contract and result,
   computes the score records for the declarer and defender. 
   In the case of rubber scoring, the part_score field is not modified.
   In the case of MP and IMP scoring, the duplicate total is computed,
   but the competitive fields are set to zero.
*/

extern void Add_scores ();
/* extern void Add_scores 
   (int score_type, Score *prev_total, Score *new_result); 
*/
/* Add the scores in the new_result record to those in the 
   prev_total record.
*/

extern float Average_Imps ();
/* extern float Average_Imps (Score *imp_score); */
/* Returns a floating point representation of the average number of
   IMPs recorded in the given score record. */

extern float Percent_Matchpoints ();
/* extern float Percent_Matchpoints (Score *mp_score); */
/* Returns a floating point representation of the average number of
   matchpoints recorded in the given score record. */
