/* scoring.c -- scoring functions for the bridge program.
 *
 ! Copyright (C) 1990-1992 by Matthew Clegg.  All Rights Reserved
 ! 
 ! OKbridge is made available as a free service to the Internet.
 ! Accordingly, the following restrictions are placed on its use:
 ! 
 ! 1.  OKbridge may not be modified in any way without the explicit 
 !     permission of Matthew Clegg.  
 ! 
 ! 2.  OKbridge may not be used in any way for commercial advantage.
 !     It may not be placed on for-profit networks or on for-profit
 !     computer systems.  It may not be bundled as part of a package
 !     or service provided by a for-profit organization.
 ! 
 ! If you have questions about restrictions on the use of OKbridge,
 ! write to mclegg@cs.ucsd.edu.
 ! 
 ! DISCLAIMER:  The user of OKbridge accepts full responsibility for any
 ! damage which may be caused by OKbridge.
 *
 * This file defines the functions used for computing scores.
 * We provide functions for scoring according to the rules of
 * rubber bridge as well as according to the rules of Chicago style
 * bridge.  Instead of being passed parameters, these functions
 * obtain most of their information from the global variables
 * defined in globals.h.
 *
 * I would like to thank Tom Kronmiller for supplying the code
 * for scoring according to the Chicago rules.  Thanks Tom!
 *
   3/26/93.  Updates to Rubber scoring according to a note
     sent by Eddie Grove:
 
     From the editorial in the November Bridge World:

     3 scoring changes:

     1: Bonus of 100 for making a redoubled contract
     2: 300/trick for nonvul doubled undertrcks beyond first 3
     3: bonus for partial in unfinished game = 100

    (paraphrased)

 */
 
#include "types.h"
#include "scoring.h"
 
static int  first_trick   [] = {20, 20, 30, 30, 40};
static int  subseq_tricks [] = {20, 20, 30, 30, 30};

/* There are three major scoring tables, one for minor suit contracts,
   another for major suit contracts, and the third for no trump contracts.
   Each table is indexed by contract level, vulnerability and penalty
   level.  There are additional tables indicating the scoring for
   overtricks.
*/
   
static int minor_suit_score [7] [2] [3] = {
  { {70,  140, 230},    {70,  140, 230}},    /* 1m nv, 1m vul */
  { {90,  180, 560},    {90,  180, 760}},    /* 2m nv, 2m vul */
  { {110, 470, 640},    {110, 670, 840}},    /* 3m nv, 3m vul */
  { {130, 510, 720},    {130, 710, 920}},    /* 4m nv, 4m vul */
  { {400, 550, 800},    {600, 750, 1000}},   /* 5m nv, 5m vul */
  { {920, 1090, 1380},  {1370, 1540, 1830}}, /* 6m nv, 6m vul */
  { {1440, 1630, 1960}, {2140, 2330, 2660}}  /* 7m nv, 7m vul */
};

static int minor_suit_inc [2] [3] =
  { {20, 100, 200}, {20, 200, 400}};

static int major_suit_score [7] [2] [3] = {
  { {80, 160, 520},     {80, 160, 720}},     /* 1M nv, 1M vul */
  { {110, 470, 640},    {110, 670, 840}},    /* 2M nv, 2M vul */
  { {140, 530, 760},    {140, 730, 960}},    /* 3M nv, 3M vul */
  { {420, 590, 880},    {620, 790, 1080}},   /* 4M nv, 4M vul */
  { {450, 650, 1000},   {650, 850, 1200}},   /* 5M nv, 5M vul */
  { {980, 1210, 1620},  {1430, 1660, 2070}}, /* 6M nv, 6M vul */
  { {1510, 1770, 2240}, {2210, 2470, 2940}}  /* 7M nv, 7M vul */
};

static int major_suit_inc [2] [3] =
  { {30, 100, 200}, {30, 200, 400}};

static notrump_score [7] [2] [3] = {
  { {90, 180, 560},     {90, 180, 760}},     /* 1N nv, 1N vul */
  { {120, 490, 680},    {120, 690, 880}},    /* 2N nv, 2N vul */
  { {400, 550, 800},    {600, 750, 1000}},   /* 3N nv, 3N vul */
  { {430, 610, 920},    {630, 810, 1120}},   /* 4N nv, 4N vul */
  { {460, 670, 1040},   {660, 870, 1240}},   /* 5N nv, 5N vul */
  { {990, 1230, 1660},  {1440, 1680, 2110}}, /* 6N nv, 6N vul */
  { {1520, 1790, 2280}, {2220, 2490, 2980}}  /* 7N nv, 7N vul */
};

static int notrump_inc [2] [3] =
  { {30, 100, 200}, {30, 200, 400}};

/* The following table defines the scores which are awarded for
   defeated contracts: */

static int defeated_score [13] [2] [3] = {
  {{50,   100,  200},  {100,  200,  400}},
  {{100,  300,  600},  {200,  500, 1000}},
  {{150,  500, 1000},  {300,  800, 1600}},
  {{200,  800, 1600},  {400, 1100, 2200}},
  {{250, 1100, 2200},  {500, 1400, 2800}},
  {{300, 1400, 2800},  {600, 1700, 3400}},
  {{350, 1700, 3400},  {700, 2000, 4000}},
  {{400, 2000, 4000},  {800, 2300, 4600}},
  {{450, 2300, 4600},  {900, 2600, 5200}},
  {{500, 2600, 5200}, {1000, 2900, 5800}},
  {{550, 2900, 5800}, {1100, 3200, 6400}},
  {{600, 3200, 6400}, {1200, 3500, 7000}},
  {{650, 3500, 7000}, {1300, 3800, 7600}}
};

/*  All of the routines in this module use the same set of parameters:
 *  
 *  vul     := a boolean flag which if true indicates that the declaring
 *             side was vulnerable.
 *  level   := the level of the contract.
 *  suit    := the trump suit (or SUIT_NOTRUMP).
 *  doubled := is 0 for an undoubled contract, 1 for a doubled contract,
 *             and 2 for a redoubled contract.
 *  made    := If the contract was made, then the number of tricks made
 *             minus 6.  Otherwise, the negative of the number of tricks set.
 *  hcp     := Number of highcard points held by the declaring side.
 */

int Rubber_score_above (vul, level, suit, doubled, made)
     int vul, level, suit, doubled, made;
/* Computes the above-the-line score for the current contract, assuming
   the contract was made. */
{
  int above = 0;  /* computed above-the-line points. */

  if (doubled) {
    above = 100 * (made - level);
    if (vul)         above *= 2;
    if (doubled > 1) above *= 2;
/*  above += 50; */
    above += (doubled == 1)? 50: 100; /* updated 26/3/93. */
  } else
    above = subseq_tricks[suit] * (made - level);

  if (level == 6)
    above += vul? 750: 500;
  else if (level == 7)
    above += vul? 1500: 1000;
  return (above);
	
}

int Rubber_score_below (vul, level, suit, doubled, made)
     int vul, level, suit, doubled, made;
/* Computes the below-the-line score for the current contract,
 * assuming the contract was made. */
{
  int below = 0;  /* computed below-the-line points. */

  below  = first_trick[suit] + 
    (level - 1) * subseq_tricks[suit];
  if (doubled > 1)
    below *= 4;
  else if (doubled)
    below *= 2;
  return (below);
}


int Rubber_score_set (vul, level, suit, doubled, made)
     int vul, level, suit, doubled, made;
/* Computes the penalty score for the current contract assuming that
 * the contract was set.  
 */
{
  int penalty = 0;  /* computed penalty points. */
  int down = -made - 1;

  if (doubled) {
    if (vul) penalty = 200 + 300 * down;
    else     
      penalty = 100 + 200 * down +
	(down > 3)? 100 * (down - 3): 0;  /* updated 26/3/93. */
    if (doubled > 1) penalty *= 2;
  } else {
    if (vul) penalty = 100 + 100 * down;
    else     penalty =  50 +  50 * down;
  }
  return (penalty);
}


int Chicago_score_made (vul, level, suit, doubled, made)
     int vul, level, suit, doubled, made;
/* Computes the score for the current contract under the Chicago scoring
 * system, assuming that it was made.
 *
 * Original version by Tom Kronmiller.
 */
{
	int result = 0, perTrick;
	int extra = made - level;

	/* How much is making the bid worth? */
	perTrick = (MINOR(suit))? 20:30;
	result = perTrick * level;
	if (suit == SUIT_NOTRUMP)	result += 10;
	if (doubled > 1)                result *= 4;
	else if (doubled == 1)          result *= 2;
	  
	/* Was it a game we made? */
	if (result >= 100)		result += (!vul)? 300:500;
/*	else					result += 50; */

	/* Was it a slam we made? */
	if (level == 6)		result += (!vul)? 500:750;
	if (level == 7)		result += (!vul)? 1000:1500;

	/* Were we insulted by a double? */
	if (doubled > 1)                  result += 100;
	else if (doubled == 1)            result += 50;

	/* How much do we get for overtricks? */
	if (doubled > 1)
	        result += (extra * 100) * (vul? 4: 2);
	else if (doubled == 1)
	        result += (extra * 100) * (vul? 2: 1);
	else
		result += extra * perTrick;

	return (result);
}


int Chicago_score_set (vul, level, suit, doubled, made)
     int vul, level, suit, doubled, made;
/* Computes the score for the given contract under the Chicago scoring
 * system, assuming that it was set.
 *
 * Original version by Tom Kronmiller.
 */
{
	int result = 0;
	int down =  -made;

	if (!doubled)
	{
		result = 50 * down;
		if (vul) result *= 2;
	}
	else
	{
		switch (down)
		{
			case 1:
				result = (!vul)? 100:200;
				break;
			case 2:
				result = (!vul)? 300:500;
				break;
			case 3:
				result = (!vul)? 500:800;
				break;
			default:
				result = 500 + (300*(down-3)) 
				  + ((!vul)? 0:300);
				break;
		}
		if (doubled > 1) result *= 2;
	}
	return (result);
}

int Duplicate_score_made (vul, level, suit, doubled, made)
     int vul, level, suit, doubled, made;
/* Computes the score for the given contract under the rules of
 * duplicate scoring. 
 */
{
  int score;
  int overtricks = made - level;

  if (made == 0)
    return (0);

  switch (suit) {
  case SUIT_CLUBS:
  case SUIT_DIAMONDS:
    score = minor_suit_score[level-1][vul][doubled] +
      minor_suit_inc[vul][doubled] * overtricks;
    break;

  case SUIT_HEARTS:
  case SUIT_SPADES:
    score = major_suit_score[level-1][vul][doubled] +
      major_suit_inc[vul][doubled] * overtricks;
    break;

  case SUIT_NOTRUMP:
    score = notrump_score[level-1][vul][doubled] +
      notrump_inc[vul][doubled] * overtricks;
    break;
  }

  return (score);
}

int Duplicate_score_set (vul, level, suit, doubled, made)
     int vul, level, suit, doubled, made;
/* Computes the score for the given contract under the rules of
 * duplicate scoring, assuming that it was set.
 */
{
  return (defeated_score[-made-1][vul][doubled]);
}


static int IMP_conversion_table [] = {
    20,   50,   90,  130,  170,  220,  270,  320,  370,  430, 
   500,  600,  750,  900, 1100, 1300, 1500, 1750, 2000, 2250,
  2500, 3000, 3500, 4000, 99999};

int IMP_rating (score_diff)
     int score_diff;
/* Returns the number of IMPs awarded for the given score difference. */
{
  int imps = 0;

  if (score_diff < 0)
    return (-IMP_rating(-score_diff));

  for (imps = 0; IMP_conversion_table[imps] <= score_diff; imps++);
  return (imps + imps);
}

static int IMP_score_conversion (pt, highcard_points)
     int pt, highcard_points;
/* Computes a 'simulated IMP' score.  This score is computed by looking
   up the IMP_rating of the pt score of the declaring team, and subtracting
   from this the number of highcard points over 20.
*/
{
  return (IMP_rating(pt) - (highcard_points - 20));
}

int Simulated_IMP_score_made (vul, level, suit, doubled, made, hcp)
     int vul, level, suit, doubled, made, hcp;
/* Computes the simulated IMP score for the given contract assuming that 
 * it was made.
 */
{
	int score;

	score =
	  IMP_score_conversion
	    (Duplicate_score_made (vul, level, suit, doubled, made), hcp);

	return (score);
	
}

int Simulated_IMP_score_set (vul, level, suit, doubled, made, hcp)
     int vul, level, suit, doubled, made, hcp;
/* Computes the simulated IMP score for the given contract assuming that 
 * it was set.
 */
{
	int score;

	score =
	  IMP_score_conversion
	    (-Duplicate_score_set(vul, level, suit, doubled, made), hcp);

	return (score);
}

int MIMP_scoring_vuln [] = {
    -2100, -2100, -2100, -2100, -1850, -1650, -1500, -1400, -1100, -950,
    -800, -750, -700, -650, -600, -450, -300, -150, -100, -50,
    0, 50, 100, 150, 300, 450, 600, 650, 700, 750,
    800, 950, 1100, 1400, 1500, 1650, 1850, 2100, 2100, 2100,
    2100
};

int MIMP_scoring_nonvuln [] = {
    -1500, -1500, -1500, -1500, -1300, -1150, -1050, -950, -800, -700,
    -600, -550, -500, -450, -400, -300, -200, -150, -100, -50,
    0, 50, 100, 150, 200, 300, 400, 450, 500, 550,
    600, 700, 800, 950, 1050, 1150, 1300, 1500, 1500, 1500,
    1500
};

int MIMP_score_conversion (score, vulnerable, highcard_points)
     int score, vulnerable, highcard_points;
/* This code contributed by David Morrison. */
{
    /* If you think the code calculating the score here is magic, you
     * are *right*.
     */

    if (vulnerable) { score -= MIMP_scoring_vuln[highcard_points]; }
    else { score -= MIMP_scoring_nonvuln[highcard_points]; }

    /* If I had a good description of the algorithm, and its
     * justification, it would go here.  For now, you'll just have to
     * accept the code.
     */

    if ( (score <= 600) && (score >= -600) ) {
        score *= 2;
    } else {
	if (score > 600) {
	    score += 600;
	} else {
	    score -= 600;
	}
    }

    if (score >= 0)
      score = (score + 25) / 50;
    else
      score = (score - 25) / 50;

    return (score);

}

int MIMP_score_made (vul, level, suit, doubled, made, hcp)
     int vul, level, suit, doubled, made, hcp;
/* Computes the MIMP score for the given contract assuming that 
 * it was made.
 *
 * This code contributed by David Morrison.
 */
{
  int score;

  score =
    MIMP_score_conversion
      (Duplicate_score_made (vul, level, suit, doubled, made), vul, hcp);

  return (score);
	
}

int MIMP_score_set (vul, level, suit, doubled, made, hcp)
     int vul, level, suit, doubled, made, hcp;
/* Computes the MIMP score for the given contract assuming that 
 * it was set.
 *
 * This code contributed by David Morrison.
 */
{
  int score;

  score =
    MIMP_score_conversion
      (-Duplicate_score_set(vul, level, suit, doubled, made), vul, hcp);
  
  return (score);
}


/* The following routines compute scores using the Score_union 
   data type as defined above:  */

void Compute_base_scores (score_type, vul, level, suit, doubled, made,
			  declarer, defender)
     int score_type, vul, level, suit, doubled, made;
     Score *declarer, *defender;
/* void Compute_scores 
   (int score_type, int vul, int level, int suit, int doubled, int made,
    Score *declarer, Score *defender);
*/
/* Based on the score_type and information on the contract and result,
   computes the score records for the declarer and defender. 
   In the case of rubber scoring, the part_score field is not modified.
   In the case of MP and IMP scoring, the duplicate total is computed,
   but the competitive fields are set to zero.
*/
{
  switch (score_type) {
  case RUBBER_SCORING:
    declarer->rubber.above_line = 0;
    declarer->rubber.below_line = 0;
    defender->rubber.above_line = 0;
    defender->rubber.below_line = 0;
    if (made < 0)
      defender->rubber.above_line = 
	Rubber_score_set (vul, level, suit, doubled, made);
    else if (made > 0) {
      declarer->rubber.above_line =
	Rubber_score_above (vul, level, suit, doubled, made);
      declarer->rubber.below_line =
	Rubber_score_below (vul, level, suit, doubled, made);
    }
    break;

  case DUPLICATE_SCORING:
  case IMP_SCORING:
  case MP_SCORING:
    declarer->duplicate.total = 0;
    defender->duplicate.total = 0;
    if (made < 0)
      defender->duplicate.total =
	Duplicate_score_set (vul, level, suit, doubled, made);
    else if (made > 0)
      declarer->duplicate.total =
	Duplicate_score_made (vul, level, suit, doubled, made);

    if (score_type == IMP_SCORING) {
      declarer->duplicate.competitive.imp.imp_total = 0;
      declarer->duplicate.competitive.imp.imp_tables = 0;
      defender->duplicate.competitive.imp.imp_total = 0;
      defender->duplicate.competitive.imp.imp_tables = 0;
    } else if (score_type == MP_SCORING) {
      declarer->duplicate.competitive.mp.mp_total = 0;
      declarer->duplicate.competitive.mp.mp_tables = 0;
      defender->duplicate.competitive.mp.mp_total = 0;
      defender->duplicate.competitive.mp.mp_tables = 0;
    }
    break;
  }
}

void Add_scores (score_type, prev_total, new_result)
     int score_type; Score *prev_total, *new_result;
/* extern void Add_scores 
   (int score_type, Score *prev_total, Score *new_result); 
*/
/* Add the scores in the new_result record to those in the 
   prev_total record.
*/
{
  switch (score_type) {
  case RUBBER_SCORING:
    prev_total->rubber.above_line += new_result->rubber.above_line;
    prev_total->rubber.below_line += new_result->rubber.below_line;
    break;

  case DUPLICATE_SCORING:
    prev_total->duplicate.total += new_result->duplicate.total;
    break;

  case IMP_SCORING:
    prev_total->duplicate.total += new_result->duplicate.total;
    prev_total->duplicate.competitive.imp.imp_total +=
      new_result->duplicate.competitive.imp.imp_total;
    prev_total->duplicate.competitive.imp.imp_tables +=
      new_result->duplicate.competitive.imp.imp_tables;
    break;

  case MP_SCORING:
    prev_total->duplicate.total += new_result->duplicate.total;
    prev_total->duplicate.competitive.mp.mp_total +=
      new_result->duplicate.competitive.mp.mp_total;
    prev_total->duplicate.competitive.mp.mp_tables +=
      new_result->duplicate.competitive.mp.mp_tables;
    break;

  }
}


float Average_Imps (imp_score)
     Score *imp_score;
/* Returns a floating point representation of the average number of
   IMPs recorded in the given score record. */
{
  float imp_avg, avg;
  int   truncated_imps;

  if (imp_score->duplicate.competitive.imp.imp_tables <= 0)
    return (0.0);

  imp_avg = 0.5 * ((float) imp_score->duplicate.competitive.imp.imp_total) /
    ((float) imp_score->duplicate.competitive.imp.imp_tables);

  if (imp_avg < 0)
    truncated_imps = (int) ((-imp_avg) * 10.0 + 0.5);
  else
    truncated_imps = (int) (imp_avg * 10.0 + 0.5);

  avg = ((float) truncated_imps) / 10.0;

  return ((imp_avg < 0)? -avg: avg);
}


float Percent_Matchpoints (mp_score)
     Score *mp_score;
/* Returns a floating point representation of the average number of
   matchpoints recorded in the given score record. */
{
  float mp_pct;

  if (mp_score->duplicate.competitive.mp.mp_tables <= 0)
    return (0.0);

  mp_pct = 0.5 * ((float) mp_score->duplicate.competitive.mp.mp_total) /
    ((float) mp_score->duplicate.competitive.mp.mp_tables);
  
  return (mp_pct);
}

