/* command.h -- Data structures and procedures used for communicating
 *              between the client and server.
 *
 ! Copyright (C) 1990-1992 by Matthew Clegg.  All Rights Reserved
 ! 
 ! OKbridge is made available as a free service to the Internet.
 ! Accordingly, the following restrictions are placed on its use:
 ! 
 ! 1.  OKbridge may not be modified in any way without the explicit 
 !     permission of Matthew Clegg.  
 ! 
 ! 2.  OKbridge may not be used in any way for commercial advantage.
 !     It may not be placed on for-profit networks or on for-profit
 !     computer systems.  It may not be bundled as part of a package
 !     or service provided by a for-profit organization.
 ! 
 ! If you have questions about restrictions on the use of OKbridge,
 ! write to mclegg@cs.ucsd.edu.
 ! 
 ! DISCLAIMER:  The user of OKbridge accepts full responsibility for any
 ! damage which may be caused by OKbridge.
 *
 * This module defines the data structures and procedures used for
 * communicating with between the client and server.  When a command
 * is received from a remote machine, this module unpacks it into
 * a data structure representing the command.  When we wish to send
 * a command to a remote machine, we invoke a procedure which is
 * tailored for that command. 
 */

#ifndef SCORING_INCLUDED
#include "scoring.h"
#endif

#define PROTOCOL_INCLUDED

enum
{
  CMD_ERROR = 0,
  CMD_ACK,
  CMD_ALERT,
  CMD_BEGIN,
  CMD_BID,
  CMD_BOARD,
  CMD_CC,
  CMD_CLAIM,
  CMD_CLAIMREQ,
  CMD_CLAIMRESP,
  CMD_COMMENT,
  CMD_CONNERR,
  CMD_DEAL,
  CMD_ECHO,
  CMD_EMAIL,
  CMD_END,
  CMD_FULLNAME,
  CMD_HELLO,
  CMD_MODE,
  CMD_NAME,
  CMD_PING,
  CMD_PLAY,
  CMD_PLAYREQ,
  CMD_QUIT,
  CMD_RECONNECT,
  CMD_RECORD,
  CMD_RESET,
  CMD_SCORE,
  CMD_SEAT,
  CMD_SEATERR,
  CMD_SEATPOS,
  CMD_SEATREQ,
  CMD_SERVEREQ,
  CMD_SKIP,
  CMD_SKIPACK,
  CMD_SPEC,
  CMD_TABLE,
  CMD_TABLEREQ,
  CMD_TALK,
  CMD_USEREC,
  CMD_WAKEUP,
  CMD_WHO,
  CMD_WHOIS,
  CMD_WHORESP,

  CMD_MAX
};

#define TALK_RCPT_LHO  0
#define TALK_RCPT_RHO  1
#define TALK_RCPT_OPPS 2
#define TALK_RCPT_SPEC 3
#define TALK_RCPT_ALL  4

#define DUMMY  int

typedef struct Connection_struct *Conn;
typedef struct Board_struct *board_ptr;
typedef struct Play_record_struct *play_ptr;
typedef struct Table_struct *Table_ptr;

typedef struct player_command_struct {
  int command;
  int player_no;
  name_buffer    player_name;
  command_buffer command_text;
  union {
    struct { message_buffer message;    }  error;
    struct { name_buffer   player_name;
	     int           position; }     ack;
    struct { int seq_no, who;        }     alert;
    DUMMY                                  begin;
    struct { int level, alert, seq_no; }   bid;
    struct { name_buffer   board_name;
	     int           board_no; 
	     board_ptr     record; }       board;
    message_buffer                         cc;
    struct { int no_tricks; }              claim;
    struct { int no_tricks; }              claimreq;
    int                                    claimresp;
    message_buffer                         comment;
    message_buffer                         connerr;
    DUMMY                                  deal;
    struct { name_buffer ping_source; }    echo;
    struct { message_buffer addr; }        email;
    DUMMY                                  end;
    struct { message_buffer  name; }       fullname;
    struct { name_buffer  version;
	     name_buffer  player_name; 
	     int          player_id; }     hello;
    int                                    mode;
    struct { name_buffer  new_name; 
	     int          new_id;   }      name;
    DUMMY                                  ping;
    struct { int card, seq_no; }           play;
    struct { int  card;
             int  play_no; }               playreq;
    DUMMY                                  quit;
    struct {name_buffer ip;
            int         port;
            int         seat; }            reconnect;
    struct { name_buffer board_name;
	     int         board_no;
	     play_ptr    play; }           record;
    DUMMY                                  reset;
    struct { int score_type; 
             Score ns, ew; }               score;
    struct { name_buffer player_name;
             int         old_pos; 
             int         new_pos; }        seat;
    struct { int free_seats[3]; }          seaterr;
    int                                    seatpos;
    int                                    seatreq;
    struct { message_buffer command; }     servereq;
    DUMMY                                  skip;
    DUMMY                                  skipack;
    DUMMY                                  spec;
    int                                    table;
    int                                    tablereq;
    struct { int recipients;
             message_buffer  message; }    talk;
    struct { name_buffer north;
	     name_buffer east;
	     name_buffer south;
	     name_buffer west; }           userec;
    struct { name_buffer recipient; }      wakeup;
    DUMMY                                  who;
    struct { name_buffer name; }           whois;
    struct { name_buffer recipient;
	     message_buffer message; }     whoresp;
  } data;
} *player_command;


/* The following routines define the interface for sending messages
 * to remote processes. */

void Send_ack ();
void Send_alert ();
void Send_begin ();
void Send_bid ();
void Send_board ();
void Send_cc ();
void Send_claim ();
void Send_claimreq ();
void Send_claimresp ();
void Send_comment ();
void Send_deal ();
void Send_echo ();
void Send_email ();
void Send_end ();
void Send_fullname ();
void Send_hello ();
void Send_mode ();
void Send_name ();
void Send_ping ();
void Send_play ();
void Send_playreq ();
void Send_quit ();
void Send_reconnect ();
void Send_record ();
void Send_registry ();
void Send_reset ();
void Send_seat  ();
void Send_seaterr ();
void Send_seatreq ();
void Send_score ();
void Send_servereq ();
void Send_skip ();
void Send_skipack ();
void Send_spec ();
void Send_tablereq ();
void Send_table ();
void Send_talk ();
void Send_userec ();
void Send_wakeup ();
void Send_who ();
void Send_whois ();
void Send_whoresp ();

/* The following two procedures are for parsing messages which have 
 * received from remote hosts.  The difference is that in the first
 * procedure, we assume that the identifier of the sender is prefixed
 * to the message.  Both procedures assume that the input buffer is
 * stored in the player command record which is passed to the procedure.
 * Both return 0 if no errors were encountered in parsing the message
 * and -1 if an error was encountered.
 */

int Parse_Command_for_Client ();
int Parse_Command_for_Server ();


/* DESCRIPTION OF PROTOCOL.

All messages transmitted by okbridge are ASCII strings terminated by
the characters "\015\012" (^M^J).  Each message is interpreted and
converted into a data structure with the help of the parser module.
Each message has a header which identifies the origin of the message.
This is followed by a keyword which identifies the message type.
Following the keyword, there may be additional type-specific
parameters.  The fields in a message are separated by blanks.

When a client sends a message to the server, it is of the form
  <client-name> <message-type> <parameters>,
where
  <client-name>  is the 8-character nickname of the client,
  <message-type> is a keyword identifying the type of the message,
  <parameters>   is an optional list of message-specific parameters.

When a server sends a message to a client, it is of the form
  <position> <source> <message-type> <parameters>
where
  <position>     is the name of the position occupied by the person
                 who originated the message.  May be North, South,
                 East, West, Mod or Obs.
  <source>       is the nickname of the person who originated the message.
  <message-type> is a keyword identifying the type of the message.
  <parameters>   is an optional list of message-specific parameters.

To see the sequence of messages transmitted by the program, compile
with the -DLOGFILE option.

There are three kinds of messages: protocol messages, conversational
messages and game messages.  A protocol message affects the
communications behavior of the program, and is typically handled
differently according to whether we are operating in client mode or in
server mode.  For example, the initial handshakes which are performed
when a new client connects to the server are protocol messages.
Conversational messages may affect the state of the game, but not
directly.  Examples of asynchronous messages are talk messages and
claim requests.  Game messages convey changes to the state of the
game.  Examples of game messages are deals, bids and plays.

All messages are first received by the network module.  This module
then parses the message, creating a data structure as defined in
protocol.h to represent the message.  The message is then put onto the
protocol queue associated to the table.  The protocol queue is handled
by the cs module.  After a message from the protocol queue is handled,
it may be put onto the conversational queue.  The conversational queue
is handled by the conversation module.  After a message from the
conversational queue is handled, it may finally be put onto the game
queue.  The game queue is handled by the bridge module.


Protocol Messages
-------- --------

   HELLO <version> <player-name> <position>
     When a client first establishes a connection to the server, it sends
     a hello message identifying itself to the server.  If the server
     recognizes the client as using a compatible version of okbridge,
     then the HELLO message is relayed to the other clients.
     <version>     := a string identifying the major revision level
     <player-name> := an eight character name identifying the player
     <position>    := the seat requested by the player

   ACK <player-name> <position>
     After verifying that the client is using the same version of okbridge,
     the server sends to the client an ACK message for itself and for each
     of the other clients, identifying all of the current connections.
     <player-name>  := an eight character name identifying the player
     <position>     := the current position of the player.

   CONNERR <message>
     Used by the server to transmit error messages in case the client
     is not using the same version of okbridge.  The message is followed
     by closing the socket connection.

   ECHO <ping-source>
     A message sent in response to a PING request.

   PING
     Sends a message which is automatically echoed by each of the other
     players.  The purpose of this message is to determine how quickly
     the networks are operating.

   QUIT
     Transmitted by a player as the final message before closing the
     socket connection.

   RECONNECT <ip-name-or-number> <port-number> <seat-name>
     Transmitted by the server to a player, indicating that the player
     should reconnect to a new server at the given ip/port and sit
     at the specified seat.

   SEAT <old-seat> <new-seat> <player-name>
     Transmitted by the server to indicate that a player has changed seats.
     If this message names the local player, it should be ignored.
     Otherwise, problems may arise when two people choose the same name.

   SEATERR <seat1> <seat2> <seat3>
     Transmitted by the server to indicate that the requested seat
     is not available.  The seats <seat1>, <seat2>, <seat3> list the
     currently free seats.

   SEATPOS <seat-assignment>
     Indicates the seat which is assigned to the local player.
     This is a mandatory assignment, and may not necessarily be the
     seat which was requested.  Note that when a player makes a
     seat request, he will receive a SEATPOS assignment and also
     a SEAT message which is broadcast to everyone.

   SEATREQ <seat-name>
     Requests a seat assignment from the server.

   SERVEREQ <command-name>
     Makes a special request to the server which is not covered by
     the other message types.  The server is free to the <command-name>
     in an implementation-specific way.

   TABLEREQ <requested-table-number>
     Transmitted by a player to make a request to sit at the requested
     table.  

   TABLE <assigned-table-number>
     Transmitted by the server to the table number assigned to the local
     player.

   
Conversational Messages
-------------- --------

   COMMENT <message>
     Carries a message to each of the players which is from the MODERATOR.

   CC <convention-card>
     Conveys the convention card which is in use by the team of the person
     sending the message.
     
   CLAIMREQ n
     Transmits a request by the declarer to claim n additional tricks.
     The defenders should each respond with a CLAIMRESP message.

   CLAIMRESP YES|NO
     Transmits a defender's response to a claim request message.

   EMAIL <email-address>
     Transmits a player's email address to the server.

   FULLNAME <full-name>
     Transmits a players full name to the server.

   NAME <new-name>
     Informs the other players that the local player has changed his
     or her name.

   REGISTRY <identification>
     Registers a unique identity for the player with the server.
     This is independent of the name supplied with the NAME command,
     which may change.
   
   RESET
     Resets the program to its initial state.  Can only be used by server.

   SPEC
     Sent by a player to indicate that s/he has entered spectator mode.

   TALK <recipients> <message>
     Sends a message to the <recipients> from a player.

   WAKEUP [<player-name>|ALL]
     Sends a wake-up signal.  If <player-name> is specified, then
     only that player receives the signal.  Otherwise, everyone receives
     the signal.

   WHO
     Asks the server to generate a brief list of the people who are
     currently connected.

   WHOIS <player-name>
     Asks the player with name <player-name> to identify him/herself.
     The identification should consist of the player's full name and
     email address.

   WHORESP <recipient> <ident>
     A response to the WHOIS message, <recipient> is the name of the
     player who issued the WHOIS message and <ident> is the identifying
     message returned by the target of the WHOIS message.


 
Game Messages
---- --------

   ALERT <formal-mode>
     Alerts the most recent bid by the partner of the transmitting player.
     <formal-mode> is a boolean flag indicating whether or not the
     bid should be displayed to partner as well.

   BEGIN
     Sent by the server to announce that bidding should begin.

   BID <bid-name> [ALERT]
     Reports a bid to the other players.  The keyword ALERT appears
     if the bidder has alerted his own bid.

   BOARD <board-name> <sequence-number>
     Initiates the transmission of a board for play.  The board name
     and number are recorded along with the board itself in the local
     database of boards.  The board itself is then transmitted as
     in-line data following the message.  This board is then made the
     Local_board, so it will be used in subsequent play.
     Any subsequent RECORD messages apply to this board.

   CLAIM n
     Claims n of the remaining tricks.  Can only be used by the declarer
     when it is his turn to play.  

   DEAL
     Sent by the server to announce the beginning of a new hand.
     This message will be followed by a BOARD message plus one or more
     RECORD messages from the server.  Bidding will be allowed to
     commence when the server sends a BEGIN command.

   END
     Sent by the server at the end of each hand to cause each client
     to re-enter the initial mode.

   MODE CLUB|PRACTICE|FORMAL
     Sets the playing mode.  Club mode is the usual mode of play.
     Practice mode is used by two or more people to practice bidding
     and planning play.  In practice mode, the following behaviors occur:
       1. During bidding, the players who are sitting may bid.  When it
          is time for a bid to be made from an unoccupied seat, the program
          automatically supplies a bid of "PASS".
       2. During play, the cards are revealed to everyone, and anyone may
          specify the next card to be played.
       3. A hand may be ended with the /claim command.  Confirmation is not
          requested for claims.
     In formal mode, the following actions are taken:
       1. Alerts are not shown to the partner of the person making the alert.
       2. Talk messages are not transmitted to partner.
       3. The dummy is not displayed to anyone until the opening lead has
          been made.
       4. The bidding cannot be reviewed after play has begun.

   PLAY <card-name>
     Reports a play of a card to the other players.

   PLAYREQ <card-name> <sequence-number>
     Requests from the server that the given card be played for the
     given play number.  The PLAYREQ message is needed only during
     practice mode, when multiple players may attempt to play 
     simultaneously.

   RECORD <board-name> <sequence-number>
     Initiates the transmission of a record of play.  The record itself
     is transmitted as in-line data following the message.  This record
     pertains to the board having the given name and number.  If this is
     the name of the current board.

   SCORE <above-ns> <above-ew> <below-ns> <below-ew>
     A message sent by the server to indicate the current scoring totals.
     This may be independent of the scores which have been computed for
     the hands that have been played.

   SKIP
     Ends the current hand prematurely, causing all players to return
     to STARTUP_MODE.  Does not record a score for the hand.

   SKIPACK
     Sent by each player in response to a skip message, indicating that
     the local player has reset his state to startup mode.

   USEREC <north-name> <east-name> <south-name> <west-name>
     A message sent by the server to indicate which play record should
     is being used in the current hand.
*/
