/* command.c -- Command processor for communications between client
 * and server.
 *
 ! Copyright (C) 1990-1992 by Matthew Clegg.  All Rights Reserved
 ! 
 ! OKbridge is made available as a free service to the Internet.
 ! Accordingly, the following restrictions are placed on its use:
 ! 
 ! 1.  OKbridge may not be modified in any way without the explicit 
 !     permission of Matthew Clegg.  
 ! 
 ! 2.  OKbridge may not be used in any way for commercial advantage.
 !     It may not be placed on for-profit networks or on for-profit
 !     computer systems.  It may not be bundled as part of a package
 !     or service provided by a for-profit organization.
 ! 
 ! If you have questions about restrictions on the use of OKbridge,
 ! write to mclegg@cs.ucsd.edu.
 ! 
 ! DISCLAIMER:  The user of OKbridge accepts full responsibility for any
 ! damage which may be caused by OKbridge.
 *
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#define PROTOCOL
#include "types.h"
#include "protocol.h"
#include "state.h"
#include "parser.h"

#ifdef GCC
extern sscanf ();
#endif

extern float Average_Imps (), Percent_Matchpoints ();

void 
  Parse_ack (), 
  Parse_alert (), 
  Parse_begin (), 
  Parse_bid (), 
  Parse_board (),
  Parse_claim (),
  Parse_claimreq (),
  Parse_claimresp (),
  Parse_cc (), 
  Parse_comment (), 
  Parse_connerr (), 
  Parse_deal (),
  Parse_echo (), 
  Parse_email (),
  Parse_end (), 
  Parse_fullname (),
  Parse_hello (), 
  Parse_mode (),
  Parse_name (), 
  Parse_ping (), 
  Parse_play (), 
  Parse_playreq (),
  Parse_quit (), 
  Parse_reconnect (),
  Parse_record (),
  Parse_reset (),
  Parse_seat (), 
  Parse_seaterr (), 
  Parse_seatpos (), 
  Parse_seatreq (),
  Parse_servereq (),
  Parse_score (), 
  Parse_skip (), 
  Parse_skipack (),
  Parse_spec (),
  Parse_table (),
  Parse_tablereq (),
  Parse_talk (), 
  Parse_userec (),
  Parse_wakeup (),
  Parse_who (), 
  Parse_whois (), 
  Parse_whoresp ();

static Command_Descriptor Commands [] = {
  {"ACK",    {{F_NAME, "<player-name>"}, 
	      {F_KEYWORD | F_OPTIONAL, PARSER_SEATS}, NOP},  Parse_ack},
  {"ALERT",  {{F_KEYWORD, "BIDDER|PARTNER"}, {F_INT, "<sequence-number>"},NOP},
     Parse_alert},
  {"BEGIN",  C_EMPTY, Parse_begin},
  {"BID",    {{F_BID, "<bid-name>"}, {F_INT, "<sequence-number>"},
		{F_KEYWORD | F_OPTIONAL, "ALERT"}}, Parse_bid},
  {"BOARD",  {{F_NAME, "<board-name>"}, {F_INT, "<board-number>"}, NOP},
              Parse_board},

  {"CC",     C_PARAM(F_STRING, "<convention-card>"), Parse_cc},
  {"CLAIM",  C_INT("<no-tricks>"), Parse_claim},
  {"CLAIMREQ",  C_INT("<no-tricks>"), Parse_claimreq},
  {"CLAIMRESP", C_KEY("REJECT|ACCEPT"), Parse_claimresp},
  {"COMMENT",C_STRING("<message>"), Parse_comment},

  {"CONNERR",C_STRING("<error-message>"), Parse_connerr},
  {"DEAL",   C_EMPTY, Parse_deal},
  {"ECHO",   C_PARAM(F_NAME, "<ping-source>"), Parse_echo},
  {"EMAIL",  C_STRING("<email-address>"), Parse_email},
  {"END",    C_EMPTY, Parse_end},

  {"FULLNAME", C_STRING("<full-name>"), Parse_fullname},
  {"HELLO",  {{F_NAME, "<version>"}, {F_NAME, "<player-name>"}, 
	      {F_INT, "<player-id>"}}, Parse_hello},
  {"MODE",   C_KEY("CLUB|PRACTICE|FORMAL|VUGRAPH"), Parse_mode},
  {"NAME",   {{F_NAME, "<new-player-name>"}, {F_INT, "<player-id>"}, NOP},
     Parse_name},
  {"PING",   C_EMPTY, Parse_ping},

  {"PLAY",   {{F_CARD, "<card-name>"}, {F_INT, "<sequence-number>"}, NOP},
     Parse_play},
  {"PLAYREQ", {{F_CARD, "<card-name>"}, {F_INT, "<sequence-number>"}, NOP}, 
     Parse_playreq},
  {"QUIT",   C_EMPTY, Parse_quit},
  {"RECONNECT", {{F_NAME, "<ip-number>"}, {F_INT, "<port-number>"},
		   {F_KEYWORD, PARSER_SEATS}}, Parse_reconnect},
  {"RECORD", {{F_NAME, "<board-name>"}, {F_INT, "<board-number>"}, NOP}, 
              Parse_record},

  {"RESET",  C_EMPTY, Parse_reset},
  {"SCORE",  {{F_STRING, "<current-scores>"}, NOP, NOP}, Parse_score},
  {"SEAT",   {{F_KEYWORD, PARSER_SEATS}, {F_KEYWORD, PARSER_SEATS},
		{F_NAME, "<player-name>"}}, Parse_seat},
  {"SEATERR",{{F_KEYWORD | F_OPTIONAL, PARSER_SEATS},
              {F_KEYWORD | F_OPTIONAL, PARSER_SEATS},
	      {F_KEYWORD | F_OPTIONAL, PARSER_SEATS}}, Parse_seaterr},

  {"SEATPOS", {{F_KEYWORD, PARSER_SEATS}, NOP, NOP}, Parse_seatpos},
  {"SEATREQ", C_SEAT, Parse_seatreq},
  {"SERVEREQ",C_PARAM(F_STRING, "<server-command>"), Parse_servereq},
  {"SKIP",    C_EMPTY, Parse_skip},
  {"SKIPACK",C_EMPTY, Parse_skipack},

  {"SPEC",   C_EMPTY, Parse_spec},
  {"TABLE",    C_INT("<table-number>"), Parse_table},
  {"TABLEREQ", C_INT("<requested-table-number>"), Parse_tablereq},
  {"TALK",   {{F_KEYWORD, "LHO|RHO|OPPS|SPEC|ALL"}, {F_STRING, "<message>"}, 
		NOP}, Parse_talk},
  {"USEREC", {{F_STRING, "<player-names>"}, NOP, NOP}, Parse_userec},

  {"WAKEUP", C_PARAM(F_NAME, "<player-name> | ALL"), Parse_wakeup},
  {"WHO",    C_EMPTY, Parse_who},
  {"WHOIS",  C_PARAM(F_NAME, "<player-name>"), Parse_whois},
  {"WHORESP",{{F_NAME, "<recipient>"}, {F_STRING, "<ident>"}, NOP},
     Parse_whoresp},
  END_PARSE_TABLE
};


static char send_buffer [132];
static player_command p;

void Send_ack (t, player_name, pos)
     Table_ptr t; name_buffer player_name; int pos;
{
  sprintf (send_buffer, "ACK %s %s", player_name,
	   (pos < 4)? seat_names[pos]: "");
  send_message (t, CMD_ACK, send_buffer);
}

void Send_alert (t, who, seq_no)
     Table_ptr t; int who, seq_no;
{
  sprintf (send_buffer, "ALERT %s %d", who? "PARTNER": "BIDDER", seq_no);
  send_message (t, CMD_ALERT, send_buffer);
}

void Send_begin (t)
     Table_ptr t;
{
  send_message (t, CMD_BEGIN, "BEGIN");
}

void Send_bid (t, bid, seq_no, alert)
     Table_ptr t;
     int bid;
     int seq_no;
     int alert;
{
  sprintf (send_buffer, "BID %s %d %s", bid_names[bid],  seq_no,
	   alert? "ALERT": "");
  send_message (t, CMD_BID, send_buffer);
}

void Send_board (t, b)
     Table_ptr t; board_ptr b;
{
  Message m;

  sprintf (send_buffer, "BOARD %s %d", b->source, b->serial_no);
  m = send_message (t, CMD_BOARD, send_buffer);
  m->p.data.board.record = b;
  if (client_mode) 
    Transmit_board (t, b);
  
}

void Send_claim (t, no_tricks)
     Table_ptr t; int no_tricks;
{
  sprintf (send_buffer, "CLAIM %d", no_tricks);
  send_message (t, CMD_CLAIM, send_buffer);
}

void Send_claimreq (t, no_tricks)
     Table_ptr t; int no_tricks;
{
  sprintf (send_buffer, "CLAIMREQ %d", no_tricks);
  send_message (t, CMD_CLAIMREQ, send_buffer);
}

void Send_claimresp (t, resp)
     Table_ptr t; int resp;
{
  sprintf (send_buffer, "CLAIMRESP %s", resp? "ACCEPT": "REJECT");
  send_message (t, CMD_CLAIMRESP, send_buffer);
}

void Send_cc (t, convention_card)
     Table_ptr t; char *convention_card;
{
  sprintf (send_buffer, "CC %s", convention_card);
  send_message (t, CMD_CC, send_buffer);
}

void Send_comment (t, comment)
     Table_ptr t; message_buffer comment;
{
  sprintf (send_buffer, "COMMENT %s", comment);
  send_message (t, CMD_COMMENT, send_buffer);
}

void Send_deal (t)
     Table_ptr t;
{
  send_message (t, CMD_DEAL, "DEAL");
}

void Send_echo (t, ping_source)
     Table_ptr t; name_buffer ping_source;
{
  sprintf (send_buffer, "ECHO %s", ping_source);
  send_message (t, CMD_ECHO, send_buffer);
}

void Send_email (t, email_addr)
     Table_ptr t; char *email_addr;
{
  sprintf (send_buffer, "EMAIL %s", email_addr);
  send_message (t, CMD_EMAIL, send_buffer);
}

void Send_end (t)
     Table_ptr t;
{
  send_message (t, CMD_END, "END");
}

void Send_fullname (t, fullname)
     Table_ptr t; message_buffer fullname;
{
  sprintf (send_buffer, "FULLNAME %s", fullname);
  send_message (t, CMD_FULLNAME, send_buffer);
}

void Send_hello (t, version, player_name, player_id)
     Table_ptr t; name_buffer version, player_name; int player_id;
{
  sprintf (send_buffer, "HELLO %s %s %d", version, player_name, player_id);
  send_message (t, CMD_HELLO, send_buffer);
}

void Send_mode (t, m)
     Table_ptr t; int m;
{
  switch (m) {
  case CLUB_PLAYING_MODE: 
    send_message (t, CMD_MODE, "MODE CLUB");
    break;
  case PRACTICE_PLAYING_MODE:
    send_message (t, CMD_MODE, "MODE PRACTICE");
    break;
  case FORMAL_PLAYING_MODE:
    send_message (t, CMD_MODE, "MODE FORMAL");
    break;
  case VUGRAPH_PLAYING_MODE:
    send_message (t, CMD_MODE, "MODE VUGRAPH");
    break;
  }
}

void Send_name (t, n, id)
     Table_ptr t; char *n; int id;
{
  sprintf (send_buffer, "NAME %s %d", n, id);
  send_message (t, CMD_NAME, send_buffer);
}

void Send_ping (t)
     Table_ptr t;
{
  send_message (t, CMD_PING, "PING");
}

void Send_play (t, play, seq_no)
     Table_ptr t; int play, seq_no;
{
  sprintf (send_buffer, "PLAY %s %d", card_names[play], seq_no);
  send_message (t, CMD_PLAY, send_buffer);
}

void Send_playreq (t, play, playno)
     Table_ptr t; int play, playno;
{
  sprintf (send_buffer, "PLAYREQ %s %d", card_names[play], playno);
  send_message (t, CMD_PLAYREQ, send_buffer);
}

void Send_quit (t)
     Table_ptr t;
{
  send_message (t, CMD_QUIT, "QUIT");
}

void Send_reconnect (t, c, ip, port, seat)
     Table_ptr t;
     Connection c;
     char *ip;
     int port;
     int seat;
{
  sprintf (send_buffer, "RECONNECT %s %d %s", ip, port, seat_names[seat]);
  send_private_message (c, send_buffer);
}

void Send_record (t, b, p)
     Table_ptr t;
     board_ptr b;
     play_ptr p;
{
  Message m;

  sprintf (send_buffer, "RECORD %s %d", b->source, b->serial_no);
  m = send_message (t, CMD_RECORD, send_buffer);
  m->p.data.record.play = p;
  if (client_mode)
    Transmit_play_record (t, p);

}

void Send_reset (t)
     Table_ptr t;
{
  send_message (t, CMD_RESET, "RESET");
}

void Send_hacked_score (t, ns, ew)
     Table_ptr t; Score *ns, *ew;
{
  Score *cns = (Local_play == NULL)? ns: &Local_play->score[SIDE_NS];
  Score *cew = (Local_play == NULL)? ew: &Local_play->score[SIDE_EW];

  switch (t->scoring_mode) {
  case RUBBER_SCORING:
    sprintf (send_buffer, "SCORE %d %d %d %d",
	     ns->rubber.above_line, ew->rubber.above_line,
	     ns->rubber.below_line, ew->rubber.below_line);
    break;

  case DUPLICATE_SCORING:
    sprintf (send_buffer, "SCORE %d %d %d %d",
	     t->scoring_mode,
	     ns->duplicate.total, 
	     ew->duplicate.total,
	     Local_play->score[SIDE_NS].duplicate.total,
	     Local_play->score[SIDE_EW].duplicate.total);
    break;

  case IMP_SCORING:
    sprintf (send_buffer, "SCORE %d %d %d %d",
	     (int) (Average_Imps (ns) * 100.0),
	     (int) (Average_Imps (ew) * 100.0),
	     (int) (Average_Imps (cns) * 100.0),
	     (int) (Average_Imps (cew) * 100.0));
    break;

  case MP_SCORING:
    sprintf (send_buffer, "SCORE %d %d %d %d",
	     (int) (Percent_Matchpoints (ns) * 100.0),
	     (int) (Percent_Matchpoints (ew) * 100.0),
	     (int) (Percent_Matchpoints (cns) * 100.0),
	     (int) (Percent_Matchpoints (cew) * 100.0));
    break;
  }

  send_server_message (t, CMD_SCORE, send_buffer);
}

void Format_Score_Message (buffer, t, ns, ew)
     char *buffer; Table_ptr t; Score *ns, *ew;
{
  switch (t->scoring_mode) {
  case RUBBER_SCORING:
    sprintf (buffer, "SCORE %d %d %d %d %d",
	     t->scoring_mode,
	     ns->rubber.above_line, ew->rubber.above_line,
	     ns->rubber.below_line, ew->rubber.below_line);
    break;

  case DUPLICATE_SCORING:
    sprintf (buffer, "SCORE %d %d %d",
	     t->scoring_mode,
	     ns->duplicate.total, ew->duplicate.total);
    break;

  case IMP_SCORING:
    sprintf (buffer, "SCORE %d %d %d %d %d",
	     t->scoring_mode,
	     ns->duplicate.competitive.imp.imp_total,
	     ns->duplicate.competitive.imp.imp_tables,
	     ew->duplicate.competitive.imp.imp_total,
	     ew->duplicate.competitive.imp.imp_tables);
    break;

  case MP_SCORING:
    sprintf (buffer, "SCORE %d %d %d %d %d",
	     t->scoring_mode,
	     ns->duplicate.competitive.mp.mp_total,
	     ns->duplicate.competitive.mp.mp_tables,
	     ew->duplicate.competitive.mp.mp_total,
	     ew->duplicate.competitive.mp.mp_tables);
    break;
  }
}

void Send_score (t, ns, ew)
     Table_ptr t; Score *ns, *ew;
{
  Format_Score_Message (send_buffer, t, ns, ew);
  send_server_message (t, CMD_SCORE, send_buffer);
}

void Send_seat (t, oldseat, newseat, name)
     Table_ptr t; int oldseat, newseat; char *name; 
{
  sprintf (send_buffer, "SEAT %s %s %s", seat_names[oldseat], 
	   seat_names[newseat], name);
  send_message (t, CMD_SEAT, send_buffer);
}

void Send_seaterr (t, message)
     Table_ptr t; message_buffer message;
{
  sprintf (send_buffer, "SEATERR %s", message);
  send_message (t, CMD_SEATERR, send_buffer);
}

void Send_seatreq (t, seat)
     Table_ptr t; int seat;
{
  sprintf (send_buffer, "SEATREQ %s", seat_names[seat]);
  send_message (t, CMD_SEAT, send_buffer);
}

void Send_servereq (t, request)
     Table_ptr t; char *request;
{
  sprintf (send_buffer, "SERVEREQ %s", request);
  send_message (t, CMD_SERVEREQ, send_buffer);
}

void Send_skip (t)
     Table_ptr t;
{
  send_message (t, CMD_SKIP, "SKIP");
}

void Send_skipack (t)
     Table_ptr t;
{
  send_message (t, CMD_SKIPACK, "SKIPACK");
}

void Send_spec (t)
     Table_ptr t;
{
  send_message (t, CMD_SPEC, "SPEC");
}

void Send_table (t, c, table_no)
     Table_ptr t;
     Connection c;
     int table_no;
{
  sprintf (send_buffer, "TABLE %d", table_no);
  send_private_message (c, send_buffer);
}

void Send_tablereq (t, table_no)
     Table_ptr t;
     int table_no;
{
  sprintf (send_buffer, "TABLEREQ %d", table_no);
  send_message (t, CMD_TABLEREQ, send_buffer);
}

static char *talk_rcpts [] = {"LHO", "RHO", "OPPS", "SPEC", "ALL"};

void Send_talk (t, destination, message)
     Table_ptr t; int destination; message_buffer message;
{

  sprintf (send_buffer, "TALK %s %s", talk_rcpts[destination], message);
  send_message (t, CMD_TALK, send_buffer);
}

void Send_userec (t, n, e, s, w)
     Table_ptr t;
     char *n;
     char *e;
     char *s;
     char *w;
{
  sprintf (send_buffer, "USEREC %s %s %s %s", n, e, s, w);
  send_message (t, CMD_USEREC, send_buffer);
}

void Send_wakeup (t, p)
     Table_ptr t; char *p;
{
  sprintf (send_buffer, "WAKEUP %s", p);
  send_message (t, CMD_WAKEUP, send_buffer);
}

void Send_who (t)
     Table_ptr t;
{
  send_message (t, CMD_WHO, "WHO");
}

void Send_whois (t, p)
     Table_ptr t; char *p;
{
  sprintf (send_buffer, "WHOIS %s", p);
  send_message (t, CMD_WHOIS, send_buffer);
}

void Send_whoresp (t, r, i)
     Table_ptr t; char *r, *i;
{
  sprintf (send_buffer, "WHORESP %s %s", r, i);
  send_message (t, CMD_WHORESP, send_buffer);
}

void Parse_ack (player_name, pos)
     char *player_name; int *pos;
{
  p->command = CMD_ACK;
  Lcopy (player_name, p->data.ack.player_name, NAME_LENGTH);
  p->data.ack.position = (pos == NULL)? 4: *pos;
}

void Parse_alert (alerter, seq_no)
     int *alerter; int *seq_no;
{
  p->command = CMD_ALERT;
  p->data.alert.who = *alerter;
  p->data.alert.seq_no = *seq_no;
}

void Parse_begin ()
{
  p->command = CMD_BEGIN;
}

void Parse_bid (bid, seq_no, alert)
     int *bid, *seq_no, *alert;
{
  p->command = CMD_BID;
  p->data.bid.level = *bid;
  p->data.bid.seq_no = *seq_no;
  p->data.bid.alert = (alert == NULL)? 0: 1;
}

void Parse_board (name, no)
     char *name; int *no;
{
  p->command = CMD_BOARD;
  Lcopy (name, p->data.board.board_name, NAME_LENGTH);
  p->data.board.board_no = *no;
}

void Parse_cc (convention_card)
     char *convention_card;
{
  p->command = CMD_CC;
  Lcopy (convention_card, p->data.cc, MESSAGE_LENGTH);
}

void Parse_claim (no_tricks)
     int *no_tricks;
{
  p->command = CMD_CLAIM;
  p->data.claim.no_tricks = *no_tricks;
}

void Parse_claimreq (no_tricks)
     int *no_tricks;
{
  p->command = CMD_CLAIMREQ;
  p->data.claimreq.no_tricks = *no_tricks;
}

void Parse_claimresp (resp)
     int *resp;
{
  p->command = CMD_CLAIMRESP;
  p->data.claimresp = *resp;
}

void Parse_comment (message)
     char *message;
{
  p->command = CMD_COMMENT;
  Lcopy (message, p->data.comment, MESSAGE_LENGTH);
}

void Parse_connerr (message)
     char *message;
{
  p->command = CMD_CONNERR;
  Lcopy (message, p->data.connerr, MESSAGE_LENGTH);
}

void Parse_deal ()
{
  p->command = CMD_DEAL;
}

void Parse_echo (ping_source)
     char *ping_source;
{
  p->command = CMD_ECHO;
  Lcopy (ping_source, p->data.echo.ping_source, NAME_LENGTH);
}

void Parse_email (email_addr)
     char *email_addr;
{
  p->command = CMD_EMAIL;
  Lcopy (email_addr, p->data.email.addr, MESSAGE_LENGTH);
}

void Parse_end ()
{
  p->command = CMD_END;
}

void Parse_fullname (fullname)
     char *fullname;
{
  p->command = CMD_FULLNAME;
  Lcopy (fullname, p->data.fullname.name, MESSAGE_LENGTH);
}

void Parse_hello (version, player_name, player_id)
     char *version, *player_name; int *player_id;
{
  p->command = CMD_HELLO;
  Lcopy (version, p->data.hello.version, NAME_LENGTH);
  Lcopy (player_name, p->data.hello.player_name, NAME_LENGTH);
  p->data.hello.player_id = *player_id;
}

void Parse_mode (m)
     int *m;
{
  p->command = CMD_MODE;
  p->data.mode = *m;
}

void Parse_name (c, id)
     char *c; int *id;
{
  p->command = CMD_NAME;
  Lcopy (c, p->data.name.new_name, NAME_LENGTH);
  p->data.name.new_id = *id;
}

void Parse_ping ()
{
  p->command = CMD_PING;
}

void Parse_play (play, seq_no)
     int *play, *seq_no;
{
  p->command = CMD_PLAY;
  p->data.play.card = *play;
  p->data.play.seq_no = *seq_no;
}

void Parse_playreq (play, seqno)
     int *play, *seqno;
{
  p->command = CMD_PLAYREQ;
  p->data.playreq.card = *play;
  p->data.playreq.play_no = *seqno;
}

void Parse_quit ()
{
  p->command = CMD_QUIT;
}

void Parse_reconnect (ip, port, seat)
     char *ip; int *port; int *seat;
{
  p->command = CMD_RECONNECT;
  Lcopy (ip, p->data.reconnect.ip, NAME_LENGTH);
  p->data.reconnect.port = *port;
  p->data.reconnect.seat = *seat;
}


void Parse_record (name, no)
     char *name; int *no;
{
  p->command = CMD_RECORD;
  Lcopy (name, p->data.record.board_name, NAME_LENGTH);
  p->data.record.board_no = *no;
  p->data.record.play = NULL;
}

void Parse_reset ()
{
  p->command = CMD_RESET;
}

void Parse_score (current_scores)
     char *current_scores;
{
  Score *ns = &(p->data.score.ns);
  Score *ew = &(p->data.score.ew);
  int tmp;

  p->command = CMD_SCORE;
  sscanf (current_scores, "%d", &p->data.score.score_type);
  switch (p->data.score.score_type) {
  case RUBBER_SCORING:
    sscanf (current_scores, "%d %d %d %d %d", &tmp,
	    &ns->rubber.above_line, &ew->rubber.above_line,
	    &ns->rubber.below_line, &ew->rubber.below_line);
    break;

  case DUPLICATE_SCORING:
    sscanf (current_scores, "%d %d %d", &tmp,
	    &ns->duplicate.total, &ew->duplicate.total);
    break;

  case IMP_SCORING:
    sscanf (current_scores, "%d %d %d %d %d", &tmp,
	    &ns->duplicate.competitive.imp.imp_total,
	    &ns->duplicate.competitive.imp.imp_tables,
	    &ew->duplicate.competitive.imp.imp_total,
	    &ew->duplicate.competitive.imp.imp_tables);
    break;

  case MP_SCORING:
    sscanf (current_scores, "%d %d %d %d %d", &tmp,
	    &ns->duplicate.competitive.mp.mp_total,
	    &ns->duplicate.competitive.mp.mp_tables,
	    &ew->duplicate.competitive.mp.mp_total,
	    &ew->duplicate.competitive.mp.mp_tables);
    break;

  default:
    break;
  }
}

void Parse_seat (old, new, name)
     int *old, *new; char *name;
{
  p->command = CMD_SEAT;
  p->data.seat.old_pos = *old;
  p->data.seat.new_pos = *new;
  Lcopy (name, p->data.seat.player_name, NAME_LENGTH);
}

void Parse_seaterr (free0, free1, free2)
     int *free0, *free1, *free2;
{
  p->command = CMD_SEATERR;
  p->data.seaterr.free_seats[0] = (free0 == NULL)? 4: *free0;
  p->data.seaterr.free_seats[1] = (free1 == NULL)? 4: *free1;
  p->data.seaterr.free_seats[2] = (free2 == NULL)? 4: *free2;
}

void Parse_seatpos (s)
     int *s;
{
  p->command = CMD_SEATPOS;
  p->data.seatpos = *s;
}

void Parse_seatreq (s)
     int *s;
{
  p->command = CMD_SEATREQ;
  p->data.seatreq = *s;
}

void Parse_servereq (request)
     char *request;
{
  p->command = CMD_SERVEREQ;
  Lcopy (request, p->data.servereq.command, MESSAGE_LENGTH);
}

void Parse_skip ()
{
  p->command = CMD_SKIP;
}

void Parse_skipack ()
{
  p->command = CMD_SKIPACK;
}

void Parse_spec ()
{
  p->command = CMD_SPEC;
}

void Parse_table (table_no)
     int *table_no;
{
  p->command = CMD_TABLE;
  p->data.table = *table_no;
}

void Parse_tablereq (table_no)
     int *table_no;
{
  p->command = CMD_TABLEREQ;
  p->data.tablereq = *table_no;
}

void Parse_talk (recpts, message)
     int *recpts; char *message;
{
  p->command = CMD_TALK;
  p->data.talk.recipients = *recpts;
  Lcopy (message, p->data.talk.message, MESSAGE_LENGTH);
}

void Parse_userec (player_names)
     char *player_names;
{
  char north_buf[80], east_buf[80], south_buf[80], west_buf[80];

  p->command = CMD_USEREC;
  sscanf (player_names, "%s %s %s %s", north_buf, east_buf, 
	  south_buf, west_buf);
  Lcopy (north_buf, p->data.userec.north, NAME_LENGTH);
  Lcopy (east_buf,  p->data.userec.east,  NAME_LENGTH);
  Lcopy (south_buf, p->data.userec.south, NAME_LENGTH);
  Lcopy (west_buf,  p->data.userec.west,  NAME_LENGTH);
}

void Parse_wakeup (c)
     char *c;
{
  p->command = CMD_WAKEUP;
  Lcopy (c, p->data.wakeup.recipient, NAME_LENGTH);
}

void Parse_who ()
{
  p->command = CMD_WHO;
}

void Parse_whois (n)
     char *n;
{
  p->command = CMD_WHOIS;
  Lcopy (n, p->data.whois.name, NAME_LENGTH);
}

void Parse_whoresp (r, i)
     char *r, *i;
{
  p->command = CMD_WHORESP;
  Lcopy (r, p->data.whoresp.recipient, NAME_LENGTH);
  Lcopy (i, p->data.whoresp.message, MESSAGE_LENGTH);
}

int Parse_Command_for_Client (pc)
     player_command pc;
{
  int i, n;
  p = pc;

  switch (p->command_text[0]) {
  case 'N':
    p->player_no = PLAYER_NORTH; break;
  case 'E':
    p->player_no = PLAYER_EAST;  break;
  case 'S':
    p->player_no = PLAYER_SOUTH; break;
  case 'W':
    p->player_no = PLAYER_WEST;  break;
  case 'M':
    p->player_no = PLAYER_SERVER; break;
  case 'O':
    p->player_no = PLAYER_OBS;   break;
  default:
    p->command = CMD_ERROR;
    sprintf (p->data.error.message, 
	     "ERROR IN MESSAGE SOURCE");
    return (-1);
  }

  i = 2;
  while ((p->command_text[i] != '\0') && !isspace(p->command_text[i])) i++;
  while ((p->command_text[i] != '\0') && isspace(p->command_text[i])) i++;
  n = 0;
  while (p->command_text[i] && !isspace(p->command_text[i])) {
    if (n < NAME_LENGTH-1)
      p->player_name[n++] = p->command_text[i];
    i++;
  }
  p->player_name[n] = '\0';

  if (Parse_Command (Commands, CMD_MAX-1, p->command_text+i)) {
    p->command = CMD_ERROR;
    Lcopy (Parser_Error_Buf, p->data.error.message, MESSAGE_LENGTH);
    return (-1);
  }
  return (0);

}

int Parse_Command_for_Server (pc)
     player_command pc;
{
  int i, n;
  p = pc;

  i = 0;
  while ((p->command_text[i] != '\0') && isspace(p->command_text[i])) i++;
  n = 0;
  while (p->command_text[i] && !isspace(p->command_text[i])) {
    if (n < NAME_LENGTH-1)
      p->player_name[n++] = p->command_text[i];
    i++;
  }
  p->player_name[n] = '\0';

  if (Parse_Command (Commands, CMD_MAX-1, p->command_text+i)) {
    p->command = CMD_ERROR;
    Lcopy (Parser_Error_Buf, p->data.error.message, MESSAGE_LENGTH);
    return (-1);
  }
  return (0);

}

void Initialize_protocols ()
/* Initializes the array of message attributes. */
{
  ;
}
