/* parser.h -- generalized parser
 *
 ! Copyright (C) 1990-1992 by Matthew Clegg.  All Rights Reserved
 ! 
 ! OKbridge is made available as a free service to the Internet.
 ! Accordingly, the following restrictions are placed on its use:
 ! 
 ! 1.  OKbridge may not be modified in any way without the explicit 
 !     permission of Matthew Clegg.  
 ! 
 ! 2.  OKbridge may not be used in any way for commercial advantage.
 !     It may not be placed on for-profit networks or on for-profit
 !     computer systems.  It may not be bundled as part of a package
 !     or service provided by a for-profit organization.
 ! 
 ! If you have questions about restrictions on the use of OKbridge,
 ! write to mclegg@cs.ucsd.edu.
 ! 
 ! DISCLAIMER:  The user of OKbridge accepts full responsibility for any
 ! damage which may be caused by OKbridge.
 *
 * This module contains the code which is used for parsing commands
 * to the okbridge program.  There are several command languages which
 * are used by the program: (1) the language used for communicating
 * over the network with other copies of the bridge program,
 * (2) the language used by the local player for specifying commands
 * to the program, (3) the language used for storing information in
 * the startup files, and (4) the language used for communicating
 * requests to the GPS.
 *
 * The syntax of each of these languages is very simple.  Each language
 * consists of a series of commands.  A command begins with a keyword
 * and then up to three command-specific parameters.  The possible
 * parameter types are also very restricted.  Here is a list of the
 * parameter types:
 *
 *  F_KEYWORD.  The parameter is one of a list of keywords.  The keywords
 *    are specified as a list of uppercase words, separated by
 *    vertical bars, e.g., "RUBBER|CHICAGO|DUPLICATE|EMAIL|IMP".
 *    Each keyword is indexed implicitly by its order in the array,
 *   
 *  F_BID.  The parameter is the name of a bid.
 *
 *  F_CARD.  The parameter is the name of a card.
 *
 *  F_INT.  The parameter is an integer.
 *
 *  F_FILENAME.  The parameter is the name of a file.
 *
 *  F_STRING.  The file is a string which terminates at the end of the
 *    buffer.
 *
 *  F_NAME.  The parameter is a name of a player.  This is currently
 *    treated in the same way as a filename.
 *
 *  F_DEAL.  The parameter is a deal of the cards.
 *
 * Consequently, we can encode each command with its keyword, the
 * parameters it may accept, and a procedure for encoding a data
 * structure to represent the command.  The action of the parser
 * can then be described as follows.  We begin by identifying the
 * command keyword.  Having done this, we attempt to parse the
 * command specific parameters.  If this is successful, then we
 * call the command specific procedure which creates the data structure
 * representing the command.
 *
 * NOTE: The generalized parser is not re-entrant.
 *
 */

typedef void (*command_handler) ();

#define F_KEYWORD   000
#define F_BID       001
#define F_CARD      002
#define F_INT       003
#define F_FILENAME  004
#define F_STRING    005
#define F_NAME      006
#define F_DEAL      007
#define F_END       010

#define F_OPTIONAL 0100


typedef struct {
  int    field_type;
  char   *field_descriptor;
} Field_Descriptor;

typedef struct Command_Descriptor_Struct {
  char *command_name;
  Field_Descriptor fields[3];
  command_handler chandler;
  int *save_field;
} Command_Descriptor;

#define NO_PARAM {F_END, NULL}
#define NOP NO_PARAM
#define NOS NULL
#define RCOK ((int *) -1)
#define END_PARSE_TABLE {NULL, {NO_PARAM, NO_PARAM, NO_PARAM}, NULL, NOS}

/* The following macros define commonly used option lists: */

#define PARSER_PLAYERS "NORTH|EAST|SOUTH|WEST"
#define PARSER_SEATS   "NORTH|EAST|SOUTH|WEST|OBS"
#define PARSER_ANSWER  "NO|YES"
#define PARSER_STATE   "OFF|ON|TOGGLE"
#define PARSER_VULN    "NONE|EW|NS|BOTH"
#define PARSER_SCORE   "RUBBER|DUPLICATE|IMP|MP"
#define PARSER_LOG     "TEXT|EMAIL|BOTH"

/* The following macros are for commands that use PARSER_STATE */
#define OFF 0
#define ON 1
#define TOGGLE 2

/* The following macros define commonly used arguments to commands: */
#define A_SWITCH  {F_KEYWORD, PARSER_ANSWER}
#define A_STATE   {F_KEYWORD|F_OPTIONAL, PARSER_STATE}
#define A_PLAYER  {F_KEYWORD, PARSER_PLAYERS}
#define A_SEAT    {F_KEYWORD, PARSER_SEATS}
#define A_VULN    {F_KEYWORD, PARSER_VULN}
#define A_SCORE   {F_KEYWORD, PARSER_SCORE}

/* The following macros define command lines where there is only one
   parameter: */
#define C_PARAM(x,y) {{x,y}, NOP, NOP}
#define C_EMPTY      {NOP, NOP, NOP}
#define C_KEY(x)     C_PARAM(F_KEYWORD, x)
#define C_INT(x)     C_PARAM(F_INT, x)
#define C_STRING(x)  C_PARAM(F_STRING, x)

#define C_SWITCH     C_PARAM(F_KEYWORD, PARSER_STATE)
#define C_STATE      C_PARAM(F_KEYWORD|F_OPTIONAL, PARSER_STATE)
#define C_PLAYER     C_PARAM(F_KEYWORD, PARSER_PLAYERS)
#define C_SEAT       C_PARAM(F_KEYWORD, PARSER_SEATS)
#define C_SCORE      C_PARAM(F_KEYWORD, PARSER_SCORE)
#define C_VULN       C_PARAM(F_KEYWORD, PARSER_VULN)

#ifdef PARSER
  char Parser_Error_Buf [80];
    /* A buffer where error messages are written if an error is
       encountered in processing the command. */
  int Parser_Error_Location;
    /* The index in the input buffer where trouble began. */

#else

  extern char Parser_Error_Buf [];
  extern int  Parser_Error_Location;

#endif

extern int Parse_Command ();
/* int Parse_Command (Command_Descriptor *command_table, 
   int table_length, char *buffer); */
/* Reads a command from the buffer and attempts to parse it according
   to the given table.  If successful, then the command specific procedure
   is called with the parsed parameters.  In this case, zero is returned.
   If unsuccessful, records some error information in Parser_Error_Buf
   and Parser_Error_Location and returns 1.
*/

extern int Parse_Command_Dont_Execute ();
/* int Parse_Command (Command_Descriptor *command_table, 
   int table_length, char *buffer); */
/* Just like Parse_Command, except that the command procedure is NOT invoked
   after successful parsing. 
*/

extern void Lcopy ();
/* void Lcopy (char *from, char *to, int len); */
/* Copies at most len-1 characters from from to to. */

   
#define PARSER_SUCCESS 0
#define PARSER_UNKNOWN 1
#define PARSER_AMBIGUOUS 2
#define PARSER_WRONG_PARMS 3
