/* parser.c -- implementaton of generalized parser.
 *
 ! Copyright (C) 1990-1992 by Matthew Clegg.  All Rights Reserved
 ! 
 ! OKbridge is made available as a free service to the Internet.
 ! Accordingly, the following restrictions are placed on its use:
 ! 
 ! 1.  OKbridge may not be modified in any way without the explicit 
 !     permission of Matthew Clegg.  
 ! 
 ! 2.  OKbridge may not be used in any way for commercial advantage.
 !     It may not be placed on for-profit networks or on for-profit
 !     computer systems.  It may not be bundled as part of a package
 !     or service provided by a for-profit organization.
 ! 
 ! If you have questions about restrictions on the use of OKbridge,
 ! write to mclegg@cs.ucsd.edu.
 ! 
 ! DISCLAIMER:  The user of OKbridge accepts full responsibility for any
 ! damage which may be caused by OKbridge.
 *
 */

#include <stdio.h>
#include <ctype.h>
#include <string.h>

extern int atoi ();

#include "types.h"
#define PARSER
#include "parser.h"

#define MAPUPPER(c) ((('a' <= (c)) && ((c) <= 'z'))? ((c) + 'A' - 'a') : (c))

static char *input_buffer;
  /* The current input buffer being examined. */
static int   buffer_pos;
  /* The current position in the parser buffer. */
static char  keyword_buffer[80];
  /* A buffer used for holding one keyword. */

static char command_word [80];
typedef char parameter_string_element[80];
static int *parameter_list[3];
static int parameter_keyword [3];
static parameter_string_element parameter_string[3];
static hand parameter_hand[3];

#define NOW     input_buffer[buffer_pos]
#define NEXT    input_buffer[buffer_pos++]
#define ADVANCE buffer_pos++;

#define IGNORE_CASE   1
#define PRESERVE_CASE 0

void Lcopy (from, to, len)
     char *from, *to; int len;
/* Copies at most len-1 characters from from to to. */
{
  int i = 0;

  len -= 1;
  for (i = 0; i < len; i++) {
    to [i] = from[i];
    if (to[i] == '\0') return;
  }
  to[len] = '\0';
}

static void Skip_whitespace ()
/* Skips any whitespace characters in the input_buffer. */
{
  while ((NOW == ' ') || (NOW == '\t')) ADVANCE;
}

static int Parse_Field (dest_buffer, dest_buflen, uppercase)
     char *dest_buffer; int uppercase, dest_buflen;
/* Reads a field from the current input buffer.  A field is a sequence
   of characters delimited by whitespace.  Copies the field to the
   keyword_buffer.  Returns the number of characters in the field,
   or 0 if the end of the input buffer has been reached.  If uppercase
   is true, then the characters in the field are automatically converted
   to upper case as they are read.
*/
{
  int n = 0;
  int c;

  Skip_whitespace ();
  dest_buflen -= 1;
  while ((NOW != ' ') && (NOW != '\t') && (NOW != '\0')) {
    c = NEXT;
    if (n < dest_buflen)
      dest_buffer[n++] = uppercase? MAPUPPER(c): c;
  }

  dest_buffer[n] = '\0';
  return (n);
}

static int Parse_keyword (keyword_list)
     char *keyword_list;
/* Reads a keyword from the input buffer and tries to find it in
   the keyword_list.  If no match can be found, then returns -1.
   Otherwise, returns the index of the keyword in the list of keywords,
   with the first keyword having index 0.
*/
{
  int l, n, keyword_index, maybe_index;

  n = Parse_Field (keyword_buffer, 80, IGNORE_CASE);
  keyword_index = 0;
  maybe_index = -1;
  while (*keyword_list) {
    l = 0;
    while ((keyword_list[l] != '|') && (keyword_list[l] != '\0')) l++;
    if ((l == 0) && (n == -1))
      return (keyword_index);
    else if ((l == n) && !strncmp(keyword_buffer, keyword_list, n))
      return (keyword_index);
    else if ((l > n) && !strncmp(keyword_buffer, keyword_list, n))
      maybe_index = (maybe_index == -1 ? keyword_index : -2);
    keyword_index++;
    if (keyword_list[l] != '\0')
      keyword_list += l+1;
    else
      return (maybe_index < 0 ? -1 : maybe_index);
  }
  return (maybe_index < 0 ? -1 : maybe_index);
}

static int Parse_bid ()
/* Reads a field from the input buffer and interprets it as a bid.
   Returns the index of the bid in the bid table.  If unsuccessful,
   returns -1.
*/
{
  int n = Parse_Field (keyword_buffer, 80, IGNORE_CASE);
  int i = 0;

  if (n == 0) return (-1);

  if (!strcmp(keyword_buffer, "PASS"))
    return (BID_PASS);
  else if (!strcmp(keyword_buffer, "DOUBLE"))
    return (BID_DOUBLE);
  else if (!strcmp(keyword_buffer, "REDOUBLE"))
    return (BID_REDOUBLE);

  if (keyword_buffer[n-1] == 'N') {
    keyword_buffer[n++] = 'T';
    keyword_buffer[n] = '\0';
  }

  while (bid_names[i] != NULL) {
    if (!strcmp(keyword_buffer, bid_names[i]))
      return (i);
    else
      i++;
  }
  return (-1);
}

static int Parse_Card ()
/* Reads the next field from the input buffer and interprets it as a
   card name.  Returns the index of the card in the card array,
   or -1 if no match is found.
 */
{
  int n = Parse_Field (keyword_buffer, 80, IGNORE_CASE);
  int i = 0;

  if (n != 2) return (-1);

  while (card_names[i] != NULL) {
    if (!strcmp(keyword_buffer, card_names[i]))
      return (i);
    else
      i++;
  }

  i = keyword_buffer[0];
  keyword_buffer[0] = keyword_buffer[1];
  keyword_buffer[1] = i;
  i = 0;
  while (card_names[i] != NULL) {
    if (!strcmp(keyword_buffer, card_names[i]))
      return (i);
    else
      i++;
  }

  return (-1);
}

static int Parse_int (ivalue)
     int *ivalue;
/* Parses an integer from the input stream.  Returns 0 if an integer
   was found, or -1 if not.  The value is placed into ivalue. */
{
  int i = 0, d = 0;

  Skip_whitespace ();
  if ((NOW == '+') || (NOW == '-')) {
    keyword_buffer[i++] = NOW;
    ADVANCE;
  }

  while (('0' <= NOW) && (NOW <= '9')) {
    keyword_buffer[i++] = NOW;
    ADVANCE;
    d++;
  }
  keyword_buffer[i] = '\0';

  if (d == 0) return (-1);
  *ivalue = atoi (keyword_buffer);
  return (0);
}

static int Parse_filename (fn_buffer, buflen)
     char *fn_buffer; int buflen;
/* Reads a filename into the buffer fn_buffer.  If a possible
   filename is found, then returns 0.  Otherwise, returns 1.
*/
{
  int n = Parse_Field (fn_buffer, buflen, PRESERVE_CASE);
  return (n == 0);
}

static int Parse_string (string_buffer)
     char *string_buffer;
/* Simply copies the rest of the input buffer to the string_buffer. 
   Returns 0 if at least one character was copied.
*/
{
  int i = 0;

  Skip_whitespace ();
  while (NOW != '\0')
    string_buffer[i++] = NEXT;
  string_buffer[i] = '\0';
  return (i == 0);
}

static int Parse_name (name_buf, name_buflen)
     char *name_buf; int name_buflen;
/* Copies a blank delimited "name" to the buffer. */
{
  int n = Parse_Field (name_buf, name_buflen, PRESERVE_CASE);
  return (n == 0);
}

static int Parse_hand (h)
     hand h;
/* Parses a hand from the buffer.  If an error occurs, returns -1.
   Otherwise, returns the hand in h. */
{
  int n = Parse_Field (keyword_buffer, 80, PRESERVE_CASE);
  int i;

  if (n != 52) return (-1);

  for (i = 0; i < 52; i++) {
    if (('0' <= keyword_buffer[i]) && (keyword_buffer[i] <= '3'))
      h[i] = keyword_buffer[i] - '0';
    else
      return (-1);
  }
  return (0);

}

static int Command_Table_Search (table, length, command)
     Command_Descriptor *table; int length; char *command;
{
  int b, m, t, r, maybe;

  b = 0;
  t = length;
  maybe = 0;
  while (b < t) {
    m = (b + t) / 2;
    r = strncmp (command, table[m].command_name, strlen(command));
/*    printf ("%d %d %d Comparing %s to %s\n", b, m, t,
	    command, table[m].command_name); */
    if (r < 0)
      t = m-1;
    else if (r > 0)
      b = m+1;
    else {
      maybe = 1;
      break;
    }
  }
  if (maybe == 0) {
    if (b == length)
      return (-1);
    else if (strncmp(command, table[b].command_name, strlen(command)))
      return (-1);
    else
      m = b;
  }
  if (m > 1)
  while (strncmp(command, table[m-1].command_name, strlen(command)) == 0
   && m > 1)
    m--;
  if (strlen(command) == strlen(table[m].command_name))
    return (m);
  if (m != 0)
    if (strncmp(command, table[m-1].command_name, strlen(command)) == 0)
      return (-2);
  if (m != length - 1)
    if (strncmp(command, table[m+1].command_name, strlen(command)) == 0)
      return (-2);
  return (m);
}

Command_Descriptor *Search_for_Command (command_table, table_length, buffer)
     Command_Descriptor *command_table; int table_length; char *buffer;
/* Searches the command table for the command given in the buffer.
   If the command exists, then returns the corresponding command table
   entry.  Otherwise, returns NULL.
*/
{
  int i = 0;  /* The index of the command in the command table. */
  int n;

  input_buffer = buffer;
  buffer_pos = 0;

  n = Parse_Field (command_word, 80, IGNORE_CASE);
  i = Command_Table_Search (command_table, table_length, command_word);
  if (i < 0)
    return (NULL);
  else
    return (command_table+i);
}

int Parse_Command_May_Execute (command_table, table_length, buffer, ex)
     Command_Descriptor *command_table; int table_length; char *buffer; int ex;
/* int Parse_Command (Command_Descriptor *command_table, 
   int table_length, char *buffer, int ex); */
/* Reads a command from the buffer and attempts to parse it according
   to the given table.  If successful, then the command specific procedure
   is called with the parsed parameters if the flag ex is true.  In this case, 
   zero is returned.  If unsuccessful parse, records some error information in 
   Parser_Error_Buf and Parser_Error_Location and returns 1.
*/
{
  int i = 0;  /* The index of the command in the command table. */
  int n, error_flag;
  int code; /* return code */

  input_buffer = buffer;
  buffer_pos = 0;

  n = Parse_Field (command_word, 80, IGNORE_CASE);
  i = Command_Table_Search (command_table, table_length, command_word);
  if (i < 0) {
    if (i == -1) {
      sprintf (Parser_Error_Buf, "UNRECOGNIZED COMMAND NAME: %s",
	     command_word);
      code = PARSER_UNKNOWN;
    }
    else if (i == -2) {
      sprintf (Parser_Error_Buf, "AMBIGUOUS COMMAND NAME: %s",
	     command_word);
      code = PARSER_AMBIGUOUS;
    }
    Parser_Error_Location = 0;
    return (code);
  }

  n = 0;
  while ((command_table[i].fields[n].field_type != F_END) && (n < 3)) {
    Skip_whitespace ();
    Parser_Error_Location = buffer_pos;
    if ((NOW == '\0') && (command_table[i].fields[n].field_type & F_OPTIONAL))
      parameter_list[n] = NULL;
    else {
      switch (command_table[i].fields[n].field_type & 077) {
      case F_KEYWORD:
	parameter_keyword[n] = 
	  Parse_keyword(command_table[i].fields[n].field_descriptor);
	if (parameter_keyword[n] < 0) {
	  sprintf (Parser_Error_Buf, "UNRECOGNIZED KEYWORD -- EXPECTED %s",
/*		   command_word, "COMMAND -- EXPECTED ", */
		   command_table[i].fields[n].field_descriptor);
	  return (PARSER_WRONG_PARMS);
	}
	parameter_list[n] = &parameter_keyword[n];
	break;
      case F_BID:
	parameter_keyword[n] = Parse_bid ();
	if (parameter_keyword[n] < 0) {
	  sprintf (Parser_Error_Buf, "%s %s",
		   "THE FORMAT OF A CORRECT BID IS <LEVEL> <TRUMPSUIT>", 
		   " OR P OR X OR XX");
	  return (PARSER_WRONG_PARMS);
	}
	parameter_list[n] = &parameter_keyword[n];
	break;
      case F_CARD:
	parameter_keyword[n] = Parse_Card ();
	if (parameter_keyword[n] < 0) {
	  sprintf (Parser_Error_Buf, "%s",
		   "ERROR - THE FORMAT OF A CORRECT PLAY IS <SUIT> <RANK>");
	  return (PARSER_WRONG_PARMS);
	}
	parameter_list[n] = parameter_keyword + n;
	break;
      case F_INT:
	error_flag = Parse_int (&parameter_keyword[n]);
	parameter_list[n] = &parameter_keyword[n];
	if (error_flag) {
	  sprintf (Parser_Error_Buf, "%s",
		   "ERROR -- EXPECTED INTEGER VALUE.");
	  return (PARSER_WRONG_PARMS);
	}
	break;
      case F_FILENAME:
	error_flag = Parse_filename (parameter_string[n], 80);
	parameter_list[n] = (int *) (parameter_string + n);
	if (error_flag) {
	  sprintf (Parser_Error_Buf, "%s",
		   "ERROR -- EXPECTED FILE NAME.");
	  return (PARSER_WRONG_PARMS);
	}
	break;
      case F_STRING:
	error_flag = Parse_string (parameter_string[n]);
	parameter_list[n] = (int *) (parameter_string + n);
	if (error_flag) {
	  sprintf (Parser_Error_Buf, "%s %s %s",
		   "ERROR --",
		   command_table[i].fields[n].field_descriptor,
		   "MISSING FROM INPUT");
	  return (PARSER_WRONG_PARMS);
	}
	break;
      case F_NAME:
	error_flag = Parse_name (parameter_string[n], 80);
	parameter_list[n] = (int *) (parameter_string + n);
	if (error_flag) {
	  sprintf (Parser_Error_Buf, "%s %s",
		   "ERROR -- EXPECTED ",
		   command_table[i].fields[n].field_descriptor);
	  return (PARSER_WRONG_PARMS);
	}
	break;
      case F_DEAL:
	error_flag = Parse_hand (parameter_hand + n);
	parameter_list[n] = (int *) (parameter_hand + n);
	if (error_flag) {
	  sprintf (Parser_Error_Buf, "%s", "ERROR IN DEAL");
	  return (PARSER_WRONG_PARMS);
	}
	break;
      default:
	printf ("Internal Parser Error !!\n");
	printf ("Input Buffer: %s\n", buffer);
	printf ("Command Name: %s\n", command_table[i].command_name);
	printf ("Parameter No: %d\n", n);
	printf ("Field Type:   %d\n", 
		command_table[i].fields[n].field_type);
	printf ("Field Desc:   %s\n",
		command_table[i].fields[n].field_descriptor);
	sprintf (Parser_Error_Buf, "Internal Error!");
	return (PARSER_WRONG_PARMS);
      }
    }
    n++;
  }

  Skip_whitespace ();
  if (NOW != '\0') {
    sprintf (Parser_Error_Buf, "TOO MANY PARAMETERS");
    return (PARSER_WRONG_PARMS);
  }

  /* At this point, we have now parsed the command successfully,
     so we need only call the procedure with the appropriate number
     of parameters. 
  */
  if (ex) switch (n) {
  case 0: 
    command_table[i].chandler ();
    break;
  case 1:
    command_table[i].chandler (parameter_list[0]);
    break;
  case 2:
    command_table[i].chandler (parameter_list[0],parameter_list[1]);
    break;
  case 3:
    command_table[i].chandler (parameter_list[0], parameter_list[1],
			       parameter_list[2]);
    break;
  }
  return (PARSER_SUCCESS);
}

int Parse_Command(command_table, table_length, buffer)
     Command_Descriptor *command_table; int table_length; char *buffer;
/* int Parse_Command (Command_Descriptor *command_table, 
   int table_length, char *buffer); */
/* Reads a command from the buffer and attempts to parse it according
   to the given table.  If successful, then the command specific procedure
   is called with the parsed parameters.  In this case, 
   zero is returned.  If unsuccessful parse, records some error information in 
   Parser_Error_Buf and Parser_Error_Location and returns the parser error code.
*/
{
   return Parse_Command_May_Execute (command_table, table_length, buffer, 1);
}

int Parse_Command_Dont_Execute(command_table, table_length, buffer)
     Command_Descriptor *command_table; int table_length; char *buffer;
/* int Parse_Command (Command_Descriptor *command_table, 
   int table_length, char *buffer); */
/* Reads a command from the buffer and attempts to parse it according
   to the table.  If a successful parse, then zero is returned.  If 
   unsuccessful parse, records some error information in Parser_Error_Buf 
   and Parser_Error_Location and returns the parser error code.
*/
{
   return Parse_Command_May_Execute (command_table, table_length, buffer, 0);
}


