/* network.h
 *
 ! Copyright (C) 1990-1992 by Matthew Clegg.  All Rights Reserved
 ! 
 ! OKbridge is made available as a free service to the Internet.
 ! Accordingly, the following restrictions are placed on its use:
 ! 
 ! 1.  OKbridge may not be modified in any way without the explicit 
 !     permission of Matthew Clegg.  
 ! 
 ! 2.  OKbridge may not be used in any way for commercial advantage.
 !     It may not be placed on for-profit networks or on for-profit
 !     computer systems.  It may not be bundled as part of a package
 !     or service provided by a for-profit organization.
 ! 
 ! If you have questions about restrictions on the use of OKbridge,
 ! write to mclegg@cs.ucsd.edu.
 ! 
 ! DISCLAIMER:  The user of OKbridge accepts full responsibility for any
 ! damage which may be caused by OKbridge.
 *
 * The network module provides an interface between the network and
 * the bridge program which is mostly independent of whether or not
 * the local player is the server or a client.  In version 1.6, the
 * network module has much more intelligence -- i.e., this module
 * now parses all incoming messages and even processes some of them,
 * only passing up to the input module those messages which are relevant
 * to the play.
 *
 */  

#ifndef TYPES_INCLUDED
#include "types.h"
#endif

#ifndef PROTOCOL_INCLUDED
#include "protocol.h"
#endif

#ifndef BOARDS_INCLUDED
#include "boards.h"
#endif

#ifndef SOCKET_INCLUDED
#include "socket.h"
#endif

#ifndef SCORING_INCLUDED
#include "scoring.h"
#endif
 
#define NETWORK_INCLUDED

/* The following definitions define the possible states of a connection. */
#define CSTATE_CONNECTED  1  /* Connected but waiting for handshakes to
				complete. */
#define CSTATE_PLAYING    2  /* Playing or ready to play. */


#define DEFAULT_PORT     1729 /* The default port to which we will try
				 to connect. */

typedef struct Message_struct {
  struct Message_struct *next;
  struct Connection_struct *source;
    /* The connection from which the message was received. */
  int    privat;
    /* TRUE if this message should not be forwarded to the other players. */
  int    loopback;
    /* TRUE if this message originated from the local player. */
  struct player_command_struct p;
} *Message;

typedef struct Message_queue_struct {
  Message head, tail;
} *message_queue;

/* If we are in server mode, then we maintain a Connection structure for
   each client connection.  All of the Connection structures are maintained
   on a global input list which is continuously monitored for incoming
   messages.  In addition, each Connection structure will be on an
   output list which is determined by the table and seat occupied by
   the player.  All of the connection lists contain dummy header elements,
   so that a connection may be deleted from a list without reference to
   the head of the list.
*/

/* typedef struct Table_struct; */

typedef struct Connection_struct {
  int  channel;            /* The socket descriptor for communication */
  message_queue incoming;  /* A queue of incoming messages ... only the
			      first will be parsed. */
  Non_blocking_buffer *inbuf;  /* Incoming data yet to be processed. */
  Non_blocking_buffer *outbuf; /* Data pending on the output queue. */
  int  local;              /* True if this connection represents the local
			      player.  In this case, channel is meaningless */
  int  state;              /* The state of the connection, as above. */
  name_buffer player_name; /* The name of the player. */
  int         player_id;   /* A unique ID assigned by the GPS to the player. */
  message_buffer fullname; /* The full name of the player. */
  message_buffer email;    /* The email address of the player. */
  int  seat;               /* The seat assigned to the player. */
  int  spectator;          /* True if this player is in spectator mode. */
  struct Table_struct *table;  /* The table where this player is located */
  struct Connection_struct *iprev, *inext;
  struct Connection_struct *oprev, *onext;
} *Connection;

typedef struct Seat_struct {
  char  player_name[10];
  int   occupied;          /* A boolean flag indicating this seat is occupied*/
  int   skipping;          /* A boolean flag indicating that this player has
			      not yet acknowledge a skip request. */
  Connection connection;   /* NULL if there is no one sitting here. */
} *Seat;

typedef struct Table_struct {
  int table_no;                            /* A unique number identifying
					      this table. */
  struct Seat_struct Seats[PLAYER_TYPES];  /* The sitting players. */
  Connection    Players;                   /* The list of players,
					      including observers. */
  message_queue protocol_queue;            /* Pending messages on the
					      protocol queue. */
  message_queue conversation_queue;        /* Pending messages on the 
					      conversation queue. */
  message_queue game_queue;                /* Pending messages on the
					      game queue. */
  int    game_mode;                        /* The current state of the game:
					      STARTUP, BIDDING, PLAYING
					      or SCORING. */
  int  playing_mode;                       /* The current mode of play:
					      CLUB, PRACTICE or FORMAL. */
  int  scoring_mode;                       /* Scoring mode in use */
  Score score[2];                          /* Cumulative score records for
					      N-S and E-W. */
  Board  *board;                           /* The current board which is
					      being played. */
  Play_record *play_record;                /* The record of play for the
					      current board. */
  int   play_request_sequence_number;      /* A request to play a card is
					      granted only if its sequence
					      number matches this number. */
  struct Table_struct *next, *prev;
} *Table;

#define OCCUPIED(t,s)    (((s) < 4) && t->Seats[s].occupied)
#define VACANT(t,s)      !OCCUPIED(t,s)
#define PLAYER_NAME(t,s) Local_table->Seats[s].player_name
#define MODERATOR(t)     t->Players

#define CLUB(t)          ((t)->playing_mode == CLUB_PLAYING_MODE)
#define FORMAL(t)        ((t)->playing_mode == FORMAL_PLAYING_MODE)
#define PRACTICE(t)      ((t)->playing_mode == PRACTICE_PLAYING_MODE)
#define VUGRAPH(t)       ((t)->playing_mode == VUGRAPH_PLAYING_MODE)

#ifdef _NETWORK_

  Table Table_List = NULL;
    /* If we are the server, then a list of the currently playing tables.
       If we are a client, then a descriptor of the particular table
       where we are sitting. */

  Connection Connections = NULL;
    /* Connections which have been established. */

  Connection Local_Player_Connection = NULL;
    /* A dummy connection representing the local player. */

  Connection Shadow_Connections [4] = {NULL, NULL, NULL, NULL};
    /* A set of 4 shadow player connections which are used for representing
       the players in a VUGRAPH tournament. */

  int    server_mode = 0;
    /* A boolean flag which is true if we are acting as the server. */
  int    client_mode = 0;
    /* A boolean flag which is true if we are a client. */
  int    quiet_mode = 0;
    /* A boolean flag which if true indicates that we will not display
       JOIN, SPEC and QUIT messages. */

  char  *server_name = NULL;
    /* The server to which we will try to connect by default. */
  int    network_port = DEFAULT_PORT;
    /* The port number to be used for network communications. */

  char Server_Start_time [40];
    /* The time at which we entered server mode. */

  /* The following information is maintained in order so that we
     can quickly identify ourselves to the GPS server. */

  char Host_IP [60];
    /* The IP number of the local host, in ascii format. */
  int  Host_IP_is_known = 0;
    /* A boolean flag indicating that the host IP number (or name) is known. */
  char Host_name [100];
    /* The name of the local host. */
  char User_name [20];
    /* The login name of the local user. */
  char User_fullname [60];
    /* The full name of the local user, as taken from the password file. */

#else

  extern Table Table_List;
  extern Connection Connections;
  extern Connection Local_Player_Connection;
  extern Connection Shadow_Connections [];
  extern int server_mode, client_mode, quiet_mode, network_port;
  extern char *server_name;
  extern char Host_IP[], Host_name[], User_name[], User_fullname[],
              Server_Start_time[];
  extern int Host_IP_is_known;

#endif

/* Procedures for manipulating command queues: */

extern int message_available ();
/* extern int message_available (message_queue queue); */
/* Returns true if the given queue is nonempty. */

extern Message dequeue_message ();
/* extern Message dequeue_message (message_queue queue); */
/* Deletes the first message from the given queue and returns a pointer
   to that message. 
*/

extern void enqueue_message ();
/* extern void enqueue_message (message_queue queue, Message m); */
/* Appends m to the list of messages stored in the given message queue. */

extern Message allocate_message ();
/* extern Message allocate_message (void); */
/* Allocates a message structure and returns a pointer to the structure.
   Copies the message text mtest into the message structure. */

extern void deallocate_message ();
/* extern void deallocate_message (Message m); */
/* Returns the message structure m to the pool of free message structs. */

extern void clear_message_queue ();
/* extern void clear_message_queue (message_queue queue); */
/* Deletes all messages from the named queue. */

extern void clear_all_message_queues ();
/* extern void clear_all_message_queues (Table t); */
/* Clears all of the message queues for the table t. */



/*  SEND ROUTINES
 *
 *  The following routines provide various options for sending messages
 *  to the other players.  
 *
 *  The client and also any routine at the conversation level or above 
 *  will use exclusively the send_message procedure.  However, server
 *  routines in the cs module may need to use some of these other
 *  send routines. 
 */

extern Message send_message ();
/* void send_message (Table t, int msg_type, char *message); */
/* If in server mode, sends a message to each of the clients at table t.
   If in client mode, sends a message to the server.  Returns a pointer
   to the message which is put onto the protocol queue.
  
   NOTE 1: This is the main send routine, and should be used by all
   routines operating at the conversation level or higher.
 
   NOTE 2: As a side effect, the message sent by this routine is
   also queued onto the protocol queue of the table t.
*/

/* SEND Primitives.

   NOTE:  The following routines should only be used if we are in
   server mode.
*/

extern int server_send_unformatted ();
/* int server_send_unformatted (Connection c, char *message); */
/* Transmits the message to the given player.  Returns 0 if successful or 1
   if an error.  If an error occurs, then the connection is closed.

   NOTE:  If the connection c is a local connection, then the message
   is dropped.  To cause the message to be received at the local table,
   the routine send_private_message must be used.   
*/

extern int server_send ();
/* int server_send (Connection c, int source, char *player_name, 
     char *message); */
/* Formats a message and sends it across the given connection.
   Returns 0 if successful or 1 if an error.  If an error occurs,
   then the connection is closed.  If the connection c is that
   of the local player, then the message is dropped.  If this is
   not desired, then send_private_message should be used.
*/

extern void Relay_message ();
/* void Relay_message (Table t, Connection c, int msg_type, char *message); */
/* Relays the message from c to all of the other players at table t who
   are allowed to receive the message.  Does NOT put the message onto
   the local protocol queue.
*/

extern Message loopback_message_unformatted ();
/* void loopback_message_unformatted (Table t, Connection c, char *message); */
/* Constructs a message structure using the the given message.
   Assumes that the message is in the format which is used by the
   client in transmitting messages.  Appends the message structure
   to the protocol queue for the table t.  Returns a pointer to the
   message which is enqueued.
*/

extern Message loopback_message ();
/* void loopback_message (Table t, Connection c, char *message); */
/* Appends the message to the protocol queue of the table t. 
   Returns a pointer to the message which is enqueued.
*/ 

extern void server_broadcast_unformatted ();
/* void server_broadcast_unformatted (Table t, int msg_type, char *message); */
/* Broadcasts the message to all of the people sitting at table t who are
   entitled to receive the message.  This includes putting the message
   onto the local protocol queue if we are sitting at the table t.
*/

extern void server_broadcast ();
/* void server_broadcast (Table t, int source, msg_type, char *player_name, 
     char *message); */
/* Formats and broadcasts the message to all of the people sitting at table t
   who are entitled to receive the message.  Puts the message onto the
   local protocol queue if we are sitting at the table t.
*/

extern void send_server_message ();
/* void send_server_message (Table t, int msg_type, char *message); */
/* Sends a message which originates from the 'SERVER'. */

extern void send_private_message ();
/* void send_private_message (Connection c, char *message); */
/* Only to be used in server mode.  Sends a message from the 'server'
   to the connection c.  If the connection c is the local player,
   then appends the message to the protocol queue of the table of
   the local player.
*/


/* The following routines are for transmitting and receiving boards and
   play records.  These routines are special in that boards and
   play records are transmitted and received as in-line data, and so
   are not handled by the normal parsing mechanisms.
*/

extern void Transmit_board ();
/* void Transmit_board (Table t, Board *b); */
/* If we are the server, then transmits the board b to each of the players
   at the table t.  If we are a client, then transmits the board b to
   the server. */

extern void Transmit_play_record ();
/* void Transmit_play_record (Table t, Play_record *p); */
/* If we are the server, then transmits the play record p to each of the
   players at the table t.  If we are a client, then transmits the play
   record p to the server. */

extern void Relay_board ();
/* void Relay_board (Table t, Connection c, Board *b); */
/* Relays the in-line board data from the connection c to the other players
   at the table t.
*/

extern void Relay_play_record ();
/* void Relay_play_record (Table t, Connection c, Play_record *p); */
/* Relays the in-line play record from the connection c to the other players
   at the table t.
*/


/* Top level routines for accessing the network: */ 

extern void Initialize_Network ();
/* Initializes the network data structures. */

extern void Attempt_to_connect ();
/* Attempts to connect to the server named above as server_name at
   the port specified above as network_port.  
*/

extern void Setup_server ();
/* Sets up the local player as a server.  Attempts to use the specified
   port, but if this does not work, then a new port will be chosen.
*/

extern void Wait_for_network_event ();
/* Waits for an input event to occur.  If a network event occurs, then
   parses the message and places it onto the proper player queue.
*/

extern void close_connection ();
/* void close_connection (Connection c) */
/* Closes the connection to c. */

extern void Close_all_connections ();
/* Closes all current network connections. */

extern void Switch_Table ();
/* void Switch_Table (Connection c, Table new_table); */
/* Unlinks the connection c from the current table and links it into
   the new table.
*/

/* The following two macros are for iterating through the set of all
   connections and through the set of all players at a given table,
   respectively.
*/

#define FOREACH_CONNECTION(x) for(x = Connections->inext; x != NULL; x = x->inext)
#define FOREACH_PLAYER(x,t)   for(x = t->Players->onext; x != NULL; x = x->onext)
