/* log.c -- module for recording boards to log files.
 *
 ! Copyright (C) 1990-1992 by Matthew Clegg.  All Rights Reserved
 ! 
 ! OKbridge is made available as a free service to the Internet.
 ! Accordingly, the following restrictions are placed on its use:
 ! 
 ! 1.  OKbridge may not be modified in any way without the explicit 
 !     permission of Matthew Clegg.  
 ! 
 ! 2.  OKbridge may not be used in any way for commercial advantage.
 !     It may not be placed on for-profit networks or on for-profit
 !     computer systems.  It may not be bundled as part of a package
 !     or service provided by a for-profit organization.
 ! 
 ! If you have questions about restrictions on the use of OKbridge,
 ! write to mclegg@cs.ucsd.edu.
 ! 
 ! DISCLAIMER:  The user of OKbridge accepts full responsibility for any
 ! damage which may be caused by OKbridge.
 *
 */
#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>
#ifdef AIX
#include <time.h>
#endif

#ifdef SCO
#include <time.h>
#endif

#include "types.h"
#include "boards.h"
#include "state.h"

#ifdef GCC
extern time_t time ();
extern fprintf ();
extern fflush ();
#endif

extern void qsort ();
extern int  strlen ();
extern void bcopy ();

typedef int (* play_comparator) ();
typedef int sorted_play_array [4][13];

typedef char        output_line[81];
typedef output_line picture_buffer [30];

static Play_record *Play_record_buffer [1000];
picture_buffer Play_picture;

void Write_summary_of_play ();
void Sort_play_records_by_matchpoints ();
void Sort_play_records_by_imps ();

int sort_direction = 1;
  /* This variable should be either 1 or -1.  If it is 1, then
     scores are sorted in decreasing order by N-S.  If it is -1,
     then scores are sorted in decreasing order by E-W.
  */

static void format_suit (buffer, h, p, s)
/* Searches the hand h for cards held by the player p in the suit s.
   For each such card found, places an ascii character representing its
   rank into the buffer.
*/
     char *buffer; hand h; int p, s;
{
  int i;

  for (i = 12; i >= 0; i--)
    if (h[card_code(s, i)] == p)
      *(buffer++) = *rank_names[i];
  *buffer = '\0';
}

static void Sort_cards_by_trick 
  (trumps, leader, plays, no_plays, s, leaders)
     int trumps;
     int leader;
     card_type *plays;
     int no_plays;
     sorted_play_array s;
     int *leaders;
/* Sorts the set of plays for a hand by trick and player.  */
{
  int player = leader;
  int trick_no = 0;
  int i, j;

  for (i = 0; i < 4; i++)
    for (j = 0; j < 13; j++)
      s[i][j] = -1;

  leaders[0] = leader;
  for (i = 0; i < no_plays; i++) {
    s[player][trick_no] = plays[i];
    player = player_next[player];
    if (i % 4 == 3) {
      player =
	Winning_card (trumps, player,
		      s[player][trick_no], 
		      s[player_next[player]][trick_no],
		      s[player_partner[player]][trick_no],
		      s[player_prev[player]][trick_no]);
      if (trick_no < 12)
	leaders[++trick_no] = player;
    }
  }
}

static void Clear_picture ()
{
  int i, j;

  for (i = 0; i < 30; i++)
    for (j = 0; j < 81; j++)
      Play_picture[i][j] = ' ';

}

static void pprint (row, col, s)
     int row, col; char *s;
{
  int len;

  len = strlen(s);
  if (col + len > 79)
    len = 79 - col;

  bcopy (s, &(Play_picture[row][col]), len);
}

void Flush_picture (fp, lines)
     FILE *fp; int lines;
{
  int i;

  for (i = 0; i < lines; i++)
    fprintf (fp, "%s\n", Play_picture[i]);
}

static void Print_hand (y, x, seat_name, player_name, b, position)
     int y, x; char *seat_name, *player_name; Board *b; int position;
{
  char buf[100];

  pprint (y + 0, x, seat_name);
  pprint (y + 1, x, player_name);
  pprint (y + 2, x, "S");
  pprint (y + 3, x, "H");
  pprint (y + 4, x, "D");
  pprint (y + 5, x, "C");

  format_suit (buf, b->deal, position, SUIT_SPADES);
  pprint (y + 2, x + 2, buf);
  format_suit (buf, b->deal, position, SUIT_HEARTS);
  pprint (y + 3, x + 2, buf);
  format_suit (buf, b->deal, position, SUIT_DIAMONDS);
  pprint (y + 4, x + 2, buf);
  format_suit (buf, b->deal, position, SUIT_CLUBS);
  pprint (y + 5, x + 2, buf);
}

static char *lc_seat_names [] = {"North", "East", "South", "West"};

static char *lc_seat_letters [] = {"N", "E", "S", "W"};

int Create_Picture_of_Hand (b, p)
     Board *b; Play_record *p;
/* Creates an image of the board b and play record p in the picture array.
 * Returns the number of lines used in the array. 
 */
{
  char buf[100];
  int i, j;

  /* We always make the declarer appear as south: */
  int vsouth = p->declarer;
  int vwest  = player_next[p->declarer];
  int vnorth = player_partner[p->declarer];
  int veast  = player_prev[p->declarer];

  char *north_name = p->player_names[vnorth],
       *west_name  = p->player_names[vwest],
       *south_name = p->player_names[vsouth],
       *east_name  = p->player_names[veast],
       *player_names[4];

  sorted_play_array sorted_plays;
    /* The list of cards each played by each player, indexed by trick no. */
  int leader[13];
    /* For each trick, the index of the player who was on lead. */
  int follow;
    /* Boolean flag indicating if the current player won the trick. */
  int round;
    /* The current round of bidding. */
  int contractor = side_of (p->declarer);
    /* The contracting side. */
  int defender = 1 - contractor;
    /* The defending side. */

  int bidder, player;
  int tricks_played = ((p->no_plays - 1) / 4) + 1;

  int lhs = 0;   /* the left border of the display of the hand. */
  int rhs = 42;  /* the right hand side of the display of the hand. */

  int cursorx, cursory;
    

  player_names[PLAYER_NORTH] = north_name;
  player_names[PLAYER_EAST]  = east_name;
  player_names[PLAYER_WEST]  = west_name;
  player_names[PLAYER_SOUTH] = south_name;

  Clear_picture ();

  switch (b->scoring_mode) {
  case RUBBER_SCORING:
    pprint (0, lhs, "Rubber");
    break;
  case DUPLICATE_SCORING:
    pprint (0, lhs, "Duplicate");
    break;
  case IMP_SCORING:
    pprint (0, lhs, "IMPs");
    break;
  case MP_SCORING:
    pprint (0, lhs, "Match Points");
    break;
  }
  sprintf (buf, "%s Board %d", b->source, b->serial_no);
  pprint (0, rhs, buf);

  if (b->dealer == PLAYER_SOUTH)
    pprint (1, lhs, "Dlr: South");
  else if (b->dealer == PLAYER_NORTH)
    pprint (1, lhs, "Dlr: North");
  else if (b->dealer == PLAYER_EAST)
    pprint (1, lhs, "Dlr: East");
  else
    pprint (1, lhs, "Dlr: West");

  if (b->vulnerable[SIDE_NS] && b->vulnerable[SIDE_EW])
    pprint (2, lhs, "Vul: Both");
  else if (b->vulnerable[SIDE_NS])
    pprint (2, lhs, "Vul: N-S");
  else if (b->vulnerable[SIDE_EW])
    pprint (2, lhs, "Vul: E-W");
  else
    pprint (2, lhs, "Vul: None");

  if (b != Local_board || revealed_hands[vnorth])
    Print_hand (0, lhs + 16, lc_seat_names[vnorth], north_name, b, vnorth);
  if (b != Local_board || revealed_hands[vwest])
    Print_hand (6, lhs +  4, lc_seat_names[vwest],  west_name,  b, vwest);
  if (b != Local_board || revealed_hands[veast])
    Print_hand (6, lhs + 27, lc_seat_names[veast],  east_name,  b, veast);
  if (b != Local_board || revealed_hands[vsouth])
    Print_hand (12,lhs + 16, lc_seat_names[vsouth], south_name, b, vsouth);

  cursory = 5;

  if (p->bidding_completed) {
    for (i = 0; i < 4; i++)
      sprintf (&buf[i*9], "%-8s ",lc_seat_names[(b->dealer+i) % 4]);
    pprint (2, rhs, buf);
    for (i = 0; i < 4; i++)
      sprintf (&buf[i*9], "%-8s ",p->player_names[(b->dealer+i) % 4]);
    pprint (3, rhs, buf);

    round = -1;
    bidder = b->dealer;
    cursorx = rhs;
    for (i = 0; i < p->no_bids - 2;) {
      if (bidder == b->dealer)
	round++;
      if (round < 0)
	pprint (cursory, cursorx, "--");
      else if (i >= p->no_bids - 3) {
	pprint (cursory, cursorx, "(all pass)");
	i++;
      } else {
	sprintf (buf, "%s%c", 
		 (p->bids[i] == BID_PASS)? "pass": bid_names[p->bids[i]],
		 p->alerts[i]? '*': ' ');
	pprint (cursory, cursorx, buf);
	i++;
      }
      bidder = player_next[bidder];
      if (bidder == b->dealer) {
	cursorx = rhs;
	cursory++;
      } else
	cursorx += 9;
    }
    if (bidder != b->dealer)
      cursory += 1;

    if (p->no_plays > 0) {
      sprintf (buf, "Opening Lead: %s",
	       card_names[(int) (p->play_list[0])]);
      pprint (++cursory, rhs, buf);
    }
    if (p->hand_completed) {
      sprintf (buf, "Result: %+2d", p->result);
      pprint (++cursory, rhs, buf);
      switch (b->scoring_mode) {
      case RUBBER_SCORING:
	sprintf (buf, "Above: %d, Below: %d", 
		 p->score[contractor].rubber.above_line -
		 p->score[defender].rubber.above_line,
		 p->score[contractor].rubber.below_line -
		 p->score[defender].rubber.below_line);
	break;
      case DUPLICATE_SCORING:
	sprintf (buf, "Score: %d",
		 p->score[contractor].duplicate.total -
		 p->score[defender].duplicate.total);
	break;
      case MP_SCORING:
	sprintf (buf, "Score: %d, Match Points: %4.2f%c",
		 p->score[contractor].duplicate.total -
		 p->score[defender].duplicate.total,
		 100.0 * Percent_Matchpoints(&p->score[contractor]), '%');
	break;
      case IMP_SCORING:
	sprintf (buf, "Score: %d,  IMPs: %5.1f",
		 p->score[contractor].duplicate.total -
		 p->score[defender].duplicate.total,
	         Average_Imps(&p->score[contractor]));
	break;
      }
      pprint (++cursory, rhs, buf);
    }
    cursory++;
  }
  
  if (cursory < 14)
    cursory = 14;
  else
    cursory++;

  if (p->hand_completed && (p->no_plays > 0)) {
    Sort_cards_by_trick (trumpsuit_of(p->contract), player_next[p->declarer],
			 p->play_list, p->no_plays, sorted_plays, leader);

    cursorx = 80 - tricks_played * 3;
    if (cursorx-2 > rhs)
      cursorx = rhs + 2;

    for (i = 0; i < tricks_played; i++) {
      sprintf (buf, "%2d", i+1);
      pprint (cursory,   cursorx + 3 * i, buf);
/*      pprint (cursory+1, cursorx + 3 * i, "---"); */
    }
    cursory += 1;
    pprint (cursory + 0, cursorx - 2, lc_seat_letters[vwest]);
    pprint (cursory + 1, cursorx - 2, lc_seat_letters[vnorth]);
    pprint (cursory + 2, cursorx - 2, lc_seat_letters[veast]);
    pprint (cursory + 3, cursorx - 2, lc_seat_letters[vsouth]);

    player = vwest;
    for (i = 0; i < 4; i++) {
      for (j = 0; j < 13; j++) {
	if (sorted_plays[player][j] != -1) {
	  if ((j < tricks_played-1) && (leader[j+1] == player))
	    follow = '-';
	  else
	    follow = ' ';
	  sprintf (buf, "%2s%c", 
		   card_names[sorted_plays[player][j]], follow);
	  pprint (cursory + i, cursorx + 3 * j, buf);
	}
      }
      player = player_next [player];
    }
  }

  if (p->no_bids < 8)
    cursory += 5;
  else
    cursory += 4;

  for (i = 0; i < cursory; i++)
    for (j = 80; (Play_picture[i][j] == ' ') && (j > 0); --j)
      Play_picture[i][j] = '\0';

  return (cursory);
}

void Write_hand (logfile, b, p, with_records)
     FILE *logfile;
     Board *b;
     Play_record *p;
     int with_records;
/* Writes the board b with play record p to the file f in an expanded
 * format which is similar to that found in bridge books.  If with_records
 * is true, then the play records are written in addition.
 */
{
  int i;
  int picture_length;

  /* We always make the declarer appear as south: */
  int vsouth = p->declarer;
/*  int vwest  = player_next[p->declarer]; */
/*  int vnorth = player_partner[p->declarer]; */
  int veast  = player_prev[p->declarer];

  if (logfile == NULL) return;

  picture_length = Create_Picture_of_Hand (b, p);

  if (picture_length < 18)
    Flush_picture (logfile, 18);
  else
    Flush_picture (logfile, picture_length);

  if (p->hand_completed) {
    fprintf (logfile, "\n");
    switch (b->scoring_mode) {
    case RUBBER_SCORING:
      fprintf (logfile, "%-10s %8s %8s\n", " ", "N-S", "E-W");
      fprintf (logfile, "%-10s %8s %8s\n", " ", "------", "------");
      fprintf (logfile, "%-10s %8d %8d\n", "Above",
	       p->score[side_of(vsouth)].rubber.above_line,
	       p->score[side_of(veast)].rubber.above_line);
      fprintf (logfile, "%-10s %8s %8s\n", " ", "------", "------");
      fprintf (logfile, "%-10s %8d %8d\n", "Part Score",
	       b->part_score[side_of(vsouth)], b->part_score[side_of(veast)]);
      fprintf (logfile, "%-10s %8d %8d\n", "Below",
	       p->score[side_of(vsouth)].rubber.below_line,
	       p->score[side_of(veast)].rubber.below_line);
      fprintf (logfile, "\n");
      break;

    case DUPLICATE_SCORING:
    case MP_SCORING:
      if (with_records) {
	Sort_play_records_by_matchpoints (b);
	Write_summary_of_play (logfile, b);
      }
      break;
      
    case IMP_SCORING:
      if (with_records) {
	Sort_play_records_by_imps (b);
	Write_summary_of_play (logfile, b);
      }
      break;
    }
  }

  fprintf (logfile, "\n\n");
  for (i = 0; i < 79; i++)
    fprintf (logfile, "=");
  fprintf (logfile, "\n%c\n", 12);
  fflush (logfile);
}


void Write_hand_compactly (zhang_logfile, table_no, b, p)
     FILE *zhang_logfile;
     int table_no;
     Board *b;
     Play_record *p;
/* Writes the board b with play record p to the file f in a brief format
 * which was introduced by S. Zhang.
 */
{ 
  int i, j;
  char format_buffer [100];
  char *bid_zhang[4][20];
  char zhang[4];
  time_t current_time;
  struct tm *decoded_time;
  sorted_play_array sorted_plays;
  int leading_play [13];
  int contractor = side_of (p->declarer);
  int trump_suit = trumpsuit_of (p->contract);
  int down, excess, round, lead, bidder;
  int contract = level_of (p->contract);

  zhang[0]='N'; zhang[1]='E';  zhang[2]='S'; zhang[3]='W';

  if (zhang_logfile == NULL) return;
  for (i = 0; i < 4; i++){fprintf(zhang_logfile, 
         " %-8.8s ", p->player_names[i]);
    for (j = 3; j >= 0; j--) {
      format_suit (format_buffer, b->deal, i, j);
      fprintf (zhang_logfile, " %-13s", format_buffer);
    }    fprintf (zhang_logfile, "\n");
  }

  fprintf(zhang_logfile, "%3d.(tab%3d)", b->serial_no, table_no);

  fprintf (zhang_logfile,"   %6.6s-%c:", p->player_names[p->declarer],
              zhang[p->declarer]);   
  fprintf (zhang_logfile, "%d%1.1s", 
	   contract, long_suit_names [trump_suit] );  

  if (p->doubled == 2)
    fprintf (zhang_logfile, "XX"); 
  else if (p->doubled)
    fprintf (zhang_logfile, "X ");
  else  
    fprintf (zhang_logfile, "  ");

  if (p->tricks[contractor] >= contract+6) {
    fprintf (zhang_logfile, " -- Made %2d  |  ", 
	     (contract == 0)? 0: p->tricks[contractor]-6);
  } else { down = contract + 6 - p->tricks[contractor];
    fprintf (zhang_logfile, " -- Down %2d  |  ", down);
  }

  if (b->vulnerable[SIDE_NS] && b->vulnerable[SIDE_EW])  
                                   fprintf (zhang_logfile, "Vul:BOTH |  ");
  else if (b->vulnerable[SIDE_NS]) fprintf (zhang_logfile, "Vul:N&S  |  ");
  else if (b->vulnerable[SIDE_EW]) fprintf (zhang_logfile, "Vul:E&W  |  ");
  else                             fprintf (zhang_logfile, "Vul:NONE |  ");


  if (p->tricks[contractor] >= contract+6) {
    if ( contractor == SIDE_NS )  {fprintf (zhang_logfile, " NS:  ");} 
                             else  fprintf (zhang_logfile, "*NS: -");
    excess = p->tricks[contractor] - contract - 6;
    if (contract == 0)
      fprintf (zhang_logfile, "0  ");
    else
      fprintf (zhang_logfile, "%d ", 
	       Duplicate_score_made 
	         (b->vulnerable[contractor], level_of(p->contract),
		  trumpsuit_of(p->contract), p->doubled, p->result));
  } else {
    down = contract + 6 - p->tricks[contractor];
    if ( contractor == SIDE_NS )  {fprintf (zhang_logfile, " NS: -");} 
                             else  fprintf (zhang_logfile, "*NS:  ");
    fprintf (zhang_logfile, "%d ", 
	     Duplicate_score_set
	     (b->vulnerable[contractor], level_of(p->contract),
	      trumpsuit_of(p->contract), p->doubled, p->result));
  }

  for (i=0; i<4; i++) for (j=0; j<20; j++) bid_zhang[i][j]="  ";

  bidder = b->dealer;
  round = 0;
  for (i = 0; i < p->no_bids; i++) {
    bid_zhang[bidder][round] = bid_names[p->bids[i]];
    bidder = player_next[bidder];
    if (bidder == PLAYER_NORTH) round++;
  }

  Sort_cards_by_trick (trumpsuit_of(p->contract), player_next[p->declarer],
		       p->play_list, p->no_plays, sorted_plays, leading_play);
  for (i = 0; i < 4; i++) {
    fprintf(zhang_logfile,"\n%c-%-8.8s",zhang[i],p->player_names[i]);
    for (j = 0; j < 13; j++) {
      if (sorted_plays[i][j] != -1) {
	lead = (leading_play[j] == i)? '*': ' ';
	if ((lead ==' ')&&(i==3) ) lead='_';
	fprintf (zhang_logfile, "%2s%c", card_names[sorted_plays[i][j]], lead);
      } else
	fprintf (zhang_logfile, "   ");
    } fprintf (zhang_logfile, "|");
    for (j = 0; j <= p->no_bids/4; j++) {
      fprintf (zhang_logfile, "%-2.2s ", bid_zhang[i][j]);
    }
  }

  fprintf (zhang_logfile, "\n==========");
  for (i = 0; i < 13; i++)    fprintf (zhang_logfile, "%2d=", i+1);
  time (&current_time);
  for (i = 0; i < 6; i++)     fprintf (zhang_logfile, "=%2d", i+1);
  decoded_time = localtime (&current_time);    
  fprintf (zhang_logfile, " %s %2d\n%c\n", 
	   month_names[decoded_time->tm_mon], decoded_time->tm_mday, 12);
  fflush (zhang_logfile);
}

void sprint_summary_header (buf)
     char *buf;
{
  sprintf (buf, "%-11s %4s %2s %-4s %-21s %-6s %2s %2s %3s %-5s %5s %4s",
	   "Board", "No", "Dl", "Vul", "Players", "Cntrct", 
	   "By", "Ld", "Res", "Score", "IMPs", "MPs");
}

static char *seat_letters [4] = {"N", "E", "S", "W"};
static char *double_names [3] = {"", "-X", "-XX"};

void sprint_summary_record (buf1, buf2, b, p)
     char *buf1;
     char *buf2;
     Board *b;
     Play_record *p;
{
  char *vul_string;
  char contract_buf[10];

  if (b->vulnerable[SIDE_NS] && b->vulnerable[SIDE_EW])
    vul_string = "both";
  else if (b->vulnerable[SIDE_NS])
    vul_string = "N-S";
  else if (b->vulnerable[SIDE_EW])
    vul_string = "E-W";
  else
    vul_string = "none";

  if (p->contract == BID_PASS)
    sprintf (contract_buf, "Passed");
  else
    sprintf (contract_buf, "%s%s", bid_names[p->contract], 
	     double_names[p->doubled]);

  sprintf (buf1, 
	   "%-11s %4d %-2s %-4s N %-8s S %-8s %-6s %-2s %-2s %+3d %5d %5.1f %4.0f",
	   b->source, b->serial_no, seat_letters[b->dealer], vul_string, 
	   p->player_names[PLAYER_NORTH], p->player_names[PLAYER_SOUTH],
	   contract_buf, seat_letters[p->declarer], 
	   (p->no_plays == 0)? " ": card_names[p->play_list[0]],
	   p->result,
	   p->score[SIDE_NS].duplicate.total,
	   Average_Imps(&p->imatch_points[SIDE_NS]),
	   100.0 * Percent_Matchpoints(&p->match_points[SIDE_NS]));

  sprintf (buf2,
	   "%-11s %4s %2s %-4s E %-8s W %-8s %-6s %2s %2s %3s %5d %5.1f %4.0f",
	   " ", " ", " ", " ", 
	   p->player_names[PLAYER_EAST], p->player_names[PLAYER_WEST],
	   " ", " ", " "," ",
	   p->score[SIDE_EW].duplicate.total,
	   Average_Imps(&p->imatch_points[SIDE_EW]),
	   100.0 * Percent_Matchpoints(&p->match_points[SIDE_EW]));

}

void Write_summary_of_play (f, b)
     FILE *f;
     Board *b;
/* For each play record associated to the board b, writes a single line
 * summary describing the result of play.
 */
{
  char buf1 [100], buf2[100];
  Play_record *p;

  sprint_summary_header (buf1);
  fprintf (f, "%s\n", buf1);

  for (p = b->play_records; p != NULL; p = p->next) {
    sprint_summary_record (buf1, buf2, b, p);
    fprintf (f, "%s\n", buf1);
    fprintf (f, "%s\n", buf2);
  }
  fprintf (f, "\n\n");
}

static void Sort_play_records (b, pc)
     Board *b;
     play_comparator pc;
{
  int i, n = 0;
  Play_record *p;

  if ((b->play_records == NULL) || (b->play_records->next == NULL))
    return;

  for (p = b->play_records; p != NULL; p = p->next)
    Play_record_buffer[n++] = p;

  qsort (Play_record_buffer, n, sizeof(Play_record *), pc);

  p = b->play_records = Play_record_buffer[0];
  for (i = 1; i < n; i++)
    p = (p->next = Play_record_buffer[i]);
  p->next = NULL;
}


static int sort_contract_lead (p1,p2)
     Play_record *p1;
     Play_record *p2;
{
  static char *double_names [3] = {"", "-X", "-XX"};
  char contract_buf1[10];
  char contract_buf2[10];
  int d;

  if (p1->contract == BID_PASS)
    sprintf (contract_buf1, "PASSED");
  else
    sprintf (contract_buf1, "%s%s", bid_names[p1->contract],
             double_names[p1->doubled]);

  if (p2->contract == BID_PASS)
    sprintf (contract_buf2, "PASSED");
  else
    sprintf (contract_buf2, "%s%s", bid_names[p2->contract],
             double_names[p2->doubled]);

  d = strcmp(contract_buf1,contract_buf2);
  if (d > 0)
    return (sort_direction);
  else if (d < 0)
    return (-sort_direction);

  d = strcmp((p1->no_plays == 0)? " ": card_names[p1->play_list[0]],
             (p2->no_plays == 0)? " ": card_names[p2->play_list[0]]);
  if (d > 0)
    return (sort_direction);
  else if (d < 0)
    return (-sort_direction);
  else
    return (0);
}


static int match_point_comparator (q1, q2)
     Play_record **q1;
     Play_record **q2;
{
  float d1, d2;
  Play_record *p1 = *q1;
  Play_record *p2 = *q2;
  int d;

  if (!p1->hand_completed && !p2->hand_completed)
    return (0);
  else if (!p2->hand_completed)
    return (-1);
  else if (!p1->hand_completed)
    return (1);

  d1 = Percent_Matchpoints(&p1->match_points[SIDE_NS]);
  d2 = Percent_Matchpoints(&p2->match_points[SIDE_NS]);

  if (d2 < d1)
    return (-sort_direction);
  else if (d1 < d2)
    return (sort_direction);
  else
    return (sort_contract_lead(p1,p2));
}


void Sort_play_records_by_matchpoints (b)
     Board *b;
/* Sorts the set of play records for the board b according to the match
 * point scores.
 */
{
  Sort_play_records (b, match_point_comparator);
}

static int imp_comparator (q1, q2)
     Play_record **q1;
     Play_record **q2;
{
  float d1, d2;
  Play_record *p1 = *q1;
  Play_record *p2 = *q2;
  int sp, sq;

  if (!p1->hand_completed && !p2->hand_completed)
    return (0);
  else if (!p2->hand_completed)
    return (-1);
  else if (!p1->hand_completed)
    return (1);

  d1 = Average_Imps(&p1->imatch_points[SIDE_NS]);
  d2 = Average_Imps(&p2->imatch_points[SIDE_NS]);
  sp = p1->imatch_points[SIDE_NS].duplicate.total -
    p1->imatch_points[SIDE_EW].duplicate.total;
  sq = p2->imatch_points[SIDE_NS].duplicate.total -
    p2->imatch_points[SIDE_EW].duplicate.total;

  if (d2 < d1)
    return (-sort_direction);
  else if (d1 < d2)
    return (sort_direction);
  else if (sq < sp)
    return (-sort_direction);
  else if (sp < sq)
    return (sort_direction);
  else
    return (sort_contract_lead(p1,p2));
}

void Sort_play_records_by_imps (b)
     Board *b;
/* Sorts the set of play records for the board b according to the
 * imp scores.
 */
{
  Sort_play_records (b, imp_comparator);
}

