/* input.h -- interface for input module of bridge game.
 *
 ! Copyright (C) 1990-1992 by Matthew Clegg.  All Rights Reserved
 ! 
 ! OKbridge is made available as a free service to the Internet.
 ! Accordingly, the following restrictions are placed on its use:
 ! 
 ! 1.  OKbridge may not be modified in any way without the explicit 
 !     permission of Matthew Clegg.  
 ! 
 ! 2.  OKbridge may not be used in any way for commercial advantage.
 !     It may not be placed on for-profit networks or on for-profit
 !     computer systems.  It may not be bundled as part of a package
 !     or service provided by a for-profit organization.
 ! 
 ! If you have questions about restrictions on the use of OKbridge,
 ! write to mclegg@cs.ucsd.edu.
 ! 
 ! DISCLAIMER:  The user of OKbridge accepts full responsibility for any
 ! damage which may be caused by OKbridge.
 *
 * This file defines the routines for handling the player-input
 * in the bridge game.
 *
 */

/* The input module handles keyboard input.  This includes reading
 * characters from the keyboard, assembling them into buffers, and
 * creating messages from completed input requests.
 *
 * There are three input buffers, called the talk buffer, the play
 * buffer and the query buffer.  The talk buffer gathers characters
 * for talk messages which will be transmitted to the other players.
 * The play buffer gathers bids and plays which will be transmitted
 * to the others.  And the query buffer gathers responses to yes/no
 * questions which may be asked at times.
 *
 * At any given time, one or more of the input buffers will be "active".
 * This means that characters can be added to the buffer.  There will
 * always be exactly one buffer which is "in focus".  This is the buffer
 * in which the next character will be placed.  Characters can be
 * erased from a buffer by pressing backspace or delete.  The entire
 * contents of a buffer can be erased by pressing escape (see Note).  A 
 * buffer will be processed when return is pressed.  And one can switch
 * between buffers by pressing return.  In addition, the following
 * characters have special meaning:
 *
 * ^A   Alerts partner's last bid.
 * ^B   When used during the play of a hand, the bidding is displayed
 *      for review.
 * ^C   Quits the program.  (The program requests confirmation first.)
 * ^D   Toggles the default input mode.  See the /DEFAULT command.
 * ^G   Toggles the bell.  See the /BELL command.
 * ^P   Toggles the prompt.  See the /PROMPT command.
 * ^R   Refreshes the screen.
 * ^T   Returns to talk mode if you have been asked to bid or play.
 * ^V	Reviews the cards during the bidding in vugraph mode.
 * ^W   Sends a wakeup signal to your partner.
 * ^X   Exits the program immediately (only if compiled in debug mode).
 *
 * Note:  The escape key actually has a dual meaning.  During normal input,
 * it will cause the buffer which is in focus to be erased.  But during
 * pause mode, it signals the program that the pause is over.
 *
 * The input processor may be placed in the following modes:
 *
 * TALK_INPUT:  Only the talk buffer is active.
 * BID_INPUT:   The talk and play buffers are active.  When return is
 *  pressed in the play buffer, the text is interpreted as a bid which
 *  if legal is transmitted to the other players.
 * PLAY_INPUT:  The talk and play buffers are active.  When return is
 *  pressed in the play buffer, the text is interpreted as a play which
 *  if legal is transmitted to the other players.
 * QUERY_INPUT: The talk and query buffers are active.  When return is
 *  is pressed in the query buffer, the text is interpreted as a yes/no
 *  answer.
 *
 * If the input text in the talk or play buffer begins with a slash '/',
 * then it is interpreted as a command.  Commands all begin with a
 * keyword and may be followed by one or more parameters, separated
 * by spaces.  For a list of the available commands, see the "command"
 * module.
 */

#define TALK_INPUT    0
#define BID_INPUT     1
#define PLAY_INPUT    2
#define QUERY_INPUT   3
#define ASK_INPUT     4

#ifdef _INPUT_
  int input_mode = TALK_INPUT;  /* The current input mode, as defined above. */
#else
  extern int input_mode;
#endif


void Initialize_Input ();
/* This routine should be called once when the program first begins,
 * in order to set up the input buffers correctly.
 */

void Reinitialize_Input ();
/* Clears all of the input buffers. */

void Set_Input_Mode ();
/* Sets the input mode to the given mode.  Redisplays the talk and
 * query buffers, if appropriate.  If the new mode is BID_INPUT (resp.
 * PLAY_INPUT), the set of legal bids (resp. plays) is computed and
 * the default bid (play) is also computed.
 */

void Refresh_Input_Buffers ();
/* Redisplays any input which may be present in the input buffers.
 * Places the cursor at the end of the current input buffer.
 */

void Clear_Focus_Buffer ();
/* Clears the focus buffer and places the cursor at the beginning of
 * the line.
 */

int Talking ();
/* Returns TRUE if the focus is currently on the talk buffer. */

void Compute_Default_Play ();
/* Computes the default play for the local player, based upon the informaion
 * contained in Local_board and Local_play. 
 */

void Clear_Default_Play ();
/* Clears a default play which may have been set earlier. */

void Install_Key ();
/* Install_Key (int keycode, char *key_mapping); */
/* Installs a mapping for a keyboard character.  If key_mapping
   is a string enclosed in double quotes, then it is sent as a 
   talk message when the corresponding key is pressed.  Otherwise,
   it is interpreted as a command.  The following additional
   special commands are available:
     CR     -- causes the current input line to be processed.
     BS     -- erases the last character from the input buffer.
     ERASE  -- causes the entire input buffer to be erased.
     META   -- causes the next key press to be interpreted as
               meta-key combination.
*/

void Install_Meta_Key ();
/* Install_Meta_Key (int keycode, char *key_mapping); */
/* Installs a mapping for a meta key.  The same conventions apply for
   the interpretation of the key_mapping as for install_key.
*/

void Initialize_Key_Mappings ();
/* Does the obvious thing. */

void Describe_Key ();
/* void Describe_Key (int keycode, char *describe_buf); */
/* Writes the current definition for the key to the buffer. */

void Describe_Meta_Key ();
/* void Describe_Meta_Key (int keycode, char *describe_buf); */
/* Write the current definition for the metakey to the buffer. */

void Write_Key_Mappings ();
/* void Write_Key_Mappings (file *f); */
/* Writes the set of non-default key mappings to the initialization
   file f.
*/

void Accept_Keyboard_Characters ();
/* If any keyboard characters are available, then reads them and adds
 * them to the current focus buffer.  This may result in a change of
 * state of the program or in messages being transmitted to the other
 * players.
 */

#ifdef VMS
#define Pause okPause
#endif

void Pause ();
/* Displays the given message on the status line, and waits for the
 * user to press the escape key.  Returns after escape has been pressed.
 */

int  Ask ();
/* Presents the question to the player and asks for a response.
 * Returns 1 if 'y' was entered and '0' otherwise.
 */

void Query ();
/* void Query (char *query_message, char *default_response, int echo_mode, 
	    char *buf, int buflen) */
/* Presents the query_message to the user on the talk line, and waits
   for the user to type a response.  If default_response is non-NULL,
   then this value is placed in the input buffer initially.  If echo_mode
   is false, then the characters typed are not displayed.  Returns up to
   buflen characters of response in buf.
*/

void Press_Return_to_Continue ();
/* Prints the message on the status line and then waits for the user
   to press return.
 */

int Reserved_message ();
/* Compares the given message to the list of card and bid names.  If a
   match is found, then returns true.  Otherwise, returns false.
   The purpose of this routine is to discourage players from sending
   talk messages which reveal intended bids or plays.
*/

typedef struct Key_Descriptor_Struct {
  char *name;   /* The name of the control key used by the KEY command. */
  char *key;    /* A pointer to the variable holding the current control char */
} Key_Descriptor;

#define END_INPUT_KEY_TABLE {NULL, NULL}
