/* input.c -- Input driver for the bridge program.
 *
 ! Copyright (C) 1990-1992 by Matthew Clegg.  All Rights Reserved
 ! 
 ! OKbridge is made available as a free service to the Internet.
 ! Accordingly, the following restrictions are placed on its use:
 ! 
 ! 1.  OKbridge may not be modified in any way without the explicit 
 !     permission of Matthew Clegg.  
 ! 
 ! 2.  OKbridge may not be used in any way for commercial advantage.
 !     It may not be placed on for-profit networks or on for-profit
 !     computer systems.  It may not be bundled as part of a package
 !     or service provided by a for-profit organization.
 ! 
 ! If you have questions about restrictions on the use of OKbridge,
 ! write to mclegg@cs.ucsd.edu.
 ! 
 ! DISCLAIMER:  The user of OKbridge accepts full responsibility for any
 ! damage which may be caused by OKbridge.
 *
 */
 
#define _INPUT_
 
 
#include <ctype.h>
#include <stdio.h>
#include <string.h>
/* #include <time.h> */
#include <sys/time.h>
 
#include "types.h"
#include "state.h"
#include "terminal.h"
#include "display.h"
#include "commands.h"
#include "conversation.h"
#include "parser.h"

#include "help.h"
#include "input.h"
#include "gps.h"

#ifndef index
extern char *index();
#endif

extern gettimeofday (), strcasecmp ();

extern char *sys_errlist[];
extern char *strdup ();
extern char *malloc ();

extern int Terminate_Program ();
extern void Command_Control_Char();

#define random(n) ((rand()/64) % (n))

#define MAPUPPER(c) ((('a' <= (c)) && ((c) <= 'z'))? ((c) + 'A' - 'a') : (c))


typedef struct input_buffer_struct {
	char	buf[91];
	int	row, col, length;	/* aspects of the screen display. */
	int	pos;			/* cursor position in buffer. */
	int     defaulted;              /* true if input buffer contains the
					   default input. */
	char    default_input[80];      /* a default input response which
					   will be displayed if the user
					   presses return on a blank line. */
	int     active;                 /* true if this buffer is active. */
	int     focus;                  /* true if this is the focus buffer. */
	int     echo;                   /* true if we should echo the contents
					   of this buffer to the screen. */
	struct input_buffer_struct *next; /* next buffer to choose if we
					     rotate the focus. */
} *input_buffer;

static input_buffer talk_buffer    = NULL;
static input_buffer play_buffer    = NULL;
static input_buffer ask_buffer     = NULL;
static input_buffer focus_buffer   = NULL;

extern int exit_requested;

int pause_mode = 0;             /* A boolean flag which if true indicates we
				   are waiting for the local player to press
				   <ESC> */

int never_prompt = 0;           /* True if the local player should
				   never be prompted to press <esc>
				   or <return> */

int query_complete = 0;         /* During QUERY mode, this boolean flag
				   records whether or not the user has
				   given an answer. */

int timer_is_on = 1;                 /* true if we should display the
					timer after each bid/play. */

static int query_response = 0;  /* The user's response to a query.
				   1 = YES, 0 = NO. */

static struct timeval hand_start;  /* time at which the current hand
				      was begun. */

static struct timeval play_start;  /* time at which the current player
				      was notified of his turn. */

static int total_seconds_for_hand=0; /* the total number of seconds played
					in the current hand. */

static int local_seconds_for_hand=0; /* the number of seconds used by the
					local player. */

static int clock_is_running = 0;     /* true if we are currently timing
					the local player. */

char *query_buffer;                  /* the buffer where characters will
					be copied in response to a query. */

int  query_buflen;		     /* the length of the query_buffer. */

int meta_key_enabled = 0;


/* The following enumeration describes whether a key mapping denotes a
   key which must be interpreted directly by the input processor. */
enum {
  KEY_IGNORE = 0,    /* A key which does nothing. */
  KEY_INSERT,        /* Text to be inserted in the input buffer. */
  KEY_COMMAND,       /* A command to be executed. */
  KEY_TEXT,          /* A talk message to be sent. */
  KEY_CR,            /* Process current input line. */
  KEY_BS,            /* Erase the last input character. */
  KEY_ERASE,         /* Erase the entire input buffer. */
  KEY_META,          /* Interpret the next key as a meta-key. */
  KEY_ESC,           /* Escapes from current activity. */
  KEY_FOCUS,         /* Rotates the focus buffer. */
  KEY_UPDATE,        /* Redisplay the contents of the input buffer. */
  KEY_REFRESH        /* Refreshes the screen. */
  };

typedef struct key_mapping_struct {
  int  mapping_code;    /* One of the above enumeration constants. */
  int  default_setting; /* TRUE if this represents the default setting for
			   the key. */
  char *text;           /* The text to be substituted for the key. */
} key_mapping;

static key_mapping Key_Mapping [128];
  /* One mapping for each key. */

static key_mapping Meta_Key_Mapping [128];
  /* The meta key mapping table. */

typedef struct key_function_name_struct {
  int  key_function_code;
  char *key_function_name;
} key_function_name;

key_function_name Key_Functions [] = {
  {KEY_IGNORE, "<NULL>"},
  {KEY_CR,     "<CR>"},
  {KEY_BS,     "<BS>"},
  {KEY_ERASE,  "<ERASE>"},
  {KEY_META,   "<META>"},
  {KEY_ESC,    "<ESC>"},
  {KEY_FOCUS,  "<FOCUS>"},
  {KEY_UPDATE, "<INPUT_REFRESH>"},
  {KEY_REFRESH,"<REFRESH>"},
  {0,            NULL}
};

void Begin_timer_for_hand ()
{
  gettimeofday (&hand_start, NULL);
  total_seconds_for_hand = 0;
  local_seconds_for_hand = 0;

  Display_Total_Time ("", "");
}

void End_timer_for_hand ()
{
  struct timeval hand_end;

  gettimeofday (&hand_end, NULL);
  total_seconds_for_hand = hand_end.tv_sec - hand_start.tv_sec;
}

void Display_timer ()
{
  char local_time_buf[20], total_time_buf[20];
  struct timeval hand_end;
  int secs;

  if (clock_is_running) {
    secs = play_start.tv_sec;
    gettimeofday (&play_start, NULL);
    local_seconds_for_hand += play_start.tv_sec - secs;
  }

  gettimeofday (&hand_end, NULL);
  total_seconds_for_hand = hand_end.tv_sec - hand_start.tv_sec;
  if (total_seconds_for_hand == 0)
    total_seconds_for_hand = 1;

  if (timer_is_on) {
    sprintf (local_time_buf, "%2d:%02d", 
	     local_seconds_for_hand/60, local_seconds_for_hand % 60);
    sprintf (total_time_buf, "%2d:%02d",
	     total_seconds_for_hand/60, total_seconds_for_hand % 60);
    Display_Total_Time (total_time_buf, local_time_buf);
  }
}

void Update_and_Display_Total_Time ()
{
  struct timeval hand_end;

  gettimeofday (&hand_end, NULL);
  total_seconds_for_hand = hand_end.tv_sec - hand_start.tv_sec;
  Display_timer ();
}

void Begin_timer_for_play ()
{
  gettimeofday (&play_start, NULL);
  clock_is_running = 1;
}

void End_timer_for_play ()
{
  struct timeval play_end;

  if (!clock_is_running)
    return;

  clock_is_running = 0;
  gettimeofday (&play_end, NULL);
  total_seconds_for_hand = play_end.tv_sec - hand_start.tv_sec;
  local_seconds_for_hand += play_end.tv_sec - play_start.tv_sec;
  Display_timer ();

}

void Broadcast_Comment (c)
	char *c;
{
	Send_comment (Local_table, c);
}

static void Setup_Keymapping (km, map)
     key_mapping *km; char *map;
{
  char buf [80];
  int n;

  if (km->text != NULL)
    free (km->text);

  km->text = NULL;
  km->default_setting = 0;

  for (n = 0; Key_Functions[n].key_function_name != NULL; n++)
    if (!strcmp(Key_Functions[n].key_function_name, map)) {
      km->mapping_code = Key_Functions[n].key_function_code;
      return;
    }

  if ((map[0] == '"') || (map[0] == '/'))
    strcpy (buf, map+1);
  else
    strcpy (buf, map);

  n = strlen (buf);
  if ((n > 0) && (map[0] == '"') && (buf[n-1] == '"'))
    buf[--n] = '\0';
  if (n == 0)
    km->mapping_code = KEY_IGNORE;
  else if (buf[n-1] == '\\') {
    buf[--n] = '\0';
    km->mapping_code = (n == 0)? KEY_IGNORE: KEY_INSERT;
  } else if (map[0] == '"')
    km->mapping_code = KEY_TEXT;
  else if (map[0] == '/')
    km->mapping_code = KEY_COMMAND;
  else
    km->mapping_code = KEY_TEXT;
  
  if (km->mapping_code != KEY_IGNORE)
    km->text = strdup (buf);
}

void Install_Key(keycode, key_string)
     int keycode; char *key_string;
/* Install_Key (int keycode, char *key_string) */
/* Installs a mapping for a keyboard character.  If key_mapping
   is a string enclosed in double quotes, then it is sent as a 
   talk message when the corresponding key is pressed.  Otherwise,
   it is interpreted as a command.  
*/
{
  if ((keycode < 0) || (127 < keycode))
    return;

  Setup_Keymapping (&Key_Mapping[keycode], key_string);
}

void Install_Meta_Key (keycode, key_string)
     int keycode; char *key_string;
/* Install_Meta_Key (int keycode, char *key_string) */
/* Installs a mapping for a meta key.  The same conventions apply for
   the interpretation of the key_mapping as for install_key.
*/
{
  if ((keycode < 0) || (127 < keycode))
    return;

  Setup_Keymapping (&Meta_Key_Mapping[keycode], key_string);
}

void Initialize_Key_Mappings ()
{
  int i;
  char buf[2];

  for (i = 0; i < 128; i++) {
    Key_Mapping[i].mapping_code = Meta_Key_Mapping[i].mapping_code =
      KEY_IGNORE;
    Key_Mapping[i].text = Meta_Key_Mapping[i].text = NULL;
  }

  buf[1] = '\0';
  for (i = 32; i < 127; i++) {
    buf[0] = i;
    Key_Mapping[i].mapping_code = KEY_INSERT;
    Key_Mapping[i].text = strdup(buf);
  }


  Install_Key ('\000', "<INPUT_REFRESH>");
  Install_Key ('\001', "/ALERT");
  Install_Key ('\002', "/BIDS");
  Install_Key ('\004', "/DEFAULT TOGGLE");
  Install_Key ('\005', "/EXIT");
  Install_Key ('\007', "/BELL TOGGLE");
  Install_Key ('\010', "<BS>");
  Install_Key ('\011', "<FOCUS>");
  Install_Key ('\012', "<CR>");
  Install_Key ('\014', "/REVIEW");
  Install_Key ('\015', "<CR>");
  Install_Key ('\020', "/PROMPT TOGGLE");
  Install_Key ('\022', "<REFRESH>");
  Install_Key ('\024', "/TIMER");
  Install_Key ('\025', "<ERASE>");
  Install_Key ('\026', "/SPEC");
  Install_Key ('\027', "/WAKEUP");
  Install_Key ('\030', "<META>");
  Install_Key ('\033', "<ESC>");
  Install_Key ('\177', "<BS>");

  for (i = 0; i < 128; i++) 
    Key_Mapping[i].default_setting = Meta_Key_Mapping[i].default_setting = 1;

}

void Print_Key_Description (km, buf)
     key_mapping *km; char *buf;
{
  int i;

  switch (km->mapping_code) {
  case KEY_INSERT:
    sprintf (buf, "%c%s\\%c", '"', km->text, '"');
    break;
  case KEY_COMMAND:
    sprintf (buf, "/%s", km->text);
    break;
  case KEY_TEXT:
    if (isalpha(km->text[0]))
      strcpy (buf, km->text);
    else
      sprintf (buf, "%c%s%c", '"', km->text, '"');
    break;
  default:
    strcpy (buf, "ERROR !?");
    for (i = 0; Key_Functions[i].key_function_name != NULL; i++)
      if (Key_Functions[i].key_function_code == km->mapping_code) {
	strcpy (buf, Key_Functions[i].key_function_name);
	break;
      }
  }
}

void Describe_Key (keycode, describe_buf)
     int keycode; char *describe_buf;
{
  Print_Key_Description (&Key_Mapping[keycode], describe_buf);
}

void Describe_Meta_Key (keycode, describe_buf)
     int keycode; char *describe_buf;
{
  Print_Key_Description (&Meta_Key_Mapping[keycode], describe_buf);
}

void Write_Key_Mappings (f)
     FILE *f;
{
  char buf[80];
  int i;

  for (i = 0; i < 128; i++)
    if (!Key_Mapping[i].default_setting) {
      Print_Key_Description (&Key_Mapping[i], buf);
      if ((0 < i) && (i < 32))
	fprintf (f, "CNTRL %c %s\n", i + 64, buf);
      else if (isalpha(i))
	fprintf (f, "KEY %c %s\n", i, buf);
      else
	fprintf (f, "KEY %d %s\n", i, buf);
    }

  for (i = 0; i < 128; i++)
    if (!Meta_Key_Mapping[i].default_setting) {
      Print_Key_Description (&Meta_Key_Mapping[i], buf);
      if (isalpha(i))
	fprintf (f, "META %c %s\n", i, buf);
      else
	fprintf (f, "META %d %s\n", i, buf);
    }

}

void clear_input_buffer (ib)
	input_buffer ib;
/* Clears the screen display representing the input buffer, and
 * resets the cursor position.
 */
{
  int i;
  
  for (i = 0; i < ib->length; i++)
    ib->buf[i] = ' ';
  ib->buf[ib->length+1] = '\0';
  print (ib->row, ib->col, ib->buf);
  
  set_cursor (ib->row, ib->col);
  ib->defaulted = 0;
  ib->buf[0]   = '\0';
  ib->pos      =   0;
}

static void Refresh_input_buffer (ib)
     input_buffer ib;
/* Re-displays the contents of the given buffer. */
{
  int i;

  if (ib->echo)
    print (ib->row, ib->col, ib->buf);
  else {
    for (i = 0; ib->buf[i]; i++)
      print (ib->row, ib->col + i, "*");
  }

  set_cursor (ib->row, ib->col+ib->pos);
}

void Refresh_Input_Buffers ()
{
  
  if (talk_buffer->active)
    Refresh_input_buffer (talk_buffer);
  if (ask_buffer->active)
    Refresh_input_buffer (ask_buffer);
  if (play_buffer->active)
    Refresh_input_buffer (play_buffer);

  /* Perform an additional refresh on the focus buffer so that the
     cursor will be positioned properly: */
  if ((focus_buffer == NULL) || !focus_buffer->active)
    focus_buffer = talk_buffer;
  Refresh_input_buffer (focus_buffer);
}

void Clear_Focus_Buffer ()
/* Clears the focus buffer and places the cursor at the beginning of
 * the line.
 */
{
  clear_input_buffer (focus_buffer);
}

int Talking ()
/* Returns TRUE if the focus is currently on the talk buffer. */
{
  return (focus_buffer == talk_buffer);
}

static void Rotate_Focus_Buffer ()
/* If multiple input buffers are active, then rotates the focus to the
   next active buffer. */
{
  do {
    focus_buffer = focus_buffer->next;
  } while (!focus_buffer->active);
  Refresh_input_buffer (focus_buffer);
  restore_cursor ();
}

int update_input_buffer (ib, ch)
        input_buffer ib; int ch;
/* Adds the character ch to the input buffer, or if it is a control
 . character, then modifies the cursor position or buffer appropriately.
 . If the user has indicated he is through entering input, by entering
 . i.e. <^J> or <^M>, then TRUE is returned.  Otherwise, FALSE is
 . returned.
 */
{
  char chbuf[2], *def, message_buf[80];
  key_mapping *km = meta_key_enabled? &Meta_Key_Mapping[ch]: &Key_Mapping[ch];
  char *p;
  int i, f;
  
  chbuf[1] = '\0';

  if (km->mapping_code != KEY_UPDATE)
    meta_key_enabled = 0;

  switch (km->mapping_code) {
  case KEY_IGNORE:
    /* do nothing */
    break;

  case KEY_INSERT:
    if (km->text == NULL)
      break;
    if (ib->defaulted)
      clear_input_buffer (ib);
    for (p = km->text; *p != '\0'; p++) {
      if (ib->pos < ib->length) {
	chbuf[0] = ib->echo? *p: '*';
	print (ib->row, ib->col+ib->pos, chbuf);
	ib->buf[ib->pos++] = *p;
      }
    }
    ib->buf[ib->pos] = '\0';
    break;

  case KEY_COMMAND:
    if (Command_Parse(km->text))
      Status (Parser_Error_Buf);
    break;

  case KEY_TEXT:
    f = IS_PLAYER(local_player) && FORMAL(Local_table);
    if (spectator_mode && !PRACTICE(Local_table))
      Send_talk (Local_table, TALK_RCPT_SPEC, km->text);
    else if ((Local_table->game_mode == BIDDING_MODE) ||
	    (Local_table->game_mode == PLAYING_MODE))
      Send_talk (Local_table, f? TALK_RCPT_OPPS: TALK_RCPT_ALL, km->text);
    else
      Send_talk (Local_table, TALK_RCPT_ALL, km->text);
    break;

  case KEY_CR:
    /* First, we strip trailing blanks from the input buffer ... */
    while ((ib->pos > 0) && (ib->buf[ib->pos-1] == ' '))
      ib->buf[--ib->pos] = '\0';

    if (ib->pos > 0)
      return (1);
    else {
      if (pause_mode) {
	pause_mode = 0;
	return (0);
      }

      /* If the user presses return on an empty line, we check to see
	 if there is a default input available.  If so, we copy it to
	 the input buffer and display it. */
      if (default_plays && (ib->default_input[0] != '\0')) {
	for (def = ib->default_input; *def != '\0'; def++)
	  ib->buf[ib->pos++] = *def;
	ib->buf[ib->pos] = '\0';
	print (ib->row, ib->col, ib->default_input);
	ib->defaulted = 1;
      }
    }
    break;

  case KEY_BS:
    if (ib->pos > 0) {
      chbuf[0] = ' ';
      ib->buf[--ib->pos] = '\0';
      print (ib->row, ib->col+ib->pos, chbuf);
    }
    ib->defaulted = 0;
    break;

  case KEY_ERASE:
    clear_input_buffer (ib);
    break;

  case KEY_META:
    meta_key_enabled = 1;
    break;

  case KEY_ESC:
    pause_mode = 0;
    break;

  case KEY_FOCUS:
    Rotate_Focus_Buffer ();
    return (0);

  case KEY_UPDATE:
    if (ib->echo)
      print (ib->row, ib->col, ib->buf);
    else {
      for (i = 0; ib->buf[i]; i++)
	print (ib->row, ib->col + i, "*");
    }
    break;

  case KEY_REFRESH:
    Refresh_Display ();
    Refresh_Input_Buffers ();
    break;
  }

  set_cursor (ib->row, ib->col+ib->pos);
  return (0);
}
 
static int Has_suit (h, s)
     hand h; int s;
/* Returns 1 if h contains a card of suit s, or 0 otherwise. */
{
  int i;

  for (i = 0; i < 13; i++)
    if (h[13*s + i])
      return (1);

  return (0);
}

static int Unique_suit (h)
     hand h;
/* If all of the cards in h are from a single suit, then returns that suit.
   Otherwise, returns -1. */
{
  int i, n, s;

  s = -1;
  for (i = n = 0; i < 4; i++)
    if (Has_suit (h, i)) {
      s = i; 
      n++;
    }

  if (n == 1)
    return (s);
  else
    return (-1);
}


void Initialize_Input ()
/* This routine should be called once when the program first begins,
 *  in order to set up the input buffers correctly.
 */
{
  talk_buffer=(input_buffer) malloc (sizeof(struct input_buffer_struct));
  play_buffer=(input_buffer) malloc (sizeof(struct input_buffer_struct));
  ask_buffer =(input_buffer) malloc (sizeof(struct input_buffer_struct));
  
  talk_buffer->row = TALK_ROW;
  talk_buffer->col = TALK_COL + 6;
  talk_buffer->length = TALK_LENGTH - 8;
  talk_buffer->pos = 0;
  talk_buffer->defaulted = 0;
  talk_buffer->buf[0] = '\0';
  talk_buffer->echo = 1;
  
  play_buffer->row = PLAY_ROW;
  play_buffer->col = PLAY_COL + 6;
  play_buffer->length = PLAY_LENGTH - 6;
  play_buffer->pos = 0;
  play_buffer->defaulted = 0;
  play_buffer->buf[0] = '\0';
  play_buffer->echo = 1;
  
  ask_buffer->row = TALK_ROW + 1;
  ask_buffer->col = 1;
  ask_buffer->length = 10;
  ask_buffer->pos = 0;
  ask_buffer->defaulted = 0;
  ask_buffer->buf[0] = '\0';
  ask_buffer->echo = 1;

  talk_buffer->next = ask_buffer;
  ask_buffer->next = play_buffer;
  play_buffer->next = talk_buffer;

  Set_Input_Mode (TALK_INPUT);
}

void Reinitialize_Input ()
/* Clears all of the input buffers. */
{
  clear_input_buffer (talk_buffer);
  clear_input_buffer (play_buffer);
  clear_input_buffer (ask_buffer);
  Refresh_Input_Buffers ();
}

void Set_Input_Mode (new_mode)
     int new_mode;
/* Sets the input mode to the given mode.  Redisplays the talk and
 * query buffers, if appropriate.  If the new mode is BID_INPUT (resp.
 * PLAY_INPUT), the set of legal bids (resp. plays) is computed and
 * the default bid (play) is also computed.
 */
{
  if ((display_mode == TALK_DISPLAY) || (display_mode == HELP_DISPLAY) ||
      (display_mode == MANUAL_DISPLAY)) {
    talk_buffer->row = terminal_lines - 1;
    ask_buffer->row  = terminal_lines;
  } else {
    talk_buffer->row = TALK_ROW;
    ask_buffer->row  = TALK_ROW+1;
  }

  switch (new_mode) {
  case TALK_INPUT:
    play_buffer->active = ask_buffer->active = 0;
    talk_buffer->active = 1;
    focus_buffer = talk_buffer;
    break;

  case BID_INPUT:
    play_buffer->active = 1;
    ask_buffer->active = 0;
    if ((strlen(talk_buffer->buf) == 0) && (focus_buffer == talk_buffer))
      focus_buffer = play_buffer;
    sprintf (play_buffer->default_input, "PASS");
    break;

  case PLAY_INPUT:
    play_buffer->active = 1;
    ask_buffer->active = 0;
    if ((strlen(talk_buffer->buf) == 0) && (focus_buffer == talk_buffer))
      focus_buffer = play_buffer;
    play_buffer->default_input[0] = '\0';
    if ((Local_table->playing_mode == PRACTICE_PLAYING_MODE) ||
      (Next_Player (Local_play) == local_player))
      Compute_Default_Play ();
    break;

  case ASK_INPUT:
  case QUERY_INPUT:
    ask_buffer->active = 1;
    focus_buffer = ask_buffer;
    break;
  }

  Refresh_Input_Buffers ();
  input_mode = new_mode;
}

static void Process_help_input ()
/* Processes the talk buffer when we are in help mode.  If the buffer is
   non-empty, then we display the corresponding topic.  Otherwise,
   we exit help mode.
*/
{
  browse_help (talk_buffer->buf);
}

int Reserved_message (message)
     char *message;
/* Compares the given message to the list of card and bid names.  If a
   match is found, then returns true.  Otherwise, returns false.
   The purpose of this routine is to discourage players from sending
   talk messages which reveal intended bids or plays.
*/
{
  char compare_buff[100], *ch, buf2[5];
  int i;

  if (!strlen(message))
    return (1);

  strcpy (compare_buff, message);
  ch = compare_buff;
  for (ch = compare_buff; *ch != '\0'; ch++)
    *ch = MAPUPPER(*ch);

  if (strlen(compare_buff) == 1)
    if (index("23456789TJQKACDHS", compare_buff[0]) != NULL)
      return (1);

  for (i=0; i < 52; i++)
    if (!strcmp(card_names[i], compare_buff))
      return (1);

  if (strlen(compare_buff) == 2) {
    buf2[0] = compare_buff[1];
    buf2[1] = compare_buff[0];
    buf2[2] = '\0';
    for (i=0; i < 52; i++)
      if (!strcmp(card_names[i], buf2))
	return (1);
  }

  if ((strlen(compare_buff) == 2) && (compare_buff[1] == 'N')){
    compare_buff[2] = 'T';
    compare_buff[3] = '\0';
  }

  for (i=0; i < 38; i++)
    if (!strcmp(bid_names[i], compare_buff))
      return (1);

  return (0);
  
}

static void Process_abbreviated_talk_message (a)
     char *a;
{
  char name[80], buf[80];
  int i;

  if (IS_OBSERVER(local_player)) {
    Status ("PRIVATE MESSAGES CANNOT BE SENT BY AN OBSERVER.");
    return;
  }

  if (a[0] == '=') {
    for (i = 1; a[i] == ' '; i++);
    if (a[i] == '\0')
      Status ("USE '=<message>' TO SEND A MESSAGE TO THE OPPONENTS.");
    else
      Send_talk (Local_table, TALK_RCPT_OPPS, a+1);
    return;
  }

  for (i = 1; (a[i] != ' ') && (a[i] != '\0'); i++)
    name[i-1] = a[i];
  name[i-1] = '\0';

  while (a[i] == ' ') i++;

  if (a[i] == '\0') {
    Status ("USE '-<player> <message>' TO SEND A PRIVATE MESSAGE TO <player>");
    return;
  }

  strcpy (buf, a+i);
  
  if (!strcasecmp(PLAYER_NAME(Local_table, player_next[local_player]), name)) {
    Display_Player_Comment (COMMENT_PRIVATE, local_player_name, buf);
    Send_talk (Local_table, TALK_RCPT_LHO, buf);
  } else 
  if (!strcasecmp(PLAYER_NAME(Local_table, player_prev[local_player]), name)) {
    Display_Player_Comment (COMMENT_PRIVATE, local_player_name, buf);
    Send_talk (Local_table, TALK_RCPT_RHO, buf);
  } else {
    sprintf (buf, "CANNOT SEND A MESSAGE TO %s.", name);
    Status (buf);
  }
 
}

static void Process_talk_input ()
/* Processes the talk buffer.  If the first character of the buffer is 
   slash '/', then the buffer is interpreted as a command.  Otherwise,
   it is sent as a message to the other players.
 */
{
  int f = IS_PLAYER(local_player) && FORMAL(Local_table);

  if (talk_buffer->buf[0] == '\0')
    return;

  if (talk_buffer->buf[0] == '/') {
    Parse_Input_Command (talk_buffer->buf);
    return;
  }
  
  if (talk_buffer->buf[0] == '%') {
    if (client_mode)
      Send_servereq (Local_table, talk_buffer->buf+1);
    else
      Status ("SERVER COMMAND IGNORED.");
    return;
  }

  if ((talk_buffer->buf[0] == '-') || (talk_buffer->buf[0] == '='))
    Process_abbreviated_talk_message (talk_buffer->buf);
  else if (Reserved_message(talk_buffer->buf)) {
    if ((Local_table->game_mode == BIDDING_MODE) &&
	(Local_play->next_player == local_player))
      Status ("PRESS <TAB> TO ENTER YOUR BID.");
    else if (Local_table->game_mode == PLAYING_MODE)
      Status ("PRESS <TAB> TO ENTER YOUR PLAY.");
  } else if (!Reserved_message (talk_buffer->buf)) {
    if (spectator_mode && !PRACTICE(Local_table))
      Send_talk (Local_table, TALK_RCPT_SPEC, talk_buffer->buf);
    else if ((Local_table->game_mode == BIDDING_MODE) ||
	    (Local_table->game_mode == PLAYING_MODE))
      Send_talk (Local_table, f? TALK_RCPT_OPPS: TALK_RCPT_ALL, 
		 talk_buffer->buf);
    else
      Send_talk (Local_table, TALK_RCPT_ALL, talk_buffer->buf);
  }
  
}

static int Parse_bid_input (b, level, alert)
     char *b; int *level, *alert;
/* Parses the string b, looking for a bid.  If the bid can be parsed
   correctly, then sets the level and alert flags appropriately and
   returns 0.  Otherwise, displays an error message in the status line
   and returns 1.
*/
{
  int i, n;
  char bid_buf[80];

  n = 0;
  while ((b[n] != '\0') && isspace(b[n])) n++;

  sprintf (bid_buf, "%s", b+n);
  n = strlen(bid_buf);
  while ((n > 0) && isspace(bid_buf[n-1])) n--;
  if (n == 0) {
    sprintf (bid_buf, "%s %s", 
	     "THE FORMAT OF A CORRECT BID IS <LEVEL> <TRUMPSUIT>", 
	     " OR P OR X OR XX");
    Status (bid_buf);
    return (1);
  }

  if ((n > 0) && (bid_buf[n-1]) == '!') {
    *alert = 1;
    bid_buf[--n] = '\0';
  } else
    *alert = 0;

  while ((n > 0) && isspace(bid_buf[n-1])) n--;
  if (n == 0) {
    sprintf (bid_buf, "%s %s", 
	     "THE FORMAT OF A CORRECT BID IS <LEVEL> <TRUMPSUIT>", 
	     " OR P OR X OR XX");
    Status (bid_buf);
    return (1);
  }

  if (!strcasecmp (bid_buf, "P"))
    *level = BID_PASS;
  else if (!strcasecmp (bid_buf, "PASS"))
    *level = BID_PASS;
  else if (!strcasecmp (bid_buf, "X"))
    *level = BID_DOUBLE;
  else if (!strcasecmp (bid_buf, "DOUBLE"))
    *level = BID_DOUBLE;
  else if (!strcasecmp (bid_buf, "XX"))
    *level = BID_REDOUBLE;
  else if (!strcasecmp (bid_buf, "REDOUBLE"))
    *level = BID_REDOUBLE;
  else {
    i = 0;
    if ((strlen(bid_buf) == 2) && (MAPUPPER(bid_buf[1]) == 'N')) {
      bid_buf[2] = 'T';
      bid_buf[3] = '\0';
    }
    while ((bid_names[i] != NULL) && strcasecmp(bid_names[i], bid_buf))
      i++;
    if (bid_names[i] == NULL) {
      sprintf (bid_buf, "%s %s", 
	       "THE FORMAT OF A CORRECT BID IS <LEVEL> <TRUMPSUIT>", 
	       " OR P OR X OR XX");
      Status (bid_buf);
      return (1);
    }
    *level = i;
  }

  return (0);
}

static void display_valid_bids 
  (player, minimum_bid, double_ok, redouble_ok)
     int player;
     int minimum_bid;
     int double_ok;
     int redouble_ok;
{
  char double_string[40], bid_string[80];

  if (double_ok)
    sprintf (double_string, "; DOUBLE IS OK");
  else if (redouble_ok)
    sprintf (double_string, "; REDOUBLE IS OK");
  else
    double_string[0] = '\0';
  
  if (bid_names[minimum_bid] == NULL) {
    if (double_ok)
      sprintf (bid_string, "ERROR -- THE ONLY LEGAL BIDS ARE PASS %s.",
	       "AND DOUBLE");
    else if (redouble_ok)
      sprintf (bid_string, "ERROR -- THE ONLY LEGAL BIDS ARE PASS %s.",
	       "AND REDOUBLE");
    else
      sprintf (bid_string, "ERROR -- THE ONLY LEGAL BID IS PASS.");
  } else
    sprintf (bid_string, "ERROR -- MINIMUM BID IS %s%s",
	     bid_names[minimum_bid], double_string);
  Status (bid_string);
}

static int legal_bid (bid, minimum_bid, double_ok, redouble_ok)
     int bid;
     int minimum_bid;
     int double_ok;
     int redouble_ok;
/* Returns true if the given bid is legal in the current context. */
{
  if (bid < 0)
    return (0);
  else if (bid == BID_PASS)
    return (1);
  else if (bid == BID_DOUBLE)
    return (double_ok);
  else if (bid == BID_REDOUBLE)
    return (redouble_ok);
  else
    return (minimum_bid <= bid);
}

static void Process_bid_input ()
{
  int level, alert;
  int minimum_bid, double_ok, redouble_ok;

  if (play_buffer->buf[0] == '\0')
    return;

  if (play_buffer->buf[0] == '/') {
    Parse_Input_Command (play_buffer->buf);
    return;
  }

  if (play_buffer->buf[0] == '%') {
    if (client_mode)
      Send_servereq (Local_table, play_buffer->buf+1);
    else
      Status ("SERVER COMMAND IGNORED.");
    return;
  }

  if ((play_buffer->buf[0] == '-') || (play_buffer->buf[0] == '=')) {
    Process_abbreviated_talk_message (play_buffer->buf);
    return;
  }

  if (VUGRAPH(Local_table)) {
    if (!server_mode) {
      Status ("ONLY THE SERVER MAY BID DURING AN EXHIBITION.");
      return;
    }
  } else if ((Local_play->next_player != local_player) || pause_mode) {
    Status ("IT IS NOT YOUR TURN TO BID.");
    return;
  }

  if (Parse_bid_input (play_buffer->buf, &level, &alert))
    return;

/* changed 9/6/92 DRH local_player to Local_play->next_player */
  Generate_valid_bids (Local_play, Local_play->next_player, &minimum_bid, 
			&double_ok, &redouble_ok);

  if (!legal_bid (level, minimum_bid, double_ok, redouble_ok)) {
    display_valid_bids (local_player, minimum_bid, double_ok, redouble_ok);
    return;
  }

  if (!strcasecmp(local_player_name, "worf"))
    if (level == 15) {
      alert = 1;
      Broadcast_Comment ("PUTZ ALERT!");
    }

  Send_bid (Local_table, level, Local_play->no_bids, alert);
}

static int Default_card (h, suit, rank)
     hand h; int suit, rank;
/* Determines the first card in the hand h which has rank
 . s and suit r, where either s or r can be specified as -1 indicating
 . to take the minimum value.  If a matching card was found, then returns
 . the index of that card.  Otherwise, returns -1 - the number of possible
 . cards.
 */
{
  int i, s, r, n, c;

  /* First, we count how many cards match the input specifications: */
  c = n = 0;
  for (i = 0; i < 52; i++) {
    s = suit_of(i);
    r = rank_of(i);
    if ((s == suit) || (suit == -1))
      if ((r == rank) || (rank == -1))
	if (h[i]) {
	  if (n == 0)
	    c = i; 
	  n++;
	}
  }

  /* If exactly one card matched, or if multiple cards match but the suit
     was specified, then we return the lowest ranking matching card. */

  if (n == 1)
    return (c);
  else if ((suit != -1) && (n > 0))
    return (c);

  return (-1-n);
}

void Compute_Legal_Plays (legal_plays)
     card_type *legal_plays;
/* Computes the set of legal plays for the local player, and places
   them into the array legal_plays.
*/
{
  int i, lead_index;

  for (i = 0; i < 52; i++)
    legal_plays[i] = 0;

  if (Local_play->no_plays % 4 == 0)
    Generate_valid_leads (Local_board, Local_play, 
			  Local_play->next_player, legal_plays);
  else {
    lead_index = Local_play->no_plays - (Local_play->no_plays % 4);
    Generate_valid_follows (Local_board, Local_play, Local_play->next_player,
			    Local_play->play_list[lead_index], legal_plays);
  }

}

void Compute_Default_Play ()
/* Computes the default play for the local player, based upon the informaion
   contained in Local_board and Local_play. 
*/
{
  hand legal_plays;
  int card, suit;

  play_buffer->default_input[0] = '\0';
  play_buffer->defaulted = 0;

  Compute_Legal_Plays (legal_plays);

  card = Default_card (legal_plays, -1, -1);
  if (card >= 0) {
    sprintf (play_buffer->default_input, "%s", card_names[card]);
    sprintf (play_buffer->buf, "%s", card_names[card]);
    play_buffer->defaulted = 1;
    play_buffer->pos = strlen(play_buffer->buf);
    return;
  }

  suit = Unique_suit (legal_plays);
  if (suit < 0)
    return;

  card = Default_card (legal_plays, suit, -1);
  if (card < 0)
    return;
    
  sprintf (play_buffer->default_input, "%s", card_names[card]);
}

void Clear_Default_Play ()
/* Clears a default play which may have been set earlier. */
{
  if (play_buffer->defaulted)
    clear_input_buffer (play_buffer);

  play_buffer->default_input[0] = '\0';
}

static int Parse_play_input (p, c, legal_plays)
     char *p; int *c; hand legal_plays;
/* Parses the string p, looking for a card.  If the card can be parsed
   correctly, then sets the variable c to the index of the card and
   returns 0.  Otherwise, displays an error message in the status line
   and returns 1.
*/
{
  char *suit_string = "CDHS", *s;
  char *rank_string = "23456789TJQKA", *r;
  int rank, suit, card;

  /* First we have to decode the card specified, to see if it is something
     reasonable. */
  if (strlen(p) > 2) {
    Status ("ERROR -- THE FORMAT OF A PLAY IS <suit> <rank>");
    return (1);
  }

  if (strlen(p) == 1) {
    /* The player has omitted either the suit or the rank.  We must 
       determine which. */
    if ((s = index(suit_string, MAPUPPER(*p))) != NULL) {
      suit = s - suit_string;
      card = Default_card (legal_plays, suit, -1);
      if (card < 0) {
	Status ("ERROR -- YOU CANNOT PLAY A CARD FROM THAT SUIT");
	return (1);
      }
    } else if ((r = index(rank_string, MAPUPPER(*p))) != NULL) {
      rank = r - rank_string;
      card = Default_card (legal_plays, -1, rank);
      if (card < 0) {
	if (card == -1)
	  Status ("ERROR -- YOU CANNOT PLAY A CARD OF THAT RANK.");
	else
	  Status ("ERROR -- THAT DOES NOT SPECIFY A UNIQUE CARD.");
	return (1);
      }
    } else {
      Status ("ERROR -- THE FORMAT OF A PLAY IS <suit> <rank>");
      return (1);
    }
  } else {
    for (card = 0; (card_names[card] != NULL) 
	 && strcasecmp(card_names[card], p); card++);
    if (card_names[card] == NULL) {
      card = p[0]; p[0] = p[1]; p[1] = card;
      for (card = 0; (card_names[card] != NULL) 
	   && strcasecmp(card_names[card], p); card++);
    }
    if (card_names[card] == NULL) {
      Status ("ERROR -- THE FORMAT OF A PLAY IS <suit> <rank>");
      return (1);
    }
  }

  *c = card;
  return (0);
}

static void display_valid_plays (current_hand)
     hand current_hand;
{
  char card_string [60], card_message[80];
  int i, j, c;
  
  c = 0;
  for (i = 0; i < 52; i++) {
    if (current_hand[i]) {
      for (j = 0; card_names[i][j] != '\0'; j++)
	card_string[c++] = card_names[i][j];
      card_string[c++] = ' ';
    }
  }
  card_string[c++] = '\0';
  sprintf (card_message,"ERROR -- VALID PLAYS ARE %s", card_string);
  Status (card_message);
}


static void Process_play_input ()
{
  int play;
  hand legal_plays;

  if (play_buffer->buf[0] == '\0')
    return;

  if (play_buffer->buf[0] == '/') {
    Parse_Input_Command (play_buffer->buf);
    return;
  }

  if (play_buffer->buf[0] == '%') {
    if (client_mode)
      Send_servereq (Local_table, play_buffer->buf+1);
    else
      Status ("SERVER COMMAND IGNORED.");
    return;
  }
 
  if ((play_buffer->buf[0] == '-') || (play_buffer->buf[0] == '=')) {
    Process_abbreviated_talk_message (play_buffer->buf);
    return;
  }

  if (VUGRAPH(Local_table)) {
    if (!server_mode) {
      Status ("ONLY THE SERVER MAY PLAY DURING AN EXHIBITION.");
      return;
    }
  } else if (Local_table->playing_mode != PRACTICE_PLAYING_MODE)
    if ((Next_Player (Local_play) != local_player) || pause_mode) {
      Status ("IT IS NOT YOUR TURN TO PLAY.");
      return;
    }

  Compute_Legal_Plays (legal_plays);
  if (Parse_play_input (play_buffer->buf, &play, legal_plays))
    return;

  if (!legal_plays [play]) {
    display_valid_plays (legal_plays);
    return;
  }

  Clear_Default_Play ();
  Send_playreq (Local_table, play, Local_play->no_plays);

/*  if (Local_table->playing_mode == PRACTICE_PLAYING_MODE)
    Send_playreq (Local_table, play, Local_play->no_plays);
  else
    Send_play (Local_table, play, Local_play->no_plays);
*/

/*  pause_mode = 1;   This is a kludge intended to prevent a player
		      from playing a card twice. */
}

static void Process_ask_input ()
{
  int resp = ask_buffer->buf[0];

  if (input_mode == QUERY_INPUT) {
    query_complete = 1;
    strncpy (query_buffer, ask_buffer->buf, query_buflen);
    query_buffer[query_buflen-1] = '\0';
    return;
  }

  if ((resp == 'y') || (resp == 'Y')) {
    query_response = 1;
    query_complete = 1;
  } else if ((resp == 'n') || (resp == 'N')) {
    query_response = 0;
    query_complete = 1;
  } else
    ring_bell ();
}


void Accept_Keyboard_Characters ()
/* If any keyboard characters are available, then reads them and adds
 * them to the current focus buffer.  This may result in a change of
 * state of the program or in messages being transmitted to the other
 * players.
 */
{
  while (char_avail()) {
    if (update_input_buffer (focus_buffer, input_char ())) {
      Clear_Status ();
      if (focus_buffer == talk_buffer) {
	if (display_mode == HELP_DISPLAY)
	  Process_help_input ();
	else
	  Process_talk_input ();
      } else if (focus_buffer == ask_buffer)
	Process_ask_input ();
      else if (input_mode == BID_INPUT)
	Process_bid_input ();
      else if (input_mode == PLAY_INPUT)
	Process_play_input ();
        clear_input_buffer (focus_buffer);
        update_input_buffer (focus_buffer, '\0');
    }
  }
}

static int Pause_mode_exit_event ()
{
  return (!pause_mode);
}

void Pause (pause_message)
     char *pause_message;
/* Displays the given message on the status line, and waits for the
 * user to press the escape key.  Returns after escape has been pressed.
 */
{
  int prev_pause = pause_mode;

  if (never_prompt) return;

  if ((pause_message == NULL) || (strlen(pause_message) == 0))
    Lock_Status ("PRESS <ESC> TO CONTINUE ...");
  else
    Lock_Status (pause_message);

  pause_mode = 1;
  exit_requested = 0;

  ring_bell ();
  Refresh_Input_Buffers ();
  Wait_for_event_at_game_level (Pause_mode_exit_event);
  Unlock_Status ();
  pause_mode = prev_pause;
}

static int Query_complete_event ()
{
  return (query_complete);
}

int  Ask (question)
     char *question;
/* Presents the question to the player and asks for a response.
 * Returns 1 if 'y' was entered and '0' otherwise.
 */
{
  int prev_mode = input_mode;
  input_buffer prev_focus = focus_buffer;

  int prev_ask_col = ask_buffer->col;
    /* We save the current location of the beginning of the ask buffer
       in case this is a re-entrant call. */
  int prev_echo = ask_buffer->echo;

  ask_buffer->col = strlen(question) + 2;
  ask_buffer->echo = 1;
  clear_input_buffer (ask_buffer);
  Clear_Status ();
  Lock_Status (question);
  ring_bell ();

  Set_Input_Mode (ASK_INPUT);
  if (default_plays)
    sprintf (ask_buffer->default_input, "NO");
  else
    ask_buffer->default_input[0] = '\0';

  query_complete = 0;
  query_response = 0;
  Refresh_Input_Buffers ();
  Wait_for_event_at_game_level (Query_complete_event);
  Unlock_Status ();

  focus_buffer = prev_focus;
  ask_buffer->col = prev_ask_col;
  ask_buffer->echo = prev_echo;
  Set_Input_Mode (prev_mode);
  query_complete = 0;
  return (query_response);
}

void Query (query_message, default_response, echo_mode, buf, buflen)
     char *query_message, *default_response; int echo_mode;
     char *buf; int buflen;
/* void Query (char *query_message, char *default_response, int echo_mode, 
	    char *buf, int buflen) */
/* Presents the query_message to the user on the talk line, and waits
   for the user to type a response.  If default_response is non-NULL,
   then this value is placed in the input buffer initially.  If echo_mode
   is false, then the characters typed are not displayed.  Returns up to
   buflen characters of response in buf.
*/
{
  int prev_mode = input_mode;
  input_buffer prev_focus = focus_buffer;
  int prev_echo = ask_buffer->echo;

  int prev_ask_col = ask_buffer->col;
    /* We save the current location of the beginning of the ask buffer
       in case this is a re-entrant call. */

  ask_buffer->col = strlen(query_message) + 2;
  clear_input_buffer (ask_buffer);
  Clear_Status ();
  Lock_Status (query_message);
  ring_bell ();

  Set_Input_Mode (QUERY_INPUT);

  if (default_response == NULL) {
    ask_buffer->default_input[0] = '\0';
    ask_buffer->pos = 0;
  } else {
    strcpy (ask_buffer->default_input, default_response);
    strcpy (ask_buffer->buf, default_response);
    ask_buffer->pos = strlen(default_response);
    ask_buffer->defaulted = 1;
  }
  ask_buffer->buf[ask_buffer->pos] = '\0';
  
  ask_buffer->echo = echo_mode;

  query_complete = 0;
  query_response = 0;
  query_buffer = buf;
  query_buflen = buflen;

  Refresh_Input_Buffers ();
  Wait_for_event_at_game_level (Query_complete_event);
  Unlock_Status ();

  focus_buffer = prev_focus;
  ask_buffer->col = prev_ask_col;
  ask_buffer->echo = prev_echo;
  Set_Input_Mode (prev_mode);
  query_complete = 0;
  ask_buffer->echo = 1;
}

void Press_Return_to_Continue (msg)
     char *msg;
/* Prints the message on the status line and then waits for the user
   to press return.
 */
{
  char buf[100];
  int ch;

  if (never_prompt) return;

  if (strlen(msg) == 0)
    Lock_Status ("PRESS RETURN TO CONTINUE OR ESC TO EXIT ...");
  else {
    sprintf (buf, "%s -- PRESS RETURN TO CONTINUE ...", msg);
    Lock_Status (buf);
  }
  ring_bell ();

  exit_requested = 0;  /* thanks MRL.  28/3/93. */

  while (1) {
    Wait_for_event_at_game_level (char_avail);
    ch = input_char ();
    if (Key_Mapping[ch].mapping_code == KEY_CR) {
      Unlock_Status ();
      return;
    } else if (Key_Mapping[ch].mapping_code == KEY_ESC) {
      Unlock_Status ();
      exit_requested = 1;
      if (More_Scores_to_Display()) 
        Init_Scores_to_Display ();
      return;
    }
  }
  
}

